////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of core_2_ahb module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
// $HeadURL: $
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none
module core_2_ahb #( 
   parameter ADDR_WIDTH = 32                     // Address bus width
) (
   //$port_g Clock and reset
   input  wire                    clk,           // root Clock
   input  wire                    rst_n,         // Asynchronous reset (active low)

   //$port_g CORE Slave interface
   input  wire                    s_req,
   output wire                    s_gnt,
   output reg                     s_rvalid,
   input  wire                    s_we,
   input  wire              [3:0] s_be,
   input  wire   [ADDR_WIDTH-1:0] s_addr,
   input  wire             [31:0] s_wdata,
   output reg              [31:0] s_rdata,
   output wire                    s_err,

   //$port_g AHB Master interface
   output wire                    m_hready_in,
   output wire                    m_hsel,
   output reg    [ADDR_WIDTH-1:0] m_haddr,
   output wire              [1:0] m_htrans,
   output wire                    m_hwrite,
   output reg               [2:0] m_hsize,
   output reg              [31:0] m_hwdata,
   input  wire             [31:0] m_hrdata,
   input  wire              [1:0] m_hresp,
   input  wire                    m_hready
);


////////////////////////////////////////////////////////////////////////////////
// Local Parameters
////////////////////////////////////////////////////////////////////////////////
localparam 
   IDLE      = 2'b00,
   ADDR_DATA = 2'b01,
   DATA      = 2'b11;


////////////////////////////////////////////////////////////////////////////////
// Internal Signals
////////////////////////////////////////////////////////////////////////////////
reg            [1:0] state;
reg                  unalign;
reg            [2:0] n_hsize;
reg [ADDR_WIDTH-1:0] n_haddr;
reg            [2:0] ext_hsize;
reg [ADDR_WIDTH-1:0] ext_haddr;


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

// State Machine is used to convert unalign CORE write access into two align AHB
// write access.
//
// CORE ACCESS              AHB ACCESS
// 24bits write @0x0   ->   16bits write @0x0 +  8bits write @0x2
// 24bits write @0x1   ->    8bits write @0x1 + 16bits write @0x2
// 16bits write @0x1   ->    8bits write @0x1 +  8bits write @0x2
//
always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
      state <= IDLE;
   else
   begin
      case (state)
      IDLE:
         if (s_req & s_gnt & unalign)
            state <= ADDR_DATA;
         else if (s_req & s_gnt)
            state <= DATA;

      ADDR_DATA:
         if (m_hready)
            state <= DATA;

      DATA:
         if (s_req & s_gnt & unalign)
            state <= ADDR_DATA;
         else if (s_req & s_gnt)
            state <= DATA;
         else if (m_hready)
            state <= IDLE;
      endcase
   end
end


//------------------------------------------------------------------------------
// Second Access control in case of unalign CORE write access
//    Set the unalign flag and the second access parameters (size & addr)
//------------------------------------------------------------------------------
always @*
begin
   case  ({s_we,s_be})
   5'b1_0111:
   begin
      unalign = 1'b1;
      n_hsize = 3'b000;
      n_haddr = {s_addr[ADDR_WIDTH-1:2],2'b10};
   end
   5'b1_1110: 
   begin
      unalign = 1'b1;
      n_hsize = 3'b001;
      n_haddr = {s_addr[ADDR_WIDTH-1:2],2'b10};
   end
   5'b1_0110:
   begin
      unalign = 1'b1;
      n_hsize = 3'b000;
      n_haddr = {s_addr[ADDR_WIDTH-1:2],2'b10};
   end
   default  :
   begin
      unalign = 1'b0;
      n_hsize = 3'b000;
      n_haddr = {s_addr[ADDR_WIDTH-1:2],2'b10};
   end
   endcase
end

always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
   begin
      ext_haddr <= 32'h0;
      ext_hsize <= 3'b000;
   end
   else if (s_req & s_gnt & s_we)
   begin
      ext_haddr <= n_haddr;
      ext_hsize <= n_hsize;
   end
end


//------------------------------------------------------------------------------
// CORE Control
//------------------------------------------------------------------------------
`ifdef  CORE_2_AHB_WAITSTATE

always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
   begin
      s_rdata  <= 32'h0;
      s_rvalid <= 1'b0;
   end
   else
   begin
      s_rdata  <= m_hrdata;
      s_rvalid <= m_hready & state==DATA;
   end
end

assign s_gnt       = s_req & (m_hready & state==IDLE | s_rvalid);
assign s_err       = 1'b0;

`else //CORE_2_AHB_WAITSTATE

always @*
begin
   s_rdata     = m_hrdata;
   s_rvalid    = m_hready & state==DATA;
end
assign s_gnt       = s_req & m_hready & state!=ADDR_DATA;
assign s_err       = 1'b0;

`endif//CORE_2_AHB_WAITSTATE


//------------------------------------------------------------------------------
// AHB Control
//------------------------------------------------------------------------------
always @*
begin
   if (state==ADDR_DATA)
      m_haddr = ext_haddr;
   else
      casez ({s_we,s_be})
      5'b1_??10: m_haddr = {s_addr[ADDR_WIDTH-1:2],2'b01};
      5'b1_?100: m_haddr = {s_addr[ADDR_WIDTH-1:2],2'b10};
      5'b1_1000: m_haddr = {s_addr[ADDR_WIDTH-1:2],2'b11};
      default  : m_haddr = {s_addr[ADDR_WIDTH-1:2],2'b00};
      endcase

   if (state==ADDR_DATA)
      m_hsize = ext_hsize;
   else
      casez ({s_we,s_be})
      // 8Bits
      5'b1_0001,
      5'b1_??10,
      5'b1_0100,
      5'b1_1000: m_hsize = 3'b000;
      // 16bits
      5'b1_0?11,
      5'b1_1100: m_hsize = 3'b001;
      // 32bits
      default:   m_hsize = 3'b010;
      endcase
end

always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
      m_hwdata <= 32'h0;
   else
   begin
      if (s_req & s_gnt & s_we)
      begin
         m_hwdata <= s_wdata;
      end
      else if (state==DATA & m_hready)
         m_hwdata <= 32'h0;
   end
end

assign m_hready_in = m_hready;
assign m_hsel      = s_req & s_gnt | state==ADDR_DATA;
assign m_htrans[1] = s_req & s_gnt | state==ADDR_DATA;
assign m_htrans[0] = 1'b0;
assign m_hwrite    = s_we          | state==ADDR_DATA;


endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
