////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of core_2_ahb_slow module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
// $HeadURL: $
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none
module core_2_ahb_slow #( 
   parameter ADDR_WIDTH = 32                     // Address bus width
) (
   //$port_g Clock and reset
   input  wire                    clk,           // root Clock
   input  wire                    rst_n,         // Asynchronous reset (active low)

   //$port_g CORE Slave interface
   input  wire                    s_req,
   output wire                    s_gnt,
   output reg                     s_rvalid,
   input  wire                    s_we,
   input  wire              [3:0] s_be,
   input  wire   [ADDR_WIDTH-1:0] s_addr,
   input  wire             [31:0] s_wdata,
   output reg              [31:0] s_rdata,
   output wire                    s_err,

   //$port_g AHB Master interface
   output wire                    m_hready_in,
   output wire                    m_hsel,
   output reg    [ADDR_WIDTH-1:0] m_haddr,
   output wire              [1:0] m_htrans,
   output reg                     m_hwrite,
   output reg               [2:0] m_hsize,
   output reg              [31:0] m_hwdata,
   input  wire             [31:0] m_hrdata,
   input  wire              [1:0] m_hresp,
   input  wire                    m_hready
);


////////////////////////////////////////////////////////////////////////////////
// Local Parameters
////////////////////////////////////////////////////////////////////////////////
localparam 
   IDLE      = 2'b00,
   ADDR      = 2'b01,
   ADDR_DATA = 2'b10,
   DATA      = 2'b11;


////////////////////////////////////////////////////////////////////////////////
// Internal Signals
////////////////////////////////////////////////////////////////////////////////
reg [1:0]  state;
reg        unalign;
reg        n_size; //0:8bits / 1:16bits


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

// State Machine is used to convert unalign CORE write access into two align AHB
// write access.
//
// CORE ACCESS              AHB ACCESS
// 24bits write @0x0   ->   16bits write @0x0 +  8bits write @0x2
// 24bits write @0x1   ->    8bits write @0x1 + 16bits write @0x2
// 16bits write @0x1   ->    8bits write @0x1 +  8bits write @0x2
//
always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
      state <= IDLE;
   else
   begin
      case (state)
      IDLE:
         if (s_req & s_gnt)
            state <= ADDR;

      ADDR:
         if (m_hready & unalign)
            state <= ADDR_DATA;
         else if (m_hready)
            state <= DATA;

      ADDR_DATA:
         if (m_hready)
            state <= DATA;

      DATA:
         if (s_req & s_gnt)
            state <= ADDR;
         else if (m_hready)
            state <= IDLE;
      endcase
   end
end


//------------------------------------------------------------------------------
// Second Access control in case of unalign CORE write access
//    Set the unalign flag and the second access parameters (size & addr)
//------------------------------------------------------------------------------
always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
   begin
      unalign <= 1'b0;
      n_size  <= 1'b0;
   end
   else if (s_req & s_gnt)
   begin
      case  ({s_we,s_be})
      5'b1_0111:
      begin
         unalign <= 1'b1;
         n_size  <= 1'b0;
      end
      5'b1_1110: 
      begin
         unalign <= 1'b1;
         n_size  <= 1'b1;
      end
      5'b1_0110:
      begin
         unalign <= 1'b1;
         n_size  <= 1'b0;
      end
      default  :
      begin
         unalign <= 1'b0;
         n_size  <= 1'b0;
      end
      endcase
   end
end


always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
   begin
      m_haddr  <= 32'h0;
      m_hsize  <= 3'b000;
      m_hwrite <= 1'b0;
   end
   else if (s_req & s_gnt)
   begin
      casez ({s_we,s_be})
      5'b1_??10: m_haddr <= {s_addr[ADDR_WIDTH-1:2],2'b01};
      5'b1_?100: m_haddr <= {s_addr[ADDR_WIDTH-1:2],2'b10};
      5'b1_1000: m_haddr <= {s_addr[ADDR_WIDTH-1:2],2'b11};
      default  : m_haddr <= {s_addr[ADDR_WIDTH-1:2],2'b00};
      endcase

      casez ({s_we,s_be})
      // 8Bits
      5'b1_0001,
      5'b1_??10,
      5'b1_0100,
      5'b1_1000: m_hsize <= 3'b000;
      // 16bits
      5'b1_0?11,
      5'b1_1100: m_hsize <= 3'b001;
      // 32bits
      default:   m_hsize <= 3'b010;
      endcase

      m_hwrite <= s_we;
   end
   else if (state==ADDR & m_hready & unalign)
   begin
      m_haddr <= {m_haddr[ADDR_WIDTH-1:2],2'b10};
      m_hsize <= {2'b00,n_size};
   end
   else if (state==DATA & m_hready)
   begin
      m_haddr  <= 32'h0;
      m_hsize  <= 3'b000;
      m_hwrite <= 1'b0;
   end
end


//------------------------------------------------------------------------------
// CORE Control
//------------------------------------------------------------------------------
always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
   begin
      s_rdata  <= 32'h0;
      s_rvalid <= 1'b0;
   end
   else
   begin
      s_rdata  <= m_hrdata;
      s_rvalid <= m_hready & state==DATA;
   end
end

assign s_gnt       = s_req & (m_hready & state==IDLE | state==DATA & s_rvalid);
assign s_err       = 1'b0;


//------------------------------------------------------------------------------
// AHB Control
//------------------------------------------------------------------------------
always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
      m_hwdata <= 32'h0;
   else
   begin
      if (s_req & s_gnt & s_we)
      begin
         m_hwdata <= s_wdata;
      end
      else if (state==DATA & m_hready)
         m_hwdata <= 32'h0;
   end
end

assign m_hready_in = m_hready;
assign m_hsel      = state==ADDR | state==ADDR_DATA;
assign m_htrans[1] = state==ADDR | state==ADDR_DATA;
assign m_htrans[0] = 1'b0;


endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
