//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of riscv_subsys module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module riscv_subsys
( 
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire                por_rst_n,
  input  wire                rst_n,
  input  wire                clk,

  /*****************************************************************************
  * processor control
  *****************************************************************************/
  input  wire                boot,
  input  wire                external_wait,

  /*****************************************************************************
  * interrupts
  *****************************************************************************/
  input  wire                int0,
  input  wire                int1,
  input  wire                int2,

  /*****************************************************************************
  * JTAG interface
  *****************************************************************************/
  input   wire               tck,
  input   wire               tms,
  input   wire               tdi,
  output  wire               tdo,
  output  wire               tdo_oen,

  /*****************************************************************************
  * AHB Master interface
  *****************************************************************************/
  input  wire                plf_hready,
  output wire         [27:0] plf_haddr,
  output wire         [ 1:0] plf_htrans,
  output wire                plf_hwrite, 
  output wire         [ 1:0] plf_hsize,
  input  wire         [31:0] plf_hrdata,
  output wire         [31:0] plf_hwdata,
  input  wire         [ 1:0] plf_hresp,

  /*****************************************************************************
  * AHB Slave interface
  *****************************************************************************/
  //input
  input  wire         [31:0] target_haddr,
  input  wire          [2:0] target_hburst,
  input  wire                target_hmastlock,
  input  wire          [3:0] target_hprot,
  input  wire          [2:0] target_hsize,
  input  wire          [1:0] target_htrans,
  input  wire         [31:0] target_hwdata,
  input  wire                target_hwrite, 
  //output
  output wire         [31:0] target_hrdata,
  output wire                target_hready,
  output wire          [1:0] target_hresp,
  
  /*****************************************************************************
  * Diag
  *****************************************************************************/
  output wire         [33:0] diag0,
  output wire         [33:0] diag1
);

/*******************************************************************************
* parameters
*******************************************************************************/
`ifdef  RISCV_RAM_1M

   // 1M Bytes Memory parameters
   localparam SRAM_AWIDTH = 18; // Address Width Of The SRAM interface
   localparam CORE_AWIDTH = 20; // Address Width Of The CORE interface

`else //RISCV_RAM_1M

   // 512K Bytes Memory parameters
   localparam SRAM_AWIDTH = 17; // Address Width Of The SRAM interface
   localparam CORE_AWIDTH = 19; // Address Width Of The CORE interface

`endif//RISCV_RAM_1M

/*******************************************************************************
* declaration
*******************************************************************************/

// RISC-V Control
wire                         fetch_enable_i;     // Start the decoding

// RISC-V Interrupt
wire                  [31:0] irq_i;              // Level sensitive IR lines

// RISC-V Instruction memory interface             
wire                         instr_req_o;        // Request
wire                         instr_gnt_i;        // Grant
wire                         instr_rvalid_i;     // Read Valid
wire                  [31:0] instr_addr_o;       // Address
wire                  [31:0] instr_rdata_i;      // Read Data

// RISC-V Data memory interface                    
wire                         data_req_o;         // Request
wire                         data_gnt_i;         // Grant
wire                         data_rvalid_i;      // Read Valid
wire                         data_we_o;          // Write Enable
wire                   [3:0] data_be_o;          // Byte Enable
wire                  [31:0] data_addr_o;        // Address
wire                  [31:0] data_wdata_o;       // Write Data
wire                  [31:0] data_rdata_i;       // Read Data
wire                         data_err_i;         // Error

// RISC-V Debug Interface
wire                         debug_req_i;        // Request
wire                         debug_gnt_o;        // Grant
wire                         debug_rvalid_o;     // Read Valid
wire                  [14:0] debug_addr_i;       // Address
wire                         debug_we_i;         // Write Enable
wire                  [31:0] debug_wdata_i;      // Write Data
wire                  [31:0] debug_rdata_o;      // Read Data
wire                         debug_halted_o;     // RISC-V is stopped
wire                         debug_halt_i;       // Halt Request
wire                         debug_resume_i;     // Resume Request

// Target to Platform AHB interface
wire                         tgt_plf_hready_in;
wire                         tgt_plf_hsel;
reg                          tgt_plf_hsel_1t; 
wire                  [27:0] tgt_plf_haddr;
wire                  [ 1:0] tgt_plf_htrans;
wire                         tgt_plf_hwrite;
wire                  [ 2:0] tgt_plf_hsize;      
wire                  [31:0] tgt_plf_hwdata;
wire                  [31:0] tgt_plf_hrdata;
wire                  [ 1:0] tgt_plf_hresp;
wire                         tgt_plf_hready;

// Target to RISC-V memory AHB interface
wire                         tgt_mem_hready_in;
wire                         tgt_mem_hsel;
reg                          tgt_mem_hsel_1t;
wire                  [19:0] tgt_mem_haddr;
wire                  [ 1:0] tgt_mem_htrans;
wire                         tgt_mem_hwrite;
wire                  [ 2:0] tgt_mem_hsize;      
wire                  [31:0] tgt_mem_hwdata;
wire                  [31:0] tgt_mem_hrdata;
wire                  [ 1:0] tgt_mem_hresp;
wire                         tgt_mem_hready;

// Target to debug AHB interface
wire                         tgt_dbg_hready_in;
wire                         tgt_dbg_hsel;
reg                          tgt_dbg_hsel_1t;
wire                  [19:0] tgt_dbg_haddr;
wire                  [ 1:0] tgt_dbg_htrans;
wire                         tgt_dbg_hwrite;
wire                  [ 2:0] tgt_dbg_hsize;      
wire                  [31:0] tgt_dbg_hwdata;
wire                  [31:0] tgt_dbg_hrdata;
wire                  [ 1:0] tgt_dbg_hresp;
wire                         tgt_dbg_hready;

// Target to None AHB interface
wire                         tgt_none_hsel;
reg                          tgt_none_hsel_1t;
wire                  [31:0] tgt_none_hrdata;
wire                  [ 1:0] tgt_none_hresp;
wire                         tgt_none_hready;

// Target to RISC-V memory CORE interface
wire                         tgt_mem_req;
wire                         tgt_mem_gnt;
wire                         tgt_mem_rvalid;
wire                         tgt_mem_we;
wire                   [3:0] tgt_mem_be;
wire                  [19:0] tgt_mem_addr;
wire                  [31:0] tgt_mem_wdata;
wire                  [31:0] tgt_mem_rdata;
wire                         tgt_mem_err;

// Target to debug CORE interface
wire                         tgt_dbg_req;
wire                         tgt_dbg_gnt;
wire                         tgt_dbg_rvalid;
wire                         tgt_dbg_we;
wire                   [3:0] tgt_dbg_be;
wire                  [19:0] tgt_dbg_addr;
wire                  [31:0] tgt_dbg_wdata;
wire                  [31:0] tgt_dbg_rdata;
wire                         tgt_dbg_err;

// RISC-V to Platform CORE interface
wire                         riscv_plf_req;
wire                         riscv_plf_gnt;
wire                         riscv_plf_rvalid;
wire                         riscv_plf_we;
wire                   [3:0] riscv_plf_be;
wire                  [27:0] riscv_plf_addr;
wire                  [31:0] riscv_plf_wdata;
wire                  [31:0] riscv_plf_rdata;
wire                         riscv_plf_err;

// RISC-V to data memory CORE interface
wire                         riscv_dmem_req;
wire                         riscv_dmem_gnt;
wire                         riscv_dmem_rvalid;
wire                         riscv_dmem_we;
wire                   [3:0] riscv_dmem_be;
wire                  [27:0] riscv_dmem_addr;
wire                  [31:0] riscv_dmem_wdata;
wire                  [31:0] riscv_dmem_rdata;
wire                         riscv_dmem_err;

// Target to data memory AHB interface
wire                         riscv_plf_hready_in;
wire                         riscv_plf_hsel;
wire                  [27:0] riscv_plf_haddr;
wire                  [ 1:0] riscv_plf_htrans;
wire                         riscv_plf_hwrite;
wire                  [ 2:0] riscv_plf_hsize;      
wire                  [31:0] riscv_plf_hwdata;
wire                  [31:0] riscv_plf_hrdata;
wire                  [ 1:0] riscv_plf_hresp;
wire                         riscv_plf_hready;

// Instruction sram interface
wire                         i_ram_cs;
wire       [SRAM_AWIDTH-1:0] i_ram_a;
wire                  [31:0] i_ram_q;

// Data sram interface
wire                         d_ram_cs;
wire                  [ 3:0] d_ram_we;
wire       [SRAM_AWIDTH-1:0] d_ram_a;
wire                  [31:0] d_ram_d;
wire                  [31:0] d_ram_q;


//******************************************************************************
// RISC-V Top
//******************************************************************************
riscv_top u_riscv_top( 
   //Clock and reset
   .clk            (clk            ),
   .rst_n          (rst_n          ),
   //Instruction memory interface             
   .instr_req_o    (instr_req_o    ),
   .instr_gnt_i    (instr_gnt_i    ),
   .instr_rvalid_i (instr_rvalid_i ),
   .instr_addr_o   (instr_addr_o   ),
   .instr_rdata_i  (instr_rdata_i  ),
   //Data memory interface                    
   .data_req_o     (data_req_o     ),
   .data_gnt_i     (data_gnt_i     ),
   .data_rvalid_i  (data_rvalid_i  ),
   .data_we_o      (data_we_o      ),
   .data_be_o      (data_be_o      ),
   .data_addr_o    (data_addr_o    ),
   .data_wdata_o   (data_wdata_o   ),
   .data_rdata_i   (data_rdata_i   ),
   .data_err_i     (data_err_i     ),
   //Interrupt Interface                      
   .irq_i          (irq_i          ),
   //Debug Interface
   .debug_req_i    (debug_req_i    ),
   .debug_gnt_o    (debug_gnt_o    ),
   .debug_rvalid_o (debug_rvalid_o ),
   .debug_addr_i   (debug_addr_i   ),
   .debug_we_i     (debug_we_i     ),
   .debug_wdata_i  (debug_wdata_i  ),
   .debug_rdata_o  (debug_rdata_o  ),
   .debug_halted_o (debug_halted_o ),
   .debug_halt_i   (debug_halt_i   ),
   .debug_resume_i (debug_resume_i ),
   //CPU Control Signals
   .fetch_enable_i (fetch_enable_i ),
   .core_busy_o    (/*Open*/       )
);

assign fetch_enable_i = ~external_wait;
assign irq_i[0]       = int0;
assign irq_i[1]       = int1;
assign irq_i[2]       = int2;
assign irq_i[31:3]    = 29'h0;

assign debug_req_i    = tgt_dbg_req;
assign debug_addr_i   = tgt_dbg_addr[14:0];
assign debug_we_i     = tgt_dbg_we;
assign debug_wdata_i  = tgt_dbg_wdata;
assign debug_halt_i   = 1'b0;
assign debug_resume_i = 1'b0;
assign tgt_dbg_gnt    = debug_gnt_o;
assign tgt_dbg_rvalid = debug_rvalid_o;
assign tgt_dbg_rdata  = debug_rdata_o;
assign tgt_dbg_err    = 1'b0;


//******************************************************************************
//
//                  |-> Platform
// Target AHB Mux ->|-> RISC-V SRAM (RISC-V External Program/Data Memory)
//                  |-> RISC-V Debug Port (Not yet implemented)
//
// Platform  : 0x6000_0000 -> 0x6FFF_FFFF (256MB)
// RISCV SRAM: 0x0000_0000 -> 0x000F_FFFF   (1MB)
// DEBUG     : 0x0040_0000 -> 0x004F_FFFF   (1MB)
//
//******************************************************************************

// AHB Interface From Target To Platform
assign tgt_plf_hready_in  = target_hready;
assign tgt_plf_hsel       = target_haddr[31:28]==4'b0110;
assign tgt_plf_htrans     = target_htrans;
assign tgt_plf_haddr      = target_haddr[27:0];
assign tgt_plf_hwrite     = target_hwrite;
assign tgt_plf_hsize      = target_hsize;
assign tgt_plf_hwdata     = target_hwdata;

// AHB Interface From Target To RISC-V Memory
assign tgt_mem_hready_in  = target_hready;
assign tgt_mem_hsel       = target_haddr[31:20]==12'b0000_0000_0000;
assign tgt_mem_htrans     = target_htrans;
assign tgt_mem_haddr      = target_haddr[19:0];
assign tgt_mem_hwrite     = target_hwrite;
assign tgt_mem_hsize      = target_hsize;
assign tgt_mem_hwdata     = target_hwdata;

// AHB Interface From Target To Debug Port
assign tgt_dbg_hready_in  = target_hready;
assign tgt_dbg_hsel       = target_haddr[31:20]==12'b0000_0000_0100;
assign tgt_dbg_htrans     = target_htrans;
assign tgt_dbg_haddr      = target_haddr[19:0];
assign tgt_dbg_hwrite     = target_hwrite;
assign tgt_dbg_hsize      = target_hsize;
assign tgt_dbg_hwdata     = target_hwdata;

// AHB interface From Target To None (empty address space)
//    prevent AHB stuck when reading empty address space
assign tgt_none_hsel         = ~tgt_plf_hsel &
                               ~tgt_mem_hsel &
                               ~tgt_dbg_hsel;
assign tgt_none_hrdata       = 32'hdeadbafe;
assign tgt_none_hresp        = 2'h0;
assign tgt_none_hready       = 1'b1;

// AHB Mux
always @(posedge clk, negedge rst_n)
begin
  if(rst_n==1'b0)
  begin
    tgt_plf_hsel_1t   <= 1'b0; 
    tgt_mem_hsel_1t   <= 1'b0;
    tgt_dbg_hsel_1t   <= 1'b0;
    tgt_none_hsel_1t  <= 1'b0;
  end
  else
  begin
    if(target_hready==1'b1 && target_htrans[1]==1'b1)
    begin
      tgt_plf_hsel_1t   <= tgt_plf_hsel; 
      tgt_mem_hsel_1t   <= tgt_mem_hsel; 
      tgt_dbg_hsel_1t   <= tgt_dbg_hsel; 
      tgt_none_hsel_1t  <= tgt_none_hsel;
    end
  end
end

assign target_hrdata = {32{tgt_plf_hsel_1t}}  & tgt_plf_hrdata  |
                       {32{tgt_mem_hsel_1t}}  & tgt_mem_hrdata  |
                       {32{tgt_dbg_hsel_1t}}  & tgt_dbg_hrdata  |
                       {32{tgt_none_hsel_1t}} & tgt_none_hrdata;

assign target_hresp  = { 2{tgt_plf_hsel_1t}}  & tgt_plf_hresp   |
                       { 2{tgt_mem_hsel_1t}}  & tgt_mem_hresp   |
                       { 2{tgt_dbg_hsel_1t}}  & tgt_dbg_hresp   |
                       { 2{tgt_none_hsel_1t}} & tgt_none_hresp;

assign target_hready = tgt_plf_hready  &
                       tgt_mem_hready  &
                       tgt_dbg_hready  &
                       tgt_none_hready;


//******************************************************************************
//
//                   |-> Platform
// RISC-V DATA Mux ->|
//                   |-> DATA  SRAM (RISC-V External Data Memory)
//
// Platform  : 0x6000_0000 -> 0x6FFF_FFFF (256MB)
// DATA  SRAM: 0x0000_0000 -> 0x000F_FFFF   (1MB)
//
//******************************************************************************

// Core Interface To Platform
assign riscv_plf_req    = data_req_o & data_addr_o[31:28]==4'b0110;
assign riscv_plf_we     = data_we_o;
assign riscv_plf_be     = data_be_o;
assign riscv_plf_addr   = data_addr_o[27:0];
assign riscv_plf_wdata  = data_wdata_o;

// Core Interface To Data Memory
assign riscv_dmem_req   = data_req_o & data_addr_o[31:28]!=4'b0110;
assign riscv_dmem_we    = data_we_o;
assign riscv_dmem_be    = data_be_o;
assign riscv_dmem_addr  = data_addr_o[27:0];
assign riscv_dmem_wdata = data_wdata_o;

// Core Mux
assign data_gnt_i    = riscv_plf_gnt |
                       riscv_dmem_gnt;

assign data_rvalid_i = riscv_plf_rvalid |
                       riscv_dmem_rvalid;

assign data_rdata_i  = {32{riscv_plf_rvalid}}  & riscv_plf_rdata  |
                       {32{riscv_dmem_rvalid}} & riscv_dmem_rdata ;

assign data_err_i    = 1'b0;


//******************************************************************************
// Target->MEM AHB_2_Core Bridge
//******************************************************************************
ahb_2_core #(
   .ADDR_WIDTH  (20                )
)u_ahb_2_core_mem(
   //Clock and reset
   .clk         (clk               ),
   .rst_n       (rst_n             ),
   //AHB Slave interface
   .s_hready_in (tgt_mem_hready_in ),
   .s_hsel      (tgt_mem_hsel      ),
   .s_haddr     (tgt_mem_haddr     ),
   .s_htrans    (tgt_mem_htrans    ),
   .s_hwrite    (tgt_mem_hwrite    ),
   .s_hsize     (tgt_mem_hsize     ),
   .s_hwdata    (tgt_mem_hwdata    ),
   .s_hrdata    (tgt_mem_hrdata    ),
   .s_hresp     (tgt_mem_hresp     ),
   .s_hready    (tgt_mem_hready    ),
   //CORE Master interface
   .m_req       (tgt_mem_req       ),
   .m_gnt       (tgt_mem_gnt       ),
   .m_rvalid    (tgt_mem_rvalid    ),
   .m_we        (tgt_mem_we        ),
   .m_be        (tgt_mem_be        ),
   .m_addr      (tgt_mem_addr      ),
   .m_wdata     (tgt_mem_wdata     ),
   .m_rdata     (tgt_mem_rdata     ),
   .m_err       (tgt_mem_err       )
);


//******************************************************************************
// Target->DEBUG AHB_2_Core Bridge
//******************************************************************************
ahb_2_core #(
   .ADDR_WIDTH  (20                 )
)u_ahb_2_core_dbg(
   //Clock and reset
   .clk         (clk                ),
   .rst_n       (rst_n              ),
   //AHB Slave interface
   .s_hready_in (tgt_dbg_hready_in  ),
   .s_hsel      (tgt_dbg_hsel       ),
   .s_haddr     (tgt_dbg_haddr      ),
   .s_htrans    (tgt_dbg_htrans     ),
   .s_hwrite    (tgt_dbg_hwrite     ),
   .s_hsize     (tgt_dbg_hsize      ),
   .s_hwdata    (tgt_dbg_hwdata     ),
   .s_hrdata    (tgt_dbg_hrdata     ),
   .s_hresp     (tgt_dbg_hresp      ),
   .s_hready    (tgt_dbg_hready     ),
   //CORE Master interface          
   .m_req       (tgt_dbg_req        ),
   .m_gnt       (tgt_dbg_gnt        ),
   .m_rvalid    (tgt_dbg_rvalid     ),
   .m_we        (tgt_dbg_we         ),
   .m_be        (tgt_dbg_be         ),
   .m_addr      (tgt_dbg_addr       ),
   .m_wdata     (tgt_dbg_wdata      ),
   .m_rdata     (tgt_dbg_rdata      ),
   .m_err       (tgt_dbg_err        )
);


//******************************************************************************
// RISCV->Platform CORE_2_AHB Bridge
//******************************************************************************
`ifdef  ZERORISCY
core_2_ahb #(
`else //ZERORISCY
core_2_ahb #(
`endif//ZERORISCY
   .ADDR_WIDTH  (28                 )
)u_core_2_ahb_plf(
   //Clock and reset
   .clk         (clk                 ),
   .rst_n       (rst_n               ),
   //CORE Slave interface
   .s_req       (riscv_plf_req       ),
   .s_gnt       (riscv_plf_gnt       ),
   .s_rvalid    (riscv_plf_rvalid    ),
   .s_we        (riscv_plf_we        ),
   .s_be        (riscv_plf_be        ),
   .s_addr      (riscv_plf_addr      ),
   .s_wdata     (riscv_plf_wdata     ),
   .s_rdata     (riscv_plf_rdata     ),
   .s_err       (riscv_plf_err       ),
   //AHB Master interface
   .m_hready_in (riscv_plf_hready_in ),
   .m_hsel      (riscv_plf_hsel      ),
   .m_haddr     (riscv_plf_haddr     ),
   .m_htrans    (riscv_plf_htrans    ),
   .m_hwrite    (riscv_plf_hwrite    ),
   .m_hsize     (riscv_plf_hsize     ),
   .m_hwdata    (riscv_plf_hwdata    ),
   .m_hrdata    (riscv_plf_hrdata    ),
   .m_hresp     (riscv_plf_hresp     ),
   .m_hready    (riscv_plf_hready    )
);


//******************************************************************************
// Core Instr Bridge
//******************************************************************************
wire                   riscv_imem_req;        // Request
wire                   riscv_imem_gnt;        // Grant
wire                   riscv_imem_rvalid;     // Read Valid
wire [CORE_AWIDTH-1:0] riscv_imem_addr;       // Address
wire            [31:0] riscv_imem_rdata;      // Read Data

`ifdef  RISCV_CORE_BRIDGE

core_instr_bridge #( 
   .ADDR_WIDTH (CORE_AWIDTH                   )
) u_core_instr_bridge (
   //Clock and reset
   .clk        (clk                           ),
   .rst_n      (rst_n                         ),
   //CORE Slave interface
   .s_req      (instr_req_o                   ),
   .s_gnt      (instr_gnt_i                   ),
   .s_rvalid   (instr_rvalid_i                ),
   .s_addr     (instr_addr_o[CORE_AWIDTH-1:0] ),
   .s_rdata    (instr_rdata_i                 ),
   //CORE Master interface
   .m_req      (riscv_imem_req                ),
   .m_gnt      (riscv_imem_gnt                ),
   .m_rvalid   (riscv_imem_rvalid             ),
   .m_addr     (riscv_imem_addr               ),
   .m_rdata    (riscv_imem_rdata              )
);

`else //RISCV_CORE_BRIDGE

assign riscv_imem_req  = instr_req_o;
assign riscv_imem_addr = instr_addr_o[CORE_AWIDTH-1:0];

assign instr_gnt_i     = riscv_imem_gnt;
assign instr_rvalid_i  = riscv_imem_rvalid;
assign instr_rdata_i   = riscv_imem_rdata;

`endif//RISCV_CORE_BRIDGE


//******************************************************************************
// IMEM CORE 2 SRAM
//******************************************************************************
core_2_sram #(
   .ADDR_WIDTH (CORE_AWIDTH       )
)u_core_2_sram_imem(
   //Clock and reset
   .clk        (clk               ),
   .rst_n      (rst_n             ),
   //CORE Slave0 interface
   .s_req      (riscv_imem_req    ), //(instr_req_o         ),
   .s_gnt      (riscv_imem_gnt    ), //(instr_gnt_i         ),
   .s_rvalid   (riscv_imem_rvalid ), //(instr_rvalid_i      ),
   .s_addr     (riscv_imem_addr   ), //(instr_addr_o[17:0]  ),
   .s_rdata    (riscv_imem_rdata  ), //(instr_rdata_i       ),
   //SRAM interface
   .ram_cs     (i_ram_cs          ),
   .ram_a      (i_ram_a           ),
   .ram_q      (i_ram_q           )
);


//******************************************************************************
// DMEM CORE MPA
//******************************************************************************
core_mpa #(
   .ADDR_WIDTH (CORE_AWIDTH                      )
)u_core_mpa_dmem(
   //Clock and reset
   .clk        (clk                              ),
   .rst_n      (rst_n                            ),
   //CORE Slave0 interface
   .s0_req     (riscv_dmem_req                   ),
   .s0_gnt     (riscv_dmem_gnt                   ),
   .s0_rvalid  (riscv_dmem_rvalid                ),
   .s0_we      (riscv_dmem_we                    ),
   .s0_be      (riscv_dmem_be                    ),
   .s0_addr    (riscv_dmem_addr[CORE_AWIDTH-1:0] ),
   .s0_wdata   (riscv_dmem_wdata                 ),
   .s0_rdata   (riscv_dmem_rdata                 ),
   .s0_err     (riscv_dmem_err                   ),
   //CORE Slave1 interface
   .s1_req     (tgt_mem_req                      ),
   .s1_gnt     (tgt_mem_gnt                      ),
   .s1_rvalid  (tgt_mem_rvalid                   ),
   .s1_we      (tgt_mem_we                       ),
   .s1_be      (tgt_mem_be                       ),
   .s1_addr    (tgt_mem_addr[CORE_AWIDTH-1:0]    ),
   .s1_wdata   (tgt_mem_wdata                    ),
   .s1_rdata   (tgt_mem_rdata                    ),
   .s1_err     (tgt_mem_err                      ),
   //SRAM interface
   .ram_cs     (d_ram_cs                         ),
   .ram_we     (d_ram_we                         ),
   .ram_a      (d_ram_a                          ),
   .ram_d      (d_ram_d                          ),
   .ram_q      (d_ram_q                          )
);


//******************************************************************************
// Platform AHB MPA
//******************************************************************************
ahb_mpa #(
   .g_addr_width (28                  )
) u_ahb_mpa_plf (
   .clk          (clk                 ),
   .rst_n        (rst_n               ),
   .s0_hready_in (riscv_plf_hready_in ),
   .s0_hsel      (riscv_plf_hsel      ),
   .s0_haddr     (riscv_plf_haddr     ),
   .s0_htrans    (riscv_plf_htrans    ),
   .s0_hwrite    (riscv_plf_hwrite    ),
   .s0_hsize     (riscv_plf_hsize[1:0]),
   .s0_hwdata    (riscv_plf_hwdata    ),
   .s0_hrdata    (riscv_plf_hrdata    ),
   .s0_hresp     (riscv_plf_hresp     ),
   .s0_hready    (riscv_plf_hready    ),
   .s1_hready_in (tgt_plf_hready_in   ),
   .s1_hsel      (tgt_plf_hsel        ),
   .s1_haddr     (tgt_plf_haddr       ),
   .s1_htrans    (tgt_plf_htrans      ),
   .s1_hwrite    (tgt_plf_hwrite      ),
   .s1_hsize     (tgt_plf_hsize[1:0]  ),
   .s1_hwdata    (tgt_plf_hwdata      ),
   .s1_hrdata    (tgt_plf_hrdata      ),
   .s1_hresp     (tgt_plf_hresp       ),
   .s1_hready    (tgt_plf_hready      ),
   .m_hready_in  (                    ),
   .m_hsel       (                    ),
   .m_haddr      (plf_haddr           ),
   .m_htrans     (plf_htrans          ),
   .m_hwrite     (plf_hwrite          ),
   .m_hsize      (plf_hsize           ),
   .m_hwdata     (plf_hwdata          ),
   .m_hrdata     (plf_hrdata          ),
   .m_hresp      (plf_hresp           ),
   .m_hready     (plf_hready          )
);


//******************************************************************************
// Unused output
//******************************************************************************
assign tdo     = 1'b0;
assign tdo_oen = 1'b0;

`ifdef RW_MEM_FPGA
//******************************************************************************
// RISCV Program and Data memories
//******************************************************************************
`ifdef  RISCV_RAM_1M
sram_tdp_256kx8 sram_tdp_256kx8_i[3:0]
`else //RISCV_RAM_1M
sram_tdp_128kx8 sram_tdp_128kx8_i[3:0]
`endif//RISCV_RAM_1M
(
  .clka             (clk                 ),
//.ena              (i_ram_cs            ),
  .addra            (i_ram_a             ),
  .wea              (1'b0                ),
  .dina             (8'b0                ),
  .douta            (i_ram_q             ),
  .clkb             (clk                 ),
//.enb              (d_ram_cs            ),
  .addrb            (d_ram_a             ),
  .web              (d_ram_we            ),
  .dinb             (d_ram_d             ),
  .doutb            (d_ram_q             )
); 

`else// RW_MEM_FPGA
  sram_dual_port_model
  #(
    .g_addr_width( SRAM_AWIDTH),
    .g_data_width(          32)
  )
`ifdef  RISCV_RAM_1M
  sram_tdp_256kx32_i
`else //RISCV_RAM_1M
  sram_tdp_128kx32_i
`endif//RISCV_RAM_1M
  (
     .clk1(             clk),
     .addr1(            i_ram_a),
     .en1(              i_ram_cs),
     .wen1(             32'b0),
     .wdata1(           32'b0),
     .rdata1(           i_ram_q),
    
     .clk2(             clk),
     .addr2(            d_ram_a),
     .en2(              d_ram_cs),
     .wen2(             {{8{d_ram_we[3]}},{8{d_ram_we[2]}},{8{d_ram_we[1]}},{8{d_ram_we[0]}}}),
     .wdata2(           d_ram_d),
     .rdata2(           d_ram_q)
  );

`endif // RW_MEM_FPGA
//******************************************************************************
// Diag port
//******************************************************************************
assign diag0 = 34'd0;
assign diag1 = 34'd0;


endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
