//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : 
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module RAM2Data # (
  parameter RAM_AWIDTH =  10,
  parameter RAM_DWIDTH =  128
)( 
  /* system */
  input  wire                   nRst,
  input  wire                   Clk,

  /* control */
  input  wire                   Enable,
  input  wire                   Start,
  input  wire [RAM_AWIDTH+1:0]  StartAddress,
  input  wire [RAM_AWIDTH-1:0]  WrapAddress, // Address at which read address wraps to 0

  /* Memory interface */
  output reg                    RAMRdEn,
  output reg  [RAM_AWIDTH-1:0]  RAMRdAddr,
  input  wire [RAM_DWIDTH-1:0]  RAMRdData,
  
  /* Data interface */
  output reg [RAM_DWIDTH-1:0]   DataOut,
  output wire                   DataOutValid,
  input  wire                   DataOutReady

  );

  /******************************************************************************
  * declarations
  ******************************************************************************/
  localparam   [RAM_AWIDTH-1:0] SIG_ONE_DSP_AWIDTH_PARAM = 'b1;

  // Internal data stages
  wire [RAM_DWIDTH-1:0] s1Data;
  wire nS1Valid, loadStall, s2StallReady, s3Ready;
  wire [RAM_DWIDTH-1:0] nS2Data, nS2StallData, nS3Data;
  wire nS2Valid, nS2StallValid, nS2Ready;
  wire nS3Valid;
  wire loadS3withStall, loadS3withS2;
  reg  [RAM_DWIDTH-1:0] s2Data, s2StallData, s3Data;
  wire [RAM_DWIDTH-1:0] ns2ors3Data;
  reg  s1Valid, s2Valid, s2StallValid, s3Valid;
  reg  s2Ready;
  // ensure that after enable=0, the block does not start before Start=1
  reg busy;


  /******************************************************************************
  * Output
  ******************************************************************************/
  assign DataOutValid = s3Valid && !Start;   

  always @ (*)
  begin: DataOut_Blk
    case (StartAddress[1:0])
      2'b00 : DataOut = s3Data;
      2'b01 : DataOut = {s2Data[RAM_DWIDTH/4-1:0],s3Data[RAM_DWIDTH-1:RAM_DWIDTH/4]};
      2'b10 : DataOut = {s2Data[RAM_DWIDTH/2-1:0],s3Data[RAM_DWIDTH-1:RAM_DWIDTH/2]};
    default : DataOut = {s2Data[RAM_DWIDTH*3/4-1:0],s3Data[RAM_DWIDTH-1:RAM_DWIDTH*3/4]};
     endcase
  end //DataOut_Blk

  /******************************************************************************
  * Data processing
  ******************************************************************************/
  /* The block is organized as a pipeline 
   * - Stage 1 mimics the RAM data port. Only Valid is implemented here. 
   * - Stage 2 is used to immediately register the RAM data. As it is not possible to flow-control 
   * the RAM interface, Stage 2 also contains a stall buffer to store the data in case Stage 3
   * is not ready to accept it.
   * - Stage 3 is the output stage. 
   * A ready/valid interface is used through the stages, except that the RAM interface can't be flow controlled,
   * so the stall buffer in stage 2 is used instead. */
  /******************************************************************************
  * Stage 1
  ******************************************************************************/
  // s1Ready not used as RAM interface can't be flow controlled
  assign s1Data    = RAMRdData;         // assign RAMRdData to s1Data for readability 
  // s1Data is saved at each cycle in s2Data or s2StallData. When an adress stays several clock 
  // cycles because of s3Ready, the data is valid only until it is loaded in the next stage.
  assign nS1Valid = RAMRdEn && (s3Ready || !s2Valid || !s2StallValid);

  /******************************************************************************
  * Stage 2
  ******************************************************************************/
  /* Stage 2 is ready to accept data when 
   * - stage 2 data can be transferred out, either because stage 3 is empty (s3Valid=0)
   * or because stage 3 data can be updated (s3Ready=1). Note that stage 3 is fed in priority with
   * data from the stall buffer, but in this case the stall buffer itself is freed to accept stage 2 data.
   * - stage 2 is empty (s2Valid=0) */
  assign nS2Ready = s3Ready || !s3Valid || !s2Valid;

  assign nS2Data  = (s1Valid && s2Ready) ? s1Data : s2Data;
  assign nS2Valid = s2Ready ? s1Valid : s2Valid;

  /* As it is not possible to forward the ready to the RAM interface, implement a stall buffer
   * to save the data that can't be used at once */
  // Load stall buffer when there is valid data inside the stage, and 
  // - stage 3 is not ready to accept it.
  // - stage 3 is ready to accept data, but older data is valid inside the stall buffer, so stage 2 
  // data won't be sent to stage 3 at this cycle and must be stalled
  assign loadStall      = s2Valid && ( (!s3Ready && s3Valid)|| (s2StallValid && s3Ready) );
  // The buffer content must be frozen when it contains valid data, and stage 3 is not ready to accept it
  assign s2StallReady   = ! (s2StallValid && !s3Ready);
  assign nS2StallData  = (loadStall && s2StallReady) ? s2Data : s2StallData;
  assign nS2StallValid = s2StallReady ? (loadStall && !(s2StallValid && s2Ready)) : s2StallValid;

  /******************************************************************************
  * Stage 3
  ******************************************************************************/
  assign s3Ready  = DataOutReady;   // assign DataOutReady to s3Ready for readability
  // Stage 3 is loaded with data from Stage 2, using in priority oldest data, i.e.
  // data from stall buffer.
  assign loadS3withStall = (s2StallValid && s3Ready) || (s2StallValid && !s3Ready && !s3Valid);
  assign loadS3withS2    = (s2Valid       && s3Ready) || (s2Valid       && !s3Ready && !s3Valid);
  assign ns2ors3Data = loadS3withS2    ? s2Data       : s3Data;
  assign nS3Data     = loadS3withStall ? s2StallData  : ns2ors3Data; 
  assign nS3Valid    = (loadS3withStall || loadS3withS2) ? (s2StallValid || s2Valid) : s3Valid;

  /******************************************************************************
  * Registers
  ******************************************************************************/
  always @(posedge Clk, negedge nRst)
    if(!nRst) begin
      s1Valid       <= 1'd0;
      s2Data        <= {RAM_DWIDTH{1'b0}};
      s2Valid       <= 1'd0;
      s2Ready       <= 1'd1;
      s2StallData   <= {RAM_DWIDTH{1'b0}};
      s2StallValid  <= 1'd0;
      s3Data        <= {RAM_DWIDTH{1'b0}};
      s3Valid       <= 1'd0;
    end
    else if(!Enable) begin
      s1Valid       <= 1'd0;
      s2Data        <= {RAM_DWIDTH{1'b0}};
      s2Valid       <= 1'd0;
      s2Ready       <= 1'd1;
      s2StallData   <= {RAM_DWIDTH{1'b0}};
      s2StallValid  <= 1'd0;
      s3Data        <= {RAM_DWIDTH{1'b0}};
      s3Valid       <= 1'd0;
    end else begin
      if(Start) begin
        s1Valid       <= 1'd0;
        s2Data        <= {RAM_DWIDTH{1'b0}};
        s2Valid       <= 1'd0;
        s2Ready       <= 1'd1;
        s2StallData   <= {RAM_DWIDTH{1'b0}};
        s2StallValid  <= 1'd0;
        s3Data        <= {RAM_DWIDTH{1'b0}};
        s3Valid       <= 1'd0;
      end else begin
        s1Valid       <= nS1Valid;
        s2Data        <= nS2Data;
        s2Valid       <= nS2Valid;
        s2Ready       <= nS2Ready;
        s2StallData   <= nS2StallData;
        s2StallValid  <= nS2StallValid;
        s3Data        <= nS3Data;
        s3Valid       <= nS3Valid;
      end
    end

  /******************************************************************************
  * control
  ******************************************************************************/

  /* Registers */
  always @(posedge Clk, negedge nRst)
    if(!nRst) begin
      RAMRdAddr          <= {RAM_AWIDTH{1'b0}};
      RAMRdEn            <= 1'b0;
      busy               <= 1'b0;
    end else if(!Enable) begin
      RAMRdAddr          <= {RAM_AWIDTH{1'b0}};
      RAMRdEn            <= 1'b0;
      busy               <= 1'b0;
    end else begin

      if (Start) begin
        RAMRdAddr          <= StartAddress[RAM_AWIDTH+1:2];
        RAMRdEn            <= 1'b1;
        busy               <= 1'b1;
      end else if (busy) begin
        if(s3Ready || !s3Valid) begin
          RAMRdEn    <= 1'b1;
          if (RAMRdAddr == WrapAddress) 
            RAMRdAddr  <= {RAM_AWIDTH{1'b0}};
          else 
            RAMRdAddr  <= RAMRdAddr + SIG_ONE_DSP_AWIDTH_PARAM;
          
        end
      end
      
    end

endmodule
                 
`default_nettype wire
