//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//   This module is responsible for generating the Key for each round
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module aesKeyGenTop (  
   //$port_g Clock and reset
   input  wire          pClk,           // clock signal 
   input  wire          nPRst,          // reset signal 
   input  wire          nSRst,          // soft reset
                                        
   //$port_g aesCore                   
   input  wire          enable,         // Enable Key Generation
   input  wire          abort_p,        // Abort Cipher Computation
   input  wire    [1:0] aesKeyLen,      // Input Key Length
   input  wire  [255:0] aesKey,         // Input Key
   input  wire          roundCountDone, // Number of round is reached
   input  wire          loadKey,        // Signal to mux the input key
   output wire  [127:0] roundKey        // Key for each Round 
);

/*******************************************************************************
* Parameters
*******************************************************************************/
localparam AES128 = 2'd0;
localparam AES192 = 2'd1;
localparam AES256 = 2'd2;


/*******************************************************************************
* Internal Signals Declaration
*******************************************************************************/
reg            rConEnable;
reg      [7:0] rCon;

reg    [127:0] muxedKey;
reg    [127:0] muxedTemp;

reg     [31:0] subWordIn;
wire    [31:0] subWordOut;
wire    [31:0] subRotWord;
        
wire    [31:0] nextRoundKey3;
wire    [31:0] nextRoundKey2;
wire    [31:0] nextRoundKey1;
wire    [31:0] nextRoundKey0;
        
reg     [31:0] roundKey3;
reg     [31:0] roundKey2;
reg     [31:0] roundKey1;
reg     [31:0] roundKey0;
        
reg     [31:0] prevRoundKey3;
reg     [31:0] prevRoundKey2;
reg     [31:0] prevRoundKey1;
reg     [31:0] prevRoundKey0;
        
reg      [3:0] phase;
reg      [3:0] phaseReg;


/*******************************************************************************
* Main Code Start
*******************************************************************************/

//-----------------------------------------------------------------------------
// Rcon Generation  used for Key Generation process for
// both  Encryption
//-----------------------------------------------------------------------------
always @*
begin
   case (phase)
   {AES192,2'd1}: rConEnable = 1'b0;
   {AES256,2'd0}: rConEnable = 1'b0;
   {AES256,2'd2}: rConEnable = 1'b0;
   default:       rConEnable = enable;
   endcase
end

always @(posedge pClk or negedge nPRst)
begin
   if (nPRst==1'b0)
      rCon <= 8'h01;
   else if (nSRst==1'b0 | roundCountDone==1'b1 | abort_p==1'b1)
      rCon <= 8'h01;
   else if (rConEnable==1'b1)
      rCon <= {rCon[6:0],1'b0} ^ {3'b000,rCon[7],rCon[7],1'b0,rCon[7],rCon[7]};
end


//------------------------------------------------------------------------------
// Round Key generation
//------------------------------------------------------------------------------
assign nextRoundKey3 = muxedKey[127:96] ^ muxedTemp[127:96];
assign nextRoundKey2 = muxedKey[95:64]  ^ muxedTemp[95:64];
assign nextRoundKey1 = muxedKey[63:32]  ^ muxedTemp[63:32];
assign nextRoundKey0 = muxedKey[31:0]   ^ muxedTemp[31:0];

always @(posedge pClk or negedge nPRst)
begin
   if (nPRst==1'b0)
   begin
      prevRoundKey0 <= 32'h0;
      prevRoundKey1 <= 32'h0;
      prevRoundKey2 <= 32'h0;
      prevRoundKey3 <= 32'h0;

      roundKey0     <= 32'h0;
      roundKey1     <= 32'h0;
      roundKey2     <= 32'h0;
      roundKey3     <= 32'h0;
   end
   else if (nSRst==1'b0)
   begin
      prevRoundKey0 <= 32'h0;
      prevRoundKey1 <= 32'h0;
      prevRoundKey2 <= 32'h0;
      prevRoundKey3 <= 32'h0;

      roundKey0     <= 32'h0;
      roundKey1     <= 32'h0;
      roundKey2     <= 32'h0;
      roundKey3     <= 32'h0;
   end
   else if (enable==1'b1)
   begin
      prevRoundKey0 <= (loadKey) ? aesKey[31:0]   : roundKey0;
      prevRoundKey1 <= (loadKey) ? aesKey[63:32]  : roundKey1;
      prevRoundKey2 <= (loadKey) ? aesKey[95:64]  : roundKey2;
      prevRoundKey3 <= (loadKey) ? aesKey[127:96] : roundKey3;

      roundKey0     <= nextRoundKey0;
      roundKey1     <= nextRoundKey1;
      roundKey2     <= nextRoundKey2;
      roundKey3     <= nextRoundKey3;
   end
end


//------------------------------------------------------------------------------
// Round Key Input Selection
//   muxedKey  = w[i-Nk]
//   muxedTemp = w[i-1] with SubWord, RotWord & Rcon
//------------------------------------------------------------------------------
always @*
begin
   case (phase)
   {AES256,2'd0}: muxedKey = {aesKey[255:224], aesKey[223:192], aesKey[191:160], aesKey[159:128]};
   {AES256,2'd1},
   {AES256,2'd2},
   {AES256,2'd3}: muxedKey = {prevRoundKey3  , prevRoundKey2  , prevRoundKey1  , prevRoundKey0};
   {AES192,2'd0}: muxedKey = {aesKey[63:32]  , aesKey[31:0]   , aesKey[191:160], aesKey[159:128]};
   {AES192,2'd1},
   {AES192,2'd2},
   {AES192,2'd3}: muxedKey = {roundKey1      , roundKey0      , prevRoundKey3  , prevRoundKey2};
   {AES128,2'd0}: muxedKey = {aesKey[127:96] , aesKey[95:64]  , aesKey[63:32]  , aesKey[31:0]};
   default:       muxedKey = {roundKey3      , roundKey2      , roundKey1      , roundKey0};
   endcase
end

always @*
begin
   case (phase)
   {AES256,2'd0}: muxedTemp = {32'h0        , 32'h0         , 32'h0        , 32'h0};
   {AES256,2'd2}: muxedTemp = {nextRoundKey2, nextRoundKey1 , nextRoundKey0, subWordOut};
   {AES192,2'd0}: muxedTemp = {nextRoundKey2, subRotWord    , 32'h0        , 32'h0};
   {AES192,2'd1}: muxedTemp = {nextRoundKey2, nextRoundKey1 , nextRoundKey0, roundKey3};
   {AES192,2'd3}: muxedTemp = {nextRoundKey2, subRotWord    , nextRoundKey0, roundKey3};
   default:       muxedTemp = {nextRoundKey2, nextRoundKey1 , nextRoundKey0, subRotWord};
   endcase
end


//------------------------------------------------------------------------------
// Substituaion box (subWord)
//------------------------------------------------------------------------------
always @*
begin
   case (phase)
   {AES192,2'd0}: subWordIn = aesKey[191:160];                           //nextRoundKey1;
   {AES192,2'd3}: subWordIn = roundKey3 ^ prevRoundKey2 ^ prevRoundKey3; //nextRoundKey1;
   {AES128,2'd0}: subWordIn = aesKey[127:96];
   default:       subWordIn = roundKey3;
   endcase
end

aesSBox u_subWord[3:0] (
   .inData     (subWordIn  ),
   .outData    (subWordOut )
);


//------------------------------------------------------------------------------
// rotWord xor Rcon
//------------------------------------------------------------------------------
assign subRotWord     = {subWordOut[7:0]    ,subWordOut[31:8]}    ^{24'h000000,rCon};


//------------------------------------------------------------------------------
// Phase
//   Phase[3:2] = Key Length
//   Phase[1:0] = Phase Number
//------------------------------------------------------------------------------
always @*
begin
   if (loadKey)
   begin
      case (aesKeyLen)
      AES256:  phase = {AES256,2'd0};
      AES192:  phase = {AES192,2'd0};
      default: phase = {AES128,2'd0};
      endcase
   end
   else
   begin
      case (phaseReg)
      //AES256
      {AES256,2'd0}: phase = {AES256,2'd1};
      {AES256,2'd1}: phase = {AES256,2'd2};
      {AES256,2'd2}: phase = {AES256,2'd1};
      //AES192
      {AES192,2'd0}: phase = {AES192,2'd1};
      {AES192,2'd1}: phase = {AES192,2'd2};
      {AES192,2'd2}: phase = {AES192,2'd3};
      {AES192,2'd3}: phase = {AES192,2'd1};
      //AES128
      {AES128,2'd0}: phase = {AES128,2'd1};
      default:       phase = {AES128,2'd1};
      endcase
   end
end

always @(posedge pClk or negedge nPRst)
begin
   if (nPRst==1'b0)
      phaseReg <= 4'b000;
   else if (nSRst==1'b0 | abort_p==1'b1)
      phaseReg <= 4'b000;
   else if (enable==1'b1)
      phaseReg <= phase;
end


//------------------------------------------------------------------------------
// Output Signal Gneration 
//------------------------------------------------------------------------------
assign roundKey = {roundKey3,roundKey2,roundKey1,roundKey0};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
reg [12*8-1:0] phase_str;
always @*
begin
   case (phase)
   {AES256,2'd0} : phase_str = {"AES256_P0"};
   {AES256,2'd1} : phase_str = {"AES256_P1"};
   {AES256,2'd2} : phase_str = {"AES256_P2"};
   {AES192,2'd0} : phase_str = {"AES192_P0"};
   {AES192,2'd1} : phase_str = {"AES192_P1"};
   {AES192,2'd2} : phase_str = {"AES192_P2"};
   {AES192,2'd3} : phase_str = {"AES192_P3"};
   {AES128,2'd0} : phase_str = {"AES128_P0"};
   {AES128,2'd1} : phase_str = {"AES128_P1"};
   default       : phase_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON

////////////////////////////////////////////////////////////////////////////////
// System Verilog Assertions
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_ASSERT_ON
`endif // RW_ASSERT_ON

endmodule
//-------------------------End of file------------------------------------------
