`default_nettype none
module ahbslaveif
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire          clk,
  input wire          rst_n,
 
  /*****************************************************************************
  * AHB slave
  *****************************************************************************/
  input  wire         hready_in,
  input  wire         hsel,
  input  wire [15:0]  haddr,
  input  wire [ 1:0]  htrans,
  input  wire         hwrite,
  
  input  wire [31:0]  hwdata,
  output reg  [31:0]  hrdata,
  output reg          hready,
 
  /*****************************************************************************
  * register bus
  *****************************************************************************/
  output wire [15:0]  regbus_addr,
  output wire         regbus_rden,
  output wire         regbus_wren,

  output reg  [31:0]  regbus_wdata,
  input  wire [31:0]  regbus_rdata,
  input  wire         regbus_ready
);
  /*******************************************************************************
  * declaration
  *******************************************************************************/
  /* q0 entry */
  reg [15:0]  q0_addr,n_q0_addr;
  reg         q0_write,n_q0_write;
  reg         q0_valid,n_q0_valid;
  wire        q0_ready;
  /* q1 entry */
  reg [15:0]  q1_addr,n_q1_addr;
  reg         q1_write,n_q1_write;
  reg         q1_valid,n_q1_valid;
  /* q2 entry */
  reg [15:0]  q2_addr,n_q2_addr;
  reg         q2_write,n_q2_write;
  reg         q2_valid,n_q2_valid;
  /* qd entry */
  reg         qd_valid,n_qd_valid;
  reg         qd_write,n_qd_write;

  /* ahb request  */
  wire        req_ahb_valid;
  wire [15:0] req_ahb_addr;
  wire        req_ahb_write;
  
  /* hready state */
  wire        n_hready_set_rd,n_hready_clr_rd;
  wire        n_hready_set_wr,n_hready_clr_wr;
  reg         hready_rd, hready_wr;
  
  /* wdata queue */ 
  reg hwdata_valid, hwdata_q_valid;
  reg [31:0] hwdata_q;

  /*******************************************************************************
  * o About request data width
  *  
  *   The interface only supports access of 32 bits. Others access data widths are
  *   treated as 32 bits.
  *
  * o AHB master BUSY are indeed supported
  * 
  * o About AHB wait states
  *  
  *   The following AHB accesses cause wait states:
  *     read accesses: 2 cycles to fetch data from regbus, plus cycles where  
  *                    regbus_ready goes low.
  *     write accesses: while waiting for the regbus_ready answer, up to 3 accesses
  *   are pipelined in the ahbslaveif queues. Then, the regbus_ready is reflected on hready.  
  *     write accesses followed by read access to a regbus device inserting wait states:
  *   The regbus device will first finish all previously posted write accesses, then 
  *   proceed to the read access. hready is low during all this time. 
  *
  * o About regbus timing
  *  
  *   The bridge assumes that the timing of regbus corresponds to a SRAM timing,
  *   with the possibility to insert wait states:
  *   1) write access occurs in the cycle where addr, wren and wdata are asserted
  *      and regbus_ready is high. The agent may assert regbus_ready low if it is 
  *      not able to proceed immediately to a write request.
  *   2) read data is returned the first cycle with regbus_ready high following
  *     the addr and rden assertion.
  * 
  * o About number of queues
  * 
  *   The devices on the regbus always post the first write access they receive, but
  *   can assert regbus_ready low if another write or read access is done on the next cycle.
  *   the queues q0, q1, q2, pipe the AHB requests until regbus_ready is sampled and hready
  *   can be set low.
  *  
  *******************************************************************************/
  assign q0_ready = regbus_ready;
    
  /*******************************************************************************
  * AHB
  *******************************************************************************/
  
  /* request from AHB */
  assign {req_ahb_valid, req_ahb_addr,req_ahb_write} = {hready_in & htrans[1] & hsel, haddr[15:0], hwrite};
  
  /* hready */
  // hready goes to low if an AHB read access is performed, and then to high when read data is returned.
  assign n_hready_clr_rd = req_ahb_valid & ~req_ahb_write;
  assign n_hready_set_rd = q0_ready & qd_valid & ~qd_write;
  // hready goes to low if an AHB write access is performed while the regbus is not ready,
  // and then to high when the write access is done.
  assign n_hready_clr_wr = ~q0_ready & req_ahb_valid & req_ahb_write;
  assign n_hready_set_wr = q0_ready & q0_valid &  q0_write;

  /* queue */
  always @(*)
  begin
  
    /* qd : q0 delayed by one regbus cycle */
    if (q0_ready)
      {n_qd_valid,n_qd_write} = {q0_valid,q0_write};
    else 
      {n_qd_valid,n_qd_write} = {qd_valid,qd_write};

    /* q0,q1,q2 */
    
    // Incoming AHB request
    if (req_ahb_valid) begin
      if (q2_valid) begin // ahb goes to q2, q2 goes to q1, q1 goes to q0
        // Note: when an AHB request is received an q2 is valid, by design the regbus is ready and 
        // all queues can be shifted on the next cycle.
        {n_q0_valid,n_q0_addr,n_q0_write} = {q1_valid,q1_addr,q1_write};                  // shift
        {n_q1_valid,n_q1_addr,n_q1_write} = {q2_valid,q2_addr,q2_write};                  // shift
        {n_q2_valid,n_q2_addr,n_q2_write} = {req_ahb_valid,req_ahb_addr,req_ahb_write};   // shift and fill
      end else if (q1_valid) begin
        if (q0_ready) begin  // q1 goes to q0, ahb goes to q1
          {n_q0_valid,n_q0_addr,n_q0_write} = {q1_valid,q1_addr,q1_write};                // shift
          {n_q1_valid,n_q1_addr,n_q1_write} = {req_ahb_valid,req_ahb_addr,req_ahb_write}; // shift and fill
          {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write};                // empty
        end else begin  // ahb goes to q2, others are frozen
          {n_q0_valid,n_q0_addr,n_q0_write} = {q0_valid,q0_addr,q0_write};                // full
          {n_q1_valid,n_q1_addr,n_q1_write} = {q1_valid,q1_addr,q1_write};                // full
          {n_q2_valid,n_q2_addr,n_q2_write} = {req_ahb_valid,req_ahb_addr,req_ahb_write}; // fill
        end
      end else if (q0_ready | ~q0_valid) begin  // q0 free or free on the next cycle
        // ahb goes to q0 if read, q1 if write. 
        if (~req_ahb_write) begin
          {n_q0_valid,n_q0_addr,n_q0_write} = {req_ahb_valid,req_ahb_addr,req_ahb_write}; // fill
          {n_q1_valid,n_q1_addr,n_q1_write} = {q1_valid,q1_addr,q1_write};                // empty
          {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write};                // empty
        end else begin
          {n_q0_valid,n_q0_addr,n_q0_write} = {1'b0,q0_addr,q0_write};                    // q0 done
          {n_q1_valid,n_q1_addr,n_q1_write} = {req_ahb_valid,req_ahb_addr,req_ahb_write}; // fill
          {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write};                // empty
        end
      end else begin // q0 can not change, ahb goes to q1
        {n_q0_valid,n_q0_addr,n_q0_write} = {q0_valid,q0_addr,q0_write};                  // full
        {n_q1_valid,n_q1_addr,n_q1_write} = {req_ahb_valid,req_ahb_addr,req_ahb_write};   // fill
        {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write};                  // empty
      end

    // No AHB request, shift queues when q0 ready
    end else begin
      if (q0_ready | ~q0_valid) begin  // q0 free or free on the next cycle
        if (q2_valid) begin  // q2 goes to q1, q1 goes to q0
          {n_q0_valid,n_q0_addr,n_q0_write} = {q1_valid,q1_addr,q1_write}; // shift
          {n_q1_valid,n_q1_addr,n_q1_write} = {q2_valid,q2_addr,q2_write}; // shift
          {n_q2_valid,n_q2_addr,n_q2_write} = {1'b0,q2_addr,q2_write};     // q2 done
        end else if (q1_valid) begin  // q1 goes to q0
          {n_q0_valid,n_q0_addr,n_q0_write} = {q1_valid,q1_addr,q1_write}; // shift
          {n_q1_valid,n_q1_addr,n_q1_write} = {1'b0,q1_addr,q1_write};     // q1 done
          {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write}; // empty
        end else begin
          {n_q0_valid,n_q0_addr,n_q0_write} = {1'b0,q0_addr,q0_write};     // q0 done
          {n_q1_valid,n_q1_addr,n_q1_write} = {q1_valid,q1_addr,q1_write}; // empty
          {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write}; // empty
        end
      end else begin // wait for q0 free
        {n_q0_valid,n_q0_addr,n_q0_write} = {q0_valid,q0_addr,q0_write};
        {n_q1_valid,n_q1_addr,n_q1_write} = {q1_valid,q1_addr,q1_write};
        {n_q2_valid,n_q2_addr,n_q2_write} = {q2_valid,q2_addr,q2_write};
      end
    end
      
  end
  
  /* registers */
  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
     
      hready                      <= 1'b1;
      hready_rd                   <= 1'b1;
      hready_wr                   <= 1'b1;
      hrdata                      <= 32'b0; 
      regbus_wdata                <= 32'b0;
      
      hwdata_valid                <= 1'b0;
      hwdata_q                    <= 32'b0; 
      hwdata_q_valid              <= 1'b0;

      {q0_valid,q0_addr,q0_write} <= 18'b0;
      {q1_valid,q1_addr,q1_write} <= 18'b0;
      {q2_valid,q2_addr,q2_write} <= 18'b0;
      {qd_valid,qd_write}         <= 2'b0;  
   
    end
    else
    begin
      
      hrdata                      <= regbus_rdata;
      
      // n_hready_clr has priority over n_hready_set
      if (n_hready_clr_rd)
        hready_rd <= 1'b0;
      else if (n_hready_set_rd)
        hready_rd <= 1'b1;

      if (n_hready_clr_wr)
        hready_wr <= 1'b0;
      else if (n_hready_set_wr)
        hready_wr <= 1'b1;

      // n_hready_clr_rd/wr is always taken into account.
      // n_hready_set_rd/wr is taken into account only if hready low was due to the same access type.
      if (n_hready_clr_rd || n_hready_clr_wr)
        hready <= 1'b0;
      else if ((n_hready_set_rd & ~ hready_rd) || (n_hready_set_wr & ~ hready_wr))
        hready <= 1'b1;

      // change regbus_wdata only when regbus is ready. Use oldest data (wdata queue if valid) first.
      if (regbus_ready==1'b1) begin
        if (hwdata_q_valid) begin
          regbus_wdata   <= hwdata_q;
          hwdata_q_valid <= 1'b0;      // queue done
        end else begin
          regbus_wdata   <= hwdata;
        end
      end
      
      // wdata queue
      if (hready_in==1'b1) begin
        hwdata_valid <= req_ahb_write & req_ahb_valid;
        // if q1 already contains a write access and regbus is not ready, the incoming
        // write access will go to q2 and hwdata must be queued.
        if (hwdata_valid & q1_valid & q1_write & ~q0_ready) begin
          hwdata_q       <= hwdata;
          hwdata_q_valid <= 1'b1;
        end
      end

      {q0_valid,q0_addr,q0_write} <= {n_q0_valid,n_q0_addr,n_q0_write};
      {q1_valid,q1_addr,q1_write} <= {n_q1_valid,n_q1_addr,n_q1_write};
      {q2_valid,q2_addr,q2_write} <= {n_q2_valid,n_q2_addr,n_q2_write};
      {qd_valid,qd_write}         <= {n_qd_valid,n_qd_write};
      
    end
  
  /* regbus interface */
  assign regbus_addr   = q0_addr;                  
  assign regbus_wren   = q0_valid &  q0_write;     
  assign regbus_rden   = q0_valid & ~q0_write;     

endmodule
`default_nettype wire
