////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Counter with CBC-MAC (CCM).
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module ccm(
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          hard_rst_n,              // Hardware Reset
   input  wire          soft_rst_n,              // Software Reset
 
   //$port_g CCM AES MIC
   output reg   [127:0] aes_mic_in,              // AES MIC data in
   output reg           aes_mic_in_valid,        // Pulse valid when aes MIC data in is ready
   input  wire  [127:0] aes_mic_out,             // AES MIC data out
   input  wire          aes_mic_out_valid,       // Pulse valid when aes MIC data out is ready

   //$port_g CCM AES CRT
   output wire  [127:0] aes_crt_in,              // AES CRT data in
   output reg           aes_crt_in_valid,        // Pulse valid when aes CRT data in is ready
   input  wire  [127:0] aes_crt_out,             // AES CRT data out
   input  wire          aes_crt_out_valid,       // Pulse valid when aes CRT data out is ready

   //$port_g CONTROL
   input  wire          encrypt,                 // Encrypt/Decrypt
   input  wire          init_p,                  // Initialization pulse
   output wire          busy,                    // CCM is busy
   input  wire  [239:0] aad,                     // Additional Authenticated Data
   input  wire   [15:0] aad_len,                 // AAD Length
   input  wire  [103:0] nonce,                   // NONCE
   input  wire   [15:0] data_len,                // Plain/Crypted Data Length
   input  wire    [1:0] mic_len,                 // Indicate the CCM MIC Len (0:64, 1:96, 2:128)
   output wire          mic_passed_p,            // Pulse for MIC passed
   output wire          mic_failed_p,            // Pulse for MIC failed

   //$port_g Data interface
   input  wire  [127:0] data_in,                 // Plain/Encrypted data in
   input  wire   [15:0] data_in_byte_en,         // Data input byte enable
   input  wire          data_in_last,            // Last data input
   input  wire          data_in_valid,           // Data input is valid
   output wire          data_in_ready,           // CCM is ready to receive data
   output wire  [127:0] data_out,                // Plain/Encrypted data out
   output wire          data_out_valid,          // Data output is valid
   input  wire          data_out_ready,          // Data output ready
   output wire  [127:0] mic_out,                 // MIC output
   output wire          mic_out_valid,           // MIC output is valid
   input  wire          mic_out_ready,           // MIC output ready
// input   wire            ccmp_rxError_p,         // Pulse to indicate rx Error
// input   wire            ccmp_txError_p,         // Pulse to indicate tx Error

   //$port_g Context interface
   input  wire  [396:0] context_in,              // Context input
   input  wire          context_in_valid,        // Context input is valid
   output wire  [396:0] context_out,             // Context output

   //$port_g Debug Port
   output wire   [31:0] debug                    // Debug Port
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// ccm FSM states definition
//$fsm_sd ccm
localparam 
   IDLE      = 4'd0,
   INITB0    = 4'd1,
   INITAAD1  = 4'd2,
   INITAAD2  = 4'd3,
   MSG       = 4'd4,
   LAST_MSG  = 4'd5,
   MIC       = 4'd6,
   MIC_CHECK = 4'd7,
   ERROR     = 4'd8;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg    [3:0] ccm_cs;                   // CCM Current state
reg    [3:0] ccm_ns;                   // CCM Next State

wire         ccm_enable;               // CCM Enable (for automatic clock gating)

reg          aes_mic_done;
reg          aes_crt_done;
reg  [127:0] aes_crt_0_reg;
reg  [127:0] aes_crt_i_reg;
reg  [127:0] aes_mic_reg;

reg   [12:0] counter;
wire  [15:0] counter16;

reg  [127:0] data_in_mask;
reg  [127:0] mic_mask;

reg    [7:0] auth_flags;               // Authentication flags
//reg               ccmp_error;

wire [127:0] context_crt_0;            // CRT 0 context
wire [127:0] context_crt_i;            // CRT i context
wire [127:0] context_mic;              // MIC context
wire  [12:0] context_counter;          // Counter context


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////
//
//always @(posedge clk or negedge hard_rst_n)
//begin
//  if (hard_rst_n == 1'b0)
//    ccmp_error  <= 1'b0;
//  else
//    ccmp_error  <= ((encrypt == 1'b0) & (ccmp_rxError_p == 1'b1))                                                           | // error pulse from rx controller
//                   ((encrypt == 1'b0) & (ccm_cs   != IDLE) & (ccm_cs   != ERROR) & (data_len == 16'h0))| // error from rx length
//                   ((encrypt == 1'b1) & (ccmp_txError_p == 1'b1));                                                             // error from tx controller
//end

  // Generates Automatic Clockgating signal enable (useful to gate all the FFs' clk when the signal is low) :


//******************************************************************************
// CCM Enable
//    CCM is enabled when FSM is not in IDLE state or when context is loaded.
//******************************************************************************
assign ccm_enable = ccm_cs!=IDLE | ccm_ns!=IDLE | context_in_valid;


//******************************************************************************
// CCM FSM State Logic 
//******************************************************************************
// current state :
// Takes the value of ccm_ns when ccm_enable is high
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      ccm_cs <= IDLE;
   else if (~soft_rst_n)
      ccm_cs <= IDLE;
   else if (ccm_enable) 
      ccm_cs <= ccm_ns;
end

// next state :
// ccm_ns changes according to the ccm_cs and the right condition.
always @*
begin
   ////when not in ERROR state, and ccmp_error rise, then ccm_ns goes to ERROR.
   //if ((ccm_cs != ERROR) & (ccm_cs != IDLE) & (ccmp_error == 1'b1))
   //begin
   //  ccm_ns = ERROR;
   //end
   ////else ccmp_error can be ignored
   //else
   begin
      case(ccm_cs)
      IDLE:
      begin
         //$fsm_s In IDLE, the state waits for the init pulse
         if (init_p)
            //$fsm_t If init pulse is received then the state goes to INITB0 state
            ccm_ns = INITB0;
         else if (data_in_valid)
            //$fsm_t When data is provided, the state machine goes to MSG state
            ccm_ns = MSG;
         else
            //$fsm_t While no init pulse is received, the state stays in IDLE state
            ccm_ns = IDLE;
      end

      INITB0:
      begin
         //$fsm_s In INITB0, the state waits for the block AES MIC computation. 
         if (aes_mic_done)
            //$fsm_t when B0 computation is done, the state goes to INITAAD1 state
            ccm_ns = INITAAD1;
         else
            //$fsm_t While B0 computation is not done, the state stays in INITB0
            //state
            ccm_ns = INITB0;
      end

      INITAAD1:
      begin
         //$fsm_s In INITAAD1, the state wait for the block AES MIC computation
         if (aes_mic_done)
            //$fsm_t when AAD1 computation is done, the state goes to INITAAD2 
            //state
            ccm_ns = INITAAD2;
         else
            //$fsm_t While ADD1 computation is not done, the state waits in 
            //INITAAD1 state
            ccm_ns = INITAAD1;
      end

      INITAAD2:
      begin
         //$fsm_s In INITAAD2, the state wait for the block AES MIC computation
         if (aes_mic_done & data_in_valid)
            //$fsm_t When AAD2 computation is done and the data is provided, the
            //state goes to MSG state
            ccm_ns = MSG;
         else if (aes_mic_done & ~data_in_valid)
            //$fsm_t When AAD2 computation is done and data is not provided, the
            //state machine goes to IDLE state
            ccm_ns = IDLE;
         else
            //$fsm_t While AAD2 computation is not done, the state waits in
            //INITAAD2 state
            ccm_ns = INITAAD2;
      end

      MSG:
      begin
         //$fsm_s In MSG, the state wait for the last block
         if (data_in_valid & data_in_ready & data_in_last)
            //$fsm_t When the last block of data is provided, the state goes to 
            //LAST_MSG state
            ccm_ns = LAST_MSG;
         else if (aes_mic_done & aes_crt_done & ~data_in_valid)
            //$fsm_t When data is not provided, the state machine goes to IDLE
            //state
            ccm_ns = IDLE;
         else
            //$fsm_t While there are still payload to encrypt/decrypt, the state
            //stays in MSG state
            ccm_ns = MSG;
      end

      LAST_MSG:
      begin
         //$fsm_s In MSG, the state wait for the last block computation
         if (aes_mic_done & aes_crt_done)
            //$fsm_t When the last block of data is encrypted/decrypted, the
            //state goes to MIC state
            ccm_ns = MIC;
         else
            //$fsm_t While last block is not encrypted/decrypted, the state stays
            //in LAST_MSG state
            ccm_ns = LAST_MSG;
      end

      MIC:
      begin
         //$fsm_s In MIC, the state wait for the mic shifted out in encryption
         if (encrypt & mic_out_ready)
            //$fsm_t When in encryption and mic shifted out, the state goes to 
            //IDLE state
            ccm_ns = IDLE;
         else if (~encrypt & mic_out_ready)
            //$fsm_t When in reception, the state goes to MIC_CHECK state
            ccm_ns = MIC_CHECK;
         else
            //$fsm_t While in encryption and mic not shifted out, the state stays
            //in MIC state
            ccm_ns = MIC;
      end

      MIC_CHECK:
      begin
         //$fsm_s In MIC_CHECK, the state machine waits MIC
         if (data_in_valid & data_in_ready)
            //$fsm_t When MIC is provided, the state goes to IDLE state
            ccm_ns = IDLE;
         else
            //$fsm_t While MIC is not provided, the state machine stays in
            //MIC_CHECK state
            ccm_ns = MIC_CHECK;
      end

      //ERROR:
      //begin
      //   //$fsm_s In ERROR, the state wait for one clock cycle
      //    
      //   //$fsm_t After one clock cycle, the state goes to IDLE state
      //   ccm_ns = IDLE;
      //end

      default :
      begin
         ccm_ns = IDLE;
      end 
      endcase
   end
end


//******************************************************************************
// Authentication Flags (Reserved, Adata, M', L')
//******************************************************************************
always @*
begin
   case (mic_len)
   2'd2:   auth_flags = {1'b0,1'b1,3'd7,3'd1};
   2'd1:   auth_flags = {1'b0,1'b1,3'd5,3'd1};
   default:auth_flags = {1'b0,1'b1,3'd3,3'd1};
   endcase
end


//******************************************************************************
// AES MIC Control
//******************************************************************************
always @(*)
begin
   case (ccm_ns)
   INITB0  : aes_mic_in = {data_len[7:0],data_len[15:8],nonce[103:0], auth_flags};
   INITAAD1: aes_mic_in = aes_mic_reg ^ {aad[111:0],aad_len[7:0],aad_len[15:8]};
   INITAAD2: aes_mic_in = aes_mic_reg ^ aad[239:112];
   default://MSG
   begin
      if (encrypt)
         aes_mic_in = aes_mic_reg ^ data_in;
      else
         aes_mic_in = aes_mic_reg ^ data_out;
   end
   endcase

   // Ouput pulse to trigger aes mic block only when needed 
   if ((ccm_cs==IDLE     & ccm_ns==INITB0  )|
       (ccm_cs==INITB0   & ccm_ns==INITAAD1)|
       (ccm_cs==INITAAD1 & ccm_ns==INITAAD2))
      aes_mic_in_valid = 1'b1;
   else if (ccm_cs==MSG)
      aes_mic_in_valid = data_in_valid & data_in_ready;
   else
      aes_mic_in_valid = 1'b0;
end


//******************************************************************************
// AES CRT Control
//******************************************************************************
assign aes_crt_in = {counter16[7:0], counter16[15:8], nonce, 8'h01};

always @*
begin
   // output pulse to trigger aes crt block only when needed
   if ((ccm_cs==INITB0   & ccm_ns==INITAAD1)|
       (ccm_cs==INITAAD1 & ccm_ns==INITAAD2))
      aes_crt_in_valid = 1'b1;
   else if (ccm_cs==MSG)
      aes_crt_in_valid = data_in_valid & data_in_ready;
   else
      aes_crt_in_valid = 1'b0;
end


//******************************************************************************
// AES Done flags
//    Store valid pulse from AES MIC & AES CRT blocks
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
   begin
      aes_mic_done  <= 1'b0;
      aes_crt_done  <= 1'b0;
   end
   else if (~soft_rst_n)
   begin
      aes_mic_done  <= 1'b0;
      aes_crt_done  <= 1'b0;
   end
   else if (ccm_enable) 
   begin
      if (aes_mic_in_valid)
         aes_mic_done  <= 1'b0;
      else if (aes_mic_out_valid)
         aes_mic_done  <= 1'b1;

      if (aes_crt_in_valid)
         aes_crt_done  <= 1'b0;
      else if (aes_crt_out_valid)
         aes_crt_done  <= 1'b1;
   end
end


//******************************************************************************
// Counter used for AES CRT
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      counter <= 13'h0;
   else if (~soft_rst_n)
      counter <= 13'h0;
   else if (ccm_enable) 
   begin
      if (context_in_valid)
        counter <= context_counter;
      else if (aes_crt_in_valid)
        counter <= counter + 13'd1;
      else if (ccm_ns==INITB0)
        counter <= 13'd0;
   end
end
assign counter16 = {3'h0, counter};


//******************************************************************************
// Caputure the CRT 0 result for MIC encryption
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      aes_crt_0_reg <= 128'h0;
   else if (~soft_rst_n)
      aes_crt_0_reg <= 128'h0;
   else if (ccm_enable) 
   begin
      if (context_in_valid)
         aes_crt_0_reg <= context_crt_0;
      else if (aes_crt_done & ccm_cs==INITAAD1)
         aes_crt_0_reg <= aes_crt_out;
      else if (ccm_ns==INITB0)
         aes_crt_0_reg <= 128'h0;
   end
end


//******************************************************************************
// Caputure AES CRT i result for Message encryption, needed by context switching
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      aes_crt_i_reg <= 128'h0;
   else if (~soft_rst_n)
      aes_crt_i_reg <= 128'h0;
   else if (ccm_enable) 
   begin
      if (context_in_valid)
         aes_crt_i_reg <= context_crt_i;
      else if (aes_crt_out_valid)
         aes_crt_i_reg <= aes_crt_out;
   end
end


//******************************************************************************
// Caputure AES MIC result for Message encryption, needed by context switching
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      aes_mic_reg <= 128'h0;
   else if (~soft_rst_n)
      aes_mic_reg <= 128'h0;
   else if (ccm_enable) 
   begin
      if (context_in_valid)
         aes_mic_reg <= context_mic;
      else if (aes_mic_out_valid)
         aes_mic_reg <= aes_mic_out;
   end
end


//******************************************************************************
// Data Input
//******************************************************************************
assign data_in_ready = (ccm_cs==MIC_CHECK) |
                       ((ccm_cs==MSG) & aes_mic_done & aes_crt_done & data_out_ready);


//******************************************************************************
// Data Output
//******************************************************************************
assign data_out       = (data_in ^ aes_crt_i_reg) & data_in_mask;
assign data_out_valid = (ccm_cs==MSG) & data_in_valid & data_in_ready;

always @*
begin: DATA_MASK
   // Local Variable
   integer v_byte;

   for (v_byte=0;v_byte<16;v_byte=v_byte+1)
      data_in_mask[v_byte*8 +: 8] = {8{data_in_byte_en[v_byte]}};
end


//******************************************************************************
// MIC Output
//******************************************************************************
always @*
begin
   case (mic_len)
   2'd2:    mic_mask = {128{1'b1}};
   2'd1:    mic_mask = {128{1'b1}}>>32;
   default: mic_mask = {128{1'b1}}>>64;
   endcase
end

assign mic_out       = (aes_crt_0_reg ^ aes_mic_reg) & mic_mask;
assign mic_out_valid = (ccm_cs==MIC) ? 1'b1 : 1'b0;

// Mic Fail and Pass
assign mic_passed_p = (ccm_cs==MIC_CHECK)  & (ccm_ns==IDLE) & (mic_out == (data_in & mic_mask));
assign mic_failed_p = ((ccm_cs==MIC_CHECK) & (ccm_ns==IDLE) & ~mic_passed_p) |
                      ((ccm_cs==ERROR) & ~encrypt);



//******************************************************************************
// Context switching
//******************************************************************************
assign context_out = {aes_crt_0_reg,aes_crt_i_reg,aes_mic_reg,counter};

assign {context_crt_0,context_crt_i,context_mic,context_counter} = context_in;


//******************************************************************************
// CCM State
//******************************************************************************
assign busy = ccm_cs==IDLE ? 1'b0 : 1'b1;


//******************************************************************************
// Debug Port
//******************************************************************************
assign debug = {17'h0,
                mic_passed_p,
                mic_failed_p,
                aes_mic_done,
                aes_crt_done,
                busy,
                mic_out_valid,
                mic_out_ready,
                data_in_last,
                data_in_valid,
                data_in_ready,
                init_p,
                ccm_cs[3:0]};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// CCM FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] ccm_cs_str;
reg  [20*8:0] ccm_ns_str;
always @*
begin
   case (ccm_cs)
   IDLE     :  ccm_cs_str = {"IDLE"};
   INITB0   :  ccm_cs_str = {"INITB0"};
   INITAAD1 :  ccm_cs_str = {"INITAAD1"};
   INITAAD2 :  ccm_cs_str = {"INITAAD2"};
   MSG      :  ccm_cs_str = {"MSG"};
   LAST_MSG :  ccm_cs_str = {"LAST_MSG"};
   MIC      :  ccm_cs_str = {"MIC"};
   MIC_CHECK:  ccm_cs_str = {"MIC_CHECK"};
   ERROR    :  ccm_cs_str = {"ERROR"};
   default  :  ccm_cs_str = {"XXX"};
   endcase
end

always @*
begin
   case (ccm_ns)
   IDLE     :  ccm_ns_str = {"IDLE"};
   INITB0   :  ccm_ns_str = {"INITB0"};
   INITAAD1 :  ccm_ns_str = {"INITAAD1"};
   INITAAD2 :  ccm_ns_str = {"INITAAD2"};
   MSG      :  ccm_ns_str = {"MSG"};
   LAST_MSG :  ccm_ns_str = {"LAST_MSG"};
   MIC      :  ccm_ns_str = {"MIC"};
   MIC_CHECK:  ccm_ns_str = {"MIC_CHECK"};
   ERROR    :  ccm_ns_str = {"ERROR"};
   default  :  ccm_ns_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef  RW_ASSERT_ON
`endif//RW_ASSERT_ON

endmodule  
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
