////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of CMAC module.
//                    This module compute the AES-CMAC of the provided
//                    message.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module cmac( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          hard_rst_n,              // Asynchronous Reset
   input  wire          soft_rst_n,              // Asynchronous Reset
 
   //$port_g Control interface
   input  wire          cmac_init,               // Initialisation
   output wire          cmac_busy,               // Busy
   input  wire  [127:0] cmac_data,               // Message word
   input  wire   [15:0] cmac_data_byte_en,       // Message word byte enable
   input  wire          cmac_data_last,          // Last data to compute
   input  wire          cmac_data_valid,         // Data is valid
   output wire          cmac_data_ready,         // Ready to load data
   output wire  [127:0] cmac_mic,                // MIC
   output wire          cmac_mic_valid,          // MIC is valid
   input  wire          cmac_mic_ready,          // Ready to load MIC

   //$port_g AES interface
   input  wire  [127:0] aes_data_out,            // Data from AES 
   input  wire          aes_data_out_valid,      // Data from AES qualifier
   output reg   [127:0] aes_data_in,             // Data to AES
   output reg           aes_data_in_valid,       // Data to AES qualifier

   //$port_g Context interface
   input  wire  [255:0] context_in,              // Context input
   input  wire          context_in_valid,        // Context input is valid
   output wire  [255:0] context_out              // Context output
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// FSM states definition
//$fsm_sd cmac
localparam 
   CMAC_IDLE      = 3'd0,
   CMAC_INIT      = 3'd1,
   CMAC_INIT_WAIT = 3'd2,
   CMAC_DATA      = 3'd3,
   CMAC_DATA_WAIT = 3'd4,
   CMAC_LAST_WAIT = 3'd5,
   CMAC_MIC_DONE  = 3'd6;

// The constant string for subkey generation for a cipher with block size 128
localparam R128 = 128'h87000000_00000000_00000000_00000000;


////////////////////////////////////////////////////////////////////////////////
// Function Definitions
////////////////////////////////////////////////////////////////////////////////

// CMAC left shift (<<1)
//   The shift is done on the swapped bytes
function [127:0] shift1;
input [127:0] din;
begin
   shift1 = {din[126:120],1'b0,
             din[118:112],din[127],
             din[110:104],din[119],
             din[102:96],din[111],
             din[94:88],din[103],
             din[86:80],din[95],
             din[78:72],din[87],
             din[70:64],din[79],
             din[62:56],din[71],
             din[54:48],din[63],
             din[46:40],din[55],
             din[38:32],din[47],
             din[30:24],din[39],
             din[22:16],din[31],
             din[14:8],din[23],
             din[6:0],din[15]};
end
endfunction


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////

reg    [2:0] cmac_cs;        // CMAC FSM current state
reg    [2:0] cmac_ns;        // CMAC FSM next state
                             
reg  [127:0] ciphk;          // CIPH(k)
reg  [127:0] k1;             // subkey K1
reg  [127:0] k2;             // subkey K2
reg  [127:0] subkey;         // selected subkey
                             
reg  [127:0] masked_data;    // data with masked bytes
reg  [127:0] padded_data;    // data with padding bits
                             
wire [127:0] context_ciphk;  // CIPHk context
wire [127:0] context_k1;     // K1 context


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// Output assignment
//******************************************************************************
// FSM current state logic
always @(posedge clk or negedge hard_rst_n) 
begin
   if (~hard_rst_n)
      cmac_cs <= CMAC_IDLE;
   else if (~soft_rst_n)
      cmac_cs <= CMAC_IDLE;
   else
      cmac_cs <= cmac_ns;
end


//******************************************************************************
// FSM next state logic
//******************************************************************************
always @* 
begin  
   case(cmac_cs)
   CMAC_IDLE:
   begin
      //$fsm_s In CMAC_IDLE state, the state machine waits the cmac_init trigger
      if (cmac_init)
         //$fsm_t When cmac_init trigger is received, the state machine goes to 
         //CMAC_INIT state
         cmac_ns = CMAC_INIT;
      else if (cmac_data_valid)
         //$fsm_t When next data is valid, the state machine goes to CMAC_DATA
         //state
         cmac_ns = CMAC_DATA;
      else
         //$fsm_t While cmac_init trigger is not received, the state machine
         //stays in CMAC_IDLE state
         cmac_ns = CMAC_IDLE;
   end

   CMAC_INIT:
   begin
      //$fsm_s In CMAC_INIT state, CIPH(0) computation is requested

      //$fsm_t After one clock cycle, the state machine goes to CMAC_INIT_WAIT
      //state
      cmac_ns = CMAC_INIT_WAIT;
   end

   CMAC_INIT_WAIT:
   begin
      //$fsm_s In CMAC_INIT_WAIT state, the state machine waits CIPH(0)
      //computation done
      if (aes_data_out_valid & cmac_data_valid)
         //$fsm_t When CIPH(0) computation is done and first data is valid, the
         //state machine goes to CMAC_DATA state
         cmac_ns = CMAC_DATA;
      else if (aes_data_out_valid & ~cmac_data_valid)
         //$fsm_t When CIPH(0) computation is done and first data is not valid,
         //the state machine goes to IDLE state
         cmac_ns = CMAC_IDLE;
      else
         //$fsm_t While AES computation is not done, the state machine stays in
         //CMAC_INIT_WAIT state
         cmac_ns = CMAC_INIT_WAIT;
   end

   CMAC_DATA:
   begin
      //$fsm_s In CMAC_DATA state, the state machine waits cmac_data valid
      if (cmac_data_valid & cmac_data_last)
         //$fsm_t When cmac_data is valid, the state machine goes to 
         //CMAC_LAST_WAIT state
         cmac_ns = CMAC_LAST_WAIT;
      else if (cmac_data_valid/* & ~cmac_data_last*/)
         //$fsm_t When cmac_data is valid, the state machine goes to 
         //CMAC_DATA_WAIT state
         cmac_ns = CMAC_DATA_WAIT;
      else
         //$fsm_t While cmac_data is not valid, the state machine stays in 
         //CMAC_DATA state
         cmac_ns = CMAC_DATA;
   end

   CMAC_DATA_WAIT:
   begin
      //$fsm_s In CMAC_DATA_WAIT state, the state machine waits AES computation
      //done
      if (aes_data_out_valid & cmac_data_valid)
         //$fsm_t When AES computation is done and next data is valid, the state
         //machine goes to CMAC_DATA state
         cmac_ns = CMAC_DATA;
      else if (aes_data_out_valid & ~cmac_data_valid)
         //$fsm_t When AES computation is done and next data is not valid, the
         //state machine goes to IDLE state
         cmac_ns = CMAC_IDLE;
      else
         //$fsm_t While AES computation is not done, the state machine stays in
         //CMAC_DATA_WAIT state
         cmac_ns = CMAC_DATA_WAIT;
   end

   CMAC_LAST_WAIT:
   begin
      //$fsm_s In CMAC_LAST_WAIT state, the state machine waits AES computation
      //done
      if (aes_data_out_valid)
         //$fsm_t When AES computation is done, the state machine goes to 
         //CMAC_IDLE state
         cmac_ns = CMAC_MIC_DONE;
      else
         //$fsm_t While AES computation is not done, the state machine stays in
         //WAIT state
         cmac_ns = CMAC_LAST_WAIT;
   end

   CMAC_MIC_DONE:
   begin
      //$fsm_s In CMAC_MIC_DONE state, the state machine waits MIC ready
      if (/*cmac_mic_valid & */cmac_mic_ready)
         //$fsm_t When MIC ready is received, the state machine goes to 
         //CMAC_IDLE state
         cmac_ns = CMAC_IDLE;
      else
         //$fsm_t While MIC ready is not received, the state machine stays in
         //CMAC_MIC_DONE state
         cmac_ns = CMAC_MIC_DONE;
   end

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      cmac_ns = CMAC_IDLE;
   // pragma coverage block = on
   endcase
end


//******************************************************************************
// Subkeys K1,K2
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      k1 <= 128'h0;
   else if (~soft_rst_n)
      k1 <= 128'h0;
   else
   begin
      if (context_in_valid)
         k1 <= context_k1;
      else if ((cmac_cs==CMAC_INIT_WAIT) & aes_data_out_valid)
      begin
         if (aes_data_out[7])
            k1 <= shift1(aes_data_out) ^ R128;
         else
            k1 <= shift1(aes_data_out);
      end
   end
end

always @*
begin
   if (k1[7])
      k2 = shift1(k1) ^ R128;
   else
      k2 = shift1(k1);
end


//******************************************************************************
// Padding
//******************************************************************************
always @*
begin:CMAC_PADDING
   // Local Variable
   reg [15:0] v_padding;
   integer    v_byte;

   v_padding = cmac_data_byte_en^{cmac_data_byte_en[14:0],1'b1};

   for (v_byte=0;v_byte<16;v_byte=v_byte+1)
   begin
      masked_data[v_byte*8 +: 8] = cmac_data[v_byte*8 +: 8] & {8{cmac_data_byte_en[v_byte]}};
   end

   for (v_byte=0;v_byte<16;v_byte=v_byte+1)
   begin
      padded_data[v_byte*8 +: 8] = masked_data[v_byte*8 +: 8] | {v_padding[v_byte],7'd0};
   end
end


//******************************************************************************
// Subkey selection
//******************************************************************************
always @*
begin
   if (cmac_data_last & cmac_data_byte_en[15])
      subkey = k1;
   else if (cmac_data_last & ~cmac_data_byte_en[15])
      subkey = k2;
   else
      subkey = 128'h0;
end


//******************************************************************************
// Cipher control
//******************************************************************************
always @*
begin
   case (cmac_cs)
   CMAC_IDLE:
   begin
      aes_data_in       = 128'h0;
      aes_data_in_valid = 1'b0;
   end

   CMAC_INIT:
   begin
      aes_data_in       = 128'h0;
      aes_data_in_valid = 1'b1;
   end

   CMAC_DATA:
   begin
      aes_data_in       = padded_data ^ subkey ^ ciphk;
      aes_data_in_valid = cmac_data_valid;
   end

   default:
   begin
      aes_data_in       = 128'h0;
      aes_data_in_valid = 1'b0;
   end
   endcase
end


//******************************************************************************
// CIPHk
//    Store AES output, needed by context switching.
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      ciphk <= 128'h0;
   else if (~soft_rst_n)
      ciphk <= 128'h0;
   else// if (enable)
   begin
      if (context_in_valid)
         ciphk <= context_ciphk;
      else if ((cmac_cs==CMAC_INIT_WAIT ) & aes_data_out_valid)
         ciphk <= 128'h0;
      else if ((cmac_cs==CMAC_DATA_WAIT |
                cmac_cs==CMAC_LAST_WAIT ) & aes_data_out_valid)
         ciphk <= aes_data_out;
   end
end


//******************************************************************************
// Data
//******************************************************************************
assign cmac_data_ready = cmac_cs==CMAC_DATA ? 1'b1 : 1'b0;


//******************************************************************************
// MIC output
//******************************************************************************
assign cmac_mic       = ciphk/*[63:0]*/;
assign cmac_mic_valid = cmac_cs==CMAC_MIC_DONE ? 1'b1 : 1'b0;


//******************************************************************************
// CMAC State
//******************************************************************************
assign cmac_busy = cmac_cs==CMAC_IDLE ? 1'b0 : 1'b1;


//******************************************************************************
// Context switching
//******************************************************************************
assign context_out = {k1,ciphk};

assign {context_k1,context_ciphk} = context_in;


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] cmac_cs_str;
always @*
begin
   case (cmac_cs)
   CMAC_IDLE      : cmac_cs_str = {"CMAC_IDLE"};
   CMAC_INIT      : cmac_cs_str = {"CMAC_INIT"};
   CMAC_INIT_WAIT : cmac_cs_str = {"CMAC_INIT_WAIT"};
   CMAC_DATA      : cmac_cs_str = {"CMAC_DATA"};
   CMAC_DATA_WAIT : cmac_cs_str = {"CMAC_DATA_WAIT"};
   CMAC_LAST_WAIT : cmac_cs_str = {"CMAC_LAST_WAIT"};
   CMAC_MIC_DONE  : cmac_cs_str = {"CMAC_MIC_DONE"};
   default        : cmac_cs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON

`endif // RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
