//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of rw_coex_rc module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_coex_rc( 
   ////////////////////////////////
   //$port_g Clock And Reset
   ////////////////////////////////
   input wire           rc_rst_n,      // Asynchronous Reset (active low)
   input wire           rc_clk,        // Radio Clock
                   
   ////////////////////////////////         
   //$port_g Modem Controls                 
   ////////////////////////////////         
   input  wire          mdm_on_req,    // Phy On Request from Modem
   output reg           mdm_on_ack,    // Phy On Acknowloedge to Modem
                                            
   ////////////////////////////////         
   //$port_g Radio Controls                 
   ////////////////////////////////         
   output reg           rc_on_req,     // Phy On Request to RC
   input  wire          rc_on_ack,     // Phy On Acknowledge from RC
                                            
   ////////////////////////////////         
   //$port_g PTA Controls                   
   ////////////////////////////////         
   input  wire          rc_abort,      // Abort request

   ////////////////////////////////         
   //$port_g Diagnostic Port
   ////////////////////////////////         
   output wire   [15:0] diag
   );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

//------ FSM states definition ------
//$fsm_sd packet_cntl_fsm_state
localparam [1:0]
   IDLE     =  2'd0,  // Normal Mode (Radio Controller is controlled by the Modem)
   ABORTED  =  2'd1,  // Abort  Mode (Radio is stopped, but On is requested by the Modem)
   PENDING  =  2'd2,  // Abort Pending Mode (Radio is stopped, Off is requested by the Modem)
   PWR_ON   =  2'd3;  // Power On Mode (The Radio is powered on when abort is finished and On is requested by the Modem)

//////////////////////////////////////////////////////////////////////////////
// DEFINES 
//////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
  // String definition to display fsm current state
  reg [11*8:0] fsm_cs_str;
`endif // RW_SIMU_ON


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// Main FSM signals definition
reg                    [1:0] fsm_cs;                       // FSM Current State
reg                    [1:0] fsm_ns;                       // FSM Next State

// Resynchronisation flop
reg                          rc_abort_ff1_resync;
reg                          rc_abort_ff2_resync;
wire                         rc_abort_sync;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
//------ RC Controls ------
always @*
begin
   case (fsm_cs)
   IDLE:
   begin
      rc_on_req  = mdm_on_req;
      mdm_on_ack = rc_on_ack;
   end
   PENDING:
   begin
      rc_on_req  = 1'b0;
      mdm_on_ack = 1'b0;
   end
   ABORTED:
   begin
      rc_on_req  = 1'b0;
      mdm_on_ack = 1'b1;
   end
   PWR_ON:
   begin
      rc_on_req  = 1'b1;
      mdm_on_ack = 1'b1;
   end
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default :   
   begin
      rc_on_req  = mdm_on_req;
      mdm_on_ack = rc_on_ack;
   end
   // pragma coverage block = on 
   endcase
end

//------ Main FSM Current State Logic ------
always @ (posedge rc_clk or negedge rc_rst_n) 
begin
   // Asynchronous Reset
   if (rc_rst_n == 1'b0)
      fsm_cs <= IDLE; 
   else
      fsm_cs <= fsm_ns; 
end

//------ Main FSM Next State Logic------
always @* 
begin
   case(fsm_cs)
   IDLE :
   //$fsm_s In IDLE state, FSM waits Abort event
   begin
      if ((rc_abort_sync == 1'b1) && (mdm_on_req == 1'b1) && (rc_on_ack == 1'b1))
         //$fsm_t  If Abort Request And Phy is On, Go to Aborted State
         fsm_ns = ABORTED; 
           
      else if ((rc_abort_sync == 1'b1) && (mdm_on_req == 1'b0) && (rc_on_ack == 1'b0))
         //$fsm_t  If Abort Request And Phy is Off, Go to Pending State
         fsm_ns = PENDING; 

      else
         //$fsm_t  Stanby in Idle mode
         fsm_ns = IDLE;
   end
      
   PENDING :
   //$fsm_s In PENDING state, Control From Modem is masked
   begin
      if ((rc_abort_sync == 1'b0) && (rc_on_ack == 1'b0))
         //$fsm_t  If Abort Release And Phy is Off, Go to IDLE
         fsm_ns = IDLE; 

      else if ((rc_abort_sync == 1'b1) && (mdm_on_req == 1'b1))
         //$fsm_t  If Abort Request And Phy On Request, Go to ABORTED State
         fsm_ns = ABORTED; 

      else
         //$fsm_t  Stay in PENDING
         fsm_ns = PENDING;
   end
      
   ABORTED :
   //$fsm_s In ABORTED state, Control From Modem is masked
   begin
      if ((rc_abort_sync == 1'b0) && (mdm_on_req == 1'b1))
         //$fsm_t  If Abort Release And Phy On Request, Go to PWR_ON State
         fsm_ns = PWR_ON; 

      else if (mdm_on_req == 1'b0)
         //$fsm_t  If Phy Off Request, Go to PENDING
         fsm_ns = PENDING; 

      else
         //$fsm_t  Stay in ABORTED
         fsm_ns = ABORTED;
   end
      
   PWR_ON :
   //$fsm_s In PWR_ON state, Wait Phy PowerOn
   begin
      if ((mdm_on_req == 1'b0) || (rc_on_ack == 1'b1))
         //$fsm_t  If Phy Off Request or PHY Is On, Go to IDLE State
         fsm_ns = IDLE; 

      else
         //$fsm_t  Stay in PWR_ON
         fsm_ns = PWR_ON;
   end
      
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default :   
      fsm_ns = IDLE; 
   // pragma coverage block = on 
   endcase
end

//------ Cross Domain Resynchronisation ------
always @(posedge rc_clk or negedge rc_rst_n)
begin
   if (rc_rst_n == 1'b0)
   begin
      rc_abort_ff1_resync <= 1'b0;
      rc_abort_ff2_resync <= 1'b0;
   end
   else
   begin
      rc_abort_ff1_resync <= rc_abort;
      rc_abort_ff2_resync <= rc_abort_ff1_resync;
   end
end

assign rc_abort_sync = rc_abort_ff2_resync;

//------ Diagnostic Port ------
assign diag = {
   9'h0        ,
   rc_abort    ,  
   rc_on_req   , 
   rc_on_ack   , 
   mdm_on_req  ,
   mdm_on_ack  ,
   fsm_cs[1:0]
};

////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// ------ Main FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (fsm_cs)
  IDLE        : fsm_cs_str <= {"IDLE"};
  PENDING     : fsm_cs_str <= {"PENDING"};
  ABORTED     : fsm_cs_str <= {"ABORTED"};
  PWR_ON      : fsm_cs_str <= {"PWR_ON"};
  // Disable coverage on the default state because it cannot be reached.
  // pragma coverage block = off 
  default     : fsm_cs_str <= {"XXX"};
  // pragma coverage block = on 
  endcase
end
`endif // RW_SIMU_ON


////////////////////////////////////////////////////////////////////////////////
// System verilog Assertion
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_ASSERT_ON
// -------- Assertion --------

// -------- Cover Point --------

`endif // RW_ASSERT_ON

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
