//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: lvalla $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 38796 $
// $Date: 2019-05-17 10:16:36 +0200 (Fri, 17 May 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of rw_nx_la module
// Simulation Notes : 
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//////////////////////////////////////////////////////////////////////////////

module rw_nx_la 
            #(
              parameter LA_MEMORY_SIZE  = 16 ,
              parameter LA_MEMORY_WIDTH = 128) // 64 or 128bits
            ( 
            ///////////////////////////////////////////////
            //$port_g  AHB clock domain
            ///////////////////////////////////////////////
            input  wire                        ahb_rst_n,
            input  wire                        ahb_clk,
            
            ///////////////////////////////////////////////
            //$port_g  LA clock domain
            ///////////////////////////////////////////////
            input  wire                        la_rst_n,
            input  wire                        la_clk,
            
            ///////////////////////////////////////////////
            //$port_g  AHB interface
            ///////////////////////////////////////////////
            input  wire                        hready_in,
            input  wire                        hsel,
            input  wire                 [20:0] haddr,
            input  wire                 [1:0]  htrans,
            input  wire                        hwrite,
            input  wire                 [31:0] hwdata,
            output wire                 [31:0] hrdata,
            output wire                  [1:0] hresp,
            output wire                        hready,

            ///////////////////////////////////////////////
            //$port_g  Interrupt interface
            ///////////////////////////////////////////////
            output reg                         la_interrupt,
            
            ///////////////////////////////////////////////
            //$port_g   memory port
            ///////////////////////////////////////////////
            output wire                        clka,
            output reg                         ena,
            output reg                         wea,
            output reg   [LA_MEMORY_SIZE -1:0] addra,
            output reg   [LA_MEMORY_WIDTH-1:0] dina,
            
            output wire                        clkb,
            output wire                        enb,
            output wire  [LA_MEMORY_SIZE -1:0] addrb,
            input  wire  [LA_MEMORY_WIDTH-1:0] doutb,
            
            ///////////////////////////////////////////////
            //$port_g  input sampling port
            ///////////////////////////////////////////////
            input  wire                  [7:0] samplingfreq,
            input  wire [LA_MEMORY_WIDTH-17:0] input_data,
            input  wire                  [3:0] ext_trigger,
            output reg                         triggered
            
);

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

localparam [15:0] MAX_SAMPLE_DEPTH = 16'd2**LA_MEMORY_SIZE-16'd1;
localparam  [7:0] SAMPLE_WIDTH     = LA_MEMORY_WIDTH;
//localparam MAX_SAMPLE_DEPTH = 16'd1023; 

localparam [31:0] LA_ID  = 32'h52574c41; 

localparam VERSION_MAJOR =  8'd5;
localparam VERSION_MINOR = 16'd0;

// Data sample width
localparam DATA_WIDTH = LA_MEMORY_WIDTH-16;

// Constants for register addresses.
localparam [ 7:0] RW_LA_ID_LOW_ADDR_CT             = 8'h0,
                  RW_LA_ID_HIGH_ADDR_CT            = 8'h4,
                  RW_LA_VERSION_ADDR_CT            = 8'h8,
                  RW_LA_CNTRL_ADDR_CT              = 8'hc,
                  RW_LA_STATUS_ADDR_CT             = 8'h10,
                  RW_LA_SAMPLING_MASK0_ADDR_CT     = 8'h14,
                  RW_LA_SAMPLING_MASK1_ADDR_CT     = 8'h18,
                  RW_LA_SAMPLING_MASK2_ADDR_CT     = 8'h1c,
                  RW_LA_SAMPLING_MASK3_ADDR_CT     = 8'h40,
                  RW_LA_TRIGGER_MASK0_ADDR_CT      = 8'h20,
                  RW_LA_TRIGGER_MASK1_ADDR_CT      = 8'h24,
                  RW_LA_TRIGGER_MASK2_ADDR_CT      = 8'h28,
                  RW_LA_TRIGGER_MASK3_ADDR_CT      = 8'h44,
                  RW_LA_TRIGGER_VALUE0_ADDR_CT     = 8'h2c,
                  RW_LA_TRIGGER_VALUE1_ADDR_CT     = 8'h30,
                  RW_LA_TRIGGER_VALUE2_ADDR_CT     = 8'h34,
                  RW_LA_TRIGGER_VALUE3_ADDR_CT     = 8'h48,
                  RW_LA_TRIGGER_POINT_ADDR_CT      = 8'h38,
                  RW_LA_FIRSTSAMPLE_ADDR_CT        = 8'h3c;

localparam [127:0] ZERO_CT = 128'd0;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
reg  [31: 0] int_reg_dr;
wire [31: 0] int_reg_dw;

reg  master_soft_reset;
wire master_soft_resetLAClk;

// control signals
wire               [31: 0] version;           // version bit
reg                [31: 0] status;            // status byte  
reg                        triggered_ff1;     // trigger delayed
wire                       triggered_p;       // trigger pulse
reg  [LA_MEMORY_SIZE-1: 0] triggered_address; // current address at the trigger
wire                       start_p;           // sampling start pulse
wire                       stop_p;            // stop sampling start pulse
reg                        sample_end;        // indicate enough sample after trigger
reg                        started;           // sampling start flag
reg                        full;              // memory full
reg      [DATA_WIDTH-1: 0] sampling_mask;     // sampling mask
reg      [DATA_WIDTH-1: 0] trigger_mask;      // trigger mask
reg      [DATA_WIDTH-1: 0] trigger_value;     // trigger value
reg  [LA_MEMORY_SIZE-1: 0] trigger_point;     // trigger point

// sampling signals
reg                [23: 0] time_cnt;          // time references
wire               [23: 0] n_time_cnt;        // time references
reg  [LA_MEMORY_SIZE-1: 0] write_address;     // current dump address
//reg   [9: 0] write_address;                 // current dump address
reg [LA_MEMORY_WIDTH-1: 0] write_data;        // data to mem
reg                        write_enable;      // enable the memory write
reg                        sample;            // sample trigger
reg                        sample_time;       // sample is due to time_cnt wrap
reg                        sample_type;       // sample type:
                                              // 0: Trigger Status, time offset, Data
                                              // 1: time offset only
reg      [DATA_WIDTH-1: 0] last_data;         // time references
wire     [DATA_WIDTH-1: 0] match_vector;      // compare the input data vs trigger value and mask

// interface signals

reg   [DATA_WIDTH-1:0] input_data_ff1;
reg   [DATA_WIDTH-1:0] input_data_ff2;

// interrupt generation signals
reg trigger_int_en;    // pulse to enable the la_interrupt generation
reg trigger_int_clear; // pulse to clear the la_interrupt
reg trigger_int_set;   // pulse to genrate a la_interrupt
reg triggeredHClk_ff1; // triggeredHClk delayed of 1cc


wire startedHClk;
wire triggeredHClk;
wire fullHClk;
reg  stopHClk_p;
reg  startHClk_p;
reg  [LA_MEMORY_SIZE-1:0] write_address_ff1;
reg  [LA_MEMORY_SIZE-1:0] write_address_ff2;
reg  [LA_MEMORY_SIZE-1:0] write_address_dly;

wire [3:0]   ext_triggerLAClk;
wire [3:0]   ext_trigger_enLAClk;
reg  [3:0]   ext_trigger_en;
  
wire [20:0]  regbus_addr;
reg  [20:0]  regbus_addr_ff1;
wire         regbus_rden;
wire         regbus_wren;

wire [31:0]  regbus_wdata;
reg  [31:0]  regbus_rdata;
 
reg  [5:0]   trigger_status;
//reg  [5:0]   trigger_status_ff1;
wire [5:0]   trigger_statusHClk;
reg  [5:0]   last_trigger_status;
reg          sw_trigger;
wire         sw_triggerLAClk;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////


// assign memory
assign  clka = la_clk;
always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  begin
    dina  <= ZERO_CT[LA_MEMORY_WIDTH-1:0];
    addra <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    ena   <= 1'b0;
    wea   <= 1'b0;
  end
  else if (master_soft_resetLAClk == 1'b1)
  begin
    dina  <= ZERO_CT[LA_MEMORY_WIDTH-1:0];
    addra <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    ena   <= 1'b0;
    wea   <= 1'b0;
  end
  else 
  begin
    dina  <= write_data; 
    addra <= write_address_dly; 
    ena   <= write_enable;
    wea   <= write_enable; 
  end
end    

assign  clkb = ahb_clk;
assign addrb = (LA_MEMORY_WIDTH<=64) ? regbus_addr[18:3] : regbus_addr[19:4];
assign   enb = regbus_rden & regbus_addr[20];


/////////////////////////////////////    
// data sampling     
/////////////////////////////////////    

assign version = {samplingfreq,VERSION_MAJOR,VERSION_MINOR};

assign match_vector = (input_data_ff2 ^ trigger_value ) & trigger_mask;

// free running time counter
always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  begin
    sample_end <= 1'b0;
  end
  else 
  begin
    if ((triggered_ff1 == 1'b1) && (write_address == (triggered_address + trigger_point)))
      sample_end <= 1'b1;
    else
      sample_end <= 1'b0;
  end
end  

// Input resynchronization
always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  begin
    input_data_ff1 <= ZERO_CT[DATA_WIDTH-1:0];
    input_data_ff2 <= ZERO_CT[DATA_WIDTH-1:0];
  end
  else 
  begin
    input_data_ff1 <= input_data;
    input_data_ff2 <= input_data_ff1;
  end
end  

// free running time counter
assign n_time_cnt[23:20] = time_cnt[23:20]+{3'd0,&time_cnt[19:0]};
assign n_time_cnt[19:16] = time_cnt[19:16]+{3'd0,&time_cnt[15:0]};
assign n_time_cnt[15:12] = time_cnt[15:12]+{3'd0,&time_cnt[11:0]};
assign n_time_cnt[11: 8] = time_cnt[11: 8]+{3'd0,&time_cnt[7:0]}; 
assign n_time_cnt[ 7: 4] = time_cnt[ 7: 4]+{3'd0,&time_cnt[3:0]}; 
assign n_time_cnt[ 3: 0] = time_cnt[ 3: 0]+{3'd0,1'b1};               

always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  begin
    time_cnt <=   24'b0;
  end
  else if (master_soft_resetLAClk == 1'b1)
  begin
    time_cnt <=   24'b0;
  end
  else 
  begin
    // reset the value to 2. 1 cycle since the assertion of the data
    // + 1 cycle to write to RAM
    if (sample == 1'b1)
    begin
      time_cnt <= 24'd1;
    end
    else if (started == 1'b1)
    begin
      time_cnt <= n_time_cnt;
    end
  end
end

// trigger status
always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  begin
    triggered         <= 1'b0;
    triggered_ff1     <= 1'b0;
    triggered_address <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    trigger_status    <= 6'b0;
    //trigger_status_ff1 <= 6'b0;
  end
  else if (master_soft_resetLAClk == 1'b1)
  begin
    triggered         <= 1'b0;
    triggered_ff1     <= 1'b0;
    triggered_address <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    trigger_status    <= 6'b0;
    //trigger_status_ff1 <= 6'b0;
  end
  else 
  begin
    // sample the trigger flag to detect the edge
    triggered_ff1 <= triggered;
    //trigger_status_ff1 <= trigger_status;
    
    // reset at the start
    if (start_p == 1'b1)
    begin
      triggered         <= 1'b0;
      trigger_status    <= 6'b0;
    end
    // when the match vector is null, this means trigger if trigger mask is not null (meaning free run)
    else if ((started      == 1'b1                   ) && 
             (match_vector == ZERO_CT[DATA_WIDTH-1:0]) &&
             (trigger_mask != ZERO_CT[DATA_WIDTH-1:0]))
    begin
      triggered         <= 1'b1;
      trigger_status[0] <= 1'b1;
    end
    // when the sw_trigger is set
    else if (started && sw_triggerLAClk)
    begin
      triggered           <= 1'b1;
      trigger_status[1]   <= 1'b1;
    end
    // when one of the ext_trigger input is set
    else if (started && ((ext_triggerLAClk & ext_trigger_enLAClk) != 4'b0))
    begin
      triggered           <= 1'b1;
    end

    // As the ext_trigger input can be a pulse, always update the status of external trigger
    if (started)
    begin
      trigger_status[5:2] <= ext_triggerLAClk & ext_trigger_enLAClk;
    end
    
    if(start_p == 1'b1)
    begin
      triggered_address <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    end
    else if (triggered_p == 1'b1)
    begin
      triggered_address <= write_address;
    end
    
  end
end

assign triggered_p = triggered & !triggered_ff1;

// manage the sample signal which trigger data write to the RAM
always@*
begin
  if(start_p == 1'b1)
  begin
    sample      = 1'b1;
    sample_time = 1'b0;
  end
  else if (started == 1'b1)
  begin
    if ((last_data & sampling_mask) != (input_data_ff2 & sampling_mask))
    begin
      sample      = 1'b1;
      sample_time = 1'b0;
    end
    else if (last_trigger_status != trigger_status)
    begin
      sample      = 1'b1;
      sample_time = 1'b0;
    end
    else if (~sample_type & time_cnt[ 8:0]=={ 9{1'b1}} | // Data time offset wrap
              sample_type & time_cnt[23:0]=={24{1'b1}} ) // Time only offset wrap
    begin
      sample      = 1'b1;
      sample_time = 1'b1;
    end
    else
    begin
      sample      = 1'b0;
      sample_time = 1'b0;
    end
  end
  else
  begin
    sample      = 1'b0;
    sample_time = 1'b0;
  end
end



// write data management
always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  // reset
  begin
    last_data           <= ZERO_CT[DATA_WIDTH-1:0];
    last_trigger_status <= 6'b0;
    started             <= 1'b0;
  end
  else if (master_soft_resetLAClk == 1'b1)
  begin
    last_data           <= ZERO_CT[DATA_WIDTH-1:0];
    last_trigger_status <= 6'b0;
    started             <= 1'b0;
  end
  else 
  begin
  
    // start pulse, sample and write data
    if(start_p == 1'b1)
    begin
      last_data           <= input_data_ff2;
      last_trigger_status <= 6'b0;
      started             <= 1'b1;
    end
    else if (sample_end || stop_p)
    begin
      started             <= 1'b0;
    end
    else if (started == 1'b1)
    begin
      //data has changed, initiate sampling
      if(sample == 1'b1)
      begin
        last_data           <= input_data_ff2;
        last_trigger_status <= trigger_status;
      end
    end
  end
end

// write data management
always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
  begin
    full              <= 1'b0;
    write_address     <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    write_address_dly <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    write_data        <= ZERO_CT[LA_MEMORY_WIDTH-1:0];
    write_enable      <= 1'b0;
  end
  else if (master_soft_resetLAClk == 1'b1)
  begin
    full              <= 1'b0;
    write_address     <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    write_address_dly <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    write_data        <= ZERO_CT[LA_MEMORY_WIDTH-1:0];
    write_enable      <= 1'b0;
  end
  else
  begin
    // manage the full bit
    if(start_p == 1'b1)
    begin
      full <= 1'b0;
    end
    else if (write_address == {LA_MEMORY_SIZE{1'b1}})
    begin
      full <= 1'b1;
    end
    
    
    // manage the write enable
    if(started && sample)
    begin
      write_enable  <= 1'b1;
      write_address_dly <= write_address;
    
    end
    else
    begin
      write_enable  <= 1'b0;
    end
    
    // sample data
    if(start_p == 1'b1)
    begin
//      write_address <= {LA_MEMORY_SIZE{1'b1}};
      write_address <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    end

    if(started == 1'b1)
    begin
      if (sample)
      begin
        write_address <= write_address + {{(LA_MEMORY_SIZE-1){1'b0}},1'b1};
        if (~sample_type)
          // Data and time sample
          write_data    <= {sample_type,last_trigger_status,time_cnt[8:0],last_data};
        else
          // Time only sample
          write_data    <= {sample_type,ZERO_CT[LA_MEMORY_WIDTH-1-25:0],time_cnt[23:0]};
      end
    end

  end
end

always @(posedge la_clk or negedge la_rst_n)
begin
  if (la_rst_n == 1'b0)
    sample_type <= 1'b0;
  else if (master_soft_resetLAClk == 1'b1)
    sample_type <= 1'b0;
  else if (sample & sample_time)
    // When sample is due to timer wrap, the next sample is time only.
    sample_type <= 1'b1;
  else if (sample)
    // When sample is not due to timer wrap, the next sample is data.
    sample_type <= 1'b0;
end


pulse2PulseSynchro u_start_p_synchro (
                .srcclk       (ahb_clk),
                .srcresetn    (ahb_rst_n),
                .dstclk       (la_clk),
                .dstresetn    (la_rst_n),
                .srcdata      (startHClk_p),
                .dstdata      (start_p)
                );

pulse2PulseSynchro u_stop_p_synchro (
                .srcclk       (ahb_clk),
                .srcresetn    (ahb_rst_n),
                .dstclk       (la_clk),
                .dstresetn    (la_rst_n),
                .srcdata      (stopHClk_p),
                .dstdata      (stop_p)
                );


simpleSynchro u_full_synchro (
                .dstclk             (ahb_clk),
                .dstresetn          (ahb_rst_n),
                .srcdata            (full),
                .dstdata            (fullHClk)
                );



simpleSynchro u_triggered_synchro (
                .dstclk             (ahb_clk),
                .dstresetn          (ahb_rst_n),
                .srcdata            (triggered),
                .dstdata            (triggeredHClk)
                );



simpleSynchro u_started_synchro (
                .dstclk             (ahb_clk),
                .dstresetn          (ahb_rst_n),
                .srcdata            (started),
                .dstdata            (startedHClk)
                );


onehotSynchro #(.SYNC_WIDTH (4)) u_ext_trigger_synchro (
                .dstclk             (la_clk),
                .dstresetn          (la_rst_n),
                .srcdata            (ext_trigger),
                .dstdata            (ext_triggerLAClk)
                );

onehotSynchro #(.SYNC_WIDTH (4)) u_ext_trigger_en_synchro (
                .dstclk             (la_clk),
                .dstresetn          (la_rst_n),
                .srcdata            (ext_trigger_en),
                .dstdata            (ext_trigger_enLAClk)
                );


simpleSynchro u_master_soft_reset_synchro (
                .dstclk             (la_clk),
                .dstresetn          (la_rst_n),
                .srcdata            (master_soft_reset),
                .dstdata            (master_soft_resetLAClk)
                );

simpleSynchro u_sw_trigger_synchro (
                .dstclk             (la_clk),
                .dstresetn          (la_rst_n),
                .srcdata            (sw_trigger),
                .dstdata            (sw_triggerLAClk)
                );

onehotSynchro #(.SYNC_WIDTH (6)) u_trigger_status_synchro (
                .dstclk             (ahb_clk),
                .dstresetn          (ahb_rst_n),
                .srcdata            (trigger_status),
                .dstdata            (trigger_statusHClk)
                );


// status vector
always @(posedge ahb_clk or negedge ahb_rst_n)
begin
  if (!ahb_rst_n) 
  begin
    status <= 32'b0;
    write_address_ff1 <= ZERO_CT[LA_MEMORY_SIZE-1:0];
    write_address_ff2 <= ZERO_CT[LA_MEMORY_SIZE-1:0];
  end
  else
  begin
    write_address_ff1 <= write_address_dly;
    write_address_ff2 <= write_address_ff1;
    // when full report the memory depth
    if (fullHClk == 1'b1)
    begin
      status  <= {8'b0,MAX_SAMPLE_DEPTH,trigger_statusHClk,triggeredHClk,startedHClk};
    end
    // report the sample count
    else
    begin
      status  <= {8'b0,write_address_ff2,trigger_statusHClk,triggeredHClk,startedHClk};
    end
  end  
end

//////////////////////////////////////////////////////////////////////////////
// Interrupt Generation
//////////////////////////////////////////////////////////////////////////////
// triggeredHClk rising edge detection
always @(posedge ahb_clk or negedge ahb_rst_n)
begin
  if (!ahb_rst_n) 
    triggeredHClk_ff1    <= 1'b0;
  else
    triggeredHClk_ff1 <= triggeredHClk;
end

always @(posedge ahb_clk or negedge ahb_rst_n)
begin
  if (!ahb_rst_n) 
    la_interrupt    <= 1'b0;
  else if ((triggeredHClk && !triggeredHClk_ff1) || trigger_int_set)
    la_interrupt <= trigger_int_en;
  else if (trigger_int_clear)
    la_interrupt <= 1'b0;
end
 

/////////////////////////////////////    
// Register Interface
/////////////////////////////////////    

  assign  int_reg_dw = regbus_wdata;
  
  //////////////////////////////////////////////////////////////////////////////
  // Register write
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge ahb_clk or negedge ahb_rst_n)
  begin:AHB_WRITE
    // local variable
    integer v_i;

    if (!ahb_rst_n) 
    begin
      startHClk_p       <= 1'b0;
      stopHClk_p        <= 1'b0;
      master_soft_reset <= 1'b0;
      sw_trigger        <= 1'b0;
      
      sampling_mask     <= ZERO_CT[DATA_WIDTH-1:0];
      trigger_mask      <= ZERO_CT[DATA_WIDTH-1:0];
      trigger_value     <= ZERO_CT[DATA_WIDTH-1:0];
      trigger_point     <= ZERO_CT[LA_MEMORY_SIZE-1:0];

      trigger_int_set   <= 1'b0;
      trigger_int_clear <= 1'b0;
      trigger_int_en    <= 1'b0;
      ext_trigger_en    <= 4'b1111;

    end
    else
    begin
      stopHClk_p   <= 1'b0;

      startHClk_p  <= 1'b0;

      trigger_int_set   <= 1'b0;
      trigger_int_clear <= 1'b0;

      if (triggeredHClk)
        sw_trigger      <= 1'b0;
      
      if (!regbus_addr[20] && regbus_wren)
      begin
        case(regbus_addr[7:0])
          RW_LA_CNTRL_ADDR_CT : 
          begin
            trigger_int_en    <= int_reg_dw[14];
            trigger_int_clear <= int_reg_dw[13];
            trigger_int_set   <= int_reg_dw[12];
            ext_trigger_en    <= int_reg_dw[7:4];
            sw_trigger        <= int_reg_dw[3];
            master_soft_reset <= int_reg_dw[2];
            stopHClk_p        <= int_reg_dw[1];
            startHClk_p       <= int_reg_dw[0];
          end
          RW_LA_SAMPLING_MASK0_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+0*32)<DATA_WIDTH) sampling_mask[0*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_SAMPLING_MASK1_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+1*32)<DATA_WIDTH) sampling_mask[1*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_SAMPLING_MASK2_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+2*32)<DATA_WIDTH) sampling_mask[2*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_SAMPLING_MASK3_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+3*32)<DATA_WIDTH) sampling_mask[3*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_MASK0_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+0*32)<DATA_WIDTH) trigger_mask[0*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_MASK1_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+1*32)<DATA_WIDTH) trigger_mask[1*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_MASK2_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+2*32)<DATA_WIDTH) trigger_mask[2*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_MASK3_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+3*32)<DATA_WIDTH) trigger_mask[3*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_VALUE0_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+0*32)<DATA_WIDTH) trigger_value[0*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_VALUE1_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+1*32)<DATA_WIDTH) trigger_value[1*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_VALUE2_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+2*32)<DATA_WIDTH) trigger_value[2*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_VALUE3_ADDR_CT : 
          begin
              for(v_i=0;v_i<=31;v_i=v_i+1)
                 if((v_i+3*32)<DATA_WIDTH) trigger_value[3*32+v_i] <= int_reg_dw[v_i];
          end
          RW_LA_TRIGGER_POINT_ADDR_CT : 
          begin
              trigger_point     <= int_reg_dw[LA_MEMORY_SIZE-1:0];
          end
          // Disable coverage on the default state because it cannot be reached.
          default : begin
          end

        endcase
      end
    end
  end

  //////////////////////////////////////////////////////////////////////////////
  // Registers read
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge ahb_clk or negedge ahb_rst_n)
  begin:AHB_READ
    // local variable
    integer v_i;

    if (!ahb_rst_n) 
    begin
      int_reg_dr     <= 32'd0;

    end
    else if (regbus_rden && !regbus_addr[20])
    begin
      int_reg_dr <= 32'b0;
      case (regbus_addr[7:0])
        // Read SIGNATURE register.
        RW_LA_ID_LOW_ADDR_CT :
        begin
          int_reg_dr  <= LA_ID;
        end

        RW_LA_ID_HIGH_ADDR_CT :
        begin
          int_reg_dr  <= {8'b0,SAMPLE_WIDTH,MAX_SAMPLE_DEPTH};
        end

        RW_LA_VERSION_ADDR_CT :
        begin
          int_reg_dr  <= version;
        end
        RW_LA_CNTRL_ADDR_CT :
        begin
          int_reg_dr  <= {16'b0,1'b0,trigger_int_en,1'b0,la_interrupt,4'b0,ext_trigger_en,sw_trigger,master_soft_reset,2'b0};
        end

        RW_LA_STATUS_ADDR_CT :
        begin
          int_reg_dr  <= status;
        end

        RW_LA_SAMPLING_MASK0_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+0*32)<DATA_WIDTH) int_reg_dr[v_i] <= sampling_mask[0*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_SAMPLING_MASK1_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+1*32)<DATA_WIDTH) int_reg_dr[v_i] <= sampling_mask[1*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_SAMPLING_MASK2_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+2*32)<DATA_WIDTH) int_reg_dr[v_i] <= sampling_mask[2*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_SAMPLING_MASK3_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+3*32)<DATA_WIDTH) int_reg_dr[v_i] <= sampling_mask[3*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_MASK0_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+0*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_mask[0*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_MASK1_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+1*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_mask[1*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_MASK2_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+2*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_mask[2*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_MASK3_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+3*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_mask[3*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_VALUE0_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+0*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_value[0*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_VALUE1_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+1*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_value[1*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_VALUE2_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+2*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_value[2*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_VALUE3_ADDR_CT :
        begin
          for(v_i=0;v_i<=31;v_i=v_i+1)
             if((v_i+3*32)<DATA_WIDTH) int_reg_dr[v_i] <= trigger_value[3*32+v_i];
             else                      int_reg_dr[v_i] <= 1'b0;
        end

        RW_LA_TRIGGER_POINT_ADDR_CT :
        begin
          int_reg_dr[LA_MEMORY_SIZE-1:0]  <= trigger_point;
        end

        RW_LA_FIRSTSAMPLE_ADDR_CT :
        begin
          if (fullHClk)
          begin
            int_reg_dr[LA_MEMORY_SIZE-1:0]  <= write_address;
          end
          else  
            int_reg_dr  <= 32'd0;
        end



        // Disable coverage on the default state because it cannot be reached.
        default : int_reg_dr <= 32'b0;
      endcase
    end
    else
      int_reg_dr <= 32'b0;
  end

  //////////////////////////////////////////////////////////////////////////////
  // Data read
  //////////////////////////////////////////////////////////////////////////////
generate if (LA_MEMORY_WIDTH==64)
begin:REGBUS_RDATA
  always@(*)
  begin
    if (regbus_addr_ff1[20])
      case (regbus_addr_ff1[2])
      1'b0 : regbus_rdata = doutb[31:0];
      1'b1 : regbus_rdata = doutb[63:32];
      endcase  
    else      
      regbus_rdata = int_reg_dr;
  end
end
else // (LA_MEMORY_WIDTH!=64)
begin:REGBUS_RDATA
  always@(*)
  begin
    if (regbus_addr_ff1[20])
      case (regbus_addr_ff1[3:2])
      2'b00 : regbus_rdata = doutb[31:0];
      2'b01 : regbus_rdata = doutb[63:32];
      2'b10 : regbus_rdata = doutb[95:64];
      2'b11 : regbus_rdata = doutb[127:96];
      endcase  
    else      
      regbus_rdata = int_reg_dr;
  end
end
endgenerate

 always @(posedge ahb_clk or negedge ahb_rst_n)
  begin
    if (!ahb_rst_n) 
      regbus_addr_ff1 <= 21'd0;
    else if (regbus_rden)
      regbus_addr_ff1 <= regbus_addr;
  end

assign hresp = 2'b0;
// Instanciation of rw_nx_la_ahbif
// Name of the instance : u_rw_nx_la_ahbif
// Name of the file containing this module : rw_nx_la_ahbif.v
rw_nx_la_ahbif u_rw_nx_la_ahbif (
		.clk                              (ahb_clk),
		.rst_n                            (ahb_rst_n),
		.hready_in                        (hready_in),
		.hsel                             (hsel),
		.haddr                            (haddr),
		.htrans                           (htrans),
		.hwrite                           (hwrite),
		.hwdata                           (hwdata),
		.hrdata                           (hrdata),
		.hready                           (hready),
		.regbus_addr                      (regbus_addr),
		.regbus_rden                      (regbus_rden),
		.regbus_rdspec                    (),
		.regbus_wren                      (regbus_wren),
		.regbus_wdata                     (regbus_wdata),
		.regbus_rdata                     (regbus_rdata),
		.regbus_rdspec_invalid            (1'b0)
		);



              
endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
