module rw_nx_la_ahbif
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire          clk,
  input wire          rst_n,
 
  /*****************************************************************************
  * AHB slave
  *****************************************************************************/
  input  wire         hready_in,
  input  wire         hsel,
  input  wire [20:0]  haddr,
  input  wire [ 1:0]  htrans,
  input  wire         hwrite,
  
  input  wire [31:0]  hwdata,
  output reg  [31:0]  hrdata,
  output reg          hready,
 
  /*****************************************************************************
  * register bus
  *****************************************************************************/
  output wire [20:0]  regbus_addr,
  output wire         regbus_rden,
  output wire         regbus_rdspec,
  output wire         regbus_wren,

  output reg  [31:0]  regbus_wdata,
  input  wire [31:0]  regbus_rdata,
  input  wire         regbus_rdspec_invalid
);


  /*******************************************************************************
  * declaration
  *******************************************************************************/
  /* speculation state */
  reg         speculate;
  wire        n_speculate_set,n_speculate_clr;
  wire        hit,speculate_fail;
  
  /* q0 entry */
  reg [20:0]  q0_addr,n_q0_addr;
  reg         q0_write,n_q0_write;
  reg         q0_spec,n_q0_spec;
  reg         q0_valid,n_q0_valid;
  /* q1 entry */
  reg [20:0]  q1_addr,n_q1_addr;
  reg         q1_write,n_q1_write;
  reg         q1_spec,n_q1_spec;
  reg         q1_valid,n_q1_valid;
  /* qd entry */
  reg [20:0]  qd_addr,n_qd_addr;
  reg         qd_spec,n_qd_spec;
  reg         qd_valid,n_qd_valid;
  
  /* ahb request  */
  wire        req_ahb_valid;
  wire [20:0] req_ahb_addr;
  wire        req_ahb_write;

  /* speculated request  */
  wire [20:0] spec_addr;
  wire [20:0] n_curr_addr;
  reg  [20:0] curr_addr;
  
  /* queue request  */
  wire        req_mux;
  wire        req_valid;
  wire        req_spec;
  wire [20:0] req_addr;
  wire        req_write;
  
  /* hready state */
  wire        n_hready_set,n_hready_clr;
  
  /*******************************************************************************
  * o About request data width
  *  
  *   The interface only supports access of 32 bits. Others access data widths are
  *   treated as 32 bits.
  *
  * o AHB master BUSY are indeed supported
  * 
  * o About post-write
  *  
  *   All write accesses are posted and do not cause any wait state.
  *
  * o About speculation
  *
  *   Because we re-register address,data and also because register banks have a read
  *   back taking one cycle, a AHB read request would take several clock cycles to 
  *   complete. 
  *
  *   Then we avoid these wait states for each read request by speculating that all
  *   all read requests belong to a burst, and hence we continue to perform read 
  *   access towards banks while the AHB is stalled.
  *  
  * o About 'regbus_rdspec' and 'regbus_rdspec_invalid'
  *  
  *   The speculation WORKS ONLY IF the agents behind the bridge are SRAM or 
  *   registers banks without side effect: 
  *   (ie: a read access can't clear/set some flags or trigger any operation).
  *  
  *   => If the agent does not support speculation, then it has to ignore the read
  *   access if 'regbus_rdspec' is asserted, but it must inform the bridge that
  *   the speculative read is refused by asserting the 'regbus_rdspec_invalid'
  *   during the data cycle.
  *
  * o About regbus timing
  *  
  *   The bridge assumes that the timing of regbus corresponds to a SRAM timing
  *   1) write access occurs in the cycle where addr,wen and wdata are asserted.
  *   2) read data is returned the cycle following the addr and rden assertion.
  *      => if your register bank returns the data during the cycle of the rden
  *         assertion, then just re-register regbank rdata bus to this interface.
  * 
  *******************************************************************************/
  
  /*******************************************************************************
  * AHB
  *******************************************************************************/
  /* hit if qd entry is consistent with current ahb address cycle */
  assign hit  = hsel & ~hwrite & htrans[1] & haddr==qd_addr & ~regbus_rdspec_invalid;
  
  /* generate speculation state */
  assign speculate_fail  = speculate & hready_in & qd_valid & ~hit;
  assign n_speculate_set = hready_in & hsel & htrans[1] & ~hwrite;
  assign n_speculate_clr = speculate_fail;
  
  /* request from AHB */
  assign {req_ahb_valid, req_ahb_addr,req_ahb_write} = {hready_in & htrans[1] & hsel, haddr[20:0], hwrite};
  
  /* request to queue */
  assign req_mux = ~speculate | speculate_fail;
  
  /* address for speculation */
  assign n_curr_addr=({21{req_mux}} & haddr | {21{~req_mux}} & spec_addr);
  assign spec_addr[20:17] = curr_addr[20:17] + {3'd0,&curr_addr[16:2]};
  assign spec_addr[16:13] = curr_addr[16:13] + {3'd0,&curr_addr[12:2]};
  assign spec_addr[12:9]  = curr_addr[12:9]  + {3'd0,&curr_addr[8:2]};
  assign spec_addr[8:5]   = curr_addr[8:5]   + {3'd0,&curr_addr[4:2]};
  assign spec_addr[4:2]   = curr_addr[4:2]   + 3'd1;
  assign spec_addr[1:0]   = curr_addr[1:0];
 
  assign {req_valid,req_addr,req_write,req_spec} =
      {24{ req_mux}} & { req_ahb_valid, req_ahb_addr, req_ahb_write, 1'b0} |
      {24{~req_mux}} & {          1'b1,    spec_addr,          1'b0, 1'b1};                         

  /* hready goes high if we return a non speculated data or if the returned speculated data
  *  is confirmed.
  * hready goes *by default* to low if a read access is performed
  *
  * indeed next_hready_set supersedes next_hready_low if a wanted data is returned to ahb.
  */
  assign n_hready_set = qd_valid & (~qd_spec | qd_spec & hit);
  assign n_hready_clr = hready_in & htrans[1] & hsel & ~hwrite;

  /* queue */
  always @(*)
  begin
  
    /* qd */
    if(q0_valid & ~q0_write)
      {n_qd_valid,n_qd_addr,n_qd_spec} = {1'b1,q0_addr,q0_spec};
    else
      {n_qd_valid,n_qd_addr,n_qd_spec} = 23'b0;
    
    /* q0,q1 */
    if(q1_valid)
    begin
      {n_q0_valid,n_q0_addr,n_q0_write,n_q0_spec} = {q1_valid,q1_addr,q1_write,q1_spec};
      {n_q1_valid,n_q1_addr,n_q1_write,n_q1_spec} = {req_valid,req_addr,req_write,req_spec};
    end
    else
    begin
      if(~req_write)
      begin
        {n_q0_valid,n_q0_addr,n_q0_write,n_q0_spec} = {req_valid,req_addr,req_write,req_spec};
        {n_q1_valid,n_q1_addr,n_q1_write,n_q1_spec} = 24'b0;
      end
      else
      begin
        {n_q0_valid,n_q0_addr,n_q0_write,n_q0_spec} = 24'b0;
        {n_q1_valid,n_q1_addr,n_q1_write,n_q1_spec} = {req_valid,req_addr,req_write,req_spec};
      end
    end
    
    /* invalidate */
    if(speculate_fail)
    begin
      if(n_qd_spec) {n_qd_valid,n_qd_spec} = 2'b0;
      if(n_q0_spec) {n_q0_valid,n_q0_spec} = 2'b0;
      if(n_q1_spec) {n_q1_valid,n_q1_spec} = 2'b0;
    end
  
  end
  
  /* registers */
  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
     
      speculate                           <= 1'b0;
      curr_addr                           <= 21'd0;
      hready                              <= 1'b1;
      hrdata                              <= 32'b0; 
      regbus_wdata                        <= 32'b0;
      
      {q0_valid,q0_addr,q0_write,q0_spec} <= 24'b0;
      {q1_valid,q1_addr,q1_write,q1_spec} <= 24'b0;
      {qd_valid,qd_addr,qd_spec}          <= 23'b0;  
   
    end
    else
    begin
      
      speculate                           <= n_speculate_set | speculate & ~ n_speculate_clr;
      curr_addr                           <= n_curr_addr;
      hready                              <= n_hready_set | hready & ~n_hready_clr;
      hrdata                              <= regbus_rdata;
      regbus_wdata                        <= hwdata;
      
      {q0_valid,q0_addr,q0_write,q0_spec} <= {n_q0_valid,n_q0_addr,n_q0_write,n_q0_spec};
      {q1_valid,q1_addr,q1_write,q1_spec} <= {n_q1_valid,n_q1_addr,n_q1_write,n_q1_spec};
      {qd_valid,qd_addr,qd_spec}          <= {n_qd_valid,n_qd_addr,n_qd_spec};
      
    end
  
  /* regbus interface */
  assign regbus_addr   = q0_addr;                  
  assign regbus_wren   = q0_valid &  q0_write;     
  assign regbus_rden   = q0_valid & ~q0_write;     
  assign regbus_rdspec = q0_valid & ~q0_write & q0_spec;     

endmodule
`default_nettype wire
