/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author:$
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :             
* Description      : rw_nx_he_macbypass
* Simulation Notes :             
* Synthesis Notes  :              
* Application Note :              
* Simulator        :             
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*           
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module rw_nx_macbypass_fsm
( 
  /* system */
  input  wire         rst_n, 
  input  wire         clk,  

  /* mac mpif */
  input  wire         mpifi_rxreq,
  
  /* phy mpif */
  output reg          rxreq,
  output reg          txreq,
  input  wire         mpifo_phyrdy,   
  input  wire         mpifo_rxendfortiming,
  input  wire         mpifo_rxend,
  input  wire         mpifo_rxerr,
  input  wire         mpifo_phyerr,
  input  wire         mpifo_txend,
  
  /* regb */
  input  wire         regb_bypass,
  input  wire         regb_mode_is_rxlisten,
  input  wire         regb_mode_is_txsingle,
  input  wire         regb_mode_is_txburst,
  input  wire         regb_defrxreq,
  input  wire         regb_defkeeprfon,
  input  wire  [19:0] regb_interframe_delay,
  input  wire  [15:0] regb_frameperburst,
  
  /* rxv */
  input  wire         rxvu_ndp,
  input  wire         rxvu1_last,
  input  wire         rxvu2_last,
  
  /* payload tracking */
  output reg          pldtrck_enable,
  input  wire         pldtrck_last,
  
  /* txv */
  input  wire         txv_last,
  
  /* payload generator */
  output reg          pldgen_enable,
  input  wire         pldgen_done,
  
  /* event signaling */
  output reg          event_tx_phyerr,
  output reg          event_tx_eof,
  output reg          event_rx_eof_abort,
  output reg          event_rx_eof_normal,
  output reg          event_rx_rxerr,
  output reg          event_rx_rxend,
  output wire         event_rxv_unsupported,
  output wire         event_rxv_supported
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam   IDLE=4'd0,
               RX_LISTEN=4'd1,
               RX_RECEIVE_RXV1=4'd2,
               RX_RECEIVE_DATA=4'd4,
               RX_RECEIVE_RXV2=4'd5,
               RX_DONE=4'd6,      
               TX_SEND_TXV=4'd7,
               TX_SEND_DATA=4'd8,
               TX_WAIT_TXEND=4'd9,
               TX_PHYERR=4'd10,
               TX_DONE=4'd11;

  reg  [ 3:0]  state;
  reg          txdone;
  
  reg          rxend4timing_captured;
  reg          rxend_captured;
  
  wire [15:0]  n_burst_count;
  reg  [15:0]  burst_count;
  wire [19:0]  n_interframe_count;
  reg  [19:0]  interframe_count;

  reg          rxerr_1t,rxend_1t;
  reg          rxv_unsupported, rxv_unsupported_1t;
  reg          rxv_supported,   rxv_supported_1t;

  /*****************************************************************************
  * fsm
  *****************************************************************************/
  assign n_burst_count      = burst_count      + 16'd1;
  assign n_interframe_count = interframe_count + 20'd1;
  
  assign event_rxv_unsupported = ~rxv_unsupported_1t & rxv_unsupported;
  assign event_rxv_supported   = ~rxv_supported_1t   & rxv_supported;
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      txreq                 <= 1'b0;
      txdone                <= 1'b0;
      pldgen_enable         <= 1'b0;
      event_tx_eof          <= 1'b0;
      event_tx_phyerr       <= 1'b0; 
      
      rxreq                 <= 1'b0;
      rxend4timing_captured <= 1'b0;
      rxend_captured        <= 1'b0;
      pldtrck_enable        <= 1'b0;
      
      rxerr_1t              <= 1'b0; 
      rxend_1t              <= 1'b0; 
      event_rx_rxerr        <= 1'b0;
      event_rx_rxend        <= 1'b0;
      event_rx_eof_normal   <= 1'b0;
      event_rx_eof_normal   <= 1'b0;
      event_rx_eof_abort    <= 1'b0;
      rxv_supported         <= 1'b0;
      rxv_supported_1t      <= 1'b0;
      rxv_unsupported       <= 1'b0;
      rxv_unsupported_1t    <= 1'b0;
     
      burst_count           <= 16'd0;
      interframe_count      <= 20'd0;
      state                 <= IDLE;
    end
    else
    begin
      /* rtz */
      rxerr_1t              <= 1'b0; 
      rxend_1t              <= 1'b0; 
      event_rx_rxerr        <= 1'b0;
      event_rx_rxend        <= 1'b0;
      event_rx_eof_normal   <= 1'b0;
      event_rx_eof_abort    <= 1'b0;
      event_rx_rxerr        <= 1'b0;
      event_rx_rxend        <= 1'b0;
      event_tx_eof          <= 1'b0;
      event_tx_phyerr       <= 1'b0;
      
      /* edge detection */
      rxv_unsupported_1t    <= rxv_unsupported;
      rxv_supported_1t      <= rxv_supported;
      
      /* rxerr,rxendfortiming,rxend states */
      if(state==RX_RECEIVE_RXV1 || state==RX_RECEIVE_DATA || 
         state==RX_RECEIVE_RXV2 || state==RX_DONE)
      begin
        rxend4timing_captured <= rxend4timing_captured | mpifo_rxendfortiming;
        rxend_captured        <= rxend_captured        | mpifo_rxend;
        rxerr_1t              <= mpifo_rxerr; 
        rxend_1t              <= mpifo_rxend; 
      
        if(mpifo_rxerr)
        begin
          event_rx_rxerr <= 1'b1;
        end
        
        if(mpifo_rxend)
        begin
          event_rx_rxend<= 1'b1;
        end
      end
      else
      begin
        rxend4timing_captured <= 1'b0;
        rxend_captured        <= 1'b0;
        rxerr_1t              <= 1'b0; 
        rxend_1t              <= 1'b0; 
      end
      
      /*************************************************************************
      * fsm
      *************************************************************************/
      if(regb_bypass || regb_mode_is_rxlisten && mpifi_rxreq)
      begin
        case(state)
          /***********************************************************************
          * IDLE
          ***********************************************************************/
          IDLE: 
          begin
            rxv_supported      <= 1'b0;
            rxv_unsupported    <= 1'b0;
            rxreq              <= 1'b0;
            pldtrck_enable     <= 1'b0;
            pldgen_enable      <= 1'b0;
            burst_count        <= 16'd0;
            rxreq              <= 1'b0;
            state              <= RX_LISTEN;
          end
          
          /***********************************************************************
          * RX_LISTEN
          ***********************************************************************/
          RX_LISTEN:
          begin
            rxv_supported      <= 1'b0;
            rxv_unsupported    <= 1'b0;
            rxreq              <= regb_defrxreq;
            pldtrck_enable     <= 1'b0;
            
            if(mpifo_phyrdy)
            begin
              state          <= RX_RECEIVE_RXV1;
            end
            else if(regb_mode_is_txsingle || regb_mode_is_txburst)
            begin
              if(interframe_count==regb_interframe_delay)
              begin
                rxreq      <= 1'b0;
                txreq      <= 1'b1;
                state      <= TX_SEND_TXV;
              end
              else
              begin
                interframe_count <= n_interframe_count;
              end
            end
          end
          
          /***********************************************************************
          * RX_RECEIVE_RXV1
          ***********************************************************************/
          RX_RECEIVE_RXV1:
          begin
            if(mpifo_rxerr)
            begin
              state <= RX_DONE;
            end
            else if(mpifo_phyrdy && rxvu1_last)
            begin
              pldtrck_enable <= 1'b1;
              state          <= RX_RECEIVE_DATA;
            end
          end
          
          /***********************************************************************
          * RX_RECEIVE_DATA
          ***********************************************************************/
          RX_RECEIVE_DATA:
          begin
            if(mpifo_rxerr)
            begin
              if(!rxv_supported)
                rxv_unsupported <= 1'b1;
              else
                event_rx_eof_abort <= 1'b1;
            
              state <= RX_DONE;
            end
            else if(rxvu_ndp)
            begin
              rxv_supported  <= 1'b1;
              pldtrck_enable <= 1'b0;
              state          <= RX_RECEIVE_RXV2;
            end
            else if(mpifo_phyrdy)
            begin
              rxv_supported <= 1'b1;
              if(pldtrck_last)
              begin
                pldtrck_enable <= 1'b0;
                state          <= RX_RECEIVE_RXV2;
              end
            end
          end
          
          /***********************************************************************
          * RX_RECEIVE_RXV2
          ***********************************************************************/
          RX_RECEIVE_RXV2:
          begin
            if(mpifo_rxerr)
            begin
              event_rx_eof_abort <= 1'b1;
              state              <= RX_DONE; 
            end 
            else if(mpifo_phyrdy && rxvu2_last)
            begin              
              event_rx_eof_normal <= 1'b1;
              state               <= RX_DONE;
            end
          end
          
          /***********************************************************************
          * RX_DONE
          ***********************************************************************/
          RX_DONE:
          begin
            pldtrck_enable     <= 1'b0;
            if(rxend4timing_captured && rxend_captured)
            begin
              state <= RX_LISTEN;
            end
          end
          
          /***********************************************************************
          * TX_SEND_TXV
          ***********************************************************************/
          TX_SEND_TXV:
          begin
            if(!txreq)
            begin
              /* assert txreq and push first byte txvector */
              txreq     <= 1'b1;
            end
            else
            begin
              if(mpifo_phyerr)
              begin
                event_tx_phyerr <= 1'b1;
                state           <= TX_PHYERR;
              end
              else if(txv_last)
              begin
                pldgen_enable <= 1'b1;
                state         <= TX_SEND_DATA; 
              end
            end
          end
         
          /***********************************************************************
          * TX_SEND_DATA
          ***********************************************************************/
          TX_SEND_DATA:
          begin 
            if(mpifo_phyerr)
            begin
              event_tx_phyerr  <= 1'b1;
              pldgen_enable    <= 1'b0;
              state            <= TX_PHYERR;
            end
            else if(pldgen_done)
            begin 
              pldgen_enable    <= 1'b0;
              state            <= TX_WAIT_TXEND;
            end
          end
          
          /***********************************************************************
          * TX_WAIT_TXEND
          ***********************************************************************/
          TX_WAIT_TXEND:
          begin
            if(mpifo_txend)
            begin
              txreq            <= 1'b0;
              rxreq            <= 1'b0;
              interframe_count <= 20'd0;
              if(regb_mode_is_txsingle)
              begin
                /* end of single tx frame */
                event_tx_eof   <= 1'b1;
                state <= TX_DONE;
              end
              else
              begin
                if(regb_frameperburst!=16'd0 && regb_frameperburst==n_burst_count)
                begin
                  /* end of burst sequence  of tx frames */
                  event_tx_eof   <= 1'b1;
                  state       <= TX_DONE;
                end
                else
                begin
                  burst_count <= n_burst_count;
                  state       <= RX_LISTEN;
                end
              end
            end
          end
         
          /***********************************************************************
          * TX_DONE
          ***********************************************************************/
          TX_DONE:
          begin 
            txdone  <= 1'b1;
            /* return on IDLE */
            if(!regb_mode_is_txsingle && !regb_mode_is_txburst)
            begin
              txdone <= 1'b0;
              state  <= IDLE;
            end
          end

          /***********************************************************************
          * TX_PHYERR
          ***********************************************************************/
          default:
          begin
            if(mpifo_txend)
            begin
              txreq  <= 1'b0;
            end
            
            /* return on IDLE */
            if(!regb_mode_is_txsingle && !regb_mode_is_txburst)
            begin
              txdone <= 1'b0;
              state  <= IDLE;
            end
          end
        endcase
      end
      else
      begin
        txdone                 <= 1'b0;
        rxreq                  <= 1'b0;
        txreq                  <= 1'b0; 
        rxend4timing_captured  <= 1'b0;
        rxend_captured         <= 1'b0;
        rxerr_1t               <= 1'b0; 
        rxend_1t               <= 1'b0; 
        pldtrck_enable         <= 1'b0;
        pldgen_enable          <= 1'b0;
        state                  <= IDLE;
      end
    end
  end

`ifdef RW_SIMU_ON
  reg [32*8-1:0] s_state;
  always @(*)
  begin
    case(state)
      IDLE:            s_state = "IDLE";           
      RX_LISTEN:       s_state = "RX_LISTEN";      
      RX_RECEIVE_RXV1: s_state = "RX_RECEIVE_RXV1";
      RX_RECEIVE_DATA: s_state = "RX_RECEIVE_DATA";
      RX_RECEIVE_RXV2: s_state = "RX_RECEIVE_RXV2";
      RX_DONE:         s_state = "RX_DONE";         
      TX_SEND_TXV:     s_state = "TX_SEND_TXV";     
      TX_SEND_DATA:    s_state = "TX_SEND_DATA";    
      TX_WAIT_TXEND:   s_state = "TX_WAIT_TXEND";   
      TX_PHYERR:       s_state = "TX_PHYERR";       
      TX_DONE:         s_state = "TX_DONE";        
    endcase
  end
`endif

endmodule
`default_nettype wire
  
