/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :                                       
* Description      : mpif_payload_generator
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module rw_nx_macbypass_payload_generator
(
  input  wire         clk,
  input  wire         rst_n,
  
  input  wire         enable,
  input  wire  [ 3:0] pldtype,
  input  wire  [ 3:0] format,
  input  wire         aggregation,
  input  wire  [19:0] length,
  output reg          done,

  input  wire         ready,
  output reg   [ 7:0] data,
  output reg          valid
);

  /*****************************************************************************
  * declaration
  *****************************************************************************/
  localparam [3:0]  IDLE        = 4'd0,
                    PAYLOAD     = 4'd1,
                    DELIM_PUSH  = 4'd2,
                    FCS_COMPUTE = 4'd3,
                    FCS_PUSH    = 4'd4,
                    EOF         = 4'd5;
  
  reg  [ 7:0] n_crc8;
  reg  [23:0] n_prbs;
  wire [19:0] n_count;
  wire [19:0] length_minus_4;
  wire [19:0] length_minus_8; 
  wire [19:0] length_payload; 
  reg         is_aggregate;
  
  reg  [23:0] prbs;
  reg  [ 3:0] state;
  reg  [19:0] count;
  reg  [15:0] delim;
  reg         last;
  
  reg  [ 4:0] rqptr,wrptr;
  wire [ 4:0] n_rqptr,n_wrptr;
  wire        rq_pending;
  
  wire [31:0] fcs;
  reg         fcs_enable;
  wire        fcs_done;
  
  /*****************************************************************************
  * 
  *****************************************************************************/
  always @(*)
  begin
    case(format)
      4'd0,4'd1: is_aggregate = 1'b0;
      4'd2,4'd3: is_aggregate = aggregation;
      default:   is_aggregate = 1'b1;
    endcase
  end

  assign n_count        = count   + 20'd1;
  assign length_minus_4 = length  - 20'd4;
  assign length_minus_8 = length  - 20'd8;
  assign length_payload = (is_aggregate)?length_minus_8:length_minus_4;
  
  /*****************************************************************************
  * CRC8
  *****************************************************************************/
  always @(*)
  begin:b_crc_calc
    integer   i;
    reg [7:0] c;
  
    /* CRC8 p911 8.6.2  */
    c = 8'hff;
    for(i=0;i<16;i=i+1)
      c = {c[6:2],c[1]^c[7]^delim[i],c[0]^c[7]^delim[i],c[7]^delim[i]}; 
    n_crc8=~{c[0],c[1],c[2],c[3],c[4],c[5],c[6],c[7]};
  end

  /*****************************************************************************
  * PRBS23 1+x^18+x^23 
  *****************************************************************************/
  always @(*)
  begin:b_prbs_calc
    integer   i;
    
    n_prbs = prbs;
    for(i=0;i<8;i=i+1)
      n_prbs = { prbs[0]^prbs[18]^prbs[23],prbs[23:1]};
    
  end
  
  /*****************************************************************************
  * FSM
  *****************************************************************************/
  assign n_rqptr    = rqptr + {4'b0,ready};
  assign n_wrptr    = wrptr + 5'd1;
  
  assign rq_pending = rqptr!=wrptr;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      data          <= 8'b0;
      last          <= 1'b0;
      valid         <= 1'b0;
      done          <= 1'b0;
      prbs          <= 24'h1234d1;
      count         <= 20'd0;
      delim         <= 16'b0;
      rqptr         <= 5'd0;
      wrptr         <= 5'd0;
      fcs_enable    <= 1'b0;
      state         <= IDLE;
    end
    else if(!enable)
    begin
      data          <= 8'b0;
      last          <= 1'b0;
      valid         <= 1'b0;
      done          <= 1'b0;
      count         <= 20'd0;
      delim         <= 16'b0;
      rqptr         <= 5'd0;
      wrptr         <= 5'd0;
      fcs_enable    <= 1'b0;
      state         <= IDLE;
    end
    else 
    begin
      /* rtz */
      last  <= 1'b0;
      valid <= 1'b0;
      
      /* ptr */
      rqptr <= n_rqptr;
      
      case(state)
        IDLE:
        begin
          count         <= 20'd0;
          if(!is_aggregate)
          begin
            state <= PAYLOAD;
          end
          else
          begin
            if(format>=4'd4)
              delim <= {length_minus_4[11:0],length_minus_4[13:12],2'b01};
            else
              delim <= {length_minus_4[11:0],                 2'b0,2'b00};
            state <= DELIM_PUSH;
          end
        end
        
        DELIM_PUSH:
        begin
          if(rq_pending)
          begin
            count  <= n_count;                             
            wrptr  <= n_wrptr;
            valid  <= 1'b1;
            case(count[1:0])                              
              2'd0:    data <= delim[ 7:0];                
              2'd1:    data <= delim[15:8];                  
              2'd2:    data <= n_crc8;                       
              default:                                     
              begin                                        
                data  <= 8'h4e;                            
                count <= 20'd0;                            
                state <= PAYLOAD;                          
              end                                          
            endcase
          end                                        
        end
        
        PAYLOAD:
        begin
          fcs_enable <= 1'b1;
          if(last)
          begin
            count <= 20'd0;
            state <= FCS_COMPUTE;
          end
          else
          begin
            if(rq_pending)
            begin
              case(pldtype)
                4'd0:    data <= 8'd0;
                4'd1:    data <= count[7:0];
                4'd2:    data <= prbs[7:0];
                default: data <= 8'd0;
              endcase
              last  <= n_count==length_payload;
              wrptr <= n_wrptr;
              valid <= 1'b1;
              prbs  <= n_prbs;                                                  
              count <= n_count;
            end
          end
        end
        
        FCS_COMPUTE:
        begin
          if(fcs_done)
            state <= FCS_PUSH;
        end
        
        FCS_PUSH:
        begin
          if(rq_pending)
          begin
            wrptr <= n_wrptr;
            valid <= 1'b1;
            count <= n_count;
            case(count[1:0])
              2'd0:   data  <= fcs[31:24];
              2'd1:   data  <= fcs[23:16];
              2'd2:   data  <= fcs[15: 8];
              default:data  <= fcs[ 7: 0];
            endcase
          
            if(count[1:0]==2'd3)
            begin
              state      <= EOF;
              fcs_enable <= 1'b0;
            end
          end
        end
        
        default: 
        begin
          done <= 1'b1;
        end
      
      endcase
    end
  end
  
  /*****************************************************************************
  * FCS
  *****************************************************************************/
  rw_nx_macbypass_fcs32 u_fcs
  (
    .clk(          clk),
    .rst_n(        rst_n),
    
    .enable(       fcs_enable),
    .done(         fcs_done),
    .fcs(          fcs),

    .data(         data),
    .last(         last),
    .valid(        valid)
  );
  
`ifdef RW_SIMU_ON
  reg [16*8-1:0] str_state;
  always @(*)
    case(state)
      IDLE:        str_state = "IDLE";
      PAYLOAD:     str_state = "PAYLOAD";
      DELIM_PUSH:  str_state = "DELIM_PUSH";
      FCS_COMPUTE: str_state = "FCS_COMPUTE";
      FCS_PUSH:    str_state = "FCS_PUSH";
      EOF:         str_state = "EOF";
      default:     str_state = "XXXXXX";
    endcase
`endif

endmodule
`default_nettype wire
