/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :                                       
* Description      : mpif_payload_tracking
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module rw_nx_macbypass_payload_tracking
(
  input wire         clk,
  input wire         rst_n,
  
  input wire         enable,
  output wire        last,
  
  input wire  [ 3:0] format,
  input wire         aggregation,
  input wire  [19:0] length,
  
  input wire  [ 7:0] data,
  input wire         valid,
  
  output reg         event_fcs_update,
  output reg         event_fcs_ok,
  output reg         event_fcs_bad,
  output reg         event_delimiter_lost
);
  localparam  IDLE = 3'd0,
              D1   = 3'd1,
              D2   = 3'd2,
              D3   = 3'd3,
              MPDU = 3'd4,
              EOF  = 3'd5;
  
  reg  [ 2:0] state;
  reg  [ 1:0] pos;
  reg  [19:0] count;
  reg  [19:0] ocount;
  reg  [15:0] d;
  reg  [31:0] fcs32_field;
  reg  [31:0] fcs32;
  reg         fcs_init_done;
  reg         compare_fcs;
  wire [ 1:0] n_pos;
  wire [19:0] n_count;
  wire [19:0] n_ocount;
  reg  [ 7:0] n_crc8;
  reg  [31:0] n_fcs32;
  wire [ 7:0] atad;
  reg  [19:0] c_length; 
  wire [ 7:0] c_data;
  wire        is_fcs;
  wire [31:0] fcs32_computed;
  wire        is_vht;   

  assign is_vht   = format==4'd4;
  assign atad     = {data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7]};
  assign n_pos    = pos   + 2'd1;                        
  assign n_count  = count + 20'd1;                       
  assign n_ocount = ocount + 20'd1;                       
  
  assign last     = valid & (length==n_ocount);
  
  always @(*)
  begin
    case(format)
      4'd0,4'd1: c_length = length;
      4'd2,4'd3: c_length = (!aggregation)?length:{6'b0,   2'd0, d[15:4]};
      default:   c_length = {6'b0, d[3:2], d[15:4]};
    endcase
  end
  
  assign is_fcs   = n_count>(c_length - 20'd4);
  assign c_data   = is_fcs?8'b0:data;
  
  always @(*)
  begin:b_crc_calc
    integer   i;
    reg [7:0] c;
  
    /* CRC8 p911 8.6.2  */
    c = 8'hff;
    for(i=0;i<16;i=i+1)
      c = {c[6:2],c[1]^c[7]^d[i],c[0]^c[7]^d[i],c[7]^d[i]}; 
    n_crc8=~{c[0],c[1],c[2],c[3],c[4],c[5],c[6],c[7]};
 
    /* FCS p498 8.2.4.8 */
    n_fcs32 = fcs32;
    for(i=0;i<8;i=i+1)
      n_fcs32 = {n_fcs32[30:0],c_data[i]}^(n_fcs32[31]?32'h04c11db7:32'd0);
    
  end
  
  assign fcs32_computed = 
    ~{fcs32[24],fcs32[25],fcs32[26],fcs32[27],fcs32[28],fcs32[29],fcs32[30],fcs32[31],
      fcs32[16],fcs32[17],fcs32[18],fcs32[19],fcs32[20],fcs32[21],fcs32[22],fcs32[23],
      fcs32[ 8],fcs32[ 9],fcs32[10],fcs32[11],fcs32[12],fcs32[13],fcs32[14],fcs32[15],
      fcs32[ 0],fcs32[ 1],fcs32[ 2],fcs32[ 3],fcs32[ 4],fcs32[ 5],fcs32[ 6],fcs32[ 7]};
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      pos                        <= 2'd0;
      count                      <= 20'd0;
      ocount                     <= 20'd0;
      d                          <= 16'd0;
      fcs32_field                <= 32'd0;
      fcs32                      <= 32'd0;
      fcs_init_done              <= 1'b0;
      compare_fcs                <= 1'b0;
      event_fcs_update           <= 1'b0;
      event_fcs_ok               <= 1'b0;
      event_fcs_bad              <= 1'b0;
      event_delimiter_lost       <= 1'b0;
      state                      <= IDLE;
    end
    else
    begin
      event_fcs_update      <= 1'b0;
      event_fcs_ok          <= 1'b0;
      event_fcs_bad         <= 1'b0;
      event_delimiter_lost  <= 1'b0;
      compare_fcs           <= 1'b0;       
      
      if(compare_fcs)                    
      begin                              
        event_fcs_update <= 1'b1;         
        if(fcs32_field==fcs32_computed) 
          event_fcs_ok           <= 1'b1;       
        else
          event_fcs_bad <= 1'b1;       
      end                                
      
      if(!enable)
      begin
        pos           <= 2'd0;
        count         <= 20'd0;
        ocount        <= 20'd0;
        d             <= 16'd0;
        fcs32_field   <= 32'd0;
        fcs32         <= 32'd0;
        fcs_init_done <= 1'b0;
        compare_fcs   <= 1'b0;
        state         <= IDLE;
      end
      else if(valid)
      begin                                                              
        /* count overall bytes */
        ocount <= n_ocount;
        
        /* track mpdu/ampdu */
        pos <= n_pos;                                                    
        case(state)                                                      
          IDLE:                                                          
          begin                                                          
            fcs_init_done <= 1'b0;                                       
            fcs32_field   <= 32'b0;                                      
                                                                         
            if(aggregation || format>=4'd4)                                                    
            begin                                                        
              if(pos==2'd0)                                              
              begin                                                      
                /* HT/VHT AMPDU */                                       
                d[7:0] <= data;                                          
                count  <= 20'd0;                                         
                state  <= D1;                                            
              end                                                        
            end                                                          
            else                                                         
            begin                                                        
              /* NONHT/HT MPDU*/                                         
              fcs32[31:24] <= ~atad;                                     
              count        <= 20'd1;                                     
              state        <= MPDU;                                      
            end                                                          
          end                                                            
                                                                         
          D1:                                                            
          begin                                                          
            d[15:8] <= data;                                             
            state   <= D2;                                               
          end                                                            
                                                                         
          D2:                                                            
          begin                                                          
            if(n_crc8==data)
            begin                                             
              state <= D3;                                               
            end
            else                                                         
            begin                                                        
              event_delimiter_lost <= 1'b1;                        
              state <= IDLE;                                             
            end                                                          
          end                                                            
                                                                         
          D3:                                                            
          begin                                                          
            if(data==8'h4e)                                              
            begin                                                        
              if(d[15:2]==14'd0)
              begin
                state <= IDLE; /* blank delimiter */
              end
              else
              begin
                state <= MPDU;
              end
            end                                                          
            else                                                         
            begin                                                        
              event_delimiter_lost <= 1'b1;                        
              state                <= IDLE;                                             
            end                                                          
          end                                                            
                                                                         
          MPDU:                                          
          begin                                                          
                                                                         
            if(n_count==c_length)                                        
            begin                                                        
              compare_fcs <= 1'b1;                                
              state       <= IDLE;                                       
            end                                                          
                                                                         
            if(!fcs_init_done)
            begin                                           
              case(pos)                                                  
                2'd0:     {fcs32[31:24],fcs_init_done} <= {~atad,1'b0};  
                2'd1:     {fcs32[23:16],fcs_init_done} <= {~atad,1'b0};  
                2'd2:     {fcs32[15: 8],fcs_init_done} <= {~atad,1'b0};  
                default:  {fcs32[ 7: 0],fcs_init_done} <= {~atad,1'b1};  
              endcase                                                    
            end
            else
            begin                                                         
              fcs32 <= n_fcs32;                                          
            end                                                             
            if(is_fcs)
            begin                                                   
              fcs32_field <= {fcs32_field[23:0],data};                   
            end                                                             
            count <= n_count;                                            
          end  
          
          default: /* EOF */ ;
                                                                    
        endcase                                                          
      end                                                                
    end
  end

`ifdef RW_SIMU_ON
  reg [16*8-1:0] str_state;
  always @(*)
  begin
    case(state)
      IDLE:    str_state = "IDLE";
      D1:      str_state = "D1";
      D2:      str_state = "D2";
      D3:      str_state = "D3";
      MPDU:    str_state = "MPDU";
      EOF:     str_state = "EOF";
      default: str_state = "XXXXX";
    endcase
  end
`endif

endmodule
`default_nettype wire
