////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of MAC/PHY Interface diagnostic port control
//                    module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_nx_mpif_diag_ctrl( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          hard_rst_n,              // Hardware Reset
   input  wire          soft_rst_n,              // Software Reset

   //$port_g MAC/PHY interface
   input  wire          mpif_txreq,              // TX Request 
   input  wire    [7:0] mpif_txdata,             // TX Data     
   input  wire          mpif_macdatavalid,       // TX Data valid 
   input  wire          mpif_txend_p,            // TX End
   input  wire          mpif_rxreq,              // RX Request 
   input  wire    [7:0] mpif_rxdata,             // RX Data  
   input  wire          mpif_phyrdy,             // RX Data valid
   input  wire          mpif_rxend_p,            // RX End

   //$port_g Control interface
   input  wire          diag_mpif_mask_en,       // Enable the MPIF data masking
   input  wire    [5:0] diag_mpif_hdr_nbr,       // Indicates the number of MAC
                                                 // Header Bytes to capture
   output reg           diag_mpif_mask,          // Indicates that the MPIF data
                                                 // should be masked.
   //$port_g Debug Port
   output wire   [15:0] debug                    // Debug Port
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// FSM states definition
//$fsm_sd mpif_diag
localparam 
   IDLE         = 3'd0,
   RX_VECTOR    = 3'd1,
   RX_DATA_CAPT = 3'd2,
   RX_DATA_MASK = 3'd3,
   TX_VECTOR    = 3'd4,
   TX_DATA_CAPT = 3'd5,
   TX_DATA_MASK = 3'd6;

// Format and Modulation
localparam
   NON_HT       = 4'b0000,
   NON_HT_DUP   = 4'b0001,
   HT_MM        = 4'b0010,
   HT_GF        = 4'b0011,
   VHT          = 4'b0100,
   HE_SU        = 4'b0101,
   HE_MU        = 4'b0110,
   HE_EXT_SU    = 4'b0111,
   HE_TRIG      = 4'b1000;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg    [2:0] mpif_diag_cs;             // Current state
reg    [2:0] mpif_diag_ns;             // Next State

reg    [5:0] byte_cnt;                 // Bytes Down Counter


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// MPIF DIAG FSM State Logic 
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      mpif_diag_cs <= IDLE;
   else if (~soft_rst_n)
      mpif_diag_cs <= IDLE;
   else
      mpif_diag_cs <= mpif_diag_ns;
end

always @* 
begin  
   case (mpif_diag_cs)
   IDLE:
      //$fsm_s In IDLE state, the state machine waits TX or RX start
      if (diag_mpif_mask_en & mpif_rxreq & mpif_phyrdy)
         //$fsm_t When reception is started,
         //the state machine goes to RX_VECTOR state
         mpif_diag_ns = RX_VECTOR;
      else if (diag_mpif_mask_en & mpif_txreq & mpif_macdatavalid)
         //$fsm_t When transmission is started,
         //the state machine goes to TX_VECTOR state
         mpif_diag_ns = TX_VECTOR;
      else
         //$fsm_t While RX or TX request is not received,
         //the state machine stays in IDLE state
         mpif_diag_ns = IDLE;

   RX_VECTOR:
      //$fsm_s In RX_VECTOR state, the state machine waits end of the RX vector
      if (~mpif_rxreq | mpif_rxend_p)
         //$fsm_t When reception is done,
         //the state machine goes to IDLE state
         mpif_diag_ns = IDLE;
      else if (byte_cnt==6'd1 & mpif_phyrdy & diag_mpif_hdr_nbr==6'd0)
         //$fsm_t When RX Vector is done and MAC Header should not be captured,
         //the state machine goes to RX_DATA_MASK state
         mpif_diag_ns = RX_DATA_MASK;
      else if (byte_cnt==6'd1 & mpif_phyrdy & diag_mpif_hdr_nbr!=6'd0)
         //$fsm_t When RX Vector is done and MAC Header should be captured,
         //the state machine goes to RX_DATA_CAPT state
         mpif_diag_ns = RX_DATA_CAPT;
      else
         //$fsm_t While RX Vector is not done,
         //the state machine stays in RX_VECTOR state
         mpif_diag_ns = RX_VECTOR;

   RX_DATA_CAPT:
      //$fsm_s In RX_DATA_CAPT state, the state machine waits requiered number
      //of MAC header bytes capture
      if (~mpif_rxreq | mpif_rxend_p)
         //$fsm_t When reception is done,
         //the state machine goes to IDLE state
         mpif_diag_ns = IDLE;
      else if (byte_cnt==6'd1 & mpif_phyrdy)
         //$fsm_t When requiered number of MAC header bytes has been captured,
         //the state machine goes to RX_DATA_MASK state
         mpif_diag_ns = RX_DATA_MASK;
      else
         //$fsm_t While requiered number of MAC header bytes is not done,
         //the state machine stays in RX_DATA_CAPT state
         mpif_diag_ns = RX_DATA_CAPT;

   RX_DATA_MASK:
      //$fsm_s In RX_DATA_MASK state, the state machine waits end of the
      //reception
      if (~mpif_rxreq | mpif_rxend_p)
         //$fsm_t When reception is done,
         //the state machine goes to IDLE state
         mpif_diag_ns = IDLE;
      else
         //$fsm_t While reception is not done,
         //the state machine stays in RX_DATA_MASK state
         mpif_diag_ns = RX_DATA_MASK;

   TX_VECTOR:
      //$fsm_s In TX_VECTOR state, the state machine waits end of the TX vector
      if (~mpif_txreq | mpif_txend_p)
         //$fsm_t When transmission is done,
         //the state machine goes to IDLE state
         mpif_diag_ns = IDLE;
      else if (byte_cnt==6'd1 & mpif_macdatavalid & diag_mpif_hdr_nbr==6'd0)
         //$fsm_t When TX Vector is done and MAC Header should not be captured,
         //the state machine goes to TX_DATA_MASK state
         mpif_diag_ns = TX_DATA_MASK;
      else if (byte_cnt==6'd1 & mpif_macdatavalid & diag_mpif_hdr_nbr!=6'd0)
         //$fsm_t When TX Vector is done and MAC Header should be captured,
         //the state machine goes to TX_DATA_CAPT state
         mpif_diag_ns = TX_DATA_CAPT;
      else
         //$fsm_t While TX Vector is not done,
         //the state machine stays in TX_VECTOR state
         mpif_diag_ns = TX_VECTOR;

   TX_DATA_CAPT:
      //$fsm_s In TX_DATA_CAPT state, the state machine waits requiered number
      //of MAC header bytes capture
      if (~mpif_txreq | mpif_txend_p)
         //$fsm_t When transmission is done,
         //the state machine goes to IDLE state
         mpif_diag_ns = IDLE;
      else if (byte_cnt==6'd1 & mpif_macdatavalid)
         //$fsm_t When requiered number of MAC header bytes has been captured,
         //the state machine goes to TX_DATA_MASK state
         mpif_diag_ns = TX_DATA_MASK;
      else
         //$fsm_t While requiered number of MAC header bytes is not done,
         //the state machine stays in TX_DATA_CAPT state
         mpif_diag_ns = TX_DATA_CAPT;

   TX_DATA_MASK:
      //$fsm_s In TX_DATA_MASK state, the state machine waits end of the
      //transmission
      if (~mpif_txreq | mpif_txend_p)
         //$fsm_t When transmission is done,
         //the state machine goes to IDLE state
         mpif_diag_ns = IDLE;
      else
         //$fsm_t While transmission is not done,
         //the state machine stays in TX_DATA_MASK state
         mpif_diag_ns = TX_DATA_MASK;

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      mpif_diag_ns = IDLE; 
   // pragma coverage block = on
   endcase
end
  

//******************************************************************************
// Byte Counter
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      byte_cnt <= 6'd0;
   else if (~soft_rst_n | mpif_diag_ns==IDLE)
      byte_cnt <= 6'd0;
   else if (mpif_diag_cs==IDLE & mpif_diag_ns==RX_VECTOR)
      // Load byte counter with RX Vector Length minus one
      case (mpif_rxdata[3:0]) // RX FormatMod
      NON_HT    : byte_cnt <= 6'd6;  //  7bytes
      NON_HT_DUP: byte_cnt <= 6'd6;  //  7bytes
      HT_MM     : byte_cnt <= 6'd9;  // 10bytes
      HT_GF     : byte_cnt <= 6'd9;  // 10bytes
      VHT       : byte_cnt <= 6'd12; // 13bytes
      HE_SU     : byte_cnt <= 6'd15; // 16bytes
      HE_MU     : byte_cnt <= 6'd15; // 16bytes
      HE_EXT_SU : byte_cnt <= 6'd15; // 16bytes
      HE_TRIG   : byte_cnt <= 6'd6;  // N/A
      default   : byte_cnt <= 6'd6;  // 7bytes (Incorrect Format)
      endcase
   else if (mpif_diag_cs==IDLE & mpif_diag_ns==TX_VECTOR)
      // Load byte counter with TX Vector Length minus one
      case (mpif_txdata[3:0]) // RX FormatMod
      NON_HT    : byte_cnt <= 6'd9;  // 10bytes
      NON_HT_DUP: byte_cnt <= 6'd9;  // 10bytes
      HT_MM     : byte_cnt <= 6'd12; // 13bytes
      HT_GF     : byte_cnt <= 6'd12; // 13bytes
      VHT       : byte_cnt <= 6'd16; // 17bytes (1 user only)
      HE_SU     : byte_cnt <= 6'd17; // 18bytes
      HE_MU     : byte_cnt <= 6'd9;  // N/A
      HE_EXT_SU : byte_cnt <= 6'd17; // 18bytes
      HE_TRIG   : byte_cnt <= 6'd23; // 24bytes
      default   : byte_cnt <= 6'd9;  // 10bytes (Incorrect Format)
      endcase
   else if (mpif_diag_cs==RX_VECTOR & mpif_diag_ns==RX_DATA_CAPT |
            mpif_diag_cs==TX_VECTOR & mpif_diag_ns==TX_DATA_CAPT )
      // Load byte counter with diag_mpif_hdr_nbr
      byte_cnt <= diag_mpif_hdr_nbr;
   else if (mpif_diag_cs==RX_VECTOR    & mpif_phyrdy       |
            mpif_diag_cs==RX_DATA_CAPT & mpif_phyrdy       |
            mpif_diag_cs==TX_VECTOR    & mpif_macdatavalid |
            mpif_diag_cs==TX_DATA_CAPT & mpif_macdatavalid )
      // Decrement byte counter
      byte_cnt <= byte_cnt-6'd1;
end


//******************************************************************************
// MAC/PHY Interface data mask
//******************************************************************************
always @(posedge clk or negedge hard_rst_n)
begin
   if (~hard_rst_n)
      diag_mpif_mask <= 1'b0;
   else if (~soft_rst_n)
      diag_mpif_mask <= 1'b0;
   else
      diag_mpif_mask <= mpif_diag_ns==RX_DATA_MASK | 
                        mpif_diag_ns==TX_DATA_MASK ;
end


//******************************************************************************
// Debug Port
//******************************************************************************
assign debug = {5'h0,
                diag_mpif_mask_en,
                diag_mpif_mask,
                byte_cnt[5:0],
                mpif_diag_cs[2:0]};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef  RW_SIMU_ON
// GCM FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] mpif_diag_cs_str;
always @*
begin
   case (mpif_diag_cs)
   IDLE        : mpif_diag_cs_str = {"IDLE        "};
   RX_VECTOR   : mpif_diag_cs_str = {"RX_VECTOR   "};
   RX_DATA_CAPT: mpif_diag_cs_str = {"RX_DATA_CAPT"};
   RX_DATA_MASK: mpif_diag_cs_str = {"RX_DATA_MASK"};
   TX_VECTOR   : mpif_diag_cs_str = {"TX_VECTOR   "};
   TX_DATA_CAPT: mpif_diag_cs_str = {"TX_DATA_CAPT"};
   TX_DATA_MASK: mpif_diag_cs_str = {"TX_DATA_MASK"};
   default     : mpif_diag_cs_str = {"XXX"};
   endcase
end
`endif//RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef  RW_ASSERT_ON
`endif//RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
