/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
********************************************************************************
* $Author: $
* Company          : RivieraWaves
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     :                                       
* Description      : ucpu
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
********************************************************************************
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module rw_ucpu
(
  /* system */
  input wire         rst_n,
  input wire         clk,
  
  /* control */
  input  wire        enable,
  output reg         done,
  
  /* prog */
  output wire [ 8:0] pr_addr,
  input  wire [15:0] pr_data,

  /* cc */
  input  wire [63:0] cc,
  
  /* rb */
  output wire [ 5:0] rb_addr,
  input  wire [31:0] rb_data,
  
  /* wb */
  output wire [ 5:0] wb_addr,
  output wire [31:0] wb_x,
  output wire [31:0] wb_y
);
  /*****************************************************************************
  * declarations
  *****************************************************************************/
  reg  [ 8:0] pc;
  reg  [31:0] b;
  wire [31:0] x;
  wire [31:0] y;

  reg  [15:0] op_data;
  wire        op_jmp;
  wire [ 5:0] op_rb;
  wire [ 2:0] op_alu;
  wire [ 5:0] op_wb;
  wire [ 8:0] op_target;
  wire [ 5:0] op_cond;
  
  wire        cc_flag;

  wire        alu_clr;
  wire        alu_add;
  wire        alu_sub;
  wire        alu_mul;
  wire        alu_div;
  wire        stall;

  /*****************************************************************************
  * PC
  *****************************************************************************/
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
      pc   <= 9'd0;
    else if(!enable || done)
      pc   <= 9'd0;
    else if(!stall)
    begin
      if(op_jmp && cc_flag)
        pc <= op_target;
      else
        pc <= pc + 9'd1;
    end
  end
  
  assign pr_addr = pc;

  /*****************************************************************************
  * OP
  ******************************************************************************
  *        I-FROM       
  *   15   0            
  * 14:9   WB[5:0]  
  *  8:6  ALU[2:0]      
  *  5:0   RB[5:0]
  ******************************************************************************
  *        J-FORM   
  *   15   1
  * 14:9   COND[5:0]
  *  8:0   TARGET[8:0]      
  ******************************************************************************
  * ALU[2:0]  
  *  000 nop
  *  001 X=Y=0
  *  010 Y=add(Y,B)
  *  011 Y=sub(Y,B)
  *  100 Y=mul(Y,B)
  *  101 (Y,X)=div(Y,B)
  *  111 done
  *****************************************************************************/
  assign cc_flag   = cc[op_cond];
  
  assign op_jmp    = op_data[  15];
  assign op_wb     = op_data[14:9];
  assign op_alu    = op_data[ 8:6];
  assign op_rb     = op_data[ 5:0];
 
  assign op_cond   = op_data[14:9];
  assign op_target = op_data[ 8:0];
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      op_data  <= 16'b0;
      done     <=  1'b0;
    end
    else if(!enable)
    begin
      op_data <= 16'b0;
      done    <=  1'b0;
    end
    else if(done)
    begin
      op_data <= 16'b0;
    end
    else if(!stall)
    begin
      
      if(!op_jmp || !cc_flag)
      begin
        /* execute prefecthed opcode */
        op_data <= pr_data;
        done    <= op_alu==3'd7;
      end
      else
      begin
        /* cc is true, do a jump, prefetched opcode is dropped */
        op_data <= 16'b0;
      end
    end
  end

  /*****************************************************************************
  * RB
  *****************************************************************************/
  assign  rb_addr = op_rb;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
      b <= 32'b0;
    else if(!stall && !op_jmp && op_rb!=6'd0)
      b <= rb_data;
  end

  /*****************************************************************************
  * ALU
  *****************************************************************************/
  
  assign alu_clr = ~op_jmp & op_alu==3'd1;
  assign alu_add = ~op_jmp & op_alu==3'd2;
  assign alu_sub = ~op_jmp & op_alu==3'd3;
  assign alu_mul = ~op_jmp & op_alu==3'd4;
  assign alu_div = ~op_jmp & op_alu==3'd5;
  
  rw_ucpu_alu u_alu
  (
    /* system */
    .rst_n(     rst_n),
    .clk(       clk),
    /* */
    .stall(     stall),       
    /* operation */
    .clear(     alu_clr),
    .sel_mul(   alu_mul),
    .sel_div(   alu_div),
    .sel_add(   alu_add),
    .sel_sub(   alu_sub),
    /* operands */
    .i(         b),
    /* results */
    .x(         x),
    .y(         y)
  );

  /*****************************************************************************
  * WB
  *****************************************************************************/
  reg alu_stall_hazard;
  always @(posedge clk,negedge rst_n)
    if(!rst_n)
      alu_stall_hazard <= 1'b0;
    else
      alu_stall_hazard <= stall;
 
  assign wb_addr = (!op_jmp&&!done&&!alu_stall_hazard)?op_wb:6'd0;
  assign wb_x    = x;
  assign wb_y    = y;

endmodule
`default_nettype wire
