/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module rw_ucpu_alu
(
  /* system */
  input  wire        rst_n,
  input  wire        clk,
  
  /* operation */
  input  wire        clear,
  input  wire        sel_mul,
  input  wire        sel_div,
  input  wire        sel_add,
  input  wire        sel_sub,
 
  /* operands */
  input  wire [31:0] i,
  
  /* results */
  output wire        stall,       
  output reg  [31:0] x,
  output reg  [31:0] y
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  reg  [ 4:0] count;
  reg         mul,div;
  reg  [31:0] b_h;
  reg         div16,mul8;
  reg         busy;
  
  wire [31:0] n_y;
  wire [31:0] n_x;
  wire [31:0] n_b_h;
  wire        s_carry,s_unconnected;
  wire [31:0] s,s_a,s_b;  
  wire [ 4:0] n_count,count_max;
  wire        sel_sub_div;
  wire        n_div,n_mul;
  wire        n_div16,n_mul8;
  
  /* next y regsiter value */
  assign n_y   =  
    /* no-op, hold current value of y */
    ({32{~busy&~(sel_add|sel_sub|sel_mul|sel_div)}}  & y) | 
    /* div init, keep y as dividend/reminder, short dividend */
    ({32{~busy&sel_div&n_div16}}                     & {y[15:0],16'b0}) | 
    /* div init, keep y as dividend/reminder, long dividend */
    ({32{~busy&sel_div&~n_div16}}                    & y) | 
    /* add/sub/product, store the arith result */ 
    ({32{~busy&(sel_add|sel_sub) | mul}} & s) |      
    /* during divide, dividend/remind is pulled from left, quotient is pushed from right */   
    ({32{div}}                                           & {y[30:0],~s_carry});

  /* next x regsiter value */
  assign n_x   = 
    /* no-op, hold current value of x */
    ({32{~busy&~(sel_mul|sel_div|sel_add|sel_sub)}} & x) |
    /* mult init : load input as operands */
    ({32{~busy&sel_mul}}                   & y) |
    /* mult : shift left operand to be accumulated */ 
    ({32{mul}}                             & {x[30:0],1'b0}) |                         
    /* div  : reminder >=0, store the subtract result */
    ({32{div & ~s_carry}}                  & s) |            
    /* div  : reminder <0, keep subtractend operand */
    ({32{div &  s_carry}}                  & s_a);
                   
  /* next value of b register */
  assign n_b_h =
    /* mul/div init : b_h loaded with input operand */
    ({32{~busy&(sel_mul|sel_div)}}                   & i) |
    /* mul on-going: b_h is shifted right to scan 2**count partial product */                  
    ({32{mul}}                                       & {1'b0,b_h[31:1]}) |
    /* div on-going: b_h holds the divisor */
    ({32{div}}                                       & b_h);
 
  /* arith operand a */
  assign s_a   =  
    /* add/sub/multiply: y is the first operand */
    ({32{~busy&(sel_add|sel_sub) | mul}}                 & y) |  
    /* div : shifted dividend */
    ({32{div}}                                           & {x[30:0],y[31]});
                    
  /* arith operand b */
  assign s_b   =
    /* add/sub : input is the second operand */
    ({32{~busy&(sel_add|sel_sub)}}                       & i) |
    /* mult :  x is the partial 2**count product of input operand */
    ({32{mul & b_h[0]}}                                  & x) |
    /* div : divisor is store in b_h */
    ({32{div}}                                           & b_h);
    
  /* added is configured as subtract for div/sub operations  */
  assign sel_sub_div               = sel_sub | div;
  assign {s_carry,s,s_unconnected} = {1'b0,s_a,1'b1} + {{1'b0,s_b}^{33{sel_sub_div}},sel_sub_div};
  
  /* number of required cycles for sequential div/mul operations */
  /* full mul/div                                  32 cycles     */
  /* mul with short input operands (8bits wide)     8 cycles     */  
  /* div with short dividend operand (16bits wide) 16 cycles     */    
  assign count_max    = (5'd31 & {5{div&~div16 | mul&~mul8}}) |
                        (5'd15 & {5{div&div16}}) |
                        (5'd7  & {5{mul&mul8}});  
  
  /* next value of counter, is reset when starting div or mult */
  assign n_count      = {5{mul|div}} & (count + {4'b0,busy});               
  
  
  /* retain operation state */
  assign n_mul        = ~busy&sel_mul | mul & ~(count==count_max);                 
  assign n_div        = ~busy&sel_div | div & ~(count==count_max);                 
  assign stall        = n_mul | n_div;

  /* speed-up divide operation if dividend is 16 bits wide */
  assign n_div16      = (~busy & sel_div & y[31:16]==16'd0) | (div16 & ~(count==count_max)); 
  /* speed-up multiply operation if multiplier is 8 bits wide */
  assign n_mul8       = (~busy & sel_mul & i[31: 8]==24'b0) | (mul8 & ~(count==count_max));
  
  /* registers */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      x       <= 32'b0;
      y       <= 32'b0;
      b_h     <= 32'b0;
      count   <=  5'b0;
      mul     <=  1'b0;
      div     <=  1'b0;
      mul8    <=  1'b0;
      div16   <=  1'b0;
      busy    <=  1'b0;
    end
    else if(clear)
    begin
      x       <= 32'b0;
      y       <= 32'b0;
      b_h     <= 32'b0;
      count   <=  5'b0;
      mul     <=  1'b0;
      div     <=  1'b0;
      mul8    <=  1'b0;
      div16   <=  1'b0;
      busy    <=  1'b0;
    end
    else
    begin
      x       <= n_x;
      y       <= n_y;
      b_h     <= n_b_h;
      count   <= n_count;
      mul     <= n_mul;
      div     <= n_div;
      mul8    <= n_mul8;
      div16   <= n_div16;
      busy    <= stall;
    end
  end
  
endmodule
`default_nettype wire
