////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of TKIP MIC module.
//                    This module compute the TKIP MIC of the provided 32 bits
//                    words.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module tkip_mic( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          rst_n,                   // Asynchronous Reset
 
   //$port_g Control interface
   input  wire          tkip_init,               // Initialisation
   input  wire   [63:0] tkip_key,                // TKIP MIC Key
   input  wire   [31:0] tkip_data,               // Message word
   input  wire    [3:0] tkip_data_byte_en,       // Message word byte enable 
   input  wire          tkip_data_last,          // Last data to compute
   input  wire          tkip_data_valid,         // Data is valid
   output wire          tkip_data_ready,         // Ready to load data
   output wire   [63:0] tkip_mic,                // MIC
   output wire          tkip_mic_valid           // MIC qualifier
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// FSM states definition
//$fsm_sd padding
localparam 
   PAD_IDLE = 2'd0,
   PAD_WORD = 2'd1,
   PAD_ZERO = 2'd2;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg  [31:0] m;           // Mi
reg         m_valid;     // Mi is valid
wire        m_ready;     // ready to load Mi

wire [31:0] l;           // Michael l
wire [31:0] r;           // Michael r
wire        lr_valid;    // Michael l & r qualifier

reg   [1:0] padding_cs;  // FSM current state
reg   [1:0] padding_ns;  // FSM next state

reg  [31:0] padded_data; // data with padding bytes
reg  [31:0] masked_data; // data with invalid bytes masked


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// Output assignement
//******************************************************************************
assign tkip_mic        = {r,l};
assign tkip_mic_valid  = (padding_cs==PAD_IDLE) ? lr_valid : 1'b0;
assign tkip_data_ready = (padding_cs==PAD_IDLE) ? m_ready  : 1'b0;


//******************************************************************************
// Michael data control
//   The message is padded at the end with a single octet with value 0x5a, 
//   followed by between 4 and 7 zero octets. The number of zero octets is chosen
//   so that the overall length of the padded message is a multiple of four.
//******************************************************************************
always @*
begin: TKIP_PADDING
   // Local variable
   reg [31:0] v_padding;

   case (tkip_data_byte_en)
   4'b0000: v_padding = 32'h5a;
   4'b0001: v_padding = 32'h5a<<8;
   4'b0011: v_padding = 32'h5a<<16;
   4'b0111: v_padding = 32'h5a<<24;
   default: v_padding = 32'h0;
   endcase

   masked_data[24 +: 8] = tkip_data[24 +: 8] & {8{tkip_data_byte_en[3]}};
   masked_data[16 +: 8] = tkip_data[16 +: 8] & {8{tkip_data_byte_en[2]}};
   masked_data[8  +: 8] = tkip_data[8  +: 8] & {8{tkip_data_byte_en[1]}};
   masked_data[0  +: 8] = tkip_data[0  +: 8] & {8{tkip_data_byte_en[0]}};

   padded_data = masked_data | v_padding;

   case (padding_cs)
   PAD_IDLE:
   begin
      if (tkip_data_last)
         m    = padded_data;
      else
         m    = tkip_data;

      m_valid = tkip_data_valid;
   end
   PAD_WORD:
   begin
      m       = 32'h5a;
      m_valid = 1'b1;
   end
   default: //PAD_ZERO
   begin
      m       = 32'h00;
      m_valid = 1'b1;
   end
   endcase
end


//******************************************************************************
// FSM current state logic
//******************************************************************************
always @(posedge clk or negedge rst_n) 
begin
  if (~rst_n)
     padding_cs <= PAD_IDLE;
  else
     padding_cs <= padding_ns;
end


//******************************************************************************
// FSM next state logic
//******************************************************************************
always @* 
begin  
   case(padding_cs)
   PAD_IDLE:
   begin
      //$fsm_s In PAD_IDLE state, the state machine waits last data
      if (tkip_data_valid & tkip_data_ready &
          tkip_data_last  & tkip_data_byte_en[3])
         //$fsm_t When last data is complete, the state machine goes to PAD_WORD
         //state
         padding_ns = PAD_WORD;
      else if (tkip_data_valid & tkip_data_ready & 
               tkip_data_last  & ~tkip_data_byte_en[3])
         //$fsm_t When last data is not complete, the state machine goes to
         //PAD_ZERO state
         padding_ns = PAD_ZERO;
      else
         //$fsm_t While last data is not received, the state machine stays in
         //PAD_IDLE state
         padding_ns = PAD_IDLE;
   end

   PAD_WORD:
   begin
      //$fsm_s In PAD_WORD state, the state machine waits Michael ready
      if (m_ready)
         //$fsm_t When Michael is ready, the state machine goes to PAD_ZERO state
         padding_ns = PAD_ZERO;
      else
         //$fsm_t While Michael is not ready, the state machine stays in PAD_WORD
         //state
         padding_ns = PAD_WORD;
   end

   PAD_ZERO:
   begin
      //$fsm_s In PAD_ZERO state, the state machine waits Michael ready
      if (m_ready)
         //$fsm_t When Michael is ready, the state machine goes to PAD_IDLE state
         padding_ns = PAD_IDLE;
      else
         //$fsm_t While Michael is not ready, the state machine stays in PAD_ZERO
         //state
         padding_ns = PAD_ZERO;
   end

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      padding_ns = PAD_IDLE;
   // pragma coverage block = on
   endcase
end


//******************************************************************************
// Michael message processing
//******************************************************************************
tkip_michael u_tkip_michael( 
   // Clock and reset
   .clk        (clk             ),
   .rst_n      (rst_n           ),
   // Control interface
   .init       (tkip_init       ),
   .k0         (tkip_key[31:0]  ),
   .k1         (tkip_key[63:32] ),
   .m          (m               ),
   .m_valid    (m_valid         ),
   .m_ready    (m_ready         ),
   .l          (l               ),
   .r          (r               ),
   .lr_valid   (lr_valid        ) 
);

                 
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] padding_cs_str;
always @*
begin
   case (padding_cs)
   PAD_IDLE : padding_cs_str = {"PAD_IDLE"};
   PAD_WORD : padding_cs_str = {"PAD_WORD"};
   PAD_ZERO : padding_cs_str = {"PAD_ZERO"};
   default  : padding_cs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON

`endif // RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
