////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Michael module.
//                    Implement the Michael message processing of the TKIP MIC
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module tkip_michael( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          rst_n,                   // Asynchronous Reset
 
   //$port_g Control interface
   input  wire          init,                    // Initialization
   input  wire   [31:0] k0,                      // k0
   input  wire   [31:0] k1,                      // k1
   input  wire   [31:0] m,                       // message word
   input  wire          m_valid,                 // m is valid
   output wire          m_ready,                 // ready to load m
   output reg    [31:0] l,                       // l
   output reg    [31:0] r,                       // r
   output reg           lr_valid                 // l & r qualifier
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg  [2:0] phase;      // Processing Phase
reg [31:0] next_l;     // next value of l
reg [31:0] next_r;     // next value of r
reg  [2:0] next_phase; // next value of phase


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// Ready to load M when processing is IDLE (phase 0)
//******************************************************************************
assign m_ready = (phase==3'd0) ? 1'b1 : 1'b0;


//******************************************************************************
// Michael message processing
//******************************************************************************
always @*
begin
   // Initialize r,l registers
   if (init)
   begin
      next_r     = k1;
      next_l     = k0;
      next_phase = 3'd0;
   end

   // Load word & Start Michael Processing
   // l = l ^ Mi
   else if (m_valid & phase==3'd0)
   begin
      next_r     = r;
      next_l     = l ^ m;
      next_phase = 3'd4;
   end

   // Michael block processing
   // Phase 4: r ^ l<<<17
   //          l + r
   // Phase 3: r ^ XSWAP(l)
   //          l + r
   // Phase 2: r ^ l<<<3
   //          l + r
   // Phase 1: r ^ l>>>2
   //          l + r
   else if (phase!=3'd0)
   begin
      case (phase)
      3'd4   : next_r = r ^ {l[14:0],l[31:15]};
      3'd3   : next_r = r ^ {l[23:16],l[31:24],l[7:0],l[15:8]};
      3'd2   : next_r = r ^ {l[28:0],l[31:29]};
      default: next_r = r ^ {l[1:0],l[31:2]};
      endcase
      next_l     = l + next_r;
      next_phase = phase - 3'd1;
   end

   else
   begin
      next_r     = r;
      next_l     = l;
      next_phase = phase;
   end
end


//******************************************************************************
// Registers
//******************************************************************************
always @(posedge clk or negedge rst_n)
begin
   if (~rst_n)
   begin
      r         <= 32'h0;
      l         <= 32'h0;
      phase     <= 3'd0;
      lr_valid  <= 1'b0;
   end
   else
   begin
      r         <= next_r;
      l         <= next_l;
      phase     <= next_phase;
      lr_valid  <= (phase==3'd1) ? 1'b1 : 1'b0;
   end
end

endmodule
                 
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
