//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//   This module converts a 8/16/19 byte seed into a arbitarily long sequence
//   of pseudo random bytes. It uses RSA RC4 algorithm to generate the pseudo 
//   random bytes. It generates one pseudo random byte every 3 clocks. The cipher 
//   strength is selected on the basis of signal currCipherLen.
//   The initialisation (init) process takes 128 clocks to complete, after which 
//   the state machine waits for seedValid to be asserted. The transformation
//   process takes 768 clocks to complete, after which the state machine waits for
//   prnStrobe_p to be asserted. For every prnStrobe_p, one pseudo random byte is 
//   output with prnOutValid_p being asserted.
//   The state machine returns to init when initPRNG_p is asserted, and immediately
//   starts its init process.
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module prng(

  //$port_g Input ports
  input  wire         hardRstBbClk_n,     // reset signal from hardware
  input  wire         softRstBbClk_p,     // reset signal from software
  input  wire         bbClk,              // baseband clock signal
  input  wire         currCipherLen,      // signal to select encryption strength
  input  wire [151:0] seed,               // signal indicating validity of seed
  input  wire         seedValid,          // 152/ 128/ 64 bit seed used by PRNG
  input  wire         initPRNG_p,         // init signal, used to start init process
  input  wire         prnStrobe_p,        // strobe for pseudo random bytes to be generated
  input  wire   [15:0] MemReadData,      // data bus from internal RAM
  input  wire         rxError_p,          // signal to indicate error in reception 
  input  wire   [1:0] wt2CryptClkRatio,   // Indicate the clock ratio between WT and macCrypt Clock

  //$port_g Output ports 
  output reg         initDone_p,          // signal indicating end of init process
  output reg         initSBoxIndexDone_p, // signal indicating end of SBOX Index init process
  output wire        prngIsIdle,          // signal indicating that the FSM is in IDLE state
  output reg   [7:0] prnOut,              // output lines
  output wire        prnOutValid_p,       // indicates validity of the pseudo random byte on prnOut

  output wire   [6:0] MemReadAddr,           // address bus to read address port RAM
  output wire   [6:0] MemWriteAddr,           // address bus to write address port RAM
  output wire         MemReadEn,             // read enable signal for internal RAM
  output wire   [1:0] MemWriteEn,             // write enable signal for bit
  output wire  [15:0] MemWriteData,      // data bus to internal RAM

`ifdef SBOX_RDATA_CAPT
  output reg   [5:0] prngCs 
`else
  output reg   [4:0] prngCs 
`endif
);



//--------------------- internal registers -------------------

`ifdef SBOX_RDATA_CAPT
reg [5:0] prngNs;
`else
reg [4:0] prngNs;
`endif
reg [7:0] index_i;
reg [7:0] nextI;
reg [7:0] index_j;
reg [7:0] index_jD1;
reg [7:0] index_jD2;
`ifdef SBOX_RDATA_CAPT
reg [7:0] index_jD3;
reg [7:0] index_jD4;
`endif   
reg [7:0] nextJ;
reg [7:0] sBoxI;
reg [7:0] nextSBoxI;
reg [7:0] sBoxJ;
reg [7:0] nextSBoxJ;
reg [7:0] sBoxAAddr;
reg [7:0] sBoxBAddr;
reg       initStatus;
reg       nextInitStatus;
reg [4:0] seedPart;
reg [4:0] nextSeedPart;
reg [1:0] clkCounter2;
reg [6:0] MemReadAddrD1;
reg       MemReadAddrD1ByteSel;
reg       MemReadEnD1;
reg    prnOutValid_p_int;
   
   reg [7:0] DataReadIndexI, DataReadIndexISave;
   reg [7:0] DataReadIndexIP1, DataReadIndexIP1Save;
   reg [7:0] DataReadIndexJ, DataReadIndexJSave;
   reg [7:0] DataReadIndexJ2, DataReadIndexJ2Save;

   reg [7:0] DataReadForOut, DataReadForOutSave;
   
   reg       conflict_read_j_same_as_ip1_D1 ;
   reg       conflict_read_j2_same_as_j_D1  ;
   reg       conflict_read_j2_same_as_i_D1;
   reg       conflict_read_j_same_as_ip1_D2 ;
   reg       conflict_read_j2_same_as_j_D2  ;
   reg       conflict_read_j2_same_as_i_D2;
   reg       conflict_read_j_same_as_ip1_D3 ;

   reg    conflict_read_j_same_as_ip1, conflict_read_j2_same_as_j, conflict_read_j2_same_as_i;
   reg    conflict_read_out_same_as_i, conflict_read_out_same_as_ip1, conflict_read_out_same_as_j;
   reg    conflict_read_out_same_as_i_D1, conflict_read_out_same_as_ip1_D1, conflict_read_out_same_as_j_D1;
   
   
   
reg     data_mem_is_i_and_ip1;
reg     data_mem_is_j        ;
reg     data_mem_is_j2       ;
reg     data_mem_is_i        ;
reg     data_mem_is_out      ;

reg  sBoxAEn;
reg  sBoxBEn;
reg  [1:0] sBoxBWriteEn;
reg [15:0] sBoxBWriteData;


      
`ifdef RW_SIMU_ON
// String definition to display txControlMainFSM current state
reg [30*8:0] prngCs_str;
`endif // RW_SIMU_ON


//--------------------- prngCs declarations -------------------
  
`ifdef SBOX_RDATA_CAPT
//$fsm_sd prngCs
localparam
  IDLE                         = 6'd0, // Do nothing
  SBOX_INDEX_INIT              = 6'd1, // Initialise sboxdpram with index values
  WAIT_FOR_SEED_VALID          = 6'd2, // Wait until seedValid signal goes high
  SBOX_TRN_READ_I_AND_IP1      = 6'd3, // Generate read signals for var i and i+1
  WAIT_SBOX_TRN_READ_I_AND_IP1 = 6'd23, // Wait read signals for var i and i+1
  SBOX_TRN_READ_J              = 6'd4, // Generate read signals for var j
  WAIT_SBOX_TRN_READ_J         = 6'd24, // Wait read signals for var j
  SBOX_TRN_READ_J2             = 6'd5, // Generate write signals for vars j2
  WAIT_SBOX_TRN_READ_J2        = 6'd25, // Wait read signals for var j2
  SBOX_TRN_WRITE_I_AND_IP1     = 6'd6, // Generate write signals for vars i and j
  SBOX_TRN_WRITE_J             = 6'd7, // Generate write signals for var j
  SBOX_TRN_WRITE_J2            = 6'd8, // Generate write signals for var j2
  PRN_BYTE_READ_ADD1           = 6'd9, // Generate read signals for addr
  WAIT_PRN_BYTE_READ_ADD1      = 6'd26, // Generate read signals for addr
  PRN_BYTE_READ_ADD1OFFSET     = 6'd10, // Generate read signals for addr
  WAIT_PRN_BYTE_READ_ADD1OFFSET= 6'd27, // Generate read signals for addr
  PRN_BYTE_WRITE_ADD1          = 6'd11, // Generate read signals for addr
  PRN_BYTE_WRITE_ADD1OFFSET    = 6'd12, // Generate read signals for addr
  PRN_BYTE_READ_I_AND_IP1      = 6'd13, // Generate read signals for var i and i+1
  WAIT_PRN_BYTE_READ_I_AND_IP1 = 6'd28, // Wait signals for var i and i+1
  PRN_BYTE_READ_J              = 6'd14, // Generate read signals for var j
  WAIT_PRN_BYTE_READ_J         = 6'd29, // Generate read signals for var i
  PRN_BYTE_READ_J2             = 6'd15, // Generate read signals for var j
  WAIT_PRN_BYTE_READ_J2        = 6'd30, // Generate read signals for var i
  PRN_B_WR_I_N_IP1_N_RD_IPJ    = 6'd16, // Generate write signals for vars i and j and read result sbox IplusJ
  WAIT_PRN_B_WR_I_N_IP1_N_RD_IPJ= 6'd31, // wait signals for read result sbox IplusJ
  PRN_BYTE_WRITE_J_AND_RD_IPJ  = 6'd17, // Generate write signals for vars i and j and read result sbox IplusJ
  WAIT_PRN_BYTE_WRITE_J_AND_RD_IPJ  = 6'd32, // wait read result sbox IplusJ
  PRN_BYTE_WRITE_J2            = 6'd18, // Generate write signals for vars i and j
  WAIT_FOR_STROBE              = 6'd19, // Wait until strobe pulse comes for data
  DELAY_PRN_OUT1               = 6'd20, // Delay the PRN output by one clock
  DELAY_PRN_OUT2               = 6'd21, // Delay the PRN output by two clocks
  DELAY_PRN_OUT3               = 6'd22; // Delay the PRN output by three clocks
`else
localparam
  IDLE                         = 5'd0, // Do nothing
  SBOX_INDEX_INIT              = 5'd1, // Initialise sboxdpram with index values
  WAIT_FOR_SEED_VALID          = 5'd2, // Wait until seedValid signal goes high
  SBOX_TRN_READ_I_AND_IP1      = 5'd3, // Generate read signals for var i and i+1
  SBOX_TRN_READ_J              = 5'd4, // Generate read signals for var j
  SBOX_TRN_READ_J2             = 5'd5, // Generate write signals for vars j2
  SBOX_TRN_WRITE_I_AND_IP1     = 5'd6, // Generate write signals for vars i and j
  SBOX_TRN_WRITE_J             = 5'd7, // Generate write signals for var j
  SBOX_TRN_WRITE_J2            = 5'd8, // Generate write signals for var j2
  PRN_BYTE_READ_ADD1           = 5'd9, // Generate read signals for addr
  PRN_BYTE_READ_ADD1OFFSET     = 5'd10, // Generate read signals for addr
  PRN_BYTE_WRITE_ADD1          = 5'd11, // Generate read signals for addr
  PRN_BYTE_WRITE_ADD1OFFSET    = 5'd12, // Generate read signals for addr
  PRN_BYTE_READ_I_AND_IP1      = 5'd13, // Generate read signals for var i and i+1
  PRN_BYTE_READ_J              = 5'd14, // Generate read signals for var j
  PRN_BYTE_READ_J2             = 5'd15, // Generate read signals for var j
  PRN_B_WR_I_N_IP1_N_RD_IPJ    = 5'd16, // Generate write signals for vars i and j
  PRN_BYTE_WRITE_J_AND_RD_IPJ  = 5'd17, // Generate write signals for vars i and j
  PRN_BYTE_WRITE_J2            = 5'd18, // Generate write signals for vars i and j
  WAIT_FOR_STROBE              = 5'd19, // Wait until strobe pulse comes for data
  DELAY_PRN_OUT1               = 5'd20, // Delay the PRN output by one clock
  DELAY_PRN_OUT2               = 5'd21, // Delay the PRN output by two clocks
  DELAY_PRN_OUT3               = 5'd22; // Delay the PRN output by three clocks
`endif




wire   prnStrobe_p_raising, prnOutValid_p_raising;
reg    prnStrobe_p_d1, prnOutValid_p_d1, prnOutValid_p_d2;   

reg       prnStrobe_p_filter;

wire [3:0] prnOutBuffer_lvl;
   
   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    prnStrobe_p_filter <= 1'b1;
  else if (softRstBbClk_p == 1'b1)
    prnStrobe_p_filter <= 1'b1;
  else
    if (prnOutBuffer_lvl[2] == 1'b1) // Buffer fifo out is almost full...
      prnStrobe_p_filter <= 1'b0;
    else
      prnStrobe_p_filter <= 1'b1;
   
                          

assign prngIsIdle = (prngCs == IDLE);



always @(*)
  begin
  case (prngCs)

    IDLE:
      //$fsm_s In IDLE state, the prng waits for 1 clock cycle
      
      //$fsm_t After 1 clock cycle, the state goes to SBOX_INDEX_INIT state
      prngNs = SBOX_INDEX_INIT;

    SBOX_INDEX_INIT:  
      //$fsm_s In SBOX_INDEX_INIT, the prgn is itinializing the SBOX memories from 0 to 255 
      if (index_i[7:0] == 8'hFE && seedValid == 1'b1)
        //$fsm_t When SBOX is initialized and a seed is valid, the state goes to SBOX_TRN_READ_I state
        prngNs = SBOX_TRN_READ_I_AND_IP1;

      else if (index_i[7:0] == 8'hFE && seedValid == 1'b0)
        //$fsm_t When SBOX is initialized and a seed is not valid, the state goes to WAIT_FOR_SEED_VALID state
        prngNs = WAIT_FOR_SEED_VALID;

      else
        //$fsm_t When SBOX is not initialized, the state stays in SBOX_INDEX_INIT state
        prngNs = SBOX_INDEX_INIT;

    WAIT_FOR_SEED_VALID:
      //$fsm_s In WAIT_FOR_SEED_VALID, the prgn waits until seedValid signal is given before proceeding with transformations on sBox
      if (seedValid == 1'b1)
        //fsm_t If seed is valid, the state goes to SBOX_TRN_READ_I state
        prngNs = SBOX_TRN_READ_I_AND_IP1;
      else
        //fsm_t If seed is not valid, the state waits in WAIT_FOR_SEED_VALID state
        prngNs = WAIT_FOR_SEED_VALID;


    SBOX_TRN_READ_I_AND_IP1:
      //$fsm_s In SBOX_TRN_READ_I_AND_IP1, the prgn read the bytes at index I and I+1
      
`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_SBOX_TRN_READ_I_AND_IP1 state
      prngNs = WAIT_SBOX_TRN_READ_I_AND_IP1;

    WAIT_SBOX_TRN_READ_I_AND_IP1:
      //$fsm_s In WAIT_SBOX_TRN_READ_I_AND_IP1, the prgn wait to read the bytes at index I and I+1
`endif

      //$fsm_t After one clock cycle, the state goes to SBOX_TRN_READ_J state
      prngNs = SBOX_TRN_READ_J;

    SBOX_TRN_READ_J:
      //$fsm_s In SBOX_TRN_READ_J, the prgn read the byte at index J

`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_SBOX_TRN_READ_J state
      prngNs = WAIT_SBOX_TRN_READ_J;

    WAIT_SBOX_TRN_READ_J:
      //$fsm_s In WAIT_SBOX_TRN_READ_J, the prgn wait to read the bytes at index J
`endif
      
      //$fsm_t After one clock cycle, the state goes to SBOX_TRN_READ_J2 state
      prngNs = SBOX_TRN_READ_J2;

    SBOX_TRN_READ_J2:
      //$fsm_s In SBOX_TRN_READ_J2, the prgn read the byte at index J2
      
`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_SBOX_TRN_READ_J2 state
      prngNs = WAIT_SBOX_TRN_READ_J2;

    WAIT_SBOX_TRN_READ_J2:
      //$fsm_s In WAIT_SBOX_TRN_READ_J2, the prgn wait to read the bytes at index J2
`endif
      
      //$fsm_t After one clock cycle, the state goes to SBOX_TRN_READ_J2 state
      prngNs = SBOX_TRN_WRITE_I_AND_IP1;


    SBOX_TRN_WRITE_I_AND_IP1:
      //$fsm_s In SBOX_TRN_WRITE_I_AND_IP1, the prgn writes sBox[j] to sBox[i] and sBox[j2] to sBox[i+1]

      //$fsm_t After one clock cycle, the state goes to SBOX_TRN_READ_J2 state
      prngNs = SBOX_TRN_WRITE_J;

    SBOX_TRN_WRITE_J:
      //$fsm_s In SBOX_TRN_WRITE_J, the prgn writes sBox[i] to sBox[j]

      //$fsm_t After one clock cycle, the state goes to SBOX_TRN_WRITE_J2 state
      prngNs = SBOX_TRN_WRITE_J2;
      
    
    SBOX_TRN_WRITE_J2:
      //$fsm_s In SBOX_TRN_WRITE_I_AND_IP1, the prgn writes sBox[i+1] to sBox[j2]
      if (index_i[7:0] == 8'hFE)
        //$fsm_t If all SBOX has been swapt, the state goes to PRN_BYTE_READ_I state
        prngNs = PRN_BYTE_READ_ADD1;
      else
        //$fsm_t If not all SBOX has been swapt, the state goes to SBOX_TRN_READ_I_AND_IP1 state
        prngNs = SBOX_TRN_READ_I_AND_IP1;


    PRN_BYTE_READ_ADD1:
      //$fsm_s In PRN_BYTE_READ_ADD1, the prgn read Addr 1

      
`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_PRN_BYTE_READ_ADD1 state
      prngNs = WAIT_PRN_BYTE_READ_ADD1;

    WAIT_PRN_BYTE_READ_ADD1:
      //$fsm_s In WAIT_PRN_BYTE_READ_ADD1, the prgn wait to read the bytes at index 1
`endif

     //$fsm_t After one clock cycle, the state goes to PRN_BYTE_READ_ADD1OFFSET state
      prngNs = PRN_BYTE_READ_ADD1OFFSET;

    PRN_BYTE_READ_ADD1OFFSET:
      //$fsm_s In PRN_BYTE_READ_I, the prgn read the byte at index provided by data at addr 1

`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_PRN_BYTE_READ_ADD1OFFSET state
      prngNs = WAIT_PRN_BYTE_READ_ADD1OFFSET;

    WAIT_PRN_BYTE_READ_ADD1OFFSET:
      //$fsm_s In WAIT_PRN_BYTE_READ_ADD1OFFSET, the prgn wait to read the byte at index provided by data at addr 1
`endif
 
      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_WRITE_ADD1 state
      prngNs = PRN_BYTE_WRITE_ADD1;

    PRN_BYTE_WRITE_ADD1:
      //$fsm_s In PRN_BYTE_WRITE_ADD1, the prgn writes at addr 1 data coming from the pointer of addr 1

      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_WRITE_ADD1OFFSET state
      prngNs = PRN_BYTE_WRITE_ADD1OFFSET;
      
    PRN_BYTE_WRITE_ADD1OFFSET:
      //$fsm_s In PRN_BYTE_WRITE_ADD1OFFSET, the prgn writes at the pointer provided on addr 1, the data of addr 1

      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_READ_I_AND_IP1 state
      prngNs = PRN_BYTE_READ_I_AND_IP1;
    
    PRN_BYTE_READ_I_AND_IP1:
      //$fsm_s In PRN_BYTE_READ_I, the prgn read the byte at index I and index I+1

`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_PRN_BYTE_READ_I state
      prngNs = WAIT_PRN_BYTE_READ_I_AND_IP1;

    WAIT_PRN_BYTE_READ_I_AND_IP1:
      //$fsm_s In WAIT_PRN_BYTE_READ_I_AND_IP1, the prgn read the byte at index I and index I+1
`endif

      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_READ_J state
      prngNs = PRN_BYTE_READ_J;

    PRN_BYTE_READ_J:
      //$fsm_s In PRN_BYTE_READ_J, the prgn read the byte at index J

`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_PRN_BYTE_READ_J state
      prngNs = WAIT_PRN_BYTE_READ_J;

    WAIT_PRN_BYTE_READ_J:
      //$fsm_s In WAIT_PRN_BYTE_READ_J, the prgn read the byte at index J
`endif

      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_READ_J2 state
      prngNs = PRN_BYTE_READ_J2;

    PRN_BYTE_READ_J2:
      //$fsm_s In PRN_BYTE_READ_J2, the prgn read the byte at index J2

`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_PRN_BYTE_READ_J2 state
      prngNs = WAIT_PRN_BYTE_READ_J2;

    WAIT_PRN_BYTE_READ_J2:
      //$fsm_s In WAIT_PRN_BYTE_READ_J2, the prgn read the byte at index J2
`endif

      //$fsm_t After one clock cycle, the state goes to PRN_B_WR_I_N_IP1_N_RD_IPJ state
      prngNs = PRN_B_WR_I_N_IP1_N_RD_IPJ;

    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      //$fsm_s In PRN_B_WR_I_N_IP1_N_RD_IPJ, the prgn writes sBox[j] to sBox[i] and sBox[j2] to sBox[i+1], and at same times read out the sbox i + j
      
`ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_SBOX_TRN_READ_I_AND_IP1 state
      prngNs = WAIT_PRN_B_WR_I_N_IP1_N_RD_IPJ;

    WAIT_PRN_B_WR_I_N_IP1_N_RD_IPJ:
      //$fsm_s In WAIT_SBOX_TRN_READ_I_AND_IP1, the prgn wait to read the bytes at index I and I+1
`endif

      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_WRITE_J_AND_RD_IPJ state
      prngNs = PRN_BYTE_WRITE_J_AND_RD_IPJ;

    PRN_BYTE_WRITE_J_AND_RD_IPJ:
      //$fsm_s In PRN_BYTE_WRITE_J_AND_RD_IPJ, the prgn writes sBox[i] to sBox[j] and , and at same times read out the sbox (i+1) + j2

 `ifdef SBOX_RDATA_CAPT
      //$fsm_t After one clock cycle, the state goes to WAIT_SBOX_TRN_READ_I_AND_IP1 state
      prngNs = WAIT_PRN_BYTE_WRITE_J_AND_RD_IPJ;

    WAIT_PRN_BYTE_WRITE_J_AND_RD_IPJ:
      //$fsm_s In WAIT_SBOX_TRN_READ_I_AND_IP1, the prgn wait to read the bytes at index I and I+1
`endif
     //$fsm_t After one clock cycle, the state goes to PRN_BYTE_WRITE_J_AND_RD_IPJ state
      prngNs = PRN_BYTE_WRITE_J2;


    PRN_BYTE_WRITE_J2:  
      //$fsm_s In PRN_BYTE_WRITE_J2, the prgn writes sBox[i+1] & sBox[j2]
      if (prnOutBuffer_lvl[2] == 1'b0)
        if (wt2CryptClkRatio == 2'd3)
          //$fsm_t If the xored data byte is strobed and  the WEP/TKIP clock ratio is 3, the state goes to DELAY_PRN_OUT1 state
          prngNs = DELAY_PRN_OUT1;
        else
          //$fsm_t If the xored data byte is strobed and  the WEP/TKIP clock ratio is 1 or 2, the state goes to PRN_BYTE_READ_I_AND_IP1 state
          prngNs = PRN_BYTE_READ_I_AND_IP1;
      else
        //$fsm_t If the xored data byte is not strobed, the state goes to WAIT_FOR_STROBE state
        prngNs = WAIT_FOR_STROBE;

    WAIT_FOR_STROBE:
      //$fsm_s In WAIT_FOR_STROBE, the prgn wait for prnStrobe_p
      if (prnOutBuffer_lvl[2] == 1'b0)
        if (wt2CryptClkRatio == 2'd3)
          //$fsm_t If the xored data byte is strobed and  the WEP/TKIP clock ratio is 3, the state goes to DELAY_PRN_OUT1 state
          prngNs = DELAY_PRN_OUT1;
        else
          //$fsm_t If the xored data byte is strobed and  the WEP/TKIP clock ratio is 1 or 2, the state goes to PRN_BYTE_READ_I_AND_IP1 state
          prngNs = PRN_BYTE_READ_I_AND_IP1;
      else
        //$fsm_t If the xored data byte is not strobed, the state stays in WAIT_FOR_STROBE state
        prngNs = WAIT_FOR_STROBE;

    DELAY_PRN_OUT1:
      //$fsm_s In DELAY_PRN_OUT1, the prgn wait for prgn byte generation
      
        //$fsm_t If the WEP/TKIP clock ratio is 3, the state goes to PRN_BYTE_READ_I_AND_IP1 state
        prngNs = PRN_BYTE_READ_I_AND_IP1;

    DELAY_PRN_OUT2:
      //$fsm_s In DELAY_PRN_OUT2, the prgn wait for prgn byte generation

      //$fsm_t After one clock cycle, the state goes to DELAY_PRN_OUT3 state
      prngNs = DELAY_PRN_OUT3;

    DELAY_PRN_OUT3:
      //$fsm_s In DELAY_PRN_OUT3, the prgn wait for prgn byte generation
     
      //$fsm_t After one clock cycle, the state goes to PRN_BYTE_READ_I_AND_IP1 state
      prngNs = PRN_BYTE_READ_I_AND_IP1;

    default:
      prngNs = IDLE;

    endcase
end

// build the prngCs flip flops

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    prngCs <= IDLE;
  else if (softRstBbClk_p == 1'b1)
    prngCs <= IDLE;
  else if (rxError_p)
    prngCs <= IDLE;
  else if (initPRNG_p == 1'b1)
    prngCs <= IDLE;
  else
    prngCs <= prngNs;


// logic for initDone_p
// when transformation phase is over, assert initDone_p. it should be asserted 
// as long as clkCounter is non zero.

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    initDone_p  <= 1'b0;
  else if (softRstBbClk_p == 1'b1)
    initDone_p  <= 1'b0;
    else if(prngCs == PRN_BYTE_WRITE_ADD1OFFSET && initStatus == 1'b0)                  
    initDone_p <= 1'b1;
  else
    initDone_p <= 1'b0;

// logic for initSBoxIndexDone_p
// when SBOX Index Initialisation is over, assert initSBoxIndexDone_p. 

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    initSBoxIndexDone_p  <= 1'b0;
  else if (softRstBbClk_p == 1'b1)
    initSBoxIndexDone_p  <= 1'b0;
  else if(prngNs == WAIT_FOR_SEED_VALID)
    initSBoxIndexDone_p <= 1'b1;
  else
    initSBoxIndexDone_p <= 1'b0;


// logic for prnOut
// since the enable to port B of sBox was sent in prngCs PRN_BYTE_READ_I
// data will be available in this prngCs. This is the PRN output.

reg [7:0] prnOutBuffer[7:0];
reg [2:0] prnOutBufferPtrRd, prnOutBufferPtrWr;
   
reg [1:0] cnt_strobe_delay;
reg [1:0] cnt_pending_strobe;

  
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    begin
       prnStrobe_p_d1 <= 1'b0;
       prnOutValid_p_d1 <= 1'b0;
       prnOutValid_p_d2 <= 1'b0;
    end
  else
    if (softRstBbClk_p == 1'b1)
      begin
         prnStrobe_p_d1 <= 1'b0;
         prnOutValid_p_d1 <= 1'b0;
         prnOutValid_p_d2 <= 1'b0;
      end
    else
      begin
         prnStrobe_p_d1   <= prnStrobe_p;
         prnOutValid_p_d1 <= prnOutValid_p_int;
         prnOutValid_p_d2 <= prnOutValid_p_d1;
      end
   
   assign prnStrobe_p_raising   = !prnStrobe_p_d1   & prnStrobe_p;
   assign prnOutValid_p_raising = !prnOutValid_p_d1 & prnOutValid_p_int;

`ifdef SBOX_RDATA_CAPT
   assign prnOutValid_p = prnOutValid_p_d2;
`else
   assign prnOutValid_p = prnOutValid_p_int;
`endif

   
                             
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    cnt_pending_strobe[1:0] <= 2'b0;
  else if (softRstBbClk_p == 1'b1)
    cnt_pending_strobe[1:0] <= 2'b0;
  else
    if(prngCs == SBOX_INDEX_INIT)                  
      cnt_pending_strobe[1:0] <= 2'b0;
    else
      if (prnOutValid_p_raising && prnStrobe_p_raising)
        // read and write request at the same time: status quo
        cnt_pending_strobe[1:0] <= cnt_pending_strobe[1:0];
      else
        if (prnStrobe_p_raising)
          // New pending request 
          cnt_pending_strobe[1:0] <= cnt_pending_strobe[1:0] + 2'd1;
        else
          if (prnOutValid_p_raising)
            // Read a value
            cnt_pending_strobe[1:0] <= cnt_pending_strobe[1:0] - 2'd1;
   

   
integer i;
always @(posedge bbClk or negedge hardRstBbClk_n)
 begin 
     if (hardRstBbClk_n == 1'b0)
       begin
          prnOutBufferPtrWr <= 3'b0;
          prnOutBufferPtrRd <= 3'b0;
          for (i=0;i<8;i=i+1)
            prnOutBuffer[i] <= 8'h0;
       end
     else
       if (softRstBbClk_p == 1'b1)
         begin
            prnOutBufferPtrWr <= 3'b0;
            prnOutBufferPtrRd <= 3'b0;
            for (i=0;i<8;i=i+1)
              prnOutBuffer[i] <= 8'h0;
         end
       else
         if (prngCs == SBOX_INDEX_INIT)
           begin
              prnOutBufferPtrWr <= 3'b0;
              prnOutBufferPtrRd <= 3'b0;
              for (i=0;i<8;i=i+1)
                prnOutBuffer[i] <= 8'h0;
           end
         else
           begin
              if (prnOutValid_p_raising) 
                prnOutBufferPtrRd <= prnOutBufferPtrRd + 3'd1;
              
              if ( (prngCs == PRN_BYTE_WRITE_J_AND_RD_IPJ && initStatus == 1'b1) ||
                   (prngCs == PRN_BYTE_WRITE_J2 && initStatus == 1'b1) )
                begin
                   prnOutBuffer[prnOutBufferPtrWr] <=  DataReadForOut[7:0];
                   prnOutBufferPtrWr <= prnOutBufferPtrWr + 3'd1;
                end
           end // if ( (initStatus == 1'b1) & (cnt_pending_strobe != 2'b00) )
 end


   
   assign prnOutBuffer_lvl = (prnOutBufferPtrWr >= prnOutBufferPtrRd) ? prnOutBufferPtrWr - prnOutBufferPtrRd :
                             {1'b1, prnOutBufferPtrWr} - {1'b0, prnOutBufferPtrRd};
   

   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    cnt_strobe_delay <= 2'h1;
  else if (softRstBbClk_p == 1'b1)
    cnt_strobe_delay <= 2'h1;
  else
    if (prngCs == SBOX_INDEX_INIT)
      cnt_strobe_delay <= 2'h1;
    else
      if (cnt_pending_strobe != 2'b00)
        begin
           if (cnt_strobe_delay == 2'd2)
             cnt_strobe_delay <= 2'h0;
           else
             cnt_strobe_delay <= cnt_strobe_delay + 2'h1;
        end

   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    prnOut[7:0] <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    prnOut[7:0] <= 8'b0;
  else if (initStatus == 1'b1)
    if (cnt_strobe_delay == 2)
      prnOut[7:0] <= prnOutBuffer[prnOutBufferPtrRd];
   
        
// logic for clkCounter2
// this counter counts the number of bb2Clk edges occuring for every bbClk.
// It is used to make the pulse width of pulses produced in bb2Clk domain
// equal to pulses produced in bbClk domain for correct sampling.

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (!hardRstBbClk_n)
    clkCounter2 <= 2'd0;
  else if (softRstBbClk_p)
    clkCounter2 <= 2'd0;
  else
    if ( ((initStatus == 1'b1) && (cnt_strobe_delay == 3'h1)) && (cnt_pending_strobe != 2'b00) && (prnOutBuffer_lvl != 0) )
      clkCounter2 <= wt2CryptClkRatio;
    else 
      if (clkCounter2 == 2'd0)
        clkCounter2 <= 2'd0;
      else
        clkCounter2 <= clkCounter2 - 2'd1;


// logic for prnOutValid_p
// prnOutValid_p needs to be sent when data is available from port B of the 
// sBox RAM. it should be asserted as long as clkCounter is non zero.
 
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    prnOutValid_p_int  <= 1'b0;
  else if (softRstBbClk_p == 1'b1)
   prnOutValid_p_int  <= 1'b0;
  else
    prnOutValid_p_int  <= (clkCounter2 != 2'd0);



`ifdef SBOX_RDATA_CAPT

   reg [15:0] MemReadDataCapt;
   reg        MemReadAddrD1ByteSelCapt;
   
   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    MemReadDataCapt  <= 16'b0;
  else if (softRstBbClk_p == 1'b1)
    MemReadDataCapt  <= 16'b0;
  else
    MemReadDataCapt <= MemReadData;

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    MemReadAddrD1ByteSelCapt  <= 16'b0;
  else if (softRstBbClk_p == 1'b1)
    MemReadAddrD1ByteSelCapt  <= 16'b0;
  else
    MemReadAddrD1ByteSelCapt <= MemReadAddrD1ByteSel;

`else

   wire [15:0] MemReadDataCapt = MemReadData;
   wire        MemReadAddrD1ByteSelCapt = MemReadAddrD1ByteSel;

`endif
   

/***************************************
 Put the incoming data into the right index pointer, and store it for later use if needed
*****************************************/
always @(*)
  begin
     data_mem_is_i_and_ip1  = 1'b0;
     data_mem_is_j          = 1'b0;
     data_mem_is_j2         = 1'b0;
     data_mem_is_i          = 1'b0;
     data_mem_is_out        = 1'b0;
     
     case (prngCs)
       SBOX_TRN_READ_J             : data_mem_is_i_and_ip1  =1'b1;
       SBOX_TRN_READ_J2            : data_mem_is_j          =1'b1;
       SBOX_TRN_WRITE_I_AND_IP1    : data_mem_is_j2         =1'b1;
       PRN_BYTE_READ_ADD1OFFSET    : data_mem_is_i          =1'b1;
       PRN_BYTE_WRITE_ADD1         : data_mem_is_j          =1'b1;
       PRN_BYTE_READ_J             : data_mem_is_i_and_ip1  =1'b1;
       PRN_BYTE_READ_J2            : data_mem_is_j          =1'b1;
       PRN_B_WR_I_N_IP1_N_RD_IPJ   : data_mem_is_j2         =1'b1;
       PRN_BYTE_WRITE_J_AND_RD_IPJ : data_mem_is_out        =1'b1;
       PRN_BYTE_WRITE_J2           : data_mem_is_out        =1'b1;
     endcase
  end

wire [7:0] MemReadDataFromByteSel = MemReadAddrD1ByteSelCapt ? MemReadDataCapt[15:8] : MemReadDataCapt[7:0];
  
always @(*)
  if (data_mem_is_i_and_ip1 == 1'b1)
    DataReadIndexI = MemReadDataCapt[7:0];
  else
    if (data_mem_is_i == 1'b1)
     DataReadIndexI = MemReadDataFromByteSel;
    else 
      DataReadIndexI = 8'd0;
   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 8'h0)
     DataReadIndexISave <= 8'h0;
  else if (softRstBbClk_p == 1'b1)
     DataReadIndexISave <= 8'h0;
  else
    if (conflict_read_j_same_as_ip1_D2 == 1'b1)
        DataReadIndexISave <= DataReadIndexJ2;
    else
      if ((data_mem_is_i_and_ip1 == 1'b1) || (data_mem_is_i == 1'b1) )
        DataReadIndexISave <= DataReadIndexI;


always @(*)
  if (conflict_read_j_same_as_ip1 == 1'b1)
    DataReadIndexIP1 = DataReadIndexI;
  else
    if (data_mem_is_i_and_ip1 == 1'b1)
      DataReadIndexIP1 = MemReadDataCapt[15:8];
    else 
      DataReadIndexIP1 = 8'd0;
   
   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 8'h0)
     DataReadIndexIP1Save <= 8'h0;
  else if (softRstBbClk_p == 1'b1)
     DataReadIndexIP1Save <= 8'h0;
  else
    if (data_mem_is_i_and_ip1 == 1'b1)
      DataReadIndexIP1Save <= DataReadIndexIP1;
            

always @(*)
  if (data_mem_is_j == 1'b1)
    DataReadIndexJ = MemReadDataFromByteSel;
  else
    DataReadIndexJ = 8'd0;


  
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 8'h0)
     DataReadIndexJSave <= 8'd0;
  else if (softRstBbClk_p == 1'b1)
     DataReadIndexJSave <= 8'd0;
  else
    if (data_mem_is_j == 1'b1)
      DataReadIndexJSave <= DataReadIndexJ[7:0];

   

always @(*)
  if (conflict_read_j2_same_as_i_D1 == 1'b1)
    DataReadIndexJ2 = DataReadIndexJSave[7:0];
  else
    if (conflict_read_j2_same_as_j_D1 == 1'b1)
      DataReadIndexJ2 = DataReadIndexISave[7:0];
    else
      if (data_mem_is_j2 == 1'b1)
        DataReadIndexJ2 = MemReadDataFromByteSel;
      else
        DataReadIndexJ2 = 8'd0;
   
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 8'h0)
     DataReadIndexJ2Save <= 8'h0;
  else 
    if (softRstBbClk_p == 1'b1)
      DataReadIndexJ2Save <= 8'h0;
    else
      if (data_mem_is_j2 == 1'b1)
        DataReadIndexJ2Save <= DataReadIndexJ2;
   

always @(*)
  if (conflict_read_out_same_as_ip1_D1 == 1'b1)
    DataReadForOut = DataReadIndexIP1Save;
  else
    if ( (conflict_read_out_same_as_i_D1 == 1'b1) && (conflict_read_j_same_as_ip1_D3 == 1'b1) )
      DataReadForOut = DataReadIndexIP1Save;
    else
      if (conflict_read_out_same_as_i_D1 == 1'b1)
        DataReadForOut = DataReadIndexISave;
      else
        if (conflict_read_out_same_as_j_D1 == 1'b1)
          DataReadForOut = DataReadIndexISave; 
        else
          DataReadForOut = MemReadDataFromByteSel;


always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 8'h0)
     DataReadForOutSave <= 8'h0;
  else if (softRstBbClk_p == 1'b1)
     DataReadForOutSave <= 8'h0;
  else
    if (data_mem_is_out == 1'b1) 
      DataReadForOutSave <= DataReadForOut;


   reg [7:0] current_seed;
always @(*)
  case (seedPart[4:0])
    5'd0 : current_seed = seed[7:0];
    5'd1 : current_seed = seed[15:8];
    5'd2 : current_seed = seed[23:16];
    5'd3 : current_seed = seed[31:24];
    5'd4 : current_seed = seed[39:32];
    5'd5 : current_seed = seed[47:40];
    5'd6 : current_seed = seed[55:48];
    5'd7 : current_seed = seed[63:56];
    5'd8 : current_seed = seed[71:64];
    5'd9 : current_seed = seed[79:72];
    5'd10 : current_seed = seed[87:80];
    5'd11 : current_seed = seed[95:88];
    5'd12 : current_seed = seed[103:96];
    5'd13 : current_seed = seed[111:104];
    5'd14 : current_seed = seed[119:112];
    5'd15 : current_seed = seed[127:120];
    5'd16 : current_seed = seed[135:128];
    5'd17 : current_seed = seed[143:136];
    default : current_seed = seed[151:144];
  endcase

  


// logic for sBoxAAddr
// sBoxAAddr takes the value of i in most prngCss, and takes the value of j
// in SBOX_TRN_READ_J prngCs, for retriving value of sBox[j].

always @(*)

  case (prngCs)
    IDLE:
      sBoxAAddr[7:0] = 8'b0;
 
    SBOX_TRN_READ_I_AND_IP1:
      sBoxAAddr[7:0] = index_i[7:0];

    SBOX_TRN_READ_J:
      sBoxAAddr[7:0] = nextJ;
    
    SBOX_TRN_READ_J2:
      sBoxAAddr[7:0] = nextJ;

    PRN_BYTE_READ_ADD1:
      sBoxAAddr[7:0] = 1;

    PRN_BYTE_READ_ADD1OFFSET:
      sBoxAAddr[7:0] = nextJ;
    
    PRN_BYTE_READ_I_AND_IP1:
      sBoxAAddr[7:0] = index_i[7:0];

    PRN_BYTE_READ_J:
      sBoxAAddr[7:0] = nextJ;

    PRN_BYTE_READ_J2:
      sBoxAAddr[7:0] = nextJ;

    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      sBoxAAddr[7:0] = sBoxI + sBoxJ; 

    PRN_BYTE_WRITE_J_AND_RD_IPJ:
      sBoxAAddr[7:0] = sBoxI + sBoxJ;
    
    default:
      sBoxAAddr[7:0] = 8'h0; // for synthesis optimisation
  
  endcase



// logic for sBoxAEn
// sBoxAEn is asserted in most prngCss, it is deasserted when swapping
// needs to be done from the same location ie a particular case of i = j.
// it is asserted evert time a read needs to be performed on port A of RAM.

always @(*)
  case (prngCs)
    SBOX_INDEX_INIT:
      sBoxAEn = 1'b1;

    SBOX_TRN_READ_I_AND_IP1:
      sBoxAEn = 1'b1;

    SBOX_TRN_READ_J:
      sBoxAEn = 1'b1;

    SBOX_TRN_READ_J2:
      sBoxAEn = 1'b1;
    
    PRN_BYTE_READ_ADD1:
      sBoxAEn = 1'b1;

    PRN_BYTE_READ_ADD1OFFSET:
      sBoxAEn = 1'b1;

    PRN_BYTE_READ_I_AND_IP1:
      sBoxAEn = 1'b1;

    PRN_BYTE_READ_J:
      sBoxAEn = 1'b1;

    PRN_BYTE_READ_J2:
      sBoxAEn = 1'b1;

    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      sBoxAEn = 1'b1;

    PRN_BYTE_WRITE_J_AND_RD_IPJ:
      sBoxAEn = 1'b1;

    default:
      sBoxAEn = 1'b0;

  endcase
always @(*)
  case (prngCs)
    IDLE:
      sBoxBAddr[7:0] = 8'b0;

    SBOX_INDEX_INIT:
      sBoxBAddr[7:0] = index_i[7:0] + 8'h1;
    
    SBOX_TRN_WRITE_I_AND_IP1:
      sBoxBAddr[7:0] = index_i[7:0];

    SBOX_TRN_WRITE_J:
      `ifdef SBOX_RDATA_CAPT
        sBoxBAddr[7:0] = index_jD3;
      `else
        sBoxBAddr[7:0] = index_jD2;
      `endif 
    SBOX_TRN_WRITE_J2:
      sBoxBAddr[7:0] = index_j; 

    PRN_BYTE_WRITE_ADD1:
     sBoxBAddr[7:0] =  8'b01;
      
    PRN_BYTE_WRITE_ADD1OFFSET:
      sBoxBAddr[7:0] = index_j[7:0];
    
    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      sBoxBAddr[7:0] = index_i[7:0];

    PRN_BYTE_WRITE_J_AND_RD_IPJ:
      `ifdef SBOX_RDATA_CAPT
        sBoxBAddr[7:0] = index_jD4;
      `else
      sBoxBAddr[7:0] = index_jD2[7:0];  
      `endif 

    PRN_BYTE_WRITE_J2:
      `ifdef SBOX_RDATA_CAPT
        sBoxBAddr[7:0] = index_jD3;
      `else
      sBoxBAddr[7:0] = index_jD2;  
      `endif 

   default:
     sBoxBAddr[7:0] = 8'h0; // for synthesis optimisation;
  endcase


// logic for sBoxBEn
// port B is enabled during the init process.
// it is again enabled when sBox[i] is to be written into location j, during
// SBOX_TRN_WRITE_IJ and PRN_BYTE_WRITE_IJ.
// sBoxBEn is also asserted when the value of sBox[sBox[i] + sBox[j]] has
// to be retrived from port B of RAM.

always @(*)
  case (prngCs)
    SBOX_INDEX_INIT:
      sBoxBEn = 1'b1;
    
    SBOX_TRN_WRITE_I_AND_IP1:
      sBoxBEn = 1'b1;

    SBOX_TRN_WRITE_J:
      sBoxBEn = 1'b1;
    
    SBOX_TRN_WRITE_J2:
      sBoxBEn = 1'b1;
    
    PRN_BYTE_WRITE_ADD1:
      sBoxBEn = 1'b1;
      
    PRN_BYTE_WRITE_ADD1OFFSET:
      sBoxBEn = 1'b1;

    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      sBoxBEn = 1'b1;

    PRN_BYTE_WRITE_J_AND_RD_IPJ:
      sBoxBEn = 1'b1;

    PRN_BYTE_WRITE_J2:
      sBoxBEn = 1'b1;
    
    default:
      sBoxBEn = 1'b0;

  endcase


// logic for sBoxBWriteEn
// sBoxBWriteEn is asserted during the init process.
// it is again asserted when sBox[i] is to be written into location j, during
// SBOX_TRN_WRITE_IJ and PRN_BYTE_WRITE_IJ.

always @(*)
  case (prngCs)
     SBOX_INDEX_INIT:
      sBoxBWriteEn = 2'b11;
    
    SBOX_TRN_WRITE_I_AND_IP1:
      sBoxBWriteEn = 2'b11;

    SBOX_TRN_WRITE_J:
        if (sBoxBAddr[0] == 1'b0)
          sBoxBWriteEn = 2'b01;
        else
          sBoxBWriteEn = 2'b10;
        
    SBOX_TRN_WRITE_J2:
      if (sBoxBAddr[0] == 1'b0)
        sBoxBWriteEn = 2'b01;
      else
        sBoxBWriteEn = 2'b10;

    PRN_BYTE_WRITE_ADD1:
      if (sBoxBAddr[0] == 1'b0)
        sBoxBWriteEn = 2'b01;
      else
        sBoxBWriteEn = 2'b10;

    PRN_BYTE_WRITE_ADD1OFFSET:
      if (sBoxBAddr[0] == 1'b0)
        sBoxBWriteEn = 2'b01;
      else
        sBoxBWriteEn = 2'b10;

    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      sBoxBWriteEn = 2'b11;

    PRN_BYTE_WRITE_J_AND_RD_IPJ:
      if (sBoxBAddr[0] == 1'b0)
        sBoxBWriteEn = 2'b01;
      else
        sBoxBWriteEn = 2'b10;

    PRN_BYTE_WRITE_J2:
      if (sBoxBAddr[0] == 1'b0)
        sBoxBWriteEn = 2'b01;
      else
        sBoxBWriteEn = 2'b10;


    default:
      sBoxBWriteEn = 2'b00;

  endcase


// logic for sBoxBWriteData
// during init, data to be written via port B of RAM is the index itself.
// during SBOX_TRN_WRITE_IJ and PRN_BYTE_WRITE_IJ, data to be written via
// port B of RAM is the previously retrived data from port A of RAM, which is 
// sBox[i] to be written to location j.

always @(*)
     case (prngCs)
       SBOX_INDEX_INIT:
         sBoxBWriteData[15:0] = {index_i[7:0] + 8'h1, index_i[7:0]};
       
       SBOX_TRN_WRITE_I_AND_IP1, PRN_B_WR_I_N_IP1_N_RD_IPJ:
         sBoxBWriteData[15:0] = {DataReadIndexJ2[7:0], DataReadIndexJSave[7:0]}; // J2 was read the cycle before, while J was read several cycles ago
       
       SBOX_TRN_WRITE_J, PRN_BYTE_WRITE_J_AND_RD_IPJ:
         sBoxBWriteData[15:0] = {DataReadIndexISave[7:0], DataReadIndexISave[7:0]}; // I was read several cycles ago
       
       SBOX_TRN_WRITE_J2, PRN_BYTE_WRITE_J2:
         sBoxBWriteData[15:0] = {DataReadIndexIP1Save[7:0], DataReadIndexIP1Save[7:0]}; // I Plus 1 was read several cycles ago
       
       PRN_BYTE_WRITE_ADD1:
         sBoxBWriteData[15:0] = {DataReadIndexJ[7:0], DataReadIndexJ[7:0]};

       PRN_BYTE_WRITE_ADD1OFFSET:
         sBoxBWriteData[15:0] = {DataReadIndexISave[7:0], DataReadIndexISave[7:0]};
       
       default:
         sBoxBWriteData[15:0] = 16'd0; // for synthesis optimisation;
     endcase // case (prngCs)

   


   
// logic for nextI
// i starts with a value of 0.
// i is incremented by 2 during the SBOX_INDEX_INIT prngCs, for referencing
// every alternate location of RAM via port A.
// It is the i counter during the transformation process and later during
// calculation of PR bytes, according to the RC4 algorithm.

//always @(prngCs or index_i)
always @(*)
  case (prngCs)
    IDLE:
      nextI[7:0] = 8'b0;

    SBOX_INDEX_INIT:
      if (index_i[7:0] == 8'hFE)
        nextI[7:0] = 8'b0;
      else
        nextI[7:0] = index_i[7:0] + 8'h2;
      
    SBOX_TRN_WRITE_J2:
      if (index_i[7:0] == 8'hFE)
        nextI[7:0] = 8'd2;
      else
        nextI[7:0] = index_i[7:0] + 8'h2;

    PRN_BYTE_READ_ADD1:
      nextI[7:0] = 8'h2;
    
    PRN_BYTE_WRITE_J2:
      nextI[7:0] = index_i[7:0] + 8'h2;

  default:
    nextI[7:0] = index_i[7:0];

  endcase


// logic for i, registered on clock

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    index_i[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    index_i[7:0]  <= 8'b0;
  else
    index_i[7:0]  <= nextI[7:0];



// logic for nextJ
// j starts with a value of 0.
// j is calculated according to the RC4 algorithm in prngCs SBOX_TRN_READ_J,
// for writing sBox[i] to location j via port B or RAM.
// in prngCs PRN_BYTE_READ_J, it takes a value of sBox[i] + sBox[j] for 
// generation of a PR byte.  

//always @(prngCs or seedPart or index_j or sBoxAReadDataCapt or sBoxAReadDataCaptP1 or seed or index_i) 
always @(*)
  case (prngCs)
    IDLE:
      nextJ[7:0] = 8'b0;

    SBOX_TRN_READ_J:
      nextJ[7:0] =  index_j[7:0] + DataReadIndexI[7:0] + current_seed[7:0];

    SBOX_TRN_READ_J2:
      nextJ[7:0] =  index_j[7:0] + DataReadIndexIP1Save[7:0] + current_seed[7:0];

    PRN_BYTE_READ_ADD1:
      nextJ[7:0] = 8'b0;

    PRN_BYTE_READ_ADD1OFFSET:
      nextJ[7:0] = index_j[7:0] + DataReadIndexI[7:0];
    
    PRN_BYTE_READ_J:
      nextJ[7:0] =  index_j[7:0] + DataReadIndexI[7:0];
    
    PRN_BYTE_READ_J2:
      nextJ[7:0] =  index_j[7:0] + DataReadIndexIP1Save[7:0];
      
   default:
     nextJ[7:0] = index_j[7:0];

  endcase


// logic for j, registered on clock

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    index_j[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    index_j[7:0]  <= 8'b0;
  else
    index_j[7:0]  <= nextJ[7:0];

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    index_jD1[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    index_jD1[7:0]  <= 8'b0;
  else
    index_jD1[7:0]  <= index_j[7:0];

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    index_jD2[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    index_jD2[7:0]  <= 8'b0;
  else
    index_jD2[7:0]  <= index_jD1[7:0];

`ifdef SBOX_RDATA_CAPT
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    index_jD3[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    index_jD3[7:0]  <= 8'b0;
  else
    index_jD3[7:0]  <= index_jD2[7:0];

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    index_jD4[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    index_jD4[7:0]  <= 8'b0;
  else
    index_jD4[7:0]  <= index_jD3[7:0];

`endif

// logic for sBoxI
// sBoxI starts with an initial value of zero.
// it stores the value of sBox[i] retrived from port A of RAM temorarily
// for internal calculations.

always @(*)
  case (prngCs)
    IDLE:
      nextSBoxI[7:0] = 8'b0;

   SBOX_TRN_READ_J:
     nextSBoxI[7:0] = DataReadIndexI[7:0];

    PRN_BYTE_WRITE_ADD1:
      nextSBoxI[7:0] = DataReadIndexISave[7:0];

    PRN_B_WR_I_N_IP1_N_RD_IPJ:
`ifdef SBOX_RDATA_CAPT
      if (conflict_read_j_same_as_ip1_D2 == 1'b1)
        nextSBoxI[7:0] = DataReadIndexIP1Save[7:0];
      else
`endif    
        nextSBoxI[7:0] = DataReadIndexISave[7:0];

    PRN_BYTE_WRITE_J2:
      nextSBoxI[7:0] = DataReadIndexIP1Save[7:0]; 

  default:
    nextSBoxI[7:0] = sBoxI[7:0];

  endcase


// logic for sBoxI, registered on clock

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    sBoxI[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    sBoxI[7:0]  <= 8'b0;
  else
    sBoxI[7:0]  <= nextSBoxI[7:0];


// logic for sBoxJ
// sBoxJ starts with an initial value of zero.
// it stores the value of sBox[j] retrived from port A of RAM temorarily
// for internal calculations.

always @(*)
  case (prngCs)
    IDLE:
      nextSBoxJ[7:0] = 8'b0;

    PRN_BYTE_WRITE_ADD1:
      nextSBoxJ[7:0] = DataReadIndexJ[7:0];
    
    PRN_B_WR_I_N_IP1_N_RD_IPJ:
      nextSBoxJ[7:0] = DataReadIndexJSave[7:0];

    PRN_BYTE_WRITE_J2:
      nextSBoxJ[7:0] = DataReadIndexJ2Save[7:0];

    SBOX_TRN_WRITE_I_AND_IP1:   
      nextSBoxJ[7:0] = DataReadIndexJ2Save[7:0];

    default:
      nextSBoxJ[7:0] = sBoxJ[7:0];

  endcase


// logic for j, registered on clock

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    sBoxJ[7:0]  <= 8'b0;
  else if (softRstBbClk_p == 1'b1)
    sBoxJ[7:0]  <= 8'b0;
  else
    sBoxJ[7:0]  <= nextSBoxJ[7:0];


// logic for nextInitStatus
// initStatus is zero at start, it takes a value of 1 the first time it enters
// PRN_BYTE_WRITE_IJ prngCs. It is used to prevent a wrong value of PR byte
// being released, the first time PRN_BYTE_READ_I prngCs is entered.

//always @(prngCs or initStatus)
always @(*)
  if (prngCs == IDLE)
    nextInitStatus = 1'b0;

  else if (prngCs == PRN_BYTE_WRITE_ADD1OFFSET) 
    nextInitStatus = 1'b1;

  else
    nextInitStatus = initStatus;


// logic for initStatus, registered on clock

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    initStatus  <= 1'b0;
  else if (softRstBbClk_p == 1'b1)
    initStatus  <= 1'b0;
  else
    initStatus  <= nextInitStatus;


// logic for nextSeedPart
// seedPart starts with a value of zero. It is the seed selection index, and
// is used so that the seed value need not be stored internally in this
// module. The seed value held on the input lines can be directly indexed
// by this variable.
// it varies from 0 to 7 for a 64 bit seed.
// it varies from 0 to 15 for a 128 bit seed.

// for currCipherLen as 0, and wep on (in rivierawaves header), the default 64 bit
// encryption is used)


//always @(prngCs or currCipherLen or seedPart)
always @(*)
  case (prngCs)
    IDLE:
      nextSeedPart[4:0] = 5'b0;

    SBOX_TRN_READ_J, SBOX_TRN_READ_J2:
      if (currCipherLen == 1'b0)
      begin
        if (seedPart[4:0] == 5'd7)
          nextSeedPart[4:0] = 5'b0;
        else
          nextSeedPart[4:0] = seedPart[4:0] + 5'h1;
      end
      else
      begin
        if (seedPart[4:0] == 5'd15)
          nextSeedPart[4:0] = 5'b0;
        else
          nextSeedPart[4:0] = seedPart[4:0] + 5'h1;
      end
    default:
      nextSeedPart[4:0] = seedPart[4:0];
       
  endcase


// logic for seedPart, registered on clock

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    seedPart  <= 5'b0;
  else if (softRstBbClk_p == 1'b1)
    seedPart  <= 5'b0;
  else
    seedPart[4:0]  <= nextSeedPart[4:0];


   assign MemReadAddr    = sBoxAAddr[7:1];
   assign MemWriteAddr   = sBoxBAddr[7:1];
   assign MemReadEn      = sBoxAEn;
   assign MemWriteEn[1]  = sBoxBEn & sBoxBWriteEn[1];
   assign MemWriteEn[0]  = sBoxBEn & sBoxBWriteEn[0];
   assign MemWriteData   = sBoxBWriteData[15:0];
 
always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    MemReadAddrD1ByteSel  <= 1'b0;
  else if (softRstBbClk_p == 1'b1)
    MemReadAddrD1ByteSel  <= 1'b0;
  else
    MemReadAddrD1ByteSel  <= sBoxAAddr[0];

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    MemReadAddrD1  <= 7'd0;
  else if (softRstBbClk_p == 1'b1)
    MemReadAddrD1  <= 7'd0;
  else
    MemReadAddrD1  <= MemReadAddr;

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    MemReadEnD1  <= 1'b0;
  else if (softRstBbClk_p == 1'b1)
    MemReadEnD1  <= 1'b0;
  else
    MemReadEnD1  <= MemReadEn;
   

   reg [7:0] MemReadAddrISave, MemReadAddrJSave;
   

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    begin
       MemReadAddrISave  <= 8'b0;
       MemReadAddrJSave  <= 8'b0;
    end
  else if (softRstBbClk_p == 1'b1)
    begin
       MemReadAddrISave  <= 8'b0;
       MemReadAddrJSave  <= 8'b0;
    end
  else
    begin
       if ((prngCs == SBOX_TRN_READ_I_AND_IP1) || (prngCs == PRN_BYTE_READ_I_AND_IP1) )
         MemReadAddrISave  <= sBoxAAddr;
       if ( (prngCs == SBOX_TRN_READ_J) || (prngCs == PRN_BYTE_READ_J))
         MemReadAddrJSave  <= sBoxAAddr;

    end


`ifdef SBOX_RDATA_CAPT
   reg prngCs_on_wait_state;
always @(*)
  case (prngCs)
    WAIT_SBOX_TRN_READ_I_AND_IP1,
    WAIT_SBOX_TRN_READ_J,
    WAIT_SBOX_TRN_READ_J2,
    WAIT_PRN_BYTE_READ_I_AND_IP1,
    WAIT_PRN_BYTE_READ_J,
    WAIT_PRN_BYTE_READ_J2,
    WAIT_PRN_BYTE_READ_ADD1,
    WAIT_PRN_BYTE_READ_ADD1OFFSET,
    WAIT_PRN_B_WR_I_N_IP1_N_RD_IPJ,
    WAIT_PRN_BYTE_WRITE_J_AND_RD_IPJ:
      prngCs_on_wait_state <= 1'b1;
    
   default:
      prngCs_on_wait_state <= 1'b0;
  endcase
`endif

   

always @(*)
  begin
     conflict_read_j_same_as_ip1    =1'b0;
     conflict_read_j2_same_as_j     =1'b0;
     conflict_read_j2_same_as_i     =1'b0;
     conflict_read_out_same_as_i    =1'b0;
     conflict_read_out_same_as_ip1  =1'b0;
     conflict_read_out_same_as_j    =1'b0;
     if ( (prngCs == SBOX_TRN_READ_J) || (prngCs == PRN_BYTE_READ_J) )
       begin
          if (sBoxAAddr == MemReadAddrISave + 8'h01)
            conflict_read_j_same_as_ip1 = 1'b1;
       end

     if ( (prngCs == SBOX_TRN_READ_J2) || (prngCs == PRN_BYTE_READ_J2) )
       if (sBoxAAddr == MemReadAddrJSave)
         conflict_read_j2_same_as_j = 1'b1;

     if ( (prngCs == SBOX_TRN_READ_J2) || (prngCs == PRN_BYTE_READ_J2) )
       if (sBoxAAddr == MemReadAddrISave)
         conflict_read_j2_same_as_i = 1'b1;

     if (prngCs == PRN_B_WR_I_N_IP1_N_RD_IPJ)
       if (sBoxAAddr == MemReadAddrISave)
         conflict_read_out_same_as_i = 1'b1;

     if (prngCs == PRN_BYTE_WRITE_J_AND_RD_IPJ) 
       if (sBoxAAddr == MemReadAddrJSave)
         conflict_read_out_same_as_j = 1'b1;

     if (prngCs == PRN_BYTE_WRITE_J_AND_RD_IPJ)
       if (sBoxAAddr == MemReadAddrISave + 8'd1)
         conflict_read_out_same_as_ip1 = 1'b1;


          
  end
   

     

always @(posedge bbClk or negedge hardRstBbClk_n)
  if (hardRstBbClk_n == 1'b0)
    begin
     conflict_read_j_same_as_ip1_D1  <=1'b0;
     conflict_read_j2_same_as_j_D1   <=1'b0;
     conflict_read_j2_same_as_i_D1   <=1'b0;
     conflict_read_j_same_as_ip1_D2  <=1'b0;
     conflict_read_j_same_as_ip1_D3  <=1'b0;
     conflict_read_j2_same_as_j_D2   <=1'b0;
     conflict_read_j2_same_as_i_D2   <=1'b0;
     conflict_read_out_same_as_i_D1  <=1'b0;
     conflict_read_out_same_as_ip1_D1<=1'b0;
     conflict_read_out_same_as_j_D1  <=1'b0;
    end
  else 
    if (softRstBbClk_p == 1'b1)
      begin
         conflict_read_j_same_as_ip1_D1  <=1'b0;
         conflict_read_j2_same_as_j_D1   <=1'b0;
         conflict_read_j2_same_as_i_D1   <=1'b0;
         conflict_read_j_same_as_ip1_D2  <=1'b0;
         conflict_read_j_same_as_ip1_D3  <=1'b0;
         conflict_read_j2_same_as_j_D2   <=1'b0;
         conflict_read_j2_same_as_i_D2   <=1'b0;
         conflict_read_out_same_as_i_D1  <=1'b0;
         conflict_read_out_same_as_ip1_D1<=1'b0;
         conflict_read_out_same_as_j_D1  <=1'b0;
      end
    else

`ifdef SBOX_RDATA_CAPT
      if (prngCs_on_wait_state == 1'b0)
`endif
        begin
              
         conflict_read_j_same_as_ip1_D1  <= conflict_read_j_same_as_ip1;
         conflict_read_j2_same_as_j_D1   <= conflict_read_j2_same_as_j;
         conflict_read_j2_same_as_i_D1   <= conflict_read_j2_same_as_i;

         conflict_read_j_same_as_ip1_D2  <= conflict_read_j_same_as_ip1_D1;
         conflict_read_j2_same_as_j_D2   <= conflict_read_j2_same_as_j_D1;
         conflict_read_j2_same_as_i_D2   <= conflict_read_j2_same_as_i_D1;

         conflict_read_j_same_as_ip1_D3  <= conflict_read_j_same_as_ip1_D2;
           
         conflict_read_out_same_as_i_D1  <= conflict_read_out_same_as_i;
         conflict_read_out_same_as_ip1_D1<= conflict_read_out_same_as_ip1;
         conflict_read_out_same_as_j_D1  <= conflict_read_out_same_as_j;

      end



`ifdef RW_SIMU_ON
// Disable coverage the RW_SIMU_ON part because this code is not functional but 
// here to ease the simulation
// pragma coverage block = off 

// txControlMainFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (prngCs)


    IDLE                         : prngCs_str = {"IDLE"};
    SBOX_INDEX_INIT              : prngCs_str = {"SBOX_INDEX_INIT"};
    WAIT_FOR_SEED_VALID          : prngCs_str = {"WAIT_FOR_SEED_VALID"};
    SBOX_TRN_READ_I_AND_IP1      : prngCs_str = {"SBOX_TRN_READ_I_AND_IP1"};
    SBOX_TRN_READ_J              : prngCs_str = {"SBOX_TRN_READ_J"};
    SBOX_TRN_READ_J2             : prngCs_str = {"SBOX_TRN_READ_J2"};
    SBOX_TRN_WRITE_I_AND_IP1     : prngCs_str = {"SBOX_TRN_WRITE_I_AND_IP1"};
    SBOX_TRN_WRITE_J             : prngCs_str = {"SBOX_TRN_WRITE_J"};
    SBOX_TRN_WRITE_J2            : prngCs_str = {"SBOX_TRN_WRITE_J2"};
    PRN_BYTE_READ_ADD1           : prngCs_str = {"PRN_BYTE_READ_ADD1"};
    PRN_BYTE_READ_ADD1OFFSET     : prngCs_str = {"PRN_BYTE_READ_ADD1OFFSET"};
    PRN_BYTE_WRITE_ADD1          : prngCs_str = {"PRN_BYTE_WRITE_ADD1"};
    PRN_BYTE_WRITE_ADD1OFFSET    : prngCs_str = {"PRN_BYTE_WRITE_ADD1OFFSET"};
    PRN_BYTE_READ_I_AND_IP1      : prngCs_str = {"PRN_BYTE_READ_I_AND_IP1"};
    PRN_BYTE_READ_J              : prngCs_str = {"PRN_BYTE_READ_J"};
    PRN_BYTE_READ_J2             : prngCs_str = {"PRN_BYTE_READ_J2"};
    PRN_B_WR_I_N_IP1_N_RD_IPJ    : prngCs_str = {"PRN_B_WR_I_N_IP1_N_RD_IPJ"};
    PRN_BYTE_WRITE_J_AND_RD_IPJ  : prngCs_str = {"PRN_BYTE_WRITE_J_AND_RD_IPJ"};
    PRN_BYTE_WRITE_J2            : prngCs_str = {"PRN_BYTE_WRITE_J2"};
    WAIT_FOR_STROBE              : prngCs_str = {"WAIT_FOR_STROBE"};
    DELAY_PRN_OUT1               : prngCs_str = {"DELAY_PRN_OUT1"};
    DELAY_PRN_OUT2               : prngCs_str = {"DELAY_PRN_OUT2"};
    DELAY_PRN_OUT3               : prngCs_str = {"DELAY_PRN_OUT3"};


    
`ifdef SBOX_RDATA_CAPT
    WAIT_SBOX_TRN_READ_I_AND_IP1 : prngCs_str = {"WAIT_SBOX_TRN_READ_I_AND_IP1"};
    WAIT_SBOX_TRN_READ_J         : prngCs_str = {"WAIT_SBOX_TRN_READ_J"};
    WAIT_SBOX_TRN_READ_J2        : prngCs_str = {"WAIT_SBOX_TRN_READ_J2"};
    WAIT_PRN_BYTE_READ_I_AND_IP1 : prngCs_str = {"WAIT_PRN_BYTE_READ_I_AND_IP1"};
    WAIT_PRN_BYTE_READ_J         : prngCs_str = {"WAIT_PRN_BYTE_READ_J"};
    WAIT_PRN_BYTE_READ_J2        : prngCs_str = {"WAIT_PRN_BYTE_READ_J2"};
    WAIT_PRN_BYTE_READ_ADD1      : prngCs_str = {"WAIT_PRN_BYTE_READ_ADD1"};
    WAIT_PRN_BYTE_READ_ADD1OFFSET: prngCs_str = {"WAIT_PRN_BYTE_READ_ADD1OFFSET"};
    WAIT_PRN_B_WR_I_N_IP1_N_RD_IPJ: prngCs_str = {"WAIT_PRN_B_WR_I_N_IP1_N_RD_IPJ"};
    WAIT_PRN_BYTE_WRITE_J_AND_RD_IPJ: prngCs_str = {"WAIT_PRN_BYTE_WRITE_J_AND_RD_IPJ"};
`endif
    default               :  prngCs_str = {"XXX"};
  endcase
end
// pragma coverage block = on 
`endif // RW_SIMU_ON


endmodule


//------------------- END OF FILE ----------------//
