//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : FSM of baController module
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

`ifndef RW_PSBITMAP_SIZE
  `define RW_PSBITMAP_SIZE 4
`endif 

`define CLOG2(x) \
   (x <= 2) ? 1 : \
   (x <= 4) ? 2 : \
   (x <= 8) ? 3 : \
   (x <= 16) ? 4 : \
   (x <= 32) ? 5 : \
   (x <= 64) ? 6 : \
   (x <= 128) ? 7 : \
   (x <= 256) ? 8 : \
   -1
   
module baControllerFsm( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire         macCoreClk,              // MAC Core Clock
            input wire         macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain active low
            input wire         macCoreClkSoftRst_n,     // Soft Reset of the MAC Core Clock domain
            
            ///////////////////////////////////////////////
            //$port_g TX Controller
            ///////////////////////////////////////////////
            input wire         psBitmapReady,           // PS Bitmap field request

            output reg  [ 7:0] psBitmap,                // PS Bitmap field
            output wire        psBitmapValid,           // PS Bitmap field valid
            
            ///////////////////////////////////////////////
            //$port_g RX Controller
            ///////////////////////////////////////////////
            input wire         psBitmapUpdate_p,        // Update Bitmap request
            input wire         psBitmapDiscard_p,       // Discard Bitmap update request
            input wire         rxQoSEnd_p,              // Rx QoS field end pulse
            input wire         barRcved_p,              // When Bar SeqCntrl end pulse

            input wire         baEnable,                // Enable BA Controller procedure
            
            input wire  [ 3:0] rxTID,                   // TID
            input wire  [11:0] rxSN,                    // Seq number
            input wire  [11:0] rxBASSN,                 // BAR Seq number
            
            ///////////////////////////////////////////////
            //$port_g Key search Engine
            ///////////////////////////////////////////////
            input wire  [ 7:0] keyIndexReturnBA,        // Returned Index
            input wire         keyStorageValid_p,       // Returned Index valid
            input wire         keyStorageError_p,       // Error indicating key not found
            
            ///////////////////////////////////////////////
            //$port_g Control and Status Register
            ///////////////////////////////////////////////
            input wire         baPSBitmapReset,         // Partial state bitmap Reset
            
            output wire        baPSBitmapResetIn,       // baPSBitmapResetIn
            output wire        baPSBitmapResetInValid,  // Clear the baPSBitmapReset bit
            
            ///////////////////////////////////////////////
            //$port_g Scoreboard Controller
            ///////////////////////////////////////////////
            input wire         psBitmapUpdateDone_p,    // PS Bitmap update done
            input wire [63:0]  updatedPSBitmap,         // Partial State Bitmap from RAM
            input wire [11:0]  updatedSSN,              // SSN from RAM
            
            output wire        startUpdate_p,           // Start PS Bitmap update
            output reg  [63:0] readPSBitmap,            // Partial State Bitmap read from RAM
            output reg  [11:0] readSSN,                 // SSN read from RAM
            output reg         newPsBitmap,             // Flag new Partial State Bitmap is created
            output reg  [11:0] rxSNCapt,                // Latch the current rxSN for bitmap update
            output reg  [11:0] rxBASSNCapt,             // Latch the current rxBASSN for bitmap update

            ///////////////////////////////////////////////
            //$port_g Debug port
            ///////////////////////////////////////////////
            output wire [15:0] debugPortBAController,   // Debug port for validation
            output wire [15:0] debugPortBAController3   // Debug port for validation
            );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// baController FSM states definition
//$fsm_sd baController
localparam
          INIT_BITMAP  =  4'd0,
                 IDLE  =  4'd1,
        RX_TUPLE_READ  =  4'd2,
           RX_COMPARE  =  4'd3,
         ALLOCATE_ROW  =  4'd4,
        UPDATE_BITMAP  =  4'd5,
   WAIT_BITMAP_UPDATE  =  4'd6,
         WRITE_BITMAP  =  4'd7,
       TX_PASS_BITMAP  =  4'd8;

localparam RW_PSBITMAP_ADDR_WIDTH = `CLOG2(`RW_PSBITMAP_SIZE);

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// baController FSM signals definition
reg  [3:0] baControllerCs;    // baController FSM Current State
reg  [3:0] baControllerNs;    // baController FSM Next State

wire syncResetReq;

// Bitmap storage memory.
// As this is usually a small memory, it is implemented using FF.
reg [87:0] psBitmapMem [0:`RW_PSBITMAP_SIZE-1];

// This small array contains the order of bitmap.
//  - orderMem[0] contains the newest psBitmapMem location.
//  - orderMem[`RW_PSBITMAP_SIZE-1] contains the oldest psBitmapMem location.
reg  [RW_PSBITMAP_ADDR_WIDTH-1:0] orderMem [0:`RW_PSBITMAP_SIZE-1];
reg  [RW_PSBITMAP_ADDR_WIDTH-1:0] orderAddr;      

// Active when TID/Index tuple matches
reg                               tupleFound;        
reg  [RW_PSBITMAP_ADDR_WIDTH-1:0] tupleAddr;      

// Active when keyStorageValid_p and baEnable are active
wire       startBAProcedure;        

// Common counter
reg  [3:0] baCnt;

// Hold signal
reg        psBitmapReadyHold;
reg        psBitmapUpdateHold;

// Latch the rxQoSEnd_p until FSM moves to RX_TUPLE_READ
reg        rxQoSEnd;  
reg        rxBarRcved;

// Latch the keyStorageValid_p until FSM moves to RX_TUPLE_READ
reg        keyStorageValid;             
reg  [3:0] rxTIDCapt;
reg  [7:0] keyIndexReturnBACapt;

`ifdef RW_SIMU_ON
// String definition to display baController current state
reg [18*8-1:0] baControllerCs_str;
`endif // RW_SIMU_ON

integer i,j,k,l;

///////////////////////////////////////////////
//PS BITMAP Memory
///////////////////////////////////////////////
wire                              psBitmapWriteEn;   // Partial State Bitmap Write Enable.
                                                     // This signal is asserted for write operations to the Partial State Bitmap.
wire [RW_PSBITMAP_ADDR_WIDTH-1:0] psBitmapAddr;      // Partial State Bitmap Address bus.
reg  [87:0]                       psBitmapWriteData; // Partial State Bitmap Write Data.

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign syncResetReq = (!macCoreClkSoftRst_n || baPSBitmapReset);

// baController FSM Current State Logic 
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                                       // Asynchronous Reset
    baControllerCs <= INIT_BITMAP; 
  else if (syncResetReq) // Synchronous Reset
    baControllerCs <= INIT_BITMAP; 
  else
    baControllerCs <= baControllerNs; 
end


// baController FSM Next State Logic.
always @* 
begin
  case(baControllerCs)

    INIT_BITMAP:
      //$fsm_s In INIT_BITMAP state, the state machine waits end of PS Bitmap clear

      //$fsm_t One clock cycle after the init request, the state machine goes back to IDLE state
        baControllerNs = IDLE;

    IDLE:
      //$fsm_s In IDLE state, the state machine waits request from Key search Engine or TX Controller
      if ((psBitmapUpdate_p || psBitmapUpdateHold) && startBAProcedure) 
        //$fsm_t When psBitmapUpdate_p is received and startBAProcedure is valid, the state machine goes to RX_TUPLE_READ state
        baControllerNs = RX_TUPLE_READ; 
      else if (psBitmapReady) 
        //$fsm_t When psBitmapReady is received, the state machine goes to TX_PASS_BITMAP state
        baControllerNs = TX_PASS_BITMAP; 
      else
        //$fsm_t While keyStorageValid_p and psBitmapReady are not received, the state machine stays in IDLE state
        baControllerNs = IDLE;

    RX_TUPLE_READ:
      //$fsm_s In RX_BITMAP_READ state, the state machine read PS Bitmap RAM

        //$fsm_t After one clock cycle, the state machine goes to RX_COMPARE state
        baControllerNs = RX_COMPARE; 

    RX_COMPARE:
      //$fsm_s In RX_COMPARE state, the state machine compare index/TID tuple with RAM
      if (tupleFound)
        //$fsm_t When tupleFound is received, the state machine goes to UPDATE_BITMAP state
        baControllerNs = UPDATE_BITMAP; 
      else 
        //$fsm_t While 4 Mem locations have been checked, the state machine goes to ALLOCATE_ROW state
        baControllerNs = ALLOCATE_ROW;

    ALLOCATE_ROW:
      //$fsm_s In ALLOCATE_ROW state, the state machine allocate a new row in the PS Bitmap RAM

        //$fsm_t After one clock cycle, the state machine goes to UPDATE_BITMAP state
        baControllerNs = UPDATE_BITMAP; 

    UPDATE_BITMAP:
      //$fsm_s In UPDATE_BITMAP state, the state machine triggers Scoreboard Controller

        //$fsm_t After one clock cycle, the state machine goes to WAIT_BITMAP_UPDATE state
        baControllerNs = WAIT_BITMAP_UPDATE; 

    WAIT_BITMAP_UPDATE:
      //$fsm_s In WAIT_BITMAP_UPDATE state, the state machine waits trigger from Scoreboard Controller
      if (psBitmapUpdateDone_p) 
        //$fsm_t When psBitmapUpdateDone_p is received, the state machine goes to WRITE_BITMAP state
        baControllerNs = WRITE_BITMAP; 
      else
        //$fsm_t While psBitmapUpdateDone_p is not received, the state machine stays in WAIT_BITMAP_UPDATE state
        baControllerNs = WAIT_BITMAP_UPDATE;

    WRITE_BITMAP:
      //$fsm_s In WRITE_BITMAP state, the state machine updates PS Bitmap Mem from Scoreboard Controller result
      if (psBitmapWriteEn)
        //$fsm_t When second word is written into PS Bitmap RAM, the state machine goes to IDLE state
        baControllerNs = IDLE; 
      else
        //$fsm_t While second word is not written into PS Bitmap RAM, the state machine stays in WRITE_BITMAP state.
        baControllerNs = WRITE_BITMAP;

    TX_PASS_BITMAP:
      //$fsm_s In TX_PASS_BITMAP state, the state machine waits end of PS Bitmap sent to TX Controller
      if ((baCnt == 4'h9) && psBitmapValid) 
        //$fsm_t When baCnt is 9, the state machine goes to IDLE state.
        baControllerNs = IDLE; 
      else
        //$fsm_t When baCnt is not 9, the state machine stays in TX_PASS_BITMAP state.
        baControllerNs = TX_PASS_BITMAP;

    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
     default:   
        baControllerNs = IDLE; 
    // pragma coverage block = on 
  endcase
end


//////////////////////////////////////////////////////
// Internal control
//////////////////////////////////////////////////////

// Clear of reset from CSReg
assign baPSBitmapResetIn      = 1'b0;
assign baPSBitmapResetInValid = (baControllerCs == INIT_BITMAP);

// Active when keyStorageValid_p and baEnable are active
assign startBAProcedure = (rxBarRcved || rxQoSEnd) && baEnable && keyStorageValid;

// Set newPsBitmap Flag if there is no save bitmap for the current TID/Tupple
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    newPsBitmap  <= 1'b0;
  else if (syncResetReq)               // Synchronous Reset
    newPsBitmap <= 1'b0;
  else if (baControllerCs == ALLOCATE_ROW)
    newPsBitmap <= 1'b1;
  else if (psBitmapUpdateDone_p)
    newPsBitmap <= 1'b0;
end 

// Latch the barRcved_p until FSM moves to RX_TUPLE_READ
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    rxBarRcved  <= 1'b0;
  else if (syncResetReq)               // Synchronous Reset
    rxBarRcved <= 1'b0;
  else if (barRcved_p && baEnable && (baControllerCs == IDLE))
    rxBarRcved <= 1'b1;
  else if ((baControllerCs == WRITE_BITMAP) || (psBitmapDiscard_p))
    rxBarRcved <= 1'b0;
end        

// Latch the rxQoSEnd_p until FSM moves to RX_TUPLE_READ
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    rxQoSEnd  <= 1'b0;
  else if (syncResetReq)               // Synchronous Reset
    rxQoSEnd <= 1'b0;
  else if (rxQoSEnd_p && baEnable && (baControllerCs == IDLE))
    rxQoSEnd <= 1'b1;
  else if ((baControllerCs == WRITE_BITMAP) || (psBitmapDiscard_p))
    rxQoSEnd <= 1'b0;
end        


always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    rxTIDCapt  <= 4'b0;
  else if (syncResetReq)               // Synchronous Reset
    rxTIDCapt  <= 4'b0;
  else if (rxQoSEnd || rxBarRcved)
    rxTIDCapt  <= rxTID;
end

always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    rxSNCapt  <= 12'h0;
  else if (syncResetReq)               // Synchronous Reset
    rxSNCapt  <= 12'h0;
  else if (rxQoSEnd)
    rxSNCapt  <= rxSN;
end

always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    rxBASSNCapt  <= 12'h0;
  else if (syncResetReq)               // Synchronous Reset
    rxBASSNCapt  <= 12'h0;
  else if (rxBarRcved)
    rxBASSNCapt  <= rxBASSN;
end

always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    keyIndexReturnBACapt  <= 8'b0;
  else if (syncResetReq)               // Synchronous Reset
    keyIndexReturnBACapt  <= 8'b0;
  else if (keyStorageValid_p && !keyStorageError_p && !keyStorageValid && baEnable && (baControllerCs == IDLE))
    keyIndexReturnBACapt  <= keyIndexReturnBA;
end
    


// Latch the keyStorageValid_p until FSM moves to RX_TUPLE_READ
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                    // Asynchronous Reset
    keyStorageValid  <= 1'b0;
  else if (syncResetReq)               // Synchronous Reset
    keyStorageValid <= 1'b0;
  else if (keyStorageValid_p && !keyStorageError_p && baEnable && (baControllerCs == IDLE))
    keyStorageValid <= 1'b1;
  else if ((baControllerCs == WRITE_BITMAP) || (psBitmapDiscard_p))
    keyStorageValid <= 1'b0;
end        



always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)      // Asynchronous Reset
  begin
    tupleAddr  <=  {RW_PSBITMAP_ADDR_WIDTH,{1'b0}};
    orderAddr  <=  {RW_PSBITMAP_ADDR_WIDTH,{1'b0}};
    tupleFound <= 1'b0;
  end
  else if (!macCoreClkSoftRst_n || baPSBitmapReset || (baControllerCs == WRITE_BITMAP)) // Synchronous Reset
  begin
    orderAddr  <=  {RW_PSBITMAP_ADDR_WIDTH,{1'b0}};
    tupleAddr  <=  {RW_PSBITMAP_ADDR_WIDTH,{1'b0}};
    tupleFound <= 1'b0;
  end
  else if (baControllerCs == RX_TUPLE_READ)
  begin
    for (j=0; j < `RW_PSBITMAP_SIZE;j=j+1)
    begin
      if ({keyIndexReturnBACapt,rxTIDCapt} == psBitmapMem[orderMem[j]][87:76])
      begin
        //$display("Tuple Found  @ %0d %h == %h",j[1:0],{keyIndexReturnBACapt,rxTIDCapt},psBitmapMem[orderMem[j]][87:76]);
        tupleAddr <= orderMem[j][RW_PSBITMAP_ADDR_WIDTH-1:0];
        orderAddr <= j[RW_PSBITMAP_ADDR_WIDTH-1:0];
        tupleFound <= 1'b1;
      end
    end     
  end
end

always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)      // Asynchronous Reset
    for (k=0; k < `RW_PSBITMAP_SIZE; k=k+1)
      orderMem[k] <= `RW_PSBITMAP_SIZE-k-1;
  else if (!macCoreClkSoftRst_n || baPSBitmapReset) // Synchronous Reset
    for (k=0; k < `RW_PSBITMAP_SIZE; k=k+1)
      orderMem[k] <= `RW_PSBITMAP_SIZE-k-1;
  else if (baControllerCs == RX_COMPARE)
  begin
    // If tuple is found, the tuple address takes the first entry in orderMem (which correstpond to the newest)
    // and the orderMem cells between 0 and tuple address will be shifted
    // If tuple is not found, the oldest address will be used and take the first entry (corresponding to the newest)
    // and the orderMem cells between 0 and max address -1 will be shifted
    if (tupleFound)
      orderMem[0] <= tupleAddr;
    else
      orderMem[0] <= orderMem[`RW_PSBITMAP_SIZE-1];

    for (k=0; k < `RW_PSBITMAP_SIZE-1;k=k+1)
    begin
      if (!tupleFound || (k[RW_PSBITMAP_ADDR_WIDTH-1:0] < orderAddr))
        orderMem[k+1] <= orderMem[k];
    end

  end
end


  

// Hold rxController signals
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                                            // Asynchronous Reset
    psBitmapUpdateHold  <= 1'b0;
  else if (!macCoreClkSoftRst_n || (baControllerCs == INIT_BITMAP) || (baControllerCs == WRITE_BITMAP))           // Synchronous Reset
    psBitmapUpdateHold  <= 1'b0;
  else if (psBitmapUpdate_p)
    psBitmapUpdateHold  <= 1'b1;
end



// Common counter
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                                              // Asynchronous Reset
    baCnt <= 4'b0;
  else if ((psBitmapValid && (baCnt == 4'h9)) || !macCoreClkSoftRst_n || // Synchronous Reset
    (baControllerCs == UPDATE_BITMAP) || (baControllerCs == IDLE))
    baCnt <= 4'b0; 
  else if (psBitmapValid || (baControllerCs == WRITE_BITMAP))
    baCnt <= baCnt + 4'b1; 
end


//////////////////////////////////////////////////////
// Scoreboard Controller
//////////////////////////////////////////////////////

// Start Scoreboard Controller computing
//assign startUpdate_p = (baControllerCs == WAIT_BITMAP_UPDATE) && psBitmapEn_ff1;
assign startUpdate_p = (baControllerCs == UPDATE_BITMAP);

// Hold psBitmap[7:0] from RAM
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                                          // Asynchronous Reset
    readPSBitmap <= 64'b0;
  else if (!macCoreClkSoftRst_n) // Synchronous Reset
    readPSBitmap <= 64'b0;
  else if (tupleFound)
    readPSBitmap <= psBitmapMem[tupleAddr][63:0];
end

// Hold readSSN from Mem or scorboard
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                                          // Asynchronous Reset
    readSSN <= 12'b0;
  else if (!macCoreClkSoftRst_n) // Synchronous Reset
    readSSN <= 12'b0;
  else if (tupleFound)
    readSSN <= psBitmapMem[tupleAddr][75:64];
end


//////////////////////////////////////////////////////
// TX Controller
//////////////////////////////////////////////////////

// Data valid generation
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)                                            // Asynchronous Reset
    psBitmapReadyHold <= 1'b0;
  else if ((psBitmapValid && (baCnt == 4'h9)) || !macCoreClkSoftRst_n) // Synchronous Reset
    psBitmapReadyHold <= 1'b0;
  else if (psBitmapReady && (baCnt == 4'h0))
    psBitmapReadyHold <= 1'b1; 
end

assign psBitmapValid = psBitmapReadyHold && psBitmapReady;

always @*
begin
  case (baCnt)
    4'h0    : psBitmap = {updatedSSN[3:0], 4'b0};
    4'h1    : psBitmap = updatedSSN[11:4];
    4'h2    : psBitmap = updatedPSBitmap[7:0];
    4'h3    : psBitmap = updatedPSBitmap[15:8];
    4'h4    : psBitmap = updatedPSBitmap[23:16];
    4'h5    : psBitmap = updatedPSBitmap[31:24];
    4'h6    : psBitmap = updatedPSBitmap[39:32];
    4'h7    : psBitmap = updatedPSBitmap[47:40];
    4'h8    : psBitmap = updatedPSBitmap[55:48];
    4'h9    : psBitmap = updatedPSBitmap[63:56];
    // Disable case default state for block coverage
    // pragma coverage block = off
    default : psBitmap = updatedPSBitmap[7:0];
    // pragma coverage block = on
  endcase
end


//////////////////////////////////////////////////////
// PS BITMAP RAM
//////////////////////////////////////////////////////

always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (!macCoreClkHardRst_n)      // Asynchronous Reset
  begin
    for (i=0; i < `RW_PSBITMAP_SIZE;i=i+1)
      psBitmapMem[i] <= 88'd0;
  end
  else if (!macCoreClkSoftRst_n || baPSBitmapReset) // Synchronous Reset
  begin
    for (i=0; i < `RW_PSBITMAP_SIZE;i=i+1)
      psBitmapMem[i] <= 88'd0;
  end
  else if (psBitmapWriteEn)
    psBitmapMem[psBitmapAddr] <= psBitmapWriteData;
end


// PS BITMAP Mem write
assign psBitmapWriteEn = (baControllerCs == WRITE_BITMAP) ||
                         (baControllerCs == INIT_BITMAP);

// PS BITMAP Mem write data
// index / TID / SSN / psBitmap[39:0]
always @*
begin
  if (baControllerCs == INIT_BITMAP)
    psBitmapWriteData = 88'b0;
  else
    //                   [87:80]               [79:76]    [75:64]     [63:0]
    psBitmapWriteData = {keyIndexReturnBACapt, rxTIDCapt, updatedSSN, updatedPSBitmap[63:0]};
end

assign psBitmapAddr   = orderMem[0]; 

//////////////////////////////////////////////////////
// Debug port
//////////////////////////////////////////////////////
assign debugPortBAController = {4'b0,
                                tupleFound,
                                startUpdate_p,
                                psBitmapUpdateDone_p,
                                newPsBitmap,
                                psBitmapReady,
                                psBitmapDiscard_p,
                                psBitmapUpdate_p,
                                keyStorageError_p, 
                                baControllerCs};

assign debugPortBAController3 = { rxTIDCapt,            // 4 bits
                                  keyIndexReturnBACapt, // 8 bits
                                  psBitmapAddr[1:0],    // 2 bits  
                                  rxQoSEnd,
                                  keyStorageValid};
                                  

///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON


`endif // RW_ASSERT_ON


//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////

`ifdef RW_SIMU_ON

// baController FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (baControllerCs)
            INIT_BITMAP  :  baControllerCs_str = {"INIT_BITMAP"};
                   IDLE  :  baControllerCs_str = {"IDLE"};
          RX_TUPLE_READ  :  baControllerCs_str = {"RX_TUPLE_READ"};
             RX_COMPARE  :  baControllerCs_str = {"RX_COMPARE"};
           ALLOCATE_ROW  :  baControllerCs_str = {"ALLOCATE_ROW"};
          UPDATE_BITMAP  :  baControllerCs_str = {"UPDATE_BITMAP"};
     WAIT_BITMAP_UPDATE  :  baControllerCs_str = {"WAIT_BITMAP_UPDATE"};
           WRITE_BITMAP  :  baControllerCs_str = {"WRITE_BITMAP"};
         TX_PASS_BITMAP  :  baControllerCs_str = {"TX_PASS_BITMAP"};
                default  :  baControllerCs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON

endmodule

