////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of bfrController module
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -----------------------------------------------------------------------------
//                                                                          
// 
// 
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module bfrController( 
   ///////////////////////////////////////////////
   //$port_g Clock and reset
   ///////////////////////////////////////////////
   input  wire        macCoreClk,               // MAC Core Clock
   input  wire        macCoreClkHardRst_n,      // Hard Reset of the MAC Core Clock domain
   input  wire        macCoreClkSoftRst_n,      // Soft Reset of the MAC Core Clock domain
   
   ///////////////////////////////////////////////
   //$port_g MAC Timer Unit
   ///////////////////////////////////////////////
   input  wire        tick1us_p,                // Pulse generated every us

   ///////////////////////////////////////////////
   //$port_g A-MPDU Deaggregator
   ///////////////////////////////////////////////
   input  wire        macPhyIfRxErr_p,          // Rx Err pulse                                   
   input  wire        rxVector1Start_p,         // Indicate that the first byte of RX Vector has been received
   input  wire        rxVector1Valid_p,         // Rx vector 1 is available
   input  wire        rxEndOfFrame_p,           // End of frame information
   input  wire  [3:0] rxFormatMod,              // Format and modulation of received frame
   input  wire  [2:0] rxChBW,                   // Channel bandwidth
   input  wire        rxSTBC,                   // Space Time Block Coding of received frame
   input  wire  [1:0] rxChBWInNonHT,            // Channel bandwidth in Non-HT
   input  wire  [2:0] rxNss,                    // Number of Spatial Stream of received frame
   input  wire        rxNDP,                    // Indicate that the received frame is a Null Data Packet

   ///////////////////////////////////////////////
   //$port_g Mac Controller interface
   ///////////////////////////////////////////////
   input  wire        sendBFR_p,                // Indication that an Beamforming report packet has to be sent
   output wire        needBFRRcved_p,           // Frame received with BFR response needed pulse
   output wire        needTBBFRRcved_p,         // Frame received with Trigger Based 
                                                // BFR response needed pulse
   output reg         incorrectBFRPollRcved_p,  // Beamforming Report Poll is not correct
   output reg  [14:0] bfrLength,                // Beamforming Report frame Length

   ///////////////////////////////////////////////
   //$port_g TX Controller
   ///////////////////////////////////////////////
   output reg   [5:0] bfrSoundingDialToken,     // Beamforming Report Sounding Dialog Token Number

   ///////////////////////////////////////////////
   //$port_g RX Controller
   ///////////////////////////////////////////////
   input  wire [47:0] rxAddr1,                  // Received addr1
   input  wire [47:0] rxAddr2,                  // Received addr2
   input  wire        rxNDPAValid_p,            // NDP-A Valid pulse
   input  wire        bfrPollRcved_p,           // BFR-Poll received pulse
   input  wire        rxBFRPollValid_p,         // BFR-Poll Valid pulse
   input  wire        triggerRcved_p,           // Trigger Received pulse
   input  wire        rxTriggerValid_p,         // Trigger Valid pulse
   input  wire        rxBWSignalingTA,          // Indicate that the received frame has a Bandwidth
                                                // Signaling TA
   input  wire  [3:0] rxTriggerType,            // Type of trigger frame
   input  wire        rxNDPAHE,                 // NDP-A HE
   input  wire  [5:0] rxNDPASoundingDialToken,  // NDP-A Sounding Dialog Token Number
   input  wire  [1:0] rxNDPAFeedbackType,       // NDP-A Feedback Type
   input  wire  [6:0] rxNDPARUStartIndex,       // NDP-A RU Start Index
   input  wire  [6:0] rxNDPARUEndIndex,         // NDP-A RU End Index
   input  wire        rxNDPANg,                 // NDP-A Ng
   input  wire        rxNDPACodebook,           // NDP-A Codebook
   input  wire  [2:0] rxNDPANcIndex,            // NDP-A Nc Index
   input  wire        rxNDPASTAIndexIsFirst,    // NDP-A our STA Info Is first
   
   ///////////////////////////////////////////////
   //$port_g BFR HW Accelerator
   ///////////////////////////////////////////////
   output reg         bfrStart,                 // Start Beamforming Report computation
   input  wire        bfrDone,                  // Beamforming Report computation is done
   output reg   [1:0] bfrChBW,                  // Indicates to the beamforming HW accelerator
                                                // the bandwidth of the report to be generated
   output reg   [1:0] bfrGrouping,              // Indicates to the beamforming HW accelerator
                                                // the grouping used for the compression
   output reg         bfrCodebook,              // Indicates to the beamforming HW accelerator
                                                // the codebook used for the compression.
   output reg   [1:0] bfrFeedbackType,          // Indicates to the beamforming HW accelerator
                                                // the FeedbackType (SU or MU) used for the
                                                // compression and report generation. 
   output reg   [2:0] bfrNc,                    // Indicates to the beamforming HW accelerator
                                                // the number of columns information
   output reg   [2:0] bfrNr,                    // Indicates to the beamforming HW accelerator
                                                // the number of rows information
   output reg         bfrHE,                    // Indicates to the beamforming HW accelerator
                                                // the Beamforming Report Format (VHT or HE)
   output reg   [6:0] bfrRUStartIndex,          // Indicates to the beamforming HW accelerator
                                                // the RU Start Index (Only for HE BFR)
   output reg   [6:0] bfrRUEndIndex,            // Indicates to the beamforming HW accelerator
                                                // the RU End Index (Only for HE BFR)

   ///////////////////////////////////////////////
   //$port_g Control and Status Register
   ///////////////////////////////////////////////
   input  wire  [7:0] slotTime,                 // aSlotTime parameter
   input  wire  [7:0] rxStartDelayOFDM,         // Receive Start Delay for OFDM
   input  wire  [7:0] sifs,                     // Provide SIFS duration in us
// input  wire  [2:0] bfrFormatMod,             // Beamforming Report Format and Modulation
// input  wire        bfrShortGI,               // Beamforming Report Short GI
// input  wire  [7:0] bfrMCS,                   // Beamforming Report MCS
   input  wire  [2:0] bfmeeNc,                  // Number of supported column by the beamformee
   input  wire  [2:0] bfmeeNr,                  // Number of supported row by the beamformee
   input  wire  [1:0] bfmeeGrouping,            // Beamforming Report Grouping
   input  wire        bfmeeCodebook,            // Beamforming Report Codebook
   input  wire        bfmeeMUSupport,           // Enable Beamformee Multi User
   input  wire        bfmeeEnable,              // Enable Beamformee mode
   
   ///////////////////////////////////////////////
   //$port_g Debug port
   ///////////////////////////////////////////////
   output wire [15:0] debugPortBFRController    // Debug port for validation
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// bfrController FSM states definition
//$fsm_sd bfrController
parameter 
   IDLE            = 3'd0,
   CHECK_NDPA      = 3'd1,
   WAIT_NDP        = 3'd2,
   CHECK_NDP       = 3'd3,
   COMPUTE_BFR_LEN = 3'd4,
   WAIT_NDP_EOF    = 3'd5,
   COMPUTE_BFR     = 3'd6,
   ACTIVE          = 3'd7;

// Bemaforming Report Feedback Type definition
parameter
   SU  = 2'd0, // Single User
   MU  = 2'd1, // Multiple User
   CQI = 2'd2; // Channel Quality Information (Not Yet Supported)

////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////

// bfrController FSM signals definition
reg   [2:0] bfrControllerCs;           // FSM Current State
reg   [2:0] bfrControllerNs;           // FSM Next State

wire        isIdle;                    // FSM is in IDLE state
wire        isComputeBFRLen;           // FSM is in COMPUTE_BFR_LEN state
wire        isComputeBFR;              // FSM is in COMPUTE_BFR state
wire        isActive;                  // FSM is in ACTIVE state
wire        changeToCheckNDPA;         // FSM Transition To CHECK_NDPA state
wire        changeToWaitNDP;           // FSM Transition To WAIT_NDP state
wire        changeToCheckNDP;          // FSM Transition To CHECK_NDP state

reg   [9:0] sifsTimeoutCnt;            // SIFS Timeout Counter
wire        sifsTimeout_p;             // SIFS Timeout detected

reg  [47:0] bfrDstAddr;                // Beamforming Report Destination Address

wire        bfrLengthCompDone;         // BFR Length computation Done
reg   [2:0] bfrLengthCompPhase;        // BFR Length computation Phase
reg   [5:0] bfrNa;                     // BFR Na
reg   [4:0] bfrBPhiBPsi;               // BFR (bPhi + bPsi)
reg   [8:0] vhtBFRNs;                  // VHT BFR Ns
wire  [7:0] heBFRNs;                   // HE  BFR Ns
wire  [8:0] bfrNs;                     // BFR Ns
reg   [7:0] vhtBFRNsp;                 // VHT BFR Ns'
wire  [8:0] bfrNsp;                    // BFR Ns'

reg   [8:0] multIn1;                   // First input of the 9x8 multiplier
reg   [7:0] multIn2;                   // Second input of the 9x8 multiplier
wire [16:0] multOut;                   // Output of the 9x8 multiplier
reg  [16:0] accuIn;                    // Input of the Accumulator 
reg  [16:0] accuOut;                   // Output of the Accumulator

reg         rxEndOfFrame;              // Registered End of frame information
reg         macPhyIfRxErr;             // Registered macPhyIfRxErr_pe information

wire  [2:0] rxNsts;                    // Number of space time stream of the NDP

reg         bfrResponseAfterNDP;       // BFR Response is needed after the NDP
reg         heNDPAUnicast;             // NDPA is an HE NDPA with only one STA Info field

reg   [7:0] heBFRSTone;                // HE Beamforming Report S-Tone indice
reg   [7:0] heBFRETone;                // HE Beamforming Report E-Tone indice


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

// bfrController FSM Current State Logic 
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
     bfrControllerCs <= IDLE;
  else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
     bfrControllerCs <= IDLE;
  else
     bfrControllerCs <= bfrControllerNs;
end


// bfrController FSM Next State Logic
always @* 
begin  
   case(bfrControllerCs)
   IDLE:
      //$fsm_s In IDLE state, the state machine waits NDPA Packet reception
      if (rxNDPAValid_p && bfmeeEnable)
         //$fsm_t When NDPA is received, the state machine goes to CHECK_NDPA state
         bfrControllerNs = CHECK_NDPA;
      else
         //$fsm_t While NDPA is not received, the state machine stays in IDLE state
         bfrControllerNs = IDLE;
  
   CHECK_NDPA:
      //$fsm_s In CHECK_NDPA state, NDPA packet is verified
      if (((bfrFeedbackType == MU) && !bfmeeMUSupport) ||
          ( bfrFeedbackType == CQI) ||
          ( rxNDPAHE && rxFormatMod==4'd1 && ~rxBWSignalingTA))
         //$fsm_t When beamforming feedback type is not supported or HE NDP-A received in NON-HT
         //without bandwidth signaling, the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else if (rxEndOfFrame)
         //$fsm_t When beamforming feedback type is supported and NDP-A PPDU receive is done,
         //the state machine goes to WAIT_NDP state
         bfrControllerNs = WAIT_NDP;
      else
         //$fsm_t When beamforming feedback type is supported and NDP-A PPDU receive is not done,
         //the state machine stays in CHECK_NDPA state
         bfrControllerNs = CHECK_NDPA;

   WAIT_NDP:
      //$fsm_s In WAIT_NDP state, the state machine waits NPD Packet reception
      if (sifsTimeout_p)
         //$fsm_t When no packet is received before the timeout,
         //the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else if (rxVector1Valid_p && rxNDP && ((rxFormatMod == 4'd4) && ~bfrHE ||
                                             (rxFormatMod == 4'd5) &&  bfrHE))
         //$fsm_t When VHT NPD packet is received and the requested beamforming report is VHT or
         //HE NDP packet is received and the requested beamforming report is HE,
         //the state machine goes to CHECK_NDP state
         bfrControllerNs = CHECK_NDP;
      else if (rxVector1Valid_p)
         //$fsm_t When received packet is not a NPD or not correct type of NDP,
         //the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else
         //$fsm_t When no packet is received, the state machine stays in WAIT_NDP state
         bfrControllerNs = WAIT_NDP;

   CHECK_NDP:
      //$fsm_s In CHECK_NDP state, NDP packet is verified
      if ((bfrNr > bfmeeNr) || (bfrNr < bfrNc))
         //$fsm_t When beamforming Report Nr is not supported or invalid, the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else
         //$fsm_t When beamforming Report Nr is supported and valid, the state machine goes to COMPUTE_BFR_LEN state
         bfrControllerNs = COMPUTE_BFR_LEN;

   COMPUTE_BFR_LEN:
      //$fsm_s In COMPUTE_BFR_LEN state, Beamforming Report length is computed
      if (bfrLengthCompDone)
         //$fsm_t When beamforming Report length is ready, the state machine goes to WAIT_NDP_EOF state
         bfrControllerNs = WAIT_NDP_EOF;
      else
         //$fsm_t When beamforming Report length is not ready, the state machine stays in COMPUTE_BFR_LEN state
         bfrControllerNs = COMPUTE_BFR_LEN;

   WAIT_NDP_EOF:
      //$fsm_s In WAIT_NDP_EOF state, the state machine waits end of NDP packet reception
      if (macPhyIfRxErr)
         //$fsm_t When NDP rxErr is received, the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else if (rxEndOfFrame)
         //$fsm_t When NDP end of frame is received, the state machine goes to CHECK_NDP state
         bfrControllerNs = COMPUTE_BFR;
      else
         //$fsm_t When NDP end of frame is not received, the state machine stays in WAIT_NDP_EOF state
         bfrControllerNs = WAIT_NDP_EOF;

   COMPUTE_BFR:
      //$fsm_s In COMPUTE_BFR state, Beamforming Report Computation is requested

      //$fsm_t After one clock cycle, the state machine goes to ACTIVE state
      bfrControllerNs = ACTIVE;

   ACTIVE:
      //$fsm_s In ACTIVE state, the state machine waits beamforming report transmit
      if (sifsTimeout_p)
         //$fsm_t When no packet is received before the timeout,
         //the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else if (sendBFR_p)
         //$fsm_t When beamforming Report has been sent, the state machine goes to IDLE state
         bfrControllerNs = IDLE;
      else if (rxNDPAValid_p)
         //$fsm_t When new NDPA is received, the state machine goes to CHECK_NDPA state
         bfrControllerNs = CHECK_NDPA;
      else
         //$fsm_t When beamforming report has not been sent, the state machine stays in ACTIVE state
         bfrControllerNs = ACTIVE;

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      bfrControllerNs = IDLE; 
   // pragma coverage block = on
   endcase
end


// bfrController FSM State Decoder
assign isIdle            = (bfrControllerCs == IDLE           );
assign isComputeBFRLen   = (bfrControllerCs == COMPUTE_BFR_LEN);
assign isComputeBFR      = (bfrControllerCs == COMPUTE_BFR    );
assign isActive          = (bfrControllerCs == ACTIVE         );
assign changeToCheckNDPA = (bfrControllerCs != CHECK_NDPA  ) && (bfrControllerNs == CHECK_NDPA  );
assign changeToWaitNDP   = (bfrControllerCs != WAIT_NDP    ) && (bfrControllerNs == WAIT_NDP    );
assign changeToCheckNDP  = (bfrControllerCs != CHECK_NDP   ) && (bfrControllerNs == CHECK_NDP   );


// Beamforming Report Response request
assign needBFRRcved_p = 
   // NDP Response
   bfrResponseAfterNDP && isComputeBFRLen && bfrLengthCompDone ||
   // BFR-POLL Response
  ~bfrResponseAfterNDP && isActive && ~bfrHE && bfrPollRcved_p;

assign needTBBFRRcved_p = 
   // BFRP-Trigger Response
  ~bfrResponseAfterNDP && isActive &&  bfrHE && triggerRcved_p && rxTriggerType==4'd1;


// Number of space time stream of NDP
assign rxNsts = rxSTBC ? {rxNss[1:0],1'b1} : rxNss;


// Beamforming Report Parameter From NDP-A
//   Store parameter when valid NDP-A is received
//
//   In case of Broadcast HE NDPA, Ng and Codebook are extracted from NDPA
//   else                          Ng and Codebook are from registers
//                                 Codebook is forced to '1' in case of MU feedback with Ng=16
//                                 (MU Feedback, Ng=16 and codebook=0 is invalid)
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
   begin
      bfrDstAddr           <= 48'b0;
      bfrSoundingDialToken <= 6'b0;
      bfrFeedbackType      <= 2'b0;
      bfrHE                <= 1'b0;
      bfrCodebook          <= 1'b0;
      bfrGrouping          <= 2'b0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      bfrDstAddr           <= 48'b0;
      bfrSoundingDialToken <= 6'b0;
      bfrFeedbackType      <= 2'b0;
      bfrHE                <= 1'b0;
      bfrCodebook          <= 1'b0;
      bfrGrouping          <= 2'b0;
   end
   else if (changeToCheckNDPA)
   begin
      bfrDstAddr           <= rxAddr2;
      bfrSoundingDialToken <= rxNDPASoundingDialToken;
      bfrFeedbackType      <= rxNDPAFeedbackType;
      bfrHE                <= rxNDPAHE;

      if (rxNDPAHE && rxAddr1=={48{1'b1}})
      begin
         bfrCodebook       <= rxNDPACodebook;
         bfrGrouping       <= {1'b0,rxNDPANg};
      end
      else if (rxNDPAHE && rxAddr1!={48{1'b1}} && rxNDPAFeedbackType==2'd1 && bfmeeGrouping==2'd1)
      begin
         bfrCodebook       <= 1'b1;
         bfrGrouping       <= bfmeeGrouping;
      end
      else
      begin
         bfrCodebook       <= bfmeeCodebook;
         bfrGrouping       <= bfmeeGrouping;
      end
   end
end


// Beamforming Report Reponse should be send after NDP
// - if VHT NDPA with our User Info field is the first User Info field
// - if HE  NDPA with only one User Info field
//
// Note: rxNDPASTAIndexIsFirst is always '0' for HE NDP-A
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrResponseAfterNDP <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrResponseAfterNDP <= 1'b0;
   else if (changeToCheckNDPA)
      bfrResponseAfterNDP <= rxNDPAHE && rxAddr1!={48{1'b1}} || rxNDPASTAIndexIsFirst;
end


// HE NDPA With only one user info field flag
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      heNDPAUnicast <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      heNDPAUnicast <= 1'b0;
   else if (changeToCheckNDPA)
      heNDPAUnicast <= rxNDPAHE && rxAddr1!={48{1'b1}};
end


// Capture Beamforming Report Nc Index Parameter
// VHT NDPA
//   if MU, Nc is the smaller of Nc from NPDA and Nc supported by the beamformee
//   if SU, Nc is the smaller of Nr from NDP  and Nc supported by the beamformee
// HE NDPA
//   if Broadcast NDPA, Nc is the smaller of Nc from NPDA and Nc supported by the beamformee
//   if Unicast   NPDA, Nc is the smaller of Nr from NDP  and Nc supported by the beamformee
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrNc <= 3'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrNc <= 3'b0;
   else if ((changeToCheckNDP && ~rxNDPAHE && (rxNDPAFeedbackType == MU)) ||
            (changeToCheckNDP &&  rxNDPAHE && ~heNDPAUnicast            ))
   begin 
      if (rxNDPANcIndex <= bfmeeNc)
         bfrNc <= rxNDPANcIndex;
      else
         bfrNc <= bfmeeNc;
   end
   else if ((changeToCheckNDP && ~rxNDPAHE && (rxNDPAFeedbackType == SU)) ||
            (changeToCheckNDP &&  rxNDPAHE &&  heNDPAUnicast            ))
   begin
      if (rxNsts <= bfmeeNc)
         bfrNc <= rxNsts;
      else
         bfrNc <= bfmeeNc;
   end
end


// Capture Beamforming Report Nr Parameter
//   Store parameter when NDP is received
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrNr   <= 3'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrNr   <= 3'b0;
   else if (changeToCheckNDP)
      bfrNr   <= rxNsts;
end


// Capture Beamforming Report chBW parameter
//   VHT NDPA                        : chBW is CH_BW           of the NDP   rxVector
//   HE NDPA received in HT,VHT or HE: chBW is CH_BW           of the NDP-A rxVector
//   HE NDPA received in NON-HT-DUP  : chBW is CH_BW_IN_NON_HT of the NDP-A rxVector
//   HE NDPA received in NON-HT      : chBW is 20MHz
//
//   In HE Calibration, chBW is extracted from the HE NDP
//   (should be same as chBW annonced in the HE NDPA)
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrChBW <= 2'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrChBW <= 2'b0;
   else if (changeToCheckNDP)
      bfrChBW <= rxChBW[1:0];
/*
   else if (~rxNDPAHE & changeToCheckNDP)
      bfrChBW <= rxChBW[1:0];
   else if ( rxNDPAHE & changeToCheckNDPA & rxFormatMod==4'd1)
      bfrChBW <= rxChBWInNonHT;
   else if ( rxNDPAHE & changeToCheckNDPA)
      bfrChBW <= rxChBW[1:0];
*/
end


// Capture Beamforming Report RU Parameters
//   Store parameter when HE  NDP-A is received
//   Clear parameter when VHT NDP-A is received
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
   begin
      bfrRUStartIndex <= 7'd0;
      bfrRUEndIndex   <= 7'd0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      bfrRUStartIndex <= 7'd0;
      bfrRUEndIndex   <= 7'd0;
   end
   else if (changeToCheckNDPA &&  rxNDPAHE)
   begin
      bfrRUStartIndex <= rxNDPARUStartIndex;
      bfrRUEndIndex   <= rxNDPARUEndIndex;
   end
   else if (changeToCheckNDPA && ~rxNDPAHE)
   begin
      bfrRUStartIndex <= 7'd0;
      bfrRUEndIndex   <= 7'd0;
   end
end


// Check BFR_POLL / BFRP Trigger
//   check if BFR-POLL TA is equal to the NDPA TA
//   Segment Retransmission is not checked
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      incorrectBFRPollRcved_p <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      incorrectBFRPollRcved_p <= 1'b0;
   else if (isActive && (rxAddr2 != bfrDstAddr) && rxBFRPollValid_p ||
            isActive && (rxAddr2 != bfrDstAddr) && rxTriggerValid_p && rxTriggerType==4'd1)
      incorrectBFRPollRcved_p <= 1'b1;
   else
      incorrectBFRPollRcved_p <= 1'b0;
end


// Configure Beamforming Report HW Accelerator
//   Beamforming Report HW Accelerator is started when enter in Active State
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrStart <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrStart <= 1'b0;
   else if (isComputeBFR)
      bfrStart <= 1'b1;
   else if (bfrDone)
      bfrStart <= 1'b0;
end


// Beamforming Report Frame Length Computation phase 
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrLengthCompPhase <= 3'd7;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrLengthCompPhase <= 3'd7;
   else if (!isComputeBFRLen)
      bfrLengthCompPhase <= 3'd0;
   else if (bfrLengthCompPhase < 3'd4)
      bfrLengthCompPhase <= bfrLengthCompPhase + 3'd1;
end
assign bfrLengthCompDone = (bfrLengthCompPhase == 3'd4) ? 1'b1 : 1'b0;


// Number of angles of the Compressed Beamforming Report (Na)
always @*
begin
   case ({bfrNr,bfrNc})
   {3'd1,3'd0}: bfrNa = 6'd2;  // 2x1
   {3'd1,3'd1}: bfrNa = 6'd2;  // 2x2
   {3'd2,3'd0}: bfrNa = 6'd4;  // 3x1
   {3'd2,3'd1}: bfrNa = 6'd6;  // 3x2
   {3'd2,3'd2}: bfrNa = 6'd6;  // 3x3
   {3'd3,3'd0}: bfrNa = 6'd6;  // 4x1
   {3'd3,3'd1}: bfrNa = 6'd10; // 4x2
   {3'd3,3'd2}: bfrNa = 6'd12; // 4x3
   {3'd3,3'd3}: bfrNa = 6'd12; // 4x4
   {3'd4,3'd0}: bfrNa = 6'd8;  // 5x1
   {3'd4,3'd1}: bfrNa = 6'd14; // 5x2
   {3'd4,3'd2}: bfrNa = 6'd18; // 5x3
   {3'd4,3'd3}: bfrNa = 6'd20; // 5x4
   {3'd4,3'd4}: bfrNa = 6'd20; // 5x5
   {3'd5,3'd0}: bfrNa = 6'd10; // 6x1
   {3'd5,3'd1}: bfrNa = 6'd18; // 6x2
   {3'd5,3'd2}: bfrNa = 6'd24; // 6x3
   {3'd5,3'd3}: bfrNa = 6'd28; // 6x4
   {3'd5,3'd4}: bfrNa = 6'd30; // 6x5
   {3'd5,3'd5}: bfrNa = 6'd30; // 6x6
   {3'd6,3'd0}: bfrNa = 6'd12; // 7x1
   {3'd6,3'd1}: bfrNa = 6'd22; // 7x2
   {3'd6,3'd2}: bfrNa = 6'd30; // 7x3
   {3'd6,3'd3}: bfrNa = 6'd36; // 7x4
   {3'd6,3'd4}: bfrNa = 6'd40; // 7x5
   {3'd6,3'd5}: bfrNa = 6'd42; // 7x6
   {3'd6,3'd6}: bfrNa = 6'd42; // 7x7
   {3'd7,3'd0}: bfrNa = 6'd14; // 8x1
   {3'd7,3'd1}: bfrNa = 6'd26; // 8x2
   {3'd7,3'd2}: bfrNa = 6'd36; // 8x3
   {3'd7,3'd3}: bfrNa = 6'd44; // 8x4
   {3'd7,3'd4}: bfrNa = 6'd50; // 8x5
   {3'd7,3'd5}: bfrNa = 6'd54; // 8x6
   {3'd7,3'd6}: bfrNa = 6'd56; // 8x7
   {3'd7,3'd7}: bfrNa = 6'd56; // 8x8
   default    : bfrNa = 6'd0;  // Invalid
   endcase
end


// Number of subcarriers of the VHT Compressed Beamforming Report (Ns)
always @*
begin
   case ({bfrChBW, bfrGrouping})
   {2'd0,2'd0}: vhtBFRNs = 9'd52;   //  20Mhz, Grouping 1
   {2'd0,2'd1}: vhtBFRNs = 9'd30;   //  20Mhz, Grouping 2
   {2'd0,2'd2}: vhtBFRNs = 9'd16;   //  20Mhz, Grouping 4
   {2'd1,2'd0}: vhtBFRNs = 9'd108;  //  40Mhz, Grouping 1
   {2'd1,2'd1}: vhtBFRNs = 9'd58;   //  40Mhz, Grouping 2
   {2'd1,2'd2}: vhtBFRNs = 9'd30;   //  40Mhz, Grouping 4
   {2'd2,2'd0}: vhtBFRNs = 9'd234;  //  80Mhz, Grouping 1
   {2'd2,2'd1}: vhtBFRNs = 9'd122;  //  80Mhz, Grouping 2
   {2'd2,2'd2}: vhtBFRNs = 9'd62;   //  80Mhz, Grouping 4
   {2'd3,2'd0}: vhtBFRNs = 9'd468;  // 160Mhz, Grouping 1
   {2'd3,2'd1}: vhtBFRNs = 9'd244;  // 160Mhz, Grouping 2
   {2'd3,2'd2}: vhtBFRNs = 9'd124;  // 160Mhz, Grouping 4
   default    : vhtBFRNs = 9'd0;    // Invalid
   endcase
end


// Number of subcarriers of the VHT MU Exclusive Report (Ns')
always @*
begin
   case ({bfrChBW, bfrGrouping})
   {2'd0,2'd0}: vhtBFRNsp = 8'd30;  //  20Mhz, Grouping 1
   {2'd0,2'd1}: vhtBFRNsp = 8'd16;  //  20Mhz, Grouping 2
   {2'd0,2'd2}: vhtBFRNsp = 8'd10;  //  20Mhz, Grouping 4
   {2'd1,2'd0}: vhtBFRNsp = 8'd58;  //  40Mhz, Grouping 1
   {2'd1,2'd1}: vhtBFRNsp = 8'd30;  //  40Mhz, Grouping 2
   {2'd1,2'd2}: vhtBFRNsp = 8'd16;  //  40Mhz, Grouping 4
   {2'd2,2'd0}: vhtBFRNsp = 8'd122; //  80Mhz, Grouping 1
   {2'd2,2'd1}: vhtBFRNsp = 8'd62;  //  80Mhz, Grouping 2
   {2'd2,2'd2}: vhtBFRNsp = 8'd32;  //  80Mhz, Grouping 4
   {2'd3,2'd0}: vhtBFRNsp = 8'd244; // 160Mhz, Grouping 1
   {2'd3,2'd1}: vhtBFRNsp = 8'd124; // 160Mhz, Grouping 2
   {2'd3,2'd2}: vhtBFRNsp = 8'd64;  // 160Mhz, Grouping 4
   default    : vhtBFRNsp = 8'd0;   // Invalid
   endcase
end


// Size of angles of the Compressed Beamforming Report (bPhi + bPsi)
always @*
begin
   case ({bfrFeedbackType, bfrCodebook})
   {SU,1'b0}: bfrBPhiBPsi = 5'd6;
   {SU,1'b1}: bfrBPhiBPsi = 5'd10;
   {MU,1'b0}: bfrBPhiBPsi = 5'd12;
   {MU,1'b1}: bfrBPhiBPsi = 5'd16;
   default  : bfrBPhiBPsi = 5'd0; // Invalid
   endcase
end


// HE start 26-tone index
// Extracted from IEEE802.11ax/D2.1 Table 9-76c, Table 9-76d & Table 9-76e
always @*
begin
   case ({bfrChBW, bfrGrouping, bfrRUStartIndex})
   //BW=20MHz, Ng=4
   {2'd0,2'd0,7'd0} : heBFRSTone = 8'd0;
   {2'd0,2'd0,7'd1} : heBFRSTone = 8'd7;
   {2'd0,2'd0,7'd2} : heBFRSTone = 8'd14;
   {2'd0,2'd0,7'd3} : heBFRSTone = 8'd20;
   {2'd0,2'd0,7'd4} : heBFRSTone = 8'd27;
   {2'd0,2'd0,7'd5} : heBFRSTone = 8'd36;
   {2'd0,2'd0,7'd6} : heBFRSTone = 8'd42;
   {2'd0,2'd0,7'd7} : heBFRSTone = 8'd49;
   {2'd0,2'd0,7'd8} : heBFRSTone = 8'd56;
   //BW=20MHz, Ng=16
   {2'd0,2'd1,7'd0} : heBFRSTone = 8'd0;
   {2'd0,2'd1,7'd1} : heBFRSTone = 8'd3;
   {2'd0,2'd1,7'd2} : heBFRSTone = 8'd4;
   {2'd0,2'd1,7'd3} : heBFRSTone = 8'd5;
   {2'd0,2'd1,7'd4} : heBFRSTone = 8'd7;
   {2'd0,2'd1,7'd5} : heBFRSTone = 8'd11;
   {2'd0,2'd1,7'd6} : heBFRSTone = 8'd13;
   {2'd0,2'd1,7'd7} : heBFRSTone = 8'd15;
   {2'd0,2'd1,7'd8} : heBFRSTone = 8'd16;
   //BW=40MHz, Ng=4
   {2'd1,2'd0,7'd0} : heBFRSTone = 8'd0;
   {2'd1,2'd0,7'd1} : heBFRSTone = 8'd6;
   {2'd1,2'd0,7'd2} : heBFRSTone = 8'd13;
   {2'd1,2'd0,7'd3} : heBFRSTone = 8'd20;
   {2'd1,2'd0,7'd4} : heBFRSTone = 8'd27;
   {2'd1,2'd0,7'd5} : heBFRSTone = 8'd33;
   {2'd1,2'd0,7'd6} : heBFRSTone = 8'd40;
   {2'd1,2'd0,7'd7} : heBFRSTone = 8'd47;
   {2'd1,2'd0,7'd8} : heBFRSTone = 8'd53;
   {2'd1,2'd0,7'd9} : heBFRSTone = 8'd61;
   {2'd1,2'd0,7'd10}: heBFRSTone = 8'd67;
   {2'd1,2'd0,7'd11}: heBFRSTone = 8'd74;
   {2'd1,2'd0,7'd12}: heBFRSTone = 8'd81;
   {2'd1,2'd0,7'd13}: heBFRSTone = 8'd87;
   {2'd1,2'd0,7'd14}: heBFRSTone = 8'd94;
   {2'd1,2'd0,7'd15}: heBFRSTone = 8'd101;
   {2'd1,2'd0,7'd16}: heBFRSTone = 8'd108;
   {2'd1,2'd0,7'd17}: heBFRSTone = 8'd114;
   //BW=40MHz, Ng=16
   {2'd1,2'd1,7'd0} : heBFRSTone = 8'd0;
   {2'd1,2'd1,7'd1} : heBFRSTone = 8'd1;
   {2'd1,2'd1,7'd2} : heBFRSTone = 8'd3;
   {2'd1,2'd1,7'd3} : heBFRSTone = 8'd5;
   {2'd1,2'd1,7'd4} : heBFRSTone = 8'd6;
   {2'd1,2'd1,7'd5} : heBFRSTone = 8'd8;
   {2'd1,2'd1,7'd6} : heBFRSTone = 8'd10;
   {2'd1,2'd1,7'd7} : heBFRSTone = 8'd11;
   {2'd1,2'd1,7'd8} : heBFRSTone = 8'd13;
   {2'd1,2'd1,7'd9} : heBFRSTone = 8'd16;
   {2'd1,2'd1,7'd10}: heBFRSTone = 8'd17;
   {2'd1,2'd1,7'd11}: heBFRSTone = 8'd19;
   {2'd1,2'd1,7'd12}: heBFRSTone = 8'd21;
   {2'd1,2'd1,7'd13}: heBFRSTone = 8'd22;
   {2'd1,2'd1,7'd14}: heBFRSTone = 8'd24;
   {2'd1,2'd1,7'd15}: heBFRSTone = 8'd26;
   {2'd1,2'd1,7'd16}: heBFRSTone = 8'd27;
   {2'd1,2'd1,7'd17}: heBFRSTone = 8'd29;
   //BW=80MHz/160MHz, Ng=4
   {2'd2,2'd0,7'd0} ,{2'd3,2'd0,7'd0} ,{2'd3,2'd0,7'd37}: heBFRSTone = 8'd0;
   {2'd2,2'd0,7'd1} ,{2'd3,2'd0,7'd1} ,{2'd3,2'd0,7'd38}: heBFRSTone = 8'd6;
   {2'd2,2'd0,7'd2} ,{2'd3,2'd0,7'd2} ,{2'd3,2'd0,7'd39}: heBFRSTone = 8'd13;
   {2'd2,2'd0,7'd3} ,{2'd3,2'd0,7'd3} ,{2'd3,2'd0,7'd40}: heBFRSTone = 8'd20;
   {2'd2,2'd0,7'd4} ,{2'd3,2'd0,7'd4} ,{2'd3,2'd0,7'd41}: heBFRSTone = 8'd27;
   {2'd2,2'd0,7'd5} ,{2'd3,2'd0,7'd5} ,{2'd3,2'd0,7'd42}: heBFRSTone = 8'd33;
   {2'd2,2'd0,7'd6} ,{2'd3,2'd0,7'd6} ,{2'd3,2'd0,7'd43}: heBFRSTone = 8'd40;
   {2'd2,2'd0,7'd7} ,{2'd3,2'd0,7'd7} ,{2'd3,2'd0,7'd44}: heBFRSTone = 8'd47;
   {2'd2,2'd0,7'd8} ,{2'd3,2'd0,7'd8} ,{2'd3,2'd0,7'd45}: heBFRSTone = 8'd53;
   {2'd2,2'd0,7'd9} ,{2'd3,2'd0,7'd9} ,{2'd3,2'd0,7'd46}: heBFRSTone = 8'd60;
   {2'd2,2'd0,7'd10},{2'd3,2'd0,7'd10},{2'd3,2'd0,7'd47}: heBFRSTone = 8'd67;
   {2'd2,2'd0,7'd11},{2'd3,2'd0,7'd11},{2'd3,2'd0,7'd48}: heBFRSTone = 8'd74;
   {2'd2,2'd0,7'd12},{2'd3,2'd0,7'd12},{2'd3,2'd0,7'd49}: heBFRSTone = 8'd80;
   {2'd2,2'd0,7'd13},{2'd3,2'd0,7'd13},{2'd3,2'd0,7'd50}: heBFRSTone = 8'd87;
   {2'd2,2'd0,7'd14},{2'd3,2'd0,7'd14},{2'd3,2'd0,7'd51}: heBFRSTone = 8'd94;
   {2'd2,2'd0,7'd15},{2'd3,2'd0,7'd15},{2'd3,2'd0,7'd52}: heBFRSTone = 8'd100;
   {2'd2,2'd0,7'd16},{2'd3,2'd0,7'd16},{2'd3,2'd0,7'd53}: heBFRSTone = 8'd107;
   {2'd2,2'd0,7'd17},{2'd3,2'd0,7'd17},{2'd3,2'd0,7'd54}: heBFRSTone = 8'd114;
   {2'd2,2'd0,7'd18},{2'd3,2'd0,7'd18},{2'd3,2'd0,7'd55}: heBFRSTone = 8'd121;
   {2'd2,2'd0,7'd19},{2'd3,2'd0,7'd19},{2'd3,2'd0,7'd56}: heBFRSTone = 8'd128;
   {2'd2,2'd0,7'd20},{2'd3,2'd0,7'd20},{2'd3,2'd0,7'd57}: heBFRSTone = 8'd135;
   {2'd2,2'd0,7'd21},{2'd3,2'd0,7'd21},{2'd3,2'd0,7'd58}: heBFRSTone = 8'd142;
   {2'd2,2'd0,7'd22},{2'd3,2'd0,7'd22},{2'd3,2'd0,7'd59}: heBFRSTone = 8'd148;
   {2'd2,2'd0,7'd23},{2'd3,2'd0,7'd23},{2'd3,2'd0,7'd60}: heBFRSTone = 8'd155;
   {2'd2,2'd0,7'd24},{2'd3,2'd0,7'd24},{2'd3,2'd0,7'd61}: heBFRSTone = 8'd162;
   {2'd2,2'd0,7'd25},{2'd3,2'd0,7'd25},{2'd3,2'd0,7'd62}: heBFRSTone = 8'd168;
   {2'd2,2'd0,7'd26},{2'd3,2'd0,7'd26},{2'd3,2'd0,7'd63}: heBFRSTone = 8'd175;
   {2'd2,2'd0,7'd27},{2'd3,2'd0,7'd27},{2'd3,2'd0,7'd64}: heBFRSTone = 8'd182;
   {2'd2,2'd0,7'd28},{2'd3,2'd0,7'd28},{2'd3,2'd0,7'd65}: heBFRSTone = 8'd189;
   {2'd2,2'd0,7'd29},{2'd3,2'd0,7'd29},{2'd3,2'd0,7'd66}: heBFRSTone = 8'd195;
   {2'd2,2'd0,7'd30},{2'd3,2'd0,7'd30},{2'd3,2'd0,7'd67}: heBFRSTone = 8'd202;
   {2'd2,2'd0,7'd31},{2'd3,2'd0,7'd31},{2'd3,2'd0,7'd68}: heBFRSTone = 8'd209;
   {2'd2,2'd0,7'd32},{2'd3,2'd0,7'd32},{2'd3,2'd0,7'd69}: heBFRSTone = 8'd215;
   {2'd2,2'd0,7'd33},{2'd3,2'd0,7'd33},{2'd3,2'd0,7'd70}: heBFRSTone = 8'd222;
   {2'd2,2'd0,7'd34},{2'd3,2'd0,7'd34},{2'd3,2'd0,7'd71}: heBFRSTone = 8'd229;
   {2'd2,2'd0,7'd35},{2'd3,2'd0,7'd35},{2'd3,2'd0,7'd72}: heBFRSTone = 8'd236;
   {2'd2,2'd0,7'd36},{2'd3,2'd0,7'd36},{2'd3,2'd0,7'd73}: heBFRSTone = 8'd242;
   //BW=80MHz/160MHz, Ng=16
   {2'd2,2'd1,7'd0} ,{2'd3,2'd1,7'd0} ,{2'd3,2'd1,7'd37}: heBFRSTone = 8'd0;
   {2'd2,2'd1,7'd1} ,{2'd3,2'd1,7'd1} ,{2'd3,2'd1,7'd38}: heBFRSTone = 8'd1;
   {2'd2,2'd1,7'd2} ,{2'd3,2'd1,7'd2} ,{2'd3,2'd1,7'd39}: heBFRSTone = 8'd3;
   {2'd2,2'd1,7'd3} ,{2'd3,2'd1,7'd3} ,{2'd3,2'd1,7'd40}: heBFRSTone = 8'd5;
   {2'd2,2'd1,7'd4} ,{2'd3,2'd1,7'd4} ,{2'd3,2'd1,7'd41}: heBFRSTone = 8'd6;
   {2'd2,2'd1,7'd5} ,{2'd3,2'd1,7'd5} ,{2'd3,2'd1,7'd42}: heBFRSTone = 8'd8;
   {2'd2,2'd1,7'd6} ,{2'd3,2'd1,7'd6} ,{2'd3,2'd1,7'd43}: heBFRSTone = 8'd10;
   {2'd2,2'd1,7'd7} ,{2'd3,2'd1,7'd7} ,{2'd3,2'd1,7'd44}: heBFRSTone = 8'd11;
   {2'd2,2'd1,7'd8} ,{2'd3,2'd1,7'd8} ,{2'd3,2'd1,7'd45}: heBFRSTone = 8'd13;
   {2'd2,2'd1,7'd9} ,{2'd3,2'd1,7'd9} ,{2'd3,2'd1,7'd46}: heBFRSTone = 8'd15;
   {2'd2,2'd1,7'd10},{2'd3,2'd1,7'd10},{2'd3,2'd1,7'd47}: heBFRSTone = 8'd16;
   {2'd2,2'd1,7'd11},{2'd3,2'd1,7'd11},{2'd3,2'd1,7'd48}: heBFRSTone = 8'd18;
   {2'd2,2'd1,7'd12},{2'd3,2'd1,7'd12},{2'd3,2'd1,7'd49}: heBFRSTone = 8'd20;
   {2'd2,2'd1,7'd13},{2'd3,2'd1,7'd13},{2'd3,2'd1,7'd50}: heBFRSTone = 8'd21;
   {2'd2,2'd1,7'd14},{2'd3,2'd1,7'd14},{2'd3,2'd1,7'd51}: heBFRSTone = 8'd23;
   {2'd2,2'd1,7'd15},{2'd3,2'd1,7'd15},{2'd3,2'd1,7'd52}: heBFRSTone = 8'd25;
   {2'd2,2'd1,7'd16},{2'd3,2'd1,7'd16},{2'd3,2'd1,7'd53}: heBFRSTone = 8'd26;
   {2'd2,2'd1,7'd17},{2'd3,2'd1,7'd17},{2'd3,2'd1,7'd54}: heBFRSTone = 8'd28;
   {2'd2,2'd1,7'd18},{2'd3,2'd1,7'd18},{2'd3,2'd1,7'd55}: heBFRSTone = 8'd30;
   {2'd2,2'd1,7'd19},{2'd3,2'd1,7'd19},{2'd3,2'd1,7'd56}: heBFRSTone = 8'd32;
   {2'd2,2'd1,7'd20},{2'd3,2'd1,7'd20},{2'd3,2'd1,7'd57}: heBFRSTone = 8'd34;
   {2'd2,2'd1,7'd21},{2'd3,2'd1,7'd21},{2'd3,2'd1,7'd58}: heBFRSTone = 8'd36;
   {2'd2,2'd1,7'd22},{2'd3,2'd1,7'd22},{2'd3,2'd1,7'd59}: heBFRSTone = 8'd37;
   {2'd2,2'd1,7'd23},{2'd3,2'd1,7'd23},{2'd3,2'd1,7'd60}: heBFRSTone = 8'd39;
   {2'd2,2'd1,7'd24},{2'd3,2'd1,7'd24},{2'd3,2'd1,7'd61}: heBFRSTone = 8'd41;
   {2'd2,2'd1,7'd25},{2'd3,2'd1,7'd25},{2'd3,2'd1,7'd62}: heBFRSTone = 8'd42;
   {2'd2,2'd1,7'd26},{2'd3,2'd1,7'd26},{2'd3,2'd1,7'd63}: heBFRSTone = 8'd44;
   {2'd2,2'd1,7'd27},{2'd3,2'd1,7'd27},{2'd3,2'd1,7'd64}: heBFRSTone = 8'd46;
   {2'd2,2'd1,7'd28},{2'd3,2'd1,7'd28},{2'd3,2'd1,7'd65}: heBFRSTone = 8'd48;
   {2'd2,2'd1,7'd29},{2'd3,2'd1,7'd29},{2'd3,2'd1,7'd66}: heBFRSTone = 8'd49;
   {2'd2,2'd1,7'd30},{2'd3,2'd1,7'd30},{2'd3,2'd1,7'd67}: heBFRSTone = 8'd51;
   {2'd2,2'd1,7'd31},{2'd3,2'd1,7'd31},{2'd3,2'd1,7'd68}: heBFRSTone = 8'd53;
   {2'd2,2'd1,7'd32},{2'd3,2'd1,7'd32},{2'd3,2'd1,7'd69}: heBFRSTone = 8'd54;
   {2'd2,2'd1,7'd33},{2'd3,2'd1,7'd33},{2'd3,2'd1,7'd70}: heBFRSTone = 8'd56;
   {2'd2,2'd1,7'd34},{2'd3,2'd1,7'd34},{2'd3,2'd1,7'd71}: heBFRSTone = 8'd58;
   {2'd2,2'd1,7'd35},{2'd3,2'd1,7'd35},{2'd3,2'd1,7'd72}: heBFRSTone = 8'd59;
   {2'd2,2'd1,7'd36},{2'd3,2'd1,7'd36},{2'd3,2'd1,7'd73}: heBFRSTone = 8'd61;
   //Invalid
   default:                                               heBFRSTone = 8'd0;
   endcase
end

// HE End 26-tone index
// Extracted from IEEE802.11ax/D2.1 Table 9-76c, Table 9-76d & Table 9-76e
always @*
begin
   case ({bfrChBW, bfrGrouping, bfrRUEndIndex})
   //BW=20MHz, Ng=4
   {2'd0,2'd0,7'd0} : heBFRETone = 8'd8;
   {2'd0,2'd0,7'd1} : heBFRETone = 8'd15;
   {2'd0,2'd0,7'd2} : heBFRETone = 8'd22;
   {2'd0,2'd0,7'd3} : heBFRETone = 8'd28;
   {2'd0,2'd0,7'd4} : heBFRETone = 8'd37;
   {2'd0,2'd0,7'd5} : heBFRETone = 8'd44;
   {2'd0,2'd0,7'd6} : heBFRETone = 8'd50;
   {2'd0,2'd0,7'd7} : heBFRETone = 8'd57;
   {2'd0,2'd0,7'd8} : heBFRETone = 8'd64;
   //BW=20MHz, Ng=16
   {2'd0,2'd1,7'd0} : heBFRETone = 8'd4;
   {2'd0,2'd1,7'd1} : heBFRETone = 8'd5;
   {2'd0,2'd1,7'd2} : heBFRETone = 8'd7;
   {2'd0,2'd1,7'd3} : heBFRETone = 8'd9;
   {2'd0,2'd1,7'd4} : heBFRETone = 8'd13;
   {2'd0,2'd1,7'd5} : heBFRETone = 8'd15;
   {2'd0,2'd1,7'd6} : heBFRETone = 8'd16;
   {2'd0,2'd1,7'd7} : heBFRETone = 8'd17;
   {2'd0,2'd1,7'd8} : heBFRETone = 8'd20;
   //BW=40MHz, Ng=4
   {2'd1,2'd0,7'd0} : heBFRETone = 8'd8;
   {2'd1,2'd0,7'd1} : heBFRETone = 8'd14;
   {2'd1,2'd0,7'd2} : heBFRETone = 8'd21;
   {2'd1,2'd0,7'd3} : heBFRETone = 8'd28;
   {2'd1,2'd0,7'd4} : heBFRETone = 8'd35;
   {2'd1,2'd0,7'd5} : heBFRETone = 8'd41;
   {2'd1,2'd0,7'd6} : heBFRETone = 8'd48;
   {2'd1,2'd0,7'd7} : heBFRETone = 8'd55;
   {2'd1,2'd0,7'd8} : heBFRETone = 8'd61;
   {2'd1,2'd0,7'd9} : heBFRETone = 8'd69;
   {2'd1,2'd0,7'd10}: heBFRETone = 8'd75;
   {2'd1,2'd0,7'd11}: heBFRETone = 8'd82;
   {2'd1,2'd0,7'd12}: heBFRETone = 8'd89;
   {2'd1,2'd0,7'd13}: heBFRETone = 8'd95;
   {2'd1,2'd0,7'd14}: heBFRETone = 8'd102;
   {2'd1,2'd0,7'd15}: heBFRETone = 8'd109;
   {2'd1,2'd0,7'd16}: heBFRETone = 8'd116;
   {2'd1,2'd0,7'd17}: heBFRETone = 8'd122;
   //BW=40MHz, Ng=16
   {2'd1,2'd1,7'd0} : heBFRETone = 8'd3;
   {2'd1,2'd1,7'd1} : heBFRETone = 8'd5;
   {2'd1,2'd1,7'd2} : heBFRETone = 8'd6;
   {2'd1,2'd1,7'd3} : heBFRETone = 8'd8;
   {2'd1,2'd1,7'd4} : heBFRETone = 8'd10;
   {2'd1,2'd1,7'd5} : heBFRETone = 8'd11;
   {2'd1,2'd1,7'd6} : heBFRETone = 8'd13;
   {2'd1,2'd1,7'd7} : heBFRETone = 8'd15;
   {2'd1,2'd1,7'd8} : heBFRETone = 8'd16;
   {2'd1,2'd1,7'd9} : heBFRETone = 8'd19;
   {2'd1,2'd1,7'd10}: heBFRETone = 8'd21;
   {2'd1,2'd1,7'd11}: heBFRETone = 8'd22;
   {2'd1,2'd1,7'd12}: heBFRETone = 8'd24;
   {2'd1,2'd1,7'd13}: heBFRETone = 8'd26;
   {2'd1,2'd1,7'd14}: heBFRETone = 8'd27;
   {2'd1,2'd1,7'd15}: heBFRETone = 8'd29;
   {2'd1,2'd1,7'd16}: heBFRETone = 8'd31;
   {2'd1,2'd1,7'd17}: heBFRETone = 8'd32;
   //BW=80MHz/160MHz, Ng=4
   {2'd2,2'd0,7'd0} ,{2'd3,2'd0,7'd0} ,{2'd3,2'd0,7'd37}: heBFRETone = 8'd8;
   {2'd2,2'd0,7'd1} ,{2'd3,2'd0,7'd1} ,{2'd3,2'd0,7'd38}: heBFRETone = 8'd14;
   {2'd2,2'd0,7'd2} ,{2'd3,2'd0,7'd2} ,{2'd3,2'd0,7'd39}: heBFRETone = 8'd21;
   {2'd2,2'd0,7'd3} ,{2'd3,2'd0,7'd3} ,{2'd3,2'd0,7'd40}: heBFRETone = 8'd28;
   {2'd2,2'd0,7'd4} ,{2'd3,2'd0,7'd4} ,{2'd3,2'd0,7'd41}: heBFRETone = 8'd35;
   {2'd2,2'd0,7'd5} ,{2'd3,2'd0,7'd5} ,{2'd3,2'd0,7'd42}: heBFRETone = 8'd41;
   {2'd2,2'd0,7'd6} ,{2'd3,2'd0,7'd6} ,{2'd3,2'd0,7'd43}: heBFRETone = 8'd48;
   {2'd2,2'd0,7'd7} ,{2'd3,2'd0,7'd7} ,{2'd3,2'd0,7'd44}: heBFRETone = 8'd55;
   {2'd2,2'd0,7'd8} ,{2'd3,2'd0,7'd8} ,{2'd3,2'd0,7'd45}: heBFRETone = 8'd61;
   {2'd2,2'd0,7'd9} ,{2'd3,2'd0,7'd9} ,{2'd3,2'd0,7'd46}: heBFRETone = 8'd68;
   {2'd2,2'd0,7'd10},{2'd3,2'd0,7'd10},{2'd3,2'd0,7'd47}: heBFRETone = 8'd75;
   {2'd2,2'd0,7'd11},{2'd3,2'd0,7'd11},{2'd3,2'd0,7'd48}: heBFRETone = 8'd82;
   {2'd2,2'd0,7'd12},{2'd3,2'd0,7'd12},{2'd3,2'd0,7'd49}: heBFRETone = 8'd88;
   {2'd2,2'd0,7'd13},{2'd3,2'd0,7'd13},{2'd3,2'd0,7'd50}: heBFRETone = 8'd95;
   {2'd2,2'd0,7'd14},{2'd3,2'd0,7'd14},{2'd3,2'd0,7'd51}: heBFRETone = 8'd102;
   {2'd2,2'd0,7'd15},{2'd3,2'd0,7'd15},{2'd3,2'd0,7'd52}: heBFRETone = 8'd108;
   {2'd2,2'd0,7'd16},{2'd3,2'd0,7'd16},{2'd3,2'd0,7'd53}: heBFRETone = 8'd115;
   {2'd2,2'd0,7'd17},{2'd3,2'd0,7'd17},{2'd3,2'd0,7'd54}: heBFRETone = 8'd122;
   {2'd2,2'd0,7'd18},{2'd3,2'd0,7'd18},{2'd3,2'd0,7'd55}: heBFRETone = 8'd129;
   {2'd2,2'd0,7'd19},{2'd3,2'd0,7'd19},{2'd3,2'd0,7'd56}: heBFRETone = 8'd136;
   {2'd2,2'd0,7'd20},{2'd3,2'd0,7'd20},{2'd3,2'd0,7'd57}: heBFRETone = 8'd143;
   {2'd2,2'd0,7'd21},{2'd3,2'd0,7'd21},{2'd3,2'd0,7'd58}: heBFRETone = 8'd150;
   {2'd2,2'd0,7'd22},{2'd3,2'd0,7'd22},{2'd3,2'd0,7'd59}: heBFRETone = 8'd156;
   {2'd2,2'd0,7'd23},{2'd3,2'd0,7'd23},{2'd3,2'd0,7'd60}: heBFRETone = 8'd163;
   {2'd2,2'd0,7'd24},{2'd3,2'd0,7'd24},{2'd3,2'd0,7'd61}: heBFRETone = 8'd170;
   {2'd2,2'd0,7'd25},{2'd3,2'd0,7'd25},{2'd3,2'd0,7'd62}: heBFRETone = 8'd176;
   {2'd2,2'd0,7'd26},{2'd3,2'd0,7'd26},{2'd3,2'd0,7'd63}: heBFRETone = 8'd183;
   {2'd2,2'd0,7'd27},{2'd3,2'd0,7'd27},{2'd3,2'd0,7'd64}: heBFRETone = 8'd190;
   {2'd2,2'd0,7'd28},{2'd3,2'd0,7'd28},{2'd3,2'd0,7'd65}: heBFRETone = 8'd197;
   {2'd2,2'd0,7'd29},{2'd3,2'd0,7'd29},{2'd3,2'd0,7'd66}: heBFRETone = 8'd203;
   {2'd2,2'd0,7'd30},{2'd3,2'd0,7'd30},{2'd3,2'd0,7'd67}: heBFRETone = 8'd210;
   {2'd2,2'd0,7'd31},{2'd3,2'd0,7'd31},{2'd3,2'd0,7'd68}: heBFRETone = 8'd217;
   {2'd2,2'd0,7'd32},{2'd3,2'd0,7'd32},{2'd3,2'd0,7'd69}: heBFRETone = 8'd223;
   {2'd2,2'd0,7'd33},{2'd3,2'd0,7'd33},{2'd3,2'd0,7'd70}: heBFRETone = 8'd230;
   {2'd2,2'd0,7'd34},{2'd3,2'd0,7'd34},{2'd3,2'd0,7'd71}: heBFRETone = 8'd237;
   {2'd2,2'd0,7'd35},{2'd3,2'd0,7'd35},{2'd3,2'd0,7'd72}: heBFRETone = 8'd244;
   {2'd2,2'd0,7'd36},{2'd3,2'd0,7'd36},{2'd3,2'd0,7'd73}: heBFRETone = 8'd250;
   //BW=80MHz/160MHz, Ng=16
   {2'd2,2'd1,7'd0} ,{2'd3,2'd1,7'd0} ,{2'd3,2'd1,7'd37}: heBFRETone = 8'd3;
   {2'd2,2'd1,7'd1} ,{2'd3,2'd1,7'd1} ,{2'd3,2'd1,7'd38}: heBFRETone = 8'd5;
   {2'd2,2'd1,7'd2} ,{2'd3,2'd1,7'd2} ,{2'd3,2'd1,7'd39}: heBFRETone = 8'd6;
   {2'd2,2'd1,7'd3} ,{2'd3,2'd1,7'd3} ,{2'd3,2'd1,7'd40}: heBFRETone = 8'd8;
   {2'd2,2'd1,7'd4} ,{2'd3,2'd1,7'd4} ,{2'd3,2'd1,7'd41}: heBFRETone = 8'd10;
   {2'd2,2'd1,7'd5} ,{2'd3,2'd1,7'd5} ,{2'd3,2'd1,7'd42}: heBFRETone = 8'd11;
   {2'd2,2'd1,7'd6} ,{2'd3,2'd1,7'd6} ,{2'd3,2'd1,7'd43}: heBFRETone = 8'd13;
   {2'd2,2'd1,7'd7} ,{2'd3,2'd1,7'd7} ,{2'd3,2'd1,7'd44}: heBFRETone = 8'd15;
   {2'd2,2'd1,7'd8} ,{2'd3,2'd1,7'd8} ,{2'd3,2'd1,7'd45}: heBFRETone = 8'd16;
   {2'd2,2'd1,7'd9} ,{2'd3,2'd1,7'd9} ,{2'd3,2'd1,7'd46}: heBFRETone = 8'd18;
   {2'd2,2'd1,7'd10},{2'd3,2'd1,7'd10},{2'd3,2'd1,7'd47}: heBFRETone = 8'd20;
   {2'd2,2'd1,7'd11},{2'd3,2'd1,7'd11},{2'd3,2'd1,7'd48}: heBFRETone = 8'd22;
   {2'd2,2'd1,7'd12},{2'd3,2'd1,7'd12},{2'd3,2'd1,7'd49}: heBFRETone = 8'd23;
   {2'd2,2'd1,7'd13},{2'd3,2'd1,7'd13},{2'd3,2'd1,7'd50}: heBFRETone = 8'd25;
   {2'd2,2'd1,7'd14},{2'd3,2'd1,7'd14},{2'd3,2'd1,7'd51}: heBFRETone = 8'd27;
   {2'd2,2'd1,7'd15},{2'd3,2'd1,7'd15},{2'd3,2'd1,7'd52}: heBFRETone = 8'd28;
   {2'd2,2'd1,7'd16},{2'd3,2'd1,7'd16},{2'd3,2'd1,7'd53}: heBFRETone = 8'd30;
   {2'd2,2'd1,7'd17},{2'd3,2'd1,7'd17},{2'd3,2'd1,7'd54}: heBFRETone = 8'd32;
   {2'd2,2'd1,7'd18},{2'd3,2'd1,7'd18},{2'd3,2'd1,7'd55}: heBFRETone = 8'd34;
   {2'd2,2'd1,7'd19},{2'd3,2'd1,7'd19},{2'd3,2'd1,7'd56}: heBFRETone = 8'd36;
   {2'd2,2'd1,7'd20},{2'd3,2'd1,7'd20},{2'd3,2'd1,7'd57}: heBFRETone = 8'd38;
   {2'd2,2'd1,7'd21},{2'd3,2'd1,7'd21},{2'd3,2'd1,7'd58}: heBFRETone = 8'd39;
   {2'd2,2'd1,7'd22},{2'd3,2'd1,7'd22},{2'd3,2'd1,7'd59}: heBFRETone = 8'd41;
   {2'd2,2'd1,7'd23},{2'd3,2'd1,7'd23},{2'd3,2'd1,7'd60}: heBFRETone = 8'd43;
   {2'd2,2'd1,7'd24},{2'd3,2'd1,7'd24},{2'd3,2'd1,7'd61}: heBFRETone = 8'd44;
   {2'd2,2'd1,7'd25},{2'd3,2'd1,7'd25},{2'd3,2'd1,7'd62}: heBFRETone = 8'd46;
   {2'd2,2'd1,7'd26},{2'd3,2'd1,7'd26},{2'd3,2'd1,7'd63}: heBFRETone = 8'd48;
   {2'd2,2'd1,7'd27},{2'd3,2'd1,7'd27},{2'd3,2'd1,7'd64}: heBFRETone = 8'd49;
   {2'd2,2'd1,7'd28},{2'd3,2'd1,7'd28},{2'd3,2'd1,7'd65}: heBFRETone = 8'd51;
   {2'd2,2'd1,7'd29},{2'd3,2'd1,7'd29},{2'd3,2'd1,7'd66}: heBFRETone = 8'd53;
   {2'd2,2'd1,7'd30},{2'd3,2'd1,7'd30},{2'd3,2'd1,7'd67}: heBFRETone = 8'd54;
   {2'd2,2'd1,7'd31},{2'd3,2'd1,7'd31},{2'd3,2'd1,7'd68}: heBFRETone = 8'd56;
   {2'd2,2'd1,7'd32},{2'd3,2'd1,7'd32},{2'd3,2'd1,7'd69}: heBFRETone = 8'd58;
   {2'd2,2'd1,7'd33},{2'd3,2'd1,7'd33},{2'd3,2'd1,7'd70}: heBFRETone = 8'd59;
   {2'd2,2'd1,7'd34},{2'd3,2'd1,7'd34},{2'd3,2'd1,7'd71}: heBFRETone = 8'd61;
   {2'd2,2'd1,7'd35},{2'd3,2'd1,7'd35},{2'd3,2'd1,7'd72}: heBFRETone = 8'd63;
   {2'd2,2'd1,7'd36},{2'd3,2'd1,7'd36},{2'd3,2'd1,7'd73}: heBFRETone = 8'd64;
   //Invalid
   default:                                               heBFRETone = 8'd0;
   endcase
end


// Number of subcarriers of the HE Compressed Beamforming Report (Ns)
assign heBFRNs = heBFRETone-heBFRSTone;


// Number of subcarriers of the BFR Report Mux
assign bfrNs  = bfrHE ? {1'b0,heBFRNs} : vhtBFRNs;
assign bfrNsp = bfrHE ? {1'b0,heBFRNs} : {1'b0,vhtBFRNsp};


// Beamforming Report Length Computation
//
// vht bfrLength = FrameCtrl         + //2
//                 Duration          + //2
//                 RA                + //6
//                 TA                + //6
//                 BSSID             + //6
//                 SEQCTRL           + //2
//                 Category          + //1
//                 VHT Action        + //1
//                 VHT MIMOCtrl      + //3
//                 VHT CompressedBFR + //~
//                 MUExclusiveReport + //~
//                 FCS                 //4
//
//               = 33 + VHT CompressedBFR + MUExclusiveReport(MU Only)
//
// he bfrLength  = FrameCtrl            + //2
//                 Duration             + //2
//                 RA                   + //6
//                 TA                   + //6
//                 BSSID                + //6
//                 SEQCTRL              + //2
//                 Category             + //1
//                 HE Action            + //1
//                 HE MIMOCtrl          + //5
//                 HE CompressedBFR     + //~
//                 HE MUExclusiveReport + //~
//                 FCS                    //4
//
//               = 35 + HE CompressedBFR + HE MUExclusiveReport(MU Only)
//
// VHT compressedBFR     (in bit) =  8 * Nc + Na * vhtNs * (bPhi+bPsi)/2
// VHT MUExclusiveReport (in bit) =  4 * Nc * vhtNs'
//
// HE  compressedBFR     (in bit) =  8 * Nc + Na * heNs * (bPhi+bPsi)/2
// HE  MUExclusiveReport (in bit) =  4 * Nc * heNs
//
// VHT/HE compressedBFR     (in bit) =  8 * Nc + Na * Ns * (bPhi+bPsi)/2
// VHT/HE MUExclusiveReport (in bit) =  4 * Nc * Ns'
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
   begin
      multIn1 <= 9'd0;
      multIn2 <= 8'd0;

      accuIn  <= 17'h0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      multIn1 <= 9'd0;
      multIn2 <= 8'd0;

      accuIn  <= 17'h0;
   end
   else if (!isIdle)
   begin
      case (bfrLengthCompPhase)
      3'd0: 
      begin
         multIn1 <= {4'd0,bfrNa[5:1]};
         multIn2 <= {4'd0,bfrBPhiBPsi[4:1]};
 
         accuIn  <= bfrHE ? {15'd35,2'd0} : {15'd33,2'd0};
      end
      3'd1: 
      begin
         multIn1 <= bfrNs[8:0];
         multIn2 <= multOut[7:0];

         accuIn  <= {11'd0,{1'b0,bfrNc}+4'd1,2'b0};
      end
      3'd2:
      begin
         multIn1 <= bfrNsp[8:0];
         multIn2 <= {3'd0,{1'b0,bfrNc}+4'd1,1'b0};

         accuIn  <= multOut;
      end
      3'd3:
      begin
         multIn1 <= 9'd0;
         multIn2 <= 8'd0;

         if (bfrFeedbackType==MU)
            accuIn  <= multOut;
         else
            accuIn  <= 17'h0;
      end
      default:
      begin
         multIn1 <= 9'd0;
         multIn2 <= 8'd0;

         accuIn  <= 17'h0;
      end
      endcase
   end
end


// Multiplier
assign multOut = {8'b0,multIn1} * {9'b0,multIn2};


// Accumulator
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      accuOut <= 17'd0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      accuOut <= 17'd0;
   else
   begin
      if (isComputeBFRLen)
      begin
         if (bfrLengthCompPhase==3'b0)
            accuOut <= 17'h0;
         else
            accuOut <= accuOut + accuIn;
      end
   end
end


// Beamforming Report Frame Length
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      bfrLength <= 15'd0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      bfrLength <= 15'd0;
   else if (bfrLengthCompDone)
      bfrLength <= (|accuOut[1:0]) ? accuOut[16:2]+15'd1 : accuOut[16:2];
end


// Check SIFS timing
//  - NDPA to NDP
//  - NDP  to BFR-Poll trigger (HE Only)
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
      sifsTimeoutCnt <= 10'b0;
   else if ((macCoreClkSoftRst_n == 1'b0) || isIdle) // Synchronous Reset
      sifsTimeoutCnt <= 10'b0;
   else if (changeToWaitNDP | isComputeBFR & bfrHE & ~bfrResponseAfterNDP)
      sifsTimeoutCnt <= {2'b00,sifs} + {2'b00,rxStartDelayOFDM} + {2'b00,slotTime};
   else if (rxVector1Start_p)
      sifsTimeoutCnt <= 10'b0;
   else if (tick1us_p && sifsTimeoutCnt != 10'b0)
      sifsTimeoutCnt <= sifsTimeoutCnt - 10'b1;
end

assign sifsTimeout_p = (sifsTimeoutCnt == 10'b1) && tick1us_p && !rxVector1Start_p;


// Register rxEndOfFrame pulse
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
   begin
      rxEndOfFrame <= 1'b0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      rxEndOfFrame <= 1'b0;
   end
   else
   begin
      if (rxVector1Valid_p/*changeToCheckNDP*/)
      begin
         rxEndOfFrame <= 1'b0;
      end
      else if (rxEndOfFrame_p)
      begin
         rxEndOfFrame <= 1'b1;
      end
   end
end


// Register macPhyIfRxErr_p pulse
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0) // Asynchronous Reset
   begin
      macPhyIfRxErr <= 1'b0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      macPhyIfRxErr <= 1'b0;
   end
   else
   begin
      if (isIdle)
      begin
         macPhyIfRxErr <= 1'b0;
      end
      else if (macPhyIfRxErr_p)
      begin
         macPhyIfRxErr <= 1'b1;
      end
   end
end

// Debug Port
assign debugPortBFRController = {bfrChBW[1:0],     
                                 bfrGrouping[1:0],    
                                 bfrCodebook,    
                                 bfrFeedbackType[0],
                                 bfrNr[2:0],
                                 bfrNc[2:0],
                                 bfrStart,
                                 bfrControllerCs[2:0]};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// String definition to display bfrController current state
reg  [20*8:0] bfrControllerCs_str;

// bfrControlCs FSM states displayed in a string to easy simulation and debug
always @*
begin
   case (bfrControllerCs)
   IDLE            : bfrControllerCs_str = {"IDLE"};
   CHECK_NDPA      : bfrControllerCs_str = {"CHECK_NDPA"};
   WAIT_NDP        : bfrControllerCs_str = {"WAIT_NDP"};
   COMPUTE_BFR_LEN : bfrControllerCs_str = {"COMPUTE_BFR_LEN"};
   WAIT_NDP_EOF    : bfrControllerCs_str = {"WAIT_NDP_EOF"};
   CHECK_NDP       : bfrControllerCs_str = {"CHECK_NDP"};
   COMPUTE_BFR     : bfrControllerCs_str = {"COMPUTE_BFR"};
   ACTIVE          : bfrControllerCs_str = {"ACTIVE"};
   default         : bfrControllerCs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON

//$rw_sva This property checks if Nr,Nc values are valid
invalidNrNc: assert property (@(posedge macCoreClk) (bfrStart) |-> (bfrNr >= bfrNc));

//$rw_sva This property checks if Na value is valid
invalidNa  : assert property (@(posedge macCoreClk) (bfrStart) |-> (bfrNa !== 0));

//$rw_sva This property checks if Ns value is valid
invalidNs  : assert property (@(posedge macCoreClk) (bfrStart) |-> (bfrNs !== 0));

//$rw_sva This property checks if Ns' value is valid
invalidNsp : assert property (@(posedge macCoreClk) (bfrStart) |-> (bfrNsp !== 0));

//$rw_sva This property checks if ruStartIndex,rxEndIndex value is valid
invalidRU  : assert property (@(posedge macCoreClk) (bfrStart) |-> (bfrRUStartIndex <= bfrRUEndIndex));

//$rw_sva This property checks if HE BFR FeedbackType,Ng,codebook are valid
//HE BFR with MU Feedback, Ng=16 and codebook=0 is invalid
invalidHEBFRParam : assert property (
   @(posedge macCoreClk) (bfrStart & bfrHE) |-> (bfrFeedbackType!=1 | bfrGrouping!=1 | bfrCodebook!=0)
);

//$rw_sva This property checks if HE BFR Length is valid
//HE BFR must be segmented if length is greather than 11454 bytes. Segmented BFR is not supported
invalidLen : assert property (@(posedge macCoreClk) (bfrStart & bfrHE) |-> (bfrLength <= 11454));

`endif // RW_ASSERT_ON

endmodule
