//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: 2017-07-25 14:15:10 +0200 (Tue, 25 Jul 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of coexController module
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`define default_net_type none

module coexController( 
            //$port_g Clock and Reset
            input  wire        macCoreClk,            // mac core clock      
            input  wire        macCoreClkHardRst_n,   // mac core clock reset
            input  wire        macCoreClkSoftRst_n,   // Soft Reset of the MAC Core Clock domain
                                                      // active low

            //$port_g macController Interface
            input  wire        rxAck,                 // Force receive only ACK after Tx
            input  wire        rxBA,                  // Force receive only BA after Tx
            input  wire        rxCts,                 // Force receive only CTS after RTS Tx
            input  wire        rxBfr,                 // Force receive only BFR after NDP or BFRPoll Tx
            input  wire        sendData_p,            // Indicates that data packet has to be sent
            input  wire        sendRTS_p,             // Indicates that an RTS packet has to be sent
            input  wire        sendCTS_p,             // Indicates that a CTS packet has to be sent
            input  wire        sendACK_p,             // Indicates that an ACK packet has to be sent
            input  wire        sendCFEND_p,           // Indicates that a CF-END packet has to be sent
            input  wire        sendBA_p,              // Indicates that an Block ACK packet has to be sent
            input  wire        sendBFR_p,             // Indicates that an BeamForming report packet has to be sent
            input  wire  [1:0] txChBW,                // Transmit a Frame with Channel Bandwidth
            input  wire        updateDMAStatus_p,     // trigs the DMA Status update.
            input  wire        activeRx,              // Enables the RX frame exchange
            input  wire        stopRx_p,              // Stop Rx
            input  wire        startRx,               // Start Rx
            input  wire        skipMPDU_p,            // Indication that MPDU in TX FIFO has to be discarded

            //$port_g txParameterCache Interface
            input  wire  [1:0] fcType,                // Indicates the fcType field of the Frame Control field of the MPDU
            input  wire        tsValid,               // Indicates that fcType is valid
            
            //$port_g txController Interface
            input  wire        startTx_p,             // Start Tx trigger                 
            input  wire        txDone_p,              // Indicates that the transmission is completed
 
            //$port_g rxController Interface
            input  wire  [3:0] rxTID,                 // TID of the received frame
            input  wire        notMineRcved,          // Indicates than the received frame is not destined to this device.
            input  wire        macHeaderCompleted,    // End of MAC Header reception
            input  wire  [1:0] rxFCType,              // Indicates the fcType field of the received frame
    
            //$port_g deaggregator Interface
            input  wire        rxVector1Start_p,      // Indicate that the first byte of RX Vector has been received
            input  wire        rxVector1Valid_p,      // Rx vector 1 is available
            input  wire        rxEndOfFrame_p,        // Rx vector 2 is available
            input  wire  [1:0] rxChBW,                // Channel bandwidth
    
    
            //$port_g macPhy Interface
            input  wire        macPhyIfRxEnd_p,       // mpIF state machine is idle

            //$port_g Registers Interface
            input  wire        coexEnable,            // Enable Coex Feature
            input  wire        coexAutoPTIAdjEnable,  // Enable the automatic PTI adjustment featureCoex Feature
            input  wire [1:0]  coexAutoPTIAdjIncr,    // Automatic PTI Adjustment Increment
            input  wire [3:0]  coexPTIBKData,         // PTI default value for Data frame transmitted on BK AC
            input  wire [3:0]  coexPTIBEData,         // PTI default value for Data frame transmitted on BE AC
            input  wire [3:0]  coexPTIVIData,         // PTI default value for Data frame transmitted on VI AC
            input  wire [3:0]  coexPTIVOData,         // PTI default value for Data frame transmitted on VO AC
            input  wire [3:0]  coexPTIBcnData,        // PTI default value for Data frame transmitted on Beacon AC
            input  wire [3:0]  coexPTIMgt,            // PTI default value for Management frame
            input  wire [3:0]  coexPTICntl,           // PTI default value for Control frames other than ACK and BA
            input  wire [3:0]  coexPTIAck,            // PTI default value for ACK and BA
            input  wire [3:0]  coexForceWlanPTI,      // Forced value of coexWlanPTI
            input  wire        coexForceWlanPTIToggle,// Forced value of coexWlanPTIToggle
            input  wire        coexForceWlanChanBw,   // Forced value of coexWlanPTIToggle
            input  wire        coexForceWlanRx,       // Forced value of coexWlanPTIToggle
            input  wire        coexForceWlanTx,       // Forced value of coexWlanPTIToggle
            input  wire        coexForceEnable,       // Use coex Forced values

            //$port_g Coex Interface
            input  wire        coexWlanTxAbort,       // Requests from external arbiter abort all on-going transmission 
            input  wire        coexWlanRxAbort,       // Requests from external arbiter abort all on-going reception 
            output reg  [3:0]  coexWlanPTI,           // Indicates the priority of the on-going traffic
            output reg         coexWlanPTIToggle,     // Indicates a new priority value
            output reg         coexWlanChanBw,        // Indicates the BW of the on-going traffic
            output reg         coexWlanTx,            // Indicates that a WLAN Transmission is on-going
            output reg         coexWlanRx,            // Indicates that a WLAN Reception is on-going

            //$port_g Backoff Interface
            input  wire        trigTxAC0,             // Trigger from the MAC Controller to indicate DMA to 
                                                      // start fetching frames associated with AC0
            input  wire        trigTxAC1,             // Trigger from the MAC Controller to indicate DMA to
                                                      // start fetching frames associated with AC1
            input  wire        trigTxAC2,             // Trigger from the MAC Controller to indicate DMA to 
                                                      // start fetching frames associated with AC2
            input  wire        trigTxAC3,             // Trigger from the MAC Controller to indicate DMA to
                                                      // start fetching frames associated with AC3
            input  wire        trigTxBcn,             // Trigger from the MAC Controller to indicate DMA to
                                                      // start fetching frames associated with Bcn
            //$port_g Interrupt Generation
            input  wire        coexWlanRxAbortFallEn, // Coex Wlan Rx Abort Falling Edge Enable
            input  wire        coexWlanRxAbortRiseEn, // Coex Wlan Rx Abort Rising Edge Enable
            input  wire        coexWlanTxAbortFallEn, // Coex Wlan Tx Abort Falling Edge Enable
            input  wire        coexWlanTxAbortRiseEn, // Coex Wlan Tx Abort Rising Edge Enable
            output reg         coexEvent,             // Pulse coex event for interrupt generation

            //$port_g Debug
            output wire [15:0] debugPortCoexController// Debug Port coexController
            );

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
localparam 
  PTI_NONE       = 4'd0,
  PTI_MGT_TX     = 4'd1,
  PTI_CNTL_TX    = 4'd2,
  PTI_ACK_TX     = 4'd3,
  PTI_BKDATA_TX  = 4'd4,
  PTI_BEDATA_TX  = 4'd5,
  PTI_VIDATA_TX  = 4'd6,
  PTI_VODATA_TX  = 4'd7,
  PTI_BCNDATA_TX = 4'd8,
  PTI_MGT_RX     = 4'd9,
  PTI_CNTL_RX    = 4'd10,
  PTI_ACK_RX     = 4'd11,
  PTI_BKDATA_RX  = 4'd12,
  PTI_BEDATA_RX  = 4'd13,
  PTI_VIDATA_RX  = 4'd14,
  PTI_VODATA_RX  = 4'd15;


`ifdef RW_SIMU_ON
// String definition to display PTI Mode current state
reg [14*8-1:0] coexControllerPTIMode_str;
wire [3:0] coexPTICurValue;
wire [3:0] coexPTIInitialValue;
`endif // RW_SIMU_ON

reg macHeaderCompletedDly;
reg startRxDly_p;
reg startRxDly;
reg startRx_p;
reg coexTxAbortDetected;
reg coexRxAbortDetected;

wire [3:0] coexPTIInitial [0:15];
reg  [3:0] coexPTICur [0:15];
reg  [3:0] ptiMode;

reg coexWlanTxAbortDly;
reg coexWlanRxAbortDly;
reg coexEnableDly;


integer i;
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Indicates that a WLAN Transmission is on-going
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    coexWlanTx <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
    coexWlanTx <= 1'b0;
  else if (coexForceEnable)
    coexWlanTx <= coexForceWlanTx;
  else
    if (startTx_p)
      coexWlanTx <= 1'b1;
    else if (txDone_p)
      coexWlanTx <= 1'b0;
end

always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    coexWlanRx <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
    coexWlanRx <= 1'b0;
  else if (coexForceEnable)
    coexWlanRx <= coexForceWlanRx;
  else
    if (startTx_p)
      coexWlanRx <= 1'b0;
    else if ((!activeRx && startRxDly_p) || rxVector1Start_p)
      coexWlanRx <= 1'b1;
    else if (activeRx && rxEndOfFrame_p && coexRxAbortDetected)
      coexWlanRx <= 1'b1;
    else if ((!activeRx && updateDMAStatus_p) || rxEndOfFrame_p || stopRx_p)
      coexWlanRx <= 1'b0;
end


always @ (posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    coexWlanChanBw <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
    coexWlanChanBw <= 1'b0;
  else if (coexForceEnable)
    coexWlanChanBw <= coexForceWlanChanBw;
  else
  begin
    if (rxVector1Valid_p)
      coexWlanChanBw <= rxChBW[0];
    else if (!activeRx && startRx_p)
      coexWlanChanBw <= txChBW[0];
    else if ((startTx_p || sendRTS_p || sendCTS_p || sendACK_p || sendCFEND_p || sendBA_p || sendBFR_p) && !skipMPDU_p)
      coexWlanChanBw <= txChBW[0];
    else if ((!activeRx && updateDMAStatus_p) || txDone_p || rxEndOfFrame_p || stopRx_p)
      coexWlanChanBw <= 1'b0;
  end
end


// Generation of ptiMode. This signal indicates the current mode and associated priority.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    ptiMode          <= PTI_NONE;
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
    ptiMode          <= PTI_NONE;
  else if (updateDMAStatus_p)
    ptiMode          <= PTI_NONE;
  else if (activeRx && txDone_p)
    ptiMode          <= PTI_NONE;
  else if (activeRx && rxEndOfFrame_p && !coexRxAbortDetected)
    ptiMode          <= PTI_NONE;
  else
  begin
    if (sendACK_p || sendBA_p || sendBFR_p)
      ptiMode        <= PTI_ACK_TX;
    else if (sendRTS_p || sendCTS_p)
      ptiMode        <= PTI_CNTL_TX;
    else if (sendData_p || (startRx_p && (rxAck || rxBA || rxCts || rxBfr))) 
    begin
      if ((fcType == 2'b01) && tsValid)
        ptiMode        <= PTI_CNTL_TX;
      else if ((fcType == 2'b00) && tsValid)
        ptiMode        <= PTI_MGT_TX;
      else if (trigTxAC0)
        ptiMode        <= PTI_BKDATA_TX;
      else if (trigTxAC1)
        ptiMode        <= PTI_BEDATA_TX;
      else if (trigTxAC2)
        ptiMode        <= PTI_VIDATA_TX;
      else if (trigTxAC3)
        ptiMode        <= PTI_VODATA_TX;
      else if (trigTxBcn)
        ptiMode        <= PTI_BCNDATA_TX;
    end  
    else if (activeRx && macHeaderCompleted && !macHeaderCompletedDly && !notMineRcved)
    begin
      if (rxFCType == 2'b01)
        ptiMode        <= PTI_CNTL_RX;
      else if (rxFCType == 2'b00)
        ptiMode        <= PTI_MGT_RX;
      else if (rxFCType == 2'b10)
      begin
        case (rxTID[2:0])
          3'd1,3'd2 : ptiMode        <= PTI_BKDATA_RX;
          3'd0,3'd3 : ptiMode        <= PTI_BEDATA_RX;
          3'd4,3'd5 : ptiMode        <= PTI_VIDATA_RX;
          default   : ptiMode        <= PTI_VODATA_RX;
        endcase
      end
    end  
//     else if (macPhyIfRxEnd_p)
//       ptiMode          <= PTI_NONE;
  end
end

// This process registers the coexWlanPTI output and generates the
// coexWlanPTIToggle signal
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    coexWlanPTI       <= 4'b0;
    coexWlanPTIToggle <= 1'b0;
  end  
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
  begin
    coexWlanPTI       <= 4'b0;
    coexWlanPTIToggle <= 1'b0;
  end
  else if (coexForceEnable)
  begin
    coexWlanPTI       <= coexForceWlanPTI;
    coexWlanPTIToggle <= coexForceWlanPTIToggle;
  end
  else if (coexPTICur[ptiMode] != coexWlanPTI)
  begin
    coexWlanPTI        <= coexPTICur[ptiMode];
    coexWlanPTIToggle  <= ~coexWlanPTIToggle;
  end
end


// Generation of rxStart_p pulse based on a rising edge detection 
// of rxStart flag
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    startRx_p          <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
    startRx_p          <= 1'b0;
  else if (startRx && !startRxDly)
    startRx_p          <= 1'b1;
  else
    startRx_p          <= 1'b0;
end


// Generation of macHeaderCompletedDly, startRxDly, startRxDly_p and coexEnableDly based respectively 
// on macHeaderCompleted, startRx, startRx_p and coexEnable delayed of one clock cycle.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    macHeaderCompletedDly   <= 1'b0;
    startRxDly_p            <= 1'b0;
    startRxDly              <= 1'b0;
    coexEnableDly           <= 1'b0;
  end
  else
  begin
    macHeaderCompletedDly   <= macHeaderCompleted;
    startRxDly_p            <= startRx_p;
    startRxDly              <= startRx;
    coexEnableDly           <= coexEnable;
  end
end

// PTI Auto Adjustment
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    for (i=0;i<=15;i=i+1)
    begin
      coexPTICur[i] <= 4'b0;
    end  
  end  
  else if (macCoreClkSoftRst_n == 1'b0 || (coexEnable && !coexEnableDly))  // Synchronous Reset
  begin
    // Initialise the current PTI information of each traffic class with their Initial values coming from coexPTIInitial array
    for (i=0;i<=15;i=i+1)
    begin
      coexPTICur[i] <= coexPTIInitial[i];
    end  
  end  
  else if (coexEnable && coexAutoPTIAdjEnable)
  begin
    if ((updateDMAStatus_p && (coexTxAbortDetected || coexRxAbortDetected)) ||
        (activeRx && ((txDone_p && coexTxAbortDetected) || (macPhyIfRxEnd_p && coexRxAbortDetected))))
    begin
      if (ptiMode == PTI_NONE)
        coexPTICur[ptiMode] <= coexPTICur[ptiMode];
      else if ((coexPTICur[ptiMode] + {2'b0,coexAutoPTIAdjIncr} < 4'b1111))
        coexPTICur[ptiMode] <= coexPTICur[ptiMode] + {2'b0,coexAutoPTIAdjIncr};
      else
        coexPTICur[ptiMode] <= 4'b1111;
    end
    else if ((updateDMAStatus_p && !coexRxAbortDetected && !coexTxAbortDetected) ||
             (activeRx && txDone_p)                                              || 
             (activeRx && rxEndOfFrame_p && !coexRxAbortDetected))  // In case of Reception
      coexPTICur[ptiMode] <= coexPTIInitial[ptiMode];
  end
end


always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    coexTxAbortDetected <= 1'b0;
    coexRxAbortDetected <= 1'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0 || !coexEnable)  // Synchronous Reset
  begin
    coexTxAbortDetected <= 1'b0;
    coexRxAbortDetected <= 1'b0;
  end
  else
  begin
    if (coexWlanTx && coexWlanTxAbort)
      coexTxAbortDetected <= 1'b1;
    else if ((!activeRx && updateDMAStatus_p) || (activeRx && !coexWlanTx))
      coexTxAbortDetected <= 1'b0;
    
    if (coexWlanRx && coexWlanRxAbort && (!activeRx || !notMineRcved))
      coexRxAbortDetected <= 1'b1;
    else if ((!activeRx && updateDMAStatus_p) ||
             (activeRx && (rxEndOfFrame_p || !coexWlanRx)) ||
             stopRx_p)
      coexRxAbortDetected <= 1'b0;
  end
end


assign coexPTIInitial[PTI_NONE]       = 4'b0;
assign coexPTIInitial[PTI_MGT_TX]     = coexPTIMgt;
assign coexPTIInitial[PTI_CNTL_TX]    = coexPTICntl;
assign coexPTIInitial[PTI_ACK_TX]     = coexPTIAck;
assign coexPTIInitial[PTI_BKDATA_TX]  = coexPTIBKData;
assign coexPTIInitial[PTI_BEDATA_TX]  = coexPTIBEData;
assign coexPTIInitial[PTI_VIDATA_TX]  = coexPTIVIData;
assign coexPTIInitial[PTI_VODATA_TX]  = coexPTIVOData;
assign coexPTIInitial[PTI_BCNDATA_TX] = coexPTIBcnData;
assign coexPTIInitial[PTI_MGT_RX]     = coexPTIMgt;
assign coexPTIInitial[PTI_CNTL_RX]    = coexPTICntl;
assign coexPTIInitial[PTI_ACK_RX]     = coexPTIAck;
assign coexPTIInitial[PTI_BKDATA_RX]  = coexPTIBKData;
assign coexPTIInitial[PTI_BEDATA_RX]  = coexPTIBEData;
assign coexPTIInitial[PTI_VIDATA_RX]  = coexPTIVIData;
assign coexPTIInitial[PTI_VODATA_RX]  = coexPTIVOData;


////////////////////////////////////////////////////////////////////////
//
// Coex Event Interrupt generation 
//
///////////////////////////////////////////////////////////////////////
// Generation of coexWlanTxAbort and coexWlanTxAbort delayed of 1cc for edge detection
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    coexWlanTxAbortDly <= 1'b0;
    coexWlanRxAbortDly <= 1'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0 || !coexEnable)  // Synchronous Reset
  begin
    coexWlanTxAbortDly <= 1'b0;
    coexWlanRxAbortDly <= 1'b0;
  end
  else
  begin
    coexWlanTxAbortDly <= coexWlanTxAbort;
    coexWlanRxAbortDly <= coexWlanRxAbort;
  end
end


// Generation of coexEvent Pulse based on  coexWlanTxAbort and coexWlanTxAbort edge detection masked with
// enable from registers (coexWlanTxAbortRiseEn, coexWlanTxAbortFallEn, coexWlanRxAbortRiseEn, coexWlanRxAbortFallEn) 
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    coexEvent <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0 || !coexEnable)  // Synchronous Reset
    coexEvent <= 1'b0;
  else
  begin
    // Detection of coexWlanTxAbort rising edge
    if (coexWlanTxAbortRiseEn && coexWlanTxAbort && !coexWlanTxAbortDly)
      coexEvent <= 1'b1;
    // Detection of coexWlanTxAbort falling edge
    else if (coexWlanTxAbortFallEn && !coexWlanTxAbort && coexWlanTxAbortDly)
      coexEvent <= 1'b1;
    // Detection of coexWlanRxAbort rising edge
    else if (coexWlanRxAbortRiseEn && coexWlanRxAbort && !coexWlanRxAbortDly)
      coexEvent <= 1'b1;
    // Detection of coexWlanRxAbort falling edge
    else if (coexWlanRxAbortFallEn && !coexWlanRxAbort && coexWlanRxAbortDly)
      coexEvent <= 1'b1;
    else
      coexEvent <= 1'b0;
  end
end


// Debug Port assigment
assign debugPortCoexController = {
macHeaderCompleted,    // 15
ptiMode,               // 14:11
activeRx,              // 10
coexWlanTxAbort,       // 9
coexWlanRxAbort,       // 8
coexWlanPTI,           // 7:4
coexWlanPTIToggle,     // 3
coexWlanChanBw,        // 2
coexWlanTx,            // 1
coexWlanRx};           // 0
                       
                        
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_ASSERT_ON

// $rw_sva Checks if a coexEvent is generated (when enabled) in case of falling edge of coexWlanRxAbort
property coexWlanRxAbortFallEvent_prop;
@(posedge macCoreClk)
  ($fell(coexWlanRxAbort) && coexWlanRxAbortFallEn) |=> (coexEvent);
endproperty
coexWlanRxAbortFallEvent: assert property (disable iff (!macCoreClkHardRst_n) coexWlanRxAbortFallEvent_prop);

// $rw_sva Checks if a coexEvent is generated (when enabled) in case of rising edge of coexWlanRxAbort
property coexWlanRxAbortRiseEvent_prop;
@(posedge macCoreClk)
  ($rose(coexWlanRxAbort) && coexWlanRxAbortRiseEn) |=> (coexEvent);
endproperty
coexWlanRxAbortRiseEvent: assert property (disable iff (!macCoreClkHardRst_n) coexWlanRxAbortRiseEvent_prop);

// $rw_sva Checks if a coexEvent is generated (when enabled) in case of falling edge of coexWlanTxAbort
property coexWlanTxAbortFallEvent_prop;
@(posedge macCoreClk)
  ($fell(coexWlanTxAbort) && coexWlanTxAbortFallEn) |=> (coexEvent);
endproperty
coexWlanTxAbortFallEvent: assert property (disable iff (!macCoreClkHardRst_n) coexWlanTxAbortFallEvent_prop);

// $rw_sva Checks if a coexEvent is generated (when enabled) in case of rising edge of coexWlanTxAbort
property coexWlanTxAbortRiseEvent_prop;
@(posedge macCoreClk)
  ($rose(coexWlanTxAbort) && coexWlanTxAbortRiseEn) |=> (coexEvent);
endproperty
coexWlanTxAbortRiseEvent: assert property (disable iff (!macCoreClkHardRst_n) coexWlanTxAbortRiseEvent_prop);

`endif // RW_ASSERT_ON

`ifdef RW_SIMU_ON

// macControllerTxFSM FSM states displayed in a string to easy simulation and debug
always @ (ptiMode)
begin
  case (ptiMode)
    PTI_NONE       :  coexControllerPTIMode_str = {"PTI_NONE      "};
    PTI_MGT_TX     :  coexControllerPTIMode_str = {"PTI_MGT_TX    "};
    PTI_CNTL_TX    :  coexControllerPTIMode_str = {"PTI_CNTL_TX   "};
    PTI_ACK_TX     :  coexControllerPTIMode_str = {"PTI_ACK_TX    "};
    PTI_BKDATA_TX  :  coexControllerPTIMode_str = {"PTI_BKDATA_TX "};
    PTI_BEDATA_TX  :  coexControllerPTIMode_str = {"PTI_BEDATA_TX "};
    PTI_VIDATA_TX  :  coexControllerPTIMode_str = {"PTI_VIDATA_TX "};
    PTI_VODATA_TX  :  coexControllerPTIMode_str = {"PTI_VODATA_TX "};
    PTI_BCNDATA_TX :  coexControllerPTIMode_str = {"PTI_BCNDATA_TX"};
    PTI_MGT_RX     :  coexControllerPTIMode_str = {"PTI_MGT_RX    "};
    PTI_CNTL_RX    :  coexControllerPTIMode_str = {"PTI_CNTL_RX   "};
    PTI_ACK_RX     :  coexControllerPTIMode_str = {"PTI_ACK_RX    "};
    PTI_BKDATA_RX  :  coexControllerPTIMode_str = {"PTI_BKDATA_RX "};
    PTI_BEDATA_RX  :  coexControllerPTIMode_str = {"PTI_BEDATA_RX "};
    PTI_VIDATA_RX  :  coexControllerPTIMode_str = {"PTI_VIDATA_RX "};
    PTI_VODATA_RX  :  coexControllerPTIMode_str = {"PTI_VODATA_RX "};
          default  :  coexControllerPTIMode_str = {"XXX           "};
  endcase
end

assign coexPTICurValue     = coexPTICur[ptiMode];
assign coexPTIInitialValue = coexPTIInitial[ptiMode];

`endif // RW_SIMU_ON

endmodule
