//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of deaggregator module
//
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                   the waveform viewer.
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
//
// Application Note : 
//
// Simulator        :
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module deaggregator( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire          macCoreRxClk,            // MAC Core Receive Clock
            input wire          macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain 
                                                         // active low
            input wire          macCoreClkSoftRst_n,     // Soft Reset of the MAC Core Clock domain
                                                         // active low
            ///////////////////////////////////////////////
            //$port_g MAC Controller
            ///////////////////////////////////////////////
            input wire          startRx,                 // Start Rx
            input wire          stopRx_p,                // Stop Rx
            
            output wire         rxVector1Start_p,        // Indicate that the first byte of RX Vector has been received
            output reg          rxVector1Valid_p,        // Rx vector 1 is available
            output reg          rxEndOfFrame_p,          // End of frame information
            output wire         deaggregatorCsIsIdle,    // Indicate that the deaggregator is waiting for a reception

            output reg          ampduCorrectRcved_p,     // A-MPDU received correctly
            output reg          ampduIncorrectRcved_p,   // A-MPDU not received correctly
            output reg          rxSMPDU,                 // Detect EOF bit in delimiter in Single MPDU

            output reg  [5:0]   rxMPDUCnt,               // Count the number of MPDU detected
            output reg  [7:0]   incorrectDelCnt,         // Count the number of incorrect delimiter

            ///////////////////////////////////////////////
            //$port_g RX Controller
            ///////////////////////////////////////////////
            input wire          correctRcved_p,          // Frame received correctly
                         
            input wire          rxCntrlReady,            // Rx controller ready to receive data
                         
            output reg   [11:0] rxLegLength,             // Legacy length
            output reg   [ 3:0] rxLegRate,               // Legacy rate
            output reg   [19:0] rxHTLength,              // HT length
            output reg   [ 6:0] rxMCS,                   // MCS
            output reg          rxPreType,               // Preamble type
            output reg   [ 3:0] rxFormatMod,             // Format and modulation
            output reg   [ 2:0] rxChBW,                  // Channel bandwidth
            output reg   [ 1:0] rxGIType,                // Guard Interval Type
            output reg          rxSmoothing,             // Smoothing
            output reg          rxLSIGValid,             // L-SIG Valid of received frame
            output reg          rxSTBC,                  // Space Time Block Coding
            output reg          rxSounding,              // Sounding
            output reg   [ 1:0] rxNumExtnSS,             // Number of Extension Spatial Streams
            output reg          rxDynBWInNonHT,          // Dynamic Bandwidth in Non-HT
            output reg   [ 1:0] rxChBWInNonHT,           // Channel bandwidth in Non-HT extracted
                                                         // from the scrambler initialization
                                                         // sequence
            output reg          rxAggregation,           // MPDU aggregate
            output reg          rxFecCoding,             // FEC Coding
            output reg   [ 7:0] rxAntennaSet,            // Antenna set
            output reg   [ 2:0] rxNss,                   // Number of Spatial stream
//          output reg          rxDynBW,                 // Dynamique Bandwidth
            output reg          rxDozeNotAllowed,        // TXOP PS not allowed
            output reg   [ 8:0] rxPartialAID,            // Partial AID
            output reg   [ 5:0] rxGroupID,               // Group ID
            output reg   [ 7:0] rxRSSILegacy,            // RSSI during Legacy preamble
            output reg   [ 7:0] rxRSSI,                  // RSSI during Data
            output wire  [ 7:0] rxRCPI,                  // Receive Channel Power Indicator
            output wire  [ 7:0] rxEVM1,                  // Error Vector Magnitude for space time stream 1
            output wire  [ 7:0] rxEVM2,                  // Error Vector Magnitude for space time stream 2
            output wire  [ 7:0] rxEVM3,                  // Error Vector Magnitude for space time stream 3
            output wire  [ 7:0] rxEVM4,                  // Error Vector Magnitude for space time stream 4
            output wire         rxVector2Valid_p,        // Rx vector 2 is available
                         
            output wire  [ 7:0] rxData,                  // Rx data extracted from MAC-PHY interface FIFO
            output wire         rxDataValid,             // Rx data is valid
            output reg          rxDataStart_p,           // Start of data flow
            output wire         rxDataEnd_p,             // End of data flow
            output wire         rxDataError_p,           // Rx Data error (length null)
                         
            output reg   [15:0] rxByteCnt,               // Rx byte counter
                         
            output reg   [15:0] rxMpduLength,            // Full Rx MPDU Length
            output reg          rxNDP,                   // Indicate that the received frame is a Null Data Packet
            output wire         correctDelimiter,        // Indicate the correct Delimiter is found
            output reg   [1:0]  rxAMPDUCnt,              // Counter incremented at each A-MPDU reception
            output reg          rxBeamformed,            // Indicate that the Received frame is beamformed
            output reg          rxFirstUser,             // Indicates if the user position inside
                                                         // the received mumimo frame is the first
                                                         // one or not 
            output reg          rxUplinkFlag,            // Indicates if the HE PPDU is addressed to
                                                         // an AP
            output reg          rxBeamChange,            // Beam Change
            output reg          rxDcm,                   // Dual Carrier Modulation
            output reg    [1:0] rxHeLtfType,             // Indicates the type of HE-LTF
            output reg          rxDoppler,               // Indicates whether the doppler effect
                                                         // should be considered for the PPDU or not
            output reg          rxMidamble,              // Indicates the midamble periodicity
            output reg    [5:0] rxBssColor,              // Indicate the BSS color of the AP
            output reg    [6:0] rxTxopDuration,          // Indicates a duration that is used to update the NAV for this TXOP
            output reg    [3:0] rxPeDuration,            // Packet Extension Duration
            output reg    [3:0] rxSpatialReuse1,         // Indicates the spatial reuse parameter value
            output reg    [3:0] rxSpatialReuse2,         // Indicates the spatial reuse parameter value
            output reg    [3:0] rxSpatialReuse3,         // Indicates the spatial reuse parameter value
            output reg    [3:0] rxSpatialReuse4,         // Indicates the spatial reuse parameter value
            output reg          rxSigBCompressionMode,   // SIG B Compression Mode
            output reg          rxDcmSigB,               // Dual Carrier Modulation on SIG B
            output reg    [2:0] rxMcsSigB,               // MCS of SIG B
            output reg    [2:0] rxRUSize,                // Indicates the size of the RU allocated
                                                         // to the user
            output reg  [127:0] rxVector1,               // received Vector 1
            output reg   [63:0] rxVector2,               // received Vector 2


            ///////////////////////////////////////////////
            //$port_g MAC-PHY interface
            ///////////////////////////////////////////////
            input wire          macPhyIfRxFifoEmpty,     // MAC-PHY interface FIFO is empty
            input wire   [ 7:0] macPhyIfRxFifoData,      // MAC-PHY interface FIFO data
                         
            input wire          macPhyIfRxErr_p,         // Rx error from MAC-PHY interface resynchronized
            input wire          macPhyIfRxEnd_p,         // Rx end from MAC-PHY interface resynchronized
                         
            output reg          macPhyIfRxFifoReadEn,    // Read command to MAC-PHY interface FIFO
            output reg          rxDSSS,                  // Indicate the current reception is DSSS/CCK

            ///////////////////////////////////////////////
            //$port_g Register interface
            ///////////////////////////////////////////////
            input wire   [19:0] maxAllowedLength,        // Filter frames whose length is greater than this length

            ///////////////////////////////////////////////
            //$port_g Debug ports
            ///////////////////////////////////////////////
            output wire  [15:0] debugPortDeaggregator,   // Debug port for validation
            output wire  [15:0] debugPortDeaggregator2,  // Debug port for validation2
            output wire  [ 3:0] debugPortDeaggregatorFsm // Deaggregator FSM for validation
            );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
// deaggregator FSM states definition
//$fsm_sd deaggregator
localparam 
                 IDLE =  4'd0,
    COMP_FIFO_LATENCY =  4'd1,
      READ_RX_VECTOR1 =  4'd2,
        VECTOR1_VALID =  4'd11,
         TRIG_RX_CTRL =  4'd3,
       RX_AMPDU_DELIM =  4'd4,
   WAIT_RX_CTRL_READY =  4'd5,
         READ_MP_FIFO =  4'd6,
      READ_RX_VECTOR2 =  4'd7,
        VECTOR2_VALID =  4'd8,
         WAIT_PHY_END =  4'd9,
            RX_ERROR  =  4'd10;

// Format and Modulation
localparam
   NON_HT       = 4'b0000,
   NON_HT_DUP   = 4'b0001,
   HT_MM        = 4'b0010,
   HT_GF        = 4'b0011,
   VHT          = 4'b0100,
   HE_SU        = 4'b0101,
   HE_MU        = 4'b0110,
   HE_EXT_SU    = 4'b0111,
   HE_TRIG      = 4'b1000;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  // deaggregator FSM signals definition
  reg [3:0] deaggregatorCs;  // deaggregator FSM Current State
  reg [3:0] deaggregatorNs;  // deaggregator FSM Next State

`ifdef RW_SIMU_ON
  // String definition to dislpay deaggregator current state
  reg [19*8-1:0] deaggregatorCs_str;
`endif // RW_SIMU_ON

  // Byte counter signals
  reg         byteCntEnd;
  
  // Read enable delayed signals
  reg         rxFifoReadEnDel1;
  
  // macPhyIfRxEnd_p sampled
  reg         macPhyIfRxEndSamp;
  
  // One clk cycle data valid after WAIT_RX_CTRL_READY
  reg         rxDataValidAfterReady;
  
  // Length signals
  wire [19:0] muxRxLength;
  //reg  [19:0] htRxLength;
  reg  [15:0] mpduLength;
  wire [15:0] rxLength;  // Rx length of individual frame

  // RX Vectors
  reg   [7:0] rxVector1Mem[15:0];
  reg   [4:0] rxVector1LastIndex; // Last Rx Vector1 Index (length-1)
  //reg  [63:0] rxVector2;
  wire  [3:0] rxVector2LastIndex; // Last Rx Vector2 Index (length-1)
  
  // A-MPDU signals
  wire        signatureFound;
  reg [19:0]  ampduByteCnt;
  reg         ampduByteCntEnd;
//  wire        correctDelimiter;

  // CRC signals
  wire        crcOk;
  reg  [1:0]  crcByteCnt;
  reg  [7:0]  crcCalc;
  wire [7:0]  crcCalcInv;
  
  // Data hold signals for CRC calculation
  reg [7:0]   rxDataHold1;
  reg [7:0]   rxDataHold2;

  // MAC Controller flag signals
  reg         frameRcvedFlag;
 
  // Frame abort due to max. length has occured
  wire        maxLengthExceeded_p;

  // detect and remember the first correct delimiter
  reg         rxFirstDelimiterDone;

  // detect EOF bit in the AMPDU delimiter
  reg         rxEOF;
  
  reg         incDelFound; // The flag incDelFound is used to detect the first incorrect delimiter
  reg         inDelimiter; // Flag indicating when the deaggregatopr is receiving a AMPDU delimiter
  
  reg  [6:0]  debugMCS;
//wire [2:0]  debugnSS; // Number of nSS

    
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////
// deaggregator FSM Current State Logic 
///////////////////////////////////////
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    deaggregatorCs <= IDLE; 
  else if(macCoreClkSoftRst_n == 1'b0) // Soft reset
    deaggregatorCs <= IDLE; 
  else
    deaggregatorCs <= deaggregatorNs; 
end

assign deaggregatorCsIsIdle = (deaggregatorCs == IDLE) ? 1'b1 : 1'b0;

///////////////////////////////////////
// deaggregator FSM Next State Logic.
///////////////////////////////////////
always @* 
begin
  if (stopRx_p)
    //$fsm_t When stopRx_p or macPhyIfRxErr_p and macPhyIfRxEnd_p are true,
    // fsm goes to IDLE state
    deaggregatorNs = IDLE;
    
  else if (macPhyIfRxErr_p)
    //$fsm_t When macPhyIfRxErr_p is true and not macPhyIfRxEnd_p , fsm goes to WAIT_PHY_END state
    deaggregatorNs = WAIT_PHY_END; 
  
  else if ((deaggregatorCs != IDLE         ) &&
           (deaggregatorCs != VECTOR2_VALID) &&
           (deaggregatorCs != WAIT_PHY_END ) &&
           (deaggregatorCs != RX_ERROR     ) &&
           macPhyIfRxFifoEmpty && macPhyIfRxEndSamp && !rxNDP &&
           !((deaggregatorCs == READ_RX_VECTOR2) && byteCntEnd))
    //$fsm_t When macPhyIfRxFifoEmpty is true and macPhyIfRxEndSamp and not rxNDP, fsm goes to RX_ERROR state
    deaggregatorNs = RX_ERROR; 
  
  else
  begin
  case(deaggregatorCs)

    IDLE:
      //$fsm_s In IDLE state, fsm waits MAC Controller to request RX start.
      if (startRx && !macPhyIfRxFifoEmpty)
        //$fsm_t When startRx is true, fsm goes to COMP_FIFO_LATENCY state
        deaggregatorNs = COMP_FIFO_LATENCY; 
      else
        //$fsm_t When startRx is false, fsm stays to IDLE state
        deaggregatorNs = IDLE;

    COMP_FIFO_LATENCY:
      //$fsm_s In COMP_FIFO_LATENCY state, fsm waits MAC PHY IF FIFO to be not empty.
      if (!macPhyIfRxFifoEmpty)
        //$fsm_t When macPhyIfRxFifoEmpty is false, fsm goes to READ_RX_VECTOR1 state
        deaggregatorNs = READ_RX_VECTOR1; 
      else
        //$fsm_t When macPhyIfRxFifoEmpty is true, fsm stays to COMP_FIFO_LATENCY state
        deaggregatorNs = COMP_FIFO_LATENCY;

    READ_RX_VECTOR1:
      //$fsm_s In READ_RX_VECTOR1 state, fsm waits byteCntEnd active.
      if (byteCntEnd && rxFifoReadEnDel1)
        //$fsm_t When byteCntEnd is asserted, fsm goes to VECTOR1_VALID state
        deaggregatorNs = VECTOR1_VALID;
      else
        //$fsm_t While byteCntEnd is not asserted, fsm stays to READ_RX_VECTOR1 state
        deaggregatorNs = READ_RX_VECTOR1;

    VECTOR1_VALID:
      //$fsm_s In VECTOR1_VALID state, rx Vector1 is valid
      if (rxFormatMod>=4'd2 & rxHTLength==20'd0/*rxNDP*/)
        //$fsm_t When Null Data Packet, fsm goes to TRIG_RX_CTRL state
        deaggregatorNs = TRIG_RX_CTRL;
      else if (!rxAggregation && (muxRxLength > maxAllowedLength))
        //$fsm_t When length is null, fsm goes to RX_ERROR state
        deaggregatorNs = RX_ERROR;
      else if (rxAggregation)
        //$fsm_t When frame is a A-MPDU, fsm goes to RX_AMPDU_DELIM state
        deaggregatorNs = RX_AMPDU_DELIM; 
      else
        //$fsm_t When length is not null, fsm goes to TRIG_RX_CTRL state
        deaggregatorNs = TRIG_RX_CTRL; 

    TRIG_RX_CTRL:
      //$fsm_s In TRIG_RX_CTRL state, fsm triggers RX Controller.
      if (rxNDP |
          rxFormatMod==4'd8 & rxHTLength==20'd0)
        //$fsm_t When Null Data Packet or HE-TB as STA, fsm goes to TRIG_RX_CTRL state
        deaggregatorNs = READ_RX_VECTOR2;
      else if (rxCntrlReady)
        //$fsm_t When rxCntrlReady is true, fsm goes to WAIT_RX_CTRL_READY state
        deaggregatorNs = WAIT_RX_CTRL_READY; 
      else
        //$fsm_t When rxCntrlReady is false, fsm stays to TRIG_RX_CTRL state
        deaggregatorNs = TRIG_RX_CTRL;
      
    RX_AMPDU_DELIM:
      //$fsm_s In RX_AMPDU_DELIM state, fsm waits delimitor to be found.
      if (correctDelimiter)
        //$fsm_t When correctDelimiter is true, fsm goes to TRIG_RX_CTRL state
        deaggregatorNs = TRIG_RX_CTRL;
      else if (rxFifoReadEnDel1 && ampduByteCntEnd)
        //$fsm_t When ampduByteCntEnd is true, fsm goes to READ_RX_VECTOR2 state
        deaggregatorNs = READ_RX_VECTOR2;
      else
        //$fsm_t When correctDelimiter and ampduByteCntEnd are false,
        // fsm stays to RX_AMPDU_DELIM state
        deaggregatorNs = RX_AMPDU_DELIM;

    RX_ERROR:
      //$fsm_s In RX_ERROR state, fsm checks if macPhyIfRxEnd_p has been received.
      if (macPhyIfRxEndSamp && macPhyIfRxFifoEmpty)
        //$fsm_t When macPhyIfRxEndSamp is true, fsm goes to IDLE state
        deaggregatorNs = IDLE; 
      else
        //$fsm_t When macPhyIfRxEndSamp is not true, fsm goes to WAIT_PHY_END state
        deaggregatorNs = WAIT_PHY_END;

    WAIT_RX_CTRL_READY:
      //$fsm_s In WAIT_RX_CTRL_READY state, fsm waits rxCntrlReady active.
      if (rxCntrlReady || (frameRcvedFlag && !rxAggregation))
        //$fsm_t When rxCntrlReady is true, fsm goes to READ_MP_FIFO state
        deaggregatorNs = READ_MP_FIFO; 
      else
        //$fsm_t When rxCntrlReady is false, fsm stays to WAIT_RX_CTRL_READY state
        deaggregatorNs = WAIT_RX_CTRL_READY;

    READ_MP_FIFO:
      //$fsm_s In READ_MP_FIFO state, fsm waits until end of the frame is reached.
      if (rxDataValid && byteCntEnd && rxAggregation && !ampduByteCntEnd)
        //$fsm_t When byteCntEnd is true and frame is an A-MPDU and this is not 
        // the last A-MPDU, fsm goes to RX_AMPDU_DELIM state
        deaggregatorNs = RX_AMPDU_DELIM; 
      else if (rxDataValid && (ampduByteCntEnd || byteCntEnd && !rxAggregation))
        //$fsm_t When byteCntEnd is true and frame is an A-MPDU and this is  
        // the last A-MPDU packet, fsm goes to READ_RX_VECTOR2 state
        deaggregatorNs = READ_RX_VECTOR2;
      else if (frameRcvedFlag && !rxAggregation)
        //$fsm_t When a frame of a singleton MPDU has been received, fsm stays in 
        //READ_MP_FIFO state to pops additional bytes in case of lenght error
        deaggregatorNs = READ_MP_FIFO;
      else if (!rxCntrlReady)
        //$fsm_t When rxCntrlReady is false, fsm goes to WAIT_RX_CTRL_READY state
        deaggregatorNs = WAIT_RX_CTRL_READY;
      else
        //$fsm_t When rxCntrlReady is true, fsm stays to READ_MP_FIFO state
        deaggregatorNs = READ_MP_FIFO;

    READ_RX_VECTOR2:
      //$fsm_s In READ_RX_VECTOR2 state, fsm waits byteCntEnd is active.
      if (byteCntEnd && rxFifoReadEnDel1)
        //$fsm_t When byteCntEnd is true, fsm goes to VECTOR2_VALID state
        deaggregatorNs = VECTOR2_VALID; 
      else
        //$fsm_t When byteCntEnd is false, fsm stays
        // to READ_RX_VECTOR2 state
        deaggregatorNs = READ_RX_VECTOR2;

    VECTOR2_VALID:
      //$fsm_s In VECTOR2_VALID state, fsm goes to WAIT_PHY_END.

        //$fsm_t After one clock cycle, fsm goes to WAIT_PHY_END
        deaggregatorNs = WAIT_PHY_END;

    WAIT_PHY_END:
      //$fsm_s In WAIT_PHY_END state, fsm waits macPhyIfRxEnd_p active.
      if ((macPhyIfRxEnd_p || macPhyIfRxEndSamp) && macPhyIfRxFifoEmpty)
        //$fsm_t When macPhyIfRxEnd_p is true, fsm goes to IDLE state
        deaggregatorNs = IDLE; 
      else
        //$fsm_t When macPhyIfRxEnd_p is false, fsm stays to WAIT_PHY_END state
        deaggregatorNs = WAIT_PHY_END;
      
      // Disable case default state for block coverage
      // pragma coverage block = off
     default:   
        deaggregatorNs = IDLE;
      // pragma coverage block = on
        
  endcase
  end
end


///////////////////////////////////////
// Byte Counter control
///////////////////////////////////////
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
  begin
    rxByteCnt  <= 16'b0;
    byteCntEnd <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    rxByteCnt  <= 16'b0;
    byteCntEnd <= 1'b0;
  end
  else
  begin
    // Clear counter
    if ((deaggregatorCs == IDLE) || (deaggregatorCs == WAIT_PHY_END))
    begin
      rxByteCnt  <= 16'b0;
      byteCntEnd <= 1'b0;
    end
    else
    begin
      // Set counter
      if (deaggregatorCs == COMP_FIFO_LATENCY)
        // Set counter for RX Vector 1
        rxByteCnt <= 16'd0;
      else if (deaggregatorCs == TRIG_RX_CTRL)
      begin
        // Set counter for RX data
        if (rxLength != 16'd0)
          rxByteCnt <= rxLength-16'd1;
      end
      else if ((rxFifoReadEnDel1 && ampduByteCntEnd) ||
            ((deaggregatorCs == READ_MP_FIFO) && rxDataValid &&
              byteCntEnd && !rxAggregation))
        // Set counter for RX Vector 2
        rxByteCnt <= 16'd0;
      //else if (rxFifoReadEnDel1 && (deaggregatorCs == READ_RX_VECTOR1) && byteCntEnd && rxNDP)
      else if ((deaggregatorCs == VECTOR1_VALID) & rxHTLength==20'd0 & rxFormatMod>=4'd2) // ToDo
        // Set counter for RX Vector 2
        rxByteCnt <= 16'd0;
         
      
      // Increment counter
      if (rxFifoReadEnDel1)
      begin
        // Rx Vector 1
        if (deaggregatorCs == READ_RX_VECTOR1)
        begin
          // Byte counter control
          if (rxByteCnt != {11'h0,rxVector1LastIndex})
            rxByteCnt <= rxByteCnt + 16'd1;
          // Byte counter end control
          if ((rxByteCnt+16'd1) == {11'h0,rxVector1LastIndex})
            byteCntEnd <= 1'b1;
          else if (byteCntEnd)
            byteCntEnd <= 1'b0;
        end
        // RX Vector 2
        else if (deaggregatorCs == READ_RX_VECTOR2)
        begin
          // Byte counter control
          if (rxByteCnt != {12'h0,rxVector2LastIndex})
            rxByteCnt <= rxByteCnt + 16'd1;
          // Byte counter end control
          if ((rxByteCnt+16'd1) == {12'h0,rxVector2LastIndex})
            byteCntEnd <= 1'b1;
          else if (byteCntEnd)
            byteCntEnd <= 1'b0;
        end
      end
      // RX data
      if (rxDataValid && ((deaggregatorCs == READ_MP_FIFO)))
      begin
        // Byte counter control
        if ((rxByteCnt != 16'd0) && !(byteCntEnd || ampduByteCntEnd))
          rxByteCnt <= rxByteCnt - 16'd1;
        // Byte counter end control
        if (rxByteCnt == 16'd1)
          byteCntEnd <= 1'b1;
        else if (byteCntEnd || ampduByteCntEnd)
          byteCntEnd <= 1'b0;
      end
    end
  end
end

// macPhyIfRxFifoReadEn delayed by 1 cc due to MAC PHY IF FIFO latency
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxFifoReadEnDel1 <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxFifoReadEnDel1 <= 1'b0;
  else
    rxFifoReadEnDel1 <= macPhyIfRxFifoReadEn;
end


///////////////////////////////////////
// RX Vectors Length
///////////////////////////////////////
always @*
begin
   case (rxFormatMod)
   NON_HT:     rxVector1LastIndex=5'd6;
   NON_HT_DUP: rxVector1LastIndex=5'd6;
   HT_MM:      rxVector1LastIndex=5'd9;
   HT_GF:      rxVector1LastIndex=5'd9;
   VHT:        rxVector1LastIndex=5'd12;
   HE_SU:      rxVector1LastIndex=5'd15;
   HE_EXT_SU:  rxVector1LastIndex=5'd15;
   HE_MU:      rxVector1LastIndex=5'd15;
   HE_TRIG:    rxVector1LastIndex=5'd11; // STA Only
   // Disable case default state for block coverage
   // pragma coverage block = off
   default:    rxVector1LastIndex=5'd6;
   // pragma coverage block = on
   endcase
end
assign rxVector2LastIndex=4'd7;


///////////////////////////////////////
// RX Vectors
///////////////////////////////////////
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin:rx_vectors
  // local variable
  integer v_i;

  if (macCoreClkHardRst_n == 1'b0)
  begin
    for (v_i=0;v_i<16;v_i=v_i+1)
      rxVector1Mem[v_i] <= 8'b0;
    rxVector2  <= 64'b0;
    //htRxLength <= 20'b0;
    mpduLength <= 16'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    for (v_i=0;v_i<16;v_i=v_i+1)
      rxVector1Mem[v_i] <= 8'b0;
    rxVector2  <= 64'b0;
    //htRxLength <= 20'b0;
    mpduLength <= 16'b0;
  end
  else if ((deaggregatorCs == COMP_FIFO_LATENCY) && (deaggregatorNs == READ_RX_VECTOR1))
  begin
    for (v_i=0;v_i<16;v_i=v_i+1)
      rxVector1Mem[v_i] <= 8'b0;
    rxVector2  <= 64'b0;
    //htRxLength <= 20'b0;
    mpduLength <= 16'b0;
  end
  else
  begin
    // Update rxVector1 length with individual A-MPDU length
    if (rxAggregation)
    begin
      if ((crcByteCnt == 2'h2) && (deaggregatorCs == RX_AMPDU_DELIM))
        mpduLength <= (rxFormatMod >= 4'd4) ? {2'b0,rxDataHold2[3:2],rxDataHold1,rxDataHold2[7:4]}:
                                              {4'b0,                 rxDataHold1,rxDataHold2[7:4]};
    end
    else
      mpduLength <= rxHTLength[15:0]/*htRxLength[15:0]*/;
    
    if (rxFifoReadEnDel1)
    begin
      // RX_VECTOR1
      if (deaggregatorCs == READ_RX_VECTOR1)
      begin
        if (rxByteCnt[4:0] <= rxVector1LastIndex)
          rxVector1Mem[rxByteCnt[4:0]] <= macPhyIfRxFifoData;
        //htRxLength <= {rxVector1[4][3:0],rxVector1[3],rxVector1[2]};
      end
      // RX_VECTOR2
      else if (deaggregatorCs == READ_RX_VECTOR2)
      begin
        case (rxByteCnt[4:0])
        5'h0: rxVector2[0*8 +: 8]  <= macPhyIfRxFifoData;
        5'h1: rxVector2[1*8 +: 8]  <= macPhyIfRxFifoData;
        5'h2: rxVector2[2*8 +: 8]  <= macPhyIfRxFifoData;
        5'h3: rxVector2[3*8 +: 8]  <= macPhyIfRxFifoData;
        5'h4: rxVector2[4*8 +: 8]  <= macPhyIfRxFifoData;
        5'h5: rxVector2[5*8 +: 8]  <= macPhyIfRxFifoData;
        5'h6: rxVector2[6*8 +: 8]  <= macPhyIfRxFifoData;
        5'h7: rxVector2[7*8 +: 8]  <= macPhyIfRxFifoData;
        // Disable case default state for block coverage
        // pragma coverage block = off
        //default : rxVector2     <= rxVector2;
        // pragma coverage block = on
        endcase
      end
    end
  end
end


///////////////////////////////////////
// Parameter extraction from Rx Vectors
///////////////////////////////////////
always @*
begin
   // Default value
   rxDynBWInNonHT        = 1'b0;
   rxChBWInNonHT         = 2'b0;
   rxLSIGValid           = 1'b0;
   rxSounding            = 1'b0;
   rxSmoothing           = 1'b0;
   rxGIType              = 2'b0;
   rxAggregation         = 1'b0;
   rxSTBC                = 1'b0;
   rxNumExtnSS           = 2'b0;
   rxMCS                 = 7'b0;
   rxNss                 = 3'b0;
   rxFecCoding           = 1'b0;
   rxHTLength            = 20'h0;
   rxBeamformed          = 1'b0;
   rxDozeNotAllowed      = 1'b0;
   rxFirstUser           = 1'b0;
   rxPartialAID          = 9'b0;
   rxGroupID             = 6'b0;
   rxUplinkFlag          = 1'b0;
   rxBeamChange          = 1'b0;
   rxDcm                 = 1'b0;
   rxHeLtfType           = 2'b0;
   rxDoppler             = 1'b0;
   rxMidamble            = 1'b0;
   rxBssColor            = 6'b0;
   rxTxopDuration        = 7'b0;
   rxPeDuration          = 4'b0;
   rxSpatialReuse1       = 4'b0;
   rxSpatialReuse2       = 4'b0;
   rxSpatialReuse3       = 4'b0;
   rxSpatialReuse4       = 4'b0;
   rxSigBCompressionMode = 1'b0;
   rxDcmSigB             = 1'b0;
   rxMcsSigB             = 3'b0;
   rxRUSize              = 3'b0;

   // Receive vector common part for all frames
   rxFormatMod                  = rxVector1Mem[0][3:0];
   rxChBW                       = rxVector1Mem[0][6:4];
   rxPreType                    = rxVector1Mem[0][7];
   rxAntennaSet                 = rxVector1Mem[1][7:0];
   rxRSSILegacy                 = rxVector1Mem[2][7:0];
   rxLegLength[7:0]             = rxVector1Mem[3][7:0];
   rxLegLength[11:8]            = rxVector1Mem[4][3:0];
   rxLegRate                    = rxVector1Mem[4][7:4];
   rxRSSI                       = rxVector1Mem[5][7:0];

   
   case (rxFormatMod)
   // Receive vector specific part for NON-HT frames
   NON_HT, NON_HT_DUP:
   begin
      rxDynBWInNonHT            = rxVector1Mem[6][0];
      rxChBWInNonHT             = rxVector1Mem[6][2:1];
      rxLSIGValid               = rxVector1Mem[6][7];
   end

   // Receive vector specific part for HT frames
   HT_MM, HT_GF:
   begin
      rxSounding                = rxVector1Mem[6][0];
      rxSmoothing               = rxVector1Mem[6][1];
      rxGIType[0]               = rxVector1Mem[6][2];
      rxAggregation             = rxVector1Mem[6][3];
      rxSTBC                    = rxVector1Mem[6][4];
      rxNumExtnSS               = rxVector1Mem[6][6:5];
      rxLSIGValid               = rxVector1Mem[6][7];
      rxMCS                     = rxVector1Mem[7][6:0];
      rxFecCoding               = rxVector1Mem[7][7];
      rxHTLength[7:0]           = rxVector1Mem[8][7:0];
      rxHTLength[15:8]          = rxVector1Mem[9][7:0];
   end

   // Receive vector specific part for VHT frames
   VHT:
   begin
      rxSounding                = rxVector1Mem[6][0];
      rxBeamformed              = rxVector1Mem[6][1];
      rxGIType[0]               = rxVector1Mem[6][2];
      rxSTBC                    = rxVector1Mem[6][4];
      rxDozeNotAllowed          = rxVector1Mem[6][5];
      rxFirstUser               = rxVector1Mem[6][6];
      rxPartialAID[7:0]         = rxVector1Mem[7][7:0];
      rxPartialAID[8]           = rxVector1Mem[8][0];
      rxGroupID                 = rxVector1Mem[8][6:1];
      rxMCS[3:0]                = rxVector1Mem[9][3:0];
      rxNss                     = rxVector1Mem[9][6:4];
      rxFecCoding               = rxVector1Mem[9][7];
      rxHTLength[7:0]           = rxVector1Mem[10][7:0];
      rxHTLength[15:8]          = rxVector1Mem[11][7:0];
      rxHTLength[19:16]         = rxVector1Mem[12][3:0];

      rxAggregation             = 1'b1;
   end

   default:
   begin
      // Receive vector specific part for all HE frames
      rxSounding                = rxVector1Mem[6][0];
      rxBeamformed              = rxVector1Mem[6][1];
      rxGIType                  = rxVector1Mem[6][3:2];
      rxSTBC                    = rxVector1Mem[6][4];
      rxUplinkFlag              = rxVector1Mem[7][0];
      rxBeamChange              = rxVector1Mem[7][1];
      rxDcm                     = rxVector1Mem[7][2];
      rxHeLtfType               = rxVector1Mem[7][4:3];
      rxDoppler                 = rxVector1Mem[7][5];
      rxMidamble                = rxVector1Mem[7][6];
      rxBssColor                = rxVector1Mem[8][5:0];
      rxTxopDuration            = rxVector1Mem[9][6:0];

      case (rxFormatMod)
      // Receive vector specific part for HE-SU frames
      HE_SU, HE_EXT_SU:
      begin
         rxPeDuration           = rxVector1Mem[10][3:0];
         rxSpatialReuse1        = rxVector1Mem[10][7:4];
         rxMCS[3:0]             = rxVector1Mem[12][3:0];
         rxNss                  = rxVector1Mem[12][6:4];
         rxFecCoding            = rxVector1Mem[12][7];
         rxHTLength[7:0]        = rxVector1Mem[13][7:0];
         rxHTLength[15:8]       = rxVector1Mem[14][7:0];
         rxHTLength[19:16]      = rxVector1Mem[15][3:0];
      end

      // Receive vector specific part for HE-MU frames
      HE_MU:
      begin
         rxPeDuration           = rxVector1Mem[10][3:0];
         rxSpatialReuse1        = rxVector1Mem[10][7:4];
         rxSigBCompressionMode  = rxVector1Mem[11][0];
         rxDcmSigB              = rxVector1Mem[11][1];
         rxMcsSigB              = rxVector1Mem[11][4:2];
         rxRUSize               = rxVector1Mem[11][7:5];
         rxMCS[3:0]             = rxVector1Mem[12][3:0];
         rxNss                  = rxVector1Mem[12][6:4];
         rxFecCoding            = rxVector1Mem[12][7];
         rxHTLength[7:0]        = rxVector1Mem[13][7:0];
         rxHTLength[15:8]       = rxVector1Mem[14][7:0];
         rxHTLength[19:16]      = rxVector1Mem[15][3:0];
      end

      // Receive vector specific part for HE-TB STA frames (STA Only)
      HE_TRIG:
      begin
         rxSpatialReuse1        = rxVector1Mem[10][3:0];
         rxSpatialReuse2        = rxVector1Mem[10][7:4];
         rxSpatialReuse3        = rxVector1Mem[11][3:0];
         rxSpatialReuse4        = rxVector1Mem[11][7:4];
      end
      endcase

      rxAggregation             = 1'b1;
   end
   endcase
end


assign rxRCPI        = rxVector2[7:0];
assign rxEVM1        = rxVector2[15:8];
assign rxEVM2        = rxVector2[23:16];
assign rxEVM3        = rxVector2[31:24];
assign rxEVM4        = rxVector2[39:32];


always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxDSSS <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (deaggregatorCs == IDLE))
    rxDSSS <= 1'b0;
  else if ((deaggregatorCs == READ_RX_VECTOR1) && rxByteCnt[4:0] == 5'd5)
    rxDSSS <= (rxFormatMod == 4'd0) && (rxLegRate < 4'd4);
end


// rxVector1 vector assignment
always @*
begin:rx_vector1_out
  // local variable
  integer v_i;

  for (v_i=0;v_i<16;v_i=v_i+1)
    rxVector1[v_i*8 +: 8] = rxVector1Mem[v_i];
end

///////////////////////////////////////
// Length selection
///////////////////////////////////////
// RX length mux between HT value and Legacy value
assign muxRxLength = (rxFormatMod >= 4'd2) ? rxHTLength/*htRxLength*/ : {8'b0, rxLegLength};
// RX length mux between A-MPDU value and muxRxLength value
assign rxLength = (rxAggregation) ? mpduLength : muxRxLength[15:0];

always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxMpduLength <= 16'b0;
  else
    rxMpduLength <= rxLength;
end



// detection of NDP 
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxNDP <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (deaggregatorCs == IDLE))
    rxNDP <= 1'b0;
  else if (deaggregatorCs == VECTOR1_VALID)
  begin
    // Null Data Packet exists only in HT / VHT / HE-SU
    if ((rxHTLength == 20'b0) && (rxFormatMod >= 4'd2) && (rxFormatMod <= 4'd5))
      rxNDP <= 1'b1;
    else  
      rxNDP <= 1'b0;
  end
end


// Beamformed frame
// formatMod VHT: if rxSmoothing field is '0', the received frame is beamformed
// formatMod HT : if rxSmoothing field is '0', the received frame is considered as beamformed
//assign rxBeamformed  = (rxFormatMod[2:0] >= 3'd2) ? ~rxSmoothing : 1'b0;


///////////////////////////////////////
// Trigger to RX Controller
///////////////////////////////////////
// Inform RX Controller to start reception procedure
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxDataStart_p <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (deaggregatorCs == IDLE))
    rxDataStart_p <= 1'b0;
  else
    rxDataStart_p <= (rxCntrlReady && (deaggregatorCs == TRIG_RX_CTRL) && !macPhyIfRxErr_p);
end


// Inform RX Controller to stop reception procedure
assign rxDataEnd_p      = ((deaggregatorCs == READ_MP_FIFO) && 
                             rxDataValid && byteCntEnd) ? 1'b1 : 1'b0;

// Inform RX Controller of length error
assign rxDataError_p    = (deaggregatorCs == RX_ERROR) ? 1'b1 : 1'b0;

// Inform RX Controller that RX Vectors 2 are available
assign rxVector2Valid_p = (deaggregatorCs == VECTOR2_VALID) ? 1'b1 : 1'b0;

// RX data from MAC PHY IF FIFO
assign rxData = macPhyIfRxFifoData;

// Inform RX Controller that data is available
assign rxDataValid = ((deaggregatorCs == READ_MP_FIFO) && 
                     (rxFifoReadEnDel1 || rxDataValidAfterReady)) ? 1'b1 : 1'b0;

// One clk cycle rxDataValid after RX Controller is ready
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxDataValidAfterReady <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxDataValidAfterReady <= 1'b0;
  else if (((deaggregatorCs == WAIT_RX_CTRL_READY) ||
            (deaggregatorCs == TRIG_RX_CTRL)) && rxFifoReadEnDel1)
    rxDataValidAfterReady <= 1'b1;
  else if (deaggregatorCs == READ_MP_FIFO)
    rxDataValidAfterReady <= 1'b0;
end

// macPhyIfRxEnd_p sampled for FSM information
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    macPhyIfRxEndSamp <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    macPhyIfRxEndSamp <= 1'b0;
  else if (deaggregatorCs == IDLE)
    macPhyIfRxEndSamp <= 1'b0;
  else if (macPhyIfRxEnd_p)
    macPhyIfRxEndSamp <= 1'b1;
end

// Read enable to MAC PHY IF FIFO
always @* 
begin
  // When MAC PHY IF FIFO is empty release macPhyIfRxFifoReadEn !
  if (macPhyIfRxFifoEmpty == 1'b1)
    macPhyIfRxFifoReadEn = 1'b0;
  else
  begin
    case(deaggregatorCs)

      COMP_FIFO_LATENCY, RX_AMPDU_DELIM, READ_MP_FIFO, WAIT_PHY_END:
        // Read
        macPhyIfRxFifoReadEn = 1'b1;

      READ_RX_VECTOR1:
        // Conditional read
        if (rxByteCnt == {11'h0,rxVector1LastIndex})
        begin
          if (rxFifoReadEnDel1)
            macPhyIfRxFifoReadEn = 1'b0;
          else
            macPhyIfRxFifoReadEn = 1'b1;
        end
        else
          macPhyIfRxFifoReadEn = 1'b1;

      READ_RX_VECTOR2:
        // Conditional read
        if (rxByteCnt == {12'h0,rxVector2LastIndex})
        begin
          if (rxFifoReadEnDel1)
            macPhyIfRxFifoReadEn = 1'b0;
          else
            macPhyIfRxFifoReadEn = 1'b1;
        end
        else
          macPhyIfRxFifoReadEn = 1'b1;

      default:
        macPhyIfRxFifoReadEn = 1'b0;
          
    endcase
  end
end


///////////////////////////////////////
// Triggers to MAC Controller
///////////////////////////////////////
// Inform MAC Controller to send ACK after A-MPDU end
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
  begin
    frameRcvedFlag        <= 1'b0;
    ampduIncorrectRcved_p <= 1'b0;
    ampduCorrectRcved_p   <= 1'b0;
    rxSMPDU               <= 1'b0;
    rxFirstDelimiterDone  <= 1'b0;
    rxEOF                 <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    frameRcvedFlag        <= 1'b0;
    ampduIncorrectRcved_p <= 1'b0;
    ampduCorrectRcved_p   <= 1'b0;
    rxSMPDU               <= 1'b0;
    rxFirstDelimiterDone  <= 1'b0;
    rxEOF                 <= 1'b0;
  end
  else
  begin
    // A-MPDU received pulse generation
    if ((deaggregatorCs == VECTOR2_VALID) && rxAggregation && !rxNDP)
    begin
      ampduIncorrectRcved_p <= !frameRcvedFlag;
      ampduCorrectRcved_p   <= frameRcvedFlag;
    end
    else if ((deaggregatorCs == VECTOR2_VALID) && rxAggregation && rxNDP)
    begin
      ampduIncorrectRcved_p <= 1'b0;
      ampduCorrectRcved_p   <= 1'b1;
    end
    else if ((deaggregatorCs == RX_ERROR) && rxAggregation)
    begin
      ampduIncorrectRcved_p <= 1'b1;
      ampduCorrectRcved_p   <= 1'b0;
    end
    else
    begin
      ampduIncorrectRcved_p <= 1'b0;
      ampduCorrectRcved_p   <= 1'b0;
    end

    // Detection of bit EOF
    if ((deaggregatorCs == RX_AMPDU_DELIM) && (crcByteCnt == 2'b00) && (rxFormatMod >= 4'b0100) &&
        rxAggregation)
    begin
      rxEOF <= rxData[0];
    end
    else if((deaggregatorCs != IDLE) && (deaggregatorNs == IDLE))
    begin
      rxEOF <= 1'b0;
    end


    // Set the rxSMPDU if the delimiter has length not null and if it is first delimiter 
    // and VHT/HE mode and crcOk
    if ((deaggregatorCs == RX_AMPDU_DELIM) && correctDelimiter && (crcByteCnt == 2'b11) &&
        (rxFormatMod >= 4'b0100) && rxAggregation && crcOk && (mpduLength != 16'h0))
    begin
      rxFirstDelimiterDone <=  1'b1;
      rxSMPDU              <= !rxFirstDelimiterDone && rxEOF;
    end
    // Clear the flag if the state is going idle
    else if ((deaggregatorCs != IDLE) && (deaggregatorNs == IDLE))
    begin
      rxSMPDU              <= 1'b0;
      rxFirstDelimiterDone <= 1'b0;
    end

    if (deaggregatorCs == IDLE)
      // Clear flags
      frameRcvedFlag <= 1'b0;
    else if ((deaggregatorCs == VECTOR2_VALID) && rxAggregation && !rxNDP)
      frameRcvedFlag <= 1'b0;
    else if (correctRcved_p)
      // Generates flag
      frameRcvedFlag <= 1'b1;
  end
end

// Inform MAC Controller that RX Vectors 1 are available
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxVector1Valid_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxVector1Valid_p <= 1'b0;
  else 
  begin
    // Pulse generation
    rxVector1Valid_p <= 1'b0;
    //if ((deaggregatorCs == READ_RX_VECTOR1) && (deaggregatorNs != READ_RX_VECTOR1))
    if (deaggregatorCs == VECTOR1_VALID)
      rxVector1Valid_p <= 1'b1;
  end
end

// Inform MAC Controller about RX frame ending
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxEndOfFrame_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxEndOfFrame_p <= 1'b0;
  else 
  begin
    if (rxDataError_p || ((deaggregatorCs == WAIT_PHY_END) && (macPhyIfRxEnd_p || macPhyIfRxEndSamp)))
      rxEndOfFrame_p <= 1'b1;
    else
      rxEndOfFrame_p <= 1'b0;
  end
end

// Inform MAC Controller about the reception of the first byte of rxVector1
assign rxVector1Start_p = (deaggregatorCs == COMP_FIFO_LATENCY) && !macPhyIfRxFifoEmpty;


///////////////////////////////////////
// A-MPDU logic
///////////////////////////////////////
// Signature found generation
assign signatureFound = ((deaggregatorCs == RX_AMPDU_DELIM) &&
                         (macPhyIfRxFifoData == 8'h4e)) ? 1'b1 : 1'b0;

// Correct delimiter generation
assign correctDelimiter = (deaggregatorCs == RX_AMPDU_DELIM) &&
                           signatureFound && 
                           (crcByteCnt == 2'd3) &&
                           crcOk && 
                           rxFifoReadEnDel1 && 
                           (mpduLength != 16'b0) &&
                           ({4'b0,mpduLength} <= ampduByteCnt) &&
                           ({4'b0,mpduLength} < muxRxLength) &&
                           ({4'b0,mpduLength} <= maxAllowedLength);

// Data hold for CRC checking
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
  begin
    rxDataHold1 <= 8'b0;
    rxDataHold2 <= 8'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    rxDataHold1 <= 8'b0;
    rxDataHold2 <= 8'b0;
  end
  // Hold data in RX_AMPDU_DELIM state
//   else if ((deaggregatorCs == RX_AMPDU_DELIM) && rxFifoReadEnDel1 &&
//             !signatureFound)
  else if ((deaggregatorCs == RX_AMPDU_DELIM) && rxFifoReadEnDel1)
  begin
    rxDataHold1 <= macPhyIfRxFifoData;
    rxDataHold2 <= rxDataHold1;
  end
end

// CRC byte counter
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    crcByteCnt <= 2'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    crcByteCnt <= 2'b0;
  // Clear counter in IDLE state
  else if ((deaggregatorCs == IDLE) || 
          ((crcByteCnt == 2'h3) && rxFifoReadEnDel1))
    crcByteCnt <= 2'b0;
  // Increment counter in RX_AMPDU_DELIM state
  else if (((deaggregatorCs == RX_AMPDU_DELIM) ||
            (deaggregatorCs == READ_MP_FIFO) ||
            (deaggregatorCs == WAIT_RX_CTRL_READY) ||
            (deaggregatorCs == TRIG_RX_CTRL)) && 
           rxFifoReadEnDel1 && rxAggregation)
    crcByteCnt <= crcByteCnt + 2'h1;
end

// CRC 8-bit with polynomial x8+x2+x1+1
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    crcCalc <= 8'hff;
  else if(macCoreClkSoftRst_n == 1'b0)
    crcCalc <= 8'hff;
  // Re-initialyse CRC in IDLE state and when crcByteCnt = 3
  else if ((deaggregatorCs == IDLE) ||
          ((crcByteCnt == 2'h3) && rxFifoReadEnDel1))
    crcCalc <= 8'hff;
  else if ((deaggregatorCs == RX_AMPDU_DELIM) && rxFifoReadEnDel1 &&
           (crcByteCnt < 2'h2))
  begin
    // CRC generation
    crcCalc[0] <= macPhyIfRxFifoData[0] ^ macPhyIfRxFifoData[1] ^ macPhyIfRxFifoData[7] ^ 
                  crcCalc[0] ^ crcCalc[6] ^ crcCalc[7];
    crcCalc[1] <= macPhyIfRxFifoData[1] ^ macPhyIfRxFifoData[6] ^ macPhyIfRxFifoData[7] ^ 
                  crcCalc[0] ^ crcCalc[1] ^ crcCalc[6];
    crcCalc[2] <= macPhyIfRxFifoData[1] ^ macPhyIfRxFifoData[5] ^ macPhyIfRxFifoData[6] ^ 
                  macPhyIfRxFifoData[7] ^ crcCalc[0] ^ crcCalc[1] ^ crcCalc[2]  ^
                  crcCalc[6];
    crcCalc[3] <= macPhyIfRxFifoData[0] ^ macPhyIfRxFifoData[4] ^ macPhyIfRxFifoData[5] ^
                  macPhyIfRxFifoData[6] ^ crcCalc[1] ^ crcCalc[2] ^ crcCalc[3]  ^ 
                  crcCalc[7];
    crcCalc[4] <= macPhyIfRxFifoData[3] ^ macPhyIfRxFifoData[4] ^ macPhyIfRxFifoData[5] ^
                  crcCalc[2] ^ crcCalc[3] ^ crcCalc[4];
    crcCalc[5] <= macPhyIfRxFifoData[2] ^ macPhyIfRxFifoData[3] ^ macPhyIfRxFifoData[4] ^
                  crcCalc[3] ^ crcCalc[4] ^ crcCalc[5];
    crcCalc[6] <= macPhyIfRxFifoData[1] ^ macPhyIfRxFifoData[2] ^ macPhyIfRxFifoData[3] ^
                  crcCalc[4] ^ crcCalc[5] ^ crcCalc[6];
    crcCalc[7] <= macPhyIfRxFifoData[0] ^ macPhyIfRxFifoData[1] ^ macPhyIfRxFifoData[2] ^
                  crcCalc[5] ^ crcCalc[6] ^ crcCalc[7];
  end
end


assign crcCalcInv = {~crcCalc[0],~crcCalc[1],~crcCalc[2],~crcCalc[3],~crcCalc[4],~crcCalc[5],~crcCalc[6],~crcCalc[7]};

// CRC Ok generation
assign crcOk = ((deaggregatorCs == RX_AMPDU_DELIM) && 
                (crcCalcInv == rxDataHold1)) ? 1'b1 : 1'b0;

// A-MPDU Byte Counter control
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
  begin
    ampduByteCnt    <= 20'b0;
    ampduByteCntEnd <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    ampduByteCnt    <= 20'b0;
    ampduByteCntEnd <= 1'b0;
  end
  else
  begin
    // Clear counter in IDLE or WAIT_PHY_END state
    if ((deaggregatorCs == IDLE) || (deaggregatorCs == WAIT_PHY_END))
    begin
      ampduByteCnt    <= 20'b0;
      ampduByteCntEnd <= 1'b0;
    end
    // If frame is an A-MPDU, then control counter
    else if (rxAggregation)
    begin
      if (deaggregatorCs == VECTOR1_VALID)
        // Set counter
          ampduByteCnt <= muxRxLength - 20'h001;
      // READ_MP_FIFO || RX_AMPDU_DELIM || WAIT_RX_CTRL_READY
      else if ((rxFifoReadEnDel1 && (deaggregatorCs == RX_AMPDU_DELIM)) ||
               (rxDataValid && ((deaggregatorCs == READ_MP_FIFO))))
      begin
        // Byte counter control
        if (ampduByteCnt != 20'h000)
          // Decrement counter
          ampduByteCnt <= ampduByteCnt - 20'h001;
        // Byte counter end control
        if (ampduByteCnt == 20'h001)
          ampduByteCntEnd <= 1'b1;
        else if (ampduByteCntEnd)
          ampduByteCntEnd <= 1'b0;
      end
    end
  end
end


// Statistics
//////////////////////////////////////////////////////////////////////////////

// received MPDU counter
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxMPDUCnt <= 6'h3F;
  else if (deaggregatorCs == IDLE)
    rxMPDUCnt <= 6'h3F;
  // Increment counter every MPDU detected
  else if (rxDataStart_p)
    rxMPDUCnt <= rxMPDUCnt + 6'd1;
end

// received MPDU counter
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    rxAMPDUCnt <= 2'd0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxAMPDUCnt <= 2'd0;
  // Increment counter every A-MPDU detected
  else if (rxVector1Valid_p && rxAggregation)
    rxAMPDUCnt <= rxAMPDUCnt + 2'd1;
end




// Flag indicating the end of the AMPDU delimiter
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    inDelimiter       <= 1'b0;
  else if (deaggregatorCs != RX_AMPDU_DELIM)
    inDelimiter       <= 1'b0;
  else if ((deaggregatorCs == RX_AMPDU_DELIM) && rxFifoReadEnDel1 && (crcByteCnt == 2'd0))
    inDelimiter       <= 1'b1;
end  

// incorrect Delimiter counter
// An immediately repeated CRC error within an A-MPDU is not counted.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
  begin
    incorrectDelCnt <= 8'd0;
    incDelFound     <= 1'b0;
  end  
  else if (deaggregatorCs == IDLE)
  begin
    incorrectDelCnt <= 8'd0;
    incDelFound     <= 1'b0;
  end  
  // Each time a delimiter is received check if it is correct or not
  else if (inDelimiter && rxFifoReadEnDel1)
  begin
    // If the delimiter is correct, reset the incDelFound flag. 
    if (correctDelimiter) 
      incDelFound     <= 1'b0;
    // If the delimiter is not correct, the delimiter is not a blank delimiter (mpduLength != 16'b0) 
    // and the incDelFound flag is not set
    // The flag incDelFound is used to detect the first incorrect delimiter
    else if (!incDelFound && (mpduLength != 16'b0) && (crcByteCnt == 2'd3))
    begin
      incorrectDelCnt <= incorrectDelCnt + 8'd1;
      incDelFound     <= 1'b1;
    end
  end  
end  

// Useful for MIB counters. This will be a pulse of 1 macCoreRxClk duration
// Note: This specifically sees READ_RX_VECTOR1 state for current state, 
// instead of (current state != next state). This is due to an stopRx_p 
// issue causing a double abort. As a result of this fix, the double 
// abort will  occur, but this signal will not be generated for the second
// incorrect abort 
//assign maxLengthExceeded_p = (deaggregatorNs != deaggregatorCs) &&
//                             (deaggregatorNs == RX_ERROR) && 
//                             (deaggregatorCs == READ_RX_VECTOR1) &&
//                             (muxRxLength > maxAllowedLength);
assign maxLengthExceeded_p = (deaggregatorNs == RX_ERROR) && 
                             (deaggregatorCs == VECTOR1_VALID) &&
                             (muxRxLength > maxAllowedLength);
 

// Debug port
//////////////////////////////////////////////////////////////////////////////
assign debugPortDeaggregator = {maxLengthExceeded_p,
                                rxFormatMod[1],
                                rxFormatMod[0],
                                rxAggregation,
                                macPhyIfRxFifoEmpty,
                                macPhyIfRxFifoReadEn,
                                startRx,
                                stopRx_p,
                                ampduCorrectRcved_p,
                                ampduIncorrectRcved_p,
                                correctRcved_p,
                                rxCntrlReady,
                                rxVector2Valid_p,
                                rxDataError_p,
                                rxDataEnd_p,
                                rxDataStart_p
                                };
//assign debugnSS = (rxSTBC == 2'd1) ? {1'b0,rxnSTS[2:1]} : rxnSTS;

always @*
begin
   case (rxFormatMod)
   NON_HT,NON_HT_DUP: debugMCS = {3'd0,rxLegRate};
   HT_MM,HT_GF:       debugMCS = rxMCS;
   default:           debugMCS = {rxNss,rxMCS[3:0]};
   endcase
end


assign debugPortDeaggregator2 = {rxFormatMod[2:0],  // 3
                                 debugMCS,          // 7
                                 rxChBW,            // 3
                                 rxGIType[0],       // 1
                                 rxAggregation,     // 1
                                 rxSTBC};           // 1

assign debugPortDeaggregatorFsm = deaggregatorCs;

//////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// deaggregator FSM states displayed in a string to ease simulation and debug
always @*
begin
  case (deaggregatorCs)
                 IDLE  :  deaggregatorCs_str = {"IDLE"};
    COMP_FIFO_LATENCY  :  deaggregatorCs_str = {"COMP_FIFO_LATENCY"};
      READ_RX_VECTOR1  :  deaggregatorCs_str = {"READ_RX_VECTOR1"};
        VECTOR1_VALID  :  deaggregatorCs_str = {"VECTOR1_VALID"};
         TRIG_RX_CTRL  :  deaggregatorCs_str = {"TRIG_RX_CTRL"};
       RX_AMPDU_DELIM  :  deaggregatorCs_str = {"RX_AMPDU_DELIM"};
         WAIT_PHY_END  :  deaggregatorCs_str = {"WAIT_PHY_END"};
   WAIT_RX_CTRL_READY  :  deaggregatorCs_str = {"WAIT_RX_CTRL_READY"};
         READ_MP_FIFO  :  deaggregatorCs_str = {"READ_MP_FIFO"};
      READ_RX_VECTOR2  :  deaggregatorCs_str = {"READ_RX_VECTOR2"};
        VECTOR2_VALID  :  deaggregatorCs_str = {"VECTOR2_VALID"};
             RX_ERROR  :  deaggregatorCs_str = {"RX_ERROR"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON
default clocking cb @(posedge macCoreRxClk); endclocking

//$rw_sva Checks if Deaggregator doesn't request data when MAC IF FIFO is empty.
property noReadWhenFifoEmpty_prop;
@(posedge macCoreRxClk)
    macPhyIfRxFifoEmpty |-> !macPhyIfRxFifoReadEn;
endproperty
noReadWhenFifoEmpty: assert property (disable iff (!macCoreClkHardRst_n) noReadWhenFifoEmpty_prop);

//$rw_sva Checks if Deaggregator doesn't trigger RX Controller with data valid.
property noDataWhenTrigger_prop;
@(posedge macCoreRxClk)
    rxDataStart_p |-> !rxDataValid;
endproperty
noDataWhenTrigger: assert property (disable iff (!macCoreClkHardRst_n) noDataWhenTrigger_prop);

//$rw_sva Checks if macPhyIfRxEnd_p is always generated after macPhyIfRxErr_p.
property macPhyIfEndAftermacPhyIfError_prop;
@(posedge macCoreRxClk)
    (macPhyIfRxErr_p ##1 !macPhyIfRxErr_p) |-> (macPhyIfRxEnd_p [*0:10] ##1 !macPhyIfRxEnd_p);
endproperty
macPhyIfEndAftermacPhyIfError: assert property (disable iff (!macCoreClkHardRst_n) macPhyIfEndAftermacPhyIfError_prop);

//$rw_sva Checks if macPhyIfRxEnd_p is always generated after rxDataError_p.
property macPhyIfEndAfterRxDataError_prop;
@(posedge macCoreRxClk)
    (rxDataError_p ##1 !rxDataError_p) |-> (macPhyIfRxEnd_p [*0:10] ##1 !macPhyIfRxEnd_p);
endproperty
macPhyIfEndAfterRxDataError: assert property (disable iff (!macCoreClkHardRst_n) macPhyIfEndAfterRxDataError_prop);

`endif // RW_ASSERT_ON


endmodule
