////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of rxListProc module
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -----------------------------------------------------------------------------
//                                                                          
// 
// 
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rxListProc( 
   ///////////////////////////////////////////////
   //$port_g Clock and reset
   ///////////////////////////////////////////////
   input  wire        macPIRxClk,               // Clock synchronized to Platform domain
   input  wire        macPIRxClkHardRst_n,      // Hard Reset synchronized to macPIRxClk
   input  wire        macPIRxClkSoftRst_n,      // Soft Reset synchronized to macPIRxClk

   ///////////////////////////////////////////////
   //$port_g rxFifo interface
   ///////////////////////////////////////////////
   output reg         rxFifoRd,                 // RX FIFO Read Request
   input  wire        rxFifoEmpty,              // RX FIFO empty indication
   input  wire        rxFifoAlmEmpty,           // RX FIFO alsmost empty indication
   input  wire [31:0] rxFifoRdData,             // Data read from RX FIFO
   input  wire  [3:0] rxTagFifoRdData,          // TAG  read from RX FIFO

   ///////////////////////////////////////////////
   //$port_g RdWrCtrl interface
   ///////////////////////////////////////////////
   output reg  [31:0] rxListStartAddress,       // Starting Address from the system memory
                                                // to read or write the data
   output wire        rxListWrite,              // Write enable
   output wire  [2:0] rxListSize,               // Access size
   output reg  [31:0] rxListWriteData,          // Written data
   input  wire        rxListError,              // Indicate a transfer error
   input  wire        rxListNextData,           // Data written, push the next data        
   input  wire        rxListTransComplete,      // transaction done

   ///////////////////////////////////////////////
   //$port_g DMA arbitrer interface
   ///////////////////////////////////////////////
   output wire        rxListProcReq,            // Receive List Processor request for grant
                                                // from arbiter
   input  wire        rxListProcGrant,          // Grant for the receive list processor
        
   ///////////////////////////////////////////////
   //$port_g macController interface
   ///////////////////////////////////////////////
   input wire         frameRxed_p,              // Frame is received completely.
                                                // Generated after the FCS check is done
                                                // and FCS is correct
   output reg         rxListProcCsIsIdle,       // rxListProc FSM is in IDLE state

   ///////////////////////////////////////////////
   //$port_g rxController interface
   ///////////////////////////////////////////////
   output reg         rxFrmDiscard,             // Discard the current rx frame :
                                                // no more descriptor and rxFlowCntrlEn == 1
   input  wire        rxFrmDiscardRet,          // Return signal on rxFrmDiscard from macCLk
                                                // domain to macPIClk domain
   output reg         rxDescAvailable,          // Indicate to rxController that Current frame
                                                // has all necessaries descripstor

   ///////////////////////////////////////////////
   //$port_g interrupt controller interface
   ///////////////////////////////////////////////
   output wire        rxBuffer1Trigger,         // Trigger for receive Buffer 1 interrupt
   output wire        rxBuffer2Trigger,         // Trigger for receive Buffer 2 interrupt
   output wire        rxTrig_p,                 // Indicate that the RX Descriptor Done has been
                                                // written
   output wire        tbProtTrigger,            // Indicate that a trigger frame has been
                                                // received and stored
   output wire        rxBusErr_p,               // Indicate a bus error during RX processing
        
   ///////////////////////////////////////////////
   //$port_g registers interface
   ///////////////////////////////////////////////
   input  wire [30:0] rxBuf1StartPtr,           // Receive buffer 1 start pointer
   input  wire [30:0] rxBuf1EndPtr,             // Receive buffer 1 end   pointer
   input  wire [31:0] rxBuf1RdPtr,              // Receive buffer 1 read  pointer
   input  wire [31:0] rxBuf1WrPtr,              // Receive buffer 1 write pointer
   output wire [31:0] rxBuf1WrPtrIn,            // Receive buffer 1 write pointer
   output wire        rxBuf1WrPtrInValid,       // Receive buffer 1 write pointer update
   input  wire [30:0] rxBuf2StartPtr,           // Receive buffer 2 start pointer
   input  wire [30:0] rxBuf2EndPtr,             // Receive buffer 2 end   pointer
   input  wire [31:0] rxBuf2RdPtr,              // Receive buffer 2 read  pointer
   input  wire [31:0] rxBuf2WrPtr,              // Receive buffer 2 write pointer
   output wire [31:0] rxBuf2WrPtrIn,            // Receive buffer 1 write pointer
   output wire        rxBuf2WrPtrInValid,       // Receive buffer 1 write pointer update
   input  wire  [7:0] rxBufRHDHeader,           // Receive buffer RHD header size
   input  wire  [7:0] rxBufRHDFooter,           // Receive buffer RHD footer size
   input  wire  [7:0] rxBufRPDHeader,           // Receive buffer RHD header size
   input  wire  [7:0] rxBufRPDFooter,           // Receive buffer RHD footer size
   input  wire  [7:0] partialUnwrapSize,        // Partial Unwrap Size
   input  wire        rxFlowCntrlEn,            // Enable Rx Flow Control
   output reg  [29:0] rxHeaderTFPtr,            // Receive Header Trigger Frame Pointer
// output reg         rxHeaderTFPtrValid,       // Receive Header Trigger Frame Pointer Valid
   output reg         rxHdrBusErr,              // Receive Header Bus Error
   output reg         rxPayBusErr,              // Receive Payload Bus Error
     
   ///////////////////////////////////////////////
   //$port_g Debug port
   ///////////////////////////////////////////////
   output wire [15:0] rxListProcDebug,          // Debug
   output wire [15:0] debugPortRxListA,         // Debug
   output reg   [4:0] rxListProcCs              // Current state register
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// rxListProc FSM states definition
//$fsm_sd rxListProc
parameter 
   IDLE                    = 5'd0,
   CHECK_FRAME_START       = 5'd1,
   CHECK_RHD_BUFFER        = 5'd2,
   WRITE_RHD_HEADER        = 5'd3,
   WRITE_RHD_FOOTER        = 5'd4,
   CHECK_HEADER_BUFFER     = 5'd5,
   WRITE_HEADER            = 5'd6,
   CHECK_HEADER_LEN        = 5'd7,
   WRITE_RHD               = 5'd8,
   CHECK_RPD_BUFFER        = 5'd9,
   WRITE_RPD_HEADER        = 5'd10,
   WRITE_RPD_FOOTER        = 5'd11,
   CHECK_PAYLOAD_BUFFER    = 5'd12,
   WRITE_PAYLOAD           = 5'd13,
   WRITE_RPD               = 5'd14,
   WRITE_RHD_TRAILER       = 5'd15,
   UPDATE_PREV_RHD         = 5'd16,
   DONE                    = 5'd17,
   FIFO_FLUSH              = 5'd18;

// busrt FSM states definition
//$fsm_sd burst
parameter 
   BURST_IDLE              = 3'd0,
   BURST_REQ               = 3'd1,
   BURST_START             = 3'd2,
   BURST_STARTED           = 3'd3,
   BURST_STOP              = 3'd4,
   BURST_STOPPED           = 3'd5,
   BURST_DONE              = 3'd6,
   BURST_ERROR             = 3'd7;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////

//    [4:0] rxListProcCs;              // rxListProc FSM Current State
reg   [4:0] rxListProcNs;              // rxListProc FSM Next State

reg   [2:0] burstCs;                   // burst FSM Current State
reg   [2:0] burstNs;                   // burst FSM Next State

wire        tagMPDULength;             // Indicates that the RX FIFO TAG read is
                                       // MPDU Length
wire        tagStartOfHeader;          // Indicates that the RX FIFO TAG read is
                                       // Start Of Header
wire        tagEndOfHeader;            // Indicates that the RX FIFO TAG read is
                                       // End Of Header
wire        tagHeaderLength;           // Indicates that the RX FIFO TAG read is
                                       // HEADER Length
wire        tagIntermediate;           // Indicates that the RX FIFO TAG read is
                                       // Intermediate Header or Payload
wire        tagEndOfPayload;           // Indicates that the RX FIFO TAG read is
                                       // End Of Payload
wire        tagTriggerSW;              // Indicates that the RX FIFO TAG read is
                                       // Trigger Frame With SW response
wire        tagSaveFrame;              // Indicates that the RX FIFO TAG read is
                                       // Frame with Save Pattern
wire        tagDiscardFrame;           // Indicates that the RX FIFO TAG read is
                                       // Frame with Discard Pattern
wire        tagMPDULength_p;           // Indicates that the RX FIFO TAG read is
                                       // MPDU Length (Pulse)
wire        tagStartOfHeader_p;        // Indicates that the RX FIFO TAG read is
                                       // Start Of Header (Pulse)
wire        tagEndOfHeader_p;          // Indicates that the RX FIFO TAG read is
                                       // End Of Header (Pulse)
wire        tagHeaderLength_p;         // Indicates that the RX FIFO TAG read is
                                       // HEADER Length (Pulse)
wire        tagIntermediate_p;         // Indicates that the RX FIFO TAG read is
                                       // Intermediate Header or Payload (Pulse)
wire        tagEndOfPayload_p;         // Indicates that the RX FIFO TAG read is
                                       // End Of Payload (Pulse)
wire        tagTriggerSW_p;            // Indicates that the RX FIFO TAG read is
                                       // Trigger Frame With SW response (Pulse)
wire        tagSaveFrame_p;            // Indicates that the RX FIFO TAG read is
                                       // Frame with Save Pattern (Pulse)
wire        tagDiscardFrame_p;         // Indicates that the RX FIFO TAG read is
                                       // Frame with Discard Pattern (Pulse)

reg  [15:0] mpduLength;                // MPDULength Counter (In byte)

reg         rxBufSel;                  // RX Buffer Selection
wire [28:0] rxBufStartQuadPtr;         // RX Buffer Start Quadlet Pointer
wire [28:0] rxBufEndQuadPtr;           // RX Buffer End   Quadlet Pointer
wire [29:0] rxBufRdQuadPtr;            // RX Buffer Read  Quadlet Pointer
reg  [29:0] rxBufWrQuadPtr;            // RX Buffer Write Quadlet Pointer
reg   [4:0] rxBufWrQuadPtrOffset;      // RX Buffer Write Quadlet Pointer Offset
wire        rxBufEmpty;                // RX Buffer Empty Flag
wire        rxBufOverflow;             // RX Buffer Overflow Flag
reg         rxBufOverflowCapt;         // RX Buffer Overflow Flag Captured
wire        rxBufError;                // RX Buffer Error Flag

reg  [28:0] currentRHDQuadPtr;         // Current  Header  Descriptor Quadlet
                                       // Pointer
reg  [28:0] currentRPDQuadPtr;         // Current  Payload Descriptor Quadlet
                                       // Pointer
wire [28:0] rxBufPrevRHDQuadPtr;       // Previous Header  Descriptor Quadlet
                                       // Pointer
reg  [28:0] rxBuf1PrevRHDQuadPtr;      // RX Buffer 1 Previous Header Descriptor
                                       // Quadlet Pointer
reg  [28:0] rxBuf2PrevRHDQuadPtr;      // RX Buffer 2 Previous Header Descriptor
                                       // Quadlet Pointer

reg  [28:0] dataStartQuadPtr;          // Data Start Pointer
reg  [30:0] dataEndPtr;                // Data End   Pointer

wire [28:0] rxBufRemainQ;              // Indicates remaing Quadlets before end
                                       // of the receive Buffer
reg  [13:0] blockSizeIn1;              // Indicates the size of the 1st part
                                       // of contiguous block.
reg  [13:0] blockSizeIn2;              // Indicates the size of the 2nd part
                                       // of contiguous block.
reg   [4:0] blockSizeIn3;              // Indicates the size of the 3rd part
                                       // of contiguous block.
reg  [13:0] blockSizeInQuadlet;        // Indicates the size of next contiguous 
                                       // block needed in the RX Buffer
wire        blockWrap;                 // Indicates that there is not enough 
                                       // place for the next contiguous block
                                       // before end of RX Buffer

reg         rxFifoRdValid_p;           // Pulse when data is read from RX FIFO

reg   [4:0] rxListOffset;              // Address offset to write the data

reg         rxListWriteEn;             // Write Enable
reg   [7:0] rxListWriteLen;            // Write Length
reg         rxListWriteFromFifo;       // Write Data Is From RX FIFO
reg   [7:0] burstCnt;                  // Number Of Quadlet written during burst
wire        burstPause;                // Indicates that burst should be pause
reg         burstStop;                 // Indicates that burst should be stop
reg         burstStop_p;               // Indicates that burst should be stop
wire        burstDone_p;               // Indicates that burst is finished
wire        burstError_p;              // Indicates a error during write burst

reg         frameRxed;                 // frameRxed_p pulse capture
reg         rxTriggerFrame;            // Indicates that received frame is a
                                       // Trigger frame with software response
reg         rxNDPFrame;                // Indicates that received frams is a NDP
                                       // frame
reg         splitPayload;              // Payload is split
reg   [1:0] payloadWrapMode;           // Indicates how the payload is stored in
                                       // the receive buffer
reg         frameWithoutHeader;        // Indicates that received frame doesn't
                                       // have header


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

// rxListProc FSM Current State Logic 
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
  if (~macPIRxClkHardRst_n)  // Asynchronous Reset
     rxListProcCs <= IDLE;
  else if (~macPIRxClkSoftRst_n) // Synchronous Reset
     rxListProcCs <= IDLE;
  else
     rxListProcCs <= rxListProcNs;
end


// rxListProc FSM Next State Logic
always @* 
begin  
   case(rxListProcCs)
   IDLE:
      //$fsm_s In IDLE state, the state machine waits until data are available
      //in the RX FIFO
      if ( rxFlowCntrlEn & ~rxFifoEmpty                   |
          ~rxFlowCntrlEn & ~rxFifoEmpty & ~rxFifoAlmEmpty |
          ~rxFlowCntrlEn & frameRxed_p                    |
          ~rxFlowCntrlEn & frameRxed                      )
         //$fsm_t When data are available in the RX FIFO,
         //the state machine goes to CHECK_FRAME_START state
         rxListProcNs = CHECK_FRAME_START;
      else
         //$fsm_t While data are no available in the RX FIFO,
         //the state machine stays in IDLE state
         rxListProcNs = IDLE;

   CHECK_FRAME_START:
      //$fsm_s In CHECK_FRAME_START state, the state waits until start of frame 
      //tag has been found into the RX FIFO.
      if (tagMPDULength_p)
         //$fsm_t When start of frame tag has been found into the RX FIFO,
         //the state machine goes to CHECK_RHD_BUFFER state
         rxListProcNs = CHECK_RHD_BUFFER;
      else if (rxFifoEmpty | tagDiscardFrame_p)
         //$fsm_t When start of frame tag is not found and RX FIFO is empty,
         //the state machine goes back to IDLE state
         rxListProcNs = IDLE;
      else
         //$fsm_t When start of frame tag has not been found into the RX FIFO,
         //the state machine stays in CHECK_FRAME_START state
         rxListProcNs = CHECK_FRAME_START;

   CHECK_RHD_BUFFER:
      //$fsm_s In CHECK_RHD_BUFFER state, the state check if there is enough 
      //place for RHD before end of receive buffer.

      //$fsm_t the state machine goes to WRITE_RHD_HEADER state
      rxListProcNs = WRITE_RHD_HEADER;

   WRITE_RHD_HEADER:
      //$fsm_s In WRITE_RHD_HEADER state, the state waits until RHD header
      //has been initialized.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p)
         //$fsm_t When RHD header initialization is done,
         //the state machine goes to WRITE_RHD_FOOTER state
         rxListProcNs = WRITE_RHD_FOOTER;
      else
         //$fsm_t When RHD header initialization is not finished,
         //the state machine stays in WRITE_RHD_HEADER state
         rxListProcNs = WRITE_RHD_HEADER;

   WRITE_RHD_FOOTER:
      //$fsm_s In WRITE_RHD_FOOTER state, the state waits until RHD footer
      //has been initialized.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p &  rxNDPFrame)
         //$fsm_t When RHD footer initialization is done and frame is NDP,
         //the state machine goes to WRITE_RHD state
         rxListProcNs = WRITE_RHD;
      else if (burstDone_p & ~rxNDPFrame &  frameWithoutHeader)
         //$fsm_t When RHD footer initialization is done and frame is not NDP
         //and frame without header,
         //the state machine goes to CHECK_RPD_BUFFER state
         rxListProcNs = CHECK_RPD_BUFFER;
      else if (burstDone_p & ~rxNDPFrame & ~frameWithoutHeader)
         //$fsm_t When RHD footer initialization is done and frame is not NDP
         //and frame with header,
         //the state machine goes to CHECK_HEADER_BUFFER state
         rxListProcNs = CHECK_HEADER_BUFFER;
      else
         //$fsm_t When RHD footer initialization is not finished,
         //the state machine stays in WRITE_RHD_FOOTER state
         rxListProcNs = WRITE_RHD_FOOTER;

   CHECK_HEADER_BUFFER:
      //$fsm_s In CHECK_HEADER_BUFFER state, the state check if there is
      //enough place for Header before end of receive buffer.

      //$fsm_t the state machine goes to WRITE_HEADER state
      rxListProcNs = WRITE_HEADER;

   WRITE_HEADER:
      //$fsm_s In WRITE_HEADER state, the state waits until HEADER has
      //been written.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p & tagDiscardFrame)
         //$fsm_t When Discard Frame tag is read,
         //the state machine goes to IDLE state
         rxListProcNs = IDLE;
      else if (burstDone_p)
         //$fsm_t When HEADER is completely written,
         //the state machine goes to CHECK_HEADER_LEN state
         rxListProcNs = CHECK_HEADER_LEN;
      else
         //$fsm_t When HEADER is not completely written,
         //the state machine stays in WRITE_HEADER state
         rxListProcNs = WRITE_HEADER;

   CHECK_HEADER_LEN:
      //$fsm_s In CHECK_HEADER_LEN state, the state read HEADER_LEN in
      //the received FIFO
      if (tagDiscardFrame_p | tagSaveFrame_p)
         //$fsm_t When read tag is DISCARD Frame or Save Frame,
         //the state machine goes to IDLE state
         rxListProcNs = IDLE;
      else if (rxFifoRdValid_p & ~tagHeaderLength)
         //$fsm_t When read tag is not HEADER LEN,
         //the state machine goes to FIFO_FLUSH state
         rxListProcNs = FIFO_FLUSH;
      else if (rxFifoRdValid_p & mpduLength==rxFifoRdData[15:0])
         //$fsm_t When HEADER LEN is read and frame without payload,
         //the state machine goes to WRITE_RHD state
         rxListProcNs = WRITE_RHD;
      else if (rxFifoRdValid_p)
         //$fsm_t When HEADER LEN is read and frame has a payload,
         //the state machine goes to CHECK_RPD_BUFFER state
         rxListProcNs = CHECK_RPD_BUFFER;
      else
         //$fsm_t When HEADER is not completely written,
         //the state machine stays in CHECK_HEADER_LEN state
         rxListProcNs = CHECK_HEADER_LEN;

   WRITE_RHD:
      //$fsm_s In WRITE_RHD state, the state waits until RHD has been
      //initialized.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p & mpduLength==16'd0)
         //$fsm_t When RHD initialization is done and frame without payload,
         //the state machine goes to WRITE_RHD_TRAILER state
         rxListProcNs = WRITE_RHD_TRAILER;
      else if (burstDone_p)
         //$fsm_t When RHD initialization is done and frame with payload,
         //the state machine goes to WRITE_RPD_FOOTER state
         rxListProcNs = WRITE_RPD_FOOTER;
      else
         //$fsm_t When RHD initialization is not finished,
         //the state machine stays in WRITE_RHD state
         rxListProcNs = WRITE_RHD;

   CHECK_RPD_BUFFER:
      //$fsm_s In CHECK_RPD_BUFFER state, the state check if there is enough 
      //place for RPD before end of receive buffer.

      //$fsm_t the state machine goes to WRITE_RPD_HEADER state
      rxListProcNs = WRITE_RPD_HEADER;

   WRITE_RPD_HEADER:
      //$fsm_s In WRITE_RPD_HEADER state, the state waits until RPD header
      //has been initialized.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p & ~splitPayload)
         //$fsm_t When first RPD header initialization is done,
         //the state machine goes to WRITE_RHD state
         rxListProcNs = WRITE_RHD;
      else if (burstDone_p &  splitPayload)
         //$fsm_t When second RPD header initialization is done,
         //the state machine goes to WRITE_RPD_FOOTER state
         rxListProcNs = WRITE_RPD_FOOTER;
      else
         //$fsm_t When RPD header initialization is not finished,
         //the state machine stays in WRITE_RPD_HEADER state
         rxListProcNs = WRITE_RPD_HEADER;

   WRITE_RPD_FOOTER:
      //$fsm_s In WRITE_RPD_FOOTER state, the state waits until RPD footer
      //has been initialized.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p)
         //$fsm_t When RPD footer initialization is done,
         //the state machine goes to CHECK_PAYLOAD_BUFFER state
         rxListProcNs = CHECK_PAYLOAD_BUFFER;
      else
         //$fsm_t When RPD footer initialization is not finished,
         //the state machine stays in WRITE_RPD_FOOTER state
         rxListProcNs = WRITE_RPD_FOOTER;

   CHECK_PAYLOAD_BUFFER:
      //$fsm_s In CHECK_PAYLOAD_BUFFER state, the state check if there is enough 
      //place for the payload before end of receive buffer.

      //$fsm_t the state machine goes to WRITE_PAYLOAD state
      rxListProcNs = WRITE_PAYLOAD;

   WRITE_PAYLOAD:
      //$fsm_s In WRITE_PAYLOAD state, the state waits until PAYLOAD has been
      //written or until wrap is needed.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p & tagDiscardFrame)
         //$fsm_t When discard frame tag is read,
         //the state machine goes to IDLE state
         rxListProcNs = IDLE;
      else if (burstDone_p)
         //$fsm_t When PAYLOAD is completely written,
         //the state machine goes to WRITE_RPD state
         rxListProcNs = WRITE_RPD;
      else
         //$fsm_t When PAYLOAD is not finished,
         //the state machine stays in WRITE_PAYLOAD state
         rxListProcNs = WRITE_PAYLOAD;

   WRITE_RPD:
      //$fsm_s In WRITE_RPD state, the state waits until RPD has been
      //initialized.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p & splitPayload)
         //$fsm_t When RPD initialization is done and PAYLOAD is split,
         //the state machine stays in CHECK_RPD_BUFFER state
         rxListProcNs = CHECK_RPD_BUFFER;
      else if (burstDone_p)
         //$fsm_t When RPD initialization is done,
         //the state machine goes to WRITE_RHD_TRAILER state
         rxListProcNs = WRITE_RHD_TRAILER;
      else
         //$fsm_t When RPD initialization is not finished,
         //the state machine stays in WRITE_RPD state
         rxListProcNs = WRITE_RPD;

   WRITE_RHD_TRAILER:
      //$fsm_s In WRITE_RHD_TRAILER state, the state waits until RHD trailer has
      //been written into the RX buffer.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to FIFO_FLUSH state
         rxListProcNs = FIFO_FLUSH;
      else if (burstDone_p & tagDiscardFrame)
         //$fsm_t When RHD trailer is done and discard tag received,
         //the state machine goes to IDLE state
         rxListProcNs = IDLE;
      else if (burstDone_p & ~rxBufEmpty)
         //$fsm_t When RHD trailer is done and previous RHD should be updated,
         //the state machine goes to UPDATE_PREV_RHD state
         rxListProcNs = UPDATE_PREV_RHD;
      else if (burstDone_p)
         //$fsm_t When RHD trailer is done and previous RHD should not be updated,
         //the state machine goes to DONE state
         rxListProcNs = DONE;
      else
         //$fsm_t When RHD trailer is not finished,
         //the state machine stays in WRITE_RHD_TRAILER state
         rxListProcNs = WRITE_RHD_TRAILER;

   UPDATE_PREV_RHD:
      //$fsm_s In UPDATE_PREV_RHD state, the state waits until previous RHD has
      //been updated.
      if (burstError_p)
         //$fsm_t When data can't be written,
         //the state machine goes to ILDE state
         rxListProcNs = IDLE;
      else if (burstDone_p)
         //$fsm_t When previous RHD is updated,
         //the state machine goes to DONE state
         rxListProcNs = DONE;
      else
         //$fsm_t When previous RHD is not updated,
         //the state machine stays in UPDATE_PREV_RHD state
         rxListProcNs = UPDATE_PREV_RHD;

   DONE:
      //$fsm_s In DONE state, the receive interrupt is generated and the RX
      //buffer write pointer is updated.

      //$fsm_t the state machine goes to IDLE state
      rxListProcNs = IDLE;

   FIFO_FLUSH:
      //$fsm_s In FIFO_FLUSH state, the FIFO is read until end of discarded frame
      if (tagDiscardFrame | tagSaveFrame)
         //$fsm_t When discarded frame is read,
         //the state machine goes to IDLE state
         rxListProcNs = IDLE;
      else
         //$fsm_t While discarded frame is not read,
         //the state machine stays in FIFO_FLUSH state
         rxListProcNs = FIFO_FLUSH;

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      rxListProcNs = IDLE; 
   // pragma coverage block = on
   endcase
end


// rxListProc FSM IDLE Flag
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxListProcCsIsIdle <= 1'b1;
   else if (~macPIRxClkSoftRst_n)
      rxListProcCsIsIdle <= 1'b1;
   else if (rxListProcCs==IDLE)
      rxListProcCsIsIdle <= 1'b1;
   else
      rxListProcCsIsIdle <= 1'b0;
end


// RX FIFO TAG decoding
assign tagMPDULength      = rxTagFifoRdData==4'b1010;
assign tagStartOfHeader   = rxTagFifoRdData==4'b0001;
assign tagEndOfHeader     = rxTagFifoRdData==4'b0010;
assign tagHeaderLength    = rxTagFifoRdData==4'b0011;
assign tagIntermediate    = rxTagFifoRdData==4'b1100;
assign tagEndOfPayload    = rxTagFifoRdData==4'b0101;
assign tagTriggerSW       = rxTagFifoRdData==4'b1110;
assign tagSaveFrame       = rxTagFifoRdData==4'b0000;
assign tagDiscardFrame    = rxTagFifoRdData==4'b1111;

assign tagMPDULength_p    = rxFifoRdValid_p & tagMPDULength;
assign tagStartOfHeader_p = rxFifoRdValid_p & tagStartOfHeader;
assign tagEndOfHeader_p   = rxFifoRdValid_p & tagEndOfHeader;
assign tagHeaderLength_p  = rxFifoRdValid_p & tagHeaderLength;
assign tagIntermediate_p  = rxFifoRdValid_p & tagIntermediate;
assign tagEndOfPayload_p  = rxFifoRdValid_p & tagEndOfPayload;
assign tagTriggerSW_p     = rxFifoRdValid_p & tagTriggerSW;
assign tagSaveFrame_p     = rxFifoRdValid_p & tagSaveFrame;
assign tagDiscardFrame_p  = rxFifoRdValid_p & tagDiscardFrame;


// MPDU Length counter
//   Down counter initialized when MPDU_LENGTH tag is read from the RX FIFO
//   and decremented when Payload is read from the RX FIFO
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      mpduLength <= 16'h0;
   else if (~macPIRxClkSoftRst_n)
      mpduLength <= 16'h0;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p)
      mpduLength <= rxFifoRdData[15:0];
   else if (rxListProcCs==CHECK_HEADER_LEN & tagHeaderLength_p)
      mpduLength <= mpduLength-{8'h0,rxFifoRdData[7:0]};
   else if (rxListProcCs==WRITE_PAYLOAD & rxFifoRd & mpduLength[15:0]>16'h4)
      mpduLength <= mpduLength-16'd4;
end


// RX buffer selection
//   initialized when MPDU_LENGTH tag is read from the RX FIFO
//   0: RX buffer 1 selected
//   1: RX buffer 2 selected
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxBufSel <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      rxBufSel <= 1'b0;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p)
      rxBufSel <= rxFifoRdData[16];
end


// Frame without Header, should be stored in RPD buffer only
//   initialized when MPDU_LENGTH tag is read from the RX FIFO
//   0: Header is present in the receive FIFO
//   1: Header is not present in the receive FIFO
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      frameWithoutHeader <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      frameWithoutHeader <= 1'b0;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p)
      frameWithoutHeader <= rxFifoRdData[17];
end


// Payload Wrap Mode
//   initialized when MPDU_LENGTH tag is read from the RX FIFO
//   0: Wrap Mode         (Payload can be splitted)
//   1: No Wrap Mode      (Payload can't be splitted)
//   2: Partial Wrap Mode (Payload can be splitted after partialUnwrapSize word)
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      payloadWrapMode <= 2'b0;
   else if (~macPIRxClkSoftRst_n)
      payloadWrapMode <= 2'b0;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p)
      payloadWrapMode <= rxFifoRdData[19:18];
end


// Receive buffer registers selection
assign rxBufStartQuadPtr   = rxBufSel ? rxBuf2StartPtr[30:2] : rxBuf1StartPtr[30:2];
assign rxBufEndQuadPtr     = rxBufSel ? rxBuf2EndPtr[30:2]   : rxBuf1EndPtr[30:2];
assign rxBufRdQuadPtr      = rxBufSel ? rxBuf2RdPtr[31:2]    : rxBuf1RdPtr[31:2];
assign rxBufPrevRHDQuadPtr = rxBufSel ? rxBuf2PrevRHDQuadPtr : rxBuf1PrevRHDQuadPtr;

assign rxBuf1WrPtrIn      = {rxBufWrQuadPtr,2'h0};
assign rxBuf2WrPtrIn      = {rxBufWrQuadPtr,2'h0};
assign rxBuf1WrPtrInValid = ~rxBufSel & rxListProcCs==DONE;
assign rxBuf2WrPtrInValid =  rxBufSel & rxListProcCs==DONE;


// Receive Buffer write pointer
//   initialized when MPDU_LENGTH tag is read from the RX FIFO (Buffer selection)
//   Wrapped when not enough place for next contiguous block
//   Incremented when data is written in RX buffer
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxBufWrQuadPtr <= 30'h0;
   else if (~macPIRxClkSoftRst_n)
      rxBufWrQuadPtr <= 30'h0;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p & ~rxFifoRdData[16])
      rxBufWrQuadPtr <= rxBuf1WrPtr[31:2];
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p &  rxFifoRdData[16])
      rxBufWrQuadPtr <= rxBuf2WrPtr[31:2];
   else if (rxListProcCs==CHECK_RHD_BUFFER     & blockWrap |
            rxListProcCs==CHECK_RPD_BUFFER     & blockWrap |
            rxListProcCs==CHECK_HEADER_BUFFER  & blockWrap |
            rxListProcCs==CHECK_PAYLOAD_BUFFER & blockWrap)
      rxBufWrQuadPtr <= {~rxBufWrQuadPtr[29],rxBufStartQuadPtr};
   else if (rxListProcCs==WRITE_RHD_HEADER & burstDone_p                  |
            rxListProcCs==WRITE_RPD_HEADER & burstDone_p                  |
            rxListProcCs==WRITE_RHD_HEADER & rxListWrite & rxListNextData |
            rxListProcCs==WRITE_RHD_FOOTER & rxListWrite & rxListNextData |
            rxListProcCs==WRITE_RPD_HEADER & rxListWrite & rxListNextData |
            rxListProcCs==WRITE_RPD_FOOTER & rxListWrite & rxListNextData |
            rxListProcCs==WRITE_HEADER     & rxListWrite & rxListNextData |
            rxListProcCs==WRITE_PAYLOAD    & rxListWrite & rxListNextData )
   begin
      if (rxBufRemainQ>={24'h0,rxBufWrQuadPtrOffset})
         rxBufWrQuadPtr <= rxBufWrQuadPtr+{25'h0,rxBufWrQuadPtrOffset};
      else
         rxBufWrQuadPtr <= {~rxBufWrQuadPtr[29],rxBufStartQuadPtr};
   end
end


// Receive Buffer write pointer increment offset
always @*
begin
   if (rxListProcCs==WRITE_RHD_HEADER & burstDone_p)
      rxBufWrQuadPtrOffset = 5'd17; // RHD Size (17 quadlets)
   else if (rxListProcCs==WRITE_RPD_HEADER & burstDone_p)
      rxBufWrQuadPtrOffset = 5'd5;  // RPD Size (5 quadlets)
   else
      rxBufWrQuadPtrOffset = 5'd1;  // Next Address
end


// Size of the next contiguous block
//    Check if a wrap is needed, when not enought space before end of buffer.
always @* 
begin
   case(rxListProcCs)
   CHECK_RHD_BUFFER:
   begin
      // RHD = 17 quadlets
      blockSizeIn1 = {6'd0,rxBufRHDHeader};
      blockSizeIn2 = {6'd0,rxBufRHDFooter};
      blockSizeIn3 = 5'd16;
   end
   CHECK_RPD_BUFFER:
   begin
      // RPD = 5 quadlets
      blockSizeIn1 = {6'd0,rxBufRPDHeader};
      blockSizeIn2 = {6'd0,rxBufRPDFooter};
      blockSizeIn3 = 5'd4;
   end
   CHECK_HEADER_BUFFER:
   begin
      // Maximum Header length = 15 quadlets
      // 36 bytes for MAC Header
      // 18 bytes for Security Header
      //  6 extra bytes
      blockSizeIn1 = 14'd0;
      blockSizeIn2 = 14'd0;
      blockSizeIn3 = 5'd14;
   end
   CHECK_PAYLOAD_BUFFER:
   begin
      case (payloadWrapMode)
      2'd0:// Wrap Mode (block size = 1 quadlet)
      begin
         blockSizeIn1 = 14'd0;
         blockSizeIn2 = 14'd0;
         blockSizeIn3 = 5'd0;
      end
      2'd1:// No Wrap Mode (block size = payload length)
      begin
         blockSizeIn1 = mpduLength[15:2];
         blockSizeIn2 = {14{~|mpduLength[1:0]}};  //-1 when mpduLength[1:0]==0
         blockSizeIn3 = 5'd0;
      end
      default:// Partial Wrap Mode (block size = partialUnwrapSize)
      begin
         blockSizeIn1 = {6'd0,partialUnwrapSize};
         blockSizeIn2 = {14{|partialUnwrapSize}}; //-1 when partialUnwrapSize!=0
         blockSizeIn3 = 5'd0;
      end
      endcase
   end
   default:
   begin
      blockSizeIn1 = 14'd0;
      blockSizeIn2 = 14'd0;
      blockSizeIn3 = 5'd0;
   end
   endcase
   blockSizeInQuadlet = {blockSizeIn1}+
                        {blockSizeIn2}+
                        {9'd0,blockSizeIn3};
end


// Check RX Buffer for block that should not be split.
assign blockWrap    = rxBufRemainQ<{15'h0,blockSizeInQuadlet};


// Current Header Descriptor Pointer
//    Capture RHD pointer when RHD Header is writed.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      currentRHDQuadPtr <= 29'h0;
   else if (~macPIRxClkSoftRst_n)
      currentRHDQuadPtr <= 29'h0;
   else if (rxListProcCs==WRITE_RHD_HEADER & burstDone_p)
      currentRHDQuadPtr <= rxBufWrQuadPtr[28:0];
end


// Current Payload Descriptor Pointer
//    Initialized with '0' on start (in case of frame without payload)
//    RPD pointer Captured when RPD Header is writed.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      currentRPDQuadPtr <= 29'h0;
   else if (~macPIRxClkSoftRst_n)
      currentRPDQuadPtr <= 29'h0;
   else if (rxListProcCs==CHECK_FRAME_START)
      currentRPDQuadPtr <= 29'h0;
   else if (rxListProcCs==WRITE_RPD_HEADER & burstDone_p)
      currentRPDQuadPtr <= rxBufWrQuadPtr[28:0];
end


// Previous Header Descriptor Pointer of RX Buffer 1
//    Capture RHD pointer when frame is saved into RX Buffer 1
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxBuf1PrevRHDQuadPtr <= 29'h0;
   else if (~macPIRxClkSoftRst_n)
      rxBuf1PrevRHDQuadPtr <= 29'h0;
   else if (~rxBufSel & rxListProcCs==DONE)
      rxBuf1PrevRHDQuadPtr <= currentRHDQuadPtr;
end


// Previous Header Descriptor Pointer of RX Buffer 2
//    Capture RHD pointer when frame is saved into RX Buffer 2
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxBuf2PrevRHDQuadPtr <= 29'h0;
   else if (~macPIRxClkSoftRst_n)
      rxBuf2PrevRHDQuadPtr <= 29'h0;
   else if (rxBufSel & rxListProcCs==DONE)
      rxBuf2PrevRHDQuadPtr <= currentRHDQuadPtr;
end


// Start Pointer
//    Initialized with '0' on start (in case of frame without header/payload)
//    Header or Payload Start Address is captured when burst is started.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      dataStartQuadPtr <= 29'h0;
   else if (~macPIRxClkSoftRst_n)
      dataStartQuadPtr <= 29'h0;
   else if (rxListProcCs==CHECK_FRAME_START)
      dataStartQuadPtr <= 29'h0;
   else if (rxListProcCs==WRITE_HEADER  & burstCs==IDLE)
      dataStartQuadPtr <= rxBufWrQuadPtr[28:0];
   else if (rxListProcCs==WRITE_PAYLOAD & burstCs==IDLE)
      dataStartQuadPtr <= rxBufWrQuadPtr[28:0];
end


// End Pointer
//    Initialized with '0' on start (in case of frame without header/payload)
//    Header or Payload End Address is captured when burst is done.
//    Header End Address Byte offset is updated when Header length is
//    read from receive FIFO
//    Payload End Address Byte offset is updated when splitPayload flag is ready
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      dataEndPtr <= 31'h0;
   else if (~macPIRxClkSoftRst_n)
      dataEndPtr <= 31'h0;
   else if (rxListProcCs==CHECK_FRAME_START)
      dataEndPtr <= 31'h0;
   else if (rxListProcCs==WRITE_HEADER & burstStop_p)
   begin
      dataEndPtr[30:2] <= rxBufWrQuadPtr[28:0];
      dataEndPtr[1:0]  <= 2'b11;
   end
   else if (rxListProcCs==CHECK_HEADER_LEN & tagHeaderLength_p)
   begin
      dataEndPtr[1:0]  <= rxFifoRdData[1:0]+2'b11;
   end
   else if (rxListProcCs==WRITE_PAYLOAD & burstStop_p)
   begin
      dataEndPtr[30:2] <= rxBufWrQuadPtr[28:0];
      dataEndPtr[1:0]  <= 2'b11;
   end
   else if (rxListProcCs==WRITE_PAYLOAD & burstDone_p & ~splitPayload)
   begin
      dataEndPtr[1:0]  <= mpduLength[1:0]+2'b11;
   end
end


// RX Buffer remaining quadlet before end of buffer
assign rxBufRemainQ = rxBufEndQuadPtr-rxBufWrQuadPtr[28:0];


// RX Buffer empty
//    Indicates if the selected buffer is empty
assign rxBufEmpty = ~rxBufSel & rxBuf1RdPtr==rxBuf1WrPtr |
                     rxBufSel & rxBuf2RdPtr==rxBuf2WrPtr ;


// RX FIFO Read
//    If data is not writed into host interface, the FIFO is read when not empty
//    until required TAG is found.
//    If data is writed into host interface, the FIFO is read when not empty and
//    host interface is ready to receive new data.
always @*
begin
   case (rxListProcCs)
   CHECK_FRAME_START:
      rxFifoRd = ~rxFifoEmpty & ~tagMPDULength_p;
   CHECK_HEADER_LEN:
      rxFifoRd = ~rxFifoEmpty & ~tagHeaderLength_p;
   FIFO_FLUSH:
      rxFifoRd = ~rxFifoEmpty & ~tagDiscardFrame & ~tagSaveFrame;
   WRITE_HEADER,
   WRITE_PAYLOAD,
   WRITE_RHD_TRAILER:
      rxFifoRd = burstCs==BURST_START   & ~rxFifoEmpty |
                 burstCs==BURST_STARTED & ~rxFifoEmpty & rxListNextData & ~burstStop_p;
   default:
      rxFifoRd = 1'b0;
   endcase
end


// RX FIFO Read valid pulse
//    A Pulse is generated when data has been read from RX FIFO.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxFifoRdValid_p <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      rxFifoRdValid_p <= 1'b0;
   else
      rxFifoRdValid_p <= rxFifoRd;
end


// Host interface Write Control
always @*
begin
   case (rxListProcCs)
   WRITE_RHD_HEADER:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = rxBufRHDHeader;
      rxListWriteFromFifo = 1'b0;
      rxListWriteData     = 32'h0;
   end
   WRITE_RHD_FOOTER:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = rxBufRHDFooter;
      rxListWriteFromFifo = 1'b0;
      rxListWriteData     = 32'h0;
   end
   WRITE_RPD_HEADER:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = rxBufRPDHeader;
      rxListWriteFromFifo = 1'b0;
      rxListWriteData     = 32'h0;
   end
   WRITE_RPD_FOOTER:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = rxBufRPDFooter;
      rxListWriteFromFifo = 1'b0;
      rxListWriteData     = 32'h0;
   end
   WRITE_RHD:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = 8'd7; // Only the 7 first word are written
      rxListWriteFromFifo = 1'b0;
      case (burstCnt)
      8'd7:    // Unique Pattern
               rxListWriteData = 32'hBAADF00D;
      8'd6:    // Next Header Descriptor Pointer
               rxListWriteData = 32'h00000000;
      8'd5:    // First Payload Buffer Descriptor Pointer
               rxListWriteData = {1'b0,currentRPDQuadPtr,2'h0};
      8'd4:    // SW Descriptor Pointer
               rxListWriteData = 32'h00000000;
      8'd3:    // Data Start Pointer
               rxListWriteData = {1'b0,dataStartQuadPtr,2'h0};
      8'd2:    // Data End Pointer
               rxListWriteData = {1'b0,dataEndPtr};
      default: // Header Control Information
               rxListWriteData = 32'h00000000;
      endcase
   end
   WRITE_RPD:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = 8'd5;
      rxListWriteFromFifo = 1'b0;
      case (burstCnt)
      8'd5:    // Unique Pattern
               rxListWriteData = 32'hC0DEDBAD;
      8'd4:    // Next Payload Buffer Descriptor Pointer
               rxListWriteData = ~splitPayload ? 32'h00000000 :
                                                 {1'b0,rxBufStartQuadPtr,2'h0} +
                                                 {22'h0,rxBufRPDHeader,2'b0};
      8'd3:    // Data Start Pointer
               rxListWriteData = {1'b0,dataStartQuadPtr,2'h0};
      8'd2:    // Data End Pointer
               rxListWriteData = {1'b0,dataEndPtr};
      default: // Buffer Status Information
               rxListWriteData = {30'h0,1'b1,~splitPayload};
      endcase
   end
   WRITE_HEADER:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = 8'hFF; // Undefined, should wait specific TAG
      rxListWriteFromFifo = 1'b1;
      rxListWriteData     = rxFifoRdData;
   end
   WRITE_PAYLOAD:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = 8'hFF; // Undefined, should wait specific TAG
      rxListWriteFromFifo = 1'b1;
      rxListWriteData     = rxFifoRdData;
   end
   WRITE_RHD_TRAILER:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = 8'd10;
      rxListWriteFromFifo = 1'b1;
      rxListWriteData     = rxFifoRdData;
   end
   UPDATE_PREV_RHD:
   begin
      rxListWriteEn       = 1'b1;
      rxListWriteLen      = 8'd1;
      rxListWriteFromFifo = 1'b0;
      rxListWriteData     = {1'b0,currentRHDQuadPtr,2'h0};
   end
   default:
   begin
      rxListWriteEn       = 1'b0;
      rxListWriteLen      = 8'd0;
      rxListWriteFromFifo = 1'b0;
      rxListWriteData     = 32'h0;
   end
   endcase
end


// burst FSM Current State Logic
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      burstCs <= BURST_IDLE;
   else if (~macPIRxClkSoftRst_n)
      burstCs <= BURST_IDLE;
   else
      burstCs <= burstNs;
end


// burst FSM Next State Logic
always @* 
begin  
   case(burstCs)
   BURST_IDLE:
      //$fsm_s In BURST_IDLE state, the state machine waits until write is
      //requested
      if (rxListWriteEn & rxListWriteLen==8'd0)
         //$fsm_t When write is requested with zero length,
         //the state machine goes to BURST_DONE state
         burstNs = BURST_DONE;
      else if (rxListWriteEn & ~(rxListWriteFromFifo & rxFifoEmpty))
         //$fsm_t When write is requested with length not null and data are
         //available,
         //the state machine goes to BURST_REQ state
         burstNs = BURST_REQ;
      else
         //$fsm_t While data are no available in the RX FIFO,
         //the state machine stays in BURST_IDLE state
         burstNs = BURST_IDLE;

   BURST_REQ:
      //$fsm_s In BURST_REQ state, the state machine waits until access to the
      // bus is granted.
      if (rxListProcGrant)
         //$fsm_t When access to the bus is granted,
         //the state machine goes to BURST_START state
         burstNs = BURST_START;
      else
         //$fsm_t While access to the bus is not granted,
         //the state machine stays in BURST_REQ state
         burstNs = BURST_REQ;

   BURST_START:
      //$fsm_s In BURST_START state, the burst is started.
      if (rxBufOverflow | rxListError | rxBufError)
         //$fsm_t If buffer overflow is detected,
         //the state machine goes to BURST_ERROR state
         burstNs = BURST_ERROR;
      else
         //$fsm_t If buffer is ready,
         //the state machine goes to BURST_STARTED state
         burstNs = BURST_STARTED;

   BURST_STARTED:
      //$fsm_s In BURST_STARTED state, the state machine waits until all data
      //has been written
      if (rxListError)
         //$fsm_t If buffer access error or FIFO access error,
         //the state machine goes to BURST_ERROR state
         burstNs = BURST_ERROR;
      else if (burstStop_p | burstPause | rxBufOverflow)
         //$fsm_t When all data has been written or no more data available or
         //buffer overflow,
         //the state machine goes to BURST_STOP state
         burstNs = BURST_STOP;
      else
         //$fsm_t When more data has to be written,
         //the state machine stays in BURST_STARTED state
         burstNs = BURST_STARTED;

   BURST_STOP:
      //$fsm_s In BURST_STOP state, the state machine waits until burst is 
      //stopped.
      if (rxBufOverflowCapt | rxListError)
         //$fsm_t If buffer access error,
         //the state machine goes to BURST_ERROR state
         burstNs = BURST_ERROR;
      else if (rxListTransComplete & burstStop)
         //$fsm_t When burst is done and no more data to write,
         //the state machine goes to BURST_DONE state
         burstNs = BURST_DONE;
      else if (rxListTransComplete)
         //$fsm_t When burst is done and more data to write,
         //the state machine goes to BURST_STOPPED state
         burstNs = BURST_STOPPED;
      else
         //$fsm_t While burst is not done.
         //the state machine stays in BURST_STOP state
         burstNs = BURST_STOP;

   BURST_STOPPED:
      //$fsm_s In BURST_STOPPED state, the state machine waits until FIFO is
      //ready.
      if (~rxFifoEmpty)
         //$fsm_t When FIFO is ready,
         //the state machine goes to BURST_START state
         burstNs = BURST_START;
      else
         //$fsm_t While FIFO is empty,
         //the state machine stays in BURST_STOPPED state
         burstNs = BURST_STOPPED;

   BURST_DONE:
      //$fsm_s In BURST_DONE state, the burst is finished.

      //$fsm_t the state machine goes to BURST_IDLE state
      burstNs = BURST_IDLE;

   BURST_ERROR:
      //$fsm_s In BURST_ERROR state, the frame is discarded

      //$fsm_t the state machine goes to BURST_IDLE state
      burstNs = BURST_IDLE;

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      burstNs = BURST_IDLE; 
   // pragma coverage block = on
   endcase
end


// Burst Counter
//   Initialized with burst width, then decremented until '0'.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      burstCnt <= 8'd0;
   else if (~macPIRxClkSoftRst_n)
      burstCnt <= 8'd0;
   else if (burstCs==BURST_REQ)
      burstCnt <= rxListWriteLen;
   else if (rxListWrite & rxListNextData & burstCnt>8'd0)
      burstCnt <= burstCnt-8'd1;
end


// Burst Starting address
always @*
begin
   case (rxListProcCs)
   WRITE_RHD:         rxListStartAddress = {1'b0,currentRHDQuadPtr  ,2'h0};
   WRITE_RPD:         rxListStartAddress = {1'b0,currentRPDQuadPtr  ,2'h0};
   WRITE_RHD_TRAILER: rxListStartAddress = {1'b0,currentRHDQuadPtr  ,2'h0}+
                                           {25'h0,rxListOffset,2'h0};
   UPDATE_PREV_RHD:   rxListStartAddress = {1'b0,rxBufPrevRHDQuadPtr,2'b0}+
                                           {25'h0,rxListOffset,2'h0};
   default:           rxListStartAddress = {1'b0,rxBufWrQuadPtr[28:0],2'h0};
   endcase
end


// Burst Address offset in quadlet
//    In case of split burst during RHD Trailer, the address offset should be
//    saved.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxListOffset <= 5'd0;
   else if (~macPIRxClkSoftRst_n)
      rxListOffset <= 5'd0;
   else if (burstCs==BURST_REQ & rxListProcCs==WRITE_RHD_TRAILER)
      rxListOffset <= 5'd7;
   else if (burstCs==BURST_REQ & rxListProcCs==UPDATE_PREV_RHD)
      rxListOffset <= 5'd1;
   else if (rxListWrite & rxListNextData)
      rxListOffset <= rxListOffset + 5'd1;
end


// Host interface control
assign rxListProcReq      = burstCs==BURST_REQ     |
                            burstCs==BURST_START   |
                            burstCs==BURST_STARTED |
                            burstCs==BURST_STOP    |
                            burstCs==BURST_STOPPED ;
assign rxListWrite        = burstCs==BURST_START   & ~rxBufOverflow & ~rxBufError | 
                            burstCs==BURST_STARTED & ~rxBufOverflow & ~rxBufError;
assign rxListSize         = 3'b010;


// Burst pause
//   Burst is paused when Fifo is empty & new data is requested.
assign burstPause = rxListWriteFromFifo & rxFifoEmpty &
                    rxListWrite         & rxListNextData;


// Burst stop
//    Burst is stopped when required TAG is read from receive FIFO or all data
//    written to HOST Interface.
//    If write is payload, the burst is stopped when end of buffer is reached.
always @*
begin
   case (rxListProcCs)
   WRITE_HEADER:
      burstStop_p = rxListWrite & rxListNextData & tagDiscardFrame |
                    rxListWrite & rxListNextData & tagEndOfHeader  ;
   WRITE_PAYLOAD:
      burstStop_p = rxListWrite & rxListNextData & tagDiscardFrame |
                    rxListWrite & rxListNextData & tagEndOfPayload |
                    rxListWrite & rxListNextData & rxBufWrQuadPtr[28:0]==rxBufEndQuadPtr;
   default:
      burstStop_p = rxListWrite & rxListNextData & tagDiscardFrame |
                    rxListWrite & rxListNextData & burstCnt==8'd1;
   endcase
end


// Burst stop pulse capture
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      burstStop  <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      burstStop  <= 1'b0;
   else if (burstCs==BURST_START)
      burstStop  <= 1'b0;
   else if (burstCs==BURST_STARTED & burstStop_p)
      burstStop  <= 1'b1;
end


// Burst Flag for rxLixtProc FSM
assign burstDone_p  = burstCs==BURST_DONE;
assign burstError_p = burstCs==BURST_ERROR;


// Burst Overflow detection
//   No overflow possible in case of RHD, RPD access
assign rxBufOverflow   =  rxListProcCs!=WRITE_RHD                  &
                          rxListProcCs!=WRITE_RPD                  &
                          rxListProcCs!=WRITE_RHD_TRAILER          &
                          rxListProcCs!=UPDATE_PREV_RHD            &
                          rxBufRdQuadPtr[29]  !=rxBufWrQuadPtr[29] & 
                          rxBufRdQuadPtr[28:0]<=rxBufWrQuadPtr[28:0];


// Burst Overflow Capture
//   Used to know if burst has been stopped due to buffer overflow.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxBufOverflowCapt <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      rxBufOverflowCapt <= 1'b0;
   else if (burstCs==BURST_STARTED & rxBufOverflow)
      rxBufOverflowCapt <= 1'b1;
   else if (burstCs==BURST_ERROR)
      rxBufOverflowCapt <= 1'b0;
end


// Burst Error Detection
//   Write pointer is not between Start & End pointer
assign rxBufError = burstCs!=BURST_IDLE &
                    (rxBufWrQuadPtr[28:0]<rxBufStartQuadPtr[28:0] |
                     rxBufWrQuadPtr[28:0]>rxBufEndQuadPtr[28:0]   );


// Frame Discard
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxFrmDiscard <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      rxFrmDiscard <= 1'b0;
   else if (burstError_p & rxFlowCntrlEn)
      rxFrmDiscard <= 1'b1;
   else if (rxFrmDiscardRet & tagDiscardFrame_p |
            rxFrmDiscardRet & tagSaveFrame_p    )
      rxFrmDiscard <= 1'b0;
end


//Trigger indicating frame has been received and transfered to SRAM in buffer 1
assign rxBuffer1Trigger  = rxListProcCs==DONE & ~rxBufSel;


//Trigger indicating frame has been received and transfered to SRAM in buffer 2
assign rxBuffer2Trigger  = rxListProcCs==DONE &  rxBufSel;


// Indicate that a trigger frame has been received and stored
assign tbProtTrigger     = rxListProcCs==DONE &  rxTriggerFrame;


// Indicate that the RX Descriptor Done has been written
assign rxTrig_p          = rxListProcCs==DONE;


//Logic for frameRxed
//Latches the frameRxed_p. Used in FSM for checking if the
//frame has been received. Between every received frame the 
//FSM will move to IDLE state hence resetting this signal in
//the IDLE state to latch the next frame's received pulse 
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      frameRxed <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      frameRxed <= 1'b0;
   else if (rxListProcCs==CHECK_FRAME_START)
      frameRxed <= 1'b0;
   else if (frameRxed_p)
      frameRxed <= 1'b1;
end


// Logic for rx Trigger Frame packet
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxTriggerFrame <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      rxTriggerFrame <= 1'b0;
   else if (rxListProcCs==CHECK_FRAME_START)
      rxTriggerFrame <= 1'b0;
   else if (rxListProcCs==WRITE_RHD_TRAILER & tagTriggerSW_p)
      rxTriggerFrame <= 1'b1;
end


// Store the RX header address of the latest Trigger Frame received
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
  if (~macPIRxClkHardRst_n)
    rxHeaderTFPtr <= 30'b0;
  else if (~macPIRxClkSoftRst_n)
    rxHeaderTFPtr <= 30'b0;
  else if (rxListProcCs==DONE & rxTriggerFrame)
    rxHeaderTFPtr <= {1'b0,currentRHDQuadPtr};
end


// Logic for rx NDP Frame packet
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      rxNDPFrame <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      rxNDPFrame <= 1'b0;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p & rxFifoRdData[15:0]==16'd0)
      rxNDPFrame <= 1'b1;
   else if (rxListProcCs==CHECK_FRAME_START & tagMPDULength_p)
      rxNDPFrame <= 1'b0;
end


// Indicates that the payload has been split
//    Initialized with '0' on start (in case of frame discard just after split)
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
   if (~macPIRxClkHardRst_n)
      splitPayload <= 1'b0;
   else if (~macPIRxClkSoftRst_n)
      splitPayload <= 1'b0;
   else if (rxListProcCs==WRITE_PAYLOAD & ~tagEndOfPayload &
            rxListWrite & rxListNextData &
            rxBufWrQuadPtr[28:0]==rxBufEndQuadPtr)
      splitPayload <= 1'b1;
   else if (rxListProcCs==CHECK_FRAME_START    |
            rxListProcCs==CHECK_PAYLOAD_BUFFER )
      splitPayload <= 1'b0;
end


// Indicates that a bus error is detected during descriptor processing
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
  if (~macPIRxClkHardRst_n)
    rxHdrBusErr <= 1'b0;
  else if (~macPIRxClkSoftRst_n)
    rxHdrBusErr <= 1'b0;
  else if ((rxListError | rxBufError) & rxListProcCs==WRITE_RHD_HEADER  |
           (rxListError | rxBufError) & rxListProcCs==WRITE_RHD         |
           (rxListError | rxBufError) & rxListProcCs==WRITE_RHD_FOOTER  |
           (rxListError | rxBufError) & rxListProcCs==WRITE_RHD_TRAILER |
           (rxListError | rxBufError) & rxListProcCs==WRITE_RPD_HEADER  |
           (rxListError | rxBufError) & rxListProcCs==WRITE_RPD         |
           (rxListError | rxBufError) & rxListProcCs==WRITE_RPD_FOOTER  |
           (rxListError | rxBufError) & rxListProcCs==UPDATE_PREV_RHD   )
    rxHdrBusErr <= 1'b1;
end


// Indicates that a bus error is detected during data processing, MAC hearder or
// payload.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
  if (~macPIRxClkHardRst_n)
    rxPayBusErr <= 1'b0;
  else if (~macPIRxClkSoftRst_n)
    rxPayBusErr <= 1'b0;
  else if ((rxListError | rxBufError) & rxListProcCs==WRITE_HEADER  |
           (rxListError | rxBufError) & rxListProcCs==WRITE_PAYLOAD )
    rxPayBusErr <= 1'b1;
end


// Indicates that a bus error is detected during data processing, header or
// payload.
assign rxBusErr_p = rxHdrBusErr & burstError_p |
                    rxPayBusErr & burstError_p ;


// Receive Buffer Available
//    Receive descriptor is not available if 
//    flow control is enabled and buffer overflow is detected.
always @(posedge macPIRxClk or negedge macPIRxClkHardRst_n)
begin
  if (~macPIRxClkHardRst_n)
    rxDescAvailable <= 1'b1;
  else if (~macPIRxClkSoftRst_n)
    rxDescAvailable <= 1'b1;
  else if (rxListProcCs==IDLE)
    rxDescAvailable <= 1'b1;
  else if (rxFlowCntrlEn & burstCs==BURST_STARTED & rxBufOverflow)
    rxDescAvailable <= 1'b0;
end


////////////////////////////////////////////////////////////////////////////////
// Debug Port
////////////////////////////////////////////////////////////////////////////////
assign rxListProcDebug     = {rxDescAvailable,
                              rxBufSel,
                              rxBufOverflowCapt,
                              rxFifoRdValid_p,
                              rxTagFifoRdData[3:0],
                              burstCs[2:0],
                              rxListProcCs[4:0]};
assign debugPortRxListA    = 16'h0; // ToDo


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef  RW_SIMU_ON
// String definition to display rxListProc current state
reg  [30*8-1:0] rxListProcCs_str;
reg  [30*8-1:0] burstCs_str;
reg  [40*8-1:0] rxTagFifoRdData_str;

// rxListProcCs FSM states displayed in a string to easy simulation and debug
always @*
begin
   case (rxListProcCs)
   IDLE                 : rxListProcCs_str = {"IDLE"};
   CHECK_FRAME_START    : rxListProcCs_str = {"CHECK_FRAME_START"};
   CHECK_RHD_BUFFER     : rxListProcCs_str = {"CHECK_RHD_BUFFER"};
   WRITE_RHD_HEADER     : rxListProcCs_str = {"WRITE_RHD_HEADER"};
   WRITE_RHD_FOOTER     : rxListProcCs_str = {"WRITE_RHD_FOOTER"};
   CHECK_HEADER_BUFFER  : rxListProcCs_str = {"CHECK_HEADER_BUFFER"};
   WRITE_HEADER         : rxListProcCs_str = {"WRITE_HEADER"};
   CHECK_HEADER_LEN     : rxListProcCs_str = {"CHECK_HEADER_LEN"};
   WRITE_RHD            : rxListProcCs_str = {"WRITE_RHD"};
   CHECK_RPD_BUFFER     : rxListProcCs_str = {"CHECK_RPD_BUFFER"};
   WRITE_RPD_HEADER     : rxListProcCs_str = {"WRITE_RPD_HEADER"};
   WRITE_RPD_FOOTER     : rxListProcCs_str = {"WRITE_RPD_FOOTER"};
   CHECK_PAYLOAD_BUFFER : rxListProcCs_str = {"CHECK_PAYLOAD_BUFFER"};
   WRITE_PAYLOAD        : rxListProcCs_str = {"WRITE_PAYLOAD"};
   WRITE_RPD            : rxListProcCs_str = {"WRITE_RPD"};
   WRITE_RHD_TRAILER    : rxListProcCs_str = {"WRITE_RHD_TRAILER"};
   UPDATE_PREV_RHD      : rxListProcCs_str = {"UPDATE_PREV_RHD"};
   DONE                 : rxListProcCs_str = {"DONE"};
   FIFO_FLUSH           : rxListProcCs_str = {"FIFO_FLUSH"};
   default              : rxListProcCs_str = {"XXX"};
   endcase
end

// rxListProcCs FSM states displayed in a string to easy simulation and debug
always @*
begin
   case (burstCs)
   BURST_IDLE    : burstCs_str = {"BURST_IDLE"};
   BURST_REQ     : burstCs_str = {"BURST_REQ"};
   BURST_START   : burstCs_str = {"BURST_START"};
   BURST_STARTED : burstCs_str = {"BURST_STARTED"};
   BURST_STOP    : burstCs_str = {"BURST_STOP"};
   BURST_STOPPED : burstCs_str = {"BURST_STOPPED"};
   BURST_DONE    : burstCs_str = {"BURST_DONE"};
   BURST_ERROR   : burstCs_str = {"BURST_ERROR"};
   default       : burstCs_str = {"XXX"};
   endcase
end

// rxTagFifoRdData states displayed in a string to easy simulation and debug
always @(*)
begin
   case (rxTagFifoRdData)
   4'b1010 : rxTagFifoRdData_str = "MPDU Length";
   4'b0001 : rxTagFifoRdData_str = "Start of Header";
   4'b0010 : rxTagFifoRdData_str = "End of Header";
   4'b0011 : rxTagFifoRdData_str = "Header Length";
   4'b1100 : rxTagFifoRdData_str = "Intermediate Header or payload";
   4'b0101 : rxTagFifoRdData_str = "End of payload";
   4'b1110 : rxTagFifoRdData_str = "Trigger Frame with SW response";
   4'b0000 : rxTagFifoRdData_str = "End of frame with save pattern";
   4'b1111 : rxTagFifoRdData_str = "End of frame with discard pattern";
   default : rxTagFifoRdData_str = "XXXXXX";
   endcase
end

`endif//RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef  RW_ASSERT_ON


`endif//RW_ASSERT_ON

endmodule
