//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : This module is the Lite primary Controller for the DMA in the 
//                    TX mode. 
// Simulation Notes : System verilog assertions can be enabled with RW_ASSERT_ON
//                    Then a system verilog capable simulator is needed
// Synthesis Notes  : 
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements : 
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module txLitePrimCtrl (
          //$port_g Clock and reset interface
          input  wire        macPITxClk,          // Platform 1 Clk domain
          input  wire        macPITxClkHardRst_n, // Hard Reset Synchronized to macPITxClk
          input  wire        macPITxClkSoftRst_n, // Soft Reset synchronized to macPITxClk

          //$port_g TxListProcessor interface
          input  wire        trigTxPCHalt_p,      // trigger to move FSM to HALTED
          input  wire        trigTxPCDead_p,      // trigger to move FSM to DEAD
          input  wire        trigTxACx_p,         // trigger to move FSM to ACTIVE
          input  wire        trigTxAC0,           // Indicate frames is associated with AC0
          input  wire        trigTxAC1,           // Indicate frames is associated with AC1
          input  wire        trigTxAC2,           // Indicate frames is associated with AC2
          input  wire        trigTxAC3,           // Indicate frames is associated with AC3
          input  wire [31:0] txHeadPtr,           // Queue Head Pointer
          input  wire        updAC0StaPtr_p,      // Update the PC Status Pointer signal.
          input  wire        updAC1StaPtr_p,      // Update the PC Status Pointer signal.
          input  wire        updAC2StaPtr_p,      // Update the PC Status Pointer signal.
          input  wire        updAC3StaPtr_p,      // Update the PC Status Pointer signal.
          input  wire [31:0] nxtDescPtr,          // the next Descriptor Pointer to be updated to
          output wire        pcTrigTxListProc,    // Trigger for the Tx List Processor
          output wire        pcTrigTxAC0,         // Trigger for the Tx Status Updater
          output wire        pcTrigTxAC1,         // Trigger for the Tx Status Updater
          output wire        pcTrigTxAC2,         // Trigger for the Tx Status Updater
          output wire        pcTrigTxAC3,         // Trigger for the Tx Status Updater
          output reg  [31:0] pcStatusPtr,         // The status pointer
          output wire        txNewHeadErr,        // Transmit New Head Error
          output reg         txStartup,           // Transmit Startup      
          output reg         txEndQ,              // Transmit  End of Queue  
          output reg         txDMADead,           // Transmit Channel error

          //$port_g Debug interface
          output wire  [2:0] txDmaPCState         // Debug Signal
         ); 
 

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
// txPrimCtrl FSM states definition
//$fsm_sd txPrimCtrlCs
localparam 
           HALTED  =  3'd1, // Reset state and Waiting for the trigger
           ACTIVE  =  3'd2, // Currently performing a dma transfer.
           DEAD    =  3'd4; // Error state. 

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

//Internal Signal Definitions
//Registers
reg [2:0]  txPrimCtrlCs;
reg [2:0]  txPrimCtrlNs;


//Wires
//The Queue head Pointer is valid
wire        txHeadPtrValid;
//The Status pointer should be updated
wire        updACxStaPtr_p;


`ifdef RW_SIMU_ON
// String definition to display txPrimCtrlCs current state
reg [7*8:0] txPrimCtrlCs_str;
`endif // RW_SIMU_ON


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Current State Logic 
always @ (posedge macPITxClk or negedge macPITxClkHardRst_n) 
begin
  if (macPITxClkHardRst_n == 1'b0)
    txPrimCtrlCs <= HALTED; 
  else if (macPITxClkSoftRst_n == 1'b0)
    txPrimCtrlCs <= HALTED;
  else
    txPrimCtrlCs <= txPrimCtrlNs; 
end


assign txHeadPtrValid = ( (txHeadPtr[1:0] == 2'b00) && (txHeadPtr[31:2] != 30'b0) ) ? 1'b1 : 1'b0 ;

assign updACxStaPtr_p = updAC0StaPtr_p | updAC1StaPtr_p | updAC2StaPtr_p | updAC3StaPtr_p;


//Next state Logic.
always @* 
begin
  case(txPrimCtrlCs)

    HALTED:
      //$fsm_s The state machine starts up in this state after reset. 
      //It may also transition to this state under other conditions. 
      //In this state, the state machine waits for the trigTxACx_p bits to be asserted.
      if (trigTxACx_p == 1'b1 && txHeadPtrValid == 1'b1)
        //$fsm_t If the active trigger is set the header pointer 
        //valid bit should also be set and AC should be valid.
        txPrimCtrlNs = ACTIVE; 
      else if (trigTxACx_p == 1'b1 && txHeadPtrValid == 1'b0)
        //$fsm_t If a active trigger is received but the Head Pointer 
        //is not valid or AC is not valid, the FSM moves to DEAD
        txPrimCtrlNs = DEAD; 
      else
        //$fsm_t If nothing happens, the FSM stays in HALTED
        txPrimCtrlNs = HALTED;

    ACTIVE:
      //$fsm_s In this state the DMA triggers the Transmit List Processor
      //to start moving the frames from the Local SRAM to the TX FIFO 
      // When Transmit Status Updater asserts the updAC0StaPtr_p, updAC1StaPtr_p,
      // updAC2StaPtr_p or updAC3StaPtr_p, the value on nextDescPtr is copied to the pcStatusPtr.
      if (trigTxPCDead_p == 1'b1)
        //$fsm_t The FSM will move to DEAD if its triggered by the TxListProcessor
        //to do so.This might happen in cases of error like policy table 
        //invalid, dmaHIF bus transaction error.   
        txPrimCtrlNs = DEAD;

      else if (trigTxACx_p == 1'b1 ||
               trigTxPCHalt_p == 1'b1)
        //$fsm_t If the trigTxACx_p bit is set and the FSM is not in HALTED it 
        //moves to HALTED and begins operations afresh
        txPrimCtrlNs = HALTED;
      else
        //$fsm_t If nothing happens, the FSM stays in ACTIVE
        txPrimCtrlNs = ACTIVE;

    DEAD:
      //$fsm_s In this state, the state machine waits for error recovery from SW. 
      //The FSM enters in this state when there is an error encountered
      //during the operation of the DMA like a policy table error or
      //a dmaHIF bus transaction error. Error recovery is only by SW which
      //is a system level reset 
      
      //$fsm_t Only the SW can recover the FSM from this state using SW reset
      txPrimCtrlNs = DEAD;
   
    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
    default:
      txPrimCtrlNs = HALTED;
    // pragma coverage block = on 
  endcase
end 


always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txDMADead <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txDMADead <= 1'b0;
  else if (((txPrimCtrlCs == ACTIVE) && trigTxPCDead_p) || 
           ((txPrimCtrlCs == HALTED) && (trigTxACx_p == 1'b1 && txHeadPtrValid == 1'b0))) 
    txDMADead <= 1'b1;
  else 
    txDMADead <= 1'b0;
end

//Logic for txStartup
//The txStartup is reset when the DMA channel leaves the HALTED state 
//for the ACTIVE.
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txStartup <= 1'b1;
  else if (macPITxClkSoftRst_n == 1'b0)
    txStartup <= 1'b1; 
  else if (txPrimCtrlCs == ACTIVE)
    txStartup <= 1'b0; 
end    

//Logic for txEndQ
//The txEndQ is set when the DMA channel moves in HALTED state 
//because it reaches the end of the linked list
//The txEndQ is reset when the DMA channel leaves the HALTED state 
//for the ACTIVE.
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txEndQ <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txEndQ <= 1'b0; 
  else if (trigTxPCHalt_p)
    txEndQ <= 1'b1; 
  else if (txPrimCtrlCs == ACTIVE)
    txEndQ <= 1'b0; 
end    

//Logic for Status Pointer
//The status pointer takes the new head pointer in the 
//register when the new head bit is set and the FSM is 
//in HALTED. 
//It updates to the nextDescriptorPointer given by the 
//status updater when it is triggered by the same.
//The Status updater will trigger for the updation after
//it has completed writing the status info to the descriptor
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    pcStatusPtr <= 32'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    pcStatusPtr <= 32'b0; 
  else if (txHeadPtrValid == 1'b1 &&
           trigTxACx_p == 1'b1 && 
           txPrimCtrlCs == HALTED)
    pcStatusPtr <= txHeadPtr;
  else if (updACxStaPtr_p == 1'b1)
    pcStatusPtr <= nxtDescPtr;
end    

//Logic for pcTrigTxListProc 
//This indicates to the Transmit list Processor this particular
//channel is in ACTIVE state
assign pcTrigTxListProc = (txPrimCtrlCs == ACTIVE) ? 1'b1 : 1'b0;

//This indicates to the Transmit Status Updater that frame is associated with AC0
assign pcTrigTxAC0 = (txPrimCtrlCs == ACTIVE && trigTxAC0 == 1'b1) ? 1'b1 : 1'b0;

//This indicates to the Transmit Status Updater that frame is associated with AC1
assign pcTrigTxAC1 = (txPrimCtrlCs == ACTIVE && trigTxAC1 == 1'b1) ? 1'b1 : 1'b0;

//This indicates to the Transmit Status Updater that frame is associated with AC2
assign pcTrigTxAC2 = (txPrimCtrlCs == ACTIVE && trigTxAC2 == 1'b1) ? 1'b1 : 1'b0;

//This indicates to the Transmit Status Updater that frame is associated with AC3
assign pcTrigTxAC3 = (txPrimCtrlCs == ACTIVE && trigTxAC3 == 1'b1) ? 1'b1 : 1'b0;

//This signal gives the present state of the Transmit Lite Primary Controller
assign txDmaPCState = txPrimCtrlCs;

// Indicates that the Transmit DMA channel's newHead bit was set but the 
// queueHeadPointer did not have a valid address.
assign txNewHeadErr = trigTxACx_p & !txHeadPtrValid;


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_SIMU_ON
// Disable coverage on the default state because it cannot be reached.
// pragma coverage block = off 
// txPrimCtrl FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (txPrimCtrlCs)
  HALTED   :  txPrimCtrlCs_str = {"HALTED"};        
  ACTIVE   :  txPrimCtrlCs_str = {"ACTIVE"};        
  DEAD     :  txPrimCtrlCs_str = {"DEAD"};        
  default  :  txPrimCtrlCs_str = {"XXX"};
  endcase
end
// pragma coverage block = on 
`endif // RW_SIMU_ON


`ifdef RW_ASSERT_ON
//Checks if trigger to DEAD happens only in ACTIVE state
property deadTrigger;
disable iff (!macPITxClkHardRst_n)
@(posedge trigTxPCDead_p) (txPrimCtrlCs != HALTED); 
endproperty
deadTriggrdRongState: assert property (deadTrigger); 

`endif                      

endmodule
