//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : This block implements the Selection logic for the signals
//                    between the Primary Controller and the Transmit List 
//                    processor
// Simulation Notes : 
// Synthesis Notes  : 
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms && concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////

module txPrimCtrlSelLogic (
          //$port_g Clock and reset interface
          input wire         macPITxClk,         //Platform 1 Clk domain
          input wire         macPITxClkHardRst_n,//Hard Reset Synchronized to macPITxClk
          input wire         macPITxClkSoftRst_n,//Soft Reset synchronized to macPITxClk

          //$port_g TxList Proc interface 
          input wire         trigHalt_p,           //From TxList Proc. Triggers PrimCtrl FSM to HALTED

          input wire         trigDead_p,           //From TxList Proc. Triggers PrimCtrl FSM to DEAD

          input wire         underRunDetected_p,   //From MAC Controller. Indicates a frame Lenght mismatch
                                                   //and the active channel has to be triggered to DEAD  

          input wire         patternError,         //From TxList Proc. 
          input wire         dmaHIFError,          //From dmaHIF Master. Indicates during transacation there was an
                                                   //dmaHIF error response
          
          input wire         nxtAtomFrmPtrValid,   //From TxList Proc. Indicates the nextAtomic Frame Pointer is valid
          input wire         nxtMpduDescValid,     //From TxList Proc. Indicates the next MPDU Descriptor Pointer is valid

          //$port_g Primary Ctlr interface 
          input wire  [31:0] tbStatusPointer,      //TB Status Pointer.
          input wire         tbTrigTxListProc,     //Trig for TxListProc from the TB Primary Ctlr.

          input wire  [31:0] bcnStatusPointer,     //Beacon Status Pointer.
          input wire         bcnTrigTxListProc,    //Trig for TxListProc from the Beacon Primary Ctlr.

          input wire  [31:0] ac0StatusPointer,     //AC0 Status Pointer.
          input wire         ac0TrigTxListProc,    //Trig for TxListProc from the ACO Primary Ctlr.

          input wire  [31:0] ac1StatusPointer,     //AC1 Status Pointer.
          input wire         ac1TrigTxListProc,    //Trig for TxListProc from the AC1 Primary Ctlr.

          input wire  [31:0] ac2StatusPointer,     //AC2 Status Pointer.
          input wire         ac2TrigTxListProc,    //Trig for TxListProc from the AC2 Primary Ctlr.

          input wire  [31:0] ac3StatusPointer,     //AC3 Status Pointer
          input wire         ac3TrigTxListProc,    //Trig for TxListProc from the AC3 Primary Ctlr.

          input wire         plyTblValid,          //Indicates validity of policy table address

          input wire         txTBNewHeadErr,       //Transmit Trigger Based New Head Error
          input wire         txBcnNewHeadErr,      //Transmit Beacon New Head Error
          input wire         txAC0NewHeadErr,      //Transmit AC0 New Head Error
          input wire         txAC1NewHeadErr,      //Transmit AC1 New Head Error
          input wire         txAC2NewHeadErr,      //Transmit AC2 New Head Error
          input wire         txAC3NewHeadErr,      //Transmit AC3 New Head Error

          output reg  [31:0] statusPointer,        //multiplexed Status Pointer for the Active Chan.
          output wire        trigTxListProc,       //multiplexed Trigger for the Active Chan.

          output wire        trigTBHalt_p,         //Demultiplexed Halt trigger for the TB Channel
          output wire        trigTBDead_p,         //Demultiplexed Dead trigger for the TB Channel
          output wire        nxtAtomFrmPtrValidTB, //Next Atomic Frame Pointer Valid indication for TB Channel
          output wire        nxtMpduDescValidTB,   //Next Mpdu descriptor Pointer Valid indication for TB Channel
          output wire        trigBcnHalt_p,        //Demultiplexed Halt trigger for the Beacon Channel
          output wire        trigBcnDead_p,        //Demultiplexed Dead trigger for the Beacon Channel
          output wire        nxtAtomFrmPtrValidBcn,//Next Atomic Frame Pointer Valid indication for Beacon Channel
          output wire        nxtMpduDescValidBcn,  //Next Mpdu descriptor Pointer Valid indication for Beacon Channel

          output wire        trigAc0Halt_p,        //Demultiplexed Halt trigger for the AC0 Channel
          output wire        trigAc0Dead_p,        //Demultiplexed Dead trigger for the AC0 Channel
          output wire        nxtAtomFrmPtrValidAc0,//Next Atomic Frame Pointer Valid indication for AC0 Channel
          output wire        nxtMpduDescValidAc0,  //Next Mpdu descriptor Pointer Valid indication for AC0 Channel

          output wire        trigAc1Halt_p,        //Demultiplexed Halt trigger for the AC1 Channel
          output wire        trigAc1Dead_p,        //Demultiplexed Dead trigger for the AC1 Channel
          output wire        nxtAtomFrmPtrValidAc1,//Next Atomic Frame Pointer Valid indication for AC1 Channel
          output wire        nxtMpduDescValidAc1,  //Next Mpdu descriptor Pointer Valid indication for AC1 Channel

          output wire        trigAc2Halt_p,        //Demultiplexed Halt trigger for the AC2 Channel
          output wire        trigAc2Dead_p,        //Demultiplexed Dead trigger for the AC2 Channel
          output wire        nxtAtomFrmPtrValidAc2,//Next Atomic Frame Pointer Valid indication for AC2 Channel
          output wire        nxtMpduDescValidAc2,  //Next Mpdu descriptor Pointer Valid indication for AC2 Channel

          output wire        trigAc3Halt_p,        //Demultiplexed Halt trigger for the AC3 Channel
          output wire        trigAc3Dead_p,        //Demultiplexed Dead trigger for the AC3 Channel
          output wire        nxtAtomFrmPtrValidAc3,//Next Atomic Frame Pointer Valid indication for AC3 Channel
          output wire        nxtMpduDescValidAc3,  //Next Mpdu descriptor Pointer Valid indication for AC3 Channel

          //$port_g Status Registers indication
          output reg         txLenMismatch,        // Transmit Length Mismatch
          output reg         txUPatternErr,        // Transmit Unique Pattern Error
          output reg         txNextPointerErr,     // Transmit Next Pointer Error
          output reg         txPTAddressErr,       // Transmit Policy Table Address Error
          output reg         txBusErr,             // Transmit Bus Error
          output reg         txNewHeadErr          // Transmit New Head Error
         ); 

localparam TB     = 6'b100000,//Multiplexes signals for the TB prim ctlr
           BEACON = 6'b010000,//Multiplexes signals for the beacon prim ctlr
           AC0    = 6'b001000,//Multiplexes signals for the AC0 prim ctlr
           AC1    = 6'b000100,//Multiplexes signals for the AC1 prim ctlr
           AC2    = 6'b000010,//Multiplexes signals for the AC2 prim ctlr
           AC3    = 6'b000001;//Multiplexes signals for the AC3 prim ctlr


//Internal Signal Declarations
reg   [5:0] selectSignal;
reg         trigTxListProcDly;

//Logic for selectSignal.
//The selectSignal is generated based on the trigger for the Trnasmit List
//Processor given by the Transmit Primary Controller. And since only one 
//primary controller can be active the selectSignal is just given as a 
//concatenation of all the trigger signals. And all the trigger signals are 
//asserted high and are not pulses. So through the entire transfer when the
//channel is active the selectSignal will point to the correct Primary 
//Controller.
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    selectSignal <= 6'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    selectSignal <= 6'b0;
  else if (trigTxListProc && !trigTxListProcDly)
    selectSignal <= {tbTrigTxListProc, bcnTrigTxListProc, ac0TrigTxListProc,
                     ac1TrigTxListProc, ac2TrigTxListProc, ac3TrigTxListProc};
end

//trigTxListProc Delayedof 1 clock cycle. Used to detect the rising edge of trigTxListProc. 
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    trigTxListProcDly <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    trigTxListProcDly <= 1'b0;
  else 
    trigTxListProcDly <= trigTxListProc;
end


//When the TB channel is active as indicated by the tbTrigTxListProc
//asserted HIGH the trigHalt_p is routed to the TB channel
assign trigTBHalt_p         = trigHalt_p   && tbTrigTxListProc;

//When the TB channel is active as indicated by the tbTrigTxListProc
//asserted HIGH the trigDead_p is routed to the TB channel
assign trigTBDead_p        = trigDead_p   && tbTrigTxListProc;

//When the Beacon channel is active as indicated by the bcnTrigTxListProc
//asserted HIGH the trigHalt_p is routed to the Beacon channel
assign trigBcnHalt_p        = trigHalt_p   && bcnTrigTxListProc;

//When the Beacon channel is active as indicated by the bcnTrigTxListProc
//asserted HIGH the trigDead_p is routed to the Beacon channel
assign trigBcnDead_p        = trigDead_p   && bcnTrigTxListProc;

//When the AC0 channel is active as indicated by the ac0TrigTxListProc
//asserted HIGH the trigHalt_p is routed to the AC0 channel
assign trigAc0Halt_p        = trigHalt_p   && ac0TrigTxListProc;

//When the AC0 channel is active as indicated by the ac0TrigTxListProc
//asserted HIGH the trigDead_p is routed to the AC0 channel
assign trigAc0Dead_p        = trigDead_p   && ac0TrigTxListProc;

//When the AC1 channel is active as indicated by the ac1TrigTxListProc
//asserted HIGH the trigHalt_p is routed to the AC1 channel
assign trigAc1Halt_p        = trigHalt_p   && ac1TrigTxListProc;

//When the AC1 channel is active as indicated by the ac1TrigTxListProc
//asserted HIGH the trigDead_p is routed to the AC1 channel
assign trigAc1Dead_p        = trigDead_p   && ac1TrigTxListProc;

//When the AC2 channel is active as indicated by the ac2TrigTxListProc
//asserted HIGH the trigHalt_p is routed to the AC2 channel
assign trigAc2Halt_p        = trigHalt_p   && ac2TrigTxListProc;

//When the AC2 channel is active as indicated by the ac2TrigTxListProc
//asserted HIGH the trigDead_p is routed to the AC2 channel
assign trigAc2Dead_p        = trigDead_p   && ac2TrigTxListProc;

//When the AC3 channel is active as indicated by the ac3TrigTxListProc
//asserted HIGH the trigHalt_p is routed to the AC3 channel
assign trigAc3Halt_p        = trigHalt_p   && ac3TrigTxListProc;

//When the AC3 channel is active as indicated by the ac3TrigTxListProc
//asserted HIGH the trigDead_p is routed to the AC3 channel
assign trigAc3Dead_p        = trigDead_p   && ac3TrigTxListProc;

//The multiplexed trigger for the transmit list processor from different 
//channels is generated
assign trigTxListProc = tbTrigTxListProc  || bcnTrigTxListProc ||
                        ac0TrigTxListProc || ac1TrigTxListProc ||
                        ac2TrigTxListProc || ac3TrigTxListProc;

//Assigning the nxtAtomFrmPtrValid to the ACTIVE primary controller
assign nxtAtomFrmPtrValidAc0 = nxtAtomFrmPtrValid && ac0TrigTxListProc;

//Assigning the nxtAtomFrmPtrValid to the ACTIVE primary controller
assign nxtAtomFrmPtrValidAc1 = nxtAtomFrmPtrValid && ac1TrigTxListProc;

//Assigning the nxtAtomFrmPtrValid to the ACTIVE primary controller
assign nxtAtomFrmPtrValidAc2 = nxtAtomFrmPtrValid && ac2TrigTxListProc;

//Assigning the nxtAtomFrmPtrValid to the ACTIVE primary controller
assign nxtAtomFrmPtrValidAc3 = nxtAtomFrmPtrValid && ac3TrigTxListProc;

//Assigning the nxtAtomFrmPtrValid to the ACTIVE primary controller
assign nxtAtomFrmPtrValidBcn = nxtAtomFrmPtrValid && bcnTrigTxListProc;

//Assigning the nxtAtomFrmPtrValid to the ACTIVE primary controller
assign nxtAtomFrmPtrValidTB = nxtAtomFrmPtrValid && tbTrigTxListProc;

//Assigning the nxtMpduDescValid to the ACTIVE primary controller
assign nxtMpduDescValidAc0 = nxtMpduDescValid && ac0TrigTxListProc;

//Assigning the nxtMpduDescValid to the ACTIVE primary controller
assign nxtMpduDescValidAc1 = nxtMpduDescValid && ac1TrigTxListProc;

//Assigning the nxtMpduDescValid to the ACTIVE primary controller
assign nxtMpduDescValidAc2 = nxtMpduDescValid && ac2TrigTxListProc;

//Assigning the nxtMpduDescValid to the ACTIVE primary controller
assign nxtMpduDescValidAc3 = nxtMpduDescValid && ac3TrigTxListProc;

//Assigning the nxtMpduDescValid to the ACTIVE primary controller
assign nxtMpduDescValidBcn = nxtMpduDescValid && bcnTrigTxListProc;

//Assigning the nxtMpduDescValid to the ACTIVE primary controller
assign nxtMpduDescValidTB = nxtMpduDescValid && tbTrigTxListProc;


//Multiplexer
//This logic helps in multiplexing the status Pointer signals from 
//the different Transmit Primary Controllers to the Transmit List processor
//depending on which transmit primary controlleris currently active.
//The selection is based on the selectSignal.
always @*
begin
  case (selectSignal)
    TB:      statusPointer = tbStatusPointer;
    BEACON:  statusPointer = bcnStatusPointer;
    AC0:     statusPointer = ac0StatusPointer;
    AC1:     statusPointer = ac1StatusPointer;
    AC2:     statusPointer = ac2StatusPointer;
    AC3:     statusPointer = ac3StatusPointer;
    default: statusPointer = 32'b0;
  endcase
end


// Indication to the DMA Status2 Register when an underrun is detected on current
// transmit Channel
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txLenMismatch <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txLenMismatch <= 1'b0;
  else if (underRunDetected_p && trigTxListProc)
    txLenMismatch <= 1'b1;
end


// Indication to the DMA Status2 Register when a pattern Error is detected on 
// current Channel
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txUPatternErr <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txUPatternErr <= 1'b0;
  else if (patternError && trigTxListProc && trigDead_p)
    txUPatternErr <= 1'b1;
end


// Indication to the DMA Status2 Register when a bus Error is detected
// during a transmission on current Channel
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txBusErr <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txBusErr <= 1'b0;
  else if (dmaHIFError && trigTxListProc && trigDead_p)
    txBusErr <= 1'b1;
end


// Indicates that the Transmit DMA channel did not find a valid PT address 
// when reading the Header Descriptor on current Channel.
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txPTAddressErr <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txPTAddressErr <= 1'b0;
  else if (!plyTblValid && trigTxListProc && trigDead_p)
    txPTAddressErr <= 1'b1;
end


// Indicates that the Transmit DMA channel found an invalid Next 
// Atomic Frame Exchange Sequence Pointer or Next MPDU Descriptor Pointer
// on current Channel.
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txNextPointerErr <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txNextPointerErr <= 1'b0;
  else if ((!patternError) && (!dmaHIFError) && (plyTblValid) &&
           (!nxtAtomFrmPtrValid || !nxtMpduDescValid) && trigTxListProc && trigDead_p)
    txNextPointerErr <= 1'b1;
end

// Indicates that the current Transmit  DMA channel newHead bit was set but the
// queue HeadPointer did not have a valid address
always @(posedge macPITxClk or negedge macPITxClkHardRst_n)
begin
  if (macPITxClkHardRst_n == 1'b0)
    txNewHeadErr <= 1'b0;
  else if (macPITxClkSoftRst_n == 1'b0)
    txNewHeadErr <= 1'b0;
  else if (txTBNewHeadErr  | txBcnNewHeadErr | txAC0NewHeadErr |
           txAC1NewHeadErr | txAC2NewHeadErr | txAC3NewHeadErr )
    txNewHeadErr <= 1'b1;
end

endmodule
