//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//   This block is the top level module for ccmp implementation. this module
//   instantiates  the ccm,ccmInReg and ccmOutReg blocks and also include
//   AAD and NONCE construction logic
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module ccmp (

  //$port_g Inputs
  input  wire         macCoreClk,         // macCore clock signal (freerunning clock)
  input  wire         macCryptClk,        // macCrypt clock signal (gated clock for crypto engine)
  input  wire         nPRst,              // reset signal from hardware
  input  wire         nSRst,              // reset signal from software
  input  wire         rxCsIsIdle,         // Indicates that receive state machine is idle
  input  wire         txCsIsIdle,         // Indicates that transmit state machine is idle
  input  wire         initCG_p,           // initialization trigger pulse for CCMP/GCMP
  input  wire         txRxBufferFull,     // indicates that CCMP transmit buffer/receive FIFO is full
  input  wire         txRxErrorP,         // pulse indicating an error has been encountered
  input  wire   [2:0] cipherType,         // Cipher Type to be used for encryption/decryption
  input  wire   [1:0] cipherLen,          // Cipher Key Length to be used for encryption/decryption
  input  wire [255:0] temporalKey,        // temporal key to be used for encryption/decryption
  input  wire  [15:0] payloadLen,         // Length of the payload in bytes

  input  wire  [15:0] frameControl,       // Frame Control
  input  wire  [47:0] addr1,              // Addr1 field of the MAC Header
  input  wire  [47:0] addr2,              // Addr2 field of the MAC Header
  input  wire  [47:0] addr3,              // Addr3 field of the MAC Header
  input  wire  [47:0] addr4,              // Addr4 field of the MAC Header
  input  wire  [47:0] sfc,                // SFC Tx field of the MAC Frame payload
  input  wire  [15:0] seqControl,         // Sequence Control Field for wpi block
  input  wire         address4Pres,       // Flag indicating address 4 is present
  input  wire         qosFrame,           // Flag indicating the current frame is a QoS Frame
  input  wire   [3:0] tid,                // Priority information for QoS frames
  input  wire         qos7,               // HT spp QoS bit 7
  input  wire         htMode,             // HT Mode
  input  wire   [1:0] sppKSR,             // HT A MSDU Capability

  //$port_g CCMP Data Input Interface
  input  wire         payloadEnd_p,       // pulse indicating end of payload
  input  wire   [7:0] ccmpInData,
  input  wire         ccmpInValid,        // write enable signal to CCMP input register
  output wire         ccmpRegFull,        // flag indicating CCMP input register is full

  //$port_g CCMP Data Output Interface
  output wire   [7:0] ccmpOutData,        // output data from the CCMP block
  output wire         ccmpOutValid_p,     // pulse indicating valid data present on the ccmpOutData lines
  output wire         ccmpOutLast_p,      // Last data from CCMP
  output wire         nextccmpRegFull,    // flag indicating CCMP input register is full
  output wire         ccmpIsIdle,        // Indicate CCMP is IDLE

  output wire         micPassed_p,        // pulse indicating MIC check successful
  output wire         micFailed_p,        // pulse indicating MIC check failed
  output reg    [1:0] ccmp_cs             // ccmp current state
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
localparam 
   IDLE     = 2'd0,
   INIT     = 2'd1,
   DATA     = 2'd2,
   MIC_OUT  = 2'd3;


reg [1:0] ccmp_ns;

////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
wire         ccmpSWRst_n;
wire [255:0] aad;
reg  [127:0] aadOne;
wire [127:0] aadTwo;
reg    [7:0] aadLen;
reg   [15:0] fcMuted;
reg          qos7_saved;
wire   [3:0] maskedtid;
wire [103:0] nonce;
wire [127:0] mic;
wire         micReady;
wire         ccmpOutValid;
wire         payloadEndOut_p;
wire         intMicPassed_p;
wire         intMicFailed_p;
wire         dataFrame;
wire         mgtFrame;
wire         fc_15;
wire         loadAAD2_p;
// CCM Data input
wire [127:0] ccmInData;
wire  [15:0] ccmInByteEn;
wire         ccmInLast;
wire         ccmInValid;
wire         ccmInReady;
// CCM Data output
wire [127:0] ccmOutData;
wire         ccmOutValid;
wire         ccmOutReady;

wire         micValid_p;         // pulse indicating MIC calculation done
wire         micStoredInCcmpOutReg;

`ifdef RW_SIMU_ON
// String definition to display ccmp current state
reg [8*8-1:0] ccmp_cs_str;
`endif // RW_SIMU_ON

////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////
// CCM & GCM are reset when both RX & TX Path are idle.
// This is needed when TX or RX error occurs during encryption/decryption.
assign ccmpSWRst_n = nSRst & ~(rxCsIsIdle & txCsIsIdle);


//******************************************************************************
// ccmpInreg Instantiation
//******************************************************************************
ccmpInReg u_ccmpInReg(
   //Clock And Reset
   .pClk               (macCoreClk         ),
   .nPRst              (nPRst              ),
   .nSRst              (ccmpSWRst_n        ),
   //Controls
   .rxCsIsIdle         (rxCsIsIdle         ),
   .txCsIsIdle         (txCsIsIdle         ),
   .cipherType         (cipherType         ),
   .cipherLen          (cipherLen          ),
   .txRxErrorP         (txRxErrorP         ),
   .payloadEnd_p       (payloadEnd_p       ),
   .loadAAD2_p         (loadAAD2_p         ),
   //Data byte input
   .ccmpInData         (ccmpInData         ),
   .ccmpInValid        (ccmpInValid        ),
   //Data block output
   .ccmInData          (ccmInData          ),
   .ccmInByteEn        (ccmInByteEn        ),
   .ccmInLast          (ccmInLast          ),
   .ccmInValid         (ccmInValid         ),
   .ccmInReady         (ccmInReady         ),
   //
   .payloadEndOut_p    (payloadEndOut_p    ),
   .ccmpRegFull        (ccmpRegFull        ),
   .nextccmpRegFull    (nextccmpRegFull    )
);


//******************************************************************************
// CCM/GCM Instantiation
//******************************************************************************

ccmp_gcmp u_ccmp_gcmp(
   //Clock and reset
   .clk              (macCryptClk             ),
   .hard_rst_n       (nPRst                   ),
   .soft_rst_n       (ccmpSWRst_n             ),
   //Control interface
   .encrypt          (rxCsIsIdle              ),
   .init_p           (initCG_p                ),
   .mode             (cipherType              ),
   .busy             (                        ),
   .aad              ({aadOne,aadTwo[127:16]} ),
   .aad_len          ({8'd0,aadLen}           ),
   .nonce            (nonce                   ),
   .data_len         (payloadLen              ),
   .aes_key          (temporalKey             ),
   .aes_key_len      (cipherLen               ),
   .mic_passed_p     (intMicPassed_p          ),
   .mic_failed_p     (intMicFailed_p          ),
   //Data Interface
   .data_in          (ccmInData               ),
   .data_in_byte_en  (ccmInByteEn             ),
   .data_in_last     (ccmInLast               ),
   .data_in_valid    (ccmInValid              ),
   .data_in_ready    (ccmInReady              ),
   .data_out         (ccmOutData              ),
   .data_out_valid   (ccmOutValid             ),
   .data_out_ready   (ccmOutReady             ),
   .mic_out          (mic                     ),
   .mic_out_valid    (micValid_p              ),
   .mic_out_ready    (micReady                ),
   //Context interface
   .context_in       (794'h0                  ),
   .context_in_valid (1'b0                    ),
   .context_out      (/*Open*/                ),
   //Debug ports
   .ccm_debug        (/*Open*/                ),
   .gcm_debug        (/*Open*/                )
);


//******************************************************************************
// FSM
//******************************************************************************
always @(posedge macCoreClk or negedge nPRst)
begin
   if (!nPRst)
      ccmp_cs <= IDLE;
   else if (!ccmpSWRst_n)
      ccmp_cs <= IDLE;
   else
      ccmp_cs <= ccmp_ns;
end

always @*
begin
   case (ccmp_cs)
   IDLE:
      if (initCG_p)
         ccmp_ns = INIT;
      else
         ccmp_ns = IDLE;
   INIT:
      if (ccmInReady)
         ccmp_ns = DATA;
      else
         ccmp_ns = INIT;
   DATA:
      if ((micValid_p && rxCsIsIdle) || ((intMicPassed_p || intMicFailed_p) && txCsIsIdle))
         ccmp_ns = MIC_OUT;
      else
         ccmp_ns = DATA;
   MIC_OUT:
      if (ccmpOutLast_p)
         ccmp_ns = IDLE;
      else
         ccmp_ns = MIC_OUT;
   endcase
end

assign micStoredInCcmpOutReg = (ccmp_ns == MIC_OUT);// && (ccmp_cs == DATA);

assign ccmpIsIdle  = ccmp_cs==IDLE;
assign loadAAD2_p  = initCG_p;


//******************************************************************************
// ccmpOutReg
//******************************************************************************
ccmpOutReg u_ccmpOutReg (
   //Clock and reset
   .pClk           (macCryptClk     ),
   .nPRst          (nPRst           ),
   .nSRst          (ccmpSWRst_n     ),
   //Inputs
   .rxCsIsIdle     (1'b1            ),
   .payloadEnd_p   (payloadEndOut_p ),
   .payloadLen     (payloadLen[3:0] ),
   .txRxErrorP     (txRxErrorP      ),
   .cipherType     (cipherType      ),
   .cipherLen      (cipherLen       ),
   //MIC input
   .mic            (mic             ),
   .micValid       (micStoredInCcmpOutReg),
   .micReady       (micReady        ),
   //Data input
   .ccmOutData     (ccmOutData      ),
   .ccmOutValid    (ccmOutValid     ),
   .ccmOutReady    (ccmOutReady     ),
   //Data output
   .ccmpOutData    (ccmpOutData     ),
   .ccmpOutValid   (ccmpOutValid    ),
   .ccmpOutReady   (~txRxBufferFull ),
   .ccmpOutLast_p  (ccmpOutLast_p   )   
);


//******************************************************************************
// Control logic
//******************************************************************************

// Constuct AAD/Nonce From Data 
  assign  aadTwo = {addr2, addr1, fcMuted, aadLen, 8'h00};
  assign  aad    = {aadTwo, aadOne};

// HT QoS FC bit 15 masking
  assign  fc_15  = (htMode && qosFrame) ? 1'b0 : frameControl[15];

  always @(posedge macCoreClk,negedge nPRst)
  begin
    if(nPRst == 1'b0)
      qos7_saved         <= 1'b0;
    else if(ccmpSWRst_n == 1'b0)
      qos7_saved         <= 1'b0;
    else
    begin
      if(initCG_p)
      begin
        if(htMode && qosFrame && (sppKSR == 2'b10)) 
          qos7_saved <= qos7;
        else
          qos7_saved <= 1'b0;
      end
    end
  end

  always @*
  begin
    if (dataFrame)
      fcMuted = {fc_15, 1'b1, 3'b000, frameControl[10:7], 3'b000, frameControl[3:0]};
    else
      fcMuted = {fc_15, 1'b1, 3'b000, frameControl[10:7], frameControl[6:0]};
  end

  always @*
  begin
    if (address4Pres && qosFrame)
    begin
      aadLen = 8'h1e;
      aadOne = {8'b0,{qos7_saved,3'b0}, maskedtid, addr4, {12'h00, seqControl[3:0]},addr3};
    end
    else if (address4Pres)
    begin
      aadLen = 8'h1c;
      aadOne = {16'b0,addr4,{12'h00, seqControl[3:0]},addr3};
    end
    else if (qosFrame)
    begin
      aadLen = 8'h18;
      aadOne = {56'b0,{qos7_saved,3'b0},maskedtid,{12'h00, seqControl[3:0]},addr3};
    end
    else
    begin
      aadLen = 8'h16;
      aadOne = {64'b0,{12'h00, seqControl[3:0]},addr3};
    end
  end


  assign dataFrame = (frameControl[3:2] == 2'b10) ? 1'b1 : 1'b0;
  assign mgtFrame  = (frameControl[3:2] == 2'b00) ? 1'b1 : 1'b0;

  assign maskedtid = tid & {qosFrame,qosFrame,qosFrame,qosFrame};

//nonce
  assign nonce     = {sfc,addr2,3'h0,mgtFrame,maskedtid};



//micPassed_p 
  assign micPassed_p = intMicPassed_p; 
//micFailed_p
  assign micFailed_p = intMicFailed_p; 

// Mask the ccmpOutValid if the ccmp transmit buffer / receive FIFO is full
  assign ccmpOutValid_p  = ccmpOutValid & ~(txRxBufferFull); 


///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// ccmp_cs FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (ccmp_cs)
                  IDLE  :  ccmp_cs_str = {"IDLE"};
                  INIT  :  ccmp_cs_str = {"INIT"};
                  DATA  :  ccmp_cs_str = {"DATA"};
               MIC_OUT  :  ccmp_cs_str = {"MIC_OUT"};
               default  :  ccmp_cs_str = {"XXX"};
   endcase
end

`endif // RW_SIMU_ON

endmodule

//---- END OF FILE  ------//
