//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//   This block is input buffer which converts 8 bits stream to 128 bits data 
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module ccmpInReg (

  //$port_g Clock and Reset
  input  wire          pClk,                 // baseband clock signal
  input  wire          nPRst,                // reset signal from hardware
  input  wire          nSRst,                // reset signal from software
  //$port_g Controls
  input  wire          rxCsIsIdle,           // Encrypt/Decrypt flag
  input  wire          txCsIsIdle,           // Encrypt/Decrypt flag
  input  wire          txRxErrorP,           // Error flag
  input  wire    [2:0] cipherType,           // Cipher Type to be used for encryption/decryption
  input  wire    [1:0] cipherLen,            // Cipher Key Length to be used for encryption/decryption
  output wire          payloadEndOut_p,      // Indicates the end of payload 

  //$port_g Data input
  input  wire [7:0]    ccmpInData,           // CCMP Data byte input
  input  wire          ccmpInValid,          // CCMP Data byte is valid
  input  wire          payloadEnd_p,         // indicates last byte of payload
  output wire          ccmpRegFull,          // CCMP buffer full signal
  output wire          nextccmpRegFull,      // CCMP buffer full signal
  //$port_g Data output
  output wire [127:0]  ccmInData,            // Input data to CCM block.
  output wire  [15:0]  ccmInByteEn,          // Input byte enable to CCM block.
  output reg           ccmInLast,            // Last data to CCM block
  output reg           ccmInValid,           // Indicate the CCM input data is valid
  input  wire          ccmInReady,           // Ready to receive input data
  input  wire          loadAAD2_p            // AAD read signal from CCM block
  );


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
localparam 
   CCMP=3'b011;


  // ------------------------ Register Declarations ------------------------------
  reg [127:0] inReg;  
  reg  [15:0] byteEnReg;
  reg   [3:0] lenCtr;  
  reg         micStart;  
  reg         shiftZeroes;  
  reg         nextshiftZeroes;  
  reg         ccmpInRegFull;  
  reg         nextccmpInRegFull;  

  //--------------Logic Begins----------------------------------------

  // Buffer & pointer
  always @(posedge pClk or negedge nPRst) 
  begin
    if (!nPRst)
    begin
      inReg     <= 128'h0;
      byteEnReg <= {16{1'b1}};
      lenCtr    <= 4'h0;
    end
    else if (!nSRst)
    begin
      inReg     <= 128'h0;
      byteEnReg <= {16{1'b1}};
      lenCtr    <= 4'h0;
    end
    else
    begin
      if ((ccmInValid & ccmInReady) || loadAAD2_p || txRxErrorP || (rxCsIsIdle && txCsIsIdle))
      begin
        byteEnReg <= {16{1'b1}};
        lenCtr    <= 4'h0;
      end
      else if (shiftZeroes)
      begin
        lenCtr    <= 4'hF;
      end
      else if (ccmpInValid & !ccmInValid)
      begin
        if (lenCtr == 4'd0)
        begin
          byteEnReg <= {{15{1'b0}},1'b1};
          inReg     <= {{120{1'b0}},ccmpInData};
        end
        else  
        begin
          byteEnReg[lenCtr]   <= 1'b1;
          inReg               <= inReg | ({120'h0,ccmpInData}<<{lenCtr,3'b000});
        end
        lenCtr    <= lenCtr + 4'h1;
      end
    end
  end


  // ccmpInRegFull generation
  always @*
  begin
    if ((ccmInValid & ccmInReady) || loadAAD2_p || txRxErrorP || (rxCsIsIdle && txCsIsIdle))
      nextccmpInRegFull = 1'b0;
    else if (lenCtr[3:0]==4'hF && (ccmpInValid || shiftZeroes))
      nextccmpInRegFull = 1'b1;
    else
      nextccmpInRegFull = ccmpInRegFull;
  end


  // ccmInValid signal generation 
  always@(posedge pClk or negedge nPRst) 
  begin
    if (!nPRst)
      ccmInValid <= 1'b0;
    else if (!nSRst)
      ccmInValid <= 1'b0;
    else if ((ccmInValid && ccmInReady) || txRxErrorP || (rxCsIsIdle && txCsIsIdle))
      ccmInValid <= 1'b0;
    else if ((lenCtr[3:0]==4'hF) && ccmpInValid)
      ccmInValid <= 1'b1;
    else if (shiftZeroes)
      ccmInValid <= 1'b1;
  end

  // ccmInLast signal generation 
  always@(posedge pClk or negedge nPRst) 
  begin
    if (!nPRst)
      ccmInLast <= 1'b0;
    else if (!nSRst)
      ccmInLast <= 1'b0;
    else if ((ccmInValid && ccmInReady) || txRxErrorP || (rxCsIsIdle && txCsIsIdle))
      ccmInLast <= 1'b0;
    else if (payloadEnd_p & ccmpInValid & !ccmpRegFull)
      ccmInLast <= 1'b1;
  end

  assign payloadEndOut_p = ccmInLast & ccmInValid & ccmInReady;

  // ShiftZero signal generation 
  always @*
  begin
    if ((lenCtr[3:0]==4'hF && shiftZeroes) || txRxErrorP || (rxCsIsIdle && txCsIsIdle))
      nextshiftZeroes = 1'b0;
    else if ((lenCtr[3:0]==4'h7 & ccmpInValid & micStart & cipherType==CCMP & cipherLen==2'd0) |
             (lenCtr[3:0]!=4'hF & ccmpInValid & payloadEnd_p  & !ccmpRegFull)) 
      nextshiftZeroes = 1'b1;
    else
      nextshiftZeroes = shiftZeroes;
  end

  // MicStart signal generation 
  always@(posedge pClk or negedge nPRst) 
  begin
    if (!nPRst)
      micStart <= 1'b0;
    else if (!nSRst)
      micStart <= 1'b0;
    else if ((ccmInValid && ccmInReady && !ccmInLast) || txRxErrorP || (rxCsIsIdle && txCsIsIdle))
      micStart <= 1'b0;
    else if ( ccmInValid && ccmInReady &&  ccmInLast && !rxCsIsIdle)
      micStart <= 1'b1;
  end

  // Buffer Full signal generation to the outside World
  assign  nextccmpRegFull = nextccmpInRegFull | nextshiftZeroes;
  assign  ccmpRegFull     = ccmpInRegFull     | shiftZeroes;

  // ccmInData
  assign  ccmInData   = inReg;
  assign  ccmInByteEn = byteEnReg;


  // Sequential logic
  always@(posedge pClk or negedge nPRst)
  begin
    if (!nPRst)
    begin
      ccmpInRegFull    <= 1'b0;
      shiftZeroes      <= 1'b0;
    end
    else if (!nSRst)
    begin
      ccmpInRegFull    <= 1'b0;
      shiftZeroes      <= 1'b0;
    end
    else
    begin
      ccmpInRegFull    <= nextccmpInRegFull;
      shiftZeroes      <= nextshiftZeroes;
    end
  end

endmodule       
//------------------- END OF FILE ----------------//
