//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//   This block is output buffer which converts 128 bits o/p of aes block into 
//   the streams of 8 bits
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module ccmpOutReg (

  //$port_g Clock and reset
  input  wire         pClk,            // baseband clock signal
  input  wire         nPRst,           // reset signal from hardware
  input  wire         nSRst,           // reset signal from software
  //$port_g Controls
  input  wire         rxCsIsIdle,      // Indicates Encrypt/Decrypt
  input  wire         txRxErrorP,      // Indicates Error
  input  wire         payloadEnd_p,    // Indicates that payload End
  input  wire   [3:0] payloadLen,      // length of payload
  input  wire   [2:0] cipherType,      // Cipher Type to be used for encryption/decryption
  input  wire   [1:0] cipherLen,       // AES Key Length
  //$port_g Data input
  input  wire [127:0] ccmOutData,      // CCM output data
  input  wire         ccmOutValid,     // CCM data is valid
  output wire         ccmOutReady,     // Ready to receive CCM data
  //$port_g MIC input
  input  wire [127:0] mic,             // MIC
  input  wire         micValid,        // MIC is valid
  output wire         micReady,        // Ready to receive MIC
  //$port_g Data output
  output wire   [7:0] ccmpOutData,     // CCMP output data
  output reg          ccmpOutValid,    // CCMP output data is valid
  input  wire         ccmpOutReady,    // Ready to receive CCMP data
  output wire         ccmpOutLast_p    // Last data from CCMP
  );

////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
localparam 
   CCMP=3'b011;


// ------------------------ Register Declarations ------------------
  reg  [127:0] outReg;
  reg  [127:0] nextoutReg;
  reg    [3:0] regPtr;
  reg    [3:0] nextregPtr;
  reg          nextccmpOutValid;
  reg          nextlastBlock;
  reg          lastBlock;
  reg          micLoaded;
  reg          micLoadedDly;
  reg    [3:0] lastMicPtr;

// ------------------------ Wire Declarations ---------------------
  wire   [3:0] newpayloadLen;

//--------------Logic Begins---------------------------------------
// Data register and pointer logic
  always @*
  begin
    nextregPtr = regPtr;
    nextoutReg = outReg;
    if (txRxErrorP)
      nextregPtr = 4'h0;
    else if (ccmOutValid & ccmOutReady)
    begin
      nextregPtr = 4'h0;
      nextoutReg = ccmOutData;
    end
    else if (micValid && micReady && rxCsIsIdle)
    begin
      nextregPtr = 4'h0;
      nextoutReg = mic; 
    end
    else if (ccmpOutValid && ccmpOutReady)
    begin
      nextregPtr = regPtr + 4'h1; 
      nextoutReg = {8'h00,outReg[127:8]};
    end
  end

// ccmpOutValid generation
  always @*
  begin
    if (txRxErrorP)
      nextccmpOutValid = 1'b0;
    else if ((regPtr==4'hF ||
             (regPtr==newpayloadLen && lastBlock) |
             (regPtr==lastMicPtr    && micLoaded)) && ccmpOutValid && ccmpOutReady)
      nextccmpOutValid = 1'b0;
    else if ((ccmOutValid && ccmOutReady) || (micValid && micReady && rxCsIsIdle))
      nextccmpOutValid = 1'b1;
    else
      nextccmpOutValid = ccmpOutValid;
  end

// Last block signal generation
  always @*
  begin
    if (((regPtr==newpayloadLen) && lastBlock && ccmpOutValid && ccmpOutReady) || txRxErrorP)
      nextlastBlock = 1'b0;
    else if (payloadEnd_p)
      nextlastBlock = 1'b1;
    else
      nextlastBlock = lastBlock;
  end

// Mic Loaded signal generation
//   Indicate that the MIC is loaded into the output register
  always @(posedge pClk or negedge nPRst) 
  begin
    if (!nPRst) 
      micLoaded <= 1'b0;
    else if (!nSRst)
      micLoaded <= 1'b0;
    else if (txRxErrorP)
      micLoaded <= 1'b0;
    else if (micValid && micReady && rxCsIsIdle) 
      micLoaded <= 1'b1;
    else if (((regPtr==lastMicPtr) && ccmpOutValid && ccmpOutReady))
      micLoaded <= 1'b0;
  end

// sequential logic 
  always @(posedge pClk or negedge nPRst)
  begin
    if (!nPRst)
    begin
      regPtr        <= 4'h0;
      ccmpOutValid  <= 1'b0; 
      lastBlock     <= 1'b0; 
      micLoadedDly  <= 1'b0; 
    end
    else if (!nSRst)
    begin
      regPtr        <= 4'h0;
      ccmpOutValid  <= 1'b0; 
      lastBlock     <= 1'b0; 
      micLoadedDly  <= 1'b0; 
    end
    else
    begin
      regPtr        <= nextregPtr;
      ccmpOutValid  <= nextccmpOutValid; 
      lastBlock     <= nextlastBlock; 
      micLoadedDly  <= micLoaded;
    end
  end

  always @(posedge pClk or negedge nPRst) 
  begin
    if (!nPRst) 
      outReg <= 128'h0;
    else if (!nSRst)
      outReg <= 128'h0;
    else  
      outReg <= nextoutReg;
  end

// Reduce the original payload by one logic
  assign newpayloadLen  =  payloadLen - 4'h1;

// Last MIC Pointer, computed from the AES key length
  always @*
  begin
     case ({cipherType,cipherLen})
     {CCMP,2'd0}: lastMicPtr = 4'd7;  // MIC Length= 8bytes
     {CCMP,2'd1}: lastMicPtr = 4'd11; // MIC Length=12bytes
     default:     lastMicPtr = 4'd15; // MIC Length=16bytes
     endcase
  end

//ccmOutReady
  assign ccmOutReady    = ~ccmpOutValid;

//micReady
  assign micReady       = ~ccmpOutValid;

//ccmpOutData
  assign ccmpOutData    = outReg[7:0];

//ccmpOutLast_p
  assign ccmpOutLast_p  = (rxCsIsIdle) ? (regPtr==lastMicPtr) & micLoaded : lastBlock & !nextlastBlock;

endmodule

//------------------- END OF FILE ----------------//
