////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//   This block is the top level module for CCMP, GCMP, BIP-CMAC & BIP-GMAC
//   implementation.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -----------------------------------------------------------------------------
//                                                                          
// 
// 
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

module ccmp_gcmp(
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          hard_rst_n,              // Hardware Reset
   input  wire          soft_rst_n,              // Software Reset
                                                                  
   //$port_g Control interface
   input  wire          encrypt,                 // Encrypt/Decrypt
   input  wire          init_p,                  // Initialization pulse
   input  wire    [2:0] mode,                    // Encrypt/Decrypt mode
   output wire          busy,                    // GCM is busy         
   input  wire  [239:0] aad,                     // Additional Authenticated Data
   input  wire   [15:0] aad_len,                 // AAD Length                   
   input  wire  [103:0] nonce,                   // NONCE                        
   input  wire   [15:0] data_len,                // Plain/Crypted Data length    
   input  wire  [255:0] aes_key,                 // AES Key
   input  wire    [1:0] aes_key_len,             // AES Key Length
   output wire          mic_passed_p,            // Data is authenticated        
   output wire          mic_failed_p,            // Data is not authenticated    

   //$port_g Data Interface
   input  wire  [127:0] data_in,                 // Plain/Encrypted data input
   input  wire   [15:0] data_in_byte_en,         // Data input byte enable
   input  wire          data_in_last,            // Last data input
   input  wire          data_in_valid,           // Data input is valid
   output reg           data_in_ready,           // ready to receive data
   output reg   [127:0] data_out,                // Plain/Encrypted data out
   output reg           data_out_valid,          // Data output is valid
   input  wire          data_out_ready,          // Data output ready
   output reg   [127:0] mic_out,                 // MIC output
   output reg           mic_out_valid,           // MIC output is valid
   input  wire          mic_out_ready,           // MIC output ready

   //$port_g Context interface
   input  wire  [793:0] context_in,              // Context input
   input  wire          context_in_valid,        // Context input is valid
   output wire  [793:0] context_out,             // Context output

   //$port_g Debug ports
   output wire   [31:0] ccm_debug,               // CCM Debug port
   output wire   [31:0] gcm_debug                // GCM Debug port
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
localparam 
//   NULL=3'b000,
//   WEP =3'b001,
//   TKIP=3'b010,
   CCMP=3'b011,
//   WAPI=3'b100,
   GCMP=3'b101;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
// CCM Controls
reg           ccm_init_p;
wire          ccm_busy;
wire    [1:0] ccm_mic_len;
wire          ccm_mic_passed_p;
wire          ccm_mic_failed_p;

// GCM Controls
reg           gcm_init_p;
wire          gcm_busy;
wire          gcm_mic_passed_p;
wire          gcm_mic_failed_p;

// AES
reg  [127:0]  aes0_in;
reg           aes0_in_valid;
wire [127:0]  aes0_out;
wire          aes0_out_valid;

reg  [127:0]  aes1_in;
reg           aes1_in_valid;
wire [127:0]  aes1_out;
wire          aes1_out_valid;

// CCM AES
wire [127:0]  ccm_aes_mic_in;
wire          ccm_aes_mic_in_valid;
wire [127:0]  ccm_aes_mic_out;
wire          ccm_aes_mic_out_valid;

wire [127:0]  ccm_aes_crt_in;
wire          ccm_aes_crt_in_valid;
wire [127:0]  ccm_aes_crt_out;
wire          ccm_aes_crt_out_valid;

// GCM AES
wire [127:0]  gcm_aes_in;
wire          gcm_aes_in_valid;
wire [127:0]  gcm_aes_out;
wire          gcm_aes_out_valid;

`ifdef  RW_GCMP_EN
// GCM GHASH
wire          gcm_ghash_enable;
wire  [127:0] gcm_ghash_h;
wire  [127:0] gcm_ghash_in;
wire          gcm_ghash_in_valid;
wire  [127:0] gcm_ghash_out;
wire          gcm_ghash_out_valid;
`endif//RW_GCMP_EN

// CCM Data
wire  [127:0] ccm_data_in;
wire   [15:0] ccm_data_in_byte_en;
wire          ccm_data_in_last;
wire          ccm_data_in_valid;
wire          ccm_data_in_ready;
wire  [127:0] ccm_data_out;
wire          ccm_data_out_valid;
wire          ccm_data_out_ready;
wire  [127:0] ccm_mic_out;
wire          ccm_mic_out_valid;
wire          ccm_mic_out_ready;

// GCM Data
wire  [127:0] gcm_data_in;
wire   [15:0] gcm_data_in_byte_en;
wire          gcm_data_in_last;
wire          gcm_data_in_valid;
wire          gcm_data_in_ready;
wire  [127:0] gcm_data_out;
wire          gcm_data_out_valid;
wire          gcm_data_out_ready;
wire  [127:0] gcm_mic_out;
wire          gcm_mic_out_valid;
wire          gcm_mic_out_ready;

// CCM Context
wire  [396:0] ccm_context_in;
wire          ccm_context_in_valid;
wire  [396:0] ccm_context_out;

// GCM Context
wire  [396:0] gcm_context_in;
wire          gcm_context_in_valid;
wire  [396:0] gcm_context_out;

// Context input
wire    [2:0] context_mode;
wire  [255:0] context_aes_key;
wire    [1:0] context_aes_key_len;
wire   [15:0] context_aad_len;
wire  [103:0] context_nonce;
wire   [15:0] context_data_len;

// Context registers
reg     [2:0] mode_reg;
reg   [255:0] aes_key_reg;
reg     [1:0] aes_key_len_reg;
reg    [15:0] aad_len_reg;
reg   [103:0] nonce_reg;
reg    [15:0] data_len_reg;


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// CCM Instantiation
//******************************************************************************
ccm u_ccm(
   //Clock and reset
   .clk               (clk                   ),
   .hard_rst_n        (hard_rst_n            ),
   .soft_rst_n        (soft_rst_n            ),
   //CCM AES MIC
   .aes_mic_in        (ccm_aes_mic_in        ),
   .aes_mic_in_valid  (ccm_aes_mic_in_valid  ),
   .aes_mic_out       (ccm_aes_mic_out       ),
   .aes_mic_out_valid (ccm_aes_mic_out_valid ),
   //CCM AES CRT
   .aes_crt_in        (ccm_aes_crt_in        ),
   .aes_crt_in_valid  (ccm_aes_crt_in_valid  ),
   .aes_crt_out       (ccm_aes_crt_out       ),
   .aes_crt_out_valid (ccm_aes_crt_out_valid ),
   //CONTROL
   .encrypt           (encrypt               ),
   .init_p            (ccm_init_p            ),
   .busy              (ccm_busy              ),
   .aad               (aad                   ),
   .aad_len           (aad_len_reg           ),
   .nonce             (nonce_reg             ),
   .data_len          (data_len_reg          ),
   .mic_len           (ccm_mic_len           ),
   .mic_passed_p      (ccm_mic_passed_p      ),
   .mic_failed_p      (ccm_mic_failed_p      ),
   //Data interface
   .data_in           (ccm_data_in           ),
   .data_in_byte_en   (ccm_data_in_byte_en   ),
   .data_in_last      (ccm_data_in_last      ),
   .data_in_valid     (ccm_data_in_valid     ),
   .data_in_ready     (ccm_data_in_ready     ),
   .data_out          (ccm_data_out          ),
   .data_out_valid    (ccm_data_out_valid    ),
   .data_out_ready    (ccm_data_out_ready    ),
   .mic_out           (ccm_mic_out           ),
   .mic_out_valid     (ccm_mic_out_valid     ),
   .mic_out_ready     (ccm_mic_out_ready     ),
   //Context interface
   .context_in        (ccm_context_in        ),
   .context_in_valid  (ccm_context_in_valid  ),
   .context_out       (ccm_context_out       ),
   //Debug Port
   .debug             (ccm_debug             )
);


`ifdef  RW_GCMP_EN
//******************************************************************************
// GCM Instantiation
//******************************************************************************
gcm u_gcm(
   //Clock and reset
   .clk              (clk                    ),
   .hard_rst_n       (hard_rst_n             ),
   .soft_rst_n       (soft_rst_n             ),
   //AES interface
   .aes_in           (gcm_aes_in             ),
   .aes_in_valid     (gcm_aes_in_valid       ),
   .aes_out          (gcm_aes_out            ),
   .aes_out_valid    (gcm_aes_out_valid      ),
   //GHASH interface
   .ghash_enable     (gcm_ghash_enable       ),
   .ghash_h          (gcm_ghash_h            ),
   .ghash_in         (gcm_ghash_in           ),
   .ghash_in_valid   (gcm_ghash_in_valid     ),
   .ghash_out        (gcm_ghash_out          ),
   .ghash_out_valid  (gcm_ghash_out_valid    ),
   //Control interface
   .encrypt          (encrypt                ),
   .init_p           (gcm_init_p             ),
   .busy             (gcm_busy               ),
   .aad              (aad                    ),
   .aad_len          (aad_len_reg            ),
   .nonce            (nonce_reg[103:8]       ),
   .data_len         (data_len_reg           ),
   .mic_passed_p     (gcm_mic_passed_p       ),
   .mic_failed_p     (gcm_mic_failed_p       ),
   //Data interface
   .data_in          (gcm_data_in            ),
   .data_in_byte_en  (gcm_data_in_byte_en    ),
   .data_in_last     (gcm_data_in_last       ),
   .data_in_valid    (gcm_data_in_valid      ),
   .data_in_ready    (gcm_data_in_ready      ),
   .data_out         (gcm_data_out           ),
   .data_out_valid   (gcm_data_out_valid     ),
   .data_out_ready   (gcm_data_out_ready     ),
   .mic_out          (gcm_mic_out            ),
   .mic_out_valid    (gcm_mic_out_valid      ),
   .mic_out_ready    (gcm_mic_out_ready      ),
   //Context interface
   .context_in       (gcm_context_in[268:0]  ),
   .context_in_valid (gcm_context_in_valid   ),
   .context_out      (gcm_context_out[268:0] ),
   //Debug Port
   .debug            (gcm_debug              )
);
`else //RW_GCMP_EN
assign gcm_aes_in         = 128'b0;
assign gcm_aes_in_valid   = 1'b0;
assign gcm_busy           = 1'b0;
assign gcm_mic_passed_p   = 1'b0;
assign gcm_mic_failed_p   = 1'b0;
assign gcm_data_in_ready  = 1'b0;
assign gcm_data_out       = 128'b0;
assign gcm_data_out_valid = 1'b0;
assign gcm_mic_out        = 128'b0;
assign gcm_mic_out_valid  = 1'b0;
assign gcm_context_out    = 397'b0;
assign gcm_debug          = 32'b0;
`endif//RW_GCMP_EN


//******************************************************************************
// AES instantiation
//******************************************************************************
aes u_aes0( 
   //Clocks and Resets
   .pClk          (clk             ),
   .nPRst         (hard_rst_n      ),
   .nSRst         (soft_rst_n      ),
   //Control
   .abort_p       (1'b0            ),
   .aesKey        (aes_key_reg     ),
   .aesKeyLen     (aes_key_len_reg ),
   .aesInData     (aes0_in         ),
   .aesInValid    (aes0_in_valid   ),
   .aesOutData    (aes0_out        ),
   .aesOutValid_p (aes0_out_valid  )
);

aes u_aes1( 
   //Clocks and Resets
   .pClk          (clk             ),
   .nPRst         (hard_rst_n      ),
   .nSRst         (soft_rst_n      ),
   //Control
   .abort_p       (1'b0            ),
   .aesKey        (aes_key_reg     ),
   .aesKeyLen     (aes_key_len_reg ),
   .aesInData     (aes1_in         ),
   .aesInValid    (aes1_in_valid   ),
   .aesOutData    (aes1_out        ),
   .aesOutValid_p (aes1_out_valid  )
);


`ifdef  RW_GCMP_EN
//******************************************************************************
// GHASH instantiation
//******************************************************************************
gcm_ghash u_gcm_ghash( 
   // Clock and reset
   .clk              (clk                     ),
   .hard_rst_n       (hard_rst_n              ),
   .soft_rst_n       (soft_rst_n              ),
   // data interface
   .data_in          (gcm_ghash_in            ),
   .data_in_valid    (gcm_ghash_in_valid      ),
   .data_out         (gcm_ghash_out           ),
   .data_out_valid   (gcm_ghash_out_valid     ),
   // Context interface
   .context_in       (gcm_context_in[396:269] ),
   .context_in_valid (gcm_context_in_valid    ),
   // Control interface
   .enable           (gcm_ghash_enable        ),
   .init_p           (gcm_init_p              ),
   .h                (gcm_ghash_h             )
);
assign gcm_context_out[396:269] = gcm_ghash_out;
`endif//RW_GCMP_EN


//******************************************************************************
// CCM & GCM Controls
//******************************************************************************
// CCM & GCM are not active at the same time, output status can be ored
assign busy         = ccm_busy         | gcm_busy    /*must be removed->*/| ccm_init_p | gcm_init_p;
assign mic_passed_p = ccm_mic_passed_p | gcm_mic_passed_p;
assign mic_failed_p = ccm_mic_failed_p | gcm_mic_failed_p;

// CCM MIC Length depend of the AES Key length
assign ccm_mic_len  = aes_key_len_reg;

// Initialisation pulse
always @(posedge clk or negedge hard_rst_n)
begin
   if (!hard_rst_n)
   begin
      ccm_init_p <= 1'b0;
      gcm_init_p <= 1'b0;
   end
   else if (!soft_rst_n)
   begin
      ccm_init_p <= 1'b0;
      gcm_init_p <= 1'b0;
   end
   else
   begin
      ccm_init_p <= init_p & mode==CCMP;
      gcm_init_p <= init_p & mode==GCMP;
   end
end


//******************************************************************************
// AES Controls
//******************************************************************************
always @*
begin
   case (mode_reg)
   CCMP:
   begin
      aes0_in       = ccm_aes_mic_in;
      aes0_in_valid = ccm_aes_mic_in_valid;
      aes1_in       = ccm_aes_crt_in;
      aes1_in_valid = ccm_aes_crt_in_valid;
   end
   GCMP:
   begin
      aes0_in       = gcm_aes_in;
      aes0_in_valid = gcm_aes_in_valid;
      aes1_in       = 128'h0;
      aes1_in_valid = 1'b0;
   end
   default:
   begin
      aes0_in       = 128'h0;
      aes0_in_valid = 1'b0;
      aes1_in       = 128'h0;
      aes1_in_valid = 1'b0;
   end
   endcase
end

assign ccm_aes_mic_out       = mode_reg==CCMP ? aes0_out       : 128'h0;
assign ccm_aes_mic_out_valid = mode_reg==CCMP ? aes0_out_valid : 1'b0;
assign ccm_aes_crt_out       = mode_reg==CCMP ? aes1_out       : 128'h0;
assign ccm_aes_crt_out_valid = mode_reg==CCMP ? aes1_out_valid : 1'b0;

assign gcm_aes_out           = mode_reg==GCMP ? aes0_out       : 128'h0;
assign gcm_aes_out_valid     = mode_reg==GCMP ? aes0_out_valid : 1'b0;


//******************************************************************************
// Data
//******************************************************************************
always @*
begin
   case (mode_reg)
   CCMP:
   begin
      data_in_ready  = ccm_data_in_ready;
      data_out       = ccm_data_out;
      data_out_valid = ccm_data_out_valid;
      mic_out        = ccm_mic_out;
      mic_out_valid  = ccm_mic_out_valid;
   end
   GCMP:
   begin
      data_in_ready  = gcm_data_in_ready;
      data_out       = gcm_data_out;
      data_out_valid = gcm_data_out_valid;
      mic_out        = gcm_mic_out;
      mic_out_valid  = gcm_mic_out_valid;
   end
   default:
   begin
      data_in_ready  = 1'b0;
      data_out       = 128'h0;
      data_out_valid = 1'b0;
      mic_out        = 128'h0;
      mic_out_valid  = 1'b0;
   end
   endcase
end

assign ccm_data_in         = mode_reg==CCMP ? data_in         : 128'h0;
assign ccm_data_in_byte_en = mode_reg==CCMP ? data_in_byte_en : 16'h0;
assign ccm_data_in_last    = mode_reg==CCMP ? data_in_last    : 1'b0;
assign ccm_data_in_valid   = mode_reg==CCMP ? data_in_valid   : 1'b0;
assign ccm_data_out_ready  = mode_reg==CCMP ? data_out_ready  : 1'b0;
assign ccm_mic_out_ready   = mode_reg==CCMP ? mic_out_ready   : 1'b0;
                                                              
assign gcm_data_in         = mode_reg==GCMP ? data_in         : 128'h0;
assign gcm_data_in_byte_en = mode_reg==GCMP ? data_in_byte_en : 16'h0;
assign gcm_data_in_last    = mode_reg==GCMP ? data_in_last    : 1'b0;
assign gcm_data_in_valid   = mode_reg==GCMP ? data_in_valid   : 1'b0;
assign gcm_data_out_ready  = mode_reg==GCMP ? data_out_ready  : 1'b0;
assign gcm_mic_out_ready   = mode_reg==GCMP ? mic_out_ready   : 1'b0;


//******************************************************************************
// Context logic
//******************************************************************************
// Context registers
always @(posedge clk or negedge hard_rst_n)
begin
   if (!hard_rst_n)
   begin
      mode_reg        <= 3'h0;
      aes_key_reg     <= 256'h0;
      aes_key_len_reg <= 2'h0;
      aad_len_reg     <= 16'h0;
      nonce_reg       <= 104'h0;
      data_len_reg    <= 16'h0;
   end
   else if (!soft_rst_n)
   begin
      mode_reg        <= 3'h0;
      aes_key_reg     <= 256'h0;
      aes_key_len_reg <= 2'h0;
      aad_len_reg     <= 16'h0;
      nonce_reg       <= 104'h0;
      data_len_reg    <= 16'h0;
   end
   else if (init_p && !ccm_busy && !gcm_busy)
   begin
      // Initialisation
      mode_reg        <= mode;
      aes_key_reg     <= aes_key;
      aes_key_len_reg <= aes_key_len;
      aad_len_reg     <= aad_len;
      nonce_reg       <= nonce;
      data_len_reg    <= data_len;
   end
   else if (context_in_valid && !ccm_busy && !gcm_busy)
   begin
      // Context Load
      mode_reg        <= context_mode;
      aes_key_reg     <= context_aes_key;
      aes_key_len_reg <= context_aes_key_len;
      aad_len_reg     <= context_aad_len;
      nonce_reg       <= context_nonce;
      data_len_reg    <= context_data_len;
   end
end

// Context input
assign {context_mode,
        context_aes_key,
        context_aes_key_len,
        context_aad_len,
        context_nonce,
        context_data_len} = context_in[396:0];

assign ccm_context_in     = context_in[793:397];
assign gcm_context_in     = context_in[793:397];

assign ccm_context_in_valid = context_mode==CCMP & context_in_valid;
assign gcm_context_in_valid = context_mode==GCMP & context_in_valid;

// Context output
assign context_out[396:0] = {mode_reg,
                             aes_key_reg,
                             aes_key_len_reg,
                             aad_len_reg,
                             nonce_reg,
                             data_len_reg};
assign context_out[793:397] = context_mode==CCMP ? ccm_context_out :
                                                   gcm_context_out;

endmodule
////////////////////////////////////////////////////////////////////////////////
// END OF FILE
////////////////////////////////////////////////////////////////////////////////
