//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $RCSmodulefile   : Encryption Engine Top
// $Author          : jvanthou
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 0.1                                                             
// $Date: 4-Dec-2008                                                                
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of Encryption Engine module
//                    
// Simulation Notes : 
//    For simulation, five defines are available
//
//    RX_BUFFER_ADDR_WIDTH : Configurable address width of Encryption
//                           Buffer
//    WEP           : Enable WEP encryption / decryption 
//    TKIP          : Enable TKIP encryption / decryption 
//    CCMP          : Enable CCMP encryption / decryption 
//    RW_SIMU_ON       : which creates string signals to display the FSM states 
//                    on the waveform viewers
//
// Synthesis Notes  :
//
//     false_path from ()
//                to   (macCoreClk)
//
// Application Note : 
//     All the signals are coming from MAC Controller clock domain. Hence, no
//     resynchronizations are needed.
//
// Simulator        :
//     
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module encryptEngTop (
      //$port_g Clock and Reset interface
      input  wire                               macCoreClk,            // macCore clock signal (freerunning clock)
      input  wire                               macCryptClk,           // macCrypt clock signal (gated clock for crypto engine)
      input  wire                               macWTClk,              // Faster Clock for WEP/TKIP engine

      input  wire                               macCoreClkHardRst_n,   // Active low hard reset signal synchronized to the macCoreClk
      input  wire                               macWTClkHardRst_n,     // Active low hard reset signal synchronized to the macWTClk
      
      input  wire                               macCoreClkSoftRst_n,   // Active low soft reset signal synchronized to the macCoreClk
      input  wire                               macWTClkSoftRst_n,     // Active low soft reset signal synchronized to the macWTClk

      output wire                               macCryptClkEn,         // Enable MAC Crypt Clock
      output wire                               macWTClkEn,            // Enable Faster Clock for WEP/TKIP engine

      //$port_g Control and status registers
      input  wire [1:0]                         wt2CryptClkRatio,      // Indicate the clock ratio between WT and macCrypt Clock
      input  wire                               activeClkGating,       // Active Clock Gating This bit is set by SW to turn on 
                                                                       // active clock gating in HW. When reset, the HW does not 
                                                                       // perform active clock gating, i.e. does not turn off 
                                                                       // clocks to save power in ACTIVE state.

      //$port_g Key search Engine
      input  wire [127:0]                       cryptoKey,             // Key to be used for encryption or decryption 
                                                                       // operations
`ifdef  RW_KEY_EXTENDED
      input  wire [127:0]                       cryptoIntKey,          // Integrity key to be used
                                                                       // for encryption or  
                                                                       // decryption operations
`endif//RW_KEY_EXTENDED
      input  wire [2:0]                         cipherType,            // Cipher Type Input from TX or RX Controller 
      input  wire [1:0]                         cipherLen,             // Cipher Length Input from TX or RX Controller
      input  wire [1:0]                         sppKSR,                // HT A MSDU Capability


      //$port_g TX Controller Interface
      input  wire                               txCsIsIdle,            // State machine information from TX Controller
      input  wire                               initEncrTxCntrl_p,     // Trigger signal to crypto engines for starting Encryption
      input  wire                               txError_p,             // Indication of error signal from TX Controller 
                                                                       // during transmission operation
      input  wire                               txEnd_p,               // Pulse to indicate end of transmission

      input  wire [15:0]                        txPayLoadLen,          // Transmit payload length in TX Frame
      input  wire [15:0]                        txFrameControl,        // Frame Control field of received frame
      input  wire [47:0]                        txAddr1,               // Address1 field of TX frame
      input  wire [47:0]                        txAddr2,               // txAddr2 of the system
      input  wire [47:0]                        txAddr3,               // txAddr3 of the system
      input  wire [47:0]                        txAddr4,               // txAddr4 of the system
      input  wire [15:0]                        txSeqControl,          // Sequence control field
      input  wire [47:0]                        txSfc,                 // Transmit SFC field in TX Frame
      input  wire                               txAddress4Pres,        // Tx flag indicating address 4 is present
      input  wire                               txQoSFrame,            // Tx flag indicating the current frame is a QoS Frame
      input  wire [3:0]                         txTID,                 // Tx priority information for QoS frames
      input  wire                               txQoS7,                // HT spp Tx QoS bit 7
    `ifdef RW_WAPI_EN
      input  wire [15:0]                        txQosCF,               // QoS control field
      input  wire [7:0]                         txKeyIdx,              // Key index
      input  wire [127:0]                       txPN,                  // Packet number
    `endif // RW_WAPI_EN
      input  wire [47:0]                        txTkipSeqCntr,         // TKIP Sequence Counter for TKIP encryption
      input  wire [23:0]                        txWEPIV,               // WEP IV from SW
      input  wire                               txHTMode,              // HT Flag

      output wire                               initDone_p,            // Indicates crypto Engine has finished initialization 
      output reg                                initSBOXDone,          // Flag for WEP and TKIP init done

      //$port_g Encryption Data Input Interface
      input  wire [7:0]                         plainDataIn,           // Plain (unencrypted) data from TX Controller
      input  wire                               plainDataInValid,      // Validating signal for plain TX data
      input  wire                               plainDataInEnd_p,      // Indicates the last bytes of plain TX data
      output reg                                plainDataInReady,      // Flow control signal indicating readiness of crypto  
                                                                       // module for accepting input data 
      //$port_g Encryption Data Output Interface
      output reg [7:0]                          encryptedDataOut,      // Encrypted data output for writing into FCS block
      output reg                                encryptedDataOutValid, // Validating signal for the output encrypted data. 
      output reg                                encryptedDataOutEnd_p, // Indicates the last bytes of encrypted data (MIC or ICV)
      input  wire                               encryptedDataOutReady, // Flow control signal indicating that the next block is
                                                                       // ready for accepting data from crypto module

      //$port_g RX Controller Interface
      input  wire                               rxCsIsIdle,            // Signal to indicate whether core is in TX or RX mode
      input  wire                               initEncrRxCntrl_p,     // Trigger signal to crypto engines for starting 
                                                                       // encryption / decryption of the input data

      input  wire [15:0]                        rxFrameControl,        // Frame Control field of received frame
      input  wire [47:0]                        rxAddr1,               // Address1 field of received frame
      input  wire [47:0]                        rxAddr2,               // Address2 field of received frame
      input  wire [47:0]                        rxAddr3,               // Address3 field of received frame
      input  wire [47:0]                        rxAddr4,               // Address4 field of received frame
      input  wire                               rxAddress4Pres,        // Rx flag indicating address 4 is present
      input  wire                               rxQoSFrame,            // Rx flag indicating the current frame is a QoS Frame
      input  wire [3:0]                         rxTID,                 // Rx priority information for QoS frames
      input  wire                               rxQoS7,                // HT spp Rx QoS bit 7
      input  wire [15:0]                        rxSeqControl,          // Sequence control field of received frame
    `ifdef RW_WAPI_EN
      input  wire [15:0]                        rxQosCF,               // QoS control field of received frame
      input  wire [7:0]                         rxKeyIdx,              // Key index of received frame
      input  wire [127:0]                       rxPN,                  // Packet number of received frame
    `endif // RW_WAPI_EN
      input  wire [23:0]                        rxInitVector,          // Initialization Vector extracted from frame received
      input  wire [47:0]                        rxTkipSeqCntr,         // TKIP Sequence Counter for TKIP decryption 
      input  wire [15:0]                        rxPayLoadLen,          // RX frame length including MIC and FCS
      input  wire                               rxHTMode,              // HT Flag

      output wire                               decryptPassed_p,       // Indicates decryption was a success
      output wire                               decryptFailed_p,       // Indicates decryption was a failure
      

  
      input  wire                               rxError_p,             // Trigger indicating received frame should
                                                                       // not be deciphered and can be discarded
      //$port_g Decryption Data Input Interface
      input  wire                               encryptedDataInEnd_p,  // Signal indicating last byte of received
                                                                       // frame fed to MAC used by encryption
                                                                       // receive controller 
      input  wire [7:0]                         encryptedDataIn,       // data bus from RX Controller
      input  wire                               encryptedDataInValid,  // data bus Valid from RX Controller
      output reg                                encryptedDataInReady,  // ready to accept new data from RX Controller

      //$port_g Decryption Data Output Interface
      output reg  [7:0]                         plainDataOut,          // Decrypted (plain) data output to RX 
                                                                       // Controller for writing into RX FIFO
      output reg                                plainDataOutValid,     // Validating signal for the output plain 
                                                                       // data after decryption
      input  wire                               plainDataOutReady,     // Flow control signal indicating readiness of
                                                                       // Rx FIFO to accept the decrypted data
      output reg                                plainDataOutEnd_p,     // End of Decrypted data pulse

      //$port_g DPRAM Interface [s-Box for WEP-TKIP engine]
      input  wire  [15:0]                       sboxMemRdData,         // data bus from internal RAM
      output wire   [6:0]                       sboxMemWrAddr,         // address bus to internal RAM, port write
      output wire   [6:0]                       sboxMemRdAddr,         // address bus to internal RAM, port read
      output wire  [15:0]                       sboxMemWrData,         // data bus to internal RAM
      output wire   [1:0]                       sboxMemWrEn,           // write enable signal for internal RAM
      output wire                               sboxMemRdEn,           // read enable signal for internal RAM


      //$port_g Debug Interface
      input  wire                               debugUseDefaultKeyKSR, // debug ports
      input  wire                               debugKeyIndexTrig,     // debug ports
      input  wire  [5:0]                        debugKeyIndexKSR,      // debug ports
`ifdef RW_WAPI_EN
      output wire  [15:0]                       debugPortWapi,         // debug ports
`endif //RW_WAPI_EN
      output wire  [31:0]                       debugPortEncryptionEngine // debug ports
      );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// Cipher Type definition
localparam 
   PLAIN=3'b000,
   WEP =3'b001,
   TKIP=3'b010,
   CCMP=3'b011,
   WAPI=3'b100,
   GCMP=3'b101;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////



wire                  encrypt;  // Set when encryption is performed.
                                // Reset when decryption is performed


wire  [15:0]          payloadLen;       
wire  [15:0]          frameControl;       // Frame Control field for wpi block
wire  [47:0]          addr1;         
wire  [47:0]          addr2;
wire  [47:0]          addr3;
wire  [47:0]          addr4;
wire  [47:0]          sfc;            
wire                  address4Pres;
wire                  qosFrame;
wire   [3:0]          tid;
wire                  qos7;
wire  [15:0]          seqControl;       // Sequence Control Field for wpi block
wire  [23:0]          iv;               // WEP IV
wire                  htMode;           // HT Flag


wire                  txRxErrorP;

// WEP/TKIP

wire                  wepTkipEn;

wire                  initPRNG_p;
wire                  initPRNGSynch_p;
wire                  initPRNGDone_p;
wire                  initSBoxIndexDone_p; // signal indicating end of SBOX Index init process

wire [47:0]           tkipSeqCntr;

wire                  wepTkipPassed_p;       // WEP/TKIP decryption succesfull pulse 
wire                  wepTkipFailed_p;       // WEP/TKIP decryption failed pulse 

wire [7:0]            wepTkipDataIn;
wire                  wepTkipDataInValid_p;
wire                  wepTkipDataInReady;
wire                  wepTkipDataInLast_p;
wire [7:0]            wepTkipDataOut;
wire                  wepTkipDataOutValid_p;
wire                  wepTkipDataOutReady;
wire                  wepTkipDataOutLast_p;


// CCMP/GCMP

reg                   initCG_p;
wire                  ccmpEn;
wire                  gcmpEn;
wire                  txCG;                // Indication for CCMP encryption
wire                  rxCG;

wire [1:0]            ccmp_cs;
wire                  txrxBufferFull;
wire                  ccmpIsIdle;

wire                  ccmppayloadEnd_p;


reg  [15:0]           ccmpPayloadLen;
wire [47:0]           ccmpSfc;

reg                   ccmpPassed_p;          // CCMP decryption succesfull pulse 
reg                   ccmpFailed_p;          // CCMP decryption failed pulse 


wire                  nextccmpRegFull;
wire                  ccmpRegFull;
wire                  micPassed_p;
wire                  micFailed_p;

wire                  ccmpOutValid_p;
wire                  ccmpOutLast_p;
wire [7:0]            ccmpOutData;

wire [7:0]            ccmpInData;
wire                  ccmpInValid;
wire                  encrTxCCMPBusy;        // Indicates to txController that the ccmp is busy and stop reading the txFifo

// WAPI 

`ifdef RW_WAPI_EN
wire                  abortWPI_p;
wire                  wpiMode;               // Flag indicating the WPI Mode (0:Encryption, 1:Decryption)
wire                  wpiMICPassed;          // Flag indicating MIC check successful

wire [7:0]            wpiDataIn;             // input data bus to WPI engine
reg                   wpiDataInValid;        // Valid signal for the input WPI data
wire [7:0]            wpiDataOut;            // Output data bus from WPI engine
wire                  wpiDataOutReady;       // 
wire                  wpiDataOutLast_p;      // 
wire                  wpiMICOutLast_p;
wire                  wpiDataOutValid_p;     // Valid signal for the output WPI data
wire                  wpiRegFull;            // Flag indicating WPI input register is full and cannot accept new dataIn
wire                  wpiRegAlmostFull;      // Flag indicating WPI input register is almost full and can only accept one byte.
wire                  encrTxWAPIBusy;        // Indicates to txController that the wapi is busy and stop reading the txFifo
reg                   initWPI_p;             // Initialization trigger pulse for WPI
wire                  wpiIsIdle;             // Flag indicating that the WPI module is back to IDLE state

wire [15:0]           wpiQosCF;              // QoS control field for wpi block 
wire [15:0]           wpiPduLength;          // Payload length for WPI block
wire [7:0]            wpiKeyIdx;             // WPI Key Index
wire [127:0]          wpiPN;                 // WPI Packet Number
reg                   payloadEnd_p;          // Pulse for payload end indication


wire                  wapiPassed_p;          // WAPI decryption succesfull pulse 
wire                  wapiFailed_p;          // WAPI decryption failed pulse 
wire                  wpiBusy;

wire                  txWAPI;                // WAPI encryption active indication
wire                  rxWAPI;                // WAPI decryption active indication
wire                  wapiEn;                // WAPI active indication
wire                  wpiDone_p;             // Pulse indicating completion of the processing



`endif //RW_WAPI_EN

// Debug

reg  [7:0]            debugCnt;
reg  [31:0]           debugLshReg;
reg                   debugValid;
reg  [5:0]            debugKeyindex;
wire [31:0]           debugWepTkip;


//////////////////////////////////////////////////////////////////////////////
// Beginning of Logic part
//////////////////////////////////////////////////////////////////////////////



// Instanciation of encryptEngClkCntl
// Name of the instance : u_encryptEngClkCntl
// Name of the file containing this module : encryptEngClkCntl.v
encryptEngClkCntl u_encryptEngClkCntl (
    .macCoreClk             (macCoreClk),
    .macCoreClkHardRst_n    (macCoreClkHardRst_n),
    .macCoreClkSoftRst_n    (macCoreClkSoftRst_n),
    .macCryptClkEn          (macCryptClkEn),
    .macWTClkEn             (macWTClkEn),
    .txCsIsIdle             (txCsIsIdle),
    .initSBoxIndexDone_p    (initSBoxIndexDone_p),
    .rxCsIsIdle             (rxCsIsIdle),
    .cipherType             (cipherType),
    .activeClkGating        (activeClkGating)
    );




assign decryptPassed_p = ccmpPassed_p | 
                       `ifdef RW_WAPI_EN
                         wapiPassed_p | 
                       `endif // RW_WAPI_EN
                         wepTkipPassed_p;

assign decryptFailed_p = ccmpFailed_p | 
                       `ifdef RW_WAPI_EN
                         wapiFailed_p | 
                       `endif // RW_WAPI_EN
                         wepTkipFailed_p;


assign encrypt = (!txCsIsIdle) ? 1'b1 : 1'b0;

//******************************************************************************
// Tx/Rx Mux
//******************************************************************************

assign payloadLen   = rxCsIsIdle ? txPayLoadLen   : ccmpPayloadLen;
assign addr1        = rxCsIsIdle ? txAddr1        : rxAddr1;
assign addr2        = rxCsIsIdle ? txAddr2        : rxAddr2;
assign addr3        = rxCsIsIdle ? txAddr3        : rxAddr3;
assign addr4        = rxCsIsIdle ? txAddr4        : rxAddr4;
assign sfc          = rxCsIsIdle ? txSfc          : ccmpSfc;
assign address4Pres = rxCsIsIdle ? txAddress4Pres : rxAddress4Pres;
assign qosFrame     = rxCsIsIdle ? txQoSFrame     : rxQoSFrame;
assign tid          = rxCsIsIdle ? txTID          : rxTID;
assign qos7         = rxCsIsIdle ? txQoS7         : rxQoS7;
assign frameControl = rxCsIsIdle ? txFrameControl : rxFrameControl;            // Frame Control
assign seqControl   = rxCsIsIdle ? txSeqControl   : rxSeqControl;
assign htMode       = rxCsIsIdle ? txHTMode       : rxHTMode;


// ------------- Encrypted output with flow control logic starts ---------------


// Encryption Multiplexing between Crypto Cores.
always @* 
begin
  if (!txCsIsIdle)
  begin
    case (cipherType)
      WEP, TKIP :
      begin
        plainDataInReady        = wepTkipDataInReady;
        encryptedDataOut        = wepTkipDataOut;
        encryptedDataOutEnd_p   = wepTkipDataOutLast_p;
        encryptedDataOutValid   = wepTkipDataOutValid_p;
      end
      CCMP, GCMP :
      begin
        plainDataInReady        = !encrTxCCMPBusy;
        encryptedDataOut        = ccmpOutData;
        encryptedDataOutEnd_p   = ccmpOutLast_p;
        encryptedDataOutValid   = ccmpOutValid_p;
      end
    `ifdef RW_WAPI_EN
      WAPI :
      begin
        plainDataInReady        = !encrTxWAPIBusy;
        encryptedDataOut        = wpiDataOut;
        encryptedDataOutEnd_p   = wpiDataOutLast_p;
        encryptedDataOutValid   = wpiDataOutValid_p;
      end
    `endif //RW_WAPI_EN
      default :
      begin
        plainDataInReady        = 1'b0;
        encryptedDataOut        = 8'b0;
        encryptedDataOutEnd_p   = 1'b0;
        encryptedDataOutValid   = 1'b0;
      end
    endcase
  end
  else
  begin
    plainDataInReady        = 1'b0;
    encryptedDataOut        = 8'b0;
    encryptedDataOutEnd_p   = 1'b0;
    encryptedDataOutValid   = 1'b0;
  end  
end  

// Decryption Multiplexing between Crypto Cores.
always @* 
begin
  if (!rxCsIsIdle)
  begin
    case (cipherType)
      WEP, TKIP :
      begin
        plainDataOut         = wepTkipDataOut;
        plainDataOutEnd_p    = wepTkipDataOutLast_p;
        plainDataOutValid    = wepTkipDataOutValid_p;
        encryptedDataInReady = wepTkipDataInReady;
     
      end
      CCMP, GCMP :
      begin
        plainDataOut         = ccmpOutData;
        plainDataOutEnd_p    = ccmpOutLast_p;
        plainDataOutValid    = ccmpOutValid_p;
        encryptedDataInReady = !encrTxCCMPBusy;
      end
    `ifdef RW_WAPI_EN
      WAPI :
      begin
        plainDataOut         = wpiDataOut;
        plainDataOutEnd_p    = wpiMICOutLast_p;
        plainDataOutValid    = wpiDataOutValid_p;
        encryptedDataInReady = !wpiBusy;
      end
    `endif //RW_WAPI_EN
      default :
      begin
        plainDataOut         = 8'b0;
        plainDataOutEnd_p    = 1'b0;
        plainDataOutValid    = 1'b0;
        encryptedDataInReady = 1'b0;
      end
    endcase
  end
  else
  begin
    plainDataOut      = 8'b0;
    plainDataOutEnd_p = 1'b0;
    plainDataOutValid = 1'b0;
    encryptedDataInReady = 1'b0;
  end  
end  



//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// WEP/TKIP
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
assign wepTkipEn = ((cipherType == WEP) || (cipherType == TKIP));

// Multiplexed signal for Initializing Pseudo Random No. Generator module
assign initPRNG_p = (wepTkipEn) ? initEncrRxCntrl_p | initEncrTxCntrl_p : 1'b0;

// Multiplexed signal for TKIP Sequence counters
assign tkipSeqCntr = rxCsIsIdle ? txTkipSeqCntr : rxTkipSeqCntr;

// Multiplexed signal for WEP/TKIP IV counters
assign iv = rxCsIsIdle ? txWEPIV : rxInitVector;


assign wepTkipDataInLast_p = (wepTkipEn) ? rxCsIsIdle ? plainDataInEnd_p : encryptedDataInEnd_p : 1'b0;

assign wepTkipDataInValid_p = (wepTkipEn) ? rxCsIsIdle ? plainDataInValid : encryptedDataInValid : 1'b0;

assign wepTkipDataIn = (wepTkipEn) ? rxCsIsIdle ? plainDataIn : encryptedDataIn : 8'b0;

assign wepTkipDataOutReady = (wepTkipEn) ? rxCsIsIdle ? encryptedDataOutReady : plainDataOutReady : 1'b0;


always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    initSBOXDone <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0 || txEnd_p)
    initSBOXDone <= 1'b0;
  else 
    if(initDone_p == 1'b1)
      initSBOXDone <= 1'b1;
    else if(initSBOXDone && (txCsIsIdle || txEnd_p))
      initSBOXDone <= 1'b0;
end


assign initDone_p = (wepTkipEn) ? initPRNGDone_p : 1'b1;



// --------------------- Module instantiations start -------------------

toggleSynch1 U_initPRNGSynch_p (
    .genClk (macCryptClk),
    .synClk (macWTClk),
    .genhardReset_n (macCoreClkHardRst_n),
    .synhardReset_n (macWTClkHardRst_n),
    .dataIn (initPRNG_p),
    .dataOut (initPRNGSynch_p)
);

   

// Instanciation of wepTkipWrapper
// Name of the instance : u_wepTkipWrapper
// Name of the file containing this module : wepTkipWrapper.v
wepTkipWrapper u_wepTkipWrapper (
    .macCryptClk            (macCryptClk),
    .macWTClk               (macWTClk),
    .macCoreClkHardRst_n    (macCoreClkHardRst_n),
    .macCoreClkSoftRst_n    (macCoreClkSoftRst_n),
    .macWTClkHardRst_n      (macWTClkHardRst_n),
    .macWTClkSoftRst_n      (macWTClkSoftRst_n),
    .encrypt                (encrypt),
    .init_p                 (initPRNGSynch_p),
    .initDone_p             (initPRNGDone_p),
    .initSBoxIndexDone_p    (initSBoxIndexDone_p),
    .wt2CryptClkRatio       (wt2CryptClkRatio),
    .cipherType             (cipherType),
    .cipherLen              (cipherLen[0]),
    .cipherKey              (cryptoKey),
    .iv                     (iv),
    .txAddr                 (addr2),
    .tkipSeqCntr            (tkipSeqCntr),
    .icvPassed_p            (wepTkipPassed_p),
    .icvFailed_p            (wepTkipFailed_p),
    .dataIn                 (wepTkipDataIn),
    .dataInValid            (wepTkipDataInValid_p),
    .dataInReady            (wepTkipDataInReady),
    .dataInLast             (wepTkipDataInLast_p),
    .dataOut                (wepTkipDataOut),
    .dataOutValid           (wepTkipDataOutValid_p),
    .dataOutReady           (wepTkipDataOutReady),
    .dataOutLast            (wepTkipDataOutLast_p),
    .sboxMemRdAddr          (sboxMemRdAddr),
    .sboxMemRdData          (sboxMemRdData),          
    .sboxMemWrAddr          (sboxMemWrAddr),          
    .sboxMemRdEn            (sboxMemRdEn),            
    .sboxMemWrEn            (sboxMemWrEn),            
    .sboxMemWrData          (sboxMemWrData),      
    .debug                  (debugWepTkip)
    );


//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// CCMP
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////

// Tx or Rx Error Pulse
assign txRxErrorP   = txError_p | rxError_p;

assign txrxBufferFull = (ccmpEn || gcmpEn) ? rxCsIsIdle ? !encryptedDataOutReady : !plainDataOutReady : 1'b0;

assign ccmppayloadEnd_p = plainDataInEnd_p | encryptedDataInEnd_p;

assign ccmpInValid = (ccmpEn || gcmpEn) ? rxCsIsIdle ? plainDataInValid : encryptedDataInValid : 1'b0;

assign ccmpInData = (ccmpEn || gcmpEn) ? rxCsIsIdle ? plainDataIn : encryptedDataIn : 8'b0;


assign encrTxCCMPBusy      = ((ccmpEn || gcmpEn) &&  
                              (ccmpRegFull)) ? 1'b1 : 1'b0;



//////////////////////////////
// CCMP
//////////////////////////////
ccmp u_ccmp (
    // Inputs
    .macCoreClk         (macCoreClk),
    .macCryptClk        (macCoreClk),
    .nPRst              (macCoreClkHardRst_n),
    .nSRst              (macCoreClkSoftRst_n),

    .rxCsIsIdle         (rxCsIsIdle),
    .txCsIsIdle         (txCsIsIdle),
    .initCG_p           (initCG_p),
    .txRxErrorP         (txRxErrorP),
`ifdef  RW_KEY_EXTENDED
    .temporalKey        ({cryptoIntKey,cryptoKey}),
`else //RW_KEY_EXTENDED
    .temporalKey        ({128'h0      ,cryptoKey}),
`endif//RW_KEY_EXTENDED
    .cipherLen          (cipherLen),
    .cipherType         (cipherType),
    .payloadLen         (payloadLen),
    .frameControl       (frameControl),
    .addr1              (addr1),
    .addr2              (addr2),
    .addr3              (addr3),
    .addr4              (addr4),
    .seqControl         (seqControl),
    .sfc                (sfc),
    .address4Pres       (address4Pres),
    .qosFrame           (qosFrame),
    .tid                (tid),
    .qos7               (qos7),
    .htMode             (htMode),
    .sppKSR             (sppKSR),

    // CCMP Data Input Ports
    .payloadEnd_p       (ccmppayloadEnd_p),
    .ccmpInData         (ccmpInData),
    .ccmpInValid        (ccmpInValid),
    .ccmpRegFull        (ccmpRegFull),
    .nextccmpRegFull    (nextccmpRegFull),

    // CCMP Data output Ports
    .txRxBufferFull     (txrxBufferFull),
    .ccmpOutData        (ccmpOutData),
    .ccmpOutValid_p     (ccmpOutValid_p),
    .ccmpOutLast_p      (ccmpOutLast_p),

    .micPassed_p        (micPassed_p),
    .micFailed_p        (micFailed_p),

    .ccmpIsIdle         (ccmpIsIdle),
    .ccmp_cs            (ccmp_cs)
    );
   
assign ccmpEn   = (cipherType == CCMP);
assign gcmpEn   = (cipherType == GCMP);
assign rxCG     = ~(rxCsIsIdle) & (ccmpEn | gcmpEn);
assign txCG     = ~(txCsIsIdle) & (ccmpEn | gcmpEn);

   // Initialize pulse to CCMP/GCMP engine - set after valid key is available
always @(posedge macCryptClk or negedge macCoreClkHardRst_n) begin
  if (macCoreClkHardRst_n == 1'b0)
    initCG_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    initCG_p <= 1'b0;
  else 
    initCG_p <= (initEncrRxCntrl_p || initEncrTxCntrl_p) & (ccmpEn | gcmpEn);
end

always @(posedge macCryptClk or negedge macCoreClkHardRst_n) begin
  if (macCoreClkHardRst_n == 1'b0)
    ccmpPassed_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    ccmpPassed_p <= 1'b0;
  else
    ccmpPassed_p <= micPassed_p;
end

always @(posedge macCryptClk or negedge macCoreClkHardRst_n) begin
  if (macCoreClkHardRst_n == 1'b0)
    ccmpFailed_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    ccmpFailed_p <= 1'b0;
  else
    ccmpFailed_p <= micFailed_p;
end

// Encrypted payload length = rxPayLoadLen - MIC
always @*
begin
   if (cipherType==CCMP & cipherLen==2'd0)
      //CCMP-128 (MIC = 8)
      ccmpPayloadLen = rxPayLoadLen - 16'd8;
   else
      //GCMP-128, GCMP-256 or CCMP-256 (MIC = 16)
      ccmpPayloadLen = rxPayLoadLen - 16'd16;
end

// Construct SFC from IV and Ext IV
assign ccmpSfc = {rxInitVector[7:0],
                  rxInitVector[15:8],
                  rxTkipSeqCntr[23:16],
                  rxTkipSeqCntr[31:24],
                  rxTkipSeqCntr[39:32],
                  rxTkipSeqCntr[47:40]};


`ifdef RW_WAPI_EN
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// WAPI
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////

assign wapiEn = (cipherType == WAPI);
assign rxWAPI = wapiEn && !rxCsIsIdle;
assign txWAPI = wapiEn && !txCsIsIdle;

assign wpiMode = !encrypt;

assign wpiQosCF        = rxCsIsIdle ? txQosCF : rxQosCF;
assign wpiKeyIdx       = rxCsIsIdle ? txKeyIdx : rxKeyIdx;
assign wpiPduLength    = rxCsIsIdle ? txPayLoadLen : rxPayLoadLen - 16'd16; // rxPayload_length - MIC(16bytes) - FCS (4 bytes)
assign wpiPN           = rxCsIsIdle ? txPN : rxPN;

assign encrTxWAPIBusy  = (!wpiIsIdle && wapiEn && wpiRegFull) ? 1'b1 : 1'b0;

assign wpiDataIn       = wapiEn ? rxCsIsIdle ? plainDataIn : encryptedDataIn : 8'b0;

always @*
begin
  if (wapiEn && wpiIsIdle != 1'b1)
  begin
    wpiDataInValid  = rxCsIsIdle ? plainDataInValid & !wpiBusy : encryptedDataInValid & !wpiBusy;
    payloadEnd_p    = rxCsIsIdle ? plainDataInEnd_p : encryptedDataInEnd_p;
  end
  else
  begin
    wpiDataInValid  = 1'b0;
    payloadEnd_p    = 1'b0;
  end
end

assign wpiDataOutReady = wapiEn ? rxCsIsIdle ? encryptedDataOutReady : plainDataOutReady : 1'b0;

assign wpiBusy = wapiEn ? wpiRegFull : 1'b0;

// Initialize pulse to WAPI engine - set after valid key is available
always @(posedge macCryptClk or negedge macCoreClkHardRst_n) begin
  if (macCoreClkHardRst_n == 1'b0)
    initWPI_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    initWPI_p <= 1'b0;
  else 
    initWPI_p <= (initEncrRxCntrl_p || initEncrTxCntrl_p ) && ((rxCsIsIdle && txWAPI) || rxWAPI);
end



assign abortWPI_p = 1'b0;
assign wapiPassed_p = wpiDone_p & wpiMICPassed;
assign wapiFailed_p = wpiDone_p & !wpiMICPassed;


wpi u_wpi (
  // Clock and Resets
  .wpiClk           (macCryptClk),         
  .nPRst            (macCoreClkHardRst_n),
  .nSRst            (macCoreClkSoftRst_n),
  // Controls signals
  .initWPI_p        (initWPI_p),
  .wpiEncryptionKey (cryptoKey),
  .wpiIntegrityKey  (cryptoIntKey),
  .abortWPI_p       (abortWPI_p),
  .wpiDone_p        (wpiDone_p),
  .wpiMICPassed     (wpiMICPassed),
  .wpiMode          (wpiMode),
  .wpiIsIdle        (wpiIsIdle),
  .payloadEnd_p     (payloadEnd_p),
  // MAC Header parameters
  .frameControl     (frameControl),
  .addr1            (addr1),
  .addr2            (addr2),
  .addr3            (addr3),
  .addr4            (addr4),
  .address4Pres     (address4Pres),
  .qosFrame         (qosFrame),
  .qosCF            (wpiQosCF),
  .seqControl       (seqControl),
  .keyIdx           (wpiKeyIdx),
  .pduLength        (wpiPduLength),
  .pn               (wpiPN),
  // Data Interface
  .wpiDataIn        (wpiDataIn),
  .wpiDataInValid   (wpiDataInValid),
  .wpiRegFull       (wpiRegFull),
  .wpiRegAlmostFull (wpiRegAlmostFull),
  .wpiDataOut       (wpiDataOut),
  .wpiDataOutValid_p(wpiDataOutValid_p),
  .wpiDataOutLast_p (wpiDataOutLast_p),
  .wpiDataOutReady  (wpiDataOutReady),
  .wpiMICOutLast_p  (wpiMICOutLast_p),
  // Debug port
  .wpiDebug         (debugPortWapi)
);

`endif //RW_WAPI_EN


//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// DEBUG
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////


   // --------------------- Debug Ports -----------------
   assign debugPortEncryptionEngine[31:29] = cipherType;
   assign debugPortEncryptionEngine[28:27] = cipherLen;
   assign debugPortEncryptionEngine[26]    = debugUseDefaultKeyKSR;
   assign debugPortEncryptionEngine[25]    = 1'b0;
   assign debugPortEncryptionEngine[24]    = decryptFailed_p || rxError_p ||txError_p ;
   assign debugPortEncryptionEngine[23]    = decryptPassed_p;
   assign debugPortEncryptionEngine[22:18] = {debugValid,debugLshReg[3:0]};
   assign debugPortEncryptionEngine[17]    = (!rxCsIsIdle) ? encryptedDataIn    : (!txCsIsIdle) ? encryptedDataOutValid : 1'b0;
   assign debugPortEncryptionEngine[16]    = (!rxCsIsIdle) ? plainDataOutValid : (!txCsIsIdle) ? plainDataInValid : 1'b0;
   assign debugPortEncryptionEngine[15:12] = debugWepTkip[7:4];
   assign debugPortEncryptionEngine[11:10] = 2'b0;
   assign debugPortEncryptionEngine[9:8]   = ccmp_cs;
   assign debugPortEncryptionEngine[7:4]   = debugWepTkip[3:0];
   assign debugPortEncryptionEngine[3]     = 1'b0;
   assign debugPortEncryptionEngine[2]     = initEncrTxCntrl_p || initEncrRxCntrl_p || initDone_p || initCG_p;
   assign debugPortEncryptionEngine[1]     = initPRNGSynch_p;
   assign debugPortEncryptionEngine[0]     = 1'b0;

  always @(posedge macCoreClk, negedge macCoreClkHardRst_n)
  begin
    if(macCoreClkHardRst_n == 1'b0)
    begin
      debugCnt                 <=   8'h0;
      debugLshReg              <=  32'h0;
      debugValid               <=   1'b0;
      debugKeyindex            <=   6'h0;
    end
    else
    begin
      if(debugKeyIndexTrig)
        debugKeyindex <= debugKeyIndexKSR;

      // Rx
      if(!rxCsIsIdle || !txCsIsIdle)
      begin
        // IV extIV Key  
        if((initDone_p && cipherType != CCMP) || initEncrRxCntrl_p || (initCG_p && ((cipherType == CCMP) || (cipherType == GCMP))) || (debugCnt != 8'h0))
        begin
          if(debugCnt < 8'd57)
          begin
            debugCnt <= debugCnt + 8'h1;
          end
          else
          begin
            debugCnt <= 8'h0;
          end

          case(debugCnt)
            8'd1 :
            begin
                debugLshReg[15:0]  <= {8'hFF,debugKeyindex[3:0],2'h0,debugKeyindex[5:4]};
                debugValid         <= 1'b1;
            end

            8'd3 :
            begin
                debugValid          <= 1'b0;
            end

            8'd4 :
            begin
                debugValid          <= 1'b1;
                if(!rxCsIsIdle)
                  debugLshReg[31:0] <= {8'hFF,rxInitVector[23:0]};
                else
                  debugLshReg[31:0] <= {8'hFF,txWEPIV[23:0]};
              end

            8'd10:
            begin
                debugValid          <= 1'b0;
              end

            8'd11:
            begin
              if(cipherType > 3'd1)
              begin
                debugValid          <= 1'b1;
                if(!rxCsIsIdle)
                  debugLshReg[31:0] <= rxTkipSeqCntr[31:0];
                else
                  debugLshReg[31:0] <= txTkipSeqCntr[31:0];
              end
            end

            8'd19:
            begin
              if(cipherType > 3'd1)
              begin
                debugValid          <= 1'b1;
                if(!rxCsIsIdle)
                  debugLshReg[31:0] <= {16'h0,rxTkipSeqCntr[47:32]};
                else
                  debugLshReg[31:0] <= {16'h0,txTkipSeqCntr[47:32]};
              end
            end

            8'd23:
            begin
              debugValid            <= 1'b0;
            end
 
            8'd24:
            begin
              debugLshReg[31:0]  <= {cryptoKey[31:0]};
              debugValid         <= 1'b1;
            end

            8'd32:
            begin
              debugLshReg[31:0]  <= {cryptoKey[63:32]};
              debugValid         <= 1'b1;
            end

            8'd40:
            begin
              if(cipherType > 3'd1)
              begin
                debugLshReg[31:0]  <= {cryptoKey[95:64]};
                debugValid         <= 1'b1;
              end
            end

            8'd41:
            begin
              debugLshReg         <= {4'h0,debugLshReg[31:4]};
              debugValid          <= (cipherLen>2'b0);
            end

            8'd48:
            begin
              if(cipherType > 3'd1)
              begin
                debugLshReg[31:0]  <= {cryptoKey[127:96]};
                debugValid         <= (cipherLen>2'b0);
              end
            end

            8'd56:
            begin
              debugValid          <= 1'b0;
            end

            default : 
            begin
              debugLshReg  <= {4'h0,debugLshReg[31:4]};
              debugValid   <= debugValid;
            end
          endcase
        end
        else
        begin
          debugLshReg                       <=  32'h0;
          debugValid                        <=   1'b0;
        end
      end
      // Idle
      else
      begin
        debugCnt                         <=   8'h0;
        debugLshReg                      <=  32'h0;
        debugValid                       <=   1'b0;
      end
    end
  end
endmodule
