////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : WEP/TKIP Wrapper 
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module wepTkipWrapper( 
   //$port_g Clock and reset
   input  wire          macCryptClk,             // Clock
   input  wire          macWTClk,                // WEP/TKIP Clock
   input  wire          macCoreClkHardRst_n,     // Active low hard reset signal synchronized to the macCoreClk
   input  wire          macWTClkHardRst_n,       // Active low hard reset signal synchronized to the macWTClk
   input  wire          macCoreClkSoftRst_n,     // Active low soft reset signal synchronized to the macCoreClk
   input  wire          macWTClkSoftRst_n,       // Active low soft reset signal synchronized to the macWTClk
 
   //$port_g WEP/TKIP Control interface
   input  wire          encrypt,                 // Encryption/Decryption Mode
   input  wire          init_p,                  // Initialisation pulse
   output wire          initDone_p,              // WEP/TKIP initialisation is done
   output wire          initSBoxIndexDone_p,     // Indicate the completion of SBOX initialisation
   input  wire    [1:0] wt2CryptClkRatio,        // Indicate the clock ratio between WT and macCrypt Clock
   input  wire    [2:0] cipherType,              // Cipher Type
   input  wire          cipherLen,               // Cipher Length
   input  wire  [127:0] cipherKey,               // Cipher Key
   input  wire   [23:0] iv,                      // WEP Initialisation Vector
   input  wire   [47:0] txAddr,                  // Transmitter Address
   input  wire   [47:0] tkipSeqCntr,             // TKIP Sequence Counter
   output reg           icvPassed_p,             // Correct ICV
   output reg           icvFailed_p,             // Incorrect ICV

   //$port_g WEP/TKIP Data Interface
   input  wire    [7:0] dataIn,                  // Data input
   input  wire          dataInValid,             // Data input is valid
   output wire          dataInReady,             // Ready to receive data
   input  wire          dataInLast,              // Last data input
   output wire    [7:0] dataOut,                 // Data output
   output wire          dataOutValid,            // Data output is valid
   input  wire          dataOutReady,            // Ready to receive data
   output wire          dataOutLast,             // Last data output

   //$port_g SBOX Memory Interface
   input  wire   [15:0] sboxMemRdData,           // data bus from internal RAM
   output wire    [6:0] sboxMemWrAddr,           // address bus to internal RAM, port write
   output wire    [6:0] sboxMemRdAddr,           // address bus to internal RAM, port read
   output wire   [15:0] sboxMemWrData,           // data bus to internal RAM
   output wire    [1:0] sboxMemWrEn,             // write enable signal for internal RAM
   output wire          sboxMemRdEn,             // read enable signal for internal RAM


   //$port_g Debug Port
   output wire   [31:0] debug                    // Debug Port
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
//$fsm_sd wt_cs
localparam
   INIT_SBOX       = 4'd0,
   IDLE            = 4'd1,
   INIT            = 4'd2,
   DATA_FIRST_WAIT = 4'd3,
   DATA_FIRST      = 4'd4,
   DATA_WAIT       = 4'd5,
   DATA            = 4'd6,
   DATA_LAST       = 4'd7,
   ICV_WAIT        = 4'd8,
   ICV             = 4'd9,
   ICV_CHECK       = 4'd10;

localparam 
   //NULL = 3'b000,
   //WEP  = 3'b001,
   TKIP = 3'b010;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg    [3:0] wt_cs;                    // WEP/TKIP Current state
reg    [3:0] wt_ns;                    // WEP/TKIP Next state

reg          encryptReg;               // Encryption/Decryption Register
reg    [2:0] cipherTypeReg;            // Cipher Type Register
reg          cipherLenReg;             // Cipher Length Register
reg  [127:0] cipherKeyReg;             // Cipher Key Register
reg   [23:0] ivReg;                    // WEP Initialisation Vector Register
reg   [47:0] txAddrReg;                // Transmitter Address Register
reg   [47:0] tkipSeqCntrReg;           // TKIP Sequence Counter Register

wire         init_prng_p;              // PRNG Intialisation request
wire         cryptoKeyValid;           // Crypto Key is valid
wire         txWepIVValid;             // WEP IV is valid
wire         cipherTypeTkip;           // Cipher Type is TKIP
wire         icvStart_p;               // ICV initialisation request
wire         icvEnable_p;              // ICV enable pulse
wire         icvDrain_p;               // ICV start draining pulse
wire         icvCrcOk;                 // ICV CRC is correct
reg    [2:0] icvCnt;                   // ICV Byte counter
wire         prnStrobe_p;              // next PRN generation request
wire         prnOutValid_p;            // next PRN is generated
reg          prnOutValid;              // prnOutValid_p registered
wire         prnOutReady;              // PRN output is ready

wire   [7:0] byteIn;                   // WEP/TKIP Byte Input 
wire         byteInLast;               // WEP/TKIP Byte Input Last Flag 
wire         byteInValid;              // WEP/TKIP Byte Input is valid
reg          byteInValidDel;           // WEP/TKIP Byte Input valid delayed
reg          byteInLastDel;            // WEP/TKIP Byte Input Last delayed
wire         byteInReady;              // WEP/TKIP Byte Input ready

wire   [7:0] byteOut;                  // WEP/TKIP Byte Output
reg          byteOutLast;              // WEP/TKIP Byte Output Last flag
wire         byteOutValid;             // WEP/TKIP Byte Output is valid
wire         byteOutReady;             // WEP/TKIP Byte Output ready

`ifdef SBOX_RDATA_CAPT
wire   [5:0] prng_cs;                  // PRNG block current state
`else 
wire   [4:0] prng_cs;                  // PRNG block current state
`endif


////////////////////////////////////////////////////////////////////////////////
// Function Definitions
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// WEP/TKIP Instantiation
//******************************************************************************
wepTkip u_wepTkip(
  .macWTClk                 (macWTClk                        ),
  .macWTClkHardRst_n        (macWTClkHardRst_n               ),
  .macWTClkSoftRst_p        (!macWTClkSoftRst_n              ),
  .macCoreClk               (macCryptClk                     ),
  .hardRstBbClk_n           (macCoreClkHardRst_n             ),
  .rxCsIsIdle               (encryptReg                      ),
  .wt2CryptClkRatio         (wt2CryptClkRatio                ),
  .cryptoKey                (cipherKeyReg                    ),
  .cipherLen                (cipherLenReg                    ),
  .cipherType               (cipherTypeReg                   ),
  .txTKIP                   (cipherTypeTkip                  ),
  .initPRNG_p               (init_prng_p                     ),
  .initSBoxIndexDone_p      (initSBoxIndexDone_p             ),
  .cryptoKeyValid           (cryptoKeyValid                  ),
  .txWEPIVValid             (txWepIVValid                    ),
  .initDone_p               (initDone_p                      ),
  .txWEPIV                  (ivReg                           ),
  .rxInitVector             (ivReg                           ),
  .txRxAddress              (txAddrReg                       ),
  .macAddr                  (txAddrReg                       ),
  .tkipSeqCntr              (tkipSeqCntrReg                  ),
  .icvStart_p               (icvStart_p                      ),
  .prnStrobe_p              (prnStrobe_p                     ),
  .icvEnable_p              (icvEnable_p                     ),
  .prnOutValid_p            (prnOutValid_p                   ),
  .icvSDrain_p              (icvDrain_p                      ),
  .plainText                (byteIn                          ),
  .dataFromBuffer           (byteIn                          ),
  .dataFromXor              (byteOut                         ),
  .rxError_p                (1'b0                            ),
  .icvCRCOk                 (icvCrcOk                        ),
  //SBOX Memory interface
  .sboxMemRdAddr            (sboxMemRdAddr                   ),        
  .sboxMemRdData            (sboxMemRdData                   ),        
  .sboxMemWrAddr            (sboxMemWrAddr                   ),        
  .sboxMemRdEn              (sboxMemRdEn                     ),          
  .sboxMemWrEn              (sboxMemWrEn                     ),          
  .sboxMemWrData            (sboxMemWrData                   ),    
  //Debug Ports
  .prngCs                   (prng_cs                         ),
  .debug_initDonemacWTClk_p (/*Open*/                        )
);


//******************************************************************************
// Configuration Registers
//******************************************************************************
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
   begin
      encryptReg       <= 1'b0;
      cipherTypeReg    <= 3'h0;
      cipherLenReg     <= 1'b0;
      cipherKeyReg     <= 128'h0;
      ivReg            <= 24'h0;
      txAddrReg        <= 48'h0;
      tkipSeqCntrReg   <= 48'h0;
   end
   else if (!macCoreClkSoftRst_n)
   begin
      encryptReg       <= 1'b0;
      cipherTypeReg    <= 3'h0;
      cipherLenReg     <= 1'b0;
      cipherKeyReg     <= 128'h0;
      ivReg            <= 24'h0;
      txAddrReg        <= 48'h0;
      tkipSeqCntrReg   <= 48'h0;
   end
   else if (init_p)
   begin
      encryptReg       <= encrypt;
      cipherTypeReg    <= cipherType;
      cipherLenReg     <= cipherLen;
      cipherKeyReg     <= cipherKey;
      ivReg            <= iv;
      txAddrReg        <= txAddr;
      tkipSeqCntrReg   <= tkipSeqCntr;
   end
end


//******************************************************************************
// WEP/TKIP Wrapper FSM State Logic 
//******************************************************************************
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
      wt_cs <= IDLE;
   else if (!macCoreClkSoftRst_n)
      wt_cs <= IDLE;
   else
      wt_cs <= wt_ns;
end

always @* 
begin  
   case(wt_cs)
   INIT_SBOX:
      //$fsm_s In INIT_SBOX state, the SBOX initialisation is requested
      //$fsm_t The state machine goes to IDLE state
      wt_ns = IDLE;

   IDLE:
      //$fsm_s In IDLE state, the state machine waits initialization request
      if (init_p)
         //$fsm_t When initialisation is requested, the state machine goes to
         //INIT state
         wt_ns = INIT;
      else
         //$fsm_t While initialisation is not requested, the state machine stays
         //in IDLE state
         wt_ns = IDLE;

   INIT:
      //$fsm_s In INIT state, the state machine waits end of initialisation
      if (initDone_p)
         //$fsm_t When initialisation is done, the state machine goes to
         //DATA_FIRST_WAIT state
         wt_ns = DATA_FIRST_WAIT;
      else
         //$fsm_t While initialisation is not done, the state machine stays in
         //INIT state
         wt_ns = INIT;

   DATA_FIRST_WAIT:
      //$fsm_s In DATA_FIRST_WAIT state, the state machine waits first data
      if (!encryptReg || byteInValidDel)
	  begin    
          //$fsm_t When first input data is ready or in decryption mode, the 
          //state machine goes to DATA_FIRST state
          wt_ns = DATA_FIRST;
      end
      else
         //$fsm_t While first input data is not ready, the state machine stays
         //in DATA_FIRST_WAIT state
         wt_ns = DATA_FIRST_WAIT;

   DATA_FIRST:
      //$fsm_s In DATA_FIRST state, 1st data is provided to WEP/TKIP
      //$fsm_t The state machine goes to DATA_WAIT state
      if (byteInLastDel && encryptReg)
	    wt_ns = ICV_WAIT;
      else
        wt_ns = DATA_WAIT;
      

   DATA_WAIT:
      //$fsm_s In DATA_WAIT state, the state machine waits PRN & buffers ready
      if (prnOutReady && byteOutReady && byteInValidDel && byteInLast)
         //$fsm_t When PRN is ready, data buffers are ready and last DATA, the
         //state machine goes to DATA_LAST state
         wt_ns = DATA_LAST;
      else if (prnOutReady && byteOutReady && byteInValidDel && !byteInLast)
         //$fsm_t When PRN is ready, data buffers are ready and not the last 
         //DATA, the state machine goes to DATA state
         wt_ns = DATA;
      else
         //$fsm_t While PRN is not ready or data buffers are not ready, the
         //state machine stays in DATA_WAIT state
         wt_ns = DATA_WAIT;

   DATA:
      //$fsm_s In DATA state, data is provided to WEP/TKIP
      //$fsm_t The state machine goes to DATA_WAIT state
      wt_ns = DATA_WAIT;

   DATA_LAST:
      //$fsm_s In DATA state, last data is provided to WEP/TKIP
      //$fsm_t The state machine goes to ICV_WAIT state
       wt_ns = ICV_WAIT;

   ICV_WAIT:
      //$fsm_s In ICV_WAIT state, the state machine waits PRN & buffers ready
      if (prnOutReady && byteOutReady && (byteInValidDel || encryptReg))
         //$fsm_t When PRN is ready, data buffers are ready, the state machine
         //goes to ICV state
         wt_ns = ICV;
      else
         //$fsm_t While PRN is not ready or data buffers are not ready, the
         //state machine stays in ICV_WAIT state
         wt_ns = ICV_WAIT;

   ICV:
      //$fsm_s In ICV state, the state machine waits last byte of the ICV 
      if ((icvCnt==3'd4) && prnOutReady && encryptReg)
         //$fsm_t When last byte of ICV is read, the state machine goes to
         //INIT_SBOX state
         wt_ns = INIT_SBOX;
      else if ((icvCnt==3'd3) && prnOutReady && !encryptReg)
         //$fsm_t When last byte of ICV is written, the state machine goes to
         //ICV_CHECK state
         wt_ns = ICV_CHECK;
      else
         //$fsm_t While not the last byte of ICV, the state machine goes in 
         //ICV_WAIT state
         wt_ns = ICV_WAIT;

   ICV_CHECK:
      //$fsm_s In ICV_CHECK state, the ICV is verified
      //$fsm_t the state machine goes to INIT_SBOX state
         wt_ns = INIT_SBOX;

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      wt_ns = INIT_SBOX; 
   // pragma coverage block = on
   endcase
end


//******************************************************************************
// WEP/TKIP controls
//******************************************************************************
assign init_prng_p      = (wt_cs==INIT_SBOX);
assign cryptoKeyValid   = (wt_cs==INIT);
assign txWepIVValid     = (wt_cs==INIT);
assign icvStart_p       = (wt_cs==INIT) & (wt_ns==DATA_FIRST_WAIT);
assign prnStrobe_p      = (wt_ns==DATA_FIRST)                                |
                          (wt_ns==DATA)                                      |
                          (wt_ns==DATA_LAST)                                 |
                          (wt_ns==ICV)        & (icvCnt!=3'd4) &  encryptReg |
                          (wt_ns==ICV)        & (icvCnt!=3'd3) & !encryptReg;
assign icvEnable_p      = (wt_ns==DATA_FIRST) & encryptReg |
                          (wt_ns==DATA)                    |
                          (wt_ns==DATA_LAST)               |
                          (wt_ns==ICV)        & icvCnt!=3'd4;
assign icvDrain_p       = ((wt_ns==DATA_LAST)  || ((wt_ns==DATA_FIRST) & byteInLastDel)) & encryptReg;

assign cipherTypeTkip = (cipherTypeReg==TKIP);

// Generate prnOutReady
//   prnOutValid_p pulse is saved, in case of data buffers are not ready 
//   during this pulse.
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
      prnOutValid <= 1'b0;
   else if (!macCoreClkSoftRst_n)
      prnOutValid <= 1'b0;
   else if (prnStrobe_p || (wt_cs==IDLE))
      prnOutValid <= 1'b0;
   else if (prnOutValid_p)
      prnOutValid <= 1'b1;
end
assign prnOutReady = prnOutValid_p | prnOutValid;


//******************************************************************************
// Input Data
//******************************************************************************
wepTkipBuffer #(
   .SIZE           (9)
) wepTkipBufferIn (
   //Clock and reset
   .clk            (macCryptClk),
   .hardRst_n      (macCoreClkHardRst_n),
   .softRst_n      (macCoreClkSoftRst_n),
   //Data interface
   .dataIn         ({dataInLast,dataIn}),
   .dataInValid    (dataInValid),
   .dataInReady    (dataInReady),
   .dataOut        ({byteInLast,byteIn}),
   .dataOutValid   (byteInValid),
   .dataOutReady   (byteInReady)
);

assign byteInReady = (wt_ns==DATA_FIRST) &  encryptReg             |
                     (wt_ns==DATA)                                 |
                     (wt_ns==DATA_LAST)                            |
                     (wt_ns==ICV)        & !encryptReg & (icvCnt!=3'd4);

// Delayed Valid Flag, this flag is set one cycle after valid flag.
//   In decryption, Data must be available one cycle before provided to the ICV.
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
      byteInValidDel <= 1'b0;
   else if (!macCoreClkSoftRst_n)
      byteInValidDel <= 1'b0;
   else if (byteInReady)
      byteInValidDel <= 1'b0;
   else
      byteInValidDel <= byteInValid;
end

// Delayed Valid Flag, this flag is set one cycle after valid flag.
//   In decryption, Data must be available one cycle before provided to the ICV.
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
      byteInLastDel <= 1'b0;
   else if (!macCoreClkSoftRst_n)
      byteInLastDel <= 1'b0;
   else
      byteInLastDel <= byteInLast;
end


//******************************************************************************
// Output Data
//******************************************************************************
// Last flag from input buffer registered.
//    In encryption, 1 data delay between in & out
//    In decryption, 1 clock cycle delay between in & out
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
      byteOutLast <= 1'b0;
   else if (!macCoreClkSoftRst_n)
      byteOutLast <= 1'b0;
//    else if (!encryptReg)
//       byteOutLast <= byteInLast;
   else if (!encryptReg && (icvCnt==3'd2) & (wt_cs==ICV))
      byteOutLast <= 1'b1;
   else if (encryptReg && (icvCnt==3'd3) & (wt_cs==ICV))
      byteOutLast <= 1'b1;
   else if (byteOutValid & byteOutReady)
      byteOutLast <= 1'b0;
end

assign byteOutValid = (wt_ns==DATA)      |
                      (wt_ns==DATA_LAST) |
                      (wt_ns==ICV)       ;//& encryptReg;

//assign byteOutLast  = (wt_ns==ICV) & (icvCnt==3'd3) & encryptReg) | byteInLastFF1;
wire [7:0] byteOutInt;
assign byteOutInt = (!encryptReg && (wt_ns==ICV)) ? byteIn : byteOut;

wepTkipBuffer #(
   .SIZE           (9)
) wepTkipBufferOut (
   //Clock and reset
   .clk            (macCryptClk),
   .hardRst_n      (macCoreClkHardRst_n),
   .softRst_n      (macCoreClkSoftRst_n),
   //Data interface
   .dataIn         ({byteOutLast,byteOutInt}),
   .dataInValid    (byteOutValid),
   .dataInReady    (byteOutReady),
   .dataOut        ({dataOutLast,dataOut}),
   .dataOutValid   (dataOutValid),
   .dataOutReady   (dataOutReady)
);


//******************************************************************************
// ICV Counter
//******************************************************************************
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
      icvCnt <= 3'd0;
   else if (!macCoreClkSoftRst_n)
      icvCnt <= 3'd0;
   else if (wt_cs==ICV)
      icvCnt <= icvCnt+3'd1;
   else if (wt_cs==IDLE)
      icvCnt <= 3'd0;
end


//******************************************************************************
// Status
//******************************************************************************
always @(posedge macCryptClk or negedge macCoreClkHardRst_n)
begin
   if (!macCoreClkHardRst_n)
   begin
      icvPassed_p <= 1'b0;
      icvFailed_p <= 1'b0;
   end
   else if (!macCoreClkSoftRst_n)
   begin
      icvPassed_p <= 1'b0;
      icvFailed_p <= 1'b0;
   end
   else
   begin
      icvPassed_p <= (wt_cs==ICV_CHECK) &  icvCrcOk;
      icvFailed_p <= (wt_cs==ICV_CHECK) & !icvCrcOk;
   end
end


//******************************************************************************
// Debug Port
//******************************************************************************
assign debug = {24'h0, wt_cs[3:0], prng_cs[3:0]};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef  RW_SIMU_ON
// ECU WT FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] wt_cs_str;
always @*
begin
  case (wt_cs)
    INIT_SBOX       : wt_cs_str = {"INIT_SBOX"};
    IDLE            : wt_cs_str = {"IDLE"};
    INIT            : wt_cs_str = {"INIT"};
    DATA_FIRST_WAIT : wt_cs_str = {"DATA_FIRST_WAIT"};
    DATA_FIRST      : wt_cs_str = {"DATA_FIRST"};
    DATA_WAIT       : wt_cs_str = {"DATA_WAIT"};
    DATA            : wt_cs_str = {"DATA"};
    DATA_LAST       : wt_cs_str = {"DATA_LAST"};
    ICV_WAIT        : wt_cs_str = {"ICV_WAIT"};
    ICV             : wt_cs_str = {"ICV"};
    ICV_CHECK       : wt_cs_str = {"ICV_CHECK"};
    default         : wt_cs_str = {"XXX"};
  endcase
end
`endif//RW_SIMU_ON


//////////////////////////////////////////////////////////
// System Verilog Assertions
//////////////////////////////////////////////////////////
`ifdef  RW_ASSERT_ON
`endif//RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
