//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Host Interface Multi Port Arbiter module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module hostIfMpa( 
   //$port_g Resets and Clocks
   input  wire               macPIClkHardRst_n,            // Active low hard reset signal
                                                           // synchronized to the macPIClk.
   input  wire               macPIClk,                     // Platform clock
   
   //$port_g Master host interface port 0 
   input  wire               m0HIFRead,                    // host interface read enable
   input  wire               m0HIFWrite,                   // host interface write enable
   input  wire        [31:0] m0HIFAddressIn,               // host interface address
   input  wire         [2:0] m0HIFSize,                    // host interface byte enable
   input  wire        [31:0] m0HIFWriteDataIn,             // host interface write data
   output wire        [31:0] m0HIFReadDataOut,             // host interface read data
   output wire               m0HIFReadDataValid,           // host interface read data valid
   output wire               m0HIFReady,                   // host interface next data
   output wire               m0HIFTransComplete,           // host interface transfer complete
   output wire               m0HIFError,                   // host interface error
   
`ifdef  RW_MUMIMO_SEC_USER1_EN
   //$port_g Master host interface port 1 
   input  wire               m1HIFRead,                    // host interface read enable
   input  wire               m1HIFWrite,                   // host interface write enable
   input  wire        [31:0] m1HIFAddressIn,               // host interface address
   input  wire         [2:0] m1HIFSize,                    // host interface byte enable
   input  wire        [31:0] m1HIFWriteDataIn,             // host interface write data
   output wire        [31:0] m1HIFReadDataOut,             // host interface read data
   output wire               m1HIFReadDataValid,           // host interface read data valid
   output wire               m1HIFReady,                   // host interface next data
   output wire               m1HIFTransComplete,           // host interface transfer complete
   output wire               m1HIFError,                   // host interface error
`endif//RW_MUMIMO_SEC_USER1_EN
   
`ifdef  RW_MUMIMO_SEC_USER2_EN
   //$port_g Master host interface port 2 
   input  wire               m2HIFRead,                    // host interface read enable
   input  wire               m2HIFWrite,                   // host interface write enable
   input  wire        [31:0] m2HIFAddressIn,               // host interface address
   input  wire         [2:0] m2HIFSize,                    // host interface byte enable
   input  wire        [31:0] m2HIFWriteDataIn,             // host interface write data
   output wire        [31:0] m2HIFReadDataOut,             // host interface read data
   output wire               m2HIFReadDataValid,           // host interface read data valid
   output wire               m2HIFReady,                   // host interface next data
   output wire               m2HIFTransComplete,           // host interface transfer complete
   output wire               m2HIFError,                   // host interface error
`endif//RW_MUMIMO_SEC_USER2_EN
   
`ifdef  RW_MUMIMO_SEC_USER3_EN
   //$port_g Master host interface port 3 
   input  wire               m3HIFRead,                    // host interface read enable
   input  wire               m3HIFWrite,                   // host interface write enable
   input  wire        [31:0] m3HIFAddressIn,               // host interface address
   input  wire         [2:0] m3HIFSize,                    // host interface byte enable
   input  wire        [31:0] m3HIFWriteDataIn,             // host interface write data
   output wire        [31:0] m3HIFReadDataOut,             // host interface read data
   output wire               m3HIFReadDataValid,           // host interface read data valid
   output wire               m3HIFReady,                   // host interface next data
   output wire               m3HIFTransComplete,           // host interface transfer complete
   output wire               m3HIFError,                   // host interface error
`endif//RW_MUMIMO_SEC_USER3_EN
   
   //$port_g Slave host interface port
   output wire               sHIFRead,                     // host interface read enable
   output wire               sHIFWrite,                    // host interface write enable
   output wire        [31:0] sHIFAddressIn,                // host interface address
   output wire         [2:0] sHIFSize,                     // host interface byte enable
   output wire        [31:0] sHIFWriteDataIn,              // host interface write data
   input  wire        [31:0] sHIFReadDataOut,              // host interface read data
   input  wire               sHIFReadDataValid,            // host interface read data valid
   input  wire               sHIFReady,                    // host interface next data
   input  wire               sHIFTransComplete,            // host interface transfer complete
   input  wire               sHIFError                     // host interface error
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
wire  [3:0] busRequest;      // Bus request
wire  [3:0] busRelease;      // Bus release
reg   [3:0] busGrant;        // Bus grant
reg   [3:0] busOwner;        // Bus owner
wire        busFree;         // Bus is free

wire  [3:0] priRequest;      // Priotity request
reg   [3:0] priUpdate;       // Priotity update enable
reg   [3:0] priTable[3:0];   // Priority table (0: highest priority)


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// request
////////////////////////////////////////////////////////////////////////////////
// Bus request
assign busRequest[0] = m0HIFRead | m0HIFWrite;

// Bus release
assign busRelease[0] = m0HIFTransComplete;

`ifdef  RW_MUMIMO_SEC_USER1_EN
assign busRequest[1] = m1HIFRead | m1HIFWrite;
assign busRelease[1] = m1HIFTransComplete;
`else //RW_MUMIMO_SEC_USER1_EN
assign busRequest[1] = 1'b0;
assign busRelease[1] = 1'b0;
`endif//RW_MUMIMO_SEC_USER1_EN

`ifdef  RW_MUMIMO_SEC_USER2_EN
assign busRequest[2] = m2HIFRead | m2HIFWrite;
assign busRelease[2] = m2HIFTransComplete;
`else //RW_MUMIMO_SEC_USER2_EN
assign busRequest[2] = 1'b0;
assign busRelease[2] = 1'b0;
`endif//RW_MUMIMO_SEC_USER2_EN

`ifdef  RW_MUMIMO_SEC_USER3_EN
assign busRequest[3] = m3HIFRead | m3HIFWrite;
assign busRelease[3] = m3HIFTransComplete;
`else //RW_MUMIMO_SEC_USER3_EN
assign busRequest[3] = 1'b0;
assign busRelease[3] = 1'b0;
`endif//RW_MUMIMO_SEC_USER3_EN

////////////////////////////////////////////////////////////////////////////////
// arbitration
////////////////////////////////////////////////////////////////////////////////
always @(posedge macPIClk , negedge macPIClkHardRst_n)
begin
   if(macPIClkHardRst_n==1'b0)
   begin
      busOwner     <= 4'b0;
       
      priTable[0]  <= 4'b0001;
      priTable[1]  <= 4'b0010;
      priTable[2]  <= 4'b0100;
      priTable[3]  <= 4'b1000;
   end
   else if (busFree==1'b1)
   begin
      // track which master owns the data read bus
      busOwner     <= busGrant;
      
      // priority queue
      if (priUpdate[0]==1'b1) priTable[0] <= priTable[1];
      if (priUpdate[1]==1'b1) priTable[1] <= priTable[2];
      if (priUpdate[2]==1'b1) priTable[2] <= priTable[3];
      if (priUpdate[3]==1'b1) priTable[3] <= busGrant;
   end
end
  
// determine when the bus is free
assign busFree = (~busOwner[0] | busRelease[0]) &
                 (~busOwner[1] | busRelease[1]) &
                 (~busOwner[2] | busRelease[2]) &
                 (~busOwner[3] | busRelease[3]) ;

// determine which entry of the priority table is request pending
assign priRequest[0] = (priTable[0]==4'b0001) & busRequest[0] |
                       (priTable[0]==4'b0010) & busRequest[1] |
                       (priTable[0]==4'b0100) & busRequest[2] |
                       (priTable[0]==4'b1000) & busRequest[3] ;
  
assign priRequest[1] = (priTable[1]==4'b0001) & busRequest[0] |
                       (priTable[1]==4'b0010) & busRequest[1] |
                       (priTable[1]==4'b0100) & busRequest[2] |
                       (priTable[1]==4'b1000) & busRequest[3] ;

assign priRequest[2] = (priTable[2]==4'b0001) & busRequest[0] |
                       (priTable[2]==4'b0010) & busRequest[1] |
                       (priTable[2]==4'b0100) & busRequest[2] |
                       (priTable[2]==4'b1000) & busRequest[3] ;

assign priRequest[3] = (priTable[3]==4'b0001) & busRequest[0] |
                       (priTable[3]==4'b0010) & busRequest[1] |
                       (priTable[3]==4'b0100) & busRequest[2] |
                       (priTable[3]==4'b1000) & busRequest[3] ;

// select the request with the highest priority
always @(*)
begin    
   if      (priRequest[0]==1'b1) {priUpdate,busGrant} = {4'b1111,priTable[0]};
   else if (priRequest[1]==1'b1) {priUpdate,busGrant} = {4'b1110,priTable[1]};
   else if (priRequest[2]==1'b1) {priUpdate,busGrant} = {4'b1100,priTable[2]};
   else if (priRequest[3]==1'b1) {priUpdate,busGrant} = {4'b1000,priTable[3]};
   else                          {priUpdate,busGrant} = {4'b0000,4'b0000};
end
  

////////////////////////////////////////////////////////////////////////////////
// Multiplexer
////////////////////////////////////////////////////////////////////////////////
assign sHIFRead          = (busOwner[0] & m0HIFRead)
`ifdef  RW_MUMIMO_SEC_USER1_EN
                         | (busOwner[1] & m1HIFRead)
`endif//RW_MUMIMO_SEC_USER1_EN
`ifdef  RW_MUMIMO_SEC_USER2_EN
                         | (busOwner[2] & m2HIFRead)
`endif//RW_MUMIMO_SEC_USER2_EN
`ifdef  RW_MUMIMO_SEC_USER3_EN
                         | (busOwner[3] & m3HIFRead)
`endif//RW_MUMIMO_SEC_USER3_EN
                         ;

assign sHIFWrite         = (busOwner[0] & m0HIFWrite)
`ifdef  RW_MUMIMO_SEC_USER1_EN
                         | (busOwner[1] & m1HIFWrite)
`endif//RW_MUMIMO_SEC_USER1_EN
`ifdef  RW_MUMIMO_SEC_USER2_EN
                         | (busOwner[2] & m2HIFWrite)
`endif//RW_MUMIMO_SEC_USER2_EN
`ifdef  RW_MUMIMO_SEC_USER3_EN
                         | (busOwner[3] & m3HIFWrite)
`endif//RW_MUMIMO_SEC_USER3_EN
                         ;

assign sHIFAddressIn     = ({32{busOwner[0]}} & m0HIFAddressIn)
`ifdef  RW_MUMIMO_SEC_USER1_EN
                         | ({32{busOwner[1]}} & m1HIFAddressIn)
`endif//RW_MUMIMO_SEC_USER1_EN
`ifdef  RW_MUMIMO_SEC_USER2_EN
                         | ({32{busOwner[2]}} & m2HIFAddressIn)
`endif//RW_MUMIMO_SEC_USER2_EN
`ifdef  RW_MUMIMO_SEC_USER3_EN
                         | ({32{busOwner[3]}} & m3HIFAddressIn)
`endif//RW_MUMIMO_SEC_USER3_EN
                         ;

assign sHIFSize          = ({3{busOwner[0]}} & m0HIFSize)
`ifdef  RW_MUMIMO_SEC_USER1_EN
                         | ({3{busOwner[1]}} & m1HIFSize)
`endif//RW_MUMIMO_SEC_USER1_EN
`ifdef  RW_MUMIMO_SEC_USER2_EN
                         | ({3{busOwner[2]}} & m2HIFSize)
`endif//RW_MUMIMO_SEC_USER2_EN
`ifdef  RW_MUMIMO_SEC_USER3_EN
                         | ({3{busOwner[3]}} & m3HIFSize)
`endif//RW_MUMIMO_SEC_USER3_EN
                         ;

assign sHIFWriteDataIn   = ({32{busOwner[0]}} & m0HIFWriteDataIn)
`ifdef  RW_MUMIMO_SEC_USER1_EN
                         | ({32{busOwner[1]}} & m1HIFWriteDataIn)
`endif//RW_MUMIMO_SEC_USER1_EN
`ifdef  RW_MUMIMO_SEC_USER2_EN
                         | ({32{busOwner[2]}} & m2HIFWriteDataIn)
`endif//RW_MUMIMO_SEC_USER2_EN
`ifdef  RW_MUMIMO_SEC_USER3_EN
                         | ({32{busOwner[3]}} & m3HIFWriteDataIn)
`endif//RW_MUMIMO_SEC_USER3_EN
                         ;

assign m0HIFReadDataOut   = sHIFReadDataOut;
assign m0HIFReadDataValid = busOwner[0] & sHIFReadDataValid;
assign m0HIFReady         = busOwner[0] & sHIFReady;
assign m0HIFTransComplete = busOwner[0] & sHIFTransComplete;
assign m0HIFError         = busOwner[0] & sHIFError;

`ifdef  RW_MUMIMO_SEC_USER1_EN
assign m1HIFReadDataOut   = sHIFReadDataOut;
assign m1HIFReadDataValid = busOwner[1] & sHIFReadDataValid;
assign m1HIFReady         = busOwner[1] & sHIFReady;
assign m1HIFTransComplete = busOwner[1] & sHIFTransComplete;
assign m1HIFError         = busOwner[1] & sHIFError;
`endif//RW_MUMIMO_SEC_USER1_EN

`ifdef  RW_MUMIMO_SEC_USER2_EN
assign m2HIFReadDataOut   = sHIFReadDataOut;
assign m2HIFReadDataValid = busOwner[2] & sHIFReadDataValid;
assign m2HIFReady         = busOwner[2] & sHIFReady;
assign m2HIFTransComplete = busOwner[2] & sHIFTransComplete;
assign m2HIFError         = busOwner[2] & sHIFError;
`endif//RW_MUMIMO_SEC_USER2_EN

`ifdef  RW_MUMIMO_SEC_USER3_EN
assign m3HIFReadDataOut   = sHIFReadDataOut;
assign m3HIFReadDataValid = busOwner[3] & sHIFReadDataValid;
assign m3HIFReady         = busOwner[3] & sHIFReady;
assign m3HIFTransComplete = busOwner[3] & sHIFTransComplete;
assign m3HIFError         = busOwner[3] & sHIFError;
`endif//RW_MUMIMO_SEC_USER3_EN

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
