//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $RCSmodulefile   : Key Search Engine
// $Author          : $ 
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 0.1                                                             
// $Date: 19-Nov-2008                                                                
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of Key Search Engine module
//                    
// Simulation Notes : 
//    RW_KEY_INDEX_WIDTH : Configurable depth of Key Storage RAM
//    RW_KEY_DATA_WIDTH  : Configurable width of Key Storage RAM
//
//    For simulation, two defines are available
//
//    RW_SIMU_ON      : which creates string signals to display the FSM states on  
//                      the waveform viewers
//    RW_ASSERT_ON    : which enables System Verilog Assertions.
//
//
// Synthesis Notes  :
//
// Application Note : 
//     All the signals are coming from MAC Controller clock domain. Hence, no
//     resynchronizations are needed.
//
// Simulator        :
//     
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module keySearchEngine ( 
   //$port_g Clock and Reset interface
   input wire          macCoreClk,                   // MAC Core Clock
   input wire          macCoreClkHardRst_n,          // Hard Reset of the MAC WT Clock domain 
                                                     // active low
   input wire          macCoreClkSoftRst_n,          // Soft Reset of the MAC WT Clock domain 
                                                     // active low

   //$port_g CS Register Interface
   input wire          keyStoRAMReset,               // Active high reset signal to clear 
                                                     // RAM contents initiated by Software
   output wire         keyStoRAMResetIn,             // keyStoRAMReset clear value
   output reg          keyStoRAMResetInValid,        // Clearing signal to CSReg, when the RAM reset
                                                     // operation is completed it is pulse
                                                     // for duration equal to 1 macCoreClk
   input wire  [127:0] encrKeyRAM,                   // Crypto key configured by software
                                                     // into Encryption registers
`ifdef  RW_KEY_EXTENDED
   input wire  [127:0] encrIntKeyRAM,                // Integrity Crypto key configured by software
                                                     // into Encryption registers
`endif//RW_KEY_EXTENDED
   input wire [`RW_KEY_INDEX_WIDTH-1:0] keyIndexRAM, // Index of Key Storage RAM into which
                                                     // the encryption register contents
                                                     // are to be written
   input wire   [47:0] encrMACAddr,                  // MAC address configured by software
                                                     // into Encryption registers
                                                     // It is reserved when keyIndexRAM points
                                                     // to default Key location
   input wire          newWrite,                     // Trigger for write operation
                                                     // KS engine writes the encryption register
                                                     // inputs into RAM location pointed by
                                                     // keyIndexRAM
   input wire          newRead,                      // Trigger for read operation
                                                     // KS engine reads the contents of RAM location
                                                     // pointed by keyIndexRAM and copies them 
                                                     // to encryption register It provides the
                                                     // RAM contents on output lines of this
                                                     // module along with a validating signal
                                                     // CSR module should sample the contents 
                                                     // based on that signal
                                                     // CSReg shall not initiate a read operation
                                                     // without asserting the debugKSR signal below
   input wire          newSearch,                    // Trigger for search operation SW API
   output wire         searchErrorIn,                // Error signal indicating to SW MAC address
                                                     // has not been found,CSR should samples this 
                                                     // signal on newSearchInValid
   input wire          debugKSR,                     // Signal to be asserted for debug purpose
                                                     // only initiated by Software. During a
                                                     // read operation, when this signal is
                                                     // asserted, valid key will be read out.
   output wire         newWriteInValid,              // Pulse signal to clear the newWrite
                                                     // signal initiated by CSReg
   output reg          newReadInValid,               // Pulse signal to clear the newRead
                                                     // signal initiated by CSReg
   output reg          newSearchInValid,             // Pulse signal to clear the newSearch
                                                     // signal initiated by CSReg
   input wire    [1:0] cLenRAM,                      // Cipher length indication
                                                     // 0 - 64-bits
                                                     // 1 - 128-bits
                                                     // 2 - reserved
                                                     // 3 - 256-bits
   input wire          useDefKeyRAM,                 // Indication for usage of Default Keys.
   input wire    [1:0] sppRAM,                       // Indication for handling A-MSDU frames
   input wire    [3:0] vlanIDRAM,                    // Virtual LAN ID for searching default key
   input wire    [2:0] cTypeRAM,                     // Cipher type
                                                     
   output wire   [2:0] cTypeRAMIn,                   // Cipher Type found at Index or value at
                                                     // at MAC address found location
   output wire   [3:0] vlanIDRAMIn,                  // Virtual LAN ID found at Index or value 
                                                     // at MAC address found location
   output wire   [1:0] sppRAMIn,                     // SPP RAM value found at Index or value 
                                                     // at MAC address found location
   output wire         useDefKeyRAMIn,               // Use Default Key Value found at Index 
                                                     // or value at MAC address found location
   output wire   [1:0] cLenRAMIn,                    // Cipher length found at Index or value 
                                                     // at MAC address found location
   output wire  [47:0] encrMACAddrIn,                // MAC address value found at Index
   output wire [127:0] encrKeyRAMIn,                 // Value of crypto key at Index found or
                                                     // value at MAC address found location
`ifdef  RW_KEY_EXTENDED
   output wire [127:0] encrIntKeyRAMIn,              // Value of integrity crypto key at Index found
                                                     // or value at MAC address found location
`endif//RW_KEY_EXTENDED

   input  wire   [7:0] staKeyStartIndex,             // Configures the index of the first STA in the
                                                     // Key Storage RAM.
   input  wire   [7:0] staKeyEndIndex,               // Configures the index of the latest STA in
                                                     // the Key Storage RAM. 
   input  wire   [3:0] nVAP,                         // Configures the number of VAP.
   output wire   [7:0] staKeyMaxIndex,               // Contains the index max which can be used in
                                                     // the key Storage RAM
          
   //$port_g Controller Interface [TX or RX or Block ACK]
   input wire [`RW_KEY_INDEX_WIDTH-1:0] keySearchIndex, // RAM Index to be searched
   input wire         keySearchIndexTrig_p,          // Trigger for searching RAM index and
                                                     // return contents (along with MAC addr)
   input wire  [47:0] macAddressIn,                  // MAC address to be searched in RAM
   input wire         indexSearchTrig_p,             // Trigger for searching MAC address in
                                                     // KS RAM and return contents (along with
                                                     // Index)
   output wire        keyStorageError_p,             // Error indicating key NOT found /
                                                     // MAC address NOT found
   output reg         keyStorageValid_p,             // Pulse signal indicating completion of 
                                                     // search operation Validates output 
                                                     // contents like cTypeKSR, 
                                                     // keyIndexReturn, cLenKSR, 
                                                     // macAddressKSR etc.
   output reg [`RW_KEY_INDEX_WIDTH-1:0] keyIndexReturn, // RAM Index output corresponding to
                                                     // location where MAC address was found
   output reg   [2:0] cTypeKSR,                      // Cipher Type found at Index or value at
                                                     // at MAC address found location
   output reg   [3:0] vlanIDKSR,                     // Virtual LAN ID found at Index or value 
                                                     // at MAC address found location
   output reg   [1:0] sppKSR,                        // SPP RAM value found at Index or value 
                                                     // at MAC address found location
   output reg         useDefKeyKSR,                  // Use Default Key Value found at Index 
                                                     // or value at MAC address found location
   output reg   [1:0] cLenKSR,                       // Cipher length found at Index or value 
                                                     // at MAC address found location
   output reg  [47:0] macAddressKSR,                 // MAC address value found at Index
   output reg [127:0] cryptoKeyKSR,                  // Value of crypto key at Index found or
                                                     // value at MAC address found location
`ifdef  RW_KEY_EXTENDED
   output reg [127:0] cryptoIntKeyKSR,               // Value of integrity crypto key at Index found
                                                     // or value at MAC address found location
`endif//RW_KEY_EXTENDED
          
   //$port_g Single Port RAM Interface [Key Storage RAM]
   input  wire  [`RW_KEY_DATA_WIDTH-1:0] keyStorageReadData,  // Read data bus from KS RAM
   output reg   [`RW_KEY_DATA_WIDTH-1:0] keyStorageWriteData, // Write data bus to KS RAM
   output reg  [`RW_KEY_INDEX_WIDTH-1:0] keyStorageAddr,      // Address bus to KS RAM
   output reg                            keyStorageEn,        // Enable (access) to KS RAM
                                                              // Active high for both
                                                              // read/write
   output reg                            keyStorageWriteEn,   // Write enable to KS RAM
                                                              // Active high only for write

   //$port_g Debug Port
   output wire  [15:0] debugPortKeySearch            // Debug port for validation
);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// Key Search FSM Description
//$fsm_sd keySearchEngine
localparam    
   KS_IDLE            =  4'd0,
   KS_INIT            =  4'd1,
   HW_SEARCH_MAC_ADDR =  4'd2,
   HW_RD_KS_RAM       =  4'd3,
   SW_WR_KS_RAM       =  4'd4,
   SW_RD_KS_RAM       =  4'd5,
   SW_SEARCH_MAC_ADDR =  4'd6,
   SW_REMOVE_INDEX    =  4'd7,
   SW_INSERT_INDEX    =  4'd8;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

   reg                     [3:0] keySearchEngCs;
   reg                     [3:0] keySearchEngNs;
   
   wire                          KSEFSMIsIdle;              // Key Search Engine FSM is in IDLE state
   wire                          HWReadInProgress;          // Key Search Engine FSM is in IDLE state
   
   reg                           indexSearchTrigPending;    // HW indexSearch request is pending
   reg                           keySearchIndexTrigPending; // HW keysearchindex request is pending

   wire                          HWSearchInProgress;
   wire                          SWSearchInProgress;
   
   reg                           KeyStorageReadDataValid;
   reg [`RW_KEY_INDEX_WIDTH-1:0] currAddr;
   
   reg                    [47:0] macAddressInLat;
   reg [`RW_KEY_INDEX_WIDTH-1:0] keySearchIndexLat;
  
   wire                          macAddrHWFound;
   wire                          macAddrSWFound;
   wire[`RW_KEY_INDEX_WIDTH-1:0] keyStorageHighestAddr;
   wire[`RW_KEY_INDEX_WIDTH-1:0] keyStorageLowestAddr;

   reg                           keyStorageError_HW_p;
   reg                           keyStorageError_SW_p;
   
   wire                          staIndex;

   reg [`RW_KEY_INDEX_WIDTH-1:0] sTable[`RW_KEY_INDEX_MAX-3:1];
   reg [`RW_KEY_INDEX_WIDTH-1:0] sTableShiftUp[`RW_KEY_INDEX_MAX-3:1];
   reg [`RW_KEY_INDEX_WIDTH-1:0] sTableShiftDn[`RW_KEY_INDEX_MAX-3:1];
   reg [`RW_KEY_INDEX_MAX-3  :1] sTableShiftEn;
   reg [`RW_KEY_INDEX_MAX-3  :1] sTableWriteEn;
   reg [`RW_KEY_INDEX_WIDTH-1:0] sTableAddrReg;
   reg [`RW_KEY_INDEX_WIDTH-1:0] sTableAddr;
   reg [`RW_KEY_INDEX_WIDTH-1:0] sTableData;
   reg                           sTableInsert;
   reg                           sTableRemove;
   reg                           sTableRemoveDone;
   reg [`RW_KEY_INDEX_WIDTH+1:0] bitSel;
   reg                           searchInProgress;
   wire                          searchMatch;
   wire                          searchDone;
   wire                          miss;
   wire                   [47:0] searchMacAddr;
   wire                   [47:0] readMacAddr;
   reg                           readMacAddrValid;

   
//////////////////////////////////////////////////////////////////////////////
// Beginning of Logic part
//////////////////////////////////////////////////////////////////////////////

assign debugPortKeySearch = {indexSearchTrig_p,
                             keySearchIndexTrig_p,
                             keyStorageAddr[5:0],
                             keyStorageError_p,
                             keyStorageValid_p,
                             keyIndexReturn[5:0]
                            };

assign keyStorageHighestAddr  = staKeyEndIndex[`RW_KEY_INDEX_WIDTH-1:0];
assign keyStorageLowestAddr   = staKeyStartIndex[`RW_KEY_INDEX_WIDTH-1:0];
assign staKeyMaxIndex         = 8'd`RW_KEY_INDEX_MAX;


// Indicates that the index is a STA index
assign staIndex  = keyIndexRAM>=keyStorageLowestAddr/* &
                   keyIndexRAM<=keyStorageHighestAddr*/;


//////////////////////////////
// Key Search FSM
//////////////////////////////

// Key Search FSM Current State Logic 
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    keySearchEngCs <= KS_IDLE; 
  else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
    keySearchEngCs <= KS_IDLE;
  else
    keySearchEngCs <= keySearchEngNs; 
end

// Key Search FSM Next State Logic
always @*
begin
  case(keySearchEngCs)
    KS_IDLE:
      //$fsm_s in KS_IDLE the KS Engine wait either for reset RAM (reinit keyRAM) or triggers from controllers to read the keyRAM, or SW read/write/search to the keyRAM
      if( keyStoRAMReset == 1'b1 )
        //$fsm_t If keyStoRAMReset is set then the KS Engine jumps to KS_INIT
        keySearchEngNs =  KS_INIT;
      else if( (indexSearchTrig_p == 1'b1) || (indexSearchTrigPending == 1'b1) )
        //$fsm_t If indexSearchTrig is requested then the KS Engine jumps to HW_SEARCH_MAC_ADDR
        keySearchEngNs =  HW_SEARCH_MAC_ADDR;
      else if( (keySearchIndexTrig_p == 1'b1) || (keySearchIndexTrigPending == 1'b1) )
        //$fsm_t If keySearchIndexTrig is requested then the KS Engine jumps to HW_RD_KS_RAM
        keySearchEngNs =  HW_RD_KS_RAM;
      else if( newWrite == 1'b1 && newWriteInValid == 1'b0 & staIndex == 1'b0)
        //$fsm_t If SW write access to VLAN index then KS Engine jumps to SW_WR_KS_RAM
        keySearchEngNs =  SW_WR_KS_RAM;
      else if( newWrite == 1'b1 && newWriteInValid == 1'b0 & staIndex == 1'b1)
        //$fsm_t If SW write access to STA index then KS Engine jumps to SW_REMOVE_INDEX
        keySearchEngNs =  SW_REMOVE_INDEX;
      else if( newRead == 1'b1 && newReadInValid == 1'b0 )
        //$fsm_t If SW read access then KS Engine jumps to SW_RD_KS_RAM
        keySearchEngNs =  SW_RD_KS_RAM;
      else if( newSearch == 1'b1 && newSearchInValid == 1'b0 )
        //$fsm_t If SW search access then KS Engine jumps to SW_SEARCH_MAC_ADDR
        keySearchEngNs =  SW_SEARCH_MAC_ADDR;
      else
        //$fsm_t If no triggers or RAM reset or SW access then it remains in KS_IDLE
        keySearchEngNs = KS_IDLE;

    KS_INIT:
      //$fsm_s in KS_INIT the KS Engine writes default value to all keyRAM address.
      if(keyStoRAMResetInValid == 1'b1)
        //$fsm_t When the keyRAM address reaches the last one, then KS Engines goes back to KS_IDLE
        keySearchEngNs =  KS_IDLE;
      else 
        //$fsm_t When the keyRAM address is not the last one the KS Engines stays in KS_INIT
        keySearchEngNs =  KS_INIT;
 
    HW_SEARCH_MAC_ADDR:
      //$fsm_s This state is reached when MAC HW requests for a MAC address search
      //[Transmitter address as input trigger]
      if(searchDone)
        //$fsm_t When search is done, the state machine goes back to KS_IDLE.
        keySearchEngNs = KS_IDLE;
      else
        //$fsm_t While search is not done, the state machine remains in HW_SEARCH_MAC_ADDR.
        keySearchEngNs = HW_SEARCH_MAC_ADDR;
    
    HW_RD_KS_RAM:
      //$fsm_s This state is reached when MAC HW requests for an Index search [RAM Index as input trigger]
      if(keyStorageValid_p == 1'b1)
        //$fsm_t if keySearchComplete (MAC address search found) then goes back to KS_IDLE.
        keySearchEngNs = KS_IDLE;
      else
        //$fsm_t if index read not done or MAC address search not done then remains in HW_RD_KS_RAM.
        keySearchEngNs = HW_RD_KS_RAM;
    
    SW_WR_KS_RAM:
      //$fsm_s The SW_WR_KS_RAM is reached when there is a write access is initiated by CSR module
      if( indexSearchTrig_p == 1'b1 )
        //$fsm_t If indexSearchTrig_p is pulsed then the KS Engine jumps to HW_SEARCH_MAC_ADDR
        keySearchEngNs =  HW_SEARCH_MAC_ADDR;
      else if( keySearchIndexTrig_p == 1'b1 )
        //$fsm_t If keySearchIndexTrig_p is pulsed then the KS Engine jumps to HW_RD_KS_RAM
        keySearchEngNs =  HW_RD_KS_RAM;
      else
        //$fsm_t If no search is requested, the state machine goes back to KS_IDLE
        keySearchEngNs = KS_IDLE;
    
    SW_REMOVE_INDEX:
      //$fsm_s The SW_REMOVE_INDEX is reached when there is a write access to STA index is
      //initiated by CSR module
      if (sTableRemoveDone & encrMACAddr=={48{1'b1}})
        //$fsm_t When KS_RAM index is removed and new entry is empty then the KS Engine jumps
        //to SW_WR_KS_RAM
        keySearchEngNs = SW_WR_KS_RAM;
      else if (sTableRemoveDone)
        //$fsm_t When KS_RAM index is removed and new entry is not empty then the KS Engine jumps
        //to SW_INSERT_INDEX
        keySearchEngNs = SW_INSERT_INDEX;
      else
        //$fsm_t while KS_RAM index is not removed, the state machine stay in SW_REMOVE_INDEX
        keySearchEngNs = SW_REMOVE_INDEX;
    
    SW_INSERT_INDEX:
      //$fsm_s The SW_INSERT_INDEX is reached when there is a write access to STA Index is
      //initiated by CSR module
      if( searchDone )
        //$fsm_t If searchDone is pulsed then the KS Engine jumps to SW_WR_KS_RAM
        keySearchEngNs = SW_WR_KS_RAM;
      else
        //$fsm_t While searchDone is low, the state machine stays in SW_INSERT_INDEX
        keySearchEngNs = SW_INSERT_INDEX;
    
    SW_RD_KS_RAM:
      //$fsm_s The SW_RD_KS_RAM is reached when there is a write access is initiated by CSR module
      if( indexSearchTrig_p == 1'b1 )
        //$fsm_t If  indexSearchTrig_p is pulsed then the KS Engine jumps to HW_SEARCH_MAC_ADDR
        keySearchEngNs =  HW_SEARCH_MAC_ADDR;
      else if( keySearchIndexTrig_p == 1'b1 )
        //$fsm_t If keySearchIndexTrig_p  is pulsed then the KS Engine jumps to HW_RD_KS_RAM
        keySearchEngNs =  HW_RD_KS_RAM;
      else 
        //$fsm_t If no search is requested, the state machine goes back to KS_IDLE
        keySearchEngNs = KS_IDLE;
        
    SW_SEARCH_MAC_ADDR:
      //$fsm_s This state is reached when SW request for a MAC address search
      //[Transmitter address as input trigger]
      if(searchDone)
        //$fsm_t When search is done, the state machine goes back to KS_IDLE.
        keySearchEngNs = KS_IDLE;
      else
        //$fsm_t While seach is not done, the state machine remains in SW_SEARCH_MAC_ADDR.
        keySearchEngNs = SW_SEARCH_MAC_ADDR;

     // Disable coverage on the default state because it cannot be reached.
     // pragma coverage block = off 
    default:   
        keySearchEngNs = KS_IDLE;
     // pragma coverage block = on 
  endcase
end

// KSE FSM is in idle mode indication
assign KSEFSMIsIdle = (keySearchEngCs == KS_IDLE);
// KSE FSM is in HW Read mode indication
assign HWReadInProgress = ((keySearchEngCs == HW_RD_KS_RAM));

// Logic to detect if the keySearchEngine is busy when indexSearchTrig_p
// request is received. This situation can happen if a SW research is on going.
// In this case a pending request signal is set.
// It is reset when the keySearchEngine starts the indexSearch process. 
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
    indexSearchTrigPending <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    indexSearchTrigPending <= 1'b0;
  else if (indexSearchTrig_p == 1'b1 && KSEFSMIsIdle == 1'b0)
    indexSearchTrigPending <= 1'b1;
  else if (keySearchEngNs == HW_SEARCH_MAC_ADDR)
    indexSearchTrigPending <= 1'b0;
end

// Logic to detect if the keySearchEngine is busy when
// keySearchIndexTrig_p request is received.
// This situation can happen if a SW research or other HW research
// is on going.
// In this case a pending request signal is set.
// It is reset when the keySearch starts the HW index read process. 
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
    keySearchIndexTrigPending <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    keySearchIndexTrigPending <= 1'b0;
  else if (keySearchEngNs == HW_RD_KS_RAM)
    keySearchIndexTrigPending <= 1'b0;
  else if (keySearchIndexTrig_p == 1'b1)
    keySearchIndexTrigPending <= 1'b1;
end

// Logic to CSR for clearing keyStoRAMReset
// Asserted once RAM reset operation is completed by KS engine
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  
    keyStoRAMResetInValid <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    keyStoRAMResetInValid <= 1'b0;
  // Assert clear pulse when address reaches 'hFF [63 - Highest port no.]
  else if( (keySearchEngCs == KS_INIT) && (keyStorageAddr == `RW_KEY_INDEX_MAX) ) 
    keyStoRAMResetInValid <= 1'b1;
  else
    keyStoRAMResetInValid <= 1'b0;
end

// Reset value to be set in CSR
assign keyStoRAMResetIn = 1'b0;

// Pulse to clear CSR signal 'newWrite' if write operation is complete
assign newWriteInValid = keySearchEngCs==SW_WR_KS_RAM & keyStorageEn & keyStorageWriteEn;
   
// Pulse to clear CSR signal 'newRead' if read operation is complete
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
    newReadInValid <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    newReadInValid <= 1'b0;
  else if (keySearchEngCs == SW_RD_KS_RAM)
    newReadInValid <= 1'b1;
  else
    newReadInValid <= 1'b0;
end

// Logic for latching macAddressIn - Input MAC address to be searched
// in the Key Storage RAM
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
     macAddressInLat <= 48'h0;
  else if (macCoreClkSoftRst_n == 1'b0) 
     macAddressInLat <= 48'h0;
  else if (indexSearchTrig_p == 1'b1)
     macAddressInLat[47:0] <= macAddressIn[47:0];
end

// Logic for latching keyIndex - Input MAC address to be searched
// in the Key Storage RAM
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
     keySearchIndexLat <= `RW_KEY_INDEX_WIDTH'd0;
  else if (macCoreClkSoftRst_n == 1'b0) 
     keySearchIndexLat <= `RW_KEY_INDEX_WIDTH'd0;
  else if (keySearchIndexTrig_p == 1'b1)
     keySearchIndexLat <= keySearchIndex;
end

// Logic to indicate HW MAC address search is in progress
// This is asserted only during MAC address search operation
// That is, it is a signal that stays high during the
// entire search operation (upto a max. of 2*RW_KEY_INDEX_WIDTH clock cycles)
assign HWSearchInProgress = keySearchEngCs==HW_SEARCH_MAC_ADDR;

// Logic to indicate SW MAC address search is in progress
// This is asserted only during MAC address search operation
// That is, it is a signal that stays high during the
// entire search operation (upto a max. of 2*RW_KEY_INDEX_WIDTH clock cycles)
assign SWSearchInProgress = keySearchEngCs==SW_SEARCH_MAC_ADDR;

// Logic to detect the event of finding MAC address in RAM
// Added here to reduce the complexity in block where it is used
assign macAddrHWFound = HWSearchInProgress & searchMatch;
assign macAddrSWFound = SWSearchInProgress & searchMatch;

// Pulse to clear newSearch bit in CSR
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    newSearchInValid <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    newSearchInValid <= 1'b0;
  else if(SWSearchInProgress & searchDone)
    newSearchInValid <= 1'b1;
  else
    newSearchInValid <= 1'b0;
end
 
// Logic for the event - Null key / MAC address not found in RAM   
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    keyStorageError_HW_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    keyStorageError_HW_p <= 1'b0;
  else if (HWSearchInProgress && searchDone && ~searchMatch)
    keyStorageError_HW_p <= 1'b1;
  else
    keyStorageError_HW_p <= 1'b0;
end

assign keyStorageError_p = keyStorageError_HW_p; 

// Logic for the event - Null key / MAC address not found in RAM   
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    keyStorageError_SW_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    keyStorageError_SW_p <= 1'b0;
  else if (SWSearchInProgress && searchDone && ~searchMatch)
    keyStorageError_SW_p <= 1'b1;
  else
    keyStorageError_SW_p <= 1'b0;
end

// Logic for the event - Null key / MAC address not found in RAM   
assign searchErrorIn = keyStorageError_SW_p;

// Key Storage - Write operation from CSR
// CSR Write operations take single clock for updating the RAM.
// RAM reset mechanism may also be initiated by software through the
// signal keyStoRAMReset, for initialization of RAM contents
// to default values.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
  begin
     keyStorageWriteData <= {`RW_KEY_DATA_WIDTH{1'b0}};
  end
  else if (keySearchEngNs == KS_INIT)
  begin
     // All zero, except macAddress (all one)
`ifdef  RW_KEY_EXTENDED
     keyStorageWriteData[315:188] <= 128'h0;
`endif//RW_KEY_EXTENDED
     keyStorageWriteData[187:60]  <= 128'h0;
     keyStorageWriteData[59:12]   <= {48{1'b1}};
     keyStorageWriteData[11:10]   <= 2'h0;
     keyStorageWriteData[9]       <= 1'b0;
     keyStorageWriteData[8:7]     <= 2'h0;
     keyStorageWriteData[6:3]     <= 4'h0;
     keyStorageWriteData[2:0]     <= 3'h0;
  end
  else if (keySearchEngNs == SW_WR_KS_RAM) 
  begin
`ifdef  RW_KEY_EXTENDED
     keyStorageWriteData[315:188] <= encrIntKeyRAM[127:0];
`endif//RW_KEY_EXTENDED
     keyStorageWriteData[187:60]  <= encrKeyRAM[127:0];
     keyStorageWriteData[59:12]   <= encrMACAddr[47:0];
     keyStorageWriteData[11:10]   <= cLenRAM[1:0];
     keyStorageWriteData[9]       <= useDefKeyRAM;
     keyStorageWriteData[8:7]     <= sppRAM[1:0];
     keyStorageWriteData[6:3]     <= vlanIDRAM[3:0];
     keyStorageWriteData[2:0]     <= cTypeRAM[2:0];
  end
end

// Logic for driving enable signal for accessing RAM port
// (for either read or write or search operations)
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  
    keyStorageEn <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    keyStorageEn <= 1'b0;
  else if ( (keySearchEngNs != KS_IDLE) &&
            (keySearchEngNs != SW_REMOVE_INDEX) &&
            (HWReadInProgress == 1'b0) && !macAddrHWFound )
    keyStorageEn <= 1'b1;
  else 
    keyStorageEn <= 1'b0;
end

// Logic for driving Write enable signal for accessing RAM port
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)
    keyStorageWriteEn <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    keyStorageWriteEn <= 1'b0;
  else if( (keySearchEngNs == KS_INIT     ) ||
           (keySearchEngNs == SW_WR_KS_RAM) )
    keyStorageWriteEn <= 1'b1;
  else
    keyStorageWriteEn <= 1'b0;
end

// Logic for driving address signals to Key Storage RAM
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  
     keyStorageAddr <= {`RW_KEY_INDEX_WIDTH'b0};
  else if (macCoreClkSoftRst_n == 1'b0)
     keyStorageAddr <= {`RW_KEY_INDEX_WIDTH'b0};
  else
  begin
    case (keySearchEngNs)
    KS_INIT           : keyStorageAddr <= (keySearchEngCs==KS_IDLE) ?
                                          {`RW_KEY_INDEX_WIDTH'b0}  :
                                          keyStorageAddr + `RW_KEY_INDEX_WIDTH'b1;
    SW_RD_KS_RAM      : keyStorageAddr <= keyIndexRAM;
    SW_WR_KS_RAM      : keyStorageAddr <= keyIndexRAM;
    SW_INSERT_INDEX   : keyStorageAddr <= sTableData;
    SW_SEARCH_MAC_ADDR: keyStorageAddr <= sTableData;
    HW_SEARCH_MAC_ADDR: keyStorageAddr <= sTableData;
    HW_RD_KS_RAM      : keyStorageAddr <= (keySearchIndexTrigPending) ?
                                          keySearchIndexLat           :
                                          keySearchIndex;
    endcase
  end
end

// KeyStorageReadDataValid indicates valid data on Read Data bus.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    KeyStorageReadDataValid <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    KeyStorageReadDataValid <= 1'b0;
  else if (HWReadInProgress && keyStorageEn && !keyStorageWriteEn )
    KeyStorageReadDataValid <= 1'b1;
  else
    KeyStorageReadDataValid <= 1'b0;
end

// Current Address - Latched version of keyStorageAddr RAM 
// These counters are maintained during MAC address search
// operation to store the index where MAC address has been found
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n == 1'b0) 
     currAddr <= {`RW_KEY_INDEX_WIDTH'b0};
   else if (macCoreClkSoftRst_n == 1'b0)
     currAddr <= {`RW_KEY_INDEX_WIDTH'b0};
   else if ( (HWSearchInProgress && !searchMatch) ||
             (SWSearchInProgress && !searchMatch) )
     currAddr <= keyStorageAddr;
end

// Indicates valid outputs to macCore
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    keyStorageValid_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    keyStorageValid_p <= 1'b0;
  else if (macAddrHWFound == 1'b1) 
    keyStorageValid_p <= 1'b1;
  else if (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM) 
    keyStorageValid_p <= 1'b1;
  else
    keyStorageValid_p <= 1'b0;
end


// Key Index pointing to the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    keyIndexReturn <= {`RW_KEY_INDEX_WIDTH'b0};
  else if (macCoreClkSoftRst_n == 1'b0) 
    keyIndexReturn <= {`RW_KEY_INDEX_WIDTH'b0};
  else if (keyStorageError_HW_p == 1'b1) 
    keyIndexReturn <= {`RW_KEY_INDEX_WIDTH'b0};
  else if (macAddrHWFound || macAddrSWFound ) 
    keyIndexReturn <= currAddr;
end

// Cipher Type Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    cTypeKSR <= 3'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    cTypeKSR <= 3'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    cTypeKSR <= 3'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM))) 
    cTypeKSR <= keyStorageReadData[2:0];
end

 
assign cTypeRAMIn = (newReadInValid) ? keyStorageReadData[2:0] : 3'b0;

// Virtual LanID Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    vlanIDKSR <= 4'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    vlanIDKSR <= 4'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    vlanIDKSR <= 4'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM))) 
    vlanIDKSR <= keyStorageReadData[6:3];
end

assign vlanIDRAMIn = (newReadInValid) ? keyStorageReadData[6:3] : 4'b0;

// SPP Type Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    sppKSR <= 2'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    sppKSR <= 2'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    sppKSR <= 2'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM))) 
    sppKSR <= keyStorageReadData[8:7];
end


assign sppRAMIn = (newReadInValid) ? keyStorageReadData[8:7] : 2'b0;

// UseDefaultKey Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    useDefKeyKSR <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    useDefKeyKSR <= 1'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    useDefKeyKSR <= 1'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM))) 
    useDefKeyKSR <= keyStorageReadData[9];
end

assign useDefKeyRAMIn = (newReadInValid) ? keyStorageReadData[9] : 1'b0;

// Cipher Length Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    cLenKSR <= 2'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    cLenKSR <= 2'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    cLenKSR <= 2'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM))) 
    cLenKSR <= keyStorageReadData[11:10];
end

assign cLenRAMIn = (newReadInValid) ? keyStorageReadData[11:10] : 2'b0;

  // MAC Address Field of the last successfully searched
  // RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    macAddressKSR <= 48'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    macAddressKSR <= 48'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    macAddressKSR <= 48'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM))) 
    macAddressKSR <= keyStorageReadData[59:12];
end

assign encrMACAddrIn = (newReadInValid) ? keyStorageReadData[59:12] : 48'b0;

// Encryption Key Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    cryptoKeyKSR <= 128'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    cryptoKeyKSR <= 128'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    cryptoKeyKSR <= 128'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM)))
    cryptoKeyKSR <= keyStorageReadData[187:60];
end

 
assign encrKeyRAMIn = (newReadInValid && debugKSR) ? keyStorageReadData[187:60]  : 128'b0 ;

`ifdef  RW_KEY_EXTENDED
// Encryption Key Field of the last successfully searched
// RAM location; registered version of the RAM's row content
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0) 
    cryptoIntKeyKSR <= 128'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    cryptoIntKeyKSR <= 128'b0;
  else if (keyStorageError_HW_p == 1'b1) 
    cryptoIntKeyKSR <= 128'b0;
  else if ((macAddrHWFound || (KeyStorageReadDataValid && keySearchEngCs == HW_RD_KS_RAM)))
    cryptoIntKeyKSR <= keyStorageReadData[315:188];
end

assign encrIntKeyRAMIn = (newReadInValid && debugKSR) ? keyStorageReadData[315:188] : 128'b0 ;
`endif//RW_KEY_EXTENDED


// Search Table
//    Contains keyStorage RAM index sorted by macAddress value, increasing order.
//    Initialized with '0' (empty) when keyStorage is initialized.
//    Can not be reseted by software.
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin: SEARCHTABLE
  integer i;

  if (macCoreClkHardRst_n == 1'b0) 
    for (i=1;i<=`RW_KEY_INDEX_MAX-3;i=i+1)
      sTable[i] <= `RW_KEY_INDEX_WIDTH'd0;
  else if (keySearchEngCs==KS_INIT)
    for (i=1;i<=`RW_KEY_INDEX_MAX-3;i=i+1)
      sTable[i] <= `RW_KEY_INDEX_WIDTH'd0;
  else
    for (i=1;i<=`RW_KEY_INDEX_MAX-3;i=i+1)
      if (sTableWriteEn[i])
         sTable[i] <= keyIndexRAM;
      else if (sTableShiftEn[i] & sTableInsert)
         sTable[i] <= sTableShiftUp[i];
      else if (sTableShiftEn[i] & ~sTableInsert)
         sTable[i] <= sTableShiftDn[i];
end


// Search Table Shifted Up   : Used to insert index into the sTable
// Search Table Shifted Down : Used to delete index from the sTable
always @*
begin: SEARCHTABLESHIFT
   integer i;

   sTableShiftUp[1]                   = `RW_KEY_INDEX_WIDTH'd0;
   sTableShiftDn[`RW_KEY_INDEX_MAX-3] = `RW_KEY_INDEX_WIDTH'd0;

   for (i=1;i<=`RW_KEY_INDEX_MAX-3-1;i=i+1)
      sTableShiftUp[i+1] = sTable[i];

   for (i=1;i<=`RW_KEY_INDEX_MAX-3-1;i=i+1)
      sTableShiftDn[i]   = sTable[i+1];
end


// Search Table Shift Enable
//    Used to select sTable entry to shift
always @*
begin: SEACHTABLESHIFTEN
   if (sTableInsert | sTableRemove)
   begin
      sTableShiftEn  = {`RW_KEY_INDEX_MAX-3{1'b1}};
      sTableShiftEn  = sTableShiftEn<<(sTableAddrReg);
      if (sTableRemove)
         sTableShiftEn  = {1'b1,sTableShiftEn[`RW_KEY_INDEX_MAX-3:2]};
   end
   else
   begin
      sTableShiftEn  = {`RW_KEY_INDEX_MAX-3{1'b0}};
   end
end


// Search Table Write Enable
//    Used to select sTable entry to write
always @*
begin
   if (sTableInsert)
   begin
      sTableWriteEn = {{`RW_KEY_INDEX_MAX-4{1'b0}},1'b1};
      sTableWriteEn = sTableWriteEn<<(sTableAddrReg);
   end
   else
   begin
      sTableWriteEn = {`RW_KEY_INDEX_MAX-3{1'b0}};
   end
end


// Search Table Data
//    Key Search index store at the search table address into search Table
//    Force to empty ('0') if address is outside sTable
always @*
  if (sTableAddr>0 & sTableAddr<=(`RW_KEY_INDEX_MAX-3))
    sTableData = sTable[sTableAddr];
  else
    sTableData = `RW_KEY_INDEX_WIDTH'd0;


// Bit selected during search process
//    Loaded with 2nd MSB set, then shifted down until LSB is '1' or macAddress found
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    bitSel <= {`RW_KEY_INDEX_WIDTH+2{1'h0}};
  else if (macCoreClkSoftRst_n == 1'b0) 
    bitSel <= {`RW_KEY_INDEX_WIDTH+2{1'b0}};
  else if (~searchInProgress | searchDone)
    bitSel <= {1'b0,1'b1,{`RW_KEY_INDEX_WIDTH{1'b0}}};
  else if (~miss)
    bitSel <= {1'b0,bitSel[`RW_KEY_INDEX_WIDTH+1:1]};
end


// search table address registered
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    sTableAddrReg <= `RW_KEY_INDEX_WIDTH'd0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    sTableAddrReg <= `RW_KEY_INDEX_WIDTH'd0;
  else if (keySearchEngCs==KS_IDLE & keySearchEngNs==SW_REMOVE_INDEX)
    sTableAddrReg <= `RW_KEY_INDEX_WIDTH'd0;
  else
    sTableAddrReg <= sTableAddr;
end


// search table address
always @*
begin
   if (searchInProgress & miss)
      //Address search prefetch not correct (step up)
      sTableAddr = sTableAddrReg | bitSel[`RW_KEY_INDEX_WIDTH-1:0];
   else if (searchInProgress & bitSel[0])
      //Last Address search
      sTableAddr = sTableAddrReg;
   else if (searchInProgress)
      //Address Search Step Down
      sTableAddr = (sTableAddrReg | bitSel[`RW_KEY_INDEX_WIDTH+1:2])
                                  ^ bitSel[`RW_KEY_INDEX_WIDTH:1];
   else if (keySearchEngCs==SW_REMOVE_INDEX & ~sTableRemoveDone)
      //Index Search Increment
      sTableAddr = sTableAddrReg+`RW_KEY_INDEX_WIDTH'd1;
   else
      //Start Address
      sTableAddr = bitSel[`RW_KEY_INDEX_WIDTH:1];
end


// search Table insert flag
//    Indicates that index should be inserted into sTable
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    sTableInsert <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    sTableInsert <= 1'b0;
  else if (keySearchEngCs==SW_INSERT_INDEX & searchInProgress & searchDone)
    sTableInsert <= 1'b1;
  else
    sTableInsert <= 1'b0;
end


// search Table Remove flag
//    Indicates that index should be removed from sTable
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    sTableRemove <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    sTableRemove <= 1'b0;
  else if (sTableRemove)
    sTableRemove <= 1'b0;
  else if (keySearchEngCs==SW_REMOVE_INDEX & sTableData==keyIndexRAM)
    sTableRemove <= 1'b1;
end


// search table remove done flag
//    indicates that the expected index has been removed from table
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    sTableRemoveDone <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    sTableRemoveDone <= 1'b0;
  else if (sTableRemoveDone)
    sTableRemoveDone <= 1'b0;
  else if (keySearchEngCs==SW_REMOVE_INDEX)
    sTableRemoveDone <= sTableRemove                       |  // Match Index
                        sTableData==0                      |  // Empty Index
                        sTableAddrReg==(`RW_KEY_INDEX_MAX-3); // Last  Index
end


// Search flag
//   Indicates that search is in progress
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    searchInProgress <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    searchInProgress <= 1'b0;
  else if (keySearchEngNs==SW_INSERT_INDEX   |
           keySearchEngNs==HW_SEARCH_MAC_ADDR|
           keySearchEngNs==SW_SEARCH_MAC_ADDR)
    searchInProgress <= 1'b1;
  else
    searchInProgress <= 1'b0;
end


// Read MAC Address Valid flag
//    Indicates that the value from keyStorage RAM is valid.
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0) 
    readMacAddrValid <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0) 
    readMacAddrValid <= 1'b0;
  else if (searchDone | miss)
    readMacAddrValid <= 1'b0;
  else if (searchInProgress & keyStorageAddr>3)
    readMacAddrValid <= 1'b1;
end


// Search MAC address
assign searchMacAddr = (keySearchEngCs==HW_SEARCH_MAC_ADDR) ? macAddressInLat : encrMACAddr;


// MAC Address read from keyStorage RAM
assign readMacAddr   = keyStorageReadData[59:12];


// Search Match
//    Indicates that the read MAC Address match the expected MAC Address
assign searchMatch  = readMacAddrValid & readMacAddr==searchMacAddr;


// Search Done
//    Indicates that the MAC Address search is finished.
//    MAC address is found (expect for Key write which should not find MAC Address)
//    last address reached.
assign searchDone   = searchMatch & keySearchEngCs==HW_SEARCH_MAC_ADDR |
                      searchMatch & keySearchEngCs==SW_SEARCH_MAC_ADDR |
                      bitSel[0];


// Miss
//    Indicates that the preloaded keyStorage index (search down by default) is
//    not correct and step up is requested.
assign miss         = readMacAddrValid & readMacAddr<searchMacAddr;


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// Disable coverage.
// pragma coverage toggle = off 
  // This signal is defined to ease the simulation and waveform analysis.
  // String definition to display keySearchEng current state
   reg [27*8:0] keySearchEngCs_str;
   
// Key Search FSM states displayed in a string to easy simulation and debug
always @ (keySearchEngCs)
begin
  case (keySearchEngCs)
    KS_IDLE             :  keySearchEngCs_str = {"KS_IDLE"};           
    KS_INIT             :  keySearchEngCs_str = {"KS_INIT"};           
    HW_SEARCH_MAC_ADDR  :  keySearchEngCs_str = {"HW_SEARCH_MAC_ADDR"};
    HW_RD_KS_RAM        :  keySearchEngCs_str = {"HW_RD_KS_RAM"};      
    SW_WR_KS_RAM        :  keySearchEngCs_str = {"SW_WR_KS_RAM"};      
    SW_REMOVE_INDEX     :  keySearchEngCs_str = {"SW_REMOVE_INDEX"};
    SW_INSERT_INDEX     :  keySearchEngCs_str = {"SW_INSERT_INDEX"};
    SW_RD_KS_RAM        :  keySearchEngCs_str = {"SW_RD_KS_RAM"};      
    SW_SEARCH_MAC_ADDR  :  keySearchEngCs_str = {"SW_SEARCH_MAC_ADDR"};   
    // pragma coverage block = off 
    default             :  keySearchEngCs_str = {"XXX"};
    // pragma coverage block = on 
  endcase
end
// pragma coverage toggle = on 

`endif // RW_SIMU_ON


// System Verilog Assertions
////////////////////////////

`ifdef RW_ASSERT_ON

   // Checks if there is no simultaneous read or write initiated by Software/CSR
   property noSimultaneousCsrReadWrite_prop;
   @(posedge macCoreClk)
     (newWrite == 1'b1) |-> (newRead == 1'b0);
   endproperty
   noSimultaneousCsrReadWrite: assert property (noSimultaneousCsrReadWrite_prop);
  
   // Checks if there is no simultaneous read or write initiated by Software/CSR
   property noSimultaneousCsrReadSearch_prop;
   @(posedge macCoreClk)
     (newSearch == 1'b1) |-> (newRead == 1'b0);
   endproperty
   noSimultaneousCsrReadSearch: assert property (noSimultaneousCsrReadSearch_prop);
  
    // Checks if there is no simultaneous read or write initiated by Software/CSR
   property noSimultaneousCsrSearchWrite_prop;
   @(posedge macCoreClk)
     (newWrite == 1'b1) |-> (newSearch == 1'b0);
   endproperty
   noSimultaneousCsrSearchWrite: assert property (noSimultaneousCsrSearchWrite_prop);
   
  
   // Checks if there is no simultaneous assertions of Index and MAC address search
   property checkIndexAndMACAddressSearch_prop;
   @(posedge macCoreClk)
     (indexSearchTrig_p == 1'b1) |-> (keySearchIndexTrig_p == 1'b0);
   endproperty
   checkIndexAndMACAddressSearch: assert property (checkIndexAndMACAddressSearch_prop);

`endif
endmodule
                 

