//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Controls the Tx part of the Phy interface
//
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module TxStateCtl(
         //$port_g clock and Reset
         input  wire        mpIFClk,                   // PHY clock
         input  wire        mpIFClkHardRst_n,          // PHY clock reset
         input  wire        mpIFClkSoftRst_n,          // PHY clock soft reset
         input  wire        macCoreTxClk,              // mac core Tx clock
         input  wire        macCoreClkHardRst_n,       // mac core Tx clock hard reset

         //$port_g Tx  Controller If
         input  wire        startTx_p,                 // Start Tx trigger
         input  wire        stopTx_p,                  // Stop Tx trigger
         input  wire [ 7:0] txPwrLevel,                // Transmit Power Level
         input  wire [ 2:0] chBW,                      // TX Vector Channel Bandwidth
         input  wire        smoothing,                 // TX Vector Smoothing recommended
         input  wire [ 7:0] antennaSet,                // TX Vector Antenna Set
         input  wire [ 7:0] smmIndex,                  // TX Vector Spatial Map Matrix Index
         input  wire [ 6:0] mcs,                       // TX Vector Modulation Coding Scheme
         input  wire        preType,                   // TX Vector Preamble Type
         input  wire [ 3:0] formatMod,                 // TX Vector Format and Modulation
         input  wire [ 1:0] numExtnSS,                 // TX Vector Number of Extension Spatial Streams
         input  wire        stbc,                      // TX Vector Space Time Block Coding
         input  wire        fecCoding,                 // TX Vector FEC Coding
         input  wire        sounding,                  // TX Vector Indicates whether this PPDU is Sounding
         input  wire [11:0] legLength,                 // TX Vector Legacy Length of the PPDU
         input  wire [ 3:0] legRate,                   // TX Vector Legacy Rate of the PPDU
         input  wire [15:0] service,                   // TX Vector Service
         input  wire [19:0] htLength,                  // TX Vector Length of the HT and VHT PPDU
         input  wire        htLengthValid,             // TX Vector Length of the HT/VHT/HE PPDU is
                                                       // valid
         input  wire [ 2:0] nTx,                       // TX Vector Number of Transmit Chains.
         input  wire        aggreation,                // TX Vector MPDU Aggregate
         input  wire        beamFormed,                // TX Vector BeamFormed
         input  wire  [8:0] partialAID,                // TX Vector partial AID
         input  wire  [5:0] groupID,                   // TX Vector group ID
         input  wire        dozeNotAllowed,            // TX Vector TXOP PS Not Allowed
         input  wire        continuousTx,              // TX Vector continuous transmit
         input  wire  [4:0] userPosition,              // TX Vector user position
         input  wire        timeOfDepartureReq,        // TX Vector Time of departure requested
         input  wire  [1:0] giType,                    // TX Vector Guard Interval Type
         input  wire        upLinkFlag,                // TX Vector UP Link Flag
         input  wire        beamChange,                // TX Vector Beam Change
         input  wire        dcm,                       // TX Vector Dual Carrier Modulation
         input  wire  [1:0] heLTFType,                 // TX Vector Type of HE LTF
         input  wire        doppler,                   // TX Vector Doppler
         input  wire        midamble,                  // TX Vector Midamble
         input  wire  [5:0] bssColor,                  // TX Vector BSS Color
         input  wire  [6:0] txopDuration,              // TX Vector Duration of TXOP
         input  wire        triggerResponding,         // TX Vector Trigger Responding
         input  wire  [7:0] nUser,                     // TX Vector Number of User
         input  wire [10:0] staID,                     // TX Vector STA ID
         input  wire  [3:0] spatialReuse1,             // TX Vector Spatial Reuse 1
         input  wire  [3:0] spatialReuse2,             // TX Vector Spatial Reuse 2
         input  wire  [3:0] spatialReuse3,             // TX Vector Spatial Reuse 3
         input  wire  [3:0] spatialReuse4,             // TX Vector Spatial Reuse 4
         input  wire  [8:0] heSigAReserved,            // TX Vector HE SIG-A Reserved bits
         input  wire  [2:0] numHeLtf,                  // TX Vector Number of HE LTF
         input  wire        heLtfMode,                 // TX Vector HE LTF Mode
         input  wire        ldpcExtraSymbol,           // TX Vector LDPC extra symbol
         input  wire  [2:0] startingSTSNum,            // TX Vector Starting STS Number
         input  wire        sigBCompressionMode,       // TX Vector SIG-B compression mode
         input  wire        dcmSigB,                   // TX Vector Dual carrier modulation on SIG-B
         input  wire  [2:0] mcsSigB,                   // TX Vector MCS of SIG-B
         input  wire [31:0] ruAllocation,              // TX Vector RU allocation
         input  wire  [2:0] packetExtension,           // TX Vector Packet extension
         input  wire        triggerMethod,             // TX Vector trigger method
         input  wire  [6:0] ruToneSetIndex,            // TX Vector RU tone set index
         input  wire        feedBackStatus,            // TX Vector feedback status
         //
         output wire [31:0] timeOfDeparture,           // TX Vector2 Time Of Departure
         output wire [15:0] timeOfDepartureClkRate,    // TX Vector2 Time Of Departure Clock Rate
         output wire        mpIfTxErr_p,               // indicate a phy error to Tx controller

         //$port_g Phy If
         input  wire        phyRdy,                    // data valid signal
         input  wire        txEnd_p,                   // data enable
         input  wire        phyErr_p,                  // phy error
         input  wire [ 7:0] rxData,                    // Rx data
         //
         output reg         txReq,                     // Tx request
         output wire [ 7:0] txData,                    // Tx data
         output reg         macDataValid,              // Tx data valid

         //$port_g registers
         input  wire        macPHYIFFIFOReset,         // Flush the FIFO
         input  wire        rateControllerMPIF,        // select transmit mechanism
                                                       // 0 phy request (i.e PHY fetches at anyrate)
                                                       // 1 phy strobe (i.e PHY fetches at air datarate)

         //$port_g FIFO Interface
         input  wire [ 7:0] mpIfTxFifoDataout,         // Tx FIFO input
         input  wire        mpIfTxFifoEmtpy,           // Tx FIFO empty
         //
         output reg         mpIfFifoReadEn,            // Read enable
         output reg         txFifoRdClkFlush,          // flush FIFO

         //$port_g Keep RF
         input  wire        mackeepRFOn,               // input Keep RF
         //
         output reg         TxkeepRFOn,                // output Keep RF

         //$port_g Debug
         output wire [15:0] debugPortMACPhyIf3,        // Debug Port
         output reg         mpifUnderRun               // underrun
         );

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
//$fsm_sd txstate
localparam
   IDLE            = 3'd0, //
   TX_VECTOR       = 3'd1, // Tx Vector
   TX_DATA         = 3'd2, // Tx Data
   TX_END          = 3'd3, // Tx End
   TX_VECTOR2      = 3'd4, // Tx Vector Response
   TX_VECTOR_PAUSE = 3'd5; // Tx Vector Pause

// Format and Modulation
localparam
   NON_HT       = 4'b0000,
   NON_HT_DUP   = 4'b0001,
   HT_MM        = 4'b0010,
   HT_GF        = 4'b0011,
   VHT          = 4'b0100,
   HE_SU        = 4'b0101,
   HE_MU        = 4'b0110,
   HE_EXT_SU    = 4'b0111,
   HE_TRIG      = 4'b1000;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
// txstate FSM signals definition
reg   [2:0] txstateCs;                 // txstate FSM Current State
reg   [2:0] txstateNs;                 // txstate FSM Next State

reg   [7:0] txVector;                  // formatted tx vector byte
reg  [47:0] txVector2;                 // Tx vector2
reg   [4:0] txVectorCnt;               // Tx vector counter
wire        txVectorPause;             // Tx vector pause

wire        fifo_reset;                // fifo reset from registers

reg         phyRdy_reg;                // registered phyRdy
reg   [7:0] continuousTxData;          // Continuous Tx Data Byte
reg         stopTxFlg;                 // Memorize the pulse stopTx_p when it happended
                                       // in TX_VECTOR state
wire        continuousTxMPIF;          // continuousTx resynchronized on mpIFClk.

reg   [4:0] phyRdyCnt;                 // Count the number of phyRdy pending request in
                                       // case of rateControllerMPIF = 0.
                                       // The PHY guaranties a maximum of 16 pending requests.
wire        txVectorEnd;               // Last Byte of TX vector flag
wire        txVector2End;              // Last Byte of TX vector2 flag

wire  [7:0] txVectorCommon[0:7];       // TX Vector Common Part for all frames
wire  [7:0] txVectorHeCommon[8:12];    // TX Vector HE Common Part
reg   [7:0] txVectorNonHt;             // TX Vector for NON-HT frames
reg   [7:0] txVectorHt;                // TX Vector for HT frames
reg   [7:0] txVectorVht;               // TX Vector for VHT frames
reg   [7:0] txVectorHeSu;              // TX Vector for HE-SU frames
reg   [7:0] txVectorHeMu;              // TX Vector for HE-MU frames
reg   [7:0] txVectorHeTrig;            // TX Vector for HE-TRIG frames


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

assign txData = (txstateCs == IDLE)            ? 8'h0             :
                (txstateCs == TX_VECTOR2)      ? 8'h0             :
                (txstateCs == TX_VECTOR ||
                 txstateCs == TX_VECTOR_PAUSE) ? txVector         :
                (continuousTxMPIF)             ? continuousTxData :
                                                 mpIfTxFifoDataout;

always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    continuousTxData <= 8'b0;
  else if(continuousTxMPIF && phyRdy_reg)
    continuousTxData <= continuousTxData + 8'b1;
end

always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    stopTxFlg <= 1'b0;
  else if((mpIFClkSoftRst_n == 1'b0) || (txstateCs == TX_END) || startTx_p)
    stopTxFlg <= 1'b0;
  else if((stopTx_p) && ((txstateCs == TX_VECTOR) || (txstateCs == TX_VECTOR_PAUSE)))
    stopTxFlg <= 1'b1;
end


pulse2PulseSynchro U_stoptTx_p_synchro (
                .srcclk (mpIFClk),
                .srcresetn (mpIFClkHardRst_n),
                .dstclk (macCoreTxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (phyErr_p),
                .dstdata (mpIfTxErr_p)
                );

pulse2PulseSynchro U_fiforeset_synchro (
                .srcclk (macCoreTxClk),
                .srcresetn (macCoreClkHardRst_n),
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (macPHYIFFIFOReset),
                .dstdata (fifo_reset)
                );

simpleSynchro U_continuousTx_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (continuousTx),
                .dstdata (continuousTxMPIF)
                );


//------------------------------------------------------------------------------
// txstate FSM Current State Logic
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    txstateCs <= IDLE;
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
    txstateCs <= IDLE;
  else
    txstateCs <= txstateNs;
end


//------------------------------------------------------------------------------
// txstate FSM Next State Logic.
//------------------------------------------------------------------------------
always @*
begin
  case(txstateCs)
    IDLE:
      //$fsm_s In IDLE state, the state machine waits for a Transmit transaction
      if (startTx_p)
        //$fsm_t When startTx_p is true, the state machine goes to the
        //TX_VECTOR state
        txstateNs = TX_VECTOR;
      else
        txstateNs = txstateCs;

    TX_VECTOR:
      //$fsm_s In TX_VECTOR state, the state machine waits for the end of
      // the vector transmission or for an abort
      if (txVectorPause)
        //$fsm_t When next byte of txVector is not ready, the state machine
        //goes to TX_VECTOR_PAUSE
        txstateNs = TX_VECTOR_PAUSE;
      else if (txVectorEnd)
      begin
        if(stopTxFlg | stopTx_p)
          //$fsm_t When the is a MAC aborted durring the TX_VECTOR, the state
          //machine goes to TX_END state at the end of TX_VECTOR
          txstateNs = TX_END;
        else
          //$fsm_t When Last Tx Vector byte, the Tx vectors state ends and the
          //state machine goes to TX_DATA state
          txstateNs = TX_DATA;
      end
      else if (phyErr_p)
        //$fsm_t When the is a MAC or a PHY abort, the state machine goes to
        //TX_END state
        txstateNs = TX_END;
      else
        txstateNs = txstateCs;

    TX_VECTOR_PAUSE:
      //$fsm_s In TX_VECTOR_PAUSE state, the state machine waits until HTLength
      //is valid
      if (htLengthValid)
        //$fsm_t When next byte of txVector is ready, the state machine
        //goes to TX_VECTOR
        txstateNs = TX_VECTOR;
      else
        txstateNs = txstateCs;

    TX_DATA:
      //$fsm_s In TX_DATA state, the state machine waits for the end of
      //transmission
      if (txEnd_p)
        //$fsm_t When txend is true, the state machine goes to TX_VECTOR2 state
        txstateNs = TX_VECTOR2;
      else if (stopTx_p | phyErr_p)
        //$fsm_t When stopTx_p or phyErr_p occurs, the state machine goes to
        //TX_END state
        txstateNs = TX_END;
      else
        txstateNs = txstateCs;

    TX_END:
      //$fsm_s In TX_END state, the state machine waits for the tx end signal
      // from the phy
      if (txEnd_p)
        //$fsm_t When txend is true, the state machine goes to IDLE state
        txstateNs = IDLE;
      else
        txstateNs = txstateCs;

    TX_VECTOR2:
      //$fsm_s In TX_VECTOR2 state, the state machine waits for the tx end of
      // the TX vector2 from the phy
      if (txVector2End)
        //$fsm_t When TX vector2 is complete, the state machine goes to IDLE
        //state
        txstateNs = IDLE;
      else
        txstateNs = txstateCs;

    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off
    default:
      txstateNs = IDLE;
    // pragma coverage block = on
   endcase
end


//------------------------------------------------------------------------------
// tx vector counter
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    txVectorCnt <= 5'b0000;
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
    txVectorCnt <= 5'b0000;
  // increment vector count each time a vector is sampled by the rx FIFO
  else if ( txstateCs == TX_VECTOR  |
            txstateCs == TX_VECTOR2 )
    txVectorCnt <= txVectorCnt + 5'd1;
  else if ( txstateCs == TX_VECTOR_PAUSE )
    txVectorCnt <= txVectorCnt;
  else
    txVectorCnt  <= 5'b0000;
end


//------------------------------------------------------------------------------
// Tx Vector pause
//------------------------------------------------------------------------------
// In case of HE TB frame, HT Length can be ready atfer the start of the
// txVector transmission. In this case, the txVector must be stopped until 
// HT Length is available.
assign txVectorPause = formatMod==HE_TRIG & txVectorCnt==5'd19 & ~htLengthValid;


//------------------------------------------------------------------------------
// Last Tx Vector Byte Flag
//------------------------------------------------------------------------------
// TX Vector End flag is generated when all byte has been provided
// (counter = number of byte)
assign txVectorEnd = formatMod==NON_HT     & txVectorCnt==5'd10 |
                     formatMod==NON_HT_DUP & txVectorCnt==5'd10 |
                     formatMod==HT_MM      & txVectorCnt==5'd13 |
                     formatMod==HT_GF      & txVectorCnt==5'd13 |
                     formatMod==VHT        & txVectorCnt==5'd17 |// MultiUser not yet supported
                     formatMod==HE_SU      & txVectorCnt==5'd18 |
                     formatMod==HE_EXT_SU  & txVectorCnt==5'd18 |
                     formatMod==HE_MU      & txVectorCnt==5'd26 |// MultiUser not yet supported
                     formatMod==HE_TRIG    & txVectorCnt==5'd24 ;

// TX Vector2 End flag is generated when the last byte is received
// (counter = last byte index)
assign txVector2End = txVectorCnt==5'd5;


//------------------------------------------------------------------------------
// Tx Vectors
//------------------------------------------------------------------------------
// TX Vector Common Part for all frames
assign txVectorCommon[0]    = {preType,chBW,formatMod};
assign txVectorCommon[1]    = antennaSet;
assign txVectorCommon[2]    = txPwrLevel;
assign txVectorCommon[3]    = {continuousTx,timeOfDepartureReq,3'b0,nTx};
assign txVectorCommon[4]    = legLength[7:0];
assign txVectorCommon[5]    = {legRate,legLength[11:8]};
assign txVectorCommon[6]    = service[7:0];
assign txVectorCommon[7]    = service[15:8];

// TX Vector HE Common Part
assign txVectorHeCommon[8]  = {3'b0,stbc,giType,beamFormed,sounding};
assign txVectorHeCommon[9]  = {1'b0,midamble,doppler,heLTFType,dcm,beamChange,upLinkFlag};
assign txVectorHeCommon[10] = {2'b0,bssColor};
assign txVectorHeCommon[11] = {1'b0,txopDuration};
assign txVectorHeCommon[12] = {4'b0,spatialReuse1};

// TX Vector for NON-HT frames
always @*
begin
   case (txVectorCnt)
   5'd0:    txVectorNonHt = txVectorCommon[0];
   5'd1:    txVectorNonHt = txVectorCommon[1];
   5'd2:    txVectorNonHt = txVectorCommon[2];
   5'd3:    txVectorNonHt = txVectorCommon[3];
   5'd4:    txVectorNonHt = txVectorCommon[4];
   5'd5:    txVectorNonHt = txVectorCommon[5];
   5'd6:    txVectorNonHt = txVectorCommon[6];
   5'd7:    txVectorNonHt = txVectorCommon[7];
   5'd8:    txVectorNonHt = {7'b0,triggerResponding};
   5'd9:    txVectorNonHt = smmIndex;
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default: txVectorNonHt = 8'b0;
   // pragma coverage block = on
   endcase
end
                       
// TX Vector for HT frames 
always @*
begin
   case (txVectorCnt)
   5'd0:    txVectorHt    = txVectorCommon[0];
   5'd1:    txVectorHt    = txVectorCommon[1];
   5'd2:    txVectorHt    = txVectorCommon[2];
   5'd3:    txVectorHt    = txVectorCommon[3];
   5'd4:    txVectorHt    = txVectorCommon[4];
   5'd5:    txVectorHt    = txVectorCommon[5];
   5'd6:    txVectorHt    = txVectorCommon[6];
   5'd7:    txVectorHt    = txVectorCommon[7];
   5'd8:    txVectorHt    = {1'b0,numExtnSS,stbc,aggreation,giType[0],smoothing,sounding};
   5'd9:    txVectorHt    = smmIndex;
   5'd10:   txVectorHt    = {fecCoding,mcs};
   5'd11:   txVectorHt    = htLength[7:0];
   5'd12:   txVectorHt    = htLength[15:8];
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default: txVectorHt    = 8'b0;
   // pragma coverage block = on
   endcase
end

// TX Vector for VHT frames
always @*
begin
   case (txVectorCnt)
   5'd0:    txVectorVht   = txVectorCommon[0];
   5'd1:    txVectorVht   = txVectorCommon[1];
   5'd2:    txVectorVht   = txVectorCommon[2];
   5'd3:    txVectorVht   = txVectorCommon[3];
   5'd4:    txVectorVht   = txVectorCommon[4];
   5'd5:    txVectorVht   = txVectorCommon[5];
   5'd6:    txVectorVht   = txVectorCommon[6];
   5'd7:    txVectorVht   = txVectorCommon[7];
   5'd8:    txVectorVht   = {2'b0,dozeNotAllowed,stbc,1'b0,giType[0],beamFormed,sounding};
   5'd9:    txVectorVht   = partialAID[7:0];
   5'd10:   txVectorVht   = {1'b0,groupID,partialAID[8]};
   5'd11:   txVectorVht   = {5'b0,nUser[2:0]};
   5'd12:   txVectorVht   = smmIndex;
   5'd13:   txVectorVht   = {fecCoding,mcs[6:4],mcs[3:0]};
   5'd14:   txVectorVht   = htLength[7:0];
   5'd15:   txVectorVht   = htLength[15:8];
   5'd16:   txVectorVht   = {userPosition[1:0],2'b0,htLength[19:16]};
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default: txVectorVht   = 8'b0;
   // pragma coverage block = on
   endcase
end

// TX Vector for HE-SU frames
always @*
begin
   case (txVectorCnt)
   5'd0:    txVectorHeSu  = txVectorCommon[0];
   5'd1:    txVectorHeSu  = txVectorCommon[1];
   5'd2:    txVectorHeSu  = txVectorCommon[2];
   5'd3:    txVectorHeSu  = txVectorCommon[3];
   5'd4:    txVectorHeSu  = txVectorCommon[4];
   5'd5:    txVectorHeSu  = txVectorCommon[5];
   5'd6:    txVectorHeSu  = txVectorCommon[6];
   5'd7:    txVectorHeSu  = txVectorCommon[7];
   5'd8:    txVectorHeSu  = txVectorHeCommon[8];
   5'd9:    txVectorHeSu  = txVectorHeCommon[9];
   5'd10:   txVectorHeSu  = txVectorHeCommon[10];
   5'd11:   txVectorHeSu  = txVectorHeCommon[11];
   5'd12:   txVectorHeSu  = txVectorHeCommon[12];
   5'd13:   txVectorHeSu  = smmIndex;
   5'd14:   txVectorHeSu  = {fecCoding,mcs[6:4],mcs[3:0]};
   5'd15:   txVectorHeSu  = htLength[7:0];
   5'd16:   txVectorHeSu  = htLength[15:8];
   5'd17:   txVectorHeSu  = {1'b0,packetExtension,htLength[19:16]};
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default: txVectorHeSu  = 8'b0;
   // pragma coverage block = on
   endcase
end

// TX Vector for HE-MU frames
always @*
begin
   case (txVectorCnt)
   5'd0:    txVectorHeMu  = txVectorCommon[0];
   5'd1:    txVectorHeMu  = txVectorCommon[1];
   5'd2:    txVectorHeMu  = txVectorCommon[2];
   5'd3:    txVectorHeMu  = txVectorCommon[3];
   5'd4:    txVectorHeMu  = txVectorCommon[4];
   5'd5:    txVectorHeMu  = txVectorCommon[5];
   5'd6:    txVectorHeMu  = txVectorCommon[6];
   5'd7:    txVectorHeMu  = txVectorCommon[7];
   5'd8:    txVectorHeMu  = txVectorHeCommon[8];
   5'd9:    txVectorHeMu  = txVectorHeCommon[9];
   5'd10:   txVectorHeMu  = txVectorHeCommon[10];
   5'd11:   txVectorHeMu  = txVectorHeCommon[11];
   5'd12:   txVectorHeMu  = txVectorHeCommon[12];
   5'd13:   txVectorHeMu  = {3'b0,mcsSigB,dcmSigB,sigBCompressionMode};
   5'd14:   txVectorHeMu  = ruAllocation[7:0];
   5'd15:   txVectorHeMu  = ruAllocation[15:8]; 
   5'd16:   txVectorHeMu  = ruAllocation[23:16];
   5'd17:   txVectorHeMu  = ruAllocation[31:24];
   5'd18:   txVectorHeMu  = nUser;
   5'd19:   txVectorHeMu  = smmIndex;
   5'd20:   txVectorHeMu  = {fecCoding,mcs[6:4],mcs[3:0]};
   5'd21:   txVectorHeMu  = htLength[7:0];
   5'd22:   txVectorHeMu  = htLength[15:8];
   5'd23:   txVectorHeMu  = {1'b0,packetExtension,htLength[19:16]};
   5'd24:   txVectorHeMu  = staID[7:0];
   5'd25:   txVectorHeMu  = {userPosition,staID[10:8]};
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default: txVectorHeMu  = 8'b0;
   // pragma coverage block = on
   endcase
end

// TX Vector for HE-TRIG frames
always @*
begin
   case (txVectorCnt)
   5'd0:    txVectorHeTrig = txVectorCommon[0];
   5'd1:    txVectorHeTrig = txVectorCommon[1];
   5'd2:    txVectorHeTrig = txVectorCommon[2];
   5'd3:    txVectorHeTrig = txVectorCommon[3];
   5'd4:    txVectorHeTrig = txVectorCommon[4];
   5'd5:    txVectorHeTrig = txVectorCommon[5];
   5'd6:    txVectorHeTrig = txVectorCommon[6];
   5'd7:    txVectorHeTrig = txVectorCommon[7];
   5'd8:    txVectorHeTrig = txVectorHeCommon[8];
   5'd9:    txVectorHeTrig = txVectorHeCommon[9];
   5'd10:   txVectorHeTrig = txVectorHeCommon[10];
   5'd11:   txVectorHeTrig = txVectorHeCommon[11];
   5'd12:   txVectorHeTrig = {spatialReuse2,spatialReuse1};
   5'd13:   txVectorHeTrig = {spatialReuse4,spatialReuse3};
   5'd14:   txVectorHeTrig = heSigAReserved[7:0];
   5'd15:   txVectorHeTrig = {2'd0,ldpcExtraSymbol,heLtfMode,numHeLtf,heSigAReserved[8]};
   5'd16:   txVectorHeTrig = {5'd0,startingSTSNum};
   5'd17:   txVectorHeTrig = ruAllocation[7:0];
   5'd18:   txVectorHeTrig = smmIndex;
   5'd19:   txVectorHeTrig = {fecCoding,mcs[6:4],mcs[3:0]};
   5'd20:   txVectorHeTrig = htLength[7:0];
   5'd21:   txVectorHeTrig = htLength[15:8];
   5'd22:   txVectorHeTrig = {triggerMethod,packetExtension,htLength[19:16]};
   5'd23:   txVectorHeTrig = {feedBackStatus,ruToneSetIndex};
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default: txVectorHeTrig = 8'b0;
   // pragma coverage block = on
   endcase
end


always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    txVector <= 8'b0;
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
    txVector <= 8'b0;
  else
  begin
     if (txstateCs == TX_VECTOR)
     begin
       case (formatMod)
       NON_HT:       txVector <= txVectorNonHt;
       NON_HT_DUP:   txVector <= txVectorNonHt;
       HT_MM:        txVector <= txVectorHt;
       HT_GF:        txVector <= txVectorHt;
       VHT:          txVector <= txVectorVht;
       HE_SU:        txVector <= txVectorHeSu;
       HE_EXT_SU:    txVector <= txVectorHeSu;
       HE_MU:        txVector <= txVectorHeMu;
       HE_TRIG:      txVector <= txVectorHeTrig;
       default:      txVector <= 8'b0;
       endcase
     end
     else
       txVector <= 8'b0;
  end
end


//------------------------------------------------------------------------------
// Tx Vector 2
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    txVector2 <= 48'b0;
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
    txVector2 <= 48'b0;
  else
  begin
     if (txstateCs == TX_VECTOR2)
       txVector2 <= {rxData,txVector2[47:8]};
  end
end

assign timeOfDeparture        = txVector2[31:0];
assign timeOfDepartureClkRate = txVector2[47:32];


//------------------------------------------------------------------------------
// control interface
// - Mac Data valid and FIFO flush
// - keepRfon latch
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    macDataValid     <= 1'b0;
    txFifoRdClkFlush <= 1'b0;
    txReq            <= 1'b0;
    TxkeepRFOn       <= 1'b0;
  end
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
  begin
    macDataValid     <= 1'b0;
    txFifoRdClkFlush <= 1'b0;
    txReq            <= 1'b0;
    TxkeepRFOn       <= 1'b0;
  end
  else
  begin
    // Sample TxkeeRFOn only when TxReq is asserted
    if (txReq)
      TxkeepRFOn <= mackeepRFOn;

    // when the State Machine is not idle, assert Tx Req
    if (txstateCs==TX_VECTOR)
      txReq            <= 1'b1;
    else if (txstateCs==IDLE                      |
             txstateCs==TX_DATA    & stopTx_p     |//Mac Abort in Tx DATA
             txstateCs==TX_END     & stopTxFlg    |//Mac Abort in Tx Vector
             txstateCs==TX_END     & txEnd_p      |//Phy Abort
             txstateCs==TX_VECTOR2 & txVector2End )
      txReq            <= 1'b0;

    // flush in case of external flush, abort of return to idle
    if  ((fifo_reset              ) |
         (txstateCs  == TX_END    ) |
         (txstateCs  == TX_VECTOR2))
      txFifoRdClkFlush <= 1'b1;
    else
      txFifoRdClkFlush <= 1'b0;

    // generate MAC data valid during TxVector and when a Read as been done
    if (txstateCs == TX_VECTOR & !txVectorEnd)
      macDataValid    <= 1'b1;
    else if  (txstateCs  == TX_DATA)
    begin
      if (mpIfFifoReadEn == 1'b1)
        macDataValid    <= 1'b1;
      else
        macDataValid    <= 1'b0;
    end
    else
    begin
      macDataValid     <= 1'b0;
    end
  end
end


//------------------------------------------------------------------------------
// Phy Ready register
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    phyRdy_reg <= 1'b0;
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
    phyRdy_reg <= 1'b0;
  else
    phyRdy_reg <= phyRdy;
end


//------------------------------------------------------------------------------
// Phy Ready Counter register
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    phyRdyCnt <= 5'b0;
  else if (mpIFClkSoftRst_n == 1'b0 || txEnd_p)  // Synchronous Reset
    phyRdyCnt <= 5'b0;
  // Enable the pending phyRdy request counter if the rateControllerMPIF is disabled
  // (i.e PHY fetches  at anyrate).
  // Note that the PHY guaranties a maximum of 16 pending requests.
  else if (!rateControllerMPIF && (txstateCs == TX_DATA))
  begin
    if (!phyRdy && mpIfFifoReadEn)
      phyRdyCnt <= phyRdyCnt - 5'b1;
    else if (phyRdy && ! mpIfFifoReadEn)
      phyRdyCnt <= phyRdyCnt + 5'b1;
  end
end


//------------------------------------------------------------------------------
// Generate FIFO read enable
//------------------------------------------------------------------------------
always @*
begin
  if  (txstateCs == TX_DATA)
  begin
    // In case of rateControllerMPIF enabled (i.e PHY fetches at the air rate),
    // a read access is requested in the TXFIFO each time the phyRdy_reg is
    // received and txFIFO is not empty (or continuousTxMPIF is set).
    if (rateControllerMPIF && phyRdy_reg && (!mpIfTxFifoEmtpy || continuousTxMPIF))
      mpIfFifoReadEn = 1'b1;
    // In case of rateControllerMPIF disabled (i.e PHY fetches  at anyrate), a
    // read access is requested in the TXFIFO each time the number of pending
    // phyRdyCnt is not null and the txFIFO is not empty (or continuousTxMPIF is set).
    else if (!rateControllerMPIF && (phyRdyCnt != 5'd0) && (!mpIfTxFifoEmtpy || continuousTxMPIF))
      mpIfFifoReadEn = 1'b1;
    else
      mpIfFifoReadEn = 1'b0;
  end
  else
  begin
    mpIfFifoReadEn   = 1'b0;
  end
end


//------------------------------------------------------------------------------
// UnderRun detection
//------------------------------------------------------------------------------
always @ (posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)  // Asynchronous Reset
    mpifUnderRun             <= 1'b0;
  else if (mpIFClkSoftRst_n == 1'b0)  // Synchronous Reset
    mpifUnderRun             <= 1'b0;
  else
  begin
    if (txstateCs != IDLE)
      // If the rateControllerMPIF is enabled (i.e PHY fetches at the air rate),
      // and a phyRdy is received with an empty Tx FIFO, an error is triggered.
      if (rateControllerMPIF && phyRdy_reg && (mpIfTxFifoEmtpy && !continuousTxMPIF))
        mpifUnderRun  <= 1'b1;
      // In case of rateControllerMPIF disabled (i.e PHY fetches at anyrate and
      // detects the underrun), and if the phyErr_p occurs during the
      // transmission of data, it indicates a mpifUnderRun.
      else if (!rateControllerMPIF && phyErr_p && (txstateCs == TX_DATA))
        mpifUnderRun  <= 1'b1;
      // In case of rateControllerMPIF, the number of prefetched bytes cannot exceed 16.
      // Thus, if the phyRdyCnt > 16 and a phyRdy is received, an error is triggered.
      else if (!rateControllerMPIF && (phyRdyCnt>5'd16) && phyRdy_reg)
        mpifUnderRun  <= 1'b1;
      else
        // Keep the mpifUnderRun status until the FSM moves back to IDLE.
        mpifUnderRun  <= mpifUnderRun;

    else
      mpifUnderRun    <= 1'b0;
  end
end



assign debugPortMACPhyIf3 = {(txstateCs == TX_DATA) ? phyRdyCnt : txVectorCnt,        // 5
                             txstateCs,                                               // 3
                             mpifUnderRun,                                            // 1
                             mpIfFifoReadEn,                                          // 1
                             txVectorEnd,                                             // 1
                             stopTx_p,                                                // 1
                             stopTxFlg,                                               // 1
                             txEnd_p,                                                 // 1
                             txVector2End,                                            // 1
                             txVectorPause                                            // 1
                             };


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_SIMU_ON
// String definition to dislpay txstate current state
reg [16*8:0] txstateCs_str;

// txstate FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (txstateCs)
  IDLE            : txstateCs_str = {"IDLE"};
  TX_VECTOR       : txstateCs_str = {"TX_VECTOR"};
  TX_VECTOR_PAUSE : txstateCs_str = {"TX_VECTOR_PAUSE"};
  TX_DATA         : txstateCs_str = {"TX_DATA"};
  TX_END          : txstateCs_str = {"TX_END"};
  TX_VECTOR2      : txstateCs_str = {"TX_VECTOR2"};
  endcase
end
`endif // RW_SIMU_ON
endmodule

