//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : This module contains all the signals resynchronization 
//                    circuitry required between macCoreClk clock domain and 
//                    macPIClk clock domain.
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module macCoreBFRResync ( 
  ///////////////////////////////////////////////
  //$port_g Resets
  ///////////////////////////////////////////////
  input  wire         mpIFClkHardRst_n,             // Active low hard reset signal synchronized to the mpIFClk.
  input  wire         macCoreClkHardRst_n,          // Active low hard reset signal synchronized to the macCoreClk.
  input  wire         macCoreClkSoftRst_n,          // Active low soft reset signal synchronized to the macCoreClk.

  ///////////////////////////////////////////////
  //$port_g Clocks
  ///////////////////////////////////////////////
  input  wire         mpIFClk,                      // Primary MAC PHY Interface Clock
  input  wire         macCoreClk,                   // Primary MAC Core Clock
 
  ///////////////////////////////////////////////
  //$port_g BFR HW Accelerator Interface
  ///////////////////////////////////////////////
  output reg          mpifBfrStart,              // BFR computation started
  input  wire         mpifBfrDone,               // Beamforming Report computation is done
  output reg    [1:0] mpifBfrChBW,               // Beamforming report Channel Width
  output reg    [1:0] mpifBfrGrouping,           // Beamforming report Grouping
  output reg          mpifBfrCodebook,           // Beamforming report Codebook Information
  output reg    [1:0] mpifBfrFeedbackType,       // Beamforming report Feedback Type
  output reg    [2:0] mpifBfrNr,                 // Beamforming report Nr Index
  output reg    [2:0] mpifBfrNc,                 // Beamforming report Nc Index
  output reg          mpifBfrHE,                 // Beamforming report Format
                                                 // (VHT or HE)
  output reg    [6:0] mpifBfrRUStartIndex,       // Beamforming report RU Start Index
                                                 // (Only for HE BFR)
  output reg    [6:0] mpifBfrRUEndIndex,         // Beamforming report RU End Index 
                                                 // (Only for HE BFR)
  input  wire   [7:0] mpifBfrData,               // Beamforming report byte
  input  wire         mpifBfrDataValid,          // Beamforming report byte valid          
  output wire         mpifBfrDataReady,          // Beamforming report byte request


  ///////////////////////////////////////////////
  //$port_g BFR Controller Interface
  ///////////////////////////////////////////////
  input  wire         macBfrStart,              // BFR computation started
  output wire         macBfrDone,               // Beamforming Report computation is done
  input  wire   [1:0] macBfrChBW,               // Beamforming report Channel Width
  input  wire   [1:0] macBfrGrouping,           // Beamforming report Grouping
  input  wire         macBfrCodebook,           // Beamforming report Codebook Information
  input  wire   [1:0] macBfrFeedbackType,       // Beamforming report Feedback Type
  input  wire   [2:0] macBfrNr,                 // Beamforming report Nr Index
  input  wire   [2:0] macBfrNc,                 // Beamforming report Nc Index
  input  wire         macBfrHE,                 // Beamforming report Format
                                                // (VHT or HE)
  input  wire   [6:0] macBfrRUStartIndex,       // Beamforming report RU Start Index
                                                // (Only for HE BFR)
  input  wire   [6:0] macBfrRUEndIndex,         // Beamforming report RU End Index 
                                                // (Only for HE BFR)
  ///////////////////////////////////////////////
  //$port_g Tx Controller Interface
  ///////////////////////////////////////////////
  output wire   [7:0] macBfrData,               // Beamforming report byte
  output reg          macBfrDataValid,          // Beamforming report byte valid          
  input  wire         macBfrDataReady,          // Beamforming report byte request
  input  wire         macPhyTxEnd_p,            // End of transmission (macCoreClk)

  ///////////////////////////////////////////////
  //$port_g PHY Interface
  ///////////////////////////////////////////////
  input  wire         txEnd_p,                  // End of transmission (mpifClk)
  
  ///////////////////////////////////////////////
  //$port_g BFR FIFO
  ///////////////////////////////////////////////
  output reg                            useRxFifo4Bfr_mpIFClk,  // Indicate that the RX FIFO is used for BFreport transfer (mpifClk)
  output reg                            useRxFifo4Bfr,          // Indicate that the RX FIFO is used for BFreport transfer (macCoreClk)
  input  wire [ 7:0]                    bfrFIFOReadData,        // data from TP RAM
  //
  output wire [ 7:0]                    bfrFIFOWriteData,       // data to TP RAM
  output wire [ (`MPIFRXADDRWIDTH-1):0] bfrFIFOWriteAddr,       // write address to TP RAM
  output wire                           bfrFIFOWriteEn,         // write enable to TP RAM
  output wire [ (`MPIFRXADDRWIDTH-1):0] bfrFIFOReadAddr,        // read address
  output wire                           bfrFIFOReadEn,          // MAC-PHY Interface Receive FIFO Read Enable.
  ///////////////////////////////////////////////
  //$port_g Debug Port
  ///////////////////////////////////////////////
  output wire                    [15:0] debugPortSVD,           // Debug Port of SVD Interface
  output wire                    [15:0] debugPortSVD2,          // Debug Port of SVD Interface
  output wire                    [15:0] debugPortSVDFifo        // Debug Port of SVD Interface
  );

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
wire        mpifBfrStartMPIFClk;    // BFR computation started (on mpifClk)
reg   [3:0] mpifBfrRequestCnt;      // Indicates number of pending request to the SVD
wire        fifoFull;               // BFR fifo full indication(on mpifClk)
wire        fifoEmpty;              // BFR fifo empty indication (on macCoreClk)
wire  [3:0] fifoWrRemain;           // Indicates remaining bytes to write before
                                    // BFR FIFO Full
wire        macBfrDataReadyMPIFClk; // macBfrDataReady (on mpifClk)


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Syncrhonization from MAC to PHY
simpleSynchro u_macBfrStart_synchro( 
   .dstclk     (mpIFClk             ),
   .dstresetn  (mpIFClkHardRst_n    ),
   .srcdata    (macBfrStart         ),
   .dstdata    (mpifBfrStartMPIFClk )
);

// Syncrhonization from PHY to MAC
pulse2PulseSynchro u_mpifBfrDone_synchro (
   .srcclk     (mpIFClk             ),
   .srcresetn  (mpIFClkHardRst_n    ),
   .dstclk     (macCoreClk          ),
   .dstresetn  (macCoreClkHardRst_n ),
   .srcdata    (mpifBfrDone         ),
   .dstdata    (macBfrDone          )
);


always @(posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
  begin
    mpifBfrChBW              <= 2'b0;
    mpifBfrGrouping          <= 2'b0;
    mpifBfrCodebook          <= 1'b0;
    mpifBfrFeedbackType      <= 2'b0;
    mpifBfrNr                <= 3'b0;
    mpifBfrNc                <= 3'b0;
    mpifBfrHE                <= 1'b0;
    mpifBfrRUStartIndex      <= 7'b0;
    mpifBfrRUEndIndex        <= 7'b0;
  end
  else if (mpifBfrStartMPIFClk && !mpifBfrStart)
  begin
    mpifBfrChBW              <= macBfrChBW;
    mpifBfrGrouping          <= macBfrGrouping;
    mpifBfrCodebook          <= macBfrCodebook;
    mpifBfrFeedbackType      <= macBfrFeedbackType;
    mpifBfrNr                <= macBfrNr;
    mpifBfrNc                <= macBfrNc;
    mpifBfrHE                <= macBfrHE;
    mpifBfrRUStartIndex      <= macBfrRUStartIndex;
    mpifBfrRUEndIndex        <= macBfrRUEndIndex;
  end
end

always @(posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
    mpifBfrStart             <= 1'b0;
  else
    mpifBfrStart             <= mpifBfrStartMPIFClk;
end


// Syncrhonization from MAC to PHY
simpleSynchro u_mmacBfrDataReady_synchro( 
   .dstclk     (mpIFClk                ),
   .dstresetn  (mpIFClkHardRst_n       ),
   .srcdata    (macBfrDataReady        ),
   .dstdata    (macBfrDataReadyMPIFClk )
);


// Indicate that the RX FIFO is used for BFreport transfer (macCoreClk)
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    useRxFifo4Bfr  <= 1'b0;
  else if (macBfrDataReady)
    useRxFifo4Bfr  <= 1'b1;
  else if (macPhyTxEnd_p)
    useRxFifo4Bfr  <= 1'b0;
end


// Indicate that the RX FIFO is used for BFreport transfer (mpifClk)
always @(posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
    useRxFifo4Bfr_mpIFClk  <= 1'b0;
  else if (macBfrDataReadyMPIFClk)
    useRxFifo4Bfr_mpIFClk  <= 1'b1;
  else if (txEnd_p)
    useRxFifo4Bfr_mpIFClk  <= 1'b0;
end


// MAC Core interface
assign bfrFIFOReadEn = macBfrDataReady & ~fifoEmpty;

always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    macBfrDataValid  <= 1'b0;
  else if (macBfrDataReady & ~fifoEmpty)
    macBfrDataValid  <= 1'b1;
  else if (macBfrDataReady &  fifoEmpty)
    macBfrDataValid  <= 1'b0;
end


// FIFO Read data is available until next read (captured in macPhyIf module)
assign macBfrData = bfrFIFOReadData;


// PHY interface
assign bfrFIFOWriteData = mpifBfrData;
assign bfrFIFOWriteEn   = mpifBfrDataValid;

always @(posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (~mpIFClkHardRst_n)
     mpifBfrRequestCnt  <= 4'b0;
  else if (~useRxFifo4Bfr_mpIFClk)
     mpifBfrRequestCnt  <= 4'b0;
  else if (mpifBfrDataReady & ~mpifBfrDataValid)
     mpifBfrRequestCnt  <= mpifBfrRequestCnt+4'h1; // +1
  else if (~mpifBfrDataReady & mpifBfrDataValid)
     mpifBfrRequestCnt  <= mpifBfrRequestCnt+4'hF; // -1
end

assign mpifBfrDataReady = useRxFifo4Bfr_mpIFClk & mpifBfrRequestCnt<fifoWrRemain;


// Fifo Controls
assign bfrFIFOWriteAddr[(`MPIFRXADDRWIDTH-1):3] = 'b0;
assign bfrFIFOReadAddr[(`MPIFRXADDRWIDTH-1):3]  = 'b0;

mpiftxRxFifoControl #(
   .ADDRWIDTH      (3                     ),
   .DEPTH          (8                     )
) U_BFRFifoControl (
   .wrClk          (mpIFClk               ),
   .wrHardReset_n  (mpIFClkHardRst_n      ),
   .rdClk          (macCoreClk            ),
   .rdHardReset_n  (macCoreClkHardRst_n   ),
   .rdFlush        (macPhyTxEnd_p         ),
   .fifoWrite      (bfrFIFOWriteEn        ),
   .fifoRead       (bfrFIFOReadEn         ),
   .fifoWrPtr      (bfrFIFOWriteAddr[2:0] ),
   .fifoRdPtr      (bfrFIFOReadAddr[2:0]  ),
   .fifoFull       (fifoFull              ),
   .fifoAlmostFull (/*Open*/              ),
   .fifoEmpty      (fifoEmpty             ),
   .fifoPtrsNull   (/*Open*/              ),
   .fifoWrRemain   (fifoWrRemain          )
);


// Debug ports
assign debugPortSVD = {
   mpifBfrCodebook,
   mpifBfrFeedbackType[1:0],
   mpifBfrHE,
   mpifBfrData[7:0],
   mpifBfrStart,
   mpifBfrDone,
   mpifBfrDataValid,
   mpifBfrDataReady};

assign debugPortSVD2 = {
   macBfrChBW[1:0],
   macBfrGrouping[1:0],
   macBfrData[7:0],
   macBfrStart,
   macBfrDone,
   macBfrDataValid,
   macBfrDataReady};

assign debugPortSVDFifo = {
   mpifBfrRequestCnt[3:0],
   fifoWrRemain[3:0],
   bfrFIFOWriteAddr[2:0],
   bfrFIFOWriteEn,
   bfrFIFOReadAddr[2:0],
   bfrFIFOReadEn
};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef  RW_ASSERT_ON

//$rw_sva Checks the BFR FIFO is not writed when it's full
property noWriteWhenBFRFifoFullCheck_prop;
@(posedge mpIFClk)
    (fifoFull == 1'b1) |-> bfrFIFOWriteEn == 1'b0;
endproperty
noWriteWhenBFRFifoFullCheck: assert property (noWriteWhenBFRFifoFullCheck_prop);

//$rw_sva Checks the BFR FIFO is not read when it's empty
property noReadWhenBFRFifoEmptyCheck_prop;
@(posedge macCoreClk)
    (fifoEmpty == 1'b1) |-> bfrFIFOReadEn == 1'b0;
endproperty
noReadWhenBFRFifoEmptyCheck: assert property (noReadWhenBFRFifoEmptyCheck_prop);

//$rw_sva Checks if the number of data request is not greather than the
//remaining place in the BFR FIFO
property bfrDataRequestNotGreaterThanBFRFifo_prop;
@(posedge mpIFClk)
    (mpifBfrRequestCnt == fifoWrRemain) |-> mpifBfrDataReady == 1'b0;
endproperty
bfrDataRequestNotGreaterThanBFRFifo: assert property (bfrDataRequestNotGreaterThanBFRFifo_prop);

`endif//RW_ASSERT_ON
endmodule
