//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : pulseSynchro, simpleSynchro, txRxFifoControl, RxStateCtl
//                    TxStateCtl
// Description      : Top level of macPhyIf module
//
// Simulation Notes : Simulated with ncsim
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on
//                   the waveform viewers
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
//    all the inpout/output are synchronous the block clock
//    the are interfaced to. I.E interface with RxController
//    is synchronous with macCoreRxClk, Tx controller
//    macCoreTxClk and so on. The main clock of the block is
//    mpIFClk.
//    All the reysnchroniztions flop are called <name>_resync
// Application Note :
// Simulator        :
//    For simulation with RW_ASSERT_ON, the simulator must support System
//    Verilog Assertions
// Parameters       : 
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`define default_net_type none

module macPhyIf (
         //$port_g clock and Reset
         input  wire                           mpIFClk,                   // PHY clock
         input  wire                           mpIFClkHardRst_n,          // PHY clock reset
         input  wire                           mpIFClkSoftRst_n,          // PHY clock soft reset
         input  wire                           macCoreRxClk,              // mac core Rx clock
         input  wire                           macCoreTxClk,              // mac core Tx clock
         input  wire                           macCoreClk,                // mac core Tx clock
         input  wire                           macCoreClkHardRst_n,       // mac core Tx clock reset
         input  wire                           macCoreClkSoftRst_n,       // mac core Tx clock reset

         //$port_g Phy If
         input  wire                           phyRdy,                    // data valid signal
         input  wire                           txEnd_p,                   // end of transaction
         input  wire [ 7:0]                    rxData,                    // rx data
         input  wire                           CCAPrimary20,              // CCA on Primary 20MHz channel
         input  wire                           CCASecondary20,            // CCA on Secondary 20MHz channel
         input  wire                           CCASecondary40,            // CCA on Secondary 40MHz channel
         input  wire                           CCASecondary80,            // CCA on Secondary 80MHz channel
         input  wire                           rxEndForTiming_p,          // end of transmission (antenna)
         input  wire                           rxErr_p,                   // Rx error
         input  wire                           rxEnd_p,                   // Rx end
         input  wire                           phyErr_p,                  // phy error
         input  wire                           rifsRxDetected,            // rifs detected
         //
         output wire                           txReq,                     // Tx request
         output wire                           rxReq,                     // Rx request
         output wire [ 7:0]                    txData,                    // Tx data
         output wire                           macDataValid,              // data valid
         output wire                           mimoCmdValid,              // mimo command
         output wire                           keepRFOn,                  // Keep RF on

         //$port_g Mac Controller If
         input  wire                           startRx,                   // Start Rx trigger
         input  wire                           stopRx_p,                  // Stop Rx trigger
         input  wire                           mpifKeepRFon,              // Keep RF On
         output wire                           macPhyIfRxStart_p,         // Start of reception pulse
         output wire                           macPhyIfRxFlush_p,         // Rx flush pulse

         //$port_g Tx  Controller If
         input  wire                           startTx_p,                 // Start Tx trigger
         input  wire                           stopTx_p,                  // Stop Tx trigger
         input  wire [ 7:0]                    mpIfTxFifoData,            // Data to transmit
         input  wire                           mpIfTxFifoWrite,           // Data valid

         input  wire [ 7:0]                    txPwrLevel,                // Transmit Power Level
         input  wire [ 2:0]                    chBW,                      // TX Vector Channel Bandwidth
         input  wire                           smoothing,                 // TX Vector Smoothing recommended
         input  wire [ 7:0]                    antennaSet,                // TX Vector Antenna Set
         input  wire [ 7:0]                    smmIndex,                  // TX Vector Spatial Map Matrix Index
         input  wire [ 6:0]                    mcs,                       // TX Vector Modulation Coding Scheme
         input  wire                           preType,                   // TX Vector Preamble Type
         input  wire [ 3:0]                    formatMod,                 // TX Vector Format and Modulation
         input  wire [ 1:0]                    numExtnSS,                 // TX Vector Number of Extension Spatial Streams
         input  wire                           stbc,                      // TX Vector Space Time Block Coding
         input  wire                           fecCoding,                 // TX Vector FEC Coding
         input  wire                           sounding,                  // TX Vector Indicates whether this PPDU is Sounding
         input  wire [11:0]                    legLength,                 // TX Vector Legacy Length of the PPDU
         input  wire [ 3:0]                    legRate,                   // TX Vector Legacy Rate of the PPDU
         input  wire [15:0]                    service,                   // TX Vector Service
         input  wire [19:0]                    htLength,                  // TX Vector Length of the HT/VHT/HE PPDU
         input  wire                           htLengthValid,             // TX Vector Length of the HT/VHT/HE PPDU is valid
         input  wire [ 2:0]                    nTx,                       // TX Vector Number of Transmit Chains.
         input  wire                           aggreation,                // TX Vector MPDU Aggregate
         input  wire                           beamFormed,                // TX Vector BeamFormed
         input  wire [8:0]                     partialAID,                // TX Vector partial AID
         input  wire [5:0]                     groupID,                   // TX Vector group ID
         input  wire                           dozeNotAllowed,            // TX Vector TXOP PS Not Allowed
         input  wire                           continuousTx,              // TX Vector continuous transmit
         input  wire [4:0]                     userPosition,              // Indicates the user
                                                                          // position of the A-MPDU
                                                                          // inside a MUMIMO 
                                                                          // transmission
         input  wire                           timeOfDepartureReq,        // TX Vector Time of departure requested
         input  wire                     [1:0] giType,                    // TX Vector Guard Interval Type
         input  wire                           upLinkFlag,                // TX Vector UP Link Flag
         input  wire                           beamChange,                // TX Vector Beam Change
         input  wire                           dcm,                       // TX Vector Dual Carrier Modulation
         input  wire                     [1:0] heLTFType,                 // TX Vector Type of HE LTF
         input  wire                           doppler,                   // TX Vector Doppler
         input  wire                           midamble,                  // TX Vector Midamble
         input  wire                     [5:0] bssColor,                  // TX Vector BSS Color
         input  wire                     [6:0] txopDuration,              // TX Vector Duration of TXOP
         input  wire                           triggerResponding,         // TX Vector Trigger Responding
         input  wire                     [7:0] nUser,                     // TX Vector Number of User
         input  wire                    [10:0] staID,                     // TX Vector STA ID
         input  wire                     [3:0] spatialReuse1,             // TX Vector Spatial Reuse 1
         input  wire                     [3:0] spatialReuse2,             // TX Vector Spatial Reuse 2
         input  wire                     [3:0] spatialReuse3,             // TX Vector Spatial Reuse 3
         input  wire                     [3:0] spatialReuse4,             // TX Vector Spatial Reuse 4
         input  wire                     [8:0] heSigAReserved,            // TX Vector HE SIG-A Reserved bits
         input  wire                     [2:0] numHeLtf,                  // TX Vector Number of HE LTF
         input  wire                           heLtfMode,                 // TX Vector HE LTF Mode
         input  wire                           ldpcExtraSymbol,           // TX Vector LDPC extra symbol
         input  wire                     [2:0] startingSTSNum,            // TX Vector Starting STS Number
         input  wire                           sigBCompressionMode,       // TX Vector SIG-B compression mode
         input  wire                           dcmSigB,                   // TX Vector Dual carrier modulation on SIG-B
         input  wire                     [2:0] mcsSigB,                   // TX Vector MCS of SIG-B
         input  wire                    [31:0] ruAllocation,              // TX Vector RU allocation
         input  wire                     [2:0] packetExtension,           // TX Vector Packet extension
         input  wire                           triggerMethod,             // TX Vector trigger method
         input  wire                     [6:0] ruToneSetIndex,            // TX Vector RU tone set index
         input  wire                           feedBackStatus,            // TX Vector feedback status
         //
         output wire                           mpIfTxFifoFull,            // FIFO status
         output wire                           mpIfTxFifoAlmostFull,      // FIFO status
         output wire                           mpIfTxErr_p,               // Transmit error
         output reg                            mpIfTxEn,                  // Transmit Enable
         output wire                    [31:0] timeOfDeparture,           // TX Vector2 Time Of Departure
         output wire                    [15:0] timeOfDepartureClkRate,    // TX Vector2 Time Of Departure Clock Rate

         //$port_g Encryption Engine
         output wire                           macPhyTxEnd_p,             // Pulse tx end resync on MacCoreCLk

         //$port_g Rx Controller
         input  wire                           rxDSSS,                    // Ongoing reception is DSSS
         input  wire                           macPhyIfRxFifoReadEn,      // FIFO read enable from the Rx controller
         //
         output wire                           macPhyIfRxFifoEmpty,       // FIFO empty signal
         output wire [ 7:0]                    macPhyIfRxFifoData,        // FIFO data
         output wire                           macPhyIfRxErr_p,           // abort pulse
         output wire                           macPhyIfRxEnd_p,           // end of reception pulse
         output wire                           macPhyIfRxEndForTiming_p,  // end of reception at the antenna
         output wire                           macPhyIfRifsRxDetected,    // RIFS detected indication
         output wire                           macPhyIfRxCca,             // CCA trigger
         output wire                           macPhyIfRxCcaSec20,        // CCA trigger Lower
         output wire                           macPhyIfRxCcaSec40,        // CCA trigger Lower
         output wire                           macPhyIfRxCcaSec80,        // CCA trigger Lower

         //$port_g Registers
         input  wire                           band5G,                    // Indicate the current operating band (0: 2.4GHz, 1:5GHz)
         input  wire                           macPHYIFFIFOReset,         // Flush the FIFO
         output wire                           macPHYIFFIFOResetInValid,  // Clear of FIFO Flush
         input  wire                           rateControllerMPIF,        // Select transmit mechanism
                                                                          // 0 phy request (i.e PHY fetches at anyrate)
                                                                          // 1 phy strobe (i.e PHY fetches at air datarate)
         input  wire                           rxReqForceDeassertion,     // Force deassertion of rxReq after each rxEnd_p
         input  wire                           rxEndForTimingErrRec,      // Enable rxEndForTiming error recovery mechanism
         input  wire [ 7:0]                    macCoreClkFreq,            // MAC Core Clock Frequency

         //$port_g Interrupt controller
         output reg                            macPHYIFUnderRun,          // underrun status
         output reg                            macPHYIFOverflow,          // overflow status

       `ifdef RW_BFMEE_EN
         //$port_g BFR HW Accelerator Interface
         output wire                           mpifBfrStart,              // BFR computation started
         input  wire                           mpifBfrDone,               // Beamforming Report computation is done
         output wire   [1:0]                   mpifBfrChBW,               // Beamforming report Channel Width
         output wire   [1:0]                   mpifBfrGrouping,           // Beamforming report Grouping
         output wire                           mpifBfrCodebook,           // Beamforming report Codebook Information
         output wire   [1:0]                   mpifBfrFeedbackType,       // Beamforming report Feedback Type
         output wire   [2:0]                   mpifBfrNr,                 // Beamforming report Nr Index
         output wire   [2:0]                   mpifBfrNc,                 // Beamforming report Nc Index
         output wire                           mpifBfrHE,                 // Beamforming report Format
                                                                          // (VHT or HE)
         output wire   [6:0]                   mpifBfrRUStartIndex,       // Beamforming report RU Start Index
                                                                          // (Only for HE BFR)
         output wire   [6:0]                   mpifBfrRUEndIndex,         // Beamforming report RU End Index 
                                                                          // (Only for HE BFR)
         input  wire   [7:0]                   mpifBfrData,               // Beamforming report byte
         input  wire                           mpifBfrDataValid,          // Beamforming report byte valid          
         output wire                           mpifBfrDataReady,          // Beamforming report byte request
         //$port_g BFR Controller Interface
         input  wire                           macBfrStart,               // BFR computation started
         output wire                           macBfrDone,                // Beamforming Report computation is done
         input  wire   [1:0]                   macBfrChBW,                // Beamforming report Channel Width
         input  wire   [1:0]                   macBfrGrouping,            // Beamforming report Grouping
         input  wire                           macBfrCodebook,            // Beamforming report Codebook Information
         input  wire   [1:0]                   macBfrFeedbackType,        // Beamforming report Feedback Type
         input  wire   [2:0]                   macBfrNr,                  // Beamforming report Nr Index
         input  wire   [2:0]                   macBfrNc,                  // Beamforming report Nc Index
         input  wire                           macBfrHE,                  // Beamforming report Format
                                                                          // (VHT or HE)
         input  wire   [6:0]                   macBfrRUStartIndex,        // Beamforming report RU Start Index
                                                                          // (Only for HE BFR)
         input  wire   [6:0]                   macBfrRUEndIndex,          // Beamforming report RU End Index 
                                                                          // (Only for HE BFR)
         output wire   [7:0]                   macBfrData,                // Beamforming report byte
         output wire                           macBfrDataValid,           // Beamforming report byte valid          
         input  wire                           macBfrDataReady,           // Beamforming report byte request
       `endif // RW_BFMEE_EN

         //$port_g Tx FIFO
         input  wire [ 7:0]                    mpIFTxFIFOReadData,        // data from TP RAM
         //
         output wire [ 7:0]                    mpIFTxFIFOWriteData,       // data to TP RAM
         output wire [ (`MPIFTXADDRWIDTH-1):0] mpIFTxFIFOWriteAddr,       // write address to TP RAM
         output wire                           mpIFTxFIFOWriteEn,         // write enable to TP RAM
         output wire [ (`MPIFTXADDRWIDTH-1):0] mpIFTxFIFOReadAddr,        // read address
         output wire                           mpIFTxFIFOReadEn,          // MAC-PHY Interface Transmit FIFO Read Enable.

         //$port_g Rx FIFO
         input  wire [ 7:0]                    mpIFRxFIFOReadData,        // data from TP RAM
         //
         output wire [ 7:0]                    mpIFRxFIFOWriteData,       // data to TP RAM
         output wire [ (`MPIFRXADDRWIDTH-1):0] mpIFRxFIFOWriteAddr,       // write address to TP RAM
         output wire                           mpIFRxFIFOWriteEn,         // write enable to TP RAM
         output wire [ (`MPIFRXADDRWIDTH-1):0] mpIFRxFIFOReadAddr,        // read address
         output wire                           mpIFRxFIFOReadEn,          // MAC-PHY Interface Receive FIFO Read Enable.

         //$port_g Debug ports
         output wire [15:0]                    debugPortMACPhyIf,         // Debug port for validation
         output wire [15:0]                    debugPortMACPhyIf2,        // Debug port for validation
         output wire [15:0]                    debugPortMACPhyIf3         // Debug port for validation
`ifdef  RW_BFMEE_EN
         ,
         output wire [15:0]                    debugPortSVD,              // Debug Port of SVD Interface
         output wire [15:0]                    debugPortSVD2,             // Debug Port of SVD Interface
         output wire [15:0]                    debugPortSVDFifo        // Debug Port of SVD Interface
`endif//RW_BFMEE_EN

);

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// Rx controller
wire       mpifOverflow;           // overflow
wire       startRx_mpIFClk;        // resynchornized start Rx
wire       stopRx_mpIFClk_p;       // resynchornized stop Rx
//wire       rxFifoWrClkFlush;       // macCore clock Tx fifo flush
wire       rxFifoRdClkFlush;       // mpif    clock Tx fifo flush

// Tx controller
wire       mpifUnderRun;           // underrun
//wire       startTx_mpIFClk_p;      // resynchornized start Tx
wire       stopTx_mpIFClk_p;       // resynchornized stop Tx
wire       txFifoRdClkFlush;       // Flush txFifo, mpficlk synchronized
//wire       txFifoWrClkFlush;       // Flush txFifo, macCoreClk synchronized

// Rx fifo
wire       fifoWriteEn;            // Rx fifo Write Enable
wire       mpIfRxFifoFull;         // Rx fifo full
wire       fifoRxPtrsNull;         // active when both write and read RX pointers are null

// Tx fifo
wire       mpIfTxFifoEmtpy;        // Tx fifo empty
wire       mpIfFifoReadEn;         // fifo read enable
wire       fifoTxPtrsNull;         // active when both write and read TX pointers are null
reg        fifoTxPtrsNull_ff1;     // FF for fifoTxPtrsNull before resync
wire       fifoTxPtrsNullSync;     // fifoTxPtrsNull resync on mac core clock

// Mac controller
wire       mackeepRFOn;            // Keep RF on from mac
wire       RxkeepRFOn;             // Keep RF on from Rx
wire       TxkeepRFOn;             // Keep RF on from Tx

wire       forceCCAPrimary;        // Force the CCA Primary high between the first byte
                                   // of the rxVector1 and the rxEndForTiming_p
reg        CCAPrimary20Int;        // This is the Logical Or between CCAPrimary20 and forceCCAPrimary
reg        CCASecondary20Int;      // Capture CCASecondary20 input
reg        CCASecondary40Int;      // Capture CCASecondary40 input
reg        CCASecondary80Int;      // Capture CCASecondary80 input
wire       rxReqForceDeassertion_mpIFClk; // rxReqForceDeassertion resynchronized on mpIFClk
wire       rxEndForTimingErrRec_mpIFClk;  // rxEndForTimingErrRec resynchronized on mpIFClk
reg        rifsRxDetectedInt;      // Capture rifsRxDetected input

wire       rateControllerMPIFResync;// rateControllerMPIF resynchronized on mpIFClk.

wire       macPHYIFUnderRunInt;     // underrun status
wire       macPHYIFOverflowInt;     // overflow status

reg       phyRdyCapt;
reg       rxEndCapt_p;
reg [7:0] rxDataCapt;

wire macPhyIfRxEndForTimingWoSigExt_p;
wire macPhyIfRxEndWoSigExt_p;
wire macPhyIfRxErrWoSigExt_p;
wire macPhyTxEndWoSigExt_p;

wire macPhyIfRxCcaInt;  // CCAPrimary20 resynchronized on macCoreClk.
reg  macPhyIfRxCcaDis;  // Force  macPhyIfRxCca low when rxStop is requested.



wire [ (`MPIFRXADDRWIDTH-1):0] mpIFRxFIFOWriteAddrInt;       // write address to TP RAM
wire [ (`MPIFRXADDRWIDTH-1):0] mpIFRxFIFOReadAddrInt;       // write address to TP RAM

wire       mpIfTxEn_mpIFClk;         // mpIfTxEn resynchronized on mpIFClk

// Trigger pulse
wire       macPhyStartTx_p;          // TX Trigger pulse
wire       macPhyTxReady;            // Indicate that MAC Phy transmission is ready
reg        macPhyTxReadyReg;         // Registred version of macPhyTxReady

wire       htLengthValid_mpIFClk;    // htLengthValid resynchronized on mpIFClk


`ifdef RW_BFMEE_EN
wire [ 7:0]                    bfrFIFOReadData;        // data from TP RAM
wire [ 7:0]                    bfrFIFOWriteData;       // data to TP RAM
wire [ (`MPIFRXADDRWIDTH-1):0] bfrFIFOWriteAddr;       // write address to TP RAM
wire                           bfrFIFOWriteEn;         // write enable to TP RAM
wire [ (`MPIFRXADDRWIDTH-1):0] bfrFIFOReadAddr;        // read address
wire                           bfrFIFOReadEn;          // MAC-PHY Interface Receive FIFO Read Enable.

wire                           useRxFifo4Bfr_mpIFClk;
wire                           useRxFifo4Bfr;
`endif // RW_BFMEE_EN
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign mpIFTxFIFOWriteEn         = mpIfTxFifoWrite;
assign mpIFTxFIFOReadEn          = mpIfFifoReadEn;
assign mpIFTxFIFOWriteData       = mpIfTxFifoData;


// output connections
///////////////////////////////////////////////////////
always @(posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
   if(macCoreClkHardRst_n == 1'b0)
      mpIfTxEn <= 1'b0;
   else if((macCoreClkSoftRst_n == 1'b0) || (macPhyTxEnd_p == 1'b1))
      mpIfTxEn <= 1'b0;
   else if(startTx_p == 1'b1)
      mpIfTxEn <= 1'b1;
end

always @(posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
   if(mpIFClkHardRst_n == 1'b0)
   begin
     rxDataCapt       <= 8'd0;
     phyRdyCapt       <= 1'b0;
     rxEndCapt_p      <= 1'b0;
   end
   else
   begin
     rxDataCapt       <= rxData;
     phyRdyCapt       <= phyRdy;
     rxEndCapt_p      <= rxEnd_p;
   end
end

// MPIF RX FIFO RAM connections
///////////////////////////////////////////////////////

`ifdef RW_BFMEE_EN
assign mpIFRxFIFOWriteEn         = (useRxFifo4Bfr_mpIFClk) ? bfrFIFOWriteEn   : fifoWriteEn;
assign mpIFRxFIFOWriteAddr       = (useRxFifo4Bfr_mpIFClk) ? bfrFIFOWriteAddr : mpIFRxFIFOWriteAddrInt;
assign mpIFRxFIFOWriteData       = (useRxFifo4Bfr_mpIFClk) ? bfrFIFOWriteData : rxDataCapt;
`else
assign mpIFRxFIFOWriteEn         = fifoWriteEn;
assign mpIFRxFIFOWriteAddr       = mpIFRxFIFOWriteAddrInt;
assign mpIFRxFIFOWriteData       = rxDataCapt;
`endif // RW_BFMEE_EN

reg mpIFRxFIFOReadEnFF1;
reg [7:0] mpIFRxFIFOReadDataCapt;
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if(macCoreClkHardRst_n == 1'b0)
   begin
     mpIFRxFIFOReadDataCapt   <= 8'd0;
     mpIFRxFIFOReadEnFF1      <= 1'b0;
   end
   else
   begin
     mpIFRxFIFOReadEnFF1      <= mpIFRxFIFOReadEn;
     if (mpIFRxFIFOReadEnFF1)
       mpIFRxFIFOReadDataCapt <= mpIFRxFIFOReadData;
   end
end


`ifdef RW_BFMEE_EN
assign mpIFRxFIFOReadAddr        = (useRxFifo4Bfr) ? bfrFIFOReadAddr : mpIFRxFIFOReadAddrInt; 
assign mpIFRxFIFOReadEn          = (useRxFifo4Bfr) ? bfrFIFOReadEn   : macPhyIfRxFifoReadEn;
`else // RW_BFMEE_EN
assign mpIFRxFIFOReadAddr        = mpIFRxFIFOReadAddrInt; 
assign mpIFRxFIFOReadEn          = macPhyIfRxFifoReadEn;
`endif // RW_BFMEE_EN


assign macPhyIfRxFifoData        = (mpIFRxFIFOReadEnFF1) ? mpIFRxFIFOReadData : mpIFRxFIFOReadDataCapt;


`ifdef RW_BFMEE_EN
assign bfrFIFOReadData           = (mpIFRxFIFOReadEnFF1) ? mpIFRxFIFOReadData : mpIFRxFIFOReadDataCapt;
`endif // RW_BFMEE_EN

assign macPhyIfRxFlush_p         = rxFifoRdClkFlush;

assign macPHYIFFIFOResetInValid  = fifoTxPtrsNullSync && fifoRxPtrsNull && macPHYIFFIFOReset;

assign mimoCmdValid              = 1'b0;         // to be implemented

assign keepRFOn                  = RxkeepRFOn || TxkeepRFOn;


// FF the fifoTxPtrsNull before resync on MAC Core Clk domain
// avoiding glitches
always @(posedge mpIFClk, negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
    fifoTxPtrsNull_ff1 <= 1'b0;
  else
    fifoTxPtrsNull_ff1 <= fifoTxPtrsNull;
end


// Tx Trigger Pulse
///////////////////////////////////////////////////////
assign macPhyTxReady = mpIfTxEn_mpIFClk;

// Register macPhyTxReady to detect posedge
always @(posedge mpIFClk or negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
    macPhyTxReadyReg <= 1'b0;
  else
  begin
    macPhyTxReadyReg <= macPhyTxReady;
  end
end

assign macPhyStartTx_p = (macPhyTxReady & !macPhyTxReadyReg);


// resynchronization cells
///////////////////////////////////////////////////////
pulse2PulseSynchro U_stoptRx_p_synchro (
                .srcclk (macCoreRxClk),
                .srcresetn (macCoreClkHardRst_n),
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (stopRx_p),
                .dstdata (stopRx_mpIFClk_p)
                );

pulse2PulseSynchro U_stoptTx_p_synchro (
                .srcclk (macCoreTxClk),
                .srcresetn (macCoreClkHardRst_n),
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (stopTx_p),
                .dstdata (stopTx_mpIFClk_p)
                );

pulse2PulseSynchro U_mpIfUnderRun_synchro (
                .srcclk (mpIFClk),
                .srcresetn (mpIFClkHardRst_n),
                .dstclk (macCoreTxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (mpifUnderRun),
                .dstdata (macPHYIFUnderRunInt)
                );

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)  
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    macPHYIFUnderRun <= 1'b0;
  else
    macPHYIFUnderRun <= macPHYIFUnderRunInt;
end

pulse2PulseSynchro U_mpifOverflow_synchro (
                .srcclk (mpIFClk),
                .srcresetn (mpIFClkHardRst_n),
                .dstclk (macCoreRxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (mpifOverflow),
                .dstdata (macPHYIFOverflowInt)
                );

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    macPHYIFOverflow <= 1'b0;
  else
    macPHYIFOverflow <= macPHYIFOverflowInt;
end

pulse2PulseSynchro U_macPhyTxEnd_synchro (
                .srcclk (mpIFClk),
                .srcresetn (mpIFClkHardRst_n),
                .dstclk (macCoreTxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (txEnd_p),
                .dstdata (macPhyTxEndWoSigExt_p)
                );

simpleSynchro U_fifoTxPtrsNull_synchro (
                .dstclk (macCoreTxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (fifoTxPtrsNull_ff1),
                .dstdata (fifoTxPtrsNullSync)
                );

always @(posedge mpIFClk, negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
  begin
    CCAPrimary20Int   <= 1'b0;
    CCASecondary20Int <= 1'b0;
    CCASecondary40Int <= 1'b0;
    CCASecondary80Int <= 1'b0;
  end
  else
  begin
    CCAPrimary20Int   <= CCAPrimary20 | forceCCAPrimary;
    CCASecondary20Int <= CCASecondary20;
    CCASecondary40Int <= CCASecondary40;
    CCASecondary80Int <= CCASecondary80;
  end
end

simpleSynchro U_mpIfCCAPrim20_synchro (
                .dstclk (macCoreRxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (CCAPrimary20Int),
                .dstdata (macPhyIfRxCcaInt)
                );
   
// Disable the CCA when stopRx is requested to avoid a stopRx and macPhyIfRxCca high simulatnuously. 
// This may happen due to the latency introduced by the clock domain crossing between macCoreClk and mpIfClk. 
// The deassertion of rxReq and as a consequence the deassertion of CCAPrimary20 takes several clocks cycle 
// after the stopRx_p request. During this time, the CCA may goes high.  
always @(posedge macCoreRxClk, negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    macPhyIfRxCcaDis   <= 1'b1;
  else if (stopRx_p)
    macPhyIfRxCcaDis   <= 1'b1;
  else if (startRx)
    macPhyIfRxCcaDis   <= 1'b0;
end

assign macPhyIfRxCca = macPhyIfRxCcaInt & !macPhyIfRxCcaDis;


simpleSynchro U_mpIfCCASec20_synchro (
                .dstclk (macCoreRxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (CCASecondary20Int),
                .dstdata (macPhyIfRxCcaSec20)
                );

simpleSynchro U_mpIfCCASec40_synchro (
                .dstclk (macCoreRxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (CCASecondary40Int),
                .dstdata (macPhyIfRxCcaSec40)
                );

simpleSynchro U_mpIfCCASec80_synchro (
                .dstclk (macCoreRxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (CCASecondary80Int),
                .dstdata (macPhyIfRxCcaSec80)
                );

simpleSynchro U_keepRF_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (mpifKeepRFon),
                .dstdata (mackeepRFOn)
                );

always @(posedge mpIFClk, negedge mpIFClkHardRst_n)
begin
  if (mpIFClkHardRst_n == 1'b0)
    rifsRxDetectedInt   <= 1'b0;
  else
    rifsRxDetectedInt   <= rifsRxDetected;
end


simpleSynchro U_rifsDetect_synchro (
                .dstclk (macCoreRxClk),
                .dstresetn (macCoreClkHardRst_n),
                .srcdata (rifsRxDetectedInt),
                .dstdata (macPhyIfRifsRxDetected)
                );

simpleSynchro U_startRx_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (startRx),
                .dstdata (startRx_mpIFClk)
                );

simpleSynchro U_rxReqForceDeassertion_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (rxReqForceDeassertion),
                .dstdata (rxReqForceDeassertion_mpIFClk)
                );

simpleSynchro U_rxEndForTimingErrRec_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (rxEndForTimingErrRec),
                .dstdata (rxEndForTimingErrRec_mpIFClk)
                );

simpleSynchro U_rateControllerMPIF_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (rateControllerMPIF),
                .dstdata (rateControllerMPIFResync)
                );

simpleSynchro U_mpifTxEn_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (mpIfTxEn),
                .dstdata (mpIfTxEn_mpIFClk)
                );

simpleSynchro U_htLengthValid_synchro (
                .dstclk (mpIFClk),
                .dstresetn (mpIFClkHardRst_n),
                .srcdata (htLengthValid),
                .dstdata (htLengthValid_mpIFClk)
                );


///////////////////////////////////////////////////////
// MAC PHY IF RX FIFO
///////////////////////////////////////////////////////
mpiftxRxFifoControl #(.ADDRWIDTH(`MPIFRXADDRWIDTH)) U_RxFifoControl (
                .wrClk                     (mpIFClk),
                .wrHardReset_n             (mpIFClkHardRst_n),
                .rdClk                     (macCoreRxClk),
                .rdHardReset_n             (macCoreClkHardRst_n),
                .rdFlush                   (rxFifoRdClkFlush),
                .fifoWrite                 (fifoWriteEn),
                .fifoRead                  (macPhyIfRxFifoReadEn),
                .fifoWrPtr                 (mpIFRxFIFOWriteAddrInt),
                .fifoRdPtr                 (mpIFRxFIFOReadAddrInt),
                .fifoFull                  (mpIfRxFifoFull),
                .fifoAlmostFull            (), // open
                .fifoEmpty                 (macPhyIfRxFifoEmpty),
                .fifoPtrsNull              (fifoRxPtrsNull),
                .fifoWrRemain              (/*Open*/)
                );

// Instanciation of RxStateCtl
// Name of the instance : u_RxStateCtl
// Name of the file containing this module : RxStateCtl.v
RxStateCtl U_RxStateCtl (
   .mpIFClk                          (mpIFClk),
   .mpIFClkHardRst_n                 (mpIFClkHardRst_n),
   .mpIFClkSoftRst_n                 (mpIFClkSoftRst_n),
   .macCoreRxClk                     (macCoreRxClk),
   .macCoreClkHardRst_n              (macCoreClkHardRst_n),
   .startRx                          (startRx_mpIFClk),
   .stopRx_p                         (stopRx_mpIFClk_p),
   .rxData                           (rxDataCapt),
   .rxErr_p                          (rxErr_p),
   .rxEnd_p                          (rxEndCapt_p),
   .phyRdy                           (phyRdyCapt),
   .rxEndForTiming_p                 (rxEndForTiming_p),
   .rxReq                            (rxReq),
   .mpIfRxFifoFull                   (mpIfRxFifoFull),
   .fifoWriteEn                      (fifoWriteEn),
   .rxFifoRdClkFlush                 (rxFifoRdClkFlush),
   .macPhyIfRxErr_p                  (macPhyIfRxErrWoSigExt_p),
   .macPhyIfRxEnd_p                  (macPhyIfRxEndWoSigExt_p),
   .macPhyIfRxEndForTiming_p         (macPhyIfRxEndForTimingWoSigExt_p),
   .forceCCAPrimary                  (forceCCAPrimary),
   .macPhyIfRxStart_p                (macPhyIfRxStart_p),
   .macPHYIFFIFOReset                (macPHYIFFIFOReset),
   .rxReqForceDeassertion            (rxReqForceDeassertion_mpIFClk),
   .rxEndForTimingErrRec             (rxEndForTimingErrRec_mpIFClk),
   .mpifOverflow                     (mpifOverflow),
   .mackeepRFOn                      (mackeepRFOn),
   .RxkeepRFOn                       (RxkeepRFOn),
   .debugPortMACPhyIf2               (debugPortMACPhyIf2)
);



// Instanciation of signalExtAdder
// Name of the instance : u_signalExtAdder
// Name of the file containing this module : signalExtAdder.v
signalExtAdder u_signalExtAdder (
   .macCoreTxClk                               (macCoreTxClk),
   .macCoreRxClk                               (macCoreRxClk),
   .macCoreClkHardRst_n                        (macCoreClkHardRst_n),
   .macCoreClkSoftRst_n                        (macCoreClkSoftRst_n),
   .rxDSSS                                     (rxDSSS),
   .band5G                                     (band5G),
   .formatMod                                  (formatMod),
   .legRate                                    (legRate),
   .stopTx_p                                   (stopTx_p),
   .macPhyIfRxEndForTimingWoSigExt_p           (macPhyIfRxEndForTimingWoSigExt_p),
   .macPhyIfRxEndForTiming_p                   (macPhyIfRxEndForTiming_p),
   .macPhyIfRxEndWoSigExt_p                    (macPhyIfRxEndWoSigExt_p),
   .macPhyIfRxEnd_p                            (macPhyIfRxEnd_p),
   .macPhyIfRxErrWoSigExt_p                    (macPhyIfRxErrWoSigExt_p),
   .macPhyIfRxErr_p                            (macPhyIfRxErr_p),
   .macPhyTxEndWoSigExt_p                      (macPhyTxEndWoSigExt_p),
   .macPhyTxEnd_p                              (macPhyTxEnd_p),
   .mpIfTxErr_p                                (mpIfTxErr_p),
   .macCoreClkFreq                             (macCoreClkFreq)
);


///////////////////////////////////////////////////////
// MAC PHY IF TX FIFO
///////////////////////////////////////////////////////
// assign mpIFTxFIFOWriteAddr[(`MPIFTXADDRWIDTH-1):5] = 'b0;
// assign mpIFTxFIFOReadAddr[(`MPIFTXADDRWIDTH-1):5] = 'b0;
// 
// mpiftxRxFifoControl #(.ADDRWIDTH(5)) U_txFifoControl (
//                 .wrClk                     (macCoreTxClk),
//                 .wrHardReset_n             (macCoreClkHardRst_n),
//                 .rdClk                     (mpIFClk),
//                 .rdHardReset_n             (mpIFClkHardRst_n),
//                 .rdFlush                   (txFifoRdClkFlush),
//                 .fifoWrite                 (mpIfTxFifoWrite),
//                 .fifoRead                  (mpIfFifoReadEn),
//                 .fifoWrPtr                 (mpIFTxFIFOWriteAddr[4:0]),
//                 .fifoRdPtr                 (mpIFTxFIFOReadAddr[4:0]),
//                 .fifoFull                  (mpIfTxFifoFull),
//                 .fifoAlmostFull            (mpIfTxFifoAlmostFull),
//                 .fifoEmpty                 (mpIfTxFifoEmtpy),
//                 .fifoPtrsNull              (fifoTxPtrsNull)
//                 );

mpiftxRxFifoControl #(.ADDRWIDTH(`MPIFTXADDRWIDTH)) U_txFifoControl (
                .wrClk                     (macCoreTxClk),
                .wrHardReset_n             (macCoreClkHardRst_n),
                .rdClk                     (mpIFClk),
                .rdHardReset_n             (mpIFClkHardRst_n),
                .rdFlush                   (txFifoRdClkFlush),
                .fifoWrite                 (mpIfTxFifoWrite),
                .fifoRead                  (mpIfFifoReadEn),
                .fifoWrPtr                 (mpIFTxFIFOWriteAddr),
                .fifoRdPtr                 (mpIFTxFIFOReadAddr),
                .fifoFull                  (mpIfTxFifoFull),
                .fifoAlmostFull            (mpIfTxFifoAlmostFull),
                .fifoEmpty                 (mpIfTxFifoEmtpy),
                .fifoPtrsNull              (fifoTxPtrsNull),
                .fifoWrRemain              (/*Open*/)
                );


TxStateCtl U_TxStateCtl (
                .mpIFClk                   (mpIFClk),
                .mpIFClkHardRst_n          (mpIFClkHardRst_n),
                .mpIFClkSoftRst_n          (mpIFClkSoftRst_n),
                .macCoreTxClk              (macCoreTxClk),
                .macCoreClkHardRst_n       (macCoreClkHardRst_n),
                .startTx_p                 (macPhyStartTx_p),
                .stopTx_p                  (stopTx_mpIFClk_p),
                .mpIfTxErr_p               (mpIfTxErr_p),
                .txPwrLevel                (txPwrLevel),
                .chBW                      (chBW),
                .smoothing                 (smoothing),
                .antennaSet                (antennaSet),
                .smmIndex                  (smmIndex),
                .mcs                       (mcs),
                .preType                   (preType),
                .formatMod                 (formatMod),
                .numExtnSS                 (numExtnSS),
                .stbc                      (stbc),
                .fecCoding                 (fecCoding),
                .sounding                  (sounding),
                .legLength                 (legLength),
                .legRate                   (legRate),
                .service                   (service),
                .htLength                  (htLength),
                .htLengthValid             (htLengthValid_mpIFClk),
                .nTx                       (nTx),
                .aggreation                (aggreation),
                .beamFormed                (beamFormed),
                .partialAID                (partialAID),
                .groupID                   (groupID),
                .dozeNotAllowed            (dozeNotAllowed),
                .continuousTx              (continuousTx),
                .userPosition              (userPosition),
                .timeOfDepartureReq        (timeOfDepartureReq),
                .giType                    (giType),
                .upLinkFlag                (upLinkFlag),
                .beamChange                (beamChange),
                .dcm                       (dcm),
                .heLTFType                 (heLTFType),
                .doppler                   (doppler),
                .midamble                  (midamble),
                .bssColor                  (bssColor),
                .txopDuration              (txopDuration),
                .triggerResponding         (triggerResponding),
                .nUser                     (nUser),
                .staID                     (staID),
                .spatialReuse1             (spatialReuse1),
                .spatialReuse2             (spatialReuse2),
                .spatialReuse3             (spatialReuse3),
                .spatialReuse4             (spatialReuse4),
                .heSigAReserved            (heSigAReserved),
                .numHeLtf                  (numHeLtf),
                .heLtfMode                 (heLtfMode),
                .ldpcExtraSymbol           (ldpcExtraSymbol),
                .startingSTSNum            (startingSTSNum),
                .sigBCompressionMode       (sigBCompressionMode),
                .dcmSigB                   (dcmSigB),
                .mcsSigB                   (mcsSigB),
                .ruAllocation              (ruAllocation),
                .packetExtension           (packetExtension),
                .triggerMethod             (triggerMethod),
                .ruToneSetIndex            (ruToneSetIndex),
                .feedBackStatus            (feedBackStatus),
                .timeOfDeparture           (timeOfDeparture),
                .timeOfDepartureClkRate    (timeOfDepartureClkRate),
                .phyRdy                    (phyRdy),
                .txEnd_p                   (txEnd_p),
                .phyErr_p                  (phyErr_p),
                .txReq                     (txReq),
                .txData                    (txData),
                .rxData                    (rxData),
                .macDataValid              (macDataValid),
                .mpIfTxFifoDataout         (mpIFTxFIFOReadData),
                .mpIfTxFifoEmtpy           (mpIfTxFifoEmtpy),
                .mpIfFifoReadEn            (mpIfFifoReadEn),
                .txFifoRdClkFlush          (txFifoRdClkFlush),
                .macPHYIFFIFOReset         (macPHYIFFIFOReset),
                .rateControllerMPIF        (rateControllerMPIFResync),
                .mackeepRFOn               (mackeepRFOn),
                .TxkeepRFOn                (TxkeepRFOn),
                .debugPortMACPhyIf3        (debugPortMACPhyIf3),
                .mpifUnderRun              (mpifUnderRun)
                );



`ifdef RW_BFMEE_EN
////////////////////////////////////////////////////
/// BFRPort Resync
////////////////////////////////////////////////////

// Instanciation of macCoreBFRResync
// Name of the instance : u_macCoreBFRResync
// Name of the file containing this module : macCoreBFRResync.v
macCoreBFRResync u_macCoreBFRResync (
   .mpIFClkHardRst_n       (mpIFClkHardRst_n),
   .macCoreClkHardRst_n    (macCoreClkHardRst_n),
   .macCoreClkSoftRst_n    (macCoreClkSoftRst_n),
   .mpIFClk                (mpIFClk),
   .macCoreClk             (macCoreClk),
   .mpifBfrStart           (mpifBfrStart),
   .mpifBfrDone            (mpifBfrDone),
   .mpifBfrChBW            (mpifBfrChBW),
   .mpifBfrGrouping        (mpifBfrGrouping),
   .mpifBfrCodebook        (mpifBfrCodebook),
   .mpifBfrFeedbackType    (mpifBfrFeedbackType),
   .mpifBfrNr              (mpifBfrNr),
   .mpifBfrNc              (mpifBfrNc),
   .mpifBfrHE              (mpifBfrHE),
   .mpifBfrRUStartIndex    (mpifBfrRUStartIndex),
   .mpifBfrRUEndIndex      (mpifBfrRUEndIndex),
   .mpifBfrData            (mpifBfrData),
   .mpifBfrDataValid       (mpifBfrDataValid),
   .mpifBfrDataReady       (mpifBfrDataReady),
   .macBfrStart            (macBfrStart),
   .macBfrDone             (macBfrDone),
   .macBfrChBW             (macBfrChBW),
   .macBfrGrouping         (macBfrGrouping),
   .macBfrCodebook         (macBfrCodebook),
   .macBfrFeedbackType     (macBfrFeedbackType),
   .macBfrNr               (macBfrNr),
   .macBfrNc               (macBfrNc),
   .macBfrHE               (macBfrHE),
   .macBfrRUStartIndex     (macBfrRUStartIndex),
   .macBfrRUEndIndex       (macBfrRUEndIndex),
   .macBfrData             (macBfrData),
   .macBfrDataValid        (macBfrDataValid),
   .macBfrDataReady        (macBfrDataReady),
   .macPhyTxEnd_p          (macPhyTxEnd_p),
   .txEnd_p                (txEnd_p),
   .useRxFifo4Bfr_mpIFClk  (useRxFifo4Bfr_mpIFClk),
   .useRxFifo4Bfr          (useRxFifo4Bfr),
   .bfrFIFOReadData        (bfrFIFOReadData),   
   .bfrFIFOWriteData       (bfrFIFOWriteData),  
   .bfrFIFOWriteAddr       (bfrFIFOWriteAddr),  
   .bfrFIFOWriteEn         (bfrFIFOWriteEn),    
   .bfrFIFOReadAddr        (bfrFIFOReadAddr),   
   .bfrFIFOReadEn          (bfrFIFOReadEn),
   .debugPortSVD           (debugPortSVD),
   .debugPortSVD2          (debugPortSVD2),
   .debugPortSVDFifo       (debugPortSVDFifo)
);

`endif // RW_BFMEE_EN


///////////////////////////////////////////////////////
// Debug port
///////////////////////////////////////////////////////
assign debugPortMACPhyIf = {macPhyIfRxFifoReadEn,
                            macPhyIfRxFifoEmpty,
                            macPhyIfRxErr_p,
                            macPhyIfRxEnd_p,
                            macPhyIfRxEndForTiming_p,
                            macPhyIfRxCcaSec20,
                            macPhyIfRxCca,
                            macPhyIfRxStart_p,
                            stopRx_p,
                            startRx,
                            aggreation,
                            mpIfTxEn,
                            mpIfTxErr_p,
                            mpIfTxFifoFull,
                            stopTx_p,
                            startTx_p
                            };


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_ASSERT_ON

///////////////////////////////////////////////////////
// PHY INTERFACE PROPERTIES
///////////////////////////////////////////////////////

//$rw_sva Check for rxReq not asserted during transmisson
property txReqAndNotRrxReq_prop;
@(posedge macCoreTxClk)
  disable iff(macCoreClkHardRst_n==0)
  txReq|->!rxReq;
endproperty
txReqAndNotRrxReq: assert property (txReqAndNotRrxReq_prop);

//$rw_sva Checks for txReq not asserted during reception
property rxReqAndNottxReq_prop;
@(posedge macCoreRxClk)
  disable iff(macCoreClkHardRst_n==0)
  rxReq|->!txReq;
endproperty
rxReqAndNottxReq: assert property (rxReqAndNottxReq_prop);

//$rw_sva Checks for phyRdy asserted only during active transmission or
//reception
property rxReq_txReq_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(phyRdy) |-> $past(txReq || rxReq);
endproperty
rxReq_txReq_phyRdy: assert property (rxReq_txReq_phyRdy_prop);

//$rw_sva Checks for phyRdy not assserted during TxVector phase
property txvector_nonHT_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(txReq) && (formatMod==0 || formatMod==1)) |-> not (##[1:10]  (phyRdy));
endproperty
txvector_nonHT_phyRdy: assert property (txvector_nonHT_phyRdy_prop);

//$rw_sva Checks for phyRdy not assserted during TxVector phase
property txvector_HT_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(txReq) && (formatMod==2 || formatMod==3)) |-> not (##[1:13]  (phyRdy));
endproperty
txvector_HT_phyRdy: assert property (txvector_HT_phyRdy_prop);

//$rw_sva Checks for phyRdy not assserted during TxVector phase
property txvector_VHT_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(txReq) && (formatMod==4)) |-> not (##[1:17]  (phyRdy));                 //ToDo: MultiUser
endproperty
txvector_VHT_phyRdy: assert property (txvector_VHT_phyRdy_prop);

//$rw_sva Checks for phyRdy not assserted during TxVector phase
property txvector_HE_SU_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(txReq) && (formatMod==5 || formatMod==7)) |-> not (##[1:18]  (phyRdy));
endproperty
txvector_HE_SU_phyRdy: assert property (txvector_HE_SU_phyRdy_prop);

//$rw_sva Checks for phyRdy not assserted during TxVector phase
property txvector_HE_MU_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(txReq) && (formatMod==6)) |-> not (##[1:26]  (phyRdy));                 //ToDo: MultiUser & RuAllocation
endproperty
txvector_HE_MU_phyRdy: assert property (txvector_HE_MU_phyRdy_prop);

//$rw_sva Checks for phyRdy not assserted during TxVector phase
property txvector_HE_TRIG_phyRdy_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(txReq) && formatMod==8) |-> not (##[1:24]  (phyRdy));
endproperty
txvector_HE_TRIG_phyRdy: assert property (txvector_HE_TRIG_phyRdy_prop);

//$rw_sva Checks for assertion of txEnd_p 1-5 clock cycles after assertion of
//phyErr_p
property phyErr_txEnd_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(phyErr_p) |-> ##[1:5] $rose(txEnd_p);
endproperty
phyErr_txEnd: assert property (phyErr_txEnd_prop);

//$rw_sva Checks for de-assertion of txReq one clock after txEnd_p is asserted.
property txEnd_txReq_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ((txReq && ~phyErr_p)[*3] ##1 (txReq && $rose(txEnd_p))) |=> ##6 $fell(txReq);
endproperty
txEnd_txReq: assert property (txEnd_txReq_prop);

//$rw_sva Checks for assertion of txEnd_p between 1-5 clock cycles after txReq
//is de-asserted.
property txReq_txEnd_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ((txReq && ~txEnd_p) [*8] ##1 $fell(txReq)) |-> ##[1:5] $rose(txEnd_p);
endproperty
txReq_txEnd: assert property (txReq_txEnd_prop);

//$rw_sva Checks for assertion of macDataValid signal one cycle after assertion
//of phyRdy. This check is valid only during transmission.
property phyRdy_macDataValid_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n== 0 || txReq == 0 || rateControllerMPIFResync)
  $fell(phyRdy) |=>  ##1(! macDataValid);
endproperty
phyRdy_macDataValid: assert property (phyRdy_macDataValid_prop);

//$rw_sva Check for macDataValid is asserted only during transmission
property macDataValid_txReq_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  macDataValid |-> txReq || $past(txReq);
endproperty
macDataValid_txReq: assert property (macDataValid_txReq_prop);

//$rw_sva Checks for LOW value on txReq line when cca lines are HIGH.
property cca_txReq_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  (CCAPrimary20)|->!txReq;
endproperty
cca_txReq: assert property (cca_txReq_prop);

//$rw_sva Check for HIGH value on rxReq line when cca lines are HIGH
property cca_rxReq_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $fell(rxReq) |-> ##[1:30] (!(CCAPrimary20));
endproperty
cca_rxReq: assert property (cca_rxReq_prop);


// //Normal Rx End
// //$rw_sva Checks for de-assertion of rxReq one clock cycle after rxEnd_p is
// //asserted.
// property rxEnd_rxReq_prop;
// @(posedge mpIFClk)
//   disable iff(mpIFClkHardRst_n==0)
//   (rxReq && $rose(rxEnd_p)) |=> $fell(rxReq);
// endproperty
// rxEnd_rxReq: assert property (rxEnd_rxReq_prop);


//$rw_sva The following properties check for keepRFOn. The keepRFOn signal
//should not be asserted or de-asserted when both txReq and rxReq are LOW
property txReq_rxReq_keepRFOn_prop;
@(posedge mpIFClk)
  (!txReq && !rxReq)|=>not ($rose(keepRFOn) || $fell(keepRFOn));
endproperty
txReq_rxReq_keepRFOn: assert property (disable iff (mpIFClkHardRst_n) txReq_rxReq_keepRFOn_prop);

//$rw_sva Checks for keepRFOn assertion atleast one clock cycle before txEnd_p is
//asserted during transmission
property keepRFOn_txReq_txEnd_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(keepRFOn) && txReq)|->(!txEnd_p ##1 !$fell(txReq));
endproperty
keepRFOn_txReq_txEnd: assert property (keepRFOn_txReq_txEnd_prop);

//$rw_sva Checks for keepRFOn not be de-asserted one clock cycle after txReq is
//asserted.
property keepRFOn_txReq_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(txReq)|->!($fell(keepRFOn));
endproperty
keepRFOn_txReq: assert property (keepRFOn_txReq_prop);

//$rw_sva Checks for keepRFOn assertion atleast a clock cycle before
//rxEndForTiming_p is asserted during reception.
property keepRFOn_rxReq_rxEndForTiming_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  ($rose(keepRFOn) && rxReq)|->!rxEndForTiming_p ##[1:10] !rxEnd_p;
endproperty
keepRFOn_rxReq_rxEndForTiming: assert property (keepRFOn_rxReq_rxEndForTiming_prop);

//$rw_sva Check for de-assertion of keepRFOn atlest a clock cycle after rxReq is
//asserted.
property keepRFOn_rxReq_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(rxReq) |->!($fell(keepRFOn));
endproperty
keepRFOn_rxReq: assert property (keepRFOn_rxReq_prop);

//$rw_sva Checks for phyErr signal asserted for a clock cycle
property phyErr_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(phyErr_p)|=>$fell(phyErr_p);
endproperty
phyErr: assert property (phyErr_prop);

//$rw_sva Checks for txEnd signal asserted for a clock cycle
property txEnd_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(txEnd_p)|=>$fell(txEnd_p);
endproperty
txEnd: assert property (txEnd_prop);

//$rw_sva Checks for rxErr signal asserted for a clock cycle
property rxErr_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(rxErr_p)|=>$fell(rxErr_p);
endproperty
rxErr: assert property (rxErr_prop);

//$rw_sva Checks for rxEndForTiming signal asserted for a clock cycle
property rxEndForTiming_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(rxEndForTiming_p)|=>$fell(rxEndForTiming_p);
endproperty
rxEndForTiming: assert property (rxEndForTiming_prop);

//$rw_sva Checks for rxEnd signal asserted for a clock cycle
property rxEnd_prop;
@(posedge mpIFClk)
  disable iff(mpIFClkHardRst_n==0)
  $rose(rxEnd_p)|=>$fell(rxEnd_p);
endproperty
rxEnd: assert property (rxEnd_prop);

///////////////////////////////////////////////////////
// FIFO HANDLING PROPERTIES
///////////////////////////////////////////////////////

//$rw_sva Checks the controller does not write into the Tx FIFO when it's full
property noWriteWhenTxFifoFullCheck_prop;
@(posedge macCoreTxClk)
    (mpIfTxFifoFull == 1'b1) |-> mpIfTxFifoWrite == 1'b0;
endproperty
noWriteWhenTxFifoFullCheck: assert property (noWriteWhenTxFifoFullCheck_prop);


//$rw_sva Checks the controller does not read into the Tx FIFO when it's empty
property noReadWhenTxFifoEmptyCheck_prop;
@(posedge macCoreTxClk)
    (mpIfTxFifoEmtpy == 1'b1) |-> mpIfFifoReadEn == 1'b0;
endproperty
noReadWhenTxFifoEmptyChecks: assert property (noReadWhenTxFifoEmptyCheck_prop);


//$rw_sva Checks the controller does not write into the Rx FIFO when it's full
property noWriteWhenRxFifoFullCheck_prop;
@(posedge mpIFClk)
    (mpIfRxFifoFull == 1'b1) |-> fifoWriteEn == 1'b0;
endproperty
//noWriteWhenRxFifoFullCheck: assert property (noWriteWhenRxFifoFullCheck_prop);

//$rw_sva Checks the controller does not read into the Rx FIFO when it's empty
property noReadWhenRxFifoEmptyCheck_prop;
@(posedge macCoreRxClk)
    (macPhyIfRxFifoEmpty == 1'b1) |-> macPhyIfRxFifoReadEn == 1'b0;
endproperty
noReadWhenRxFifoEmptyCheck: assert property (noReadWhenRxFifoEmptyCheck_prop);

`endif // RW_ASSERT_ON
endmodule

