//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//     The read and write pointers for transmit FIFO are controlled here.
//     Also the FIFO status such as, full, empty, 
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module mpiftxRxFifoControl #(parameter ADDRWIDTH = 4,parameter DEPTH = 64)(
          //$port_g Inputs are defined here..
          input  wire                       wrClk ,           // Write clock     
          input  wire                       wrHardReset_n ,   // Write clock hard reset         
          input  wire                       rdClk ,           // Read clock     
          input  wire                       rdHardReset_n ,   // Read clock hard reset
          input  wire                       rdFlush ,         // Read Flush     
          input  wire                       fifoWrite ,       // Write enables     
          input  wire                       fifoRead ,        // Read enable     

          //$port_g Outputs defined here ..
          output wire   [(ADDRWIDTH - 1):0] fifoWrPtr ,       // Write ptr
          output wire   [(ADDRWIDTH - 1):0] fifoRdPtr ,       // Read ptr
          output wire                       fifoFull ,        // fifo full indicator
          output wire                       fifoAlmostFull ,  // fifo full indicator
          output wire                       fifoEmpty ,       // fifo empty indicator
          output wire                       fifoPtrsNull,     // Indicates FIFO pointers null
          output wire   [ADDRWIDTH:0]       fifoWrRemain      // Indicates remaining bytes to write
                                                              // before FIFO Full
          );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// Output types defined here ..
// Internal variables of the module ..
reg    [ADDRWIDTH:0] fifoWrAddr ;

// Internal variables of the module ..
wire   [ADDRWIDTH:0] fifoWrAddrGray ;
wire   [ADDRWIDTH:0] fifoWrAddrSync ;
wire   [ADDRWIDTH:0] fifoWrAddrGraySync ;
wire   [ADDRWIDTH:0] fifoWrAddrAlmost ;
wire   [ADDRWIDTH:0] fifoRdAddr ;
wire   [ADDRWIDTH:0] fifoRdAddrGray ;
wire   [ADDRWIDTH:0] fifoRdAddrSync ;
wire   [ADDRWIDTH:0] fifoRdAddrGraySync ;
wire   [ADDRWIDTH:0] fifoDepth ;

wire                 wrFlush ;      
wire                 rdFlushDly ;      
reg                  rdFlushDly2 ;      
reg                  rdFlushDly3 ;      
wire                 rdFlushInt;

reg                  flushOnGoing;

`ifdef RW_SIMU_ON
integer fifoCnt;
`endif // RW_SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
// Use internal wire to avoid unequal lentgth operator in Full & AlmostFull signal generation
assign fifoDepth = DEPTH[ADDRWIDTH:0];

// This function converts Gray Code into Binary Code
function [ADDRWIDTH:0] gray2Binary ;
input    [ADDRWIDTH:0] gray ;
integer i ;
begin
   gray2Binary[ADDRWIDTH] = gray[ADDRWIDTH] ;
   for(i = 1; i <= ADDRWIDTH; i = i + 1) 
   begin
      gray2Binary[ADDRWIDTH - i] =
           gray2Binary[ADDRWIDTH - i+1] ^ gray[ADDRWIDTH - i] ;
   end
end
endfunction

// fifoRdAddrGray converted to binary
assign fifoRdAddr = gray2Binary(fifoRdAddrGray) ;

// Gray to Binary conversion for the synchronised Read Address
assign fifoRdAddrSync = gray2Binary(fifoRdAddrGraySync) ;

// Gray to Binary conversion for the synchronised Write Address
assign fifoWrAddrSync = gray2Binary(fifoWrAddrGraySync) ;

// The MSB of the address is not used as a part of Pointer.
//fifoWrPtr
assign fifoWrPtr  = fifoWrAddr[(ADDRWIDTH-1):0] ;
// Signal used to generate fifoAlmostFull signal, threshold is tied to 1
assign fifoWrAddrAlmost = fifoWrAddr + {{(ADDRWIDTH){1'b0}},1'd1};
//fifoRdPtr
assign fifoRdPtr  = fifoRdAddr[(ADDRWIDTH-1):0] ;

// The status of fifoFull is continuously driven based on the 
// fifo addresses
assign fifoFull = (fifoWrAddr[(ADDRWIDTH-1):0] == fifoRdAddrSync[(ADDRWIDTH-1):0]) &&
                  (fifoWrAddr[ADDRWIDTH] ^ fifoRdAddrSync[ADDRWIDTH]);
                  
assign fifoAlmostFull = (fifoWrAddrAlmost[ADDRWIDTH] == fifoRdAddrSync[ADDRWIDTH]) ? 
                            ((fifoWrAddrAlmost[(ADDRWIDTH-1):0] < fifoRdAddrSync[(ADDRWIDTH-1):0]) ? 
                                ({1'b0,fifoWrAddrAlmost[(ADDRWIDTH-1):0]}+fifoDepth >= {1'b0,fifoRdAddrSync[(ADDRWIDTH-1):0]}) :
                                 1'b0) :
                            (fifoWrAddrAlmost[(ADDRWIDTH-1):0] >= fifoRdAddrSync[(ADDRWIDTH-1):0]);

// Indicates remaining bytes to write before FIFO Full
assign fifoWrRemain = fifoWrAddr[ADDRWIDTH] ^ fifoRdAddrSync[ADDRWIDTH] ?
   {1'b0,fifoRdAddrSync[ADDRWIDTH-1:0]}-{1'b0,fifoWrAddr[ADDRWIDTH-1:0]}           :
   {1'b0,fifoRdAddrSync[ADDRWIDTH-1:0]}-{1'b0,fifoWrAddr[ADDRWIDTH-1:0]}+fifoDepth ;

// fifoEmpty is asserted in rdClk domain after synchronising the
// fifoWrAddr to rdClk. 
assign fifoEmpty = (fifoRdAddr == fifoWrAddrSync) || flushOnGoing || rdFlush;

// fifoPtrsNull is asserted when both write and read pointers are null
assign fifoPtrsNull = (fifoRdAddr == fifoWrAddrSync) && (fifoWrAddrSync == {(ADDRWIDTH+1){1'b0}});

// fifoRdAddrGray is synchronised to rdClk
// Following instantiation increments fifoRdAddr and
// converts this new address into corresponding Gray encoding
fifoPtrGray #(.ADDRWIDTH(ADDRWIDTH)) u_fifoRdAddrGray(
                             .clk(rdClk),
                             .hardReset_n(rdHardReset_n),
                             .flush(rdFlush),
                             .rdWrEn(fifoRead),
                             .rdWrDisable(fifoEmpty),
                             .fifoAddr(fifoRdAddr),
                             .fifoAddrGray(fifoRdAddrGray)
                            );

// fifoRdAddrGray is synchronised to wrClk
ClkSyncSimpleTop #(.SIZE(ADDRWIDTH+1)) u_fifoRdAddrSync (
                             .dstclk (wrClk) ,
                             .dstresetn (wrHardReset_n) ,
                             .srcdata (fifoRdAddrGray) ,
                             .dstdata (fifoRdAddrGraySync)
                             );

// fifoWrAddrGray is syncronised to rdClk
// Following instantiation increments fifoWrAddr and
// converts this new address into corresponding Gray encoding
fifoPtrGray #(.ADDRWIDTH(ADDRWIDTH)) u_fifoWrAddrGray(
                             .clk(wrClk),
                             .hardReset_n(wrHardReset_n),
                             .flush(wrFlush),
                             .rdWrEn(fifoWrite),
                             .rdWrDisable(fifoFull),
                             .fifoAddr(fifoWrAddr),
                             .fifoAddrGray(fifoWrAddrGray)
                            );

// fifoWrAddrGray is synchronised to rdClk
ClkSyncSimpleTop #(.SIZE(ADDRWIDTH+1)) u_fifoWrAddrSync (
                             .dstclk (rdClk) ,
                             .dstresetn (rdHardReset_n) ,
                             .srcdata (fifoWrAddrGray) ,
                             .dstdata (fifoWrAddrGraySync)
                             );


// Generation of wrFlush 
pulse2PulseSynchro U_wrFlush_synchro (
                .srcclk (rdClk),
                .srcresetn (rdHardReset_n),
                .dstclk (wrClk),
                .dstresetn (wrHardReset_n),
                .srcdata (rdFlushInt),
                .dstdata (wrFlush)
                );

pulse2PulseSynchro U_rdFlushDly_synchro (
                .srcclk (wrClk),
                .srcresetn (wrHardReset_n),
                .dstclk (rdClk),
                .dstresetn (rdHardReset_n),
                .srcdata (wrFlush),
                .dstdata (rdFlushDly)
                );
                

// Generation of flushOnGoing flag indicating that the FIFO is being flushed.
// It allows to keep the fifo empty flag set during the flush process.
always @(posedge rdClk or negedge rdHardReset_n) 
begin
   if (rdHardReset_n == 1'b0) 
     flushOnGoing <= 1'b0; 
   else if (rdFlushDly) 
     flushOnGoing <= 1'b0; 
   else if(rdFlushInt) 
     flushOnGoing <= 1'b1; 
end


always @(posedge rdClk or negedge rdHardReset_n) 
begin
   if (rdHardReset_n == 1'b0) 
   begin
     rdFlushDly2 <= 1'b0; 
     rdFlushDly3 <= 1'b0; 
   end
   else
   begin
     rdFlushDly2 <= rdFlushDly; 
     rdFlushDly3 <= rdFlushDly2; 
   end  
end

assign rdFlushInt =  rdFlush && !flushOnGoing && !rdFlushDly && !rdFlushDly3 && !rdFlushDly2;


// The fifoWrAddr is incrmented on every fifoWrite.
always @(posedge wrClk or negedge wrHardReset_n) 
begin
   if (wrHardReset_n == 1'b0) 
     fifoWrAddr <= {(ADDRWIDTH+1){1'b0}}; 
   else if (wrFlush) 
     fifoWrAddr <= {(ADDRWIDTH+1){1'b0}}; 
   else if(fifoWrite && (fifoFull == 1'b0)) 
     fifoWrAddr <= fifoWrAddr + {{(ADDRWIDTH){1'b0}},1'd1};
   else 
     fifoWrAddr <= fifoWrAddr;
end


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_SIMU_ON

always @(posedge wrClk or negedge wrHardReset_n) 
begin
   if(wrHardReset_n == 1'b0) 
     fifoCnt = 0; 
   else if (wrFlush) 
     fifoCnt = 0; 
   else if(fifoWrite && (fifoFull == 1'b0)) 
     fifoCnt = fifoCnt + 1; 
end

always @(posedge rdClk or negedge rdHardReset_n) 
begin
   if(rdHardReset_n == 1'b0) 
     fifoCnt = 0; 
   else if (wrFlush) 
     fifoCnt = 0; 
   else if(fifoRead && (fifoEmpty == 1'b0)) 
     fifoCnt = fifoCnt - 1; 
end


`endif // RW_SIMU_ON
endmodule
//---------- END OF FILE ----------
