//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: 2017-09-27 09:08:00 +0200 (Wed, 27 Sep 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Add signal extension for timing alignement on OFDM 
//                    frame in 2,.4GHz band
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module signalExtAdder ( 
            //$port_g clock and Reset
            input  wire       macCoreTxClk,                   // mac core Rx clock                  
            input  wire       macCoreRxClk,                   // mac core Rx clock                  
            input  wire       macCoreClkHardRst_n,            // mac core Rx clock hard reset            
            input  wire       macCoreClkSoftRst_n,            // mac core Tx clock reset       

            //$port_g control
            input  wire       rxDSSS,                         // Ongoing reception is DSSS
            input  wire       band5G,                         // Indicate the current operating band (0: 2.4GHz, 1:5GHz)
            input  wire [3:0] formatMod,                      // TX Vector Format and Modulation           
            input  wire [3:0] legRate,                        // TX Vector Legacy Rate of the PPDU               
            input  wire       stopTx_p,                       // Stop Tx trigger                    

            //$port_g PHY IF
            

            input  wire       macPhyIfRxEndForTimingWoSigExt_p, // end of reception (at antenna) 
            output reg        macPhyIfRxEndForTiming_p,         // end of reception (at antenna includng signal extension) 
            input  wire       macPhyIfRxEndWoSigExt_p,          // end of reception 
            output reg        macPhyIfRxEnd_p,                  // end of reception (includng signal extension) 
            input  wire       macPhyIfRxErrWoSigExt_p,
            output reg        macPhyIfRxErr_p,
            input  wire       macPhyTxEndWoSigExt_p,            // end of transmission (antenna)
            output reg        macPhyTxEnd_p,                    // end of transmission (including signal extension)
            input  wire       mpIfTxErr_p,

            //$port_g CSReg
            input  wire [7:0] macCoreClkFreq                  // MAC Core Clock Frequency
            );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////



//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

reg [10:0] rxEndForTimingSigExtCnt;                        
reg        rxEndForTimingSigExtCntStarted;                        
reg [10:0] txEndSigExtCnt;                        
reg        txEndSigExtCntStarted;                        

reg        macPhyTxErrRcved;
reg        macPhyIfRxEndRcved;
reg        macPhyIfRxErrRcved;
reg        stopTxRcved;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Signal extension in receive
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    macPhyIfRxEndForTiming_p       <= 1'b0;
    macPhyIfRxEnd_p                <= 1'b0;
    macPhyIfRxErr_p                <= 1'b0;
    macPhyIfRxEndRcved             <= 1'b0;
    macPhyIfRxErrRcved             <= 1'b0;
    rxEndForTimingSigExtCntStarted <= 1'b0; 
    rxEndForTimingSigExtCnt        <= 11'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
  begin
    macPhyIfRxEndForTiming_p       <= 1'b0;
    macPhyIfRxEnd_p                <= 1'b0;
    macPhyIfRxErr_p                <= 1'b0;
    macPhyIfRxEndRcved             <= 1'b0;
    macPhyIfRxErrRcved             <= 1'b0;
    rxEndForTimingSigExtCntStarted <= 1'b0; 
    rxEndForTimingSigExtCnt        <= 11'b0;
  end
  else
  begin 
    if (macPhyIfRxEndForTimingWoSigExt_p)
      rxEndForTimingSigExtCntStarted <= 1'b1;

    if (macPhyIfRxEndWoSigExt_p)
      macPhyIfRxEndRcved <= 1'b1;
    
    if (macPhyIfRxErrWoSigExt_p)
      macPhyIfRxErrRcved <= 1'b1;

    if (rxEndForTimingSigExtCntStarted && (macPhyIfRxErrRcved ||
                                           rxDSSS             || 
                                           band5G             || 
                                           (rxEndForTimingSigExtCnt == (3'd6*macCoreClkFreq))))
    begin
      macPhyIfRxEndForTiming_p       <= 1'b1;
      rxEndForTimingSigExtCntStarted <= 1'b0;
      rxEndForTimingSigExtCnt        <= 11'b0;
    end
    else
    begin
      macPhyIfRxEndForTiming_p       <= 1'b0;
      if (rxEndForTimingSigExtCntStarted)
        rxEndForTimingSigExtCnt <= rxEndForTimingSigExtCnt + 11'b1;
    end



    if (macPhyIfRxEndRcved && !rxEndForTimingSigExtCntStarted)
    begin
      macPhyIfRxEnd_p                  <= 1'b1;
      macPhyIfRxEndRcved               <= 1'b0;
    end
    else
      macPhyIfRxEnd_p                  <= 1'b0;

    if (macPhyIfRxErrRcved && !rxEndForTimingSigExtCntStarted)
    begin
      macPhyIfRxErr_p                  <= 1'b1;
      macPhyIfRxErrRcved               <= 1'b0;
    end
    else
      macPhyIfRxErr_p                  <= 1'b0;

  end
end


// Signal extension in transmit
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    macPhyTxEnd_p         <= 1'b0; 
    txEndSigExtCntStarted <= 1'b0; 
    txEndSigExtCnt        <= 11'b0;
    macPhyTxErrRcved      <= 1'b0;
    stopTxRcved           <= 1'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0)  // Synchronous Reset
  begin
    macPhyTxEnd_p         <= 1'b0; 
    txEndSigExtCntStarted <= 1'b0; 
    txEndSigExtCnt        <= 11'b0;
    macPhyTxErrRcved      <= 1'b0;
    stopTxRcved           <= 1'b0;
  end
  else
  begin 
    if (macPhyTxEndWoSigExt_p)
      txEndSigExtCntStarted <= 1'b1;

    if (mpIfTxErr_p)
    begin
      macPhyTxErrRcved      <= 1'b1;
    end 

    if (stopTx_p)
    begin
      stopTxRcved           <= 1'b1;
    end 

    if (txEndSigExtCntStarted && (macPhyTxErrRcved                          || // In case of TX Error
                                  stopTxRcved                               || // or if MAC stops the transmission
                                  ((legRate < 4'd4) && (formatMod == 4'd0)) || // or if Tx is DSSS
                                  band5G                                    || // or if TX is on 5GHz band
                                  (txEndSigExtCnt == (3'd6*macCoreClkFreq))))  // or if signalExtension counter reaches 6us
    begin
      macPhyTxEnd_p         <= 1'b1;
      macPhyTxErrRcved      <= 1'b0;
      stopTxRcved           <= 1'b0;
      txEndSigExtCntStarted <= 1'b0;
      txEndSigExtCnt        <= 11'b0;
    end
    else
    begin
      macPhyTxEnd_p         <= 1'b0;
      if (txEndSigExtCntStarted)
        txEndSigExtCnt <= txEndSigExtCnt + 11'b1;
    end  


  end
end

endmodule
                 
