////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of nav module
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -----------------------------------------------------------------------------
//                                                                          
// 
// 
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module nav( 
            //$port_g Clock and Reset
            input  wire        macCoreClk,                    // mac core clock      
            input  wire        macCoreClkHardRst_n,           // mac core clock reset
            input  wire        macCoreClkSoftRst_n,           // Soft Reset of the MAC Core Clock domain
                                                              // active low
            //$port_g MAC Controller Module interface
            input  wire [15:0] ctsDuration,                   // Indicates the duration of a CTS frame response of the received RTS
            input  wire        ctsDurationValid_p,            // Indicates that the ctsDuration is valid
            input  wire [15:0] ackDuration,                   // Indicates the duration of a ACK frame response of the received PS-POLL
            input  wire        ackDurationValid_p,            // Indicates that the ackDuration is valid
            input  wire  [7:0] phyRxStartDelay,               // Receive Start Delay of the PHY
            input  wire        moveToActive_p,                // Core should now move to ACTIVE state
            input  wire        stopRx_p,                      // Stop Rx
            
            //$port_g Rx controller interface
            input  wire        basicNAVClear_p,               // Clear basic NAV
            input  wire        intraNAVClear_p,               // Clear intra NAV
            input  wire        basicNAVUpdate_p,              // Update basic NAV with frame
                                                              // duration
            input  wire        intraNAVUpdate_p,              // Update intra NAV with frame
                                                              // duration
            input  wire [14:0] navUpdateDuration,             // Duration for the NAV update
            input  wire        notMinePsPollRcved_p,          // indicate reception of a pspoll
            input  wire        incorrectRcved_p,              // Frame not received correctly for PS-POLL case
            input  wire        correctRcved_p,                // Frame received correctly
            input  wire        notMineRtsRcved_p,             // indicate the reception of an RTS
            input  wire        notMineMURtsRcved_p,           // indicate the reception of an MU-RTS
                         
            //$port_g macPhyIf interface
            input  wire        macPhyIfRxEndForTiming_p,      // end of transmission (antenna) 
            input  wire        macPhyIfRxEnd_p,               // end of transmission
            input  wire        macPhyIfRxStart_p,             // Start of reception pulse
 
            //$port_g MAC Timer Unit interface
            input  wire        tick1us_p,                     // 1 us pulse
            input  wire        loadQuietDuration_p,           // load quiet duration
            input  wire [15:0] quietDuration,                 // quiet duration value
            output wire        channelToIdle_p,               // NAV will go idle

            //$port_g Spatial Reuse interface
            input  wire        basicNAVUpdateMask,            // Indicates the basic NAV should not
                                                              // be updated due to Spatial Reuse
                                                              // opportunity.

            //$port_g Registers Interface
            input  wire [ 7:0] sifs,                          // SIFS Duration in us
            input  wire [ 7:0] slotTime,                      // slot time for timeout
            input  wire [15:0] probeDelay,                    // Probe delay during active scan
            input  wire [25:0] basicNAVCounter,               // basic NAV counter value from
                                                              // registers
            output wire [25:0] basicNAVCounterIn,             // basic NAV counter value to
                                                              // registers
            output reg         basicNAVCounterInValid,        // basic NAV counter value to
                                                              // registers
            input  wire [15:0] intraNAVCounter,               // intra-BSS NAV counter value from
                                                              // registers
            output wire [15:0] intraNAVCounterIn,             // intra-BSS NAV counter value to 
                                                              // registers
            output reg         intraNAVCounterInValid,        // intra-BSS NAV counter value to 
                                                              // registers
            
            //$port_g Status
            output wire        channelBusy,                   // Channel busy indicator
            output wire        channelIdle,                   // Channel idle indicator
            output wire        basicNAVBusy,                  // Basic NAV is not null
            output wire        intraNAVBusy,                  // Intra-BSS NAV is not null
            
            //$port_g Debug
            output wire [15:0] debugPortNAV                   // Debug Port NAV
);


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg         rtsRcvdSampled;            // keep in memory the RTS received indicator
reg         notMineRtsRcved;           // Capture version of notMineRtsRcved_p or
                                       // notMineMURtsRcved_p
                                       
reg  [15:0] int_ctsDuration;           // For time out counter
reg  [ 7:0] int_phyRxStartDelay;       // For time out counter
reg         navCounterHoldsQuiet;      // hold quiet duration
reg         navCounterHoldsProbe;      // hold probe delay
                                       
reg  [ 7:0] navOffset;                 // NAV offset in case of restart
reg         navOffsetEn;               // Enable NAV offset calculation
                                       
wire [15:0] timeoutVal;                // value for the timeout
reg         timeout_p;                 // indicate a timeout occured
reg  [15:0] timeoutCount;              // time out counter
reg         psPollReceived;            // a pspoll has been received
wire [15:0] pspollAckprotection;       // a pspollprotection length
reg         basicNAVPsPollUpdate;      // Flag indicating that basic NAV update is
                                       // requested for PS-POLL protection
reg         intraNAVPsPollUpdate;      // Flag indicating that intra NAV update is
                                       // requested for PS-POLL protection

reg         basicNAVCounterEnable;     // basic NAV counter enable
wire [25:0] basicNAVCounterRemaining;  // basic NAV counter remaining
reg  [25:0] basicNAVCounterInt;        // basic NAV counter internal

reg         nextBasicNAVCounterEnable; // next value of the basic NAV counter enable
reg  [25:0] nextBasicNAVCounterInt;    // next value of the basic NAV counter internal
reg         nextBasicNAVCounterInValid;// next value of the basic NAV counter in valid

reg         intraNAVCounterEnable;     // intra NAV counter enable
wire [15:0] intraNAVCounterRemaining;  // intra NAV counter remaining
reg  [15:0] intraNAVCounterInt;        // intra NAV counter internal

reg         nextIntraNAVCounterEnable; // next value of the intra NAV counter enable
reg  [15:0] nextIntraNAVCounterInt;    // next value of the intra NAV counter internal
reg         nextIntraNAVCounterInValid;// next value of the intra NAV counter in valid

reg         basicNAVLoad_p;            // Basic NAV Load  pulse
reg         intraNAVLoad_p;            // Intra NAV Load  pulse
wire        basicNAVReset_p;           // Basic NAV Reset pulse
wire        intraNAVReset_p;           // Intra NAV Reset pulse

reg         lastUpdateIsBasicRTS;      // Indicates that the last NAV update is the basic NAV 
                                       // due to an RTS frame
reg         lastUpdateIsIntraRTS;      // Indicates that the last NAV update is the intra NAV 
                                       // due to an RTS frame

wire [15:0] pspollNAV;                 // New NAV Value from PSPOLL
wire [15:0] durationNAV;               // New NAV Value from navUpdateDuration

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// ctsDuration sampling
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    int_ctsDuration <= 16'h0000;
  else if(macCoreClkSoftRst_n == 1'b0)
    int_ctsDuration <= 16'h0000;
  else if (ctsDurationValid_p)
    int_ctsDuration <= ctsDuration;
end


// phyRxStartDelay sampling
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    int_phyRxStartDelay <= 8'h0;
  else if(macCoreClkSoftRst_n == 1'b0)
    int_phyRxStartDelay <= 8'h0;
  else if (ctsDurationValid_p)
    int_phyRxStartDelay <= phyRxStartDelay;
end


// duration of the PS-Poll ACK protection
assign pspollAckprotection  = ackDuration + {8'b0, sifs};


// Channel Status
assign basicNAVBusy = basicNAVCounterEnable;
assign intraNAVBusy = intraNAVCounterEnable;

assign channelIdle = !basicNAVCounterEnable & !intraNAVCounterEnable;
assign channelBusy =  basicNAVCounterEnable |  intraNAVCounterEnable;


// CTS Timeout value
assign timeoutVal = {7'b0, sifs,1'b0} +
                    {7'b0, slotTime,1'b0} +
                    {8'b0, int_phyRxStartDelay} +
                    int_ctsDuration;


// channelToIdle_p generation 1 cc before channel goes to idle for SIFS boundary
assign channelToIdle_p = (basicNAVCounterEnable || intraNAVCounterEnable) &&
                         !nextBasicNAVCounterEnable                       &&
                         !nextIntraNAVCounterEnable;


// NAV Clear
// reset the NAV in case of external clear or timeout
assign basicNAVReset_p = (basicNAVClear_p                                                   ) ||
                         (timeout_p         && !navCounterHoldsQuiet && lastUpdateIsBasicRTS) ||
                         (macPhyIfRxStart_p &&  navCounterHoldsProbe                        );

assign intraNAVReset_p = (intraNAVClear_p                           ) || 
                         (timeout_p         &&  lastUpdateIsIntraRTS);


// new NAV Value with offset
assign pspollNAV   = (pspollAckprotection      > {8'h0, navOffset}) ? 
                      pspollAckprotection      - {8'h0, navOffset} : 16'h0;

assign durationNAV = ({1'b0,navUpdateDuration} > {8'h0, navOffset}) ? 
                      {1'b0,navUpdateDuration} - {8'h0, navOffset} : 16'h0;


//------------------------------------------------------------------------------
// Basic NAV counter
//------------------------------------------------------------------------------
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    basicNAVCounterInt     <= 26'h0;
    basicNAVCounterEnable  <= 1'b0;
    basicNAVCounterInValid <= 1'b1;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    basicNAVCounterInt     <= 26'h0;
    basicNAVCounterEnable  <= 1'b0;
    basicNAVCounterInValid <= 1'b1;
  end
  else
  begin
    basicNAVCounterInt     <= nextBasicNAVCounterInt;
    basicNAVCounterEnable  <= nextBasicNAVCounterEnable;
    basicNAVCounterInValid <= nextBasicNAVCounterInValid;
  end
end

always @*
begin
  // update quiet duration (No check on current NAV needed)
  if (loadQuietDuration_p)
  begin
    nextBasicNAVCounterEnable  = 1'b1;
    nextBasicNAVCounterInt     = {quietDuration,10'b0};
    nextBasicNAVCounterInValid = 1'b1;

    basicNAVLoad_p             = 1'b1;
  end
  // update probe delay (No check on current NAV needed)
  else if (moveToActive_p)
  begin
    nextBasicNAVCounterEnable  = 1'b1;
    nextBasicNAVCounterInt     = {10'b0,probeDelay};
    nextBasicNAVCounterInValid = 1'b1;

    basicNAVLoad_p             = 1'b1;
  end
  // reset the NAV in case of external clear or timeout
  else if (basicNAVReset_p)
  begin
    nextBasicNAVCounterEnable  = 1'b0;
    nextBasicNAVCounterInt     = 26'h0;
    nextBasicNAVCounterInValid = 1'b1;

    basicNAVLoad_p             = 1'b0;
  end
  // PS-POLL received
  else if (basicNAVPsPollUpdate && ackDurationValid_p &&
           ({10'h0,pspollAckprotection} > basicNAVCounterRemaining))
  begin
    nextBasicNAVCounterEnable  = 1'b1;
    nextBasicNAVCounterInt     = {10'd0, pspollNAV};
    nextBasicNAVCounterInValid = 1'b1;

    basicNAVLoad_p             = 1'b1;
  end
  // update frame duration
  else if (basicNAVUpdate_p && !basicNAVUpdateMask && !psPollReceived &&
           ({11'h0,navUpdateDuration} > basicNAVCounterRemaining))
  begin
    nextBasicNAVCounterEnable  = 1'b1;
    nextBasicNAVCounterInt     = {10'd0, durationNAV};
    nextBasicNAVCounterInValid = 1'b1;

    basicNAVLoad_p             = 1'b1;
  end
  else if (basicNAVCounterEnable && tick1us_p)
  begin
    if (basicNAVCounterInt <= 26'd1)
    begin
      nextBasicNAVCounterEnable  = 1'b0;
      nextBasicNAVCounterInt     = 26'h0000;
      nextBasicNAVCounterInValid = 1'b1;

      basicNAVLoad_p             = 1'b0;
    end
    else
    begin
      nextBasicNAVCounterEnable  = basicNAVCounterEnable;
      nextBasicNAVCounterInt     = basicNAVCounterInt - 26'd1;
      nextBasicNAVCounterInValid = 1'b1;

      basicNAVLoad_p             = 1'b0;
    end
  end
  else
  begin
    nextBasicNAVCounterEnable  = basicNAVCounterEnable;
    nextBasicNAVCounterInt     = basicNAVCounterInt;
    nextBasicNAVCounterInValid = 1'b0;

    basicNAVLoad_p             = 1'b0;
  end     
end


//------------------------------------------------------------------------------
// Intra NAV counter
//------------------------------------------------------------------------------
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    intraNAVCounterInt     <= 16'h0;
    intraNAVCounterEnable  <= 1'b0;
    intraNAVCounterInValid <= 1'b1;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    intraNAVCounterInt     <= 16'h0;
    intraNAVCounterEnable  <= 1'b0;
    intraNAVCounterInValid <= 1'b1;
  end
  else
  begin
    intraNAVCounterInt     <= nextIntraNAVCounterInt;
    intraNAVCounterEnable  <= nextIntraNAVCounterEnable;
    intraNAVCounterInValid <= nextIntraNAVCounterInValid;
  end
end

always @*
begin
  // reset the NAV in case of external clear or timeout
  if (intraNAVReset_p)
  begin
    nextIntraNAVCounterEnable  = 1'b0;
    nextIntraNAVCounterInt     = 16'h0;
    nextIntraNAVCounterInValid = 1'b1;

    intraNAVLoad_p             = 1'b0;
  end
  // PS-POLL received
  else if (intraNAVPsPollUpdate && ackDurationValid_p &&
           (pspollAckprotection > intraNAVCounterRemaining))
  begin
    nextIntraNAVCounterEnable  = 1'b1;
    nextIntraNAVCounterInt     = pspollNAV;
    nextIntraNAVCounterInValid = 1'b1;

    intraNAVLoad_p             = 1'b1;
  end
  // update frame duration
  else if (intraNAVUpdate_p && !psPollReceived &&
           ({1'b0,navUpdateDuration} > intraNAVCounterRemaining))
  begin
    nextIntraNAVCounterEnable  = 1'b1;
    nextIntraNAVCounterInt     = durationNAV;
    nextIntraNAVCounterInValid = 1'b1;

    intraNAVLoad_p             = 1'b1;
  end
  else if (intraNAVCounterEnable && tick1us_p)
  begin
    if (intraNAVCounterInt <= 16'd1)
    begin
      nextIntraNAVCounterEnable  = 1'b0;
      nextIntraNAVCounterInt     = 16'h0;
      nextIntraNAVCounterInValid = 1'b1;

      intraNAVLoad_p             = 1'b0;
    end
    else
    begin
      nextIntraNAVCounterEnable  = intraNAVCounterEnable;
      nextIntraNAVCounterInt     = intraNAVCounterInt - 16'd1;
      nextIntraNAVCounterInValid = 1'b1;

      intraNAVLoad_p             = 1'b0;
    end
  end
  else
  begin
    nextIntraNAVCounterEnable  = intraNAVCounterEnable;
    nextIntraNAVCounterInt     = intraNAVCounterInt;
    nextIntraNAVCounterInValid = 1'b0;

    intraNAVLoad_p             = 1'b0;
  end     
end


//------------------------------------------------------------------------------
// NAV counter remaining
//------------------------------------------------------------------------------
assign basicNAVCounterRemaining = (basicNAVCounterEnable) ? basicNAVCounterInt + {18'd0,navOffset}
                                                          : 26'd0;
assign basicNAVCounterIn        =  basicNAVCounterInt;

assign intraNAVCounterRemaining = (intraNAVCounterEnable) ? intraNAVCounterInt +  {8'd0,navOffset}
                                                          : 16'd0;
assign intraNAVCounterIn        =  intraNAVCounterInt;


//------------------------------------------------------------------------------
// CTS timeout control
//------------------------------------------------------------------------------
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    timeoutCount    <= 16'h0000;
    timeout_p       <= 1'b0;
    rtsRcvdSampled  <= 1'b0;
    notMineRtsRcved <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    timeoutCount    <= 16'h0000;
    timeout_p       <= 1'b0;
    rtsRcvdSampled  <= 1'b0;
    notMineRtsRcved <= 1'b0;
  end
  else
  begin
    if (notMineRtsRcved_p || notMineMURtsRcved_p)
      notMineRtsRcved <= 1'b1;
    else if (macPhyIfRxEnd_p || timeout_p || macPhyIfRxStart_p || incorrectRcved_p)
      notMineRtsRcved <= 1'b0;

    if (timeout_p || macPhyIfRxStart_p || incorrectRcved_p)
      rtsRcvdSampled <= 1'b0;
    else if (notMineRtsRcved && macPhyIfRxEnd_p)
      rtsRcvdSampled <= 1'b1;

    // reset the nav timeout when packet is received 
    // or when channel is idle
    if ((basicNAVCounter==26'h0 && intraNAVCounter==26'h0) ||
        macPhyIfRxStart_p)
    begin
      timeoutCount   <= 16'h0000;
      timeout_p      <= 1'b0;
    end
    else if (tick1us_p)
    begin
      // timeout reached, assert timeout pulse
      if ((timeoutCount  == (timeoutVal - 16'd1)) && rtsRcvdSampled)
      begin
        timeoutCount   <= 16'h0000;
        timeout_p      <= 1'b1;
      end
      // stop the count when hold duration is enabled
      else if (!navCounterHoldsQuiet && rtsRcvdSampled)
      begin
        timeoutCount <= timeoutCount + 16'd1;
        timeout_p    <= 1'b0;
      end
      else
      begin
        timeoutCount <= timeoutCount;
        timeout_p    <= 1'b0;
      end
    end
    else
    begin
      timeoutCount <= timeoutCount;
      timeout_p    <= 1'b0;
    end
  end
end


//------------------------------------------------------------------------------
// input pulse latch
//------------------------------------------------------------------------------
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    psPollReceived       <= 1'b0;
    basicNAVPsPollUpdate <= 1'b0;
    intraNAVPsPollUpdate <= 1'b0;

    navCounterHoldsQuiet <= 1'b0;
    navCounterHoldsProbe <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    psPollReceived       <= 1'b0;
    basicNAVPsPollUpdate <= 1'b0;
    intraNAVPsPollUpdate <= 1'b0;

    navCounterHoldsQuiet <= 1'b0;
    navCounterHoldsProbe <= 1'b0;
  end
  else
  begin
    // PS-POLL received
    if (notMinePsPollRcved_p)
      psPollReceived <= 1'b1;
    else if (basicNAVUpdate_p || intraNAVUpdate_p || incorrectRcved_p)
      psPollReceived <= 1'b0;
    
    // basic NAV update on PS-POLL
    if (basicNAVUpdate_p && !basicNAVUpdateMask && psPollReceived)
      basicNAVPsPollUpdate <= 1'b1;
    else if (ackDurationValid_p)
      basicNAVPsPollUpdate <= 1'b0;

    // intra NAV update on PS-POLL
    if (intraNAVUpdate_p && psPollReceived)
      intraNAVPsPollUpdate <= 1'b1;
    else if (ackDurationValid_p)
      intraNAVPsPollUpdate <= 1'b0;
    
    // start quiet duration
    if (loadQuietDuration_p) 
      navCounterHoldsQuiet <= 1'b1;
    // end quiet duration
    else if (!basicNAVCounterEnable)
      navCounterHoldsQuiet <= 1'b0;

    // start probe delay
    if (moveToActive_p) 
      navCounterHoldsProbe <= 1'b1;
    // end probe delay
    else if (!basicNAVCounterEnable)
      navCounterHoldsProbe <= 1'b0;
  end
end


//------------------------------------------------------------------------------
// navOffset computation
//------------------------------------------------------------------------------
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    navOffset   <= 8'b0;
    navOffsetEn <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    navOffset   <= 8'b0;
    navOffsetEn <= 1'b0;
  end
  else
  begin
    
    if (loadQuietDuration_p                          ||
        moveToActive_p                               ||
        (basicNAVPsPollUpdate && ackDurationValid_p) ||
        (intraNAVPsPollUpdate && ackDurationValid_p) ||
        (basicNAVUpdate_p     && !psPollReceived   ) ||
        (intraNAVUpdate_p     && !psPollReceived   ))
    begin
      // Stop Offset counter when NAV is Updated
      navOffsetEn <= 1'b0;
      navOffset   <= 8'd0;
    end
    else if (macPhyIfRxEndForTiming_p)
    begin
      // Start Offset counter on reception of the rxEndForTiming signal
      navOffsetEn <= 1'b1;
      navOffset   <= 8'd0;
    end
    else if (navOffsetEn && tick1us_p && navOffset==8'hFF)
    begin
      // Stop Offset counter when NAV was not updated after end of frame and 
      // counter reach the maximum value.
      navOffsetEn <= 1'b0;
      navOffset   <= 8'd0;
    end
    else if (navOffsetEn && tick1us_p)
    begin
      // Increment the offset counter
      navOffset <= navOffset + 8'd1;
    end
  end
end


//------------------------------------------------------------------------------
// Save information about the last NAV update
//------------------------------------------------------------------------------
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    lastUpdateIsBasicRTS <= 1'b0;
    lastUpdateIsIntraRTS <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    lastUpdateIsBasicRTS <= 1'b0;
    lastUpdateIsIntraRTS <= 1'b0;
  end
  else
  begin
    if (basicNAVLoad_p)
    begin
       lastUpdateIsBasicRTS <= notMineRtsRcved | rtsRcvdSampled;
       lastUpdateIsIntraRTS <= 1'b0;
    end
    else if (intraNAVLoad_p)
    begin
       lastUpdateIsBasicRTS <= 1'b0;
       lastUpdateIsIntraRTS <= notMineRtsRcved | rtsRcvdSampled;
    end
  end
end


//------------------------------------------------------------------------------
// Debug Port assigment
//------------------------------------------------------------------------------
assign debugPortNAV = {channelBusy,
                       basicNAVClear_p  || intraNAVClear_p,
                       basicNAVUpdate_p || intraNAVUpdate_p,
                       macPhyIfRxEnd_p,
                       psPollReceived,
                       1'b0,//endForceInc,
                       notMineRtsRcved,
                       incorrectRcved_p,
                       correctRcved_p,
                       1'b0,//DEPRECATED//basicNAVLsigUpdate_p || intraNAVLsigUpdate_p,
                       macPhyIfRxEndForTiming_p,
                       timeout_p,
                       basicNAVCounterEnable || intraNAVCounterEnable,
                       1'b0,//navRestart_p,
                       stopRx_p,
                       1'b0};//macPhyIfRxErr_p};
                       
                        
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_ASSERT_ON


//$rw_sva Check check no idle while busy
property not_channelbusy_while_idle_prop;
@(posedge macCoreClk)
  disable iff(macCoreClkHardRst_n==0)
  channelBusy|->!channelIdle;
endproperty
not_channelbusy_while_idle: assert property (not_channelbusy_while_idle_prop); 

//$rw_sva Check check no busy while idle
property not_channelidle_while_busy_prop;
@(posedge macCoreClk)
  disable iff(macCoreClkHardRst_n==0)
  channelIdle|->!channelBusy;
endproperty
not_channelidle_while_busy: assert property (not_channelidle_while_busy_prop); 

//$rw_sva Check timeout_p is a pulse
property timeout_p_pulse_prop;
@(posedge macCoreClk)
  disable iff(macCoreClkHardRst_n==0)
  timeout_p|=>!timeout_p;
endproperty
timeout_p_pulse: assert property (timeout_p_pulse_prop); 

//$rw_sva Check channelToIdle_p is a pulse
property channelToIdle_p_pulse_prop;
@(posedge macCoreClk)
  disable iff(macCoreClkHardRst_n==0)
  channelToIdle_p|=>!channelToIdle_p;
endproperty
channelToIdle_p_pulse: assert property (channelToIdle_p_pulse_prop); 

//$rw_sva Check channelToIdle_p comes before channel idle
property channelToIdle_p_before_idle_prop;
@(posedge macCoreClk)
  disable iff(macCoreClkHardRst_n==0)
  $rose(channelToIdle_p) |=> channelIdle;
endproperty
channelToIdle_p_before_idle: assert property (channelToIdle_p_before_idle_prop); 


`endif // RW_ASSERT_ON

endmodule
