//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Synthesizable AHB spy trickbox (protocol checker)
//                    
// Simulation Notes : Define AHBSPY_ON in order to enable block.
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements : o two-cycles response to be implemented
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`ifdef RW_AHBSPY_ON

module ahbSpy #(parameter ahbSpyNumber = 0) (
  //$port_g system ports
  input wire         resetn,
  input wire         clk,
              
  //$port_g Spied AHB 
  input wire         hready_in,
  input wire  [8:0]  hsel,
  input wire  [31:0] haddr,
  input wire  [1:0]  htrans,
  input wire  [2:0]  hburst,
  input wire  [2:0]  hsize,
  input wire         hwrite,
  input wire  [1:0]  hresp,
  input wire         hready,
    
  //$port_g Violation detected
  output wire        hit
  );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

    wire [ 1 : 0] htrans_sel;
    
    reg [ 8 : 0] sel;
    reg [31 : 0] addr;
    reg [ 1 : 0] trans;
    reg [31 : 0] next_addr;
    reg [ 2 : 0] burst;
    reg [ 2 : 0] size;
    reg          write;
    
    reg [ 3 : 0] beat;
    reg          incr;
    
    reg [31:0] tmp_addr;
    reg [31:0] tmp_incr;
    
    //
    // hready_transition[0] is hready
    // hready_transition[1] is hready 1 hclk delayed
    //
    // Thus :
    //
    //   10 = is a transition from 1 to 0
    //   01 =    "        "   from 0 to 1
    //   00 = no transition, hready is still low
    //   11 = no transition, hready is still high
    //
    
    wire [ 1 : 0] hready_transition;
    reg           hready_transition1;
    
    //
    // error flags
    //
    
    reg  hit_hready;
    reg  hit_exceed;
    reg  hit_ahb;
    reg  hit_ahb_pulse;
    wire hit_i;
    
    // pragma translate_off
    // synopsys translate_off
    time statustime;
    // synopsys translate_on
    // pragma translate_on
       
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

      

  // //////////////////////////////////////////////////////////////////////////-
  //
  // fault detection (to trig a digtal analyzer)
  //
  // //////////////////////////////////////////////////////////////////////////-
  
  assign hit_i = hit_hready || hit_ahb || hit_exceed;
  assign hit   = hit_i;
  
  // //////////////////////////////////////////////////////////////////////////-
  //
  // htrans_sel is htrans except we consider it as IDLE if the slave is not
  // selected when incoming htrans=NONSEQ
  //
  // //////////////////////////////////////////////////////////////////////////-
  
  assign htrans_sel = (hsel == 9'b0) ? "00" : htrans;
  
  // //////////////////////////////////////////////////////////////////////////-
  // hready/hready_in check
  // //////////////////////////////////////////////////////////////////////////-

  always @(posedge clk or negedge resetn)
  begin
  
    if (!resetn)
    begin
    
      hready_transition1 <= 1'b1;
      hit_hready         <= 1'b0;
    end
    
    else
    begin
    
      hready_transition1 <= hready_transition[0];
      
      if (!hready_transition[0])
      begin
        
        if (hready_in != hready_transition[0])
        begin
        
          // pragma translate_off
          // synopsys translate_off
          statustime = $time;
          $display("ahbSpy %d ERROR: hready_in/hready discrepency at %t!", ahbSpyNumber, statustime);
          // synopsys translate_on
          // pragma translate_on
          hit_hready  <= 1'b1;
        
        end
      end
    end
  end
 
  assign hready_transition = {hready_transition1, hready}; // alias
  
  // //////////////////////////////////////////////////////////////////////////-
  //  check
  // //////////////////////////////////////////////////////////////////////////-

  task gen_hit;
  begin
    hit_ahb       <= 1'b1;
    hit_ahb_pulse <= 1'b1;                 
  end
  endtask

  task no_change_check;
  begin
  
    if (trans != htrans)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: illegal htrans transition when hready = 0 at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
  
    if (sel != hsel)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hsel has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
    
    if (burst != hburst)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hburst has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
    
    if (size != hsize)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hsize has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
  
    if (write != hwrite)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hwrite has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
    
    if (addr != haddr)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: haddr has a bad address within the burst at %t!", ahbSpyNumber, statustime);                
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
  end
  endtask
  
  task next_beat_check;
  begin
    
    if (sel != hsel)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hsel has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
    
    if (burst != hburst)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hburst has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
    
    if (size != hsize)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hsize has changed within the burst at %t!", ahbSpyNumber, statustime);
      // synopsys translate_on
      // pragma translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
  
    if (write != hwrite)
    begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: hwrite has changed within the burst at %t!", ahbSpyNumber, statustime);
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
    
   if (next_addr!=haddr) 
   begin
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d VIOLATION: haddr has a bad address within the burst at %t!", ahbSpyNumber, statustime);                
      // pragma translate_on
      // synopsys translate_on
      hit_ahb       <= 1'b1;                       
      hit_ahb_pulse <= 1'b1;                 
    end
  
    if (incr == 1'b0)
    begin
      if (beat == 4'b0) 
      begin
        // pragma translate_off
        // synopsys translate_off
        statustime = $time;
        $display("ahbSpy %d VIOLATION: the current burst length exceeds the length defined by hburst at %t!", ahbSpyNumber, statustime);
        // pragma translate_on
        // synopsys translate_on
        hit_exceed <= 1'b1;                 
      end
    end
  end
  endtask

  task control_sample;
  begin
    
    sel   <= hsel;                                                                                                   
    addr  <= haddr;                                                                                                  
    burst <= hburst;                                                                                                 
    size  <= hsize;                                                                                                  
    write <= hwrite;
    
    case (hburst)
    
      3'b000 :
      begin
        beat <= 4'b0000;
        incr <= 1'b0;
      end
      3'b001 :
      begin
        beat <= 4'b0000;
        incr <= 1'b1;
      end
      3'b010 :
      begin
        beat <= 4'b0011;
        incr <= 1'b0;
      end
      3'b011 : 
      begin
        beat <= 4'b0011;
        incr <= 1'b0;
      end
      3'b100 :
      begin
        beat <= 4'b0111;
        incr <= 1'b0;
      end
      3'b101 :
      begin
        beat <= 4'b0111;
        incr <= 1'b0;
      end
      3'b110 :
      begin
        beat <= 4'b1111;
        incr <= 1'b0;
      end
      3'b111 :
      begin
        beat <= 4'b1111;
        incr <= 1'b0;
      end
      default:
      begin
        
        // pragma translate_off
        // synopsys translate_off
        statustime = $time;
        $display("ahbSpy %d ERROR: hburst is undefined at %t!", ahbSpyNumber, statustime);
        // synopsys translate_on
        // pragma translate_on
        hit_ahb       <= 1'b1;
        hit_ahb_pulse <= 1'b1;
      end
    
    endcase
    
    case (hsize)
    
      3'b000 :
      begin
      end
        
      3'b001 :
      begin
      
        if (haddr[0] != 1'b0)
        begin
          // pragma translate_off
          // synopsys translate_off
          statustime = $time;
          $display("ahbSpy %d VIOLATION: haddr must be aligned according hsize at %t!", ahbSpyNumber, statustime);
          // pragma translate_on
          // synopsys translate_on
          hit_ahb       <= 1'b1;                 
          hit_ahb_pulse <= 1'b1;                 
        end
      end
     
      3'b010 :
      begin
        if (haddr[1:0] != 2'b00)
        begin
          // pragma translate_off
          // synopsys translate_off
          statustime = $time;
          $display("ahbSpy %d VIOLATION: haddr must be aligned according hsize at %t!", ahbSpyNumber, statustime);
          // pragma translate_on
          // synopsys translate_on
          hit_ahb       <= 1'b1;                 
          hit_ahb_pulse <= 1'b1;                 
        end
      end
       
      default:   
      begin
         
          // pragma translate_off
          // synopsys translate_off
          statustime = $time;
          $display("ahbSpy %d VIOLATION: hsize not supported at %t!", ahbSpyNumber, statustime);
          // pragma translate_on
          // synopsys translate_on
          hit_ahb       <= 1'b1;                 
          hit_ahb_pulse <= 1'b1;
      end
    endcase                
  end
  endtask
  
  task beat_update;
  begin
    
    addr <= next_addr;
    
    if (beat != 4'b0) 
      beat <= beat + 4'b1111;
  end
  endtask

      
  always @(posedge clk or negedge resetn)
  begin
  
    if (!resetn)
    begin
      sel        <= 'h0;               
      addr       <= 'h0;         
      trans      <= 'h0;              
      burst      <= 'h0;             
      size       <= 'h0;              
      write      <= 1'b0;                   
      
      beat       <= 'h0;                
      incr       <= 1'b0;                   
      
      hit_ahb       <= 1'b0;                   
      hit_ahb_pulse <= 1'b0;
      hit_exceed    <= 1'b0;
      
    end
    else
    begin
      
      trans         <= htrans_sel;
      hit_ahb_pulse <= 1'b0;
      
      case (hready_transition)
      
        // ////////////////////////////////////////////////////////-
        //
        // during a wait state, possible htrans_sel changes are :
        //
        //  trans   htrans_sel
        //
        //  IDLE -> NONSEQ
        //  BUSY -> IDLE
        //  BUSY -> NONSEQ
        //  BUSY -> SEQ
        //
        // ////////////////////////////////////////////////////////-
        
        2'b00, 2'b01 :
        begin
        
          case (trans)
          
            //
            // TRANS=IDLE
            //
            // only IDLE->NONSEQ allowed
            //
            
            2'b00 :
            begin
            
              case (htrans_sel)
               
                //
                // IDLE->IDLE
                //
                //
                // no check
                //
                
                2'b00 :
                begin
                end
                
                //
                // IDLE->BUSY
                //
                //
                // violation
                //
                  
                2'b01 :
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d VIOLATION: illegal htrans_sel transition from IDLE to BUSY at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
                
                //
                // IDLE->NONSEQ
                //
                //
                // a new access starts, control wires are sampled for monitoring
                //
               
                2'b10 :
                  control_sample;
                
                //
                // IDLE->SEQ
                //
                //
                // violation
                //
                
                2'b11 :
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d VIOLATION: illegal htrans_sel transition from IDLE to SEQ at %t!", ahbSpyNumber, statustime);
                  // pragma translate_on
                  // synopsys translate_on
                end
                
                //
                // IDLE->X
                //
                //
                // violation
                //
               
                default:   // undefined state
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // pragma translate_on
                  // synopsys translate_on
                end

              
              endcase
            end
           
            //
            // TRANS=BUSY
            //
            // only BUSY->IDLE, BUSY->NONSEQ, BUSY->SEQ transitions allowed
            //
           
            2'b01 :
            begin
           
              case (htrans_sel)
              
                //
                // BUSY->IDLE
                //
                // no check
                //
                
                2'b00 :
                begin
                end
                
                //
                // BUSY->BUSY
                //
                // control wire must not change
                //
                
                2'b01 :
                  next_beat_check;
                
                //
                // BUSY->NONSEQ
                //
                // previous burst is terminated, new access starts (no check required vs previous access) 
                // control wires are sampled for monitoring
                //
               
                2'b10 :
                  control_sample;
                
                //
                // BUSY->SEQ
                //
                // the current burst resumes, control wire must not change
                // 
                //
                
                2'b11 :
                begin
                  next_beat_check;
                  if (hready_transition == 2'b01)
                    beat_update;
                end
               
                //
                // BUSY->X
                //
                // violation
                // 
                //
                
                default :   
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
              endcase
            end
            
            //
            // TRANS=NONSEQ
            //
            //
            // control must not change
            //
            
            2'b10 :
            begin
              if (hresp != 2'b0 && hready_transition == 2'b01)  // do not perform any check during a 2-cycle response
              begin
              end
              else
                no_change_check;
              control_sample;
            end
          
            //
            // TRANS=SEQ
            //
            //
            // control must not change
            //
            
            2'b11 :
            begin
              if (hresp != 2'b0 && hready_transition == 2'b01)  // do not perform any check during a 2-cycle response
              begin
              end
              else
              begin
                if (htrans_sel != 2'b11)
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: illegal htrans_sel transition when hready = 0 at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
                else
                begin
                  next_beat_check;
                  if (hready_transition == 2'b01)
                    beat_update;
                end
              end
            end
            
            default :  // undefined state
            begin
              gen_hit;
              // pragma translate_off
              // synopsys translate_off
              statustime = $time;
              $display("ahbSpy %d ERROR: undefined state of trans at %t!", ahbSpyNumber, statustime);
              // pragma translate_on
              // synopsys translate_on
            end
          
          endcase
        end
        
        // ////////////////////////////////////////////////////////-
        //
        // entering wait state, all htrans_sel changes are possible
        // except :
        //
        //  trans   htrans_sel
        //
        //  IDLE -> all htrans_sel states
        //  BUSY -> all htrans_sel states
        //
        // a wait state must only happen when htrans_sel=NONSEQ or SEQ
        //
        // ////////////////////////////////////////////////////////-
        
        2'b10 :
        begin
          case (trans)
          
            //
            // TRANS=IDLE
            //
            
            2'b0 :
            begin
            
              gen_hit;
              // pragma translate_off
              // synopsys translate_off
              statustime = $time;
              $display("ahbSpy %d VIOLATION: hready must not be un-asserted when htrans_sel=IDLE is sampled at %t!", ahbSpyNumber, statustime);
              // synopsys translate_on
              // pragma translate_on
            end
              
            //
            // TRANS=BUSY
            //
          
            2'b01 :
            begin
              gen_hit;
              // pragma translate_off
              // synopsys translate_off
              statustime = $time;
              $display("ahbSpy %d VIOLATION: hready must not be un-asserted when htrans_sel=BUSY is sampled at %t!", ahbSpyNumber, statustime);
              // pragma translate_on
              // synopsys translate_on
            end
            
            //
            // TRANS=NONSEQ
            //
            
            2'b10 :
            begin
            
              case (htrans_sel)
              
                //
                // NONSEQ->IDLE
                //
                // no check
                //
                
                2'b0 :
              begin
              end
                
                //
                // NONSEQ->BUSY
                //
                // the control must reflects the next beat
                //
                
                2'b01 :
                  next_beat_check;
                
                //
                // NONSEQ->NONSEQ
                //
                // the previous access is terminated, a new one starts,
                // control wires are sampled for monitoring
                //
               
                2'b10 :
                  control_sample;
               
                //
                // NONSEQ->SEQ
                //
                // the control must reflects the next beat
                //
                
                2'b11 :
                  next_beat_check;
                
                //
                // NONSEQ->X
                //
                // violation
                //
                
                default:
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);                  
                  // pragma translate_on
                  // synopsys translate_on
                end
             
              endcase
            end
            
            //
            // TRANS=SEQ
            //
            
            2'b11 :
            begin
            
              case (htrans_sel)
              
                //
                // SEQ->IDLE
                //
                // no check
                //
                
                2'b0 :
              begin
              end
                
                //
                // SEQ->BUSY
                //
                // the control must reflects the next beat
                //
                
                2'b01 :
                  next_beat_check;
                
                //
                // SEQ->NONSEQ
                //
                // the access is terminated, a new one starts,
                // control wires are sampled for monitoring
                //
               
                2'b10 :
                  control_sample;
                
                //
                // SEQ->SEQ
                //
                // the control must reflects the next beat
                //
                
                2'b11 :
                  next_beat_check;
                
                //
                // SEQ->X
                //
                // violation
                //
              
                default:
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
             
              endcase
            end
            
            //
            // TRANS=undefined state
            //
           
            default :
            begin
              gen_hit;
              // pragma translate_off
              // synopsys translate_off
              statustime = $time;
              $display("ahbSpy %d ERROR: undefined state of trans at %t!", ahbSpyNumber, statustime);                  
              // synopsys translate_on
              // pragma translate_on
            end
          
          endcase
        end
        
        // ////////////////////////////////////////////////////////-
        //
        // no wait state, all htrans_sel changes are possible
        // execpt IDLE->BUSY, IDLE->SEQ
        //
        // ////////////////////////////////////////////////////////-
       
        2'b11 :
        begin
        
          case (trans)
          
            //
            // TRANS=IDLE            
            //
            
            2'b0 :
            begin
            
              case (htrans_sel)
              
                //
                // IDLE->IDLE
                //
                // no check
                //
                
                2'b0 :
                begin
                end
                
                //
                // IDLE->BUSY
                //
                // violation
                //
                  
                2'b01 :
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d VIOLATION: illegal htrans_sel transition from IDLE to BUSY at %t!", ahbSpyNumber, statustime);
                  // pragma translate_on
                  // synopsys translate_on
                end
                
                //
                // IDLE->NONSEQ
                //
                // control wires are sampled for monitoring
                //
              
                2'b10 :
                  control_sample;
                  
                //
                // IDLE->SEQ
                //
                // violation
                //
                
                2'b11 :
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d VIOLATION: illegal htrans_sel transition from IDLE to SEQ at %t!", ahbSpyNumber, statustime);
                  // pragma translate_on
                  // synopsys translate_on
                end
             
                default :  // undefined state
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
              
              endcase             
            end
            
            //
            // TRANS=BUSY            
            //
            
            2'b01 :
            begin
            
              case (htrans_sel)
              
                //
                // BUSY->IDLE
                //
                // access is terminated, no check
                //
                
                2'b0 :
                begin
                end
                
                //
                // BUSY->BUSY
                //
                // control wire must not change
                //
                
                2'b01 :
                  next_beat_check;
                  
                //
                // BUSY->NONSEQ
                //
                // access is terminated, a new one starts,
                // control data are sampled for monitoring
                //
                
                2'b10 :
                  control_sample;
                  
                //
                // BUSY->SEQ
                //
                // control must reflect the next beat
                //
                
                2'b11 :
                begin
                  next_beat_check;
                  beat_update;
                end
                  
                default :  // undefined state
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end

              endcase
            end
            
            //
            // TRANS=NONSEQ            
            //
            
            2'b10 :
            begin
              
              case (htrans_sel)
              
                //
                // NONSEQ->IDLE
                //
                // the access is terminated, no check
                //
                
                2'b0 :
                begin
                end
                
                //
                // NONSEQ->BUSY
                //
                // control must reflect the next beat 
                //
                
                2'b01 :
                  next_beat_check;
                  
                //
                // NONSEQ->NONSEQ
                //
                // access is terminated, a new one start,
                // control wires are sampled for monitoring
                //
                
                2'b10 :
                  control_sample;
                  
                //
                // NONSEQ->SEQ
                //
                // control must reflect the next beat 
                //
                
                2'b11 :
                begin
                  next_beat_check;
                  beat_update;
                end
                  
                default :  // undefined state
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
                
              endcase
            end
         
            //
            // TRANS=SEQ            
            //
            
            2'b11 :
            begin
            
              case (htrans_sel)
                
                //
                // SEQ->IDLE
                //
                // burst is terminated, no check
                //
              
                2'b0 :
                begin
                end
                
                //
                // SEQ->BUSY
                //
                // control must reflect the next beat 
                //
               
                2'b01 :
                  next_beat_check;
                
                //
                // SEQ->NONSEQ
                //
                // access is terminated, a new one start,
                // control wires are sampled for monitoring
                //
               
                2'b10 :
                  control_sample;
                
                //
                // SEQ->SEQ
                //
                // control must reflect the next beat 
                //
                
                2'b11 :
                begin
                  next_beat_check;
                  beat_update;
                end
                
                default:
                begin
                  gen_hit;
                  // pragma translate_off
                  // synopsys translate_off
                  statustime = $time;
                  $display("ahbSpy %d ERROR: undefined state of htrans_sel at %t!", ahbSpyNumber, statustime);
                  // synopsys translate_on
                  // pragma translate_on
                end
              
              endcase
            end
            
            default:
            begin
              gen_hit;
              // pragma translate_off
              // synopsys translate_off
              statustime = $time;
              $display("ahbSpy %d ERROR: undefined state of trans at %t!", ahbSpyNumber, statustime);                  
              // synopsys translate_on
              // pragma translate_on
            end
            
          endcase
        end
        
        default:
        begin
          gen_hit;
          // pragma translate_off
          // synopsys translate_off
          statustime = $time;
          $display("ahbSpy %d ERROR: undefined state of hready_transition at %t!", ahbSpyNumber, statustime);
          // synopsys translate_on
          // pragma translate_on
        end
              
      endcase 
      
    
    
    // Detects if a violation occures when crossing 1 KB boundary
    if (haddr[9:0] == 10'b0 && ((htrans_sel == 2'b11) || (htrans_sel == 2'b01))) 
    begin
      gen_hit;
      // pragma translate_off
      // synopsys translate_off
      statustime = $time;
      $display("ahbSpy %d ERROR: violation crossing a 1KB boundary at %t!", ahbSpyNumber, statustime);
      // synopsys translate_on
      // pragma translate_on
    end
  end
  end
    
 
  always @ *
  begin
  
    case (size)

      3'b000  : tmp_incr = 32'b00000000000000000000000000000001;
      3'b001  : tmp_incr = 32'b00000000000000000000000000000010;
      default : tmp_incr = 32'b00000000000000000000000000000100;

    endcase

    tmp_addr = addr + tmp_incr;

    case (burst)

      3'b010 : // wrap4
      begin
        case (size)

          3'b0    : next_addr <= {addr[31:2], tmp_addr[1:0]};
          3'b001  : next_addr <= {addr[31:3], tmp_addr[2:0]};
          default : next_addr <= {addr[31:4], tmp_addr[3:0]};

        endcase
      end

      3'b100 : // wrap8
      begin      
        case (size)

          3'b0    : next_addr <= {addr[31:3], tmp_addr[2:0]};
          3'b001  : next_addr <= {addr[31:4], tmp_addr[3:0]};
          default : next_addr <= {addr[31:5], tmp_addr[4:0]};

        endcase
      end

      3'b110 : // wrap16
      begin
        case (size)

          3'b0    : next_addr <= {addr[31:4], tmp_addr[3:0]};
          3'b001  : next_addr <= {addr[31:5], tmp_addr[4:0]};
          default : next_addr <= {addr[31:6], tmp_addr[5:0]};

        endcase
      end
                                                                                                                            
      default : // all incrementing burst types
                                                                                                                            
        next_addr <= tmp_addr;                                                                                                   
  
    endcase
  end


endmodule

`endif // RW_AHBSPY_ON

////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
