//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : FSM of Decryption procedure
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewer.
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module decryptFsm( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire         macCoreRxClk,            // MAC Core Receive Clock
            input wire         macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain 
                                                        // active low
            input wire         macCoreClkSoftRst_n,     // Soft Reset of the MAC Core Clock domain
                                                        // active low
            ///////////////////////////////////////////////
            //$port_g A-MPDU Deaggregator
            ///////////////////////////////////////////////
            input wire [ 7:0]  rxData,                  // Rx data read from MAC-PHY interface FIFO
            input wire         rxDataValid,             // Rx data is valid
            output wire        decryptCntrlReady,       // Indicate that the Crypto is ready to receive data
            
            ///////////////////////////////////////////////
            //$port_g Encryption Engine
            ///////////////////////////////////////////////
            output wire        rxError_p,               // Indicates received frame can be discarded
            output wire        initEncrRxCntrl_p,       // Starting decryption sequence
            output reg         rxCryptoKeyValid,        // Validating Key
            
            output wire        encryptedDataInValid,    // Write pulse to Encryption buffer
            output wire [7:0]  encryptedDataIn,         // Write data to Encryption buffer
            input wire         encryptedDataInReady,    // Encryption Buffer full flag
            output  wire       encryptedDataInEnd_p,    // Signal indicating last byte of received
                                                        // frame fed to MAC used by encryption
                                                        // receive controller 
            
            input wire         plainDataOutEnd_p,       // Encryption Buffer end
            output wire        plainDataOutReady,       // Flow control for receiving decrypted data

            ///////////////////////////////////////////////
            //$port_g Key Search Engine
            ///////////////////////////////////////////////
            input wire         keyStorageValid_p,       // Key Storage Index Valid
            input wire         keyStorageError_p,       // Error indicating MAC address not found
            
            input wire [1:0]   sppKSR,                  // SPP from RAM
            input wire [2:0]   cTypeKSR,                // Cipher Type from RAM
            input wire [3:0]   vlanIDKSR,               // Virtual LAN ID from RAM
            input wire         useDefKeyKSR,            // Use Default Key from RAM
            
            output reg         indexSearchTrig_p,       // Trigger for searching index from MAC address
            output reg         rxKeySearchIndexTrig_p,  // Trigger for searching parameter from index
            output wire  [9:0] rxKeySearchIndex,        // RAM Index to be searched
            
            ///////////////////////////////////////////////
            //$port_g RX Controller FSM
            ///////////////////////////////////////////////
            input wire         rxControlIdle,           // FSM in idle
            input wire         rxControlToError_p,      // RX FSM goes to error
            input wire         rxControlToFrmBody_p,    // RX FSM goes to frame body
            input wire         acceptProtected,         // Accept protected frame -> trig Encryption Engine
            input wire         rxFIFOOverFlowError,     // RX FIFO overflow
            input wire         rxInitVectorEnd_p,       // FSM IV end pulse
            input wire         rxExtIV_p,               // Indicate if there is a extended IV
            input wire         decrStatusValid_p,       // Decryption status valid pulse            
            output reg         rxIndexSearchDone,       // Key Search Done Flag
            output reg         rxDefaultKeySearchDone,  // Default Key Search Done Flag
            output reg         decryptionCheckOk,       // Indicate that the decryption Check is ok
            
            input wire         searchMACAddr_p,            // Addr Received        
            input wire         payloadEnd_p,            // Last Byte of Frame Body (without encryption tailer)
            
            ///////////////////////////////////////////////
            //$port_g Frame decoder
            ///////////////////////////////////////////////
            input wire         addr1Match,              // ADDR1 match
            input wire         bcMcRcved,               // Broadcast/Multicast
            input wire  [1:0]  rxDefKeyID,              // Default Key ID
            output reg         keyStorageValid,         // ADDR2 has been found in KSR

            ///////////////////////////////////////////////
            //$port_g RX FIFO Interface Controller
            ///////////////////////////////////////////////
            output reg         rxDataSelect,            // Select data from Encryption Engine
            output wire        keyIndexCapture_p,       // Key Index returned by keySearch capture pulse

            output wire [15:0] debugPortDecryptFsm      // Port for debug purpose
            );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// decryption FSM states definition
//$fsm_sd decryptCs
parameter 
                       IDLE  =  3'd0,
        KSR_TRIG_WITH_INDEX  =  3'd1,
             KSR_INDEX_WAIT  =  3'd2,
         KSR_TRIG_WITH_ADDR  =  3'd3,
              KSR_ADDR_WAIT  =  3'd4,
                  ENCR_TRIG  =  3'd5,
                    DECRYPT  =  3'd6;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// decryption FSM signals definition
reg [2:0] decryptCs;  // decryption FSM Current State
reg [2:0] decryptNs;  // decryption FSM Next State

// Ready to trig encryption
reg       rxProtHdrValid;
reg       rxExtIVValid;

// Flag the address search error
reg       ksrAddrError;


// RX Controller pulse delayed by one cc
reg       rxControlIdle_ff1;
reg       rxControlToError_ff1_p;

`ifdef RW_SIMU_ON
// String definition to display decrypt current state
reg [19*8:0] decryptCs_str;
`endif // RW_SIMU_ON


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////


assign debugPortDecryptFsm =  {indexSearchTrig_p,
                               rxKeySearchIndexTrig_p,
                               keyStorageError_p,
                               keyStorageValid_p,
                               decryptionCheckOk,
                               rxError_p,
                               keyIndexCapture_p,
                               rxControlIdle,
                               decryptCs[2:0],
                               bcMcRcved,
                               searchMACAddr_p,
                               1'b0,
                               rxInitVectorEnd_p,
                               rxIndexSearchDone};


// decryption FSM Current State Logic 
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    decryptCs <= IDLE;
  else if(macCoreClkSoftRst_n == 1'b0)
    decryptCs <= IDLE;
  else
    decryptCs <= decryptNs;
end


// decryption FSM Next State Logic.
always @* 
begin

  if (((decryptCs != IDLE) && rxControlToError_p) || rxControlIdle)
    //$fsm_t When rxError_p or rxControlIdle are true, then fsm goes to IDLE state
    decryptNs = IDLE; 
  
  else
  begin
    case(decryptCs)
  
      IDLE:
        //$fsm_s In IDLE state, the state machine waits encrypted frame
        if (searchMACAddr_p)
          //$fsm_t When the frame is received is potentially for this device (searchMACAddr_p), 
          //the state machine goes to KSR_TRIG_WITH_ADDR state
          decryptNs = KSR_TRIG_WITH_ADDR;
        else if(rxInitVectorEnd_p && (useDefKeyKSR || bcMcRcved))
          decryptNs = KSR_TRIG_WITH_INDEX;
        else if(rxProtHdrValid && decryptionCheckOk)
          decryptNs = ENCR_TRIG;
        else
          //$fsm_t While no frame is received, the state machine stays in IDLE state
          decryptNs = IDLE;
  
      KSR_TRIG_WITH_ADDR:
        //$fsm_s In KSR_TRIG_WITH_ADDR state, the state machine goes to KSR_ADDR_WAIT state
  
        //$fsm_t After one clock cycle, the state machine goes to KSR_ADDR_WAIT state
        decryptNs = KSR_ADDR_WAIT; 

      KSR_ADDR_WAIT:
        if (keyStorageValid_p || keyStorageError_p)
          //$fsm_t When keyStorageValid_p is received, or no address found and the frame is control type, then the state machine goes back to IDLE state
          decryptNs = IDLE;
        else
          //$fsm_t While keyStorageValid_p or keyStorageError_p are not received, the state machine stays in KSR_ADDR_WAIT state
          decryptNs = KSR_ADDR_WAIT;
  
      KSR_TRIG_WITH_INDEX:
        //$fsm_s In KSR_TRIG_WITH_INDEX state, the state machine wait for the end of IV fetching

        //$fsm_t One clock cycle after rxInitVectorEnd_p, the state machine goes to KSR_INDEX_WAIT state
          decryptNs = KSR_INDEX_WAIT;
      
      KSR_INDEX_WAIT:
        //$fsm_s In KSR_INDEX_WAIT state, the state machine waits pulse from KSR
        if (keyStorageValid_p || keyStorageError_p) 
          //$fsm_t When keyStorageValid_p is received, the state machine goes back to IDLE state
          decryptNs = IDLE; 
        else
          //$fsm_t While keyStorageValid_p is not received, the state machine stays in KSR_INDEX_WAIT state
          decryptNs = KSR_INDEX_WAIT;
  
  
      ENCR_TRIG:
        //$fsm_s In ENCR_TRIG state, the state machine goes to IDLE state
  
        //$fsm_t After one clock cycle, the state machine goes to IDLE state
        decryptNs = DECRYPT; 
  
      DECRYPT:
        //$fsm_s In DECRYPT state, the state machine waits for the completion of the decryption
  
        if (plainDataOutEnd_p) 
          //$fsm_t When plainDataOutEnd_p is received, the state machine goes back to IDLE state
          decryptNs = IDLE; 
        else
          //$fsm_t While plainDataOutEnd_p is not received, the state machine stays in DECRYPT state
          decryptNs = DECRYPT; 
  
      // Disable coverage on the default state because it cannot be reached.
      // pragma coverage block = off 
       default:   
          decryptNs = IDLE; 
      // pragma coverage block = on
    endcase
  end

end

// generation of indexSearchTrig_p
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)                             // Asynchronous Reset
    indexSearchTrig_p <= 1'b0;
  else if (decryptCs == KSR_TRIG_WITH_ADDR)
    indexSearchTrig_p <= 1'b1;
  else
    indexSearchTrig_p <= 1'b0;
end

// generation of rxKeySearchIndexTrig_p
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)                             // Asynchronous Reset
    rxKeySearchIndexTrig_p <= 1'b0;
  else if (decryptCs == KSR_TRIG_WITH_INDEX)
    rxKeySearchIndexTrig_p <= 1'b1;
  else
    rxKeySearchIndexTrig_p <= 1'b0;
end

assign rxKeySearchIndex       = {4'b0,vlanIDKSR[3:0],rxDefKeyID};

// Mux between data from deaggregator or encryption Engine
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)                             // Asynchronous Reset
    rxDataSelect <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || 
           (rxControlIdle == 1'b1)       || 
           plainDataOutEnd_p             ||
           (rxFIFOOverFlowError == 1'b1)) // Synchronous Reset
    rxDataSelect <= 1'b0;
  else if (decryptionCheckOk && rxControlToFrmBody_p)
    rxDataSelect <= 1'b1;
end


assign keyIndexCapture_p = (decryptCs == KSR_ADDR_WAIT) && (keyStorageValid_p || keyStorageError_p);

// By default the flag is valid
// When start searching it goes in invalid until keySearchValid ot keySearchError
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    rxIndexSearchDone <= 1'b1;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1)) // Synchronous Reset
    rxIndexSearchDone <= 1'b1;
  else if (searchMACAddr_p)
    rxIndexSearchDone <= 1'b0;
  else if (keyIndexCapture_p)
    rxIndexSearchDone <= 1'b1;
end

// By default the flag is valid
// When start searching it goes in invalid until keySearchValid ot keySearchError
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    rxDefaultKeySearchDone <= 1'b1;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1)) // Synchronous Reset
    rxDefaultKeySearchDone <= 1'b1;
  else if (rxInitVectorEnd_p && (useDefKeyKSR || bcMcRcved))
    rxDefaultKeySearchDone <= 1'b0;
  else if ((decryptCs == KSR_INDEX_WAIT) && (keyStorageValid_p || keyStorageError_p))
    rxDefaultKeySearchDone <= 1'b1;
end

// Trigger to Encryption Engine
assign rxError_p          = rxDataSelect && rxControlToError_ff1_p;
assign initEncrRxCntrl_p  = (decryptCs == ENCR_TRIG);



always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    decryptionCheckOk      <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1) || rxKeySearchIndexTrig_p || searchMACAddr_p || 
          `ifndef RW_WAPI_EN
           (cTypeKSR == 3'd4) ||
          `endif
           ( rxExtIVValid && (cTypeKSR < 3'd2)) ||
           (!rxExtIVValid && (cTypeKSR >= 3'd2)))
    decryptionCheckOk      <= 1'b0;
  else if (acceptProtected && keyStorageValid && (addr1Match || bcMcRcved) && (cTypeKSR != 3'd0))
    decryptionCheckOk      <= 1'b1;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxControlIdle_ff1      <= 1'b1;
    rxControlToError_ff1_p <= 1'b1;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    rxControlIdle_ff1      <= 1'b1;
    rxControlToError_ff1_p <= 1'b1;
  end
  else
  begin
    rxControlIdle_ff1      <= rxControlIdle;
    rxControlToError_ff1_p <= rxControlToError_p;
  end
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    rxProtHdrValid <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1) || (decryptCs == ENCR_TRIG) ) // Synchronous Reset
    rxProtHdrValid <= 1'b0;
  else if (rxControlToFrmBody_p)
    rxProtHdrValid <= 1'b1;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    keyStorageValid <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1) || rxKeySearchIndexTrig_p || searchMACAddr_p) // Synchronous Reset
    keyStorageValid <= 1'b0;
  else if (keyStorageValid_p)
    keyStorageValid <= 1'b1;
end


// Null Key information to Encryption Engine
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    rxCryptoKeyValid <= 1'b0;
  
  else if((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1) || decrStatusValid_p) // Synchronous Reset
    rxCryptoKeyValid <= 1'b0;
  else if (decryptNs == ENCR_TRIG)
    rxCryptoKeyValid <= 1'b1;
end


reg encryptedDataInValidCapt;
reg encryptedDataInEndCapt;
reg [7:0] rxDataCapt;

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
  begin
    encryptedDataInValidCapt <= 1'b0;
    encryptedDataInEndCapt   <= 1'b0;
  end
  else if((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1)) // Synchronous Reset
  begin
    encryptedDataInValidCapt <= 1'b0;
    encryptedDataInEndCapt   <= 1'b0;
  end
  else if (!encryptedDataInReady && rxDataValid && rxDataSelect)
  begin
    encryptedDataInValidCapt <= 1'b1;
    encryptedDataInEndCapt   <= payloadEnd_p;
  end
  else if (encryptedDataInReady)
  begin
    encryptedDataInValidCapt <= 1'b0;
    encryptedDataInEndCapt   <= 1'b0;
  end
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    rxDataCapt               <= 8'b0;
  else if((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1)) // Synchronous Reset
    rxDataCapt               <= 8'b0;
  else if (rxDataValid)
    rxDataCapt               <= rxData;
end



// RX Encryption FIFO data
assign encryptedDataInValid = (rxDataValid && rxDataSelect) | encryptedDataInValidCapt;
assign encryptedDataInEnd_p = (rxDataValid && rxDataSelect && payloadEnd_p) | encryptedDataInEndCapt;
assign encryptedDataIn      = (rxDataSelect) ? (encryptedDataInValidCapt) ? rxDataCapt : rxData : 8'b0;

assign decryptCntrlReady    = (rxDataSelect | rxControlToFrmBody_p) ? encryptedDataInReady & (decryptCs == DECRYPT) : 1'b1;


// RX FIFO Interface Controller -> Ready to receive data from Encryption Engine
assign plainDataOutReady        = rxDataSelect;

// Flag the keySearchEngine error on Address Search
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    ksrAddrError <= 1'b0;
  else if((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1)) // Synchronous Reset
    ksrAddrError <= 1'b0;
  else if ((decryptCs == KSR_ADDR_WAIT) && (keyStorageError_p))
    ksrAddrError <= 1'b1;
end

// Flag the rxExtIV_p from rxControllerFSM
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)      // Asynchronous Reset
    rxExtIVValid <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1)) // Synchronous Reset
    rxExtIVValid <= 1'b0;
  else if (rxExtIV_p)
    rxExtIVValid <= 1'b1;
end


///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// decryptCs FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (decryptCs)
                         IDLE  :  decryptCs_str = {"IDLE"};
          KSR_TRIG_WITH_INDEX  :  decryptCs_str = {"KSR_TRIG_WITH_INDEX"};
               KSR_INDEX_WAIT  :  decryptCs_str = {"KSR_INDEX_WAIT"};
           KSR_TRIG_WITH_ADDR  :  decryptCs_str = {"KSR_TRIG_WITH_ADDR"};
                KSR_ADDR_WAIT  :  decryptCs_str = {"KSR_ADDR_WAIT"};
                    ENCR_TRIG  :  decryptCs_str = {"ENCR_TRIG"};
                      DECRYPT  :  decryptCs_str = {"DECRYPT"};
                      default  :  decryptCs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON

endmodule
