//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Frame decoder of rxController module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewer.
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module frameDecoder( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire         macCoreRxClk,                 // MAC Core Receive Clock
            input wire         macCoreClkHardRst_n,          // Hard Reset of the MAC Core Clock domain 
                                                             // active low
            input wire         macCoreClkSoftRst_n,          // Soft Reset of the MAC Core Clock domain
                                                             // active low
            ///////////////////////////////////////////////
            //$port_g MAC Controller
            ///////////////////////////////////////////////
            input wire         activeRx,                     // Current state Master is active Rx and Rx FSM is not IDLE
            input wire         rxCts,                        // Force receive only CTS after RTS Tx
                                                             
            output wire        triggerRcved_p,               // Trigger received pulse
            output wire        trsRcved_p,                   // Frame with TRS Control field
                                                             // received pulse
            output wire        bfrPollRcved_p,               // BFR-Poll received pulse
`ifdef RW_BFMER_EN                                           
            output wire        bfrRcved_p,                   // BF Report received pulse
`endif // RW_BFMER_EN                                        
            output wire        ndpaRcved_p,                  // NDP-A received pulse
            output wire        ackRcved_p,                   // ACK received pulse
            output wire        rtsRcved_p,                   // RTS received pulse
            output wire        ctsRcved_p,                   // CTS received pulse
            output wire        cfEndRcved_p,                 // CF-End received pulse
            output wire        baRcved_p,                    // BA received pulse
            output wire        barRcved_p,                   // BAR received pulse
            output wire        needAckRcved_p,               // Data or management received with ACK needed pulse
            output wire        needHTPAckRcved_p,            // Received frame that need an HTP Ack 
            output wire        bcMcRcved_p,                  // Broadcast/Multicast received pulse
            output wire        bcnRcved_p,                   // Beacon received pulse
            output wire        probRespRcved_p,              // Probe response received pulse
            output wire        notMineRcved_p,               // Frame received but not mine pulse
            output wire        unknownRcved_p,               // Unknown frame received pulse
            output wire        notMinePsPollRcved_p,         // PS-Poll received but not mine pulse
            output wire        notMineRtsRcved_p,            // RTS received but not mine pulse
            output wire        notMineMURtsRcved_p,          // MU-RTS received but not mine pulse
                                                             
            output reg [47:0]  rxAddr2,                      // Addr2
                                                             
            output wire        notMineRcved,                 // Frame received but not mine

            ///////////////////////////////////////////////
            //$port_g NAV
            ///////////////////////////////////////////////
            output reg [15:0]  rxFrameDuration,              // Frame duration

            ///////////////////////////////////////////////
            //$port_g MAC Timer unit
            ///////////////////////////////////////////////
            output reg [7:0]   dtimCnt,                      // DTIM counter
            output reg [63:0]  timeStamp,                    // TSF value
            output wire        olbcUpdate_p,                 // Update OLBC counter

            ///////////////////////////////////////////////
            //$port_g Encryption Engine
            ///////////////////////////////////////////////
            output reg [47:0]  rxAddr3,                      // Address 3 field
            output reg [47:0]  rxAddr4,                      // Address 4 field
            output reg [15:0]  rxQoSControl,                 // QoS control field
`ifdef RW_WAPI_EN
            output reg [7:0]   rxWAPIKeyIndex,               // WAPI key index field
            output reg [127:0] rxWAPIPN,                     // WAPI packet number(PN) field
`endif //RW_WAPI_EN                                          
            output reg  [47:0] rxAddr1,                      // Addr1
            output wire [23:0] rxInitVector,                 // IV
            output reg  [31:0] rxExtInitVector,              // Extended IV

            ///////////////////////////////////////////////
            //$port_g Decryption FSM
            ///////////////////////////////////////////////
            output wire  [1:0] rxDefKeyID,                   // Default Key ID
            input  wire        keyStorageValid,              // ADDR2 has been found in KSR

            ///////////////////////////////////////////////
            //$port_g RX Controller FSM
            ///////////////////////////////////////////////
            input wire         rxFrmCtrlEn,                  // Frame control field
            input wire         rxCarFrmCtrlEn,               // Carried Frame control field
            input wire         rxDurationIdEn,               // Duration / ID field
            input wire         rxAddr1En,                    // ADDR1 field
            input wire         rxAddr2En,                    // ADDR2 field
            input wire         rxAddr3En,                    // ADDR3 field
            input wire         rxAddr4En,                    // ADDR4 field
`ifdef RW_WAPI_EN                                            
            input wire         rxWAPIKeyIdxEn,               // WAPI KeyIdx field
            input wire         rxWAPIPNEn,                   // WAPI PN field
`endif //RW_WAPI_EN                                          
            input wire         rxQoSCtrlEn,                  // QoS control field
            input wire         rxHTCtrlEn,                   // HT control field
            input wire         rxSeqCtrlEn,                  // Sequence control field
            input wire         rxBACtrlEn,                   // BA control field
            input wire         rxBASeqCtrlEn,                // BA Sequence control field
            input wire         rxInitVectorEn,               // IV field
            input wire         rxExtInitVectorEn,            // Extended IV field
                                                             
            input wire         rxDtimCntEn,                  // DTIM counter field
            input wire         rxDtimPeriodEn,               // DTIM period field
            input wire         rxTimeStampEn,                // Timestamp field
                                                             
            input wire         rxQuietCount1En,              // Quiet Count 1 field
            input wire         rxQuietPeriod1En,             // Quiet Period 1 field
            input wire         rxQuietDuration1En,           // Quiet Duration 1 field
            input wire         rxQuietOffset1En,             // Quiet Offset 1 field
                                                             
            input wire         rxFrmBodyByte1En,             // Fame Body Byte 1 field
            input wire         rxFrmBodyByte2En,             // Fame Body Byte 2 field
`ifdef RW_BFMER_EN                                           
            input wire         rxFrmBodyByte2End_p,          // Fame Body Byte 2 field end pulse
`endif                                                       
                                                             
`ifdef RW_BFMEE_EN                                           
            input wire         rxBFRPollSegRetEn,            // BFR-Poll Feedback Segment Retransmission Bitmap Field
            input wire         rxNDPASndDialTokenEn,         // NDP-A Sounding Dialog Token field
            input wire         rxNDPASTAInfo1En,             // NDP-A STA Info 1 field
            input wire         rxNDPASTAInfoNEn,             // NDP-A STA Info N field
`endif //RW_BFMEE_EN                                         
            input wire         rxTriggerCMNInfoEn,           // Trigger Common Info field
            input wire         rxTriggerUSRInfoEn,           // Trigger User Info field
            input wire         rxTriggerDepUSRInfoEn,        // Trigger Dependent User Info field
                                                             
            input wire [7:0]   stateByteCnt,                 // Byte counter by states
            input wire         fcsCheck_p,                   // FSM FCS check
            input wire         rxDurationIdEnd_p,            // FSM Duration / ID end pulse
            input wire         rxAddr1End_p,                 // FSM ADDR1 end pulse
            input wire         rxAddr2End_p,                 // FSM ADDR2 end pulse
            input wire         carFrameCtrlEnd_p,            // FSM Carried frame control end pulse
            input wire         rxAddr3End_p,                 // FSM ADDR3 end pulse
            input wire         rxSeqCtrlEnd_p,               // FSM Sequence Control end pulse
            input wire         rxQoSEnd_p,                   // FSM QoS end pulse
            input wire         rxHTCtrlEnd_p,                // FSM HT Control end pulse
            input wire         rxBACtrlEnd_p,                // FSM BA Control end pulse
            input wire         rxBARInfoEnd_p,               // FSM BAR Info end pulse
`ifdef RW_BFMEE_EN                                           
            input wire         rxNDPAStaInfoEnd_p,           // FSM NDPA STA Info end pulse
`endif //RW_BFMEE_EN                                         
            input wire         rxTriggerUSRInfoEnd_p,        // FSM Trigger User Info end pulse
            input wire         fcsOkRcved_p,                 // Frame received with correct FCS 
                                                             
            input wire         rxControlIdle,                // FSM in idle
            input wire         rxTriggerFound,               // Indicate that a valid trigger has 
                                                             // been found
                                                             
            output wire        mgmtFrame,                    // Management frame
            output wire        ctrlFrame,                    // Control frame
            output wire        dataFrame,                    // Data frame
            output wire        extnFrame,                    // Extension frame
            output wire        qosFrame,                     // QoS frame
            output wire        htcFrame,                     // HT or VHT frame with HTC field
            output reg         bssIDFrame,                   // Frame contains BSSID field
                                                             
            output reg         broadCastRcved,               // Broadcast frame
            output wire        bcMcRcved,                    // Broadcast/Multicast
                                                             
            output wire        bcnRcved,                     // Beacon
            output wire        probRespRcved,                // Probe response
                                                             
            output wire        triggerRcved,                 // Trigger Frame
            output wire        triggerHWRcved,               // Trigger frame received with HW
                                                             // response expected
            output wire        triggerSWRcved,               // Trigger frame received with SW
                                                             // response expected
`ifdef RW_BFMEE_EN                                           
            output wire        bfrPollRcved,                 // BFR-Poll
            output wire        ndpaRcved,                    // NPD-A
`endif //RW_BFMEE_EN                                         
`ifdef RW_BFMER_EN                                           
            output wire        bfrRcved,                     // BFR
`endif // RW_BFMER_EN                                        
            output wire        ftmRcved,                     // FTM received
            output wire        publicActionRcved,            // public Action frame
                                                             // received
            output wire        ctrlWrapRcved,                // Control Wrapper
            output wire        baRcved,                      // BA
            output wire        barRcved,                     // BAR
            output wire        psPollRcved,                  // PS-Poll
            output wire        rtsRcved,                     // RTS
            output wire        ctsRcved,                     // CTS
            output wire        ackRcved,                     // ACK
            output wire        cfEndRcved,                   // CF-End
                                                             
            output reg         ctrlWrapFlag,                 // Control Wrapper flag
            output wire        ctrlWrapError_p,              // Control wrapper error pulse
                                                             
            output wire        dataRcved,                    // Data
            output wire        qosDataRcved,                 // QoS Data
            output wire        qosNullRcved,                 // QoS null
                                                             
            output wire        reservedRcved,                // Reserved
            output wire        otherCtrlRcved,               // Other control frame
                                                             
            output wire        toDS,                         // To DS
            output wire        fromDS,                       // From DS
            output wire        rxMoreFrag,                   // More fragment
            output wire        rxRetry,                      // Retry
            output wire        pwrMgt,                       // Power management
            output wire        rxMoreData,                   // More data
            output wire        protectedBit,                 // Protected bit
            output wire        rxAMSDUPresent,               // A-MSDU Present
            output wire [1:0]  rxAckPolicy,                  // Ack policy (Ack, No Ack, BA)
            output wire [3:0]  rxTID,                        // TID
                                                             
            output reg         rxBWSignalingTA,              // Indicate that the received frame has a Bandwidth Signaling TA
                                                             
            output reg [31:0]  rxHTCtrl,                     // HT control
                                                             
            output wire        basicBA,                      // Basic BA
            output wire        compressedBA8,                // Compressed BA with
                                                             // 8 Bytes Bitmap
            output wire        compressedBA32,               // Compressed BA with
                                                             // 32 Bytes Bitmap
                                                             
            output reg         acceptRcved,                  // Accept current frame
                                                             
            output reg         addr1Match,                   // ADDR1 match
            output reg         bssIDMatch,                   // BSSID match
            output reg         wildcardBSSID,                // Wildcard BSSID
`ifdef RW_BFMEE_EN                                           
            output reg         rxNDPAAIDMatch,               // NDPA AID12 match
`endif //RW_BFMEE_EN
            output reg         addr1MatchBssID,              // ADDR1 is equal to the BSS ID
            output reg         addr2MatchBssID,              // ADDR2 is equal to the BSS ID

            output wire  [7:0] trigDepUsrInfoLen,            // Length of the Trigger Dependent
                                                             // User Info Field
            output wire  [3:0] rxTriggerType,                // Trigger Type
            output wire [11:0] rxTriggerULLength,            // Trigger frame, UL Length
            output wire        rxTriggerCS,                  // Trigger frame, CS
            output wire  [1:0] rxTriggerULBW,                // Trigger frame, UL BW
            output wire  [1:0] rxTriggerGIAndLTFType,        // Trigger frame, GI And LTF Type
            output wire        rxTriggerHELTFMode,           // Trigger frame, HE-LTF Mode
            output wire  [2:0] rxTriggerNumHELTFAndMidPeriod,// Trigger frame, Number of HE-LTF
                                                             // symbols And Midamble Periodicity
            output wire        rxTriggerULSTBC,              // Trigger frame, UL STBC
            output wire        rxTriggerLDPCExtraSymbol,     // Trigger frame, LDPC Extra Symbol
                                                             // Segment
            output wire  [5:0] rxTriggerAPTxPower,           // Trigger frame, AP TX Power
            output wire  [2:0] rxTriggerULPacketExtension,   // Trigger frame, UL Packet Extension
            output wire [15:0] rxTriggerULSpatialReuse,      // Trigger frame, UL Spatial Reuse
            output wire        rxTriggerDoppler,             // Trigger frame, Doppler
            output wire  [8:0] rxTriggerULHESigA2Reserved,   // Trigger frame, UL HE-SIG-A2 Reserved
            output wire        rxTriggerUSRInfoMatch,        // Trigger frame, Valid user info field
            output wire        rxTriggerAIDMatch,            // Trigger frame, AID12 Match
            output wire        rxTriggerAID4095,             // Trigger frame, AID12 is 4095
            output wire  [7:0] rxTriggerRUAllocation,        // Trigger frame, RU Allocation
            output wire        rxTriggerULFECCoding,         // Trigger frame, UL Coding Type
            output wire  [3:0] rxTriggerULMCS,               // Trigger frame, UL MCS
            output wire        rxTriggerULDCM,               // Trigger frame, UL DCM
            output wire  [2:0] rxTriggerStartingSSNum,       // Trigger frame, Starting Spatial 
                                                             // Stream
            output wire  [2:0] rxTriggerNumSS,               // Trigger frame, Number Of Spatial
                                                             // Streams
            output wire  [6:0] rxTriggerULTargetRSSSI,       // Trigger frame, UL Target RSSI
            output wire [79:0] rxTriggerDepUserInfo,         // Trigger frame, Trigger Dependent
                                                             // User Info Field
            output wire        rxTriggerULRUType,            // Trigger Frame, RU Type
            
            ///////////////////////////////////////////////
            //$port_g BA Controller
            ///////////////////////////////////////////////
            output reg  [15:0] rxSeqCtrl,               // Sequence control
            output wire [11:0] rxBASSN,                 // BA Starting Sequence Number
            output wire        muBARTriggerRcved,       // MU-BAR Trigger Received flag
            output wire        muBARTriggerRcved_p,     // MU-BAR Trigger Received pulse

            ///////////////////////////////////////////////
            //$port_g A-MPDU Deaggregator
            ///////////////////////////////////////////////
            input wire [ 3:0]  rxFormatMod,             // Format and modulation
            input wire [15:0]  rxMpduLength,            // Rx MPDU full length
            input wire         rxSMPDU,                 // Indicate that the received frame is a 
                                                        // S-MPDU
            
            input wire [ 7:0]  rxData,                  // Rx data read from MAC-PHY interface FIFO
            input wire         rxDataValid,             // Rx data is valid

`ifdef RW_BFMEE_EN
            ///////////////////////////////////////////////
            //$port_g BFR Controller
            ///////////////////////////////////////////////
            output reg   [7:0] rxBFRPollSegRet,           // BFR-Poll Feedback Segment Retransmission Bitmap
            output reg         rxNDPAHE,                  // NDP-A HE
            output reg   [5:0] rxNDPASoundingDialToken,   // NDP-A Sounding Dialog Token Number
            output reg   [1:0] rxNDPAFeedbackType,        // NDP-A Feedback Type
            output reg   [6:0] rxNDPARUStartIndex,        // NDP-A RU Start Index
            output reg   [6:0] rxNDPARUEndIndex,          // NDP-A RU End Index
            output reg         rxNDPANg,                  // NDP-A Ng
            output reg         rxNDPACodebook,            // NDP-A Codebook
            output reg   [2:0] rxNDPANcIndex,             // NDP-A Nc Index
            output reg         rxNDPASTAIndexIsFirst,     // NDP-A our STA Info field Position
`endif //RW_BFMEE_EN

            ///////////////////////////////////////////////
            //$port_g SR Controller
            ///////////////////////////////////////////////
            output reg   [5:0] partialBSSID,            // BSSID[44:39] of the received Frame

            ///////////////////////////////////////////////
            //$port_g Control and Status Register
            ///////////////////////////////////////////////
            input wire [47:0]  macAddr,                 // MAC Addr
            input wire [47:0]  macAddrMask,             // MAC Addr mask
            
            input wire [47:0]  bssID,                   // BSSID
            input wire [47:0]  bssIDMask,               // BSSID Mask
            
            input wire  [15:0] aid,                     // Association ID

            input wire         dontDecrypt,             // Don't decrypt frames
            input wire         excUnencrypted,          // Exclude unencrypted frames
            input wire         acceptMulticast,         // Accept Multicast frames
            input wire         acceptBroadcast,         // Accept Broadcast frames
            input wire         acceptOtherBSSID,        // Accept other BSSID frames
            input wire         acceptErrorFrames,       // Accept error frames
            input wire         acceptUnicast,           // Accept Unicast frames
            input wire         acceptMyUnicast,         // Accept frames directed to this device
            input wire         acceptProbeReq,          // Accept Probe Request frames
            input wire         acceptProbeResp,         // Accept Probe Response frames
            input wire         acceptBeacon,            // Accept Beacon frames
            input wire         acceptAllBeacon,         // Accept all Beacon frames 
`ifdef RW_BFMEE_EN
            input wire         acceptBfmeeFrames,       // Accept Beamformee Frames
`endif //RW_BFMEE_EN
            input wire         acceptOtherMgmtFrames,   // Accept other management frames
            input wire         acceptNotExpectedBA,     // Accept Block Ack Request frames received after SIFS
            input wire         acceptBAR,               // Accept Block Ack Request frames
            input wire         acceptBA,                // Accept Block Ack frames
            input wire         acceptPSPoll,            // Accept PS-Poll frames
            input wire         acceptRTS,               // Accept RTS frames
            input wire         acceptCTS,               // Accept CTS frames
            input wire         acceptACK,               // Accept ACK frames
            input wire         acceptCFEnd,             // Accept CF-End frames
            input wire         acceptOtherCntrlFrames,  // Accept other control frames
            input wire         acceptData,              // Accept Data frames
            input wire         acceptCFWOData,          // Accept CF WithOut Data frames
            input wire         acceptQData,             // Accept QoS Data frames
            input wire         acceptQCFWOData,         // Accept QoS CF WithOut Data frames
            input wire         acceptQoSNull,           // Accept QoS Null frames
            input wire         acceptOtherDataFrames,   // Accept other Data frames
            input wire         acceptUnknown,           // Accept unknown frames
            input wire         acceptTriggerHWFrames,   // Accept Trigger frame with HW response
            input wire         acceptTriggerSWFrames,   // Accept Trigger frame with SW response
            input wire         acceptAllTriggerFrames,  // Accept All Trigger frames

            output reg   [7:0] quietCount1In,           // Quiet Count 1
            output reg   [7:0] quietPeriod1In,          // Quiet Period 1
            output reg  [15:0] quietDuration1In,        // Quiet Duration 1
            output reg  [15:0] quietOffset1In,          // Quiet Offset 1
            
            output reg   [7:0] dtimPeriodIn,            // DTIM period

            input wire         dynBWEn,                 // Enable the Dynamic Bandwidth

            input wire   [2:0] primaryChPosition,       // Primary Channel Position
            input wire   [1:0] maxSupportedBW,          // max Supported BW
            input wire         raRUType,                // Random Access RU Type
            input wire         raRUEnable,              // Random Access RU Enable
            input wire   [3:0] maxMCSInHETB,            // Max MCS in HE TB
            input wire   [2:0] eOCW,                    // Exposent of OFDMA Contention Window
            input wire   [2:0] eOCWMin,                 // Exposent of OFDMA Contention Window Minimum
            output wire  [2:0] eOCWIn,                  // Exposent of OFDMA Contention Window In
            output wire        eOCWInValid,             // Exposent of OFDMA Contention Window In Valid
            input  wire  [6:0] ocwLFSR,                 // OFDMA Contention Window LFSR
            output wire  [6:0] ocwLFSRIn,               // OFDMA Contention Window LFSR In
            output wire        ocwLFSRInValid,          // OFDMA Contention Window LFSR In Valid
            input wire         dopplerSupport,          // Doppler Support
            input wire         dcmSupport,              // Dual Carrier Modulation Support
            ///////////////////////////////////////////////
            //$port_g RX FIFO Interface Controller
            ///////////////////////////////////////////////
            input  wire        rxBADly,                 // Capture of the rxBA
            output reg [15:0]  rxFrmCtrl,               // Frame control field
            output wire [3:0]  subTypeFrame,            // Subtype
            output wire [1:0]  typeFrame,               // Type
            output wire        protocolError,           // Protocol version
            output wire        ackRespRcved,            // Ack Response received
            ///////////////////////////////////////////////
            //$port_g RX FIFO Interface Controller
            ///////////////////////////////////////////////
            output wire [15:0] debugPortTriggerDecoder  // Debug port
            );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
reg         multiCastRcved;   // Multicast frame
reg         ctrlWrapError;    // Control wrapper error flag
reg  [15:0] rxBACtrl;         // BA or BAR control field
reg  [15:0] rxBASeqCtrl;      // BA or BAR Starting Sequence control field
wire [47:0] macAddrMasked;    // MAC Addr masked
wire [47:0] bssIDMasked;      // BSSID masked
wire        rxOrder;          // Order
                                                             
reg  [31:0] rxInitVectorFull; // IV field
reg   [7:0] rxFrmBodyByte1;      // Frame Bofy Byte 1
reg   [7:0] rxFrmBodyByte2;      // Frame Bofy Byte 2

wire compressedBA;            // Compressed BA
wire multiSTABA;              // Multi-STA BA

wire probReqRcved;            // Probe request
wire actNoAckRcved;           // Action no Ack
wire pubActRcved;             // Public Action frame
wire otherMgmtRcved;          // Other management frame
wire cfRcved;                 // CF frame
wire qosCFRcved;              // QoS CF frame
wire otherDataRcved;          // Other data frame
wire dataNullRcved;           // Null (no data)
`ifndef RW_BFMEE_EN
wire bfrPollRcved;            // BFR-Poll
wire ndpaRcved;               // NPD-A
`endif //RW_BFMEE_EN

wire encryptionFilter;        // Encryption filter
wire errorFilter;             // Error filter
wire ksrFilter;               // Key storage RAM filter
wire addressFilter;           // Address filter
wire bssIDFilter;             // bssID filter
wire frameTypeFilter;         // Frame type filter
wire monitorFilter;           // monitor mode filter
wire duplicateFilter;         // duplicate filter

wire myFrameRcved_p;          // my Frame correctly received pulse
wire myQoSFrameRcved_p;       // my QoS Frame correctly received pulse
wire barIncorrectRcved_p;     // BAR not supported (Compressed BA only is supported)

`ifdef RW_BFMEE_EN
reg  rxNDPAMyStaInfoDone;     // Indicates that my NDP-A STA INFO Field is found
`endif //RW_BFMEE_EN

wire mgmtEncryptFrame;        // Flag for management frame allowed to be encrypted

wire rxTriggerUSRInfoMatch_p; // Trigger frame with a valid user info


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Fields sampling

// Frame control field
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrmCtrl <= 16'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxFrmCtrl[1:0] <= 2'b0;
  else if (rxDataValid && rxFrmCtrlEn   ||
           rxDataValid && rxCarFrmCtrlEn)
  begin
    if (stateByteCnt[0])
      rxFrmCtrl[15:8] <= rxData;
    else
      rxFrmCtrl[7:0]  <= rxData;
  end
end

// MPDU Frame Type/SubType Decoding information
assign protocolError   = (rxFrmCtrl[1:0] != 2'h0) | ctrlWrapError;

// Incase fo protocol Error, frame type is forced to Reserved
assign typeFrame       = (protocolError) ? 2'h3 : rxFrmCtrl[3:2];
assign subTypeFrame    = (protocolError) ? 4'h0 : rxFrmCtrl[7:4];

// Frame type
assign mgmtFrame       = (typeFrame == 2'h0);
assign ctrlFrame       = (typeFrame == 2'h1);
assign dataFrame       = (typeFrame == 2'h2);
assign extnFrame       = (typeFrame == 2'h3);

// Indicate a Managment Frame which can be encrypted 
assign mgmtEncryptFrame = mgmtFrame && ((subTypeFrame == 4'b1010) ||  // Disassociation
                                        (subTypeFrame == 4'b1011) ||  // Authentication
                                        (subTypeFrame == 4'b1100) ||  // Deauthentication
                                        (subTypeFrame == 4'b1101) ||  // Action
                                        (subTypeFrame == 4'b1110));   // Action No Ack

assign qosFrame        = (dataFrame && rxFrmCtrl[7]);
// It is set to 1 in a QoS Data or Management frame transmitted with rxOrder = 1 and a value of HT_GF, HT_MF, or
// VHT to indicate that the frame contains an HT Control field
assign htcFrame        = ((qosFrame || mgmtFrame) && (rxFormatMod >= 4'd2) && rxOrder);

// Management frame
assign probReqRcved    = mgmtFrame &&  (subTypeFrame == 4'h4);
assign probRespRcved   = mgmtFrame &&  (subTypeFrame == 4'h5);
assign bcnRcved        = mgmtFrame &&  (subTypeFrame == 4'h8);
assign actNoAckRcved   = mgmtFrame &&  (subTypeFrame == 4'he);
assign otherMgmtRcved  = mgmtFrame && !(bcnRcved || probRespRcved ||
                                        probReqRcved);
`ifdef RW_BFMER_EN
  // VHT Compressed Beamforming Report
assign bfrRcved        = mgmtFrame &&  (subTypeFrame == 4'he) 
                                   &&  (rxFrmBodyByte1 == 8'd21) // Action field 
                                   &&  (rxFrmBodyByte2 == 8'd0); // Category field 
`endif // RW_BFMER_EN
assign ftmRcved        = mgmtFrame &&  (subTypeFrame == 4'hd) 
                                   &&  (rxFrmBodyByte1 == 8'd04)  // Action field
                                   &&  (rxFrmBodyByte2 == 8'd33); // Public Action field 

assign pubActRcved     = mgmtFrame &&  (subTypeFrame == 4'hd) 
                                   &&  (rxFrmBodyByte1 == 8'd04); // Action field

assign publicActionRcved = pubActRcved;


// Control frame
assign otherCtrlRcved  = ctrlFrame && ((subTypeFrame <= 4'h1) ||
                                       (subTypeFrame == 4'h3) ||
                                       (subTypeFrame == 4'h6));
assign triggerRcved    = ctrlFrame &&  (subTypeFrame == 4'h2);
assign triggerHWRcved  = triggerRcved && ~(rxTriggerType==4'd0 || rxTriggerType==4'd4);
assign triggerSWRcved  = triggerRcved &&  (rxTriggerType==4'd0 || rxTriggerType==4'd4);
assign bfrPollRcved    = ctrlFrame &&  (subTypeFrame == 4'h4) && (rxMpduLength == 16'd21);
assign ndpaRcved       = ctrlFrame &&  (subTypeFrame == 4'h5);
assign ctrlWrapRcved   = ctrlFrame &&  (subTypeFrame == 4'h7);
assign barRcved        = ctrlFrame &&  (subTypeFrame == 4'h8) && ((ctrlWrapFlag && (rxMpduLength == 16'd30)) || (!ctrlWrapFlag && (rxMpduLength == 16'd24))); // todo in case of Multi-TID and GCR BlockAckReq
assign baRcved         = ctrlFrame &&  (subTypeFrame == 4'h9);
assign psPollRcved     = ctrlFrame &&  (subTypeFrame == 4'ha) && ((ctrlWrapFlag && (rxMpduLength == 16'd26)) || (!ctrlWrapFlag && (rxMpduLength == 16'd20)));
assign rtsRcved        = ctrlFrame &&  (subTypeFrame == 4'hb) && ((ctrlWrapFlag && (rxMpduLength == 16'd26)) || (!ctrlWrapFlag && (rxMpduLength == 16'd20)));
assign ctsRcved        = ctrlFrame &&  (subTypeFrame == 4'hc) && ((ctrlWrapFlag && (rxMpduLength == 16'd20)) || (!ctrlWrapFlag && (rxMpduLength == 16'd14)));
assign ackRcved        = ctrlFrame &&  (subTypeFrame == 4'hd) && ((ctrlWrapFlag && (rxMpduLength == 16'd20)) || (!ctrlWrapFlag && (rxMpduLength == 16'd14)));
assign cfEndRcved      = ctrlFrame && ((subTypeFrame == 4'he) ||
                                       (subTypeFrame == 4'hf)) &&
                                      (( ctrlWrapFlag && (rxMpduLength == 16'd26)) ||
                                       (!ctrlWrapFlag && (rxMpduLength == 16'd20)));


// Control Wrapper flag
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    ctrlWrapFlag <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    ctrlWrapFlag <= 1'b0;
  else if (ctrlWrapRcved)
    ctrlWrapFlag <= 1'b1;
end

// Control Wrapper error
//    Detection after 1st byte of frame control field of the carried frame, when
//    frame type & subtype are available. This allow error be handled in the
//    receive carried frame control state.
assign ctrlWrapError_p = rxCarFrmCtrlEn & rxDataValid & stateByteCnt[0] &
                         (ctrlWrapRcved || !ctrlFrame);

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    ctrlWrapError <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    ctrlWrapError <= 1'b0;
  //else if (ctrlWrapFlag && carFrameCtrlEnd_p && (ctrlWrapRcved || !ctrlFrame))
  else if (ctrlWrapError_p)
    ctrlWrapError <= 1'b1;
end

// Data frame
assign dataRcved       = dataFrame && ((subTypeFrame == 4'h0) || 
                                       (subTypeFrame == 4'h1) || 
                                       (subTypeFrame == 4'h2) || 
                                       (subTypeFrame == 4'h3));
assign otherDataRcved  = dataFrame && (subTypeFrame == 4'hd);
assign dataNullRcved   = dataFrame && (subTypeFrame == 4'h4);
assign cfRcved         = dataFrame && ((subTypeFrame == 4'h5) || 
                                       (subTypeFrame == 4'h6) || 
                                       (subTypeFrame == 4'h7));
assign qosDataRcved    = dataFrame && ((subTypeFrame == 4'h8) || 
                                       (subTypeFrame == 4'h9) ||
                                       (subTypeFrame == 4'ha) ||
                                       (subTypeFrame == 4'hb));
assign qosNullRcved    = dataFrame &&  (subTypeFrame == 4'hc);
assign qosCFRcved      = dataFrame && ((subTypeFrame == 4'he) || 
                                       (subTypeFrame == 4'hf));

// Reserved frame
assign reservedRcved   = (dataFrame &&  (subTypeFrame == 4'hd))  ||
                         (mgmtFrame && ((subTypeFrame == 4'h7)   ||
                                        (subTypeFrame == 4'hf))) ||
                         (dataFrame &&  (subTypeFrame == 4'hd))  ||
                         (ctrlFrame && otherCtrlRcved);

// Frame control fields
assign toDS            = rxFrmCtrl[8] && (dataFrame || mgmtFrame);
assign fromDS          = rxFrmCtrl[9] && (dataFrame || mgmtFrame);
assign rxMoreFrag      = rxFrmCtrl[10];
assign rxRetry         = rxFrmCtrl[11];
assign pwrMgt          = rxFrmCtrl[12];
assign rxMoreData      = rxFrmCtrl[13];
assign protectedBit    = rxFrmCtrl[14] && (dataFrame || mgmtEncryptFrame) && !reservedRcved;
assign rxOrder         = rxFrmCtrl[15];


// Duration / ID field
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrameDuration <= 16'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxFrameDuration <= 16'b0;
  else if (rxDataValid && rxDurationIdEn)
  begin
    if (stateByteCnt[0])
      rxFrameDuration[15:8] <= rxData;
    else
      rxFrameDuration[7:0]  <= rxData;
  end
end

// Addr1 matching & sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    addr1Match <= 1'b0;
    rxAddr1    <= 48'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0)
  begin
    addr1Match <= 1'b0;
    rxAddr1    <= 48'b0;
  end
  else if (rxControlIdle == 1'b1) // Reset only addr1Match and keep rxAddr1
    addr1Match <= 1'b0;
  else if (rxDataValid && rxAddr1En)
  begin
    case (stateByteCnt[2:0])
      3'h0: begin
        rxAddr1[7:0] <= rxData;
        if (macAddrMasked[7:0] == (rxData | macAddrMask[7:0]))
          addr1Match <= 1'b1;
        else
          addr1Match <= 1'b0;
      end

      3'h1: begin
        rxAddr1[15:8] <= rxData;
        if (macAddrMasked[15:8] == (rxData | macAddrMask[15:8]))
          addr1Match <= addr1Match;
        else
          addr1Match <= 1'b0;
      end

      3'h2: begin
        rxAddr1[23:16] <= rxData;
        if (macAddrMasked[23:16] == (rxData | macAddrMask[23:16]))
          addr1Match <= addr1Match;
        else
          addr1Match <= 1'b0;
      end

      3'h3: begin
        rxAddr1[31:24] <= rxData;
        if (macAddrMasked[31:24] == (rxData | macAddrMask[31:24]))
          addr1Match <= addr1Match;
        else
          addr1Match <= 1'b0;
      end

      3'h4: begin
        rxAddr1[39:32] <= rxData;
        if (macAddrMasked[39:32] == (rxData | macAddrMask[39:32]))
          addr1Match <= addr1Match;
        else
          addr1Match <= 1'b0;
      end

      3'h5: begin
        rxAddr1[47:40] <= rxData;
        if (macAddrMasked[47:40] == (rxData | macAddrMask[47:40]))
          addr1Match <= addr1Match;
        else
          addr1Match <= 1'b0;
      end

      // Disable case default state for block coverage
      // pragma coverage block = off
      default: begin
        addr1Match <= addr1Match;
        rxAddr1    <= rxAddr1;
      end
      // pragma coverage block = on
      
    endcase
  end
end

// Addr1 Mask
assign macAddrMasked = (macAddr | macAddrMask);

// Broadcast frame
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    broadCastRcved <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    broadCastRcved <= 1'b0;
  else if (rxDataValid && rxAddr1En)
  begin
    case (stateByteCnt[2:0])
      3'h0:
        if (rxData == 8'hff)
          broadCastRcved <= 1'b1;
        else
          broadCastRcved <= 1'b0;

      3'h1, 3'h2, 3'h3, 3'h4, 3'h5:
        if (rxData == 8'hff)
          broadCastRcved <= broadCastRcved;
        else
          broadCastRcved <= 1'b0;

      // Disable case default state for block coverage
      // pragma coverage block = off
      default:
        broadCastRcved <= broadCastRcved;
      // pragma coverage block = on
        
    endcase
  end
end

// Multicast frame
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    multiCastRcved <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    multiCastRcved <= 1'b0;
  else if (rxDataValid && (stateByteCnt == 8'h00) && rxAddr1En)
    multiCastRcved <= rxData[0];
  else if (rxAddr2En && broadCastRcved)
    multiCastRcved <= 1'b0;
end


// Broadcast/Multicast frame
assign bcMcRcved = (multiCastRcved || broadCastRcved);

// Frame filtering
assign encryptionFilter = (excUnencrypted && !protectedBit) ||
                          !excUnencrypted;

assign errorFilter      = (!acceptErrorFrames && !protocolError) ||
                          acceptErrorFrames;

// KSR Filter
//    The received frame doesn't have BSSID information and ADDR2 has been found
//    in the key storage RAM
assign ksrFilter        = ~bssIDFrame & keyStorageValid;

// If BSSID match
// acceptOtherBSSID is set
// a Probe Request is received
assign bssIDFilter      = bssIDMatch                        ||
                          (!bssIDMatch && acceptOtherBSSID) ||
                          (acceptProbeReq && probReqRcved)  ||
                          (acceptAllBeacon && bcnRcved);

assign addressFilter    = (acceptMulticast && multiCastRcved && bssIDFilter) ||
                          (acceptBroadcast && broadCastRcved && bssIDFilter) ||
                          (acceptMulticast && multiCastRcved && ksrFilter  ) ||
                          (acceptBroadcast && broadCastRcved && ksrFilter  ) ||
                          (acceptUnicast && !addr1Match && !bcMcRcved && (bssIDFilter || (ctrlFrame && !cfEndRcved && !psPollRcved))) ||
                          (acceptMyUnicast && addr1Match && !bcMcRcved);

assign frameTypeFilter  = (acceptProbeReq && probReqRcved) ||
                          (acceptProbeResp && probRespRcved) ||
                          ((acceptBeacon || acceptAllBeacon) && bcnRcved) ||
`ifdef RW_BFMER_EN
                          (acceptOtherMgmtFrames && (otherMgmtRcved || bfrRcved)) ||
`else
                          (acceptOtherMgmtFrames && otherMgmtRcved) ||
`endif //RW_BFMER_EN
                          (acceptTriggerHWFrames && triggerHWRcved) ||
                          (acceptTriggerSWFrames && triggerSWRcved && rxTriggerUSRInfoMatch) ||
`ifdef RW_BFMEE_EN
                          (acceptBfmeeFrames && (bfrPollRcved || ndpaRcved)) ||
`else //RW_BFMEE_EN
                          (acceptOtherCntrlFrames && (bfrPollRcved || ndpaRcved)) ||
`endif //RW_BFMEE_EN
                          (acceptBAR && (barRcved || ctrlWrapRcved)) ||
                          (((acceptBA && rxBADly) || acceptNotExpectedBA) && (baRcved || ctrlWrapRcved)) ||
                          (acceptPSPoll && (psPollRcved || ctrlWrapRcved)) ||
                          (acceptRTS && (rtsRcved || ctrlWrapRcved)) ||
                          (acceptCTS && (ctsRcved || ctrlWrapRcved)) ||
                          (acceptACK && (ackRcved || ctrlWrapRcved)) ||
                          (acceptCFEnd && (cfEndRcved || ctrlWrapRcved)) ||
                          (acceptOtherCntrlFrames && (otherCtrlRcved || ctrlWrapRcved)) ||
                          (acceptData && dataRcved) ||
                          (acceptCFWOData && cfRcved) ||
                          (acceptQData && qosDataRcved) ||
                          (acceptQCFWOData && qosCFRcved) ||
                          (acceptQoSNull && qosNullRcved) ||
                          (acceptQoSNull && dataNullRcved) ||
                          (acceptOtherDataFrames && otherDataRcved) ||
                          (acceptUnknown && (rxFrmCtrl[3:2] == 2'h3));

assign monitorFilter    = (acceptAllTriggerFrames && triggerRcved);

assign duplicateFilter  = ~(rxTriggerFound & triggerRcved);

// Accept received frame
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    acceptRcved <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    acceptRcved <= 1'b0;
  else if (monitorFilter)
    acceptRcved <= 1'b1;
  else if (encryptionFilter && errorFilter     && addressFilter &&
           frameTypeFilter  && duplicateFilter )
    acceptRcved <= 1'b1;
  else
    acceptRcved <= 1'b0;
end

// Addr2 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxAddr2          <= 48'b0;
    rxBWSignalingTA  <= 1'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
  begin
    rxAddr2          <= 48'b0;
    rxBWSignalingTA  <= 1'b0;
  end
  else if (rxDataValid && rxAddr2En)
  begin
    case (stateByteCnt[2:0])
      3'h0:    
        begin
          rxAddr2[7:1]    <= rxData[7:1];
          rxBWSignalingTA <= dynBWEn && rxData[0];
        end
      3'h1:    rxAddr2[15:8]  <= rxData;
      3'h2:    rxAddr2[23:16] <= rxData;
      3'h3:    rxAddr2[31:24] <= rxData;
      3'h4:    rxAddr2[39:32] <= rxData;
      3'h5:    rxAddr2[47:40] <= rxData;
      // Disable case default state for block coverage
      // pragma coverage block = off
      default: rxAddr2        <= rxAddr2;
      // pragma coverage block = on
    endcase
  end
end


// Addr3 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxAddr3 <= 48'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxAddr3 <= 48'b0;
  else if (rxDataValid && rxAddr3En)
  begin
    case (stateByteCnt[2:0])
      3'h0:    rxAddr3[7:0]   <= rxData;
      3'h1:    rxAddr3[15:8]  <= rxData;
      3'h2:    rxAddr3[23:16] <= rxData;
      3'h3:    rxAddr3[31:24] <= rxData;
      3'h4:    rxAddr3[39:32] <= rxData;
      3'h5:    rxAddr3[47:40] <= rxData;
      // Disable case default state for block coverage
      // pragma coverage block = off
      default: rxAddr3        <= rxAddr3;
      // pragma coverage block = on
    endcase
  end
end

// Addr4 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxAddr4 <= 48'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxAddr4 <= 48'b0;
  else if (rxDataValid && rxAddr4En)
  begin
    case (stateByteCnt[2:0])
      3'h0:    rxAddr4[7:0]   <= rxData;
      3'h1:    rxAddr4[15:8]  <= rxData;
      3'h2:    rxAddr4[23:16] <= rxData;
      3'h3:    rxAddr4[31:24] <= rxData;
      3'h4:    rxAddr4[39:32] <= rxData;
      3'h5:    rxAddr4[47:40] <= rxData;
      // Disable case default state for block coverage
      // pragma coverage block = off
      default: rxAddr4        <= rxAddr4;
      // pragma coverage block = on
    endcase
  end
end

`ifdef RW_WAPI_EN
// rxWAPIKeyIndex sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxWAPIKeyIndex <= 8'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxWAPIKeyIndex <= 8'b0;
  else if (rxDataValid && rxWAPIKeyIdxEn)
    rxWAPIKeyIndex <= rxData;
end


// rxWAPIPN sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxWAPIPN <= 128'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxWAPIPN <= 128'b0;
  else if (rxDataValid && rxWAPIPNEn)
  begin
    case (stateByteCnt[3:0])
      4'h0:    rxWAPIPN[7:0]     <= rxData;
      4'h1:    rxWAPIPN[15:8]    <= rxData;
      4'h2:    rxWAPIPN[23:16]   <= rxData;
      4'h3:    rxWAPIPN[31:24]   <= rxData;
      4'h4:    rxWAPIPN[39:32]   <= rxData;
      4'h5:    rxWAPIPN[47:40]   <= rxData;
      4'h6:    rxWAPIPN[55:48]   <= rxData;
      4'h7:    rxWAPIPN[63:56]   <= rxData;
      4'h8:    rxWAPIPN[71:64]   <= rxData;
      4'h9:    rxWAPIPN[79:72]   <= rxData;
      4'hA:    rxWAPIPN[87:80]   <= rxData;
      4'hB:    rxWAPIPN[95:88]   <= rxData;
      4'hC:    rxWAPIPN[103:96]  <= rxData;
      4'hD:    rxWAPIPN[111:104] <= rxData;
      4'hE:    rxWAPIPN[119:112] <= rxData;
      4'hF:    rxWAPIPN[127:120] <= rxData;
      // Disable case default state for block coverage
      // pragma coverage block = off
      default: rxWAPIPN          <= rxWAPIPN;
      // pragma coverage block = on
    endcase
  end
end

`endif //RW_WAPI_EN

// BSSID field matching
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    bssIDMatch    <= 1'b0;
    bssIDFrame    <= 1'b0;
    partialBSSID  <= 6'd0;
    wildcardBSSID <= 1'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
  begin
    bssIDMatch    <= 1'b0;
    bssIDFrame    <= 1'b0;
    partialBSSID  <= 6'd0;
    wildcardBSSID <= 1'b0;
  end
  else if (rxDataValid && 
         ((rxAddr1En && (( toDS && !fromDS &&  dataFrame) || psPollRcved)) ||
       //   (rxAddr2En && ((!toDS &&  fromDS &&  dataFrame) || cfEndRcved || triggerRcved)) ||
          (rxAddr2En && ((!toDS &&  fromDS &&  dataFrame) || cfEndRcved)) ||
`ifdef RW_BFMEE_EN
          (rxAddr2En && broadCastRcved && ndpaRcved) ||
`endif//RW_BFMEE_EN
          (rxAddr3En &&    toDS &&  fromDS &&  dataFrame) ||
          (rxAddr3En &&   !toDS && !fromDS && (dataFrame || mgmtFrame))))
  begin
    // Frame contains bssID field
    bssIDFrame <= 1'b1;
    
    // Check bssID field
    case (stateByteCnt[2:0])
      3'h0:
        if (rxAddr2En && dynBWEn)
        begin
          // In case of dynBWEn and TA (Addr2), the Individual/Group bit shall be masked
	        if (bssIDMasked[7:0] == ({rxData[7:1],1'b0} | bssIDMask[7:0]))
	          bssIDMatch <= 1'b1;
	        else
	          bssIDMatch <= 1'b0;
        end
        else
        begin
          // otherwise, the check is done on the full rxData
	        if (bssIDMasked[7:0] == (rxData | bssIDMask[7:0]))
  	        bssIDMatch <= 1'b1;
	        else
	          bssIDMatch <= 1'b0;
        end

      3'h1:
        if (bssIDMasked[15:8] == (rxData | bssIDMask[15:8]))
          bssIDMatch <= bssIDMatch;
        else
          bssIDMatch <= 1'b0;

      3'h2:
        if (bssIDMasked[23:16] == (rxData | bssIDMask[23:16]))
          bssIDMatch <= bssIDMatch;
        else
          bssIDMatch <= 1'b0;

      3'h3:
        if (bssIDMasked[31:24] == (rxData | bssIDMask[31:24]))
          bssIDMatch <= bssIDMatch;
        else
          bssIDMatch <= 1'b0;

      3'h4:
        if (bssIDMasked[39:32] == (rxData | bssIDMask[39:32]))
          bssIDMatch <= bssIDMatch;
        else
          bssIDMatch <= 1'b0;

      3'h5:
        if (bssIDMasked[47:40] == (rxData | bssIDMask[47:40]))
          bssIDMatch <= bssIDMatch;
        else
          bssIDMatch <= 1'b0;

      // Disable case default state for block coverage
      // pragma coverage block = off
      default:
        bssIDMatch <= bssIDMatch;
      // pragma coverage block = on
      
    endcase

    // Check Wildcasd bssID
    case (stateByteCnt[2:0])
    3'h0:    wildcardBSSID <= rxData[7:0]==8'hFF;
    default: wildcardBSSID <= wildcardBSSID & rxData[7:0]==8'hFF;
    endcase


    // Store Partial bssID field (bssID[44:39])
    case (stateByteCnt[2:0])
    3'd4: partialBSSID[0]   <= rxData[7];
    3'd5: partialBSSID[5:1] <= rxData[4:0];
    endcase
  end
end

// Address1 field matching the BSSID
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    addr1MatchBssID <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    addr1MatchBssID <= 1'b0;
  else if (rxDataValid && rxAddr1En)
  begin
    // Compare addr1 field with BSS ID
    case (stateByteCnt[2:0])
      3'h0:
	     if (bssIDMasked[7:0] == (rxData | bssIDMask[7:0]))
  	       addr1MatchBssID <= 1'b1;
	     else
	       addr1MatchBssID <= 1'b0;

      3'h1:
        if (bssIDMasked[15:8] == (rxData | bssIDMask[15:8]))
          addr1MatchBssID <= addr1MatchBssID;
        else
          addr1MatchBssID <= 1'b0;

      3'h2:
        if (bssIDMasked[23:16] == (rxData | bssIDMask[23:16]))
          addr1MatchBssID <= addr1MatchBssID;
        else
          addr1MatchBssID <= 1'b0;

      3'h3:
        if (bssIDMasked[31:24] == (rxData | bssIDMask[31:24]))
          addr1MatchBssID <= addr1MatchBssID;
        else
          addr1MatchBssID <= 1'b0;

      3'h4:
        if (bssIDMasked[39:32] == (rxData | bssIDMask[39:32]))
          addr1MatchBssID <= addr1MatchBssID;
        else
          addr1MatchBssID <= 1'b0;

      3'h5:
        if (bssIDMasked[47:40] == (rxData | bssIDMask[47:40]))
          addr1MatchBssID <= addr1MatchBssID;
        else
          addr1MatchBssID <= 1'b0;

      // Disable case default state for block coverage
      // pragma coverage block = off
      default:
        addr1MatchBssID <= addr1MatchBssID;
      // pragma coverage block = on
    endcase
  end
end

// Address2 field matching the BSSID
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    addr2MatchBssID <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    addr2MatchBssID <= 1'b0;
  else if (rxDataValid && rxAddr2En)
  begin
    // Check bssID field
    case (stateByteCnt[2:0])
      3'h0:
        if (dynBWEn)
        begin
          // In case of dynBWEn and TA (Addr2), the Individual/Group bit shall be masked
	       if (bssIDMasked[7:0] == ({rxData[7:1],1'b0} | bssIDMask[7:0]))
	         addr2MatchBssID <= 1'b1;
	       else
	         addr2MatchBssID <= 1'b0;
        end
        else
        begin
          // otherwise, the check is done on the full rxData
	       if (bssIDMasked[7:0] == (rxData | bssIDMask[7:0]))
  	         addr2MatchBssID <= 1'b1;
	       else
	         addr2MatchBssID <= 1'b0;
        end

      3'h1:
        if (bssIDMasked[15:8] == (rxData | bssIDMask[15:8]))
          addr2MatchBssID <= addr2MatchBssID;
        else
          addr2MatchBssID <= 1'b0;

      3'h2:
        if (bssIDMasked[23:16] == (rxData | bssIDMask[23:16]))
          addr2MatchBssID <= addr2MatchBssID;
        else
          addr2MatchBssID <= 1'b0;

      3'h3:
        if (bssIDMasked[31:24] == (rxData | bssIDMask[31:24]))
          addr2MatchBssID <= addr2MatchBssID;
        else
          addr2MatchBssID <= 1'b0;

      3'h4:
        if (bssIDMasked[39:32] == (rxData | bssIDMask[39:32]))
          addr2MatchBssID <= addr2MatchBssID;
        else
          addr2MatchBssID <= 1'b0;

      3'h5:
        if (bssIDMasked[47:40] == (rxData | bssIDMask[47:40]))
          addr2MatchBssID <= addr2MatchBssID;
        else
          addr2MatchBssID <= 1'b0;

      // Disable case default state for block coverage
      // pragma coverage block = off
      default:
        addr2MatchBssID <= addr2MatchBssID;
      // pragma coverage block = on
      
    endcase
  end
end

// BSSID Mask
assign bssIDMasked = (bssID | bssIDMask);

// OLBC counter update
assign olbcUpdate_p = bssIDFrame && !bssIDMatch && fcsCheck_p;

// QoS Control sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxQoSControl <= 16'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxQoSControl <= 16'b0;
  else if (rxControlIdle && !activeRx)
    rxQoSControl <= 16'b0;
  else if (rxDataValid && rxQoSCtrlEn)
  begin
    if (stateByteCnt[0])
      rxQoSControl[15:8] <= rxData;
    else
      rxQoSControl[7:0] <= rxData;
  end
end

assign rxAMSDUPresent = rxQoSControl[7];
assign rxAckPolicy    = rxQoSControl[6:5];
assign rxTID          = (barRcved || baRcved) ? rxBACtrl[15:12] :
                        (muBARTriggerRcved)   ? rxTriggerDepUserInfo[15:12] : 
                                                rxQoSControl[3:0];

assign rxBASSN        = (muBARTriggerRcved)   ? rxTriggerDepUserInfo[31:20] :
                                                rxBASeqCtrl[15:4];

// HT Control sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxHTCtrl <= 32'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxHTCtrl <= 32'b0;
  else if (rxDataValid && rxHTCtrlEn)
  begin
    case (stateByteCnt[1:0])
      2'h0:    rxHTCtrl[7:0]   <= rxData;
      2'h1:    rxHTCtrl[15:8]  <= rxData;
      2'h2:    rxHTCtrl[23:16] <= rxData;
      default: rxHTCtrl[31:24] <= rxData;
    endcase
  end
end

// Sequence Control sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxSeqCtrl <= 16'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxSeqCtrl <= 16'b0;
  else if (rxDataValid && rxSeqCtrlEn)
  begin
    if (stateByteCnt[0])
      rxSeqCtrl[15:8] <= rxData;
    else
      rxSeqCtrl[7:0]  <= rxData;
  end
end

// BA Control sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxBACtrl <= 16'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxBACtrl <= 16'b0;
  else if (rxDataValid && rxBACtrlEn)
  begin
    if (stateByteCnt[0])
      rxBACtrl[15:8] <= rxData;
    else
      rxBACtrl[7:0]  <= rxData;
  end
end

assign basicBA      = (rxBACtrl[4:1] == 4'd0);
assign compressedBA = (rxBACtrl[4:1] == 4'd2);
assign multiSTABA   = (rxBACtrl[4:1] == 4'd11);

// BA Starting Sequence Number sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxBASeqCtrl[15:0] <= 16'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxBASeqCtrl[15:0] <= 16'b0;
  else if (rxDataValid && rxBASeqCtrlEn)
  begin
    if (stateByteCnt[0])
      rxBASeqCtrl[15:8] <= rxData;
    else
      rxBASeqCtrl[7:0]  <= rxData;
  end
end

assign compressedBA8  = compressedBA & ~rxBASeqCtrl[2];
assign compressedBA32 = compressedBA &  rxBASeqCtrl[2];

// IV sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxInitVectorFull <= 32'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxInitVectorFull <= 32'b0;
  else if (rxDataValid && rxInitVectorEn)
  begin
    case (stateByteCnt[1:0])
      2'h0:    rxInitVectorFull[7:0]   <= rxData;
      2'h1:    rxInitVectorFull[15:8]  <= rxData;
      2'h2:    rxInitVectorFull[23:16] <= rxData;
      default: rxInitVectorFull[31:24] <= rxData;
    endcase
  end
end

assign rxInitVector = rxInitVectorFull[23:0];

// Extended IV sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxExtInitVector <= 32'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxExtInitVector <= 32'b0;
  else if (rxDataValid && rxExtInitVectorEn)
  begin
    case (stateByteCnt[1:0])
      2'h0:    rxExtInitVector[7:0]   <= rxData;
      2'h1:    rxExtInitVector[15:8]  <= rxData;
      2'h2:    rxExtInitVector[23:16] <= rxData;
      default: rxExtInitVector[31:24] <= rxData;
    endcase
  end
end

// DTIM counter sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    dtimCnt <= 8'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    dtimCnt <= 8'b0;
  else if (rxDataValid && rxDtimCntEn)
    dtimCnt <= rxData;
end

// DTIM period sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    dtimPeriodIn <= 8'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    dtimPeriodIn <= 8'b0;
  else if (rxDataValid && rxDtimPeriodEn)
    dtimPeriodIn <= rxData;
end

// Timestamp sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    timeStamp <= 64'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    timeStamp <= 64'b0;
  else if (rxDataValid && rxTimeStampEn)
  begin
    case (stateByteCnt[2:0])
      3'h0:    timeStamp[7:0]   <= rxData;
      3'h1:    timeStamp[15:8]  <= rxData;
      3'h2:    timeStamp[23:16] <= rxData;
      3'h3:    timeStamp[31:24] <= rxData;
      3'h4:    timeStamp[39:32] <= rxData;
      3'h5:    timeStamp[47:40] <= rxData;
      3'h6:    timeStamp[55:48] <= rxData;
      default: timeStamp[63:56] <= rxData;
    endcase
  end
end

// Quiet count 1 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    quietCount1In <= 8'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    quietCount1In <= 8'b0;
  else if (rxDataValid && rxQuietCount1En)
    quietCount1In <= rxData;
end

// Quiet period 1 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    quietPeriod1In <= 8'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    quietPeriod1In <= 8'b0;
  else if (rxDataValid && rxQuietPeriod1En)
    quietPeriod1In <= rxData;
end

// Quiet duration 1 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    quietDuration1In <= 16'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    quietDuration1In <= 16'b0;
  else if (rxDataValid && rxQuietDuration1En)
  begin
    if (stateByteCnt[0])
      quietDuration1In[15:8] <= rxData;
    else
      quietDuration1In[7:0]  <= rxData;
  end
end

// Quiet offset 1 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    quietOffset1In <= 16'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    quietOffset1In <= 16'b0;
  else if (rxDataValid && rxQuietOffset1En)
  begin
    if (stateByteCnt[0])
      quietOffset1In[15:8] <= rxData;
    else
      quietOffset1In[7:0]  <= rxData;
  end
end

`ifdef RW_BFMEE_EN
// BFR-Poll Feedback Segment Retransmission Bitmap sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxBFRPollSegRet <= 8'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1)) // Synchronous Reset
    rxBFRPollSegRet <= 8'b0;
  else if (rxDataValid && rxBFRPollSegRetEn)
  begin
    rxBFRPollSegRet <= rxData;
  end
end

// NDP-A Sounding Dialog Token field sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxNDPAHE                <= 1'b0;
    rxNDPASoundingDialToken <= 6'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1)) // Synchronous Reset
  begin
    rxNDPAHE                <= 1'b0;
    rxNDPASoundingDialToken <= 6'b0;
  end
  else if (rxDataValid && rxNDPASndDialTokenEn)
  begin
    rxNDPAHE                <= rxData[1];
    rxNDPASoundingDialToken <= rxData[7:2];
  end
end

// NDP-A STA Info sampling
// Store NDP-A STA Info
//  - if my MAC Address
//  - if my AID
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxNDPAMyStaInfoDone         <= 1'b0;
    rxNDPAAIDMatch              <= 1'b0;
    rxNDPASTAIndexIsFirst       <= 1'b0;
    rxNDPAFeedbackType          <= 2'b0;
    rxNDPARUStartIndex          <= 7'b0;
    rxNDPARUEndIndex            <= 7'b0;
    rxNDPANg                    <= 1'b0;
    rxNDPACodebook              <= 1'b0;
    rxNDPANcIndex               <= 3'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1)) // Synchronous Reset
  begin
    rxNDPAMyStaInfoDone         <= 1'b0;
    rxNDPAAIDMatch              <= 1'b0;
    rxNDPASTAIndexIsFirst       <= 1'b0;
    rxNDPAFeedbackType          <= 2'b0;
    rxNDPARUStartIndex          <= 7'b0;
    rxNDPARUEndIndex            <= 7'b0;
    rxNDPANg                    <= 1'b0;
    rxNDPACodebook              <= 1'b0;
    rxNDPANcIndex               <= 3'b0;
  end
  else if (rxDataValid && (rxNDPASTAInfo1En || rxNDPASTAInfoNEn) && ~rxNDPAMyStaInfoDone)
  begin
    // Byte0 of VHT NDP-A / HE NDP-A
    if (stateByteCnt[1:0]==2'd0)
    begin
      rxNDPAAIDMatch            <= (aid[7:0]==rxData);
    end
    // Byte1 of VHT NDP-A
    else if (stateByteCnt[1:0]==2'd1 && ~rxNDPAHE)
    begin
      rxNDPAAIDMatch            <= rxNDPAAIDMatch && (aid[11:8]==rxData[3:0]);

      if ((rxNDPAAIDMatch && (aid[11:8] == rxData[3:0])) || (addr1Match && rxNDPASTAInfo1En))
      begin
        rxNDPAMyStaInfoDone     <= 1'b1;
        rxNDPASTAIndexIsFirst   <= rxNDPASTAInfo1En;
        rxNDPAFeedbackType      <= {1'b0,rxData[4]};
        rxNDPANcIndex           <= rxData[7:5];
      end
    end
    // Byte1 of HE NDP-A
    else if (stateByteCnt[1:0]==2'd1 && rxNDPAHE)
    begin
      rxNDPAAIDMatch            <= rxNDPAAIDMatch && (aid[10:8] == rxData[2:0]);

      if ((rxNDPAAIDMatch && (aid[10:8] == rxData[2:0])) || (addr1Match && rxNDPASTAInfo1En))
      begin
        rxNDPARUStartIndex[4:0] <= rxData[7:3];
      end
    end
    // Byte2 of HE NDP-A
    else if (stateByteCnt[1:0]==2'd2 && rxNDPAAIDMatch)
    begin
      rxNDPARUStartIndex[6:5]   <= rxData[1:0];
      rxNDPARUEndIndex[5:0]     <= rxData[7:2];
    end
    // Byte3 of HE NDP-A
    else if (stateByteCnt[1:0]==2'd3 && rxNDPAAIDMatch)
    begin
      // Ng, Codebook & NcIndex are valid only in case of broadcast NDPA
      // Feedback type can't be determined when NDPA is not broadcast.
      // ToDo:11ax, update feedback type decoding when clarified in 802.11ax spec.
      rxNDPAMyStaInfoDone       <= 1'b1;
      rxNDPARUEndIndex[6]       <= rxData[0];
      rxNDPAFeedbackType        <= (~addr1Match & rxData[2] & ~rxData[4]) ? 2'd2 : {1'b0,rxData[1]};
      rxNDPANg                  <= rxData[2];
      rxNDPACodebook            <= rxData[4];
      rxNDPANcIndex             <= rxData[7:5];
    end
  end
end
`endif //RW_BFMEE_EN

// rxFrmBodyByte1 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrmBodyByte1 <= 8'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxFrmBodyByte1 <= 8'b0;
  else if (rxDataValid && rxFrmBodyByte1En)
    rxFrmBodyByte1 <= rxData;
end

// rxFrmBodyByte2 sampling
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrmBodyByte2 <= 8'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxDurationIdEnd_p == 1'b1))       // Synchronous Reset
    rxFrmBodyByte2 <= 8'b0;
  else if (rxDataValid && rxFrmBodyByte2En)
    rxFrmBodyByte2 <= rxData;
end


// Trigger frame sampling
triggerDecoder u_triggerDecoder( 
   //Clock and reset
   .macCoreRxClk                  (macCoreRxClk                  ),
   .macCoreClkHardRst_n           (macCoreClkHardRst_n           ),
   .macCoreClkSoftRst_n           (macCoreClkSoftRst_n           ),
   //RX Controller FSM
   .stateByteCnt                  (stateByteCnt                  ),
   .rxTriggerCMNInfoEn            (rxTriggerCMNInfoEn            ),
   .rxTriggerUSRInfoEn            (rxTriggerUSRInfoEn            ),
   .rxTriggerDepUSRInfoEn         (rxTriggerDepUSRInfoEn         ),
   .rxTriggerUSRInfoEnd_p         (rxTriggerUSRInfoEnd_p         ),
   .rxTriggerFound                (rxTriggerFound                ),
   .fcsOkRcved_p                  (fcsOkRcved_p                  ),
   .trigDepUsrInfoLen             (trigDepUsrInfoLen             ),
   .rxTriggerType                 (rxTriggerType                 ),
   .rxTriggerULLength             (rxTriggerULLength             ),
   .rxTriggerCS                   (rxTriggerCS                   ),
   .rxTriggerULBW                 (rxTriggerULBW                 ),
   .rxTriggerGIAndLTFType         (rxTriggerGIAndLTFType         ),
   .rxTriggerHELTFMode            (rxTriggerHELTFMode            ),
   .rxTriggerNumHELTFAndMidPeriod (rxTriggerNumHELTFAndMidPeriod ),
   .rxTriggerULSTBC               (rxTriggerULSTBC               ),
   .rxTriggerLDPCExtraSymbol      (rxTriggerLDPCExtraSymbol      ),
   .rxTriggerAPTxPower            (rxTriggerAPTxPower            ),
   .rxTriggerULPacketExtension    (rxTriggerULPacketExtension    ),
   .rxTriggerULSpatialReuse       (rxTriggerULSpatialReuse       ),
   .rxTriggerDoppler              (rxTriggerDoppler              ),
   .rxTriggerULHESigA2Reserved    (rxTriggerULHESigA2Reserved    ),
   .rxTriggerUSRInfoMatch_p       (rxTriggerUSRInfoMatch_p       ),
   .rxTriggerUSRInfoMatch         (rxTriggerUSRInfoMatch         ),
   .rxTriggerAIDMatch             (rxTriggerAIDMatch             ),
   .rxTriggerAID4095              (rxTriggerAID4095              ),
   .rxTriggerRUAllocation         (rxTriggerRUAllocation         ),
   .rxTriggerULFECCoding          (rxTriggerULFECCoding          ),
   .rxTriggerULMCS                (rxTriggerULMCS                ),
   .rxTriggerULDCM                (rxTriggerULDCM                ),
   .rxTriggerStartingSSNum        (rxTriggerStartingSSNum        ),
   .rxTriggerNumSS                (rxTriggerNumSS                ),
   .rxTriggerULTargetRSSSI        (rxTriggerULTargetRSSSI        ),
   .rxTriggerDepUserInfo          (rxTriggerDepUserInfo          ),
   .rxTriggerULRUType             (rxTriggerULRUType             ),
   //Frame Decoder
   .addr2MatchBssID               (addr2MatchBssID               ),
   .triggerRcved                  (triggerRcved                  ),
   //A-MPDU Deaggregator
   .rxData                        (rxData                        ),
   .rxDataValid                   (rxDataValid                   ),
   //Control and Status Register
   .aid                           (aid                           ),
   .primaryChPosition             (primaryChPosition             ),
   .maxSupportedBW                (maxSupportedBW                ),
   .raRUType                      (raRUType                      ),
   .raRUEnable                    (raRUEnable                    ),
   .maxMCSInHETB                  (maxMCSInHETB                  ),
   .eOCW                          (eOCW                          ),
   .eOCWMin                       (eOCWMin                       ),
   .eOCWIn                        (eOCWIn                        ),
   .eOCWInValid                   (eOCWInValid                   ),
   .ocwLFSR                       (ocwLFSR                       ),
   .ocwLFSRIn                     (ocwLFSRIn                     ),
   .ocwLFSRInValid                (ocwLFSRInValid                ),
   .dopplerSupport                (dopplerSupport                ),
   .dcmSupport                    (dcmSupport                    ),
   .debugPortTriggerDecoder       (debugPortTriggerDecoder       )
);


// my Frame received pulse after the end of ADDR1 field
assign myFrameRcved_p = (rxAddr1End_p || carFrameCtrlEnd_p) &&
                         addr1Match && !qosFrame && !barRcved;

// my QoS Frame received pulse after the end of QoS field
assign myQoSFrameRcved_p = rxQoSEnd_p && addr1Match && 
                           qosFrame;

// Incorrect BAR received pulse after the end of BAR Control field
assign barIncorrectRcved_p = rxBARInfoEnd_p && addr1Match && 
                             barRcved && !compressedBA;

// Outputs to MAC Controller
assign bfrPollRcved_p       = myFrameRcved_p && bfrPollRcved;
`ifdef RW_BFMEE_EN
assign ndpaRcved_p          = (myFrameRcved_p && ndpaRcved) || 
                              (broadCastRcved && ndpaRcved && rxNDPAAIDMatch && rxNDPAStaInfoEnd_p);
`else // RW_BFMEE_EN
assign ndpaRcved_p          = (myFrameRcved_p && ndpaRcved);
`endif // RW_BFMEE_EN
assign ackRcved_p           = myFrameRcved_p && ackRcved;
assign rtsRcved_p           = myFrameRcved_p && rtsRcved;
assign ctsRcved_p           = myFrameRcved_p && ctsRcved;
assign cfEndRcved_p         = myFrameRcved_p && cfEndRcved;
assign baRcved_p            = myFrameRcved_p && baRcved ||
                              rxBACtrlEnd_p  && baRcved && multiSTABA && bcMcRcved;
assign barRcved_p           = rxBARInfoEnd_p && addr1Match &&
                              barRcved && compressedBA;
assign needAckRcved_p       = barIncorrectRcved_p ||
                              (rxAddr3End_p && probRespRcved) || 
                              (rxDurationIdEnd_p && reservedRcved) ||
                             ((myFrameRcved_p || myQoSFrameRcved_p) && 
                              (!bcMcRcved) && (!bcnRcved) && (!probRespRcved) &&
                              (rxAckPolicy == 2'b00) &&
                             ((dataFrame && (subTypeFrame != 4'h5)) ||
                               psPollRcved                          ||
                              (mgmtFrame && !actNoAckRcved)));


assign needHTPAckRcved_p    = // HE MU PPDU or HE SU PPDU with HTP Ack ack policy
                              myQoSFrameRcved_p && rxFormatMod>=4'd5 && rxFormatMod<=4'd7 &&
                              (
                               // QoS Data with HTP-ACK ackPolicy
                               rxAckPolicy==2'b10 && dataFrame && subTypeFrame==4'b1000 ||
                               // QoS Null with HTP-ACK ackPolicy
                               rxAckPolicy==2'b10 && dataFrame && subTypeFrame==4'b1100
                              ) ||
                              // Action Frame receive in an S-MPDU with TRS Control field
                              trsRcved_p && rxSMPDU && mgmtFrame && subTypeFrame==4'b1101;
                              
assign bcMcRcved_p          = myFrameRcved_p && bcMcRcved;
assign bcnRcved_p           = rxAddr3End_p && bssIDMatch &&
                              bcnRcved;
assign probRespRcved_p      = rxAddr3End_p && bssIDMatch &&
                              probRespRcved;
assign notMineRcved         = !addr1Match && !bcnRcved && 
                              !bcMcRcved;
assign notMineRcved_p       = rxAddr1End_p && notMineRcved;
assign unknownRcved_p       = rxDurationIdEnd_p && reservedRcved;
assign notMinePsPollRcved_p = (rxAddr1End_p || carFrameCtrlEnd_p) && psPollRcved &&
                              !addr1Match && !qosFrame;
assign notMineRtsRcved_p    = (rxAddr1End_p || carFrameCtrlEnd_p) && rtsRcved &&
                              !addr1Match && !qosFrame;
`ifdef RW_BFMER_EN
assign bfrRcved_p           = rxFrmBodyByte2End_p && bfrRcved &&
                              addr1Match && !qosFrame;
`endif // RW_BFMER_EN
assign trsRcved_p           = rxHTCtrlEnd_p          && 
                              rxHTCtrl[1:0]==2'b11   && // HE variant HT Control field
                              rxHTCtrl[5:2]==4'b0000 && // TRS Control
                              addr1Match             &&
                              !protocolError         &&
                              rxFormatMod>=4'd5      && // TRS Control field should be 
                              rxFormatMod<=4'd7;        // received in HE PPDU

// Outputs Decryption FSM
assign rxDefKeyID         = rxInitVectorFull[31:30];


// Trigger Received pulse after end of user info fields.
assign triggerRcved_p      = triggerRcved && rxTriggerUSRInfoMatch && rxTriggerUSRInfoEnd_p &&
                             (broadCastRcved || addr1Match) &&
                             !protocolError;
   
// MU-BAR Trigger received pulse after end of Trigger User info field or Trigger Dependent info
// field if present
//   - Only Compressed Block Ack Request is supported
assign muBARTriggerRcved_p = triggerRcved_p &&
                             rxTriggerType==4'd2 &&           /*MU-BAR*/
                             rxTriggerDepUserInfo[2:1]==2'd2; /*Compressed BlockAckReq*/

assign muBARTriggerRcved   = triggerRcved && rxTriggerType==4'd2;


// Not mine MU-RTS received pulse.
assign notMineMURtsRcved_p = triggerRcved && rxTriggerUSRInfoEnd_p && rxTriggerType==4'd3 &&
                             !protocolError && broadCastRcved && !rxTriggerAIDMatch;


// ACK response received, used for RHD mpdu status information in an Rx Exchange
assign ackRespRcved = rxBADly & (ackRcved | baRcved);


///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display Frame type in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// String definition to display the type of frame
reg [23*8:0] rxFrameType_str;

// Type of received frame displayed in a string to easy simulation and debug
always @*
begin
  casex ({subTypeFrame, typeFrame})
    6'b000000  :  rxFrameType_str = {"ASSOCIATION REQUEST"};
    6'b000100  :  rxFrameType_str = {"ASSOCIATION RESPONSE"};
    6'b001000  :  rxFrameType_str = {"REASSOCIATION REQUEST"};
    6'b001100  :  rxFrameType_str = {"REASSOCIATION RESPONSE"};
    6'b001110  :  rxFrameType_str = {"ACTION NO ACK"};
    6'b010000  :  rxFrameType_str = {"PROBE RESPONSE"};
    6'b010100  :  rxFrameType_str = {"PROBE RESPONSE"};
    6'b011000  :  rxFrameType_str = {"TIMING ADVERTISEMENT"};
    6'b100000  :  rxFrameType_str = {"BEACON"};
    6'b100100  :  rxFrameType_str = {"ATIM"};
    6'b101000  :  rxFrameType_str = {"DISSASSOCIATION"};
    6'b101100  :  rxFrameType_str = {"AUTHENTICATION"};
    6'b110000  :  rxFrameType_str = {"DEAUTHENTICATION"};
    6'b110100  :  rxFrameType_str = {"ACTION"};
    6'b001001  :  rxFrameType_str = {"TRIGGER"};
    6'b010001  :  rxFrameType_str = {"BFR-POLL"};
    6'b010101  :  rxFrameType_str = {"NDP-A"};
    6'b011101  :  rxFrameType_str = {"CONTROL WRAPPER"};
    6'b100001  :  rxFrameType_str = {"BA REQUEST"};
    6'b100101  :  rxFrameType_str = {"BA"};
    6'b101001  :  rxFrameType_str = {"PS-POLL"};
    6'b101101  :  rxFrameType_str = {"RTS"};
    6'b110001  :  rxFrameType_str = {"CTS"};
    6'b110101  :  rxFrameType_str = {"ACK"};
    6'b111001  :  rxFrameType_str = {"CF-END"};
    6'b111101  :  rxFrameType_str = {"CF-END+CF-ACK"};
    6'b000010  :  rxFrameType_str = {"DATA"};
    6'b000110  :  rxFrameType_str = {"DATA+CF-ACK"};
    6'b001010  :  rxFrameType_str = {"DATA+CF-POLL"};
    6'b001110  :  rxFrameType_str = {"DATA+CF-ACK+CF-POLL"};
    6'b010010  :  rxFrameType_str = {"Null"};
    6'b010110  :  rxFrameType_str = {"CF-ACK"};
    6'b011010  :  rxFrameType_str = {"CF-POLL"};
    6'b011110  :  rxFrameType_str = {"CF-ACK+CF-POLL"};
    6'b100010  :  rxFrameType_str = {"QoS DATA"};
    6'b100110  :  rxFrameType_str = {"QoS DATA+CF-ACK"};
    6'b101010  :  rxFrameType_str = {"QoS DATA+CF-POLL"};
    6'b101110  :  rxFrameType_str = {"QoS DATA+CF-ACK+CF-POLL"};
    6'b110010  :  rxFrameType_str = {"QoS Null"};
    6'b111010  :  rxFrameType_str = {"QoS CF-POLL"};
    6'b111110  :  rxFrameType_str = {"QoS CF-ACK+CF-POLL"};
      default  :  rxFrameType_str = {"UNKNOWN"};
  endcase
end
`endif // RW_SIMU_ON

endmodule
