//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of rxController module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON    : which creates string signals to display the FSM states on  
//                 the waveform viewer.
//
//    RW_ASSERT_ON  : which enables System Verilog Assertions.
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rxController( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire         macCoreRxClk,            // MAC Core Receive Clock
            input wire         macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain 
                                                        // active low
            input wire         macCoreClkSoftRst_n,     // Soft Reset of the MAC Core Clock domain
                                                        // active low
            ///////////////////////////////////////////////
            //$port_g DMA Engine
            ///////////////////////////////////////////////
            input wire         rxFrmDiscard,            // Discard current frame
            input wire         rxDescAvailable,         // Indicate that the DMA has all the needed desc for the current rxFrame

            ///////////////////////////////////////////////
            //$port_g MAC Controller
            ///////////////////////////////////////////////
            input wire         stopRx_p,                // Stop Rx
            output wire        rxEndOfFrameRC_p,        // End of frame information to macController

            input wire         activeRx,                // Current state Master is active Rx and Rx FSM is not IDLE
            input wire         rxAck,                   // Force receive only ACK after Tx
            input wire         rxCts,                   // Force receive only CTS after RTS Tx
`ifdef RW_BFMER_EN
            input wire         rxBfr,                   // Force receive only BFR after VHT NDP Tx
`endif // RW_BFMER_EN
            input wire         frameExpectingResp,      // Indicates that the received frame expect
                                                        // an immediate response
            input wire         txInProgress,            // Tx is ongoing
            
            output wire        dataFrame,               // Type of frame is data
            output wire        reservedRcved,           // Reserved received
            output wire        triggerRcved_p,          // Trigger Received
            output wire        trsRcved_p,              // Frame with TRS Control field
                                                        // received pulse
            output wire        bfrPollRcved_p,          // BFR-Poll received pulse
            output wire        ndpaRcved_p,             // NDP-A received pulse
            output wire        ackRcved_p,              // ACK received
            output wire        rtsRcved_p,              // RTS received
            output wire        ctsRcved_p,              // CTS received
            output wire        cfEndRcved_p,            // CF-End received
            output wire        baRcved_p,               // BA received
            output wire        barRcved_p,              // BAR received
            output wire        needAckRcved_p,          // Data or management received with ACK needed
            output wire        needHTPAckRcved_p,       // Received frame that need an HTP Ack 
            output wire        bcMcRcved_p,             // Broadcast/Multicast received
            output wire        bcnRcved_p,              // Beacon received
            output wire        probRespRcved_p,         // Probe response received
            output wire        notMineRcved_p,          // Frame received but not mine
            output wire        unknownRcved_p,          // Unknown frame received
            output wire        macHeaderCompleted,      // End of MAC Header reception
            output wire        notMinePsPollRcved_p,    // PS-Poll received but not mine
            output wire        notMineRtsRcved_p,       // RTS received but not mine
            output wire        notMineMURtsRcved_p,     // MU-RTS received but not mine
`ifdef RW_BFMER_EN
            output wire        bfrRcved_p,              // BFR received pulse 
`endif // RW_BFMER_EN
            output wire [1:0]  rxFCType,                // Indicates the fcType field of the received frame
            output wire        correctRcved_p,          // Frame received correctly
            output wire        incorrectRcved_p,        // Frame not received correctly
            
            output wire [47:0] rxAddr2,                 // Receiver addr
            output wire        rxAMSDUPresent,          // A-MSDU Present
            output wire [1:0]  rxAckPolicy,             // Ack policy (Ack, No Ack, BA)
            output wire [3:0]  rxTID,                   // TID
            output wire [31:0] rxHTCtrl,                // HT control
            output wire        rxRetry,                 // Retry
            output wire        bcMcRcved,               // Broadcast/Multicast
            output wire        notMineRcved,            // Frame received but not mine
            output wire        rxBWSignalingTA,         // Indicate that the received frame has a Bandwidth Signaling TA
            output wire        rxTriggerHWValid_p,      // Trigger HW Valid pulse
            output wire        rxTriggerSWValid_p,      // Trigger SW Valid pulse
            output wire  [3:0] rxTriggerType,           // Trigger frame, Type
            output wire [11:0] rxTriggerULLength,       // Trigger frame, UL Length
            output wire        rxTriggerCS,             // Trigger frame, CS
            output wire  [1:0] rxTriggerULBW,           // Trigger frame, UL BW
            output wire  [1:0] rxTriggerGIAndLTFType,   // Trigger frame, GI And LTF Type
            output wire        rxTriggerHELTFMode,      // Trigger frame, HE-LTF Mode
            output wire  [2:0] rxTriggerNumHELTFAndMidPeriod,// Trigger frame, Number of HE-LTF
                                                             // symbols And Midamble Periodicity
            output wire        rxTriggerULSTBC,         // Trigger frame, UL STBC
            output wire        rxTriggerLDPCExtraSymbol,// Trigger frame, LDPC Extra Symbol Segment
            output wire  [5:0] rxTriggerAPTxPower,      // Trigger frame, AP TX Power
            output wire  [2:0] rxTriggerULPacketExtension,// Trigger frame, UL Packet Extension
            output wire [15:0] rxTriggerULSpatialReuse, // Trigger frame, UL Spatial Reuse
            output wire        rxTriggerDoppler,        // Trigger frame, Doppler
            output wire  [8:0] rxTriggerULHESigA2Reserved, // Trigger frame, UL HE-SIG-A2 Reserved
            output wire  [7:0] rxTriggerRUAllocation,   // Trigger frame, RU Allocation
            output wire        rxTriggerULFECCoding,    // Trigger frame, UL Coding Type
            output wire  [3:0] rxTriggerULMCS,          // Trigger frame, UL MCS
            output wire        rxTriggerULDCM,          // Trigger frame, UL DCM
            output wire  [2:0] rxTriggerStartingSSNum,  // Trigger frame, Starting Spatial 
                                                        // Stream
            output wire  [2:0] rxTriggerNumSS,          // Trigger frame, Number Of Spatial
                                                        // Streams
            output wire  [6:0] rxTriggerULTargetRSSSI,  // Trigger frame, UL Target RSSI
            output wire [79:0] rxTriggerDepUserInfo,    // Trigger frame, Trigger Dependent User
                                                        // Info Field
            output wire        rxTriggerULRUType,       // Trigger Frame, RU Type
            
            ///////////////////////////////////////////////
            //$port_g Backoff Engine
            ///////////////////////////////////////////////
            input wire         backOffDone_p,           // Backoff completed
            input wire [2:0]   activeAC,                // Indicates which AC is currently selected.

            
            ///////////////////////////////////////////////
            //$port_g NAV
            ///////////////////////////////////////////////
            output wire        basicNAVClear_p,         // Clear basic NAV
            output wire        intraNAVClear_p,         // Clear intra NAV
            output wire        basicNAVUpdate_p,        // Update basic NAV with Frame duration
            output wire        intraNAVUpdate_p,        // Update intra NAV with Frame duration
            output wire [15:0] rxFrameDuration,         // Frame duration
            output wire [14:0] navUpdateDuration,       // Duration for the NAV update
            
            ///////////////////////////////////////////////
            //$port_g MAC Timer unit
            ///////////////////////////////////////////////
            output wire        dtimUpdate_p,            // Update DTIM element
            output wire [7:0]  dtimCnt,                 // DTIM counter
            
            output wire        tsfUpdate_p,             // Update TSF
            output wire        tsfOffsetUpdate_p,       // Update TSF Offset
            output wire [63:0] timeStamp,               // Timestamp
            
            output wire        olbcUpdate_p,            // Update OLBC counter

            output wire        lastRxWrong,             // Last received packet wrong

            ///////////////////////////////////////////////
            //$port_g Encryption Engine
            ///////////////////////////////////////////////
            input wire         decryptPassed_p,         // Indicates decryption was a success
            input wire         decryptFailed_p,         // Indicates decryption was a failure
            
            input wire [7:0]   plainDataOut,            // Decrypted data for writing into RX FIFO
            input wire         plainDataOutValid,       // Decrypted data valid
            
            input wire         plainDataOutEnd_p,       // End of Decrypted data pulse

            
            output wire        rxControlIdle,           // RX Controller FSM in idle
            output wire [15:0] rxFrmCtrl,               // Frame control field      
            output wire [47:0] rxAddr3,                 // Address 3 field      
            output wire [47:0] rxAddr4,                 // Address 4 field                
            output wire [15:0] rxQoSControl,            // QoS field
            output wire [15:0] rxSeqCtrl,               // Sequence control
`ifdef RW_WAPI_EN
            output wire [7:0]  rxWAPIKeyIndex,          // WAPI key index field           
            output wire [127:0] rxWAPIPN,               // WAPI packet number(PN) field   
`endif //RW_WAPI_EN
            output wire [47:0] rxAddr1,                 // Addr1
            output wire [23:0] rxInitVector,            // IV
            output wire [47:0] rxTkipSeqCntr,           // TKIP Sequence Counter for TKIP decryption
            output wire [15:0] rxPayLoadLen,            // Frame length including ICV or MIC and FCS
            output reg         rxAddress4Pres,          // Rx flag indicating address 4 is present
            output reg         rxQoSFrame,              // Rx flag indicating the current frame is a QoS Frame
            output wire        plainDataOutReady,       // Flow control for receiving decrypted data
            output wire        rxDataSelect,            // Select data from Encryption Engine
            output wire        rxError_p,               // Indicates received frame can be discarded
            output wire        initEncrRxCntrl_p,       // Starting decryption sequence
            output wire        rxCryptoKeyValid,        // Validating Key

            output wire        encryptedDataInValid,    // Write pulse to Encryption buffer
            output wire [7:0]  encryptedDataIn,         // Write data to Encryption buffer
            input wire         encryptedDataInReady,    // Encryption Buffer full flag
            output wire        encryptedDataInEnd_p,    // Encryption data end pulse

            ///////////////////////////////////////////////
            //$port_g Key Search Engine
            ///////////////////////////////////////////////
            input wire         keyStorageValid_p,       // Key Storage Index Valid
            input wire         keyStorageError_p,       // Error indicating MAC address not found
           
            input wire [9:0]   rxKeyIndexReturn,        // Key Storage Index
            input wire [1:0]   sppKSR,                  // SPP from RAM
            input wire [2:0]   cTypeKSR,                // Cipher Type from RAM
            input wire [1:0]   cLenKSR,                 // Cipher length from RAM
            input wire [3:0]   vlanIDKSR,               // Virtual LAN ID from RAM
            input wire         useDefKeyKSR,            // Use Default Key from RAM

            output wire        indexSearchTrig_p,       // Trigger for searching index from MAC address
            output wire        rxKeySearchIndexTrig_p,  // Trigger for searching parameter from index
            output wire [9:0]  rxKeySearchIndex,        // RAM Index to be searched
            
            ///////////////////////////////////////////////
            //$port_g MAC-PHY interface
            ///////////////////////////////////////////////
            input wire         macPhyIfRxErr_p,         // Rx error from MAC-PHY if. resynchronized
            input wire         macPhyIfRxEndForTiming_p,// Rx end for timing from MAC-PHY if. resynchronized
            input wire         mpIfTxEn,                // Indicates that TX is in progress
            input wire         macPHYIFOverflow,        // Overflow Indication
            
            ///////////////////////////////////////////////
            //$port_g A-MPDU Deaggregator
            ///////////////////////////////////////////////
            input wire [11:0]  rxLegLength,             // Legacy length
            input wire [ 3:0]  rxFormatMod,             // Format and modulation
            input wire         rxAggregation,           // MPDU aggregate
            input wire [ 8:0]  rxPartialAID,            // Partial AID         
            input wire [ 5:0]  rxGroupID,               // Group ID            
            input wire         rxUplinkFlag,            // Indicates if the HE PPDU is addressed to
                                                        // an AP
            input wire         rxVector1Valid_p,        // Rx vector 1 is available
            input wire         rxVector2Valid_p,        // Rx vector 2 is available
            
            input wire [7:0]   rxData,                  // Rx data extracted from MAC-PHY interface FIFO
            input wire         rxDataValid,             // Rx data is valid
            input wire         rxDataStart_p,           // Start of data flow
            input wire         rxDataEnd_p,             // End of data flow
            input wire         rxDataError_p,           // Rx data error (incorrect length)
            input wire         rxEndOfFrameDA_p,        // End of frame information from Deaggregator

            input wire         ampduIncorrectRcved_p,   // A-MPDU not received correctly
            
            input wire [15:0]  rxByteCnt,               // Rx byte counter
            input wire [15:0]  rxMpduLength,            // Rx MPDU full length
            input wire         rxNDP,                   // Indicate that the received frame is a Null Data Packet
            input wire         rxSMPDU,                 // Indicate that the received frame is a 
                                                        // S-MPDU
            input wire  [5:0]  rxBssColor,              // Indicate the BSS color of the AP
            input wire  [6:0]  rxTxopDuration,          // Indicates a duration that is used to
                                                        // update the NAV for this TXOP
            input wire         correctDelimiter,        // Indicate that correct delimiter is found

            input wire  [1:0]  rxAMPDUCnt,              // Counter incremented at each A-MPDU reception
            input wire  [5:0]  rxMPDUCnt,               // Count the number of MPDU detected
            input wire [127:0] rxVector1,               // received Vector 1
            input wire  [63:0] rxVector2,               // received Vector 2
            output wire        rxCntrlReady,            // Rx controller ready to receive data
            
            ///////////////////////////////////////////////
            //$port_g BA Controller
            ///////////////////////////////////////////////
            output wire        psBitmapUpdate_p,        // Update Bitmap request
            output wire        psBitmapDiscard_p,       // Discard Bitmap update request
            output wire        rxQoSEnd_p,              // FSM QoS end pulse
            
            output wire        baEnable,                // Enable BA Controller procedure

            output wire        barRcved,                // BAR received
            
            output wire [11:0] rxSN,                    // Sequence Number
            output wire [11:0] rxBASSN,                 // BA Starting Sequence Number

            output wire        muBARTriggerRcved,       // MU-BAR Trigger Received flag
            output wire        muBARTriggerRcved_p,     // MU-BAR Trigger Received pulse
            
`ifdef RW_BFMEE_EN
            ///////////////////////////////////////////////
            //$port_g BFR Controller
            ///////////////////////////////////////////////
            output wire        rxNDPAValid_p,           // NDP-A Valid pulse
            output wire        rxBFRPollValid_p,        // BFR-Poll Valid pulse
            output wire  [7:0] rxBFRPollSegRet,         // BFR-Poll Feedback Segment Retransmission Bitmap
            output wire        rxNDPAHE,                // NDP-A HE
            output wire  [5:0] rxNDPASoundingDialToken, // NDP-A Sounding Dialog Token Number
            output wire  [1:0] rxNDPAFeedbackType,      // NDP-A Feedback Type
            output wire  [6:0] rxNDPARUStartIndex,      // NDP-A RU Start Index
            output wire  [6:0] rxNDPARUEndIndex,        // NDP-A RU End Index
            output wire        rxNDPANg,                // NDP-A Ng
            output wire        rxNDPACodebook,          // NDP-A Codebook
            output wire  [2:0] rxNDPANcIndex,           // NDP-A Nc Index
            output wire        rxNDPASTAIndexIsFirst,   // NDP-A our STA Info Is first
`endif //RW_BFMEE_EN

            ///////////////////////////////////////////////
            //$port_g Control and Status Register
            ///////////////////////////////////////////////
            input wire [3:0]   currentState,            // Current state of the core
            
            input wire         bssType,                 // BSS Type
            input wire         ap,                      // Access Point
            input wire         tsfMgtDisable,           // Disable HW management of TSF
            
            input wire [31:0]  tsfTimerHigh,            // TSF[63:32]
            input wire [31:0]  tsfTimerLow,             // TSF[31:0]
            
            input wire [47:0]  macAddr,                 // MAC Addr
            input wire [47:0]  macAddrMask,             // MAC Addr mask
            
            input wire [47:0]  bssID,                   // BSSID
            input wire [47:0]  bssIDMask,               // BSSID Mask

            input wire [15:0]  aid,                     // Association ID
            input wire  [5:0]  bssColorCSReg,           // BSS Color
            input wire         bssColorEnCSReg,         // BSS Color Enable
            input wire         partialBSSColorEnCSReg,  // Partial BSS Color Enable
            input wire         bssHE,                   // Connected to an HE AP
            
            input wire         excUnencrypted,          // Exclude unencrypted frames
            input wire         dontDecrypt,             // Don't decrypt frames
            input wire         acceptMulticast,         // Accept Multicast frames
            input wire         acceptBroadcast,         // Accept Broadcast frames
            input wire         acceptOtherBSSID,        // Accept other BSSID frames
            input wire         acceptErrorFrames,       // Accept error frames
            input wire         acceptUnicast,           // Accept Unicast frames
            input wire         acceptMyUnicast,         // Accept frames directed to this device
            input wire         acceptProbeReq,          // Accept Probe Request frames
            input wire         acceptProbeResp,         // Accept Probe Response frames
            input wire         acceptBeacon,            // Accept Beacon frames
            input wire         acceptAllBeacon,         // Accept all Beacon frames 
`ifdef RW_BFMEE_EN
            input wire         acceptBfmeeFrames,       // Accept Beamformee Frames
`endif //RW_BFMEE_EN
            input wire         acceptDecryptErrorFrames,// Accept frames which failed decryption
            input wire         acceptOtherMgmtFrames,   // Accept other management frames
            input wire         acceptBAR,               // Accept Block Ack Request frames
            input wire         acceptBA,                // Accept Block Ack frames
            input wire         acceptNotExpectedBA,     // Accept Block Ack Request frames received after SIFS
            input wire         acceptPSPoll,            // Accept PS-Poll frames
            input wire         acceptRTS,               // Accept RTS frames
            input wire         acceptCTS,               // Accept CTS frames
            input wire         acceptACK,               // Accept ACK frames
            input wire         acceptCFEnd,             // Accept CF-End frames
            input wire         acceptOtherCntrlFrames,  // Accept other control frames
            input wire         acceptData,              // Accept Data frames
            input wire         acceptCFWOData,          // Accept CF WithOut Data frames
            input wire         acceptQData,             // Accept QoS Data frames
            input wire         acceptQCFWOData,         // Accept QoS CF WithOut Data frames
            input wire         acceptQoSNull,           // Accept QoS Null frames
            input wire         acceptOtherDataFrames,   // Accept other Data frames
            input wire         acceptUnknown,           // Accept unknown frames
            input wire         acceptTriggerHWFrames,   // Accept Trigger frame with HW response
            input wire         acceptTriggerSWFrames,   // Accept Trigger frame with SW response
            input wire         acceptAllTriggerFrames,  // Accept All Trigger frames

            input wire         dataMHStoredwithPld,     // Data Frame Mac Header Stored With Payload
            input wire         mgtMHStoredwithPld,      // Management Frame Mac Header Stored With
                                                        // Payload
            input wire         ctrlMHStoredwithPld,     // Control Frame Mac Header Stored With
                                                        // Payload
            input wire   [1:0] dataFrmWrapMode,         // Data Frame Wrap Mode
            input wire   [1:0] mgtFrmWrapMode,          // Management Frame Wrap Mode
            input wire   [1:0] ctrlFrmWrapMode,         // Control Frame Wrap Mode
            input wire         disableRxBuffer2,        // Disable Rx Buffer 2

            output wire [7:0]  quietCount1In,           // Quiet Count 1
            output wire [7:0]  quietPeriod1In,          // Quiet Period 1
            output wire [15:0] quietDuration1In,        // Quiet Duration 1
            output wire [15:0] quietOffset1In,          // Quiet Offset 1
            output wire        quietElement1InValid,    // Indicates that the Quiet 1 registers have to be updated with Quiet 1 In elements

            output wire [7:0]  dtimPeriodIn,            // DTIM period
            output wire        dtimPeriodInValid,       // Indicates that the dtimPeriod register has to be updated with dtimPeriodIn
 
            input wire         dynBWEn,                 // Enable the Dynamic Bandwidth

            input wire   [2:0] primaryChPosition,       // Primary Channel Position
            input wire   [1:0] maxSupportedBW,          // max Supported BW
            input wire         raRUType,                // Random Access RU Type
            input wire         raRUEnable,              // Random Access RU Enable
            input wire   [3:0] maxMCSInHETB,            // Max MCS in HE TB
            input wire   [2:0] eOCW,                    // Exposent of OFDMA Contention Window
            input wire   [2:0] eOCWMin,                 // Exposent of OFDMA Contention Window Minimum
            output wire  [2:0] eOCWIn,                  // Exposent of OFDMA Contention Window In
            output wire        eOCWInValid,             // Exposent of OFDMA Contention Window In Valid
            input  wire  [6:0] ocwLFSR,                 // OFDMA Contention Window LFSR
            output wire  [6:0] ocwLFSRIn,               // OFDMA Contention Window LFSR In
            output wire        ocwLFSRInValid,          // OFDMA Contention Window LFSR In Valid
            input wire         dopplerSupport,          // Doppler Support
            input wire         dcmSupport,              // Dual Carrier Modulation Support
            
            output wire [5:0]  rxControlLs,             // RX Controller latched state
            output wire [5:0]  rxControlCs,             // RX Controller current state
            
            ///////////////////////////////////////////////
            //$port_g RX FIFO
            ///////////////////////////////////////////////
            input wire         rxFIFOAlmostFull,        // RX FIFO is almost full
            input wire         rxFIFOFull,              // RX FIFO is full
            
            output wire [3:0]  rxFIFOWrTag,             // RX Tag FIFO write data
            output wire [31:0] rxFIFOWrData,            // RX FIFO write data
            output wire        rxFIFOWrite,             // RX FIFO write enable
            
            ///////////////////////////////////////////////
            //$port_g Interrupt Controller
            ///////////////////////////////////////////////
            output wire        rxFIFOOverFlow,          // RX FIFO overflow
            
            ///////////////////////////////////////////////
            //$port_g HW Error detected
            ///////////////////////////////////////////////
            input wire         hwErr,                   // HW error detected
            
            ///////////////////////////////////////////////
            //$port_g FCS
            ///////////////////////////////////////////////
            input wire         fcsOk,                   // FCS is ok
            
            output wire        fcsStartRx_p,            // Start FCS calculation
            output wire        fcsEnableRx,             // Enable FCS

            ///////////////////////////////////////////////
            //$port_g txParameters Cache
            ///////////////////////////////////////////////
            input wire         rxBA,                    // Indicate current tx frame is a single VHT Mpdu

            ///////////////////////////////////////////////
            //$port_g SR Controller
            ///////////////////////////////////////////////
            output wire        rtsRcved,                // RTS Received
            output wire        ctsRcved,                // CTS Received
            output wire        ackRcved,                // ACK Received
            output wire        baRcved,                 // BA Received
            output wire        ftmRcved,                // FTM received
`ifdef  RW_BFMEE_EN
            output wire        ndpaRcved,               // NPD-A Received
`endif//RW_BFMEE_EN
            output wire        publicActionRcved,       // Public Action frame received
            output wire  [5:0] partialBSSID,            // BSSID[44:39] of the received Frame
            output wire        bssIDFrame,              // Frame contains BSSID field
            output wire        fcsOkRcved_p,            // Frame received with correct FCS 
            output wire        interBSSRcved,           // The received frame is classified as an
                                                        // inter-BSS frame

            ///////////////////////////////////////////////
            //$port_g Debug ports
            ///////////////////////////////////////////////
            output wire [15:0] debugPortRxController,    // Debug port for validation
            output wire  [6:0] debugPortAMPDURxC,
            output wire [15:0] debugPortRxFrameDebug1,   // Debug port for validation
            output wire [15:0] debugPortRxFrameDebug2,   // Debug port for validation
            output wire [15:0] debugPortrxFIFOCntrl,     // Debug port for validation
            output wire [15:0] debugPortDecryptFsm,      // Debug port for validation
            output wire [15:0] debugPortTriggerDecoder   // Debug port for validation
            );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
wire [63:0]  tsf;                     // TSF

wire         mgmtFrame;               // Management frame
wire         ctrlFrame;               // Control frame
wire         extnFrame;               // Extension frame
wire         qosFrame;                // QoS frame
wire         htcFrame;                // HT or VHT frame with HTC field

wire         broadCastRcved;          // Broadcast frame

wire         bcnRcved;                // Beacon
wire         probRespRcved;           // Probe response

wire         triggerRcved;            // Trigger Frame
wire         triggerHWRcved;          // Trigger frame received with HW response expected
wire         triggerSWRcved;          // Trigger frame received with SW response expected
`ifdef RW_BFMEE_EN
wire         bfrPollRcved;            // BFR-Poll
`endif //RW_BFMEE_EN
`ifdef RW_BFMER_EN
wire         bfrRcved;            // BFR
`endif //RW_BFMER_EN
wire         ctrlWrapRcved;           // Control Wrapper
wire         psPollRcved;             // PS-Poll
wire         cfEndRcved;              // CF-End

wire         ctrlWrapFlag;            // Control Wrapper flag
wire         ctrlWrapError_p;         // Control wrapper error pulse

wire         dataRcved;               // Data
wire         qosDataRcved;            // QoS Data
wire         qosNullRcved;            // QoS null

wire         otherCtrlRcved;          // Other control frame

wire         basicBA;                 // Basic BA
wire         compressedBA8;           // Compressed BA with 08 Bytes Bitmap
wire         compressedBA32;          // Compressed BA with 32 Bytes Bitmap

wire         toDS;                    // To DS
wire         fromDS;                  // From DS
wire         pwrMgt;                  // Power management
wire         protectedBit;            // Protected bit

wire         acceptRcved;             // Accept current frame

wire         addr1Match;              // ADDR1 match
wire         bssIDMatch;              // BSSID match
wire         wildcardBSSID;           // Wildcard BSSID
`ifdef RW_BFMEE_EN
wire         rxNDPAAIDMatch;          // NDPA AID12 match
`endif //RW_BFMEE_EN
wire         addr1MatchBssID;         // ADDR1 is equal to the BSS ID
wire         addr2MatchBssID;         // ADDR2 is equal to the BSS ID
wire   [7:0] trigDepUsrInfoLen;       // Length of the Trigger Dependent
                                      // User Info Field
wire         rxTriggerUSRInfoMatch;   // Trigger: Valid user info field
wire         rxTriggerAIDMatch;       // Trigger: AID12 Match
wire         rxTriggerAID4095;        // Trigger: AID12 is 4095

wire         rxFrmCtrlEn;             // Frame control field
wire         rxCarFrmCtrlEn;          // Carried Frame control field
wire         rxDurationIdEn;          // Duration / ID field
wire         rxAddr1En;               // ADDR1 field
wire         rxAddr2En;               // ADDR2 field
wire         rxAddr3En;               // ADDR3 field
wire         rxQoSCtrlEn;             // QoS control field
wire         rxHTCtrlEn;              // HT control field
wire         rxSeqCtrlEn;             // Sequence control field
wire         rxBACtrlEn;              // BA control field
wire         rxBASeqCtrlEn;           // BA Sequence control field
wire         rxInitVectorEn;          // IV field
wire         rxExtInitVectorEn;       // Extended IV field
wire         rxDtimCntEn;             // DTIM counter field
wire         rxDtimPeriodEn;          // DTIM period field
wire         rxTimeStampEn;           // Timestamp field
wire         rxQuietCount1En;         // Quiet Count 1 field
wire         rxQuietPeriod1En;        // Quiet Period 1 field
wire         rxQuietDuration1En;      // Quiet Duration 1 field
wire         rxQuietOffset1En;        // Quiet Offset 1 field
`ifdef RW_BFMEE_EN
wire         rxBFRPollSegRetEn;       // BFR-Poll Feedback Segment Retransmission Bitmap Field
wire         rxNDPASndDialTokenEn;    // NDP-A Sounding Dialog Token field
wire         rxNDPASTAInfo1En;        // NDP-A STA Info 1 field
wire         rxNDPASTAInfoNEn;        // NDP-A STA Info N field
`endif //RW_BFMEE_EN
wire         rxTriggerCMNInfoEn;      // Trigger Common Info field
wire         rxTriggerUSRInfoEn;      // Trigger User Info field
wire         rxTriggerDepUSRInfoEn;   // Trigger Dependent User Info field
wire         rxFrmBodyByte1En;
wire         rxFrmBodyByte2En;
`ifdef RW_BFMER_EN
wire         rxFrmBodyByte2End_p;
`endif //RW_BFMER_EN

wire [7:0]   stateByteCnt;            // Byte counter by states

wire         fcsCheck_p;              // FSM FCS check pulse
wire         rxDurationIdEnd_p;       // FSM Duration / ID end pulse
wire         rxAddr1End_p;            // FSM ADDR1 end pulse
wire         rxAddr2End_p;            // FSM ADDR1 end pulse
wire         carFrameCtrlEnd_p;       // FSM Carried frame control end pulse
wire         rxAddr3End_p;            // FSM ADDR3 end pulse
wire         rxSeqCtrlEnd_p;          // FSM Sequence Control end pulse
wire         rxHTCtrlEnd_p;           // FSM HT Control end pulse
wire         rxBACtrlEnd_p;           // FSM BA Control end pulse
wire         rxBARInfoEnd_p;          // FSM BAR Info end pulse
`ifdef RW_BFMEE_EN
wire         rxNDPAStaInfoEnd_p;      // FSM NDPA STA Info end pulse
`endif //RW_BFMEE_EN
wire         rxTriggerUSRInfoEnd_p;   // FSM Trigger User Info end pulse

wire         rxFIFODone_p;            // RX FIFO status done
wire         rxEndOfFrame;            // RX end of frame holded

wire         decrStatusValid_p;        // Decryption status valid pulse

wire         frmSuccessfulRcved;      // Frame successful for RX DMA
wire         fcsErrorRcved;           // FCS Error
wire         phyErrorRcved;           // PHY Error
wire         undefErrorRcved;         // Undefined Error
wire         rxFIFOOverFlowError;     // RX FIFO overflow
wire         decryptionError;         // Decryption Error
wire [3:0]   decryptionType;          // Decryption Type
wire         frmBody;                 // Frame body completion

wire         startMacHdr;             // Start of the MAC Header
wire         macHdr;                  // MAC Header processing
wire         endMacHdr_p;             // End of the MAC Header pulse
wire         discardFrm;              // End of frame with discard command

wire         writeTrailer;            // Write trailer info
wire         rxTriggerSWValid;        // Valid trigger frame with SW response flag


wire         rxControlToError_p;      // RX FSM goes to error
wire         rxControlToFrmBody_p;    // RX FSM goes to frame body
wire         acceptProtected;         // Accept protected frame
wire         rxInitVectorEnd_p;       // FSM IV end pulse
wire         rxExtIV_p;               // indicate if there is a ExtIV field
wire         rxIndexSearchDone;       // Key search Engine done searching
wire         rxDefaultKeySearchDone;  // Default Key search Engine done searching
wire         searchMACAddr_p;         // Request a Key search Engine search based on MAC addr (requested by FSm after reception of ADDR2 if the frame is accepted or BAR)
wire [1:0]   rxDefKeyID;              // Default Key ID
wire         keyStorageValid;         // ADDR2 has been found in KSR


wire         rxMoreData;              // More data
wire         rxMoreFrag;              // More fragment


wire [3:0]   subTypeFrame;            // Subtype
wire [1:0]   typeFrame;               // Type
wire         protocolError;           // Protocol error
wire         ackRespRcved;            // Ack Response received

wire [31:0]  rxExtInitVector;         // Extended IV

wire         keyIndexCapture_p;       // Key Index returned by keySearch capture pulse

wire         fcsOknDiscard;             // FCS OK and no discard frame from DMA
wire         acceptErrorFramesnDiscard; // accept  Error frame and no discard frame from DMA

wire         txInProgressDly;         // captured of txInProgress on rxVector1_valid
wire         rxControlBackToIdle_p;   // FSM changed to idle state
wire         rxPayloadEnd_p;          // Last Byte of Frame Body

wire         rxBADly;                 // capture of rxBA

wire         acceptError_p;           // Pulse to indicate that writeTrailer to accept Error frame (others than FCS error)


wire         rxCntrlReadyInt;         // Rx controller ready to receive data
wire         decryptCntrlReady;       // Indicate that the Crypto is ready to receive data

wire         rxAddr4En;               // ADDR4 field          
`ifdef RW_WAPI_EN
wire         rxWAPIKeyIdxEn;          // WAPI KeyIdx field    
wire         rxWAPIPNEn;              // WAPI PN field        
`endif //RW_WAPI_EN
wire         decryptionCheckOk;       // Indicate that the decryption Check is ok
wire         rxTriggerFound;          // Indicate that a valid trigger has been found
wire         payloadEnd_p;            // Last Byte of Frame Body (without encryption tailer)

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign fcsOknDiscard             = (fcsOk && !rxFrmDiscard);
assign acceptErrorFramesnDiscard = (acceptErrorFrames && !rxFrmDiscard);

assign rxCntrlReady = decryptCntrlReady & rxCntrlReadyInt;

assign rxFCType = typeFrame;


  
// RX Control FSM
rxControllerFsm U_rxControllerFsm (
        // Clock and reset
        .macCoreRxClk             (macCoreRxClk),
        .macCoreClkHardRst_n      (macCoreClkHardRst_n),
        .macCoreClkSoftRst_n      (macCoreClkSoftRst_n),
        // DMA Engine
        .rxFrmDiscard             (rxFrmDiscard),
        .rxDescAvailable          (rxDescAvailable),
        // MAC Controller
        .stopRx_p                 (stopRx_p),
        .frameExpectingResp       (frameExpectingResp),
        .rxCts                    (rxCts),
`ifdef RW_BFMER_EN
        .rxBfr                    (rxBfr),
`endif // RW_BFMER_EN
        .rxAck                    (rxAck),
        .correctRcved_p           (correctRcved_p),
        .incorrectRcved_p         (incorrectRcved_p),
    		.rxEndOfFrameRC_p         (rxEndOfFrameRC_p),
        .rxTriggerHWValid_p       (rxTriggerHWValid_p),
        .rxTriggerSWValid_p       (rxTriggerSWValid_p),
        // Backoff Engine
        .backOffDone_p            (backOffDone_p),
        // NAV
        .basicNAVClear_p          (basicNAVClear_p),
        .intraNAVClear_p          (intraNAVClear_p),
        .basicNAVUpdate_p         (basicNAVUpdate_p),
        .intraNAVUpdate_p         (intraNAVUpdate_p),
        .rxFrameDuration          (rxFrameDuration),
        .navUpdateDuration        (navUpdateDuration),
        // MAC Timer unit
        .dtimUpdate_p             (dtimUpdate_p),
        .tsfUpdate_p              (tsfUpdate_p),
        .tsfOffsetUpdate_p        (tsfOffsetUpdate_p),
        .lastRxWrong              (lastRxWrong),
        // Encryption Engine
        .decryptPassed_p          (decryptPassed_p),
        .decryptFailed_p          (decryptFailed_p),
        .plainDataOutEnd_p        (plainDataOutEnd_p),
        .rxPayLoadLen             (rxPayLoadLen),
        // Key Search Engine
        .sppKSR                   (sppKSR),
        .cTypeKSR                 (cTypeKSR),
        .cLenKSR                  (cLenKSR),
        // Decryption FSM
        .rxDataSelect             (rxDataSelect),
        .rxControlToError_p       (rxControlToError_p),
        .rxControlToFrmBody_p     (rxControlToFrmBody_p),
        .decrStatusValid_p        (decrStatusValid_p),
        .acceptProtected          (acceptProtected),
        .rxInitVectorEnd_p        (rxInitVectorEnd_p),
        .rxExtIV_p                (rxExtIV_p),
        .rxIndexSearchDone        (rxIndexSearchDone),
        .rxDefaultKeySearchDone   (rxDefaultKeySearchDone),
        .searchMACAddr_p          (searchMACAddr_p), 
        .payloadEnd_p             (payloadEnd_p),
        // MAC-PHY IF
        .mpIfTxEn                 (mpIfTxEn),
        .macPhyIfRxErr_p          (macPhyIfRxErr_p),
        .macPHYIFOverflow         (macPHYIFOverflow),
        .macPhyIfRxEndForTiming_p (macPhyIfRxEndForTiming_p),
        // A-MPDU Deaggregator
        .rxVector1Valid_p         (rxVector1Valid_p),
        .rxFormatMod              (rxFormatMod),
        .rxAggregation            (rxAggregation),
        .rxData                   (rxData),
        .rxDataValid              (rxDataValid),
        .rxDataStart_p            (rxDataStart_p),
        .rxDataEnd_p              (rxDataEnd_p),
        .rxDataError_p            (rxDataError_p),
        .ampduIncorrectRcved_p    (ampduIncorrectRcved_p),
        .rxByteCnt                (rxByteCnt),
        .rxNDP                    (rxNDP),
        .rxPartialAID             (rxPartialAID),
        .rxGroupID                (rxGroupID),
        .rxUplinkFlag             (rxUplinkFlag),
        .rxBssColor               (rxBssColor),
        .rxTxopDuration           (rxTxopDuration),
        .rxCntrlReady             (rxCntrlReadyInt),
    		.rxEndOfFrameDA_p         (rxEndOfFrameDA_p),
        // Frame decoder
        .protocolError            (protocolError),
        .mgmtFrame                (mgmtFrame),
        .ctrlFrame                (ctrlFrame),
        .dataFrame                (dataFrame),
        .extnFrame                (extnFrame),
        .qosFrame                 (qosFrame),
        .htcFrame                 (htcFrame),
        .bssIDFrame               (bssIDFrame),
        .broadCastRcved           (broadCastRcved),
        .bcMcRcved                (bcMcRcved),
        .bcnRcved                 (bcnRcved),
        .probRespRcved            (probRespRcved),
        .triggerRcved             (triggerRcved),
        .triggerHWRcved           (triggerHWRcved),
        .triggerSWRcved           (triggerSWRcved),
`ifdef RW_BFMEE_EN
        .bfrPollRcved             (bfrPollRcved),
        .ndpaRcved                (ndpaRcved),
`endif //RW_BFMEE_EN
`ifdef RW_BFMER_EN
        .bfrRcved                 (bfrRcved),
`endif //RW_BFMER_EN
        .ctrlWrapRcved            (ctrlWrapRcved),
        .baRcved                  (baRcved),
        .barRcved                 (barRcved),
        .psPollRcved              (psPollRcved),
        .rtsRcved                 (rtsRcved),
        .ctsRcved                 (ctsRcved),
        .ackRcved                 (ackRcved),
        .cfEndRcved               (cfEndRcved),
        .ctrlWrapFlag             (ctrlWrapFlag),
        .ctrlWrapError_p          (ctrlWrapError_p),
        .dataRcved                (dataRcved),
        .qosDataRcved             (qosDataRcved),
        .qosNullRcved             (qosNullRcved),
        .otherCtrlRcved           (otherCtrlRcved),
        .muBARTriggerRcved        (muBARTriggerRcved),
        .toDS                     (toDS),
        .fromDS                   (fromDS),
        .protectedBit             (protectedBit),
        .rxAMSDUPresent           (rxAMSDUPresent),
        .basicBA                  (basicBA),
        .compressedBA8            (compressedBA8),
        .compressedBA32           (compressedBA32),
        .acceptRcved              (acceptRcved),
        .addr1Match               (addr1Match),
        .bssIDMatch               (bssIDMatch),
        .wildcardBSSID            (wildcardBSSID),
        .addr1MatchBssID          (addr1MatchBssID),
        .addr2MatchBssID          (addr2MatchBssID),
`ifdef RW_BFMEE_EN
        .rxNDPAAIDMatch           (rxNDPAAIDMatch),
        .rxNDPAHE                 (rxNDPAHE),
`endif //RW_BFMEE_EN
        .trigDepUsrInfoLen        (trigDepUsrInfoLen),
        .rxTriggerType            (rxTriggerType),
        .rxTriggerUSRInfoMatch    (rxTriggerUSRInfoMatch),
        .rxTriggerAIDMatch        (rxTriggerAIDMatch),
        .rxTriggerAID4095         (rxTriggerAID4095),
        .rxAddr1                  (rxAddr1),
        .rxAddr2                  (rxAddr2),
        .rxFrmCtrlEn              (rxFrmCtrlEn),
        .rxCarFrmCtrlEn           (rxCarFrmCtrlEn),
        .rxDurationIdEn           (rxDurationIdEn),
        .rxAddr1En                (rxAddr1En),
        .rxAddr2En                (rxAddr2En),
        .rxAddr3En                (rxAddr3En),
        .rxAddr4En                (rxAddr4En),
`ifdef RW_WAPI_EN
        .rxWAPIKeyIdxEn           (rxWAPIKeyIdxEn),
        .rxWAPIPNEn               (rxWAPIPNEn),
`endif //RW_WAPI_EN
        .rxQoSCtrlEn              (rxQoSCtrlEn),
        .rxHTCtrlEn               (rxHTCtrlEn),
        .rxSeqCtrlEn              (rxSeqCtrlEn),
        .rxBACtrlEn               (rxBACtrlEn),
        .rxBASeqCtrlEn            (rxBASeqCtrlEn),
        .rxInitVectorEn           (rxInitVectorEn),
        .rxExtInitVectorEn        (rxExtInitVectorEn),
        .rxDtimCntEn              (rxDtimCntEn),
        .rxDtimPeriodEn           (rxDtimPeriodEn),
        .rxTimeStampEn            (rxTimeStampEn),
        .rxQuietCount1En          (rxQuietCount1En),
        .rxQuietPeriod1En         (rxQuietPeriod1En),
        .rxQuietDuration1En       (rxQuietDuration1En),
        .rxQuietOffset1En         (rxQuietOffset1En),
`ifdef RW_BFMEE_EN
        .rxBFRPollSegRetEn        (rxBFRPollSegRetEn),
        .rxNDPASndDialTokenEn     (rxNDPASndDialTokenEn),
        .rxNDPASTAInfo1En         (rxNDPASTAInfo1En),
        .rxNDPASTAInfoNEn         (rxNDPASTAInfoNEn),
`endif //RW_BFMEE_EN
        .rxTriggerCMNInfoEn       (rxTriggerCMNInfoEn),
        .rxTriggerUSRInfoEn       (rxTriggerUSRInfoEn),
        .rxTriggerDepUSRInfoEn    (rxTriggerDepUSRInfoEn),
        .rxFrmBodyByte1En         (rxFrmBodyByte1En),
        .rxFrmBodyByte2En         (rxFrmBodyByte2En),
`ifdef RW_BFMER_EN
        .rxFrmBodyByte2End_p      (rxFrmBodyByte2End_p),
`endif //RW_BFMER_EN
        .stateByteCnt             (stateByteCnt),
        .fcsCheck_p               (fcsCheck_p),
        .rxAddr4End_p             (),     
        .rxDurationIdEnd_p        (rxDurationIdEnd_p),
        .rxAddr1End_p             (rxAddr1End_p),
        .rxAddr2End_p             (rxAddr2End_p),
        .carFrameCtrlEnd_p        (carFrameCtrlEnd_p),
        .rxAddr3End_p             (rxAddr3End_p),
        .rxSeqCtrlEnd_p           (rxSeqCtrlEnd_p),
        .rxQoSEnd_p               (rxQoSEnd_p),
        .rxHTCtrlEnd_p            (rxHTCtrlEnd_p),
        .rxBACtrlEnd_p            (rxBACtrlEnd_p),
        .rxBARInfoEnd_p           (rxBARInfoEnd_p),
`ifdef RW_BFMEE_EN
        .rxNDPAStaInfoEnd_p       (rxNDPAStaInfoEnd_p),
`endif //RW_BFMEE_EN
        .rxTriggerUSRInfoEnd_p    (rxTriggerUSRInfoEnd_p),
        .macHeaderCompleted       (macHeaderCompleted),
        .rxControlIdle            (rxControlIdle),
        .decryptionCheckOk        (decryptionCheckOk),
        .rxTriggerFound           (rxTriggerFound),
        // BA Controller
        .psBitmapUpdate_p         (psBitmapUpdate_p),
        .psBitmapDiscard_p        (psBitmapDiscard_p),
        .baEnable                 (baEnable),
`ifdef RW_BFMEE_EN
        // BFR Controller
        .rxNDPAValid_p            (rxNDPAValid_p),
        .rxBFRPollValid_p         (rxBFRPollValid_p),
`endif //RW_BFMEE_EN
        // SR Controller
        .fcsOkRcved_p             (fcsOkRcved_p),
        .interBSSRcved            (interBSSRcved),
        // Control and Status Register
        .currentState             (currentState),
        .bssType                  (bssType),
        .ap                       (ap),
        .tsfMgtDisable            (tsfMgtDisable),
        .bssID                    (bssID),
        .bssColorCSReg            (bssColorCSReg),
        .bssColorEnCSReg          (bssColorEnCSReg),
        .partialBSSColorEnCSReg   (partialBSSColorEnCSReg),
        .bssHE                    (bssHE),
        .dontDecrypt              (dontDecrypt),
        .acceptBroadcast          (acceptBroadcast),
        .acceptErrorFrames        (acceptErrorFramesnDiscard),
`ifdef RW_BFMEE_EN
        .acceptBfmeeFrames        (acceptBfmeeFrames),
`endif //RW_BFMEE_EN
        .acceptDecryptErrorFrames (acceptDecryptErrorFrames),
        .quietElement1InValid     (quietElement1InValid),
        .dtimPeriodInValid        (dtimPeriodInValid),
        .rxControlLs              (rxControlLs),
        .rxControlCs              (rxControlCs),
        // RX FIFO Interface Controller
        .rxFIFOOverFlow           (rxFIFOOverFlow),
        .rxFIFOFull               (rxFIFOFull),
        .rxFIFODone_p             (rxFIFODone_p),
        .rxFIFOWrite              (rxFIFOWrite),
        .txInProgressDly          (txInProgressDly),
        .rxControlBackToIdle_p    (rxControlBackToIdle_p),
        .rxPayloadEnd_p           (rxPayloadEnd_p),
        .frmSuccessfulRcved       (frmSuccessfulRcved),
        .fcsErrorRcved            (fcsErrorRcved),
        .phyErrorRcved            (phyErrorRcved),
        .undefErrorRcved          (undefErrorRcved),
        .rxFIFOOverFlowError      (rxFIFOOverFlowError),
        .decryptionError          (decryptionError),
        .decryptionType           (decryptionType),
        .frmBody                  (frmBody),
        .startMacHdr              (startMacHdr),
        .macHdr                   (macHdr),
        .endMacHdr_p              (endMacHdr_p),
        .discardFrm               (discardFrm),
        .acceptError_p            (acceptError_p),
        .writeTrailer             (writeTrailer),
        .rxTriggerSWValid         (rxTriggerSWValid),
        // HW Error detected
        .hwErr                    (hwErr),
        // FCS
        .fcsOk                    (fcsOknDiscard),
        .fcsStartRx_p             (fcsStartRx_p),
        .fcsEnableRx              (fcsEnableRx)
        );


// Decryption FSM
decryptFsm U_decryptFsm (
        // Clock and reset
        .macCoreRxClk                     (macCoreRxClk),
        .macCoreClkHardRst_n              (macCoreClkHardRst_n),
        .macCoreClkSoftRst_n              (macCoreClkSoftRst_n),
        // A-MPDU Deaggregator
        .rxData                           (rxData),
        .rxDataValid                      (rxDataValid),
        .decryptCntrlReady                (decryptCntrlReady),
        // Encryption Engine
        .rxError_p                        (rxError_p),
        .initEncrRxCntrl_p                (initEncrRxCntrl_p),
        .rxCryptoKeyValid                 (rxCryptoKeyValid),
        .encryptedDataInValid             (encryptedDataInValid),
        .encryptedDataIn                  (encryptedDataIn),
        .encryptedDataInReady             (encryptedDataInReady),
        .encryptedDataInEnd_p             (encryptedDataInEnd_p),
        .plainDataOutReady                (plainDataOutReady),
    	.plainDataOutEnd_p                (plainDataOutEnd_p),
        // Key Search Engine
        .keyStorageValid_p                (keyStorageValid_p),
        .keyStorageError_p                (keyStorageError_p),
        .sppKSR                           (sppKSR),
        .cTypeKSR                         (cTypeKSR),
        .vlanIDKSR                        (vlanIDKSR),
        .useDefKeyKSR                     (useDefKeyKSR),
        .indexSearchTrig_p                (indexSearchTrig_p),
        .rxKeySearchIndexTrig_p           (rxKeySearchIndexTrig_p),
        .rxKeySearchIndex                 (rxKeySearchIndex),
        // RX Controller FSM
        .rxControlIdle                    (rxControlIdle),
        .rxControlToError_p               (rxControlToError_p),
        .rxControlToFrmBody_p             (rxControlToFrmBody_p),
        .decrStatusValid_p                (decrStatusValid_p),
        .acceptProtected                  (acceptProtected),
        .rxFIFOOverFlowError              (rxFIFOOverFlowError),
        .rxInitVectorEnd_p                (rxInitVectorEnd_p),
        .rxExtIV_p                        (rxExtIV_p),
        .rxIndexSearchDone                (rxIndexSearchDone),
        .rxDefaultKeySearchDone           (rxDefaultKeySearchDone),
        .searchMACAddr_p                  (searchMACAddr_p), 
        .decryptionCheckOk                (decryptionCheckOk),
        .payloadEnd_p                     (payloadEnd_p),
        // Frame decoder
        .addr1Match                       (addr1Match),
        .bcMcRcved                        (bcMcRcved),
        .rxDefKeyID                       (rxDefKeyID),
        .keyStorageValid                  (keyStorageValid),
        // RX FIFO Interface Controller
        .rxDataSelect                     (rxDataSelect),
        .keyIndexCapture_p                (keyIndexCapture_p),

        .debugPortDecryptFsm              (debugPortDecryptFsm)
        );


// Frame decoder
frameDecoder U_frameDecoder(
        // Clock and reset
        .macCoreRxClk                  (macCoreRxClk),
        .macCoreClkHardRst_n           (macCoreClkHardRst_n),
        .macCoreClkSoftRst_n           (macCoreClkSoftRst_n),
        // MAC Controller
        .activeRx                      (activeRx),
        .rxCts                         (rxCts),
`ifdef RW_BFMER_EN
        .bfrRcved_p                    (bfrRcved_p),
`endif // RW_BFMER_EN
        .triggerRcved_p                (triggerRcved_p),
        .trsRcved_p                    (trsRcved_p),
        .bfrPollRcved_p                (bfrPollRcved_p),
        .ndpaRcved_p                   (ndpaRcved_p),
        .ackRcved_p                    (ackRcved_p),
        .rtsRcved_p                    (rtsRcved_p),
        .ctsRcved_p                    (ctsRcved_p),
        .cfEndRcved_p                  (cfEndRcved_p),
        .baRcved_p                     (baRcved_p),
        .barRcved_p                    (barRcved_p),
        .needAckRcved_p                (needAckRcved_p),
        .needHTPAckRcved_p             (needHTPAckRcved_p),
        .bcMcRcved_p                   (bcMcRcved_p),
        .bcnRcved_p                    (bcnRcved_p),
        .probRespRcved_p               (probRespRcved_p),
        .notMineRcved_p                (notMineRcved_p),
        .unknownRcved_p                (unknownRcved_p),
        .notMinePsPollRcved_p          (notMinePsPollRcved_p),
        .notMineRtsRcved_p             (notMineRtsRcved_p),
        .notMineMURtsRcved_p           (notMineMURtsRcved_p),
        .rxAddr2                       (rxAddr2),
        .notMineRcved                  (notMineRcved),
        // NAV
        .rxFrameDuration               (rxFrameDuration),
        // MAC Timer unit
        .dtimCnt                       (dtimCnt),
        .timeStamp                     (timeStamp),
        .olbcUpdate_p                  (olbcUpdate_p),
        // Encryption Engine
        .rxAddr3                       (rxAddr3),       
        .rxAddr4                       (rxAddr4),       
        .rxQoSControl                  (rxQoSControl),
`ifdef RW_WAPI_EN
        .rxWAPIKeyIndex                (rxWAPIKeyIndex),
        .rxWAPIPN                      (rxWAPIPN),      
`endif //RW_WAPI_EN
        .rxAddr1                       (rxAddr1),
        .rxInitVector                  (rxInitVector),
        .rxExtInitVector               (rxExtInitVector),
        // Decryption FSM
        .rxDefKeyID                    (rxDefKeyID),
        .keyStorageValid               (keyStorageValid),
        // RX Controller FSM
        .rxFrmCtrlEn                   (rxFrmCtrlEn),
        .rxCarFrmCtrlEn                (rxCarFrmCtrlEn),
        .rxDurationIdEn                (rxDurationIdEn),
        .rxAddr1En                     (rxAddr1En),
        .rxAddr2En                     (rxAddr2En),
        .rxAddr3En                     (rxAddr3En),
        .rxAddr4En                     (rxAddr4En),
`ifdef RW_WAPI_EN
        .rxWAPIKeyIdxEn                (rxWAPIKeyIdxEn),
        .rxWAPIPNEn                    (rxWAPIPNEn),
`endif //RW_WAPI_EN
        .rxQoSCtrlEn                   (rxQoSCtrlEn),
        .rxHTCtrlEn                    (rxHTCtrlEn),
        .rxSeqCtrlEn                   (rxSeqCtrlEn),
        .rxBACtrlEn                    (rxBACtrlEn),
        .rxBASeqCtrlEn                 (rxBASeqCtrlEn),
        .rxInitVectorEn                (rxInitVectorEn),
        .rxExtInitVectorEn             (rxExtInitVectorEn),
        .rxDtimCntEn                   (rxDtimCntEn),
        .rxDtimPeriodEn                (rxDtimPeriodEn),
        .rxTimeStampEn                 (rxTimeStampEn),
        .rxQuietCount1En               (rxQuietCount1En),
        .rxQuietPeriod1En              (rxQuietPeriod1En),
        .rxQuietDuration1En            (rxQuietDuration1En),
        .rxQuietOffset1En              (rxQuietOffset1En),
`ifdef RW_BFMEE_EN
        .rxBFRPollSegRetEn             (rxBFRPollSegRetEn),
        .rxNDPASndDialTokenEn          (rxNDPASndDialTokenEn),
        .rxNDPASTAInfo1En              (rxNDPASTAInfo1En),
        .rxNDPASTAInfoNEn              (rxNDPASTAInfoNEn),
`endif //RW_BFMEE_EN
        .rxTriggerCMNInfoEn            (rxTriggerCMNInfoEn),
        .rxTriggerUSRInfoEn            (rxTriggerUSRInfoEn),
        .rxTriggerDepUSRInfoEn         (rxTriggerDepUSRInfoEn),
        .rxFrmBodyByte1En              (rxFrmBodyByte1En),
        .rxFrmBodyByte2En              (rxFrmBodyByte2En),
`ifdef RW_BFMER_EN
        .rxFrmBodyByte2End_p           (rxFrmBodyByte2End_p),
`endif 
        .stateByteCnt                  (stateByteCnt),
        .fcsCheck_p                    (fcsCheck_p),
        .rxDurationIdEnd_p             (rxDurationIdEnd_p),
        .rxAddr1End_p                  (rxAddr1End_p),
        .rxAddr2End_p                  (rxAddr2End_p),
        .carFrameCtrlEnd_p             (carFrameCtrlEnd_p),
        .rxAddr3End_p                  (rxAddr3End_p),
        .rxSeqCtrlEnd_p                (rxSeqCtrlEnd_p),
        .rxQoSEnd_p                    (rxQoSEnd_p),
        .rxHTCtrlEnd_p                 (rxHTCtrlEnd_p),
        .rxBACtrlEnd_p                 (rxBACtrlEnd_p),
        .rxBARInfoEnd_p                (rxBARInfoEnd_p),
`ifdef RW_BFMEE_EN
        .rxNDPAStaInfoEnd_p            (rxNDPAStaInfoEnd_p),
`endif //RW_BFMEE_EN
        .rxTriggerUSRInfoEnd_p         (rxTriggerUSRInfoEnd_p),
        .fcsOkRcved_p                  (fcsOkRcved_p),
        .rxControlIdle                 (rxControlIdle),
        .rxTriggerFound                (rxTriggerFound),
        .mgmtFrame                     (mgmtFrame),
        .ctrlFrame                     (ctrlFrame),
        .dataFrame                     (dataFrame),
        .extnFrame                     (extnFrame),
        .qosFrame                      (qosFrame),
        .htcFrame                      (htcFrame),
        .bssIDFrame                    (bssIDFrame),
        .broadCastRcved                (broadCastRcved),
        .bcMcRcved                     (bcMcRcved),
        .bcnRcved                      (bcnRcved),
        .probRespRcved                 (probRespRcved),
        .triggerRcved                  (triggerRcved),
        .triggerHWRcved                (triggerHWRcved),
        .triggerSWRcved                (triggerSWRcved),
`ifdef RW_BFMEE_EN
        .bfrPollRcved                  (bfrPollRcved),
        .ndpaRcved                     (ndpaRcved),
`endif //RW_BFMEE_EN
`ifdef RW_BFMER_EN
        .bfrRcved                      (bfrRcved),
`endif //RW_BFMER_EN
        .ftmRcved                      (ftmRcved),
        .publicActionRcved             (publicActionRcved),
        .ctrlWrapRcved                 (ctrlWrapRcved),
        .baRcved                       (baRcved),
        .barRcved                      (barRcved),
        .psPollRcved                   (psPollRcved),
        .rtsRcved                      (rtsRcved),
        .ctsRcved                      (ctsRcved),
        .ackRcved                      (ackRcved),
        .cfEndRcved                    (cfEndRcved),
        .ctrlWrapFlag                  (ctrlWrapFlag),
        .ctrlWrapError_p               (ctrlWrapError_p),
        .dataRcved                     (dataRcved),
        .qosDataRcved                  (qosDataRcved),
        .qosNullRcved                  (qosNullRcved),
        .reservedRcved                 (reservedRcved),
        .otherCtrlRcved                (otherCtrlRcved),
        .toDS                          (toDS),
        .fromDS                        (fromDS),
        .rxMoreFrag                    (rxMoreFrag),
        .rxRetry                       (rxRetry),
        .pwrMgt                        (pwrMgt),
        .rxMoreData                    (rxMoreData),
        .protectedBit                  (protectedBit),
        .rxAMSDUPresent                (rxAMSDUPresent),
        .rxAckPolicy                   (rxAckPolicy),
        .rxTID                         (rxTID),
        .rxBWSignalingTA               (rxBWSignalingTA),
        .rxHTCtrl                      (rxHTCtrl),
        .basicBA                       (basicBA),
        .compressedBA8                 (compressedBA8),
        .compressedBA32                (compressedBA32),
        .acceptRcved                   (acceptRcved),
        .addr1Match                    (addr1Match),
        .bssIDMatch                    (bssIDMatch),
        .wildcardBSSID                 (wildcardBSSID),
`ifdef RW_BFMEE_EN
        .rxNDPAAIDMatch                (rxNDPAAIDMatch),
`endif //RW_BFMEE_EN
        .addr1MatchBssID               (addr1MatchBssID),
        .addr2MatchBssID               (addr2MatchBssID),
        .trigDepUsrInfoLen             (trigDepUsrInfoLen),
        .rxTriggerType                 (rxTriggerType),
        .rxTriggerULLength             (rxTriggerULLength),
        .rxTriggerCS                   (rxTriggerCS),
        .rxTriggerULBW                 (rxTriggerULBW),
        .rxTriggerGIAndLTFType         (rxTriggerGIAndLTFType),
        .rxTriggerHELTFMode            (rxTriggerHELTFMode),
        .rxTriggerNumHELTFAndMidPeriod (rxTriggerNumHELTFAndMidPeriod),
        .rxTriggerULSTBC               (rxTriggerULSTBC),
        .rxTriggerLDPCExtraSymbol      (rxTriggerLDPCExtraSymbol),
        .rxTriggerAPTxPower            (rxTriggerAPTxPower),
        .rxTriggerULPacketExtension    (rxTriggerULPacketExtension),
        .rxTriggerULSpatialReuse       (rxTriggerULSpatialReuse),
        .rxTriggerDoppler              (rxTriggerDoppler),
        .rxTriggerULHESigA2Reserved    (rxTriggerULHESigA2Reserved),
        .rxTriggerUSRInfoMatch         (rxTriggerUSRInfoMatch),
        .rxTriggerAIDMatch             (rxTriggerAIDMatch),
        .rxTriggerAID4095              (rxTriggerAID4095),
        .rxTriggerRUAllocation         (rxTriggerRUAllocation),
        .rxTriggerULFECCoding          (rxTriggerULFECCoding),
        .rxTriggerULMCS                (rxTriggerULMCS),
        .rxTriggerULDCM                (rxTriggerULDCM),
        .rxTriggerStartingSSNum        (rxTriggerStartingSSNum),
        .rxTriggerNumSS                (rxTriggerNumSS),
        .rxTriggerULTargetRSSSI        (rxTriggerULTargetRSSSI),
        .rxTriggerDepUserInfo          (rxTriggerDepUserInfo),
        .rxTriggerULRUType             (rxTriggerULRUType),
        // BA Controller
        .rxSeqCtrl                     (rxSeqCtrl),
        .rxBASSN                       (rxBASSN),
        .muBARTriggerRcved             (muBARTriggerRcved),
        .muBARTriggerRcved_p           (muBARTriggerRcved_p),
        // A-MPDU Deaggregator
        .rxFormatMod                   (rxFormatMod),
        .rxMpduLength                  (rxMpduLength),        
        .rxSMPDU                       (rxSMPDU),
        .rxData                        (rxData),
        .rxDataValid                   (rxDataValid),
`ifdef RW_BFMEE_EN
        // BFR Controller
        .rxBFRPollSegRet               (rxBFRPollSegRet),
        .rxNDPAHE                      (rxNDPAHE),
        .rxNDPASoundingDialToken       (rxNDPASoundingDialToken),
        .rxNDPAFeedbackType            (rxNDPAFeedbackType),
        .rxNDPARUStartIndex            (rxNDPARUStartIndex),
        .rxNDPARUEndIndex              (rxNDPARUEndIndex),
        .rxNDPANg                      (rxNDPANg),
        .rxNDPACodebook                (rxNDPACodebook),
        .rxNDPANcIndex                 (rxNDPANcIndex),
        .rxNDPASTAIndexIsFirst         (rxNDPASTAIndexIsFirst),
`endif //RW_BFMEE_EN
        // SR Controller
        .partialBSSID                  (partialBSSID),
        // Control and Status Register
        .macAddr                       (macAddr),
        .macAddrMask                   (macAddrMask),
        .bssID                         (bssID),
        .bssIDMask                     (bssIDMask),
        .aid                           (aid),
        .dontDecrypt                   (dontDecrypt),
        .excUnencrypted                (excUnencrypted),
        .acceptMulticast               (acceptMulticast),
        .acceptBroadcast               (acceptBroadcast),
        .acceptOtherBSSID              (acceptOtherBSSID),
        .acceptErrorFrames             (acceptErrorFramesnDiscard),
        .acceptUnicast                 (acceptUnicast),
        .acceptMyUnicast               (acceptMyUnicast),
        .acceptProbeReq                (acceptProbeReq),
        .acceptProbeResp               (acceptProbeResp),
        .acceptBeacon                  (acceptBeacon),
        .acceptAllBeacon               (acceptAllBeacon),
`ifdef RW_BFMEE_EN
        .acceptBfmeeFrames             (acceptBfmeeFrames),
`endif //RW_BFMEE_EN
        .acceptOtherMgmtFrames         (acceptOtherMgmtFrames),
        .acceptBAR                     (acceptBAR),
        .acceptBA                      (acceptBA),
        .acceptNotExpectedBA           (acceptNotExpectedBA),
        .acceptPSPoll                  (acceptPSPoll),
        .acceptRTS                     (acceptRTS),
        .acceptCTS                     (acceptCTS),
        .acceptACK                     (acceptACK),
        .acceptCFEnd                   (acceptCFEnd),
        .acceptOtherCntrlFrames        (acceptOtherCntrlFrames),
        .acceptData                    (acceptData),
        .acceptCFWOData                (acceptCFWOData),
        .acceptQData                   (acceptQData),
        .acceptQCFWOData               (acceptQCFWOData),
        .acceptQoSNull                 (acceptQoSNull),
        .acceptOtherDataFrames         (acceptOtherDataFrames),
        .acceptUnknown                 (acceptUnknown),
        .acceptTriggerHWFrames         (acceptTriggerHWFrames),
        .acceptTriggerSWFrames         (acceptTriggerSWFrames),
        .acceptAllTriggerFrames        (acceptAllTriggerFrames),
        .quietCount1In                 (quietCount1In),
        .quietPeriod1In                (quietPeriod1In),
        .quietDuration1In              (quietDuration1In),
        .quietOffset1In                (quietOffset1In),
        .dtimPeriodIn                  (dtimPeriodIn),
        .dynBWEn                       (dynBWEn),
        .primaryChPosition             (primaryChPosition),
        .maxSupportedBW                (maxSupportedBW),
        .raRUType                      (raRUType),
        .raRUEnable                    (raRUEnable),
        .maxMCSInHETB                  (maxMCSInHETB),
        .eOCW                          (eOCW),
        .eOCWMin                       (eOCWMin),
        .eOCWIn                        (eOCWIn),
        .eOCWInValid                   (eOCWInValid),
        .ocwLFSR                       (ocwLFSR),
        .ocwLFSRIn                     (ocwLFSRIn),
        .ocwLFSRInValid                (ocwLFSRInValid),
        .dopplerSupport                (dopplerSupport),
        .dcmSupport                    (dcmSupport),
        // RX FIFO Controller
        .rxFrmCtrl                     (rxFrmCtrl),
        .rxBADly                       (rxBADly),
        .subTypeFrame                  (subTypeFrame),
        .typeFrame                     (typeFrame),
        .protocolError                 (protocolError),
        .ackRespRcved                  (ackRespRcved),
        // Debug port
        .debugPortTriggerDecoder       (debugPortTriggerDecoder)
        );


// RX FIFO Interface Controller
rxFIFOIfController U_rxFIFOIfController (
    // Clock and reset
		.macCoreRxClk             (macCoreRxClk),
		.macCoreClkHardRst_n      (macCoreClkHardRst_n),
		.macCoreClkSoftRst_n      (macCoreClkSoftRst_n),
    // Timer unit
		.tsf                      (tsf),
    // Crypto Engine
		.plainDataOut             (plainDataOut),
		.plainDataOutValid        (plainDataOutValid),
		.plainDataOutEnd_p        (plainDataOutEnd_p),
    // Key Search Engine
		.keyStorageValid_p        (keyStorageValid_p),
		.rxKeyIndexReturn         (rxKeyIndexReturn),
    // Decryption FSM
		.rxDataSelect             (rxDataSelect),
    // MAC-PHY interface
		.macPhyIfRxEndForTiming_p (macPhyIfRxEndForTiming_p),
		.macPhyIfRxErr_p          (macPhyIfRxErr_p),
    // A-MPDU Deaggregator
		.rxVector1                (rxVector1),
		.rxVector2                (rxVector2),
		.correctDelimiter         (correctDelimiter),
		.rxNDP                    (rxNDP),
		.rxLegLength              (rxLegLength),
		.rxFormatMod              (rxFormatMod),
		.rxAggregation            (rxAggregation),
 		.rxVector1Valid_p         (rxVector1Valid_p),
 		.rxVector2Valid_p         (rxVector2Valid_p),
		.rxData                   (rxData),
		.rxDataValid              (rxDataValid),
		.rxDataStart_p            (rxDataStart_p),
		.rxDataEnd_p              (rxDataEnd_p),
		.rxDataError_p            (rxDataError_p),
		.rxEndOfFrame_p           (rxEndOfFrameDA_p),
		.rxAMPDUCnt               (rxAMPDUCnt),
		.rxMPDUCnt                (rxMPDUCnt),
    // RX Controller FSM
		.rxControlIdle            (rxControlIdle),
      .rxControlBackToIdle_p    (rxControlBackToIdle_p),
		.rxMpduLength             (rxMpduLength),
		.frmSuccessfulRcved       (frmSuccessfulRcved),
		.fcsErrorRcved            (fcsErrorRcved),
		.phyErrorRcved            (phyErrorRcved),
		.undefErrorRcved          (undefErrorRcved),
		.rxFIFOOverFlowError      (rxFIFOOverFlowError),
      .decryptionError          (decryptionError),
      .decryptionType           (decryptionType),
		.frmBody                  (frmBody),
		.rxPayloadEnd_p           (rxPayloadEnd_p),
		.startMacHdr              (startMacHdr),
		.macHdr                   (macHdr),
		.endMacHdr_p              (endMacHdr_p),
		.discardFrm               (discardFrm),
		.acceptError_p            (acceptError_p),
		.writeTrailer             (writeTrailer),
		.rxTriggerSWValid         (rxTriggerSWValid),
		.rxFIFODone_p             (rxFIFODone_p),
		.rxEndOfFrame             (rxEndOfFrame),
		.txInProgressDly          (txInProgressDly),
    // Frame decoder
      .protocolError            (protocolError),
		.subTypeFrame             (subTypeFrame),
		.typeFrame                (typeFrame),
		.mgmtFrame                (mgmtFrame),
		.ctrlFrame                (ctrlFrame),
		.dataFrame                (dataFrame),
      .extnFrame                (extnFrame),
		.addr1Match               (addr1Match),
		.bcMcRcved                (bcMcRcved),
		.ackRespRcved             (ackRespRcved),
		.baRcved                  (baRcved),
		.triggerRcved             (triggerRcved),
      .otherCtrlRcved           (otherCtrlRcved),
		.rxFrmCtrl                (rxFrmCtrl),
		.rxBADly                  (rxBADly),
    // Decrypt FSM  
		.keyIndexCapture_p        (keyIndexCapture_p),
    // RX FIFO
		.rxFIFOAlmostFull         (rxFIFOAlmostFull),
		.rxFIFOWrTag              (rxFIFOWrTag),
		.rxFIFOWrData             (rxFIFOWrData),
		.rxFIFOWrite              (rxFIFOWrite),
    // MacController
		.txInProgress             (txInProgress),
		.rxBA                     (rxBA),
    // backoff
		.activeAC                 (activeAC),
    // Control and Status Register
		.acceptErrorFrames        (acceptErrorFramesnDiscard),
      .dataMHStoredwithPld      (dataMHStoredwithPld),
      .mgtMHStoredwithPld       (mgtMHStoredwithPld),
      .ctrlMHStoredwithPld      (ctrlMHStoredwithPld),
      .dataFrmWrapMode          (dataFrmWrapMode),
      .mgtFrmWrapMode           (mgtFrmWrapMode),
      .ctrlFrmWrapMode          (ctrlFrmWrapMode),
      .disableRxBuffer2         (disableRxBuffer2),
    // Interrupt Controller
		.rxFIFOOverFlow           (rxFIFOOverFlow),
    // Debug Port
		.debugPortrxFIFOCntrl     (debugPortrxFIFOCntrl)
		);

// Internal wires
assign tsf                  = {tsfTimerHigh, tsfTimerLow};

// Output ports
assign rxSN           = rxSeqCtrl[15:4];
assign rxTkipSeqCntr  = {rxExtInitVector, rxInitVector[7:0], rxInitVector[23:16]};

// Generate rxAddress4Pres and rxQoSFrame indicating respectively the reception
// of frame with Addr4 and a QoS Frame.
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxAddress4Pres <= 1'b0;
    rxQoSFrame     <= 1'b0;
  end
  else
  begin
    rxAddress4Pres <= dataFrame && toDS && fromDS;
    rxQoSFrame     <= qosFrame;
  end
end

// Debug ports
assign debugPortRxController =  {discardFrm,
                                 1'b0,
                                 endMacHdr_p,
                                 startMacHdr,
                                 addr1Match,
                                 bssIDMatch,
                                 acceptRcved,
                                 fcsOk,
                                 rxFIFOWrTag,
                                 rxFIFOWrite,
                                 rxFIFOAlmostFull,
                                 rxFIFOOverFlow,
                                 1'b0
                                 };

assign debugPortRxFrameDebug1 = {correctRcved_p || incorrectRcved_p,
                                 rxSeqCtrl[3:0],
                                 protectedBit,
                                 pwrMgt,
                                 rxMoreFrag,
                                 rxMoreData,
                                 rxRetry,
                                 subTypeFrame,
                                 typeFrame
                                 };

assign debugPortRxFrameDebug2 = {correctRcved_p,
                                 rxSeqCtrl[9:4],
                                 addr1Match,
                                 rxAddr2[47:40]
                                 };

assign debugPortAMPDURxC = {baRcved,
                            acceptRcved,
                            addr1Match,
                            bssIDMatch,
                            discardFrm,
                            fcsOk,
                            1'b0};


///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON

//$rw_sva This cover point checks if RX Controller have generated rxFIFOOverFlow
countRxFIFOOverFlow: cover property (@(posedge macCoreRxClk) (rxFIFOOverFlow));

`endif // RW_ASSERT_ON


endmodule
                 
