//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : FSM of rxController module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewer.
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module rxControllerFsm( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire         macCoreRxClk,            // MAC Core Receive Clock
            input wire         macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain 
                                                        // active low
            input wire         macCoreClkSoftRst_n,     // Soft Reset of the MAC Core Clock domain
                                                        // active low

            ///////////////////////////////////////////////
            //$port_g DMA Engine
            ///////////////////////////////////////////////
            input wire         rxFrmDiscard,            // Discard current frame
            input wire         rxDescAvailable,         // Indicate that the DMA has all the needed desc for the current rxFrame


            ///////////////////////////////////////////////
            //$port_g MAC Controller
            ///////////////////////////////////////////////
            input wire         stopRx_p,                // Stop Rx
            output reg         rxEndOfFrameRC_p,        // End of frame information to macController
            input wire         frameExpectingResp,      // Indicates that the received frame expect
                                                        // an immediate response
            input wire         rxCts,                   // Reception of a CTS is expected
`ifdef RW_BFMER_EN
            input wire         rxBfr,                   // Reception of a BFR is expected
`endif // RW_BFMER_EN
            input wire         rxAck,                   // Reception of an ACK is expected

            output wire        correctRcved_p,          // Frame received correctly
            output wire        incorrectRcved_p,        // Frame not received correctly
            output wire        rxTriggerHWValid_p,      // Trigger HW Valid pulse
            output wire        rxTriggerSWValid_p,      // Trigger SW Valid pulse

            ///////////////////////////////////////////////
            //$port_g Backoff Engine
            ///////////////////////////////////////////////
            input wire         backOffDone_p,            // Backoff completed

            
            ///////////////////////////////////////////////
            //$port_g NAV
            ///////////////////////////////////////////////
            output wire        basicNAVClear_p,         // Clear basic NAV
            output wire        intraNAVClear_p,         // Clear intra NAV
            output wire        basicNAVUpdate_p,        // Update basic NAV with Frame duration
            output wire        intraNAVUpdate_p,        // Update intra NAV with Frame duration
            input  wire [15:0] rxFrameDuration,         // Frame duration
            output reg  [14:0] navUpdateDuration,       // Duration for the NAV update
            
            ///////////////////////////////////////////////
            //$port_g MAC Timer unit
            ///////////////////////////////////////////////
            output wire        dtimUpdate_p,            // Update DTIM element
            output wire        tsfUpdate_p,             // Update TSF
            output reg         tsfOffsetUpdate_p,       // Update TSF Offset calculation
            output reg         lastRxWrong,             // Last received packet wrong
            
            ///////////////////////////////////////////////
            //$port_g Encryption Engine
            ///////////////////////////////////////////////
            input wire         decryptPassed_p,         // Indicates decryption was a success
            input wire         decryptFailed_p,         // Indicates decryption was a failure
            input wire         plainDataOutEnd_p,       // Encryption Buffer end
            output wire [15:0] rxPayLoadLen,            // Frame length including MIC & FCS
            
            ///////////////////////////////////////////////
            //$port_g Key Search Engine
            ///////////////////////////////////////////////
            input wire [1:0]   sppKSR,                  // SPP for RAM
            input wire [2:0]   cTypeKSR,                // Cipher Type for RAM
            input wire [1:0]   cLenKSR,                 // Cipher length from RAM
            
            ///////////////////////////////////////////////
            //$port_g Decryption FSM
            ///////////////////////////////////////////////
            input wire         rxDataSelect,            // Select data from Encryption Engine
            input wire         rxIndexSearchDone,       // Key Search Engine Done
            input wire         rxDefaultKeySearchDone,  // Default Key Search Done Flag

            output wire        rxControlToError_p,      // RX FSM goes to error
            output reg         rxControlToFrmBody_p,    // RX FSM goes to frame body
            output wire        acceptProtected,         // Accept protected frame
            output reg         rxInitVectorEnd_p,       // FSM IV end pulse
            output reg         rxExtIV_p,               // Indicate if there is a extended IV
            output wire        decrStatusValid_p,       // Decryption status valid pulse
            output wire        searchMACAddr_p,         // FSM ADDR2 end pulse
            output wire        payloadEnd_p,            // Last Byte of Frame Body (without encryption tailer)
            
            ///////////////////////////////////////////////
            //$port_g MAC-PHY interface
            ///////////////////////////////////////////////
            input wire         macPhyIfRxErr_p,         // Rx error from MAC-PHY if. resynchronized
            input wire         mpIfTxEn,                // Indicates that TX is in progress
            input wire         macPHYIFOverflow,        // Overflow Indication
            input wire         macPhyIfRxEndForTiming_p,// Rx end for timing from MAC-PHY if. resynchronized

            ///////////////////////////////////////////////
            //$port_g A-MPDU Deaggregator
            ///////////////////////////////////////////////
            input wire         rxVector1Valid_p,        // Rx vector 1 is available
            input wire [ 3:0]  rxFormatMod,             // Format and modulation
            input wire         rxAggregation,           // MPDU aggregate
            
            input wire [ 7:0]  rxData,                  // Rx data read from MAC-PHY interface FIFO
            input wire         rxDataValid,             // Rx data is valid
            input wire         rxDataStart_p,           // Start of data flow
            input wire         rxDataEnd_p,             // End of data flow
            input wire         rxDataError_p,           // Rx data error (incorrect length)

            input wire         ampduIncorrectRcved_p,   // A-MPDU not received correctly
            
            input wire [15:0]  rxByteCnt,               // Rx byte counter
            input wire         rxNDP,                   // Indicate that the received frame is a Null Data Packet
            input wire  [8:0]  rxPartialAID,            // Partial AID         
            input wire  [5:0]  rxGroupID,               // Group ID            
            input wire         rxUplinkFlag,            // Indicates if the HE PPDU is addressed to
                                                        // an AP
            input wire  [5:0]  rxBssColor,              // Indicates the BSS color of the AP
            input wire  [6:0]  rxTxopDuration,          // Indicates a duration that is used to
                                                        // update the NAV for this TXOP
            output wire        rxCntrlReady,            // Rx controller ready to receive data
            input wire         rxEndOfFrameDA_p,        // End of frame information from Deaggregator

            ///////////////////////////////////////////////
            //$port_g Frame decoder
            ///////////////////////////////////////////////
            input wire         protocolError,           // Protocol error
            
            input wire         mgmtFrame,               // Management frame
            input wire         ctrlFrame,               // Control frame
            input wire         dataFrame,               // Data frame
            input wire         extnFrame,               // Extension frame
            input wire         qosFrame,                // QoS frame
            input wire         htcFrame,                // HT or VHT frame with HTC field
            input wire         bssIDFrame,              // Frame contains BSSID field
            
            input wire         broadCastRcved,          // Broadcast frame
            input wire         bcMcRcved,               // Broadcast/Multicast
            
            input wire         bcnRcved,                // Beacon
            input wire         probRespRcved,           // Probe response
            
            input wire         triggerRcved,            // Trigger Frame
            input wire         triggerHWRcved,          // Trigger frame received
                                                        // with HW response expected
            input wire         triggerSWRcved,          // Trigger frame received
                                                        // with SW response expected
`ifdef RW_BFMEE_EN
            input wire         bfrPollRcved,            // BFR-Poll
            input wire         ndpaRcved,               // NPD-A
`endif //RW_BFMEE_EN
`ifdef RW_BFMER_EN
            input wire         bfrRcved,                // BFR
`endif //RW_BFMER_EN

            input wire         ctrlWrapRcved,           // Control Wrapper
            input wire         baRcved,                 // BA
            input wire         barRcved,                // BAR
            input wire         psPollRcved,             // PS-Poll
            input wire         rtsRcved,                // RTS
            input wire         ctsRcved,                // CTS
            input wire         ackRcved,                // ACK
            input wire         cfEndRcved,              // CF-End
            
            input wire         ctrlWrapFlag,            // Control Wrapper flag
            input wire         ctrlWrapError_p,         // Control wrapper error pulse
            
            input wire         dataRcved,               // Data
            input wire         qosDataRcved,            // QoS Data
            input wire         qosNullRcved,            // QoS null

            input wire         otherCtrlRcved,          // Other control frame

            input wire         muBARTriggerRcved,       // MU-BAR Trigger
            
            input wire         toDS,                    // To DS
            input wire         fromDS,                  // From DS
            input wire         protectedBit,            // Protected bit
            
            input wire         rxAMSDUPresent,          // A-MSDU present
            
            input wire         basicBA,                 // Basic BA
            input wire         compressedBA8,           // Compressed BA with
                                                        // 8 Bytes Bitmap
            input wire         compressedBA32,          // Compressed BA with
                                                        // 32 Bytes Bitmap
            input wire         acceptRcved,             // Accept current frame after filtering
            
            input wire         addr1Match,              // ADDR1 match
            input wire         bssIDMatch,              // BSSID match
            input wire         wildcardBSSID,           // Wildcard BSSID
            input wire         addr1MatchBssID,         // ADDR1 is equal to the BSS ID
            input wire         addr2MatchBssID,         // ADDR2 is equal to the BSS ID
`ifdef RW_BFMEE_EN
            input wire         rxNDPAAIDMatch,          // NDPA AID12 match
            input wire         rxNDPAHE,                // NDP-A HE
`endif //RW_BFMEE_EN
            input wire   [7:0] trigDepUsrInfoLen,       // Length of the Trigger Dependent
                                                        // User Info Field
            input wire   [3:0] rxTriggerType,           // Trigger Type
            input wire         rxTriggerUSRInfoMatch,   // Trigger: Valid user info field
            input wire         rxTriggerAIDMatch,       // Trigger: AID12 Match
            input wire         rxTriggerAID4095,        // Trigger: AID12 is 4095
            input wire  [47:0] rxAddr1,                 // Received ADDR1 field
            input wire  [47:0] rxAddr2,                 // Received ADDR2 field
            
            output wire        rxFrmCtrlEn,             // Frame control field
            output wire        rxCarFrmCtrlEn,          // Carried Frame control field
            output wire        rxDurationIdEn,          // Duration / ID field
            output wire        rxAddr1En,               // ADDR1 field
            output wire        rxAddr2En,               // ADDR2 field
            output wire        rxAddr3En,               // ADDR3 field
            output wire        rxAddr4En,               // ADDR4 field
`ifdef RW_WAPI_EN
            output wire        rxWAPIKeyIdxEn,          // WAPI KeyIdx field
            output wire        rxWAPIPNEn,              // WAPI PN field
`endif //RW_WAPI_EN
            output wire        rxQoSCtrlEn,             // QoS control field
            output wire        rxHTCtrlEn,              // HT control field
            output wire        rxSeqCtrlEn,             // Sequence control field
            output wire        rxBACtrlEn,              // BA control field
            output wire        rxBASeqCtrlEn,           // BA Sequence control field
            output wire        rxInitVectorEn,          // IV field
            output wire        rxExtInitVectorEn,       // Extended IV field

            output wire        rxDtimCntEn,             // DTIM counter field
            output wire        rxDtimPeriodEn,          // DTIM period field
            output wire        rxTimeStampEn,           // Timestamp field
            
            output wire        rxQuietCount1En,         // Quiet Count 1 field
            output wire        rxQuietPeriod1En,        // Quiet Period 1 field
            output wire        rxQuietDuration1En,      // Quiet Duration 1 field
            output wire        rxQuietOffset1En,        // Quiet Offset 1 field
`ifdef RW_BFMEE_EN
            output wire        rxBFRPollSegRetEn,       // BFR-Poll Feedback Segment Retransmission Bitmap Field
            output wire        rxNDPASndDialTokenEn,    // NDP-A Sounding Dialog Token field
            output wire        rxNDPASTAInfo1En,        // NDP-A STA Info 1 field
            output wire        rxNDPASTAInfoNEn,        // NDP-A STA Info N field
`endif //RW_BFMEE_EN
            output wire        rxTriggerCMNInfoEn,      // Trigger Common Info field
            output wire        rxTriggerUSRInfoEn,      // Trigger User Info field
            output wire        rxTriggerDepUSRInfoEn,   // Trigger Dependent User Info field
            output reg         rxFrmBodyByte1En,        // Fame Body Byte 1 field
            output reg         rxFrmBodyByte2En,        // Fame Body Byte 2 field
`ifdef RW_BFMER_EN
            output reg         rxFrmBodyByte2End_p,     // Fame Body Byte 2 field end pulse
`endif 
            
            output reg [7:0]   stateByteCnt,            // Byte counter by states

            output reg         rxAddr4End_p,            // FSM ADDR4 pulse
            output wire        fcsCheck_p,              // FSM FCS check pulse
            output wire        rxDurationIdEnd_p,       // FSM Duration / ID end pulse
            output reg         rxAddr1End_p,            // FSM ADDR1 end pulse
            output reg         rxAddr2End_p,            // FSM ADDR2 end pulse
            output reg         carFrameCtrlEnd_p,       // FSM Carried frame control end pulse
            output reg         rxAddr3End_p,            // FSM ADDR3 end pulse
            output reg         rxSeqCtrlEnd_p,          // FSM Sequence Control end pulse
            output reg         rxQoSEnd_p,              // FSM QoS end pulse
            output reg         rxHTCtrlEnd_p,           // FSM HT Control end pulse
            output wire        rxBACtrlEnd_p,           // FSM BA Control end pulse
            output wire        rxBARInfoEnd_p,          // FSM BAR Info end pulse
`ifdef RW_BFMEE_EN
            output reg         rxNDPAStaInfoEnd_p,      // FSM NDPA STA Info end pulse
`endif //RW_BFMEE_EN
            output reg         rxTriggerUSRInfoEnd_p,   // FSM Trigger User Info end pulse
            output wire        macHeaderCompleted,      // End of MAC Header reception pulse

            output wire        rxControlIdle,           // FSM in idle
            input  wire        decryptionCheckOk,       // Indicate that the decryption Check is ok
            output reg         rxTriggerFound,          // Indicate that a valid trigger has been
                                                        // found

            ///////////////////////////////////////////////
            //$port_g BA Controller
            ///////////////////////////////////////////////
            output wire        psBitmapUpdate_p,        // Update Bitmap request
            output wire        psBitmapDiscard_p,       // Discard Bitmap update request
            output wire        baEnable,                // Enable BA Controller procedure
            
`ifdef RW_BFMEE_EN
            ///////////////////////////////////////////////
            //$port_g BFR Controller
            ///////////////////////////////////////////////
            output wire        rxNDPAValid_p,           // NDP-A Valid pulse
            output wire        rxBFRPollValid_p,        // BFR-Poll Valid pulse
`endif //RW_BFMEE_EN

            ///////////////////////////////////////////////
            //$port_g SR Controller
            ///////////////////////////////////////////////
            output wire        fcsOkRcved_p,            // Frame received with correct FCS 
            output reg         interBSSRcved,           // Inter-BSS frame received

            ///////////////////////////////////////////////
            //$port_g Control and Status Register
            ///////////////////////////////////////////////
            input wire [3:0]   currentState,            // Current state of the core

            input wire         bssType,                 // BSS Type
            input wire         ap,                      // Access Point
            input wire         tsfMgtDisable,           // Disable HW management of TSF
            input wire [47:0]  bssID,                   // BSSID
            input wire  [5:0]  bssColorCSReg,           // BSS Color
            input wire         bssColorEnCSReg,         // BSS Color Enable
            input wire         partialBSSColorEnCSReg,  // Partial BSS Color Enable
            input wire         bssHE,                   // Connected to an HE AP
            
            input wire         dontDecrypt,             // Don't decrypt frames
            input wire         acceptBroadcast,         // Accept Broadcast frames
            input wire         acceptErrorFrames,       // Accept error frames
`ifdef RW_BFMEE_EN
            input wire         acceptBfmeeFrames,       // Accept Beamformee Frames
`endif //RW_BFMEE_EN
            input wire         acceptDecryptErrorFrames,// Accept frames which failed decryption

            output wire        quietElement1InValid,    // Indicates that the Quiet 1 registers have to be updated with Quiet 1 In elements
            output wire        dtimPeriodInValid,       // Indicates that the dtimPeriod register has to be updated with dtimPeriodIn
            
            output reg [ 5:0]  rxControlLs,             // RX Controller latched state
            output reg [ 5:0]  rxControlCs,             // RX Controller current state
            
            ///////////////////////////////////////////////
            //$port_g RX FIFO Interface Controller
            ///////////////////////////////////////////////
            input wire         rxFIFOOverFlow,          // RX FIFO overflow
            input wire         rxFIFOFull,              // RX FIFO is full
            input wire         rxFIFODone_p,            // RX FIFO status done
            input wire         rxFIFOWrite,             // RX FIFO write enable
            input wire         txInProgressDly,         // Captured of the txInProgress on rxVector1 valid
            output wire        rxControlBackToIdle_p,   // FSM changed to idle state
            output reg         rxPayloadEnd_p,          // Last Byte of Frame Body
            
            output wire        frmSuccessfulRcved,      // Frame successful for RX DMA
            output reg         fcsErrorRcved,           // FCS Error
            output reg         phyErrorRcved,           // PHY Error
            output reg         undefErrorRcved,         // Undefined Error
            output reg         rxFIFOOverFlowError,     // RX FIFO overflow
            output reg         decryptionError,         // Decryption Error
            output reg [3:0]   decryptionType,          // Decryption Type
            output wire        frmBody,                 // Frame body completion

            output wire        startMacHdr,             // Start of the MAC Header
            output wire        macHdr,                  // MAC Header processing
            output wire        endMacHdr_p,             // End of the MAC Header pulse
            output wire        discardFrm,              // End of frame with discard command
            output wire        acceptError_p,           // Pulse to indicate that writeTrailer to accept Error frame (others than FCS error)
            
            output wire        writeTrailer,            // Write trailer info
            output reg         rxTriggerSWValid,        // Valid trigger frame with SW response flag
            
            ///////////////////////////////////////////////
            //$port_g HW Error detected
            ///////////////////////////////////////////////
            input wire         hwErr,                   // HW error detected
            
            ///////////////////////////////////////////////
            //$port_g FCS
            ///////////////////////////////////////////////
            input wire         fcsOk,                   // FCS is ok
            
            output wire        fcsStartRx_p,            // Start FCS calculation
            output reg         fcsEnableRx              // Enable FCS
            );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// rxControl FSM states definition
//$fsm_sd rxControl
localparam 
                 IDLE  =  6'd0,
          RX_FRM_CTRL  =  6'd1,
            RX_DUR_ID  =  6'd2,
             RX_ADDR1  =  6'd3,
             RX_ADDR2  =  6'd4,
             RX_ADDR3  =  6'd5,
             RX_ADDR4  =  6'd6,
          RX_SEQ_CTRL  =  6'd7,
          RX_QOS_CTRL  =  6'd8,
      RX_CAR_FRM_CTRL  =  6'd9,
           RX_HT_CTRL  =  6'd10,
     RX_IV_WAPIKEYIDX  =  6'd11,
      RX_EXTIV_WAPIPN  =  6'd12,
          RX_FRM_BODY  =  6'd13,
               RX_ICV  =  6'd14,
     RX_CCMP_WAPI_MIC  =  6'd15,
               RX_FCS  =  6'd16,
           RX_BA_CTRL  =  6'd17,
       RX_BA_SEQ_CTRL  =  6'd18,
         RX_BA_BITMAP  =  6'd19,
         RX_TIMESTAMP  =  6'd20,
        RX_BCN_INTRVL  =  6'd21,
        RX_CAPABILITY  =  6'd22,
        RX_ELEMENT_ID  =  6'd23,
       RX_ELEMENT_LEN  =  6'd24,
      RX_ELEMENT_BODY  =  6'd25,
            CHECK_FCS  =  6'd26,
           CHECK_ENCR  =  6'd27,
         WRITE_STATUS  =  6'd28,
             RX_ERROR  =  6'd29,
        DISCARD_FRAME  =  6'd30,
   WAIT_KEYSEARCHDONE  =  6'd31,
`ifdef RW_BFMEE_EN
// As Debug Port can only probe 5 bits of the FSM State, 
// the BFMEE state has been choose in order to be identified without MSB bit.
// RX_ADDR2 -> RX_BFRPOLL_SEGRET  transition will be seen has transition to 31
  RX_BFRPOLL_SEGRET       =  6'd63, 
  RX_NDPA_SNDDIALTOKEN    =  6'd60, // WRITE_STATUS       + 32
  RX_NDPA_STAINFO1        =  6'd59, // CHECK_ENCR         + 32
  RX_NDPA_STAINFON        =  6'd58, // CHECK_FCS          + 32
`endif //RW_BFMEE_EN
  RX_TRIGGER_CMN_INFO     =  6'd57, // RX_ELEMENT_BODY    + 32
  RX_TRIGGER_DEP_CMN_INFO =  6'd56, // RX_ELEMENT_LEN     + 32
  RX_TRIGGER_USR_INFO     =  6'd55, // RX_ELEMENT_ID      + 32
  RX_TRIGGER_DEP_USR_INFO =  6'd54, // RX_CAPABILITY      + 32
  RX_TRIGGER_PADDING      =  6'd53, // RX_BCN_INTRVL      + 32
  RX_BA_INFO              =  6'd52, // RX_TIMESTAMP       + 32
  WAIT_SHIFTMIC_ICV       =  6'd51; // RX_BA_BITMAP       + 32


// Core current state
localparam 
    ACTIVE_STATE = 4'h3; // active state


// Type of encryption received
localparam 
   ENCRYPTION_TYPE_NONE = 3'd0,   // None encryption is selected
   ENCRYPTION_TYPE_WEP  = 3'd1,   // WEP  encryption is selected
   ENCRYPTION_TYPE_TKIP = 3'd2,   // TKIP encryption is selected
   ENCRYPTION_TYPE_CCMP = 3'd3,   // CCMP encryption is selected
   ENCRYPTION_TYPE_WAPI = 3'd4,   // WAPI encryption is selected
   ENCRYPTION_TYPE_GCMP = 3'd5;   // GCMP encryption is selected


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// rxControl FSM signals definition
reg [5:0] rxControlNs;  // rxControl FSM Next State

// Info element length
reg [7:0] infoElemLen;
reg [7:0] elementID;

// Payload length
reg [15:0] payloadLength;
reg [15:0] payloadLengthSub;

// DTIM parameter found
reg       dtimParameterFound;
// Quiet parameter found
reg       quietParameterFound;

// Non encrypted status information check
reg       statusInfoCheck_p;

// FCS Ok status
reg       fcsOkRcved;
wire      fcsAndLenOk;
// Local frame status
wire      localCorrectRcved_p;
wire      localIncorrectRcved_p;

// Invalid protected packet length
reg       invalidProtectPktLen;

// Invalid packet
wire      invalidPkt;

// rxByteCnt trigger
wire      rxByteCntTrig;
reg       rxByteCntTrig_ff1;

//reg       rxDataEnd;               // End of data flow holded

reg       macPhyIfRxEndForTimingHold;
wire      macPhyIfRxEndForTimingHoldPatch; // Use to work around a Synplify issue.

// Flag indicating if the frame has to be discarded due to encryption failure.
wire      discardDecrypt;

reg       macPhyIfRxErrInt_p;       // Rx error from MAC-PHY if delayed of 1 cc

// Encryption Engine specific logic
reg       decrStatusValidHold;      // Decryption status valid registered

wire      respError;                // Indicate that the type of the received frame is not an expected response.

reg       lenIs0InFCSCheck;
reg       discardDone;

// NAV logic
reg       navClear;                 // Clear NAV
reg       basicNAVUpdateFromVector; // Basic NAV Update from RX Vector flag
reg       basicNAVUpdateFromHeader; // Basic NAV Update from Header flag
reg       intraNAVUpdateFromVector; // Intra NAV Update from RX Vector flag
reg       intraNAVUpdateFromHeader; // Intra NAV Update from Header flag

// Intra-BSS / inter-BSS logic
reg        fcsOkRcvedDly_p;         // Frame received with correct FCS (1 cycle delayed)
wire       addr2Pres;               // Indicates that the received frame has an addr2 field;
reg [47:0] txopHolderOfBSS;         // Indicates the Address of the TXOP Holder
reg        intraBSSRcved;           // Intra-BSS frame received
wire       interBSSFromBssColor;    // From the RX Vector BSS Color information,
                                    // the received frame is classified as an inter-BSS frame
wire       interBSSFromRxVector;    // From the rxVector information,
                                    // the received frame is classified as an inter-BSS frame
reg        interBSSFromHeader;      // From the frame header information,
                                    // the received frame is classified as an inter-BSS frame
wire       intraBSSFromBssColor;    // From the RX Vector BSS Color information,
                                    // the received frame is classified as an intra-BSS frame
wire       intraBSSFromRxVector;    // From the rxVector information, the received frame is
                                    // classified as an intra-BSS frame
reg        intraBSSFromHeader;      // From the frame header information, the received frame is
                                    // classified as an inter-BSS frame
wire       processingEnd_p;         // End of processing

wire      cTypeNONE;                // CipherType is NONE
wire      cTypeWEP;                 // CipherType is WEP
wire      cTypeTKIP;                // CipherType is TKIP
wire      cTypeCCMP128;             // CipherType is CCMP-128
wire      cTypeCCMP256;             // CipherType is CCMP-256
`ifdef  RW_WAPI_EN
wire      cTypeWAPI;                // CipherType is WAPI
`endif//RW_WAPI_EN
wire      cTypeGCMP;                // CipherType is GCMP
wire      cTypeGCMP128;             // CipherType is GCMP-128
wire      cTypeGCMP256;             // CipherType is GCMP-256


`ifdef RW_SIMU_ON
// String definition to display rxControl current state
reg [24*8-1:0] rxControlCs_str;
reg [24*8-1:0] rxControlNs_str;
reg  [8*8-1:0] encryptionType_str;
reg [30*8-1:0] decryptionType_str;
`endif // RW_SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Cipher Type Decoding
assign cTypeNONE    = cTypeKSR==ENCRYPTION_TYPE_NONE;
assign cTypeWEP     = cTypeKSR==ENCRYPTION_TYPE_WEP;
assign cTypeTKIP    = cTypeKSR==ENCRYPTION_TYPE_TKIP;
assign cTypeCCMP128 = cTypeKSR==ENCRYPTION_TYPE_CCMP & cLenKSR==2'd0;
assign cTypeCCMP256 = cTypeKSR==ENCRYPTION_TYPE_CCMP & cLenKSR==2'd2;
`ifdef  RW_WAPI_EN
assign cTypeWAPI    = cTypeKSR==ENCRYPTION_TYPE_WAPI;
`endif//RW_WAPI_EN
assign cTypeGCMP    = cTypeKSR==ENCRYPTION_TYPE_GCMP;
assign cTypeGCMP128 = cTypeKSR==ENCRYPTION_TYPE_GCMP & cLenKSR==2'd0;
assign cTypeGCMP256 = cTypeKSR==ENCRYPTION_TYPE_GCMP & cLenKSR==2'd2;



                 
// rxControl FSM Latched State Logic 
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxControlLs <= IDLE;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxControlLs <= IDLE;
  else if (hwErr)
    rxControlLs <= rxControlCs;
end


// rxControl FSM Current State Logic 
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxControlCs <= IDLE;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxControlCs <= IDLE;
  else
    rxControlCs <= rxControlNs;
end

// rxControl FSM Next State Logic.
always @* 
begin  
  if ((rxControlCs != IDLE) && (rxControlCs != WRITE_STATUS) && 
      (rxControlCs != CHECK_ENCR)  && (rxControlCs != CHECK_FCS) &&
      ((macPhyIfRxErr_p && !macPhyIfRxErrInt_p) || rxDataError_p || 
       macPHYIFOverflow || invalidPkt))
    //$fsm_t When rxDataError_p or macPhyIfRxErr_p are true,
    // or when rxByteCnt=3 and fsm is not in RX_FCS state,
    // or when rxDataEnd_p is true and fsm is not in RX_FCS state, then
    // fsm goes to RX_ERROR state
    rxControlNs = RX_ERROR;

  else if (stopRx_p && (rxControlCs != IDLE) && (rxControlCs != RX_ERROR) && (rxControlCs != WRITE_STATUS)
           && (rxControlCs != CHECK_ENCR))
  begin
    if (rxFIFOOverFlowError)
      //$fsm_t When rxFIFOOverFlowError is active, fsm goes to IDLE state
      rxControlNs = IDLE;
    else
      //$fsm_t When stopRx_p is true and fsm is not idle or write status, fsm goes to DISCARD_FRAME state
      rxControlNs = DISCARD_FRAME;
  end    

  else
  begin  
    case(rxControlCs)
  
      IDLE:
        //$fsm_s In IDLE state, the state machine waits start from A-MPDU Deaggregator
        if (rxDataStart_p)
        begin
`ifdef RW_BFMEE_EN
          if (rxNDP && !txInProgressDly && !acceptBfmeeFrames)
            //$fsm_t When rxDataStart_p and not txInProgress and rxNDP received, the state machine goes to DISCARD_FRAME state
            rxControlNs = DISCARD_FRAME;
          else
`endif//RW_BFMEE_EN
          if (rxNDP && !txInProgressDly)
            //$fsm_t When rxDataStart_p and not txInProgress and rxNDP received, the state machine goes to WRITE_STATUS state
            rxControlNs = WRITE_STATUS;
          else if (rxNDP && txInProgressDly)
            //$fsm_t When rxDataStart_p and txInProgress and rxNDP received, the state machine goes to RX_ERROR state
            rxControlNs = RX_ERROR;
          else 
            //$fsm_t When rxDataStart_p is received, the state machine goes to RX_FRM_CTRL state
            rxControlNs = RX_FRM_CTRL;
        end 
        else
          //$fsm_t While rxDataStart_p is not received, the state machine stays in IDLE state
          rxControlNs = IDLE;
  
      RX_FRM_CTRL:
        //$fsm_s In RX_FRM_CTRL state, the state machine waits frame control field reception
        if (rxDataValid && stateByteCnt[0])
        begin
          if (!protocolError)
            //$fsm_t When rxDataValid is received after 2 bytes and not protocol error is detected, the state machine goes to RX_DUR_ID state
            rxControlNs = RX_DUR_ID;
          else
            //$fsm_t In case of protocol error detected, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_FRM_CTRL state
          rxControlNs = RX_FRM_CTRL;
  
      RX_DUR_ID:
        //$fsm_s In RX_DUR_ID state, the state machine waits frame duration field reception
        if (rxDataValid && stateByteCnt[0])
          //$fsm_t When frame is not reserved, the state machine goes to RX_ADDR1 state
          rxControlNs = RX_ADDR1;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_DUR_ID state
          rxControlNs = RX_DUR_ID;
  
      RX_ADDR1:
        //$fsm_s In RX_ADDR1 state, the state machine waits ADDR1 field reception
        if (rxDataValid && (stateByteCnt == 8'h05))
        begin
          if (otherCtrlRcved | extnFrame)
          begin
            if (rxByteCnt > 16'h0004)
              //$fsm_t When reserved frame is received, the state machine goes to RX_FRM_BODY state
              rxControlNs = RX_FRM_BODY;
            else
              //$fsm_t When reserved frame is received, the state machine goes to RX_FCS state
              rxControlNs = RX_FCS;
          end
          else if (ackRcved || ctsRcved)
            //$fsm_t When ACK or CTS is received, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if (ctrlWrapRcved)
            //$fsm_t When Control Wrapper is received, the state machine goes to RX_CAR_FRM_CTRL state
            rxControlNs = RX_CAR_FRM_CTRL;
          else
            //$fsm_t When frame is not an ACK or CTS, the state machine goes to RX_ADDR2 state
            rxControlNs = RX_ADDR2;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ADDR1 state
          rxControlNs = RX_ADDR1;
  
      RX_ADDR2:
        //$fsm_s In RX_ADDR2 state, the state machine waits ADDR2 field reception
        if (rxDataValid && (stateByteCnt == 8'h05))
        begin
          if (baRcved || barRcved)
            //$fsm_t When BA or BAR is received, the state machine goes to RX_BA_CTRL state
            rxControlNs = RX_BA_CTRL;
`ifdef RW_BFMEE_EN
          else if (ndpaRcved)
            //$fsm_t When NDP-A is received, the state machine goes to RX_NDPA_SNDDIALTOKEN state
            rxControlNs = RX_NDPA_SNDDIALTOKEN;
          else if (bfrPollRcved)
            //$fsm_t When BFR-POLL is received, the state machine goes to RX_BFRPOLL_SEGRET state
            rxControlNs = RX_BFRPOLL_SEGRET;
`endif //RW_BFMEE_EN
          else if (triggerRcved & ~rxTriggerFound)
            //$fsm_t When Trigger Frame is received, the state machine goes to
            //RX_TRIGGER_CMN_INFO state
            rxControlNs = RX_TRIGGER_CMN_INFO;
          else if (ctrlFrame)
            if (rxByteCnt > 16'h0004)
              //$fsm_t When control  frame is received, the state machine goes to RX_FRM_BODY state
              rxControlNs = RX_FRM_BODY;
            else
            //$fsm_t When a control frame is received, the state machine goes to RX_FCS state
              rxControlNs = RX_FCS;
          else
            //$fsm_t When frame is not a control frame, the state machine goes to RX_ADDR3 state
            rxControlNs = RX_ADDR3;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ADDR2 state
          rxControlNs = RX_ADDR2;
  
      RX_ADDR3:
        //$fsm_s In RX_ADDR3 state, the state machine waits ADDR3 field reception
        if (rxDataValid && (stateByteCnt == 8'h05))
          //$fsm_t When rxDataValid is received after 21 bytes, the state machine goes to RX_SEQ_CTRL state
          rxControlNs = RX_SEQ_CTRL; 
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ADDR3 state
          rxControlNs = RX_ADDR3;
  
      RX_SEQ_CTRL:
        //$fsm_s In RX_SEQ_CTRL state, the state machine waits Sequence Control field reception
        if (rxDataValid && stateByteCnt[0])
        begin
          if (rxByteCnt < 16'h0005)
            //$fsm_t When frame body is null, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if ((rxByteCnt < 16'd13) && protectedBit && !qosFrame)
            //$fsm_t When frame is encrypted and length is less than 13, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else if (dataFrame && toDS && fromDS)
            //$fsm_t When frame is data with addr field, the state machine goes to RX_ADDR4 state
            rxControlNs = RX_ADDR4;
          else if ((bcnRcved || probRespRcved) && !htcFrame)
            //$fsm_t When frame is beacon or probe response, the state machine goes to RX_TIMESTAMP state
            rxControlNs = RX_TIMESTAMP;
          else if (protectedBit && !qosFrame && !htcFrame)
            //$fsm_t When frame is encrypted, the state machine goes to RX_IV_WAPIKEYIDX state
            rxControlNs = RX_IV_WAPIKEYIDX;
          else if (!qosFrame && !htcFrame)
            //$fsm_t When frame is not QoS and not HTC, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else if (!qosFrame)
            //$fsm_t When frame contains HTC field, the state machine goes to RX_HT_CTRL state
            rxControlNs = RX_HT_CTRL;
          else
            //$fsm_t When frame is QoS, the state machine goes to RX_QOS_CTRL state
            rxControlNs = RX_QOS_CTRL;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_SEQ_CTRL state
          rxControlNs = RX_SEQ_CTRL;
  
      RX_ADDR4:
        //$fsm_s In RX_ADDR4 state, the state machine waits ADDR4 field reception
        if (rxDataValid && (stateByteCnt == 8'h05))
        begin
          if (rxByteCnt < 16'h0005)
            //$fsm_t When frame body is null, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if ((rxByteCnt < 16'd13) && protectedBit && !qosFrame)
            //$fsm_t When frame is encrypted and length is less than 13, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else if (protectedBit && !qosFrame)
            //$fsm_t When frame is encrypted, the state machine goes to RX_IV_WAPIKEYIDX state
            rxControlNs = RX_IV_WAPIKEYIDX;
          else if (!qosFrame)
            //$fsm_t When frame is not QoS, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else
            //$fsm_t When frame is QoS, the state machine goes to RX_QOS_CTRL state
            rxControlNs = RX_QOS_CTRL;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ADDR4 state
          rxControlNs = RX_ADDR4;
  
      RX_QOS_CTRL:
        //$fsm_s In RX_QOS_CTRL state, the state machine waits QoS Control field reception
        if (rxDataValid && stateByteCnt[0])
        begin
          if (rxByteCnt < 16'h0005)
            //$fsm_t When frame body is null, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if ((rxByteCnt < 16'd13) && protectedBit && !htcFrame)
            //$fsm_t When frame is encrypted and length is less than 13, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else if (protectedBit && !htcFrame)
            //$fsm_t When frame is encrypted, the state machine goes to RX_IV_WAPIKEYIDX state
            rxControlNs = RX_IV_WAPIKEYIDX;
          else if (!htcFrame)
            //$fsm_t When frame does not contain HTC field, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else
            //$fsm_t When frame contains HTC field, the state machine goes to RX_HT_CTRL state
            rxControlNs = RX_HT_CTRL;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_QOS_CTRL state
          rxControlNs = RX_QOS_CTRL;
  
      RX_CAR_FRM_CTRL:
        //$fsm_s In RX_CAR_FRM_CTRL state, the state machine waits Carried Frame Control field reception
        if (rxDataValid && stateByteCnt[0])
        begin
          if (protocolError | ctrlWrapError_p)
            //$fsm_t When rxDataValid is received after 2 bytes, the state machine goes to RX_HT_CTRL state
            rxControlNs = RX_FRM_BODY;
          else
            //$fsm_t When rxDataValid is received after 2 bytes, the state machine goes to RX_HT_CTRL state
            rxControlNs = RX_HT_CTRL;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_CAR_FRM_CTRL state
          rxControlNs = RX_CAR_FRM_CTRL;
  
      RX_HT_CTRL:
        //$fsm_s In RX_HT_CTRL state, the state machine waits HT Control field reception
        if (rxDataValid && (stateByteCnt == 8'h03))
        begin
          if ((rxByteCnt < 16'h0005) || (ctrlWrapFlag && (ackRcved || ctsRcved)))
            //$fsm_t When frame body is null or when ACK or CTS is received from a control wrapper, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if ((rxByteCnt < 16'd13) && protectedBit)
            //$fsm_t When frame is encrypted and length is less than 13, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
          else if (ctrlWrapFlag)
            //$fsm_t When frame is a control wrapper, the state machine goes to RX_ADDR2 state
            rxControlNs = RX_ADDR2;
          else if (bcnRcved || probRespRcved)
            //$fsm_t When frame is beacon or probe response, the state machine goes to RX_TIMESTAMP state
            rxControlNs = RX_TIMESTAMP;
          else if (protectedBit)
            //$fsm_t When frame is encrypted, the state machine goes to RX_IV_WAPIKEYIDX state
            rxControlNs = RX_IV_WAPIKEYIDX;
          else
            //$fsm_t When frame is not encrypted, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_HT_CTRL state
          rxControlNs = RX_HT_CTRL;
  
      RX_IV_WAPIKEYIDX:
        //$fsm_s In RX_IV_WAPIKEYIDX state, the state machine waits IV field reception for WEP/CCMP and KEYIDX field reception for WAPI 
  `ifdef RW_WAPI_EN
        if(cTypeWAPI && rxDataValid && (stateByteCnt == 8'h01))
        begin
          rxControlNs = RX_EXTIV_WAPIPN;
        end
        else
        if (!cTypeWAPI && rxDataValid && (stateByteCnt == 8'h03))
  `else
        if (rxDataValid && (stateByteCnt == 8'h03))
  `endif //RW_WAPI_EN
        begin
          if (((rxByteCnt < 16'd13) && rxData[5]) || !rxData[5])
            //$fsm_t When frame has not extended IV or length is less than 13, the state machine goes to RX_FRM_BODY state
            rxControlNs = WAIT_KEYSEARCHDONE;
          else
            //$fsm_t When frame has extended IV, the state machine goes to RX_EXTIV_WAPIPN state
            rxControlNs = RX_EXTIV_WAPIPN;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_IV_WAPIKEYIDX state
          rxControlNs = RX_IV_WAPIKEYIDX;
  
      RX_EXTIV_WAPIPN:
        //$fsm_s In RX_EXTIV_WAPIPN state, the state machine waits Extended IV or WAPI PN field reception
  `ifdef RW_WAPI_EN
        if(cTypeWAPI && rxDataValid && (stateByteCnt == 8'hF))
        begin
          rxControlNs = WAIT_KEYSEARCHDONE;
        end
        else
        if (!cTypeWAPI && rxDataValid && (stateByteCnt == 8'h03))
  `else
        if (rxDataValid && (stateByteCnt == 8'h03))
  `endif //RW_WAPI_EN
          //$fsm_t When rxDataValid is received after 4 bytes or 16 bytes for WAPI PN, the state machine goes to WAIT_KEYSEARCHDONE state
          rxControlNs = WAIT_KEYSEARCHDONE;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_EXTIV_WAPIPN state
          rxControlNs = RX_EXTIV_WAPIPN;


      WAIT_KEYSEARCHDONE:
        //$fsm_s In WAIT_KEYSEARCHDONE state, the state machine waits for the completion of encryption key search
        if (!rxInitVectorEnd_p && rxDefaultKeySearchDone)
          //$fsm_t When rxDefaultKeySearchDone is set indication the validity of encryption key, the state machine goes to RX_FRM_BODY state
          rxControlNs = RX_FRM_BODY;
        else
          //$fsm_t While rxDefaultKeySearchDone is not received, the state machine stays in WAIT_KEYSEARCHDONE state
          rxControlNs = WAIT_KEYSEARCHDONE;

  
      RX_FRM_BODY:
        //$fsm_s In RX_FRM_BODY state, the state machine waits end of frame body field reception
        if (protocolError)
        begin 
          if (rxByteCnt == 16'h0000 && rxDataValid && acceptErrorFrames)
            //$fsm_t When rxByteCnt=0 and a protocol error was detected and error frame are accepted, 
            //the state machine goes to WRITE_STATUS state
            rxControlNs = WRITE_STATUS; 
          else if (rxByteCnt == 16'h0000 && rxDataValid && !acceptErrorFrames)
            //$fsm_t When rxByteCnt=0 and a protocol error was detected and error frame are not accepted, 
            //the state machine goes to DISCARD_FRAME state
            rxControlNs = DISCARD_FRAME; 
          else  
            //$fsm_t While rxByteCnt != 0 and a protocol error was detected, 
            //the state machine stays in RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
        end    
        else if (!decryptionCheckOk && rxByteCnt == 16'h0004 && rxDataValid )
          //$fsm_t When rxByteCnt=4, the state machine goes to RX_FCS state
          rxControlNs = RX_FCS; 
        else if (decryptionCheckOk && rxDataValid && (rxByteCnt == 16'h0008) &&
               (cTypeWEP || cTypeTKIP))
          //$fsm_t When frame is WEP or TKIP, the state machine goes to RX_ICV state
          rxControlNs = RX_ICV; 
        else if (decryptionCheckOk && rxDataValid && 
                 (( cTypeCCMP128 && (rxByteCnt == 16'h000c) ) || // CCMP-128 : MIC(8) +FCS(4)
                  ( cTypeCCMP256 && (rxByteCnt == 16'h0014) ) || // CCMP-256 : MIC(16)+FCS(4)
`ifdef  RW_WAPI_EN
                  ( cTypeWAPI    && (rxByteCnt == 16'h0014) ) || // WAPI     : MIC(16)+FCS(4)  
`endif//RW_WAPI_EN
                  ( cTypeGCMP    && (rxByteCnt == 16'h0014) ) )) // GCMP     : MIC(16)+FCS(4)
          //$fsm_t When frame is CCMP, GCMP or WAPI,
          //the state machine goes to RX_CCMP_WAPI_MIC state
          rxControlNs = RX_CCMP_WAPI_MIC; 
        else
          //$fsm_t While rxByteCnt>4, the state machine stays in RX_FRM_BODY state
          rxControlNs = RX_FRM_BODY;
  
      RX_ICV:
        //$fsm_s In RX_ICV state, the state machine waits RX_ICV field reception
        if (rxDataValid && (stateByteCnt == 8'h03))
          //$fsm_t When rxDataValid is received after 4 bytes, the state machine goes to RX_FCS state
          rxControlNs = WAIT_SHIFTMIC_ICV;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ICV state
          rxControlNs = RX_ICV;
  
      RX_CCMP_WAPI_MIC:
        //$fsm_s In RX_CCMP_WAPI_MIC state, the state machine waits RX_CCMP_WAPI_MIC field reception
        if ( rxDataValid && 
             (( cTypeCCMP128 && (stateByteCnt == 8'h07)) || // MIC =  8 bytes
              (~cTypeCCMP128 && (stateByteCnt == 8'h0F)) )) // MIC = 16 bytes
          //$fsm_t When rxDataValid is received after 8 bytes, the state machine goes to RX_FCS state
          rxControlNs = WAIT_SHIFTMIC_ICV;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_CCMP_WAPI_MIC state
          rxControlNs = RX_CCMP_WAPI_MIC;
  
      WAIT_SHIFTMIC_ICV:
        //$fsm_s In RX_CCMP_WAPI_MIC state, the state machine waits RX_CCMP_WAPI_MIC field reception
        if (plainDataOutEnd_p) 
          //$fsm_t When rxDataValid is received after 8 bytes, the state machine goes to RX_FCS state
          rxControlNs = RX_FCS;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_CCMP_WAPI_MIC state
          rxControlNs = WAIT_SHIFTMIC_ICV;
  
      RX_FCS:
        //$fsm_s In RX_FCS state, the state machine waits the last 4 FCS bytes of the frame
        if (rxDataValid && (stateByteCnt == 8'h03))
          //$fsm_t When rxDataValid is received after 4 bytes, the state machine goes to CHECK_FCS state
          rxControlNs = CHECK_FCS; 
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_FCS state
          rxControlNs = RX_FCS;
  
      RX_BA_CTRL:
        //$fsm_s In RX_BA_CTRL state, the state machine waits BA Control field reception
        if (rxDataValid && stateByteCnt[0])
        begin
          if (barRcved || basicBA || compressedBA8 || compressedBA32)
            //$fsm_t When rxDataValid is received after 2 bytes and BAR, basic BA or compressed BA,
            //the state machine goes to RX_BA_SEQ_CTRL state
            rxControlNs = RX_BA_SEQ_CTRL;
          else
            //$fsm_t When rxDataValid is received after 2 bytes and not BAR, basic BA or compressed BA,
            //the state machine goes to RX_BA_INFO state
            rxControlNs = RX_BA_INFO;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_BA_CTRL state
          rxControlNs = RX_BA_CTRL;
  
      RX_BA_SEQ_CTRL:
        //$fsm_s In RX_BA_SEQ_CTRL state, the state machine waits BA Starting Sequence Control field reception
        if (rxDataValid && stateByteCnt[0])
        begin
          if (barRcved)
            //$fsm_t When frame is BA Request, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if (basicBA        && rxByteCnt==16'd132 || // Bitmap + FCS
                   compressedBA8  && rxByteCnt==16'd12  || // Bitmap + FCS
                   compressedBA32 && rxByteCnt==16'd36  )  // Bitmap + FCS
            //$fsm_t When frame is BA with correct length, the state machine goes to RX_BA_BITMAP state
            rxControlNs = RX_BA_BITMAP;
          else
            //$fsm_t When frame is BA with incorrect length, the state machine goes to RX_FRM_BODY state
            rxControlNs = RX_FRM_BODY;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_BA_SEQ_CTRL state
          rxControlNs = RX_BA_SEQ_CTRL;
  
      RX_BA_BITMAP:
        //$fsm_s In RX_BA_BITMAP state, the state machine waits BA Bitmap field reception
        if (rxDataValid && (((stateByteCnt == 8'h07) && compressedBA8 ) ||
                            ((stateByteCnt == 8'h1f) && compressedBA32) ||
                            ((stateByteCnt == 8'h7f) /* basicBA */    )))
          //$fsm_t When rxDataValid is received after 8 bytes or 32 bytes on compressed BA 
          //or after 128 bytes on normal BA, the state machine goes to RX_FCS state
          rxControlNs = RX_FCS;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_FCS state
          rxControlNs = RX_BA_BITMAP;
  
      RX_BA_INFO:
        //$fsm_s In RX_BA_INFO state, the state machine waits BA Info field reception
        if (rxDataValid && (rxByteCnt == 16'h0004))
          //$fsm_t When BA Info is received, the state machine goes to RX_FCS state
          rxControlNs = RX_FCS;
        else
          //$fsm_t While BA Info is not received, the state machine stays in RX_BA_INFO state
          rxControlNs = RX_BA_INFO;

      RX_TIMESTAMP:
        //$fsm_s In RX_TIMESTAMP state, the state machine waits Timestamp field reception
        if (rxDataValid && (stateByteCnt == 8'h07))
          //$fsm_t When rxDataValid is received after 8 bytes, the state machine goes to RX_BCN_INTRVL state
          rxControlNs = RX_BCN_INTRVL;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_TIMESTAMP state
          rxControlNs = RX_TIMESTAMP;
  
      RX_BCN_INTRVL:
        //$fsm_s In RX_BCN_INTRVL state, the state machine waits Beacon Interval field reception
        if (rxDataValid && stateByteCnt[0])
          //$fsm_t When rxDataValid is received after 2 bytes, the state machine goes to RX_CAPABILITY state
          rxControlNs = RX_CAPABILITY;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_BCN_INTRVL state
          rxControlNs = RX_BCN_INTRVL;
  
      RX_CAPABILITY:
        //$fsm_s In RX_CAPABILITY state, the state machine waits Capability info. field reception
        if (rxDataValid && stateByteCnt[0])
          //$fsm_t When rxDataValid is received after 2 bytes, the state machine goes to RX_ELEMENT_ID state
          rxControlNs = RX_ELEMENT_ID;
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_CAPABILITY state
          rxControlNs = RX_CAPABILITY;
  
      RX_ELEMENT_ID:
        //$fsm_s In RX_ELEMENT_ID state, the state machine waits Element ID field reception
        if (rxDataValid)
        begin
          if (rxByteCnt == 16'h0004)
            //$fsm_t When element body is null, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else
            //$fsm_t When rxDataValid is true, the state machine goes to RX_ELEMENT_LEN state
            rxControlNs = RX_ELEMENT_LEN;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ELEMENT_ID state
          rxControlNs = RX_ELEMENT_ID;
  
      RX_ELEMENT_LEN:
        //$fsm_s In RX_ELEMENT_LEN state, the state machine waits Element length field reception
        if (rxDataValid)
        begin
          if (rxByteCnt == 16'h0004)
            //$fsm_t When element body is null, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if (rxData == 8'h00)
            //$fsm_t When length is null, the state machine goes to RX_ELEMENT_ID state
            rxControlNs = RX_ELEMENT_ID;
          else
            //$fsm_t When length is not null, the state machine goes to RX_ELEMENT_BODY state
           rxControlNs = RX_ELEMENT_BODY;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ELEMENT_LEN state
          rxControlNs = RX_ELEMENT_LEN;
  
      RX_ELEMENT_BODY:
        //$fsm_s In RX_ELEMENT_BODY state, the state machine waits Element body field reception
        if (rxDataValid)
        begin
          if (rxByteCnt == 16'h0004)
            //$fsm_t When last element body is reached, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else if (stateByteCnt == (infoElemLen - 8'd1))
            //$fsm_t When element length is reached, the state machine goes to RX_ELEMENT_ID state
            rxControlNs = RX_ELEMENT_ID;
          else
            //$fsm_t When length is not reached, the state machine stays in RX_ELEMENT_BODY state
           rxControlNs = RX_ELEMENT_BODY;
        end
        else
          //$fsm_t While rxDataValid is not received, the state machine stays in RX_ELEMENT_BODY state
          rxControlNs = RX_ELEMENT_BODY;
  
`ifdef RW_BFMEE_EN
        RX_NDPA_SNDDIALTOKEN:
        //$fsm_s In RX_NDPA_SNDDIALTOKEN state, the state machine waits Sounding Dialog Token field reception
        if (rxDataValid)
          //$fsm_t When Sounding Dialog Token is received, the state machine goes to RX_NDPA_STAINFO1 state
          rxControlNs = RX_NDPA_STAINFO1;
        else
          //$fsm_t While Sounding Dialog Token is not received, the state machine stays in RX_NDPA_SNDDIALTOKEN state
          rxControlNs = RX_NDPA_SNDDIALTOKEN;

        RX_NDPA_STAINFO1:
        //$fsm_s In RX_NDPA_STAINFO1 state, the state machine waits STA Info 1 field reception
        if ((rxDataValid && ~rxNDPAHE && stateByteCnt[0]  ==1'd1) ||
            (rxDataValid &&  rxNDPAHE && stateByteCnt[1:0]==2'd3))
        begin
          if (rxDataValid && (rxByteCnt == 16'h0004))
            //$fsm_t When STA Info 1 is received & only one STA Info field, the state machine goes to RX_FCS state
            rxControlNs = RX_FCS;
          else
            //$fsm_t When STA Info 1 is received & more than one STA Info field, the state machine goes to RX_NDPA_STAINFON state
            rxControlNs = RX_NDPA_STAINFON;
        end
        else
          //$fsm_t While STA Info 1 is not received, the state machine stays in RX_NDPA_STAINFO1 state
          rxControlNs = RX_NDPA_STAINFO1;

        RX_NDPA_STAINFON:
        //$fsm_s In RX_NDPA_STAINFON state, the state machine waits STA Info N field reception
        if (rxDataValid && (rxByteCnt == 16'h0004))
          //$fsm_t When STA Info N is received & last STA Info field, the state machine goes to RX_FCS state
          rxControlNs = RX_FCS;
        else
          //$fsm_t While STA Info N is not received, the state machine stays in RX_NDPA_STAINFON state
          rxControlNs = RX_NDPA_STAINFON;

        RX_BFRPOLL_SEGRET:
        //$fsm_s In RX_BFRPOLL_SEGRET state, the state machine waits Feedback Segment Retransmission Bitmap field reception
        if (rxDataValid)
          //$fsm_t When Feedback Segment Retransmission Bitmap is received, the state machine goes to RX_FCS state
          rxControlNs = RX_FCS;
        else
          //$fsm_t While Feedback Segment Retransmission Bitmap is not received, the state machine stays in RX_BFRPOLL_SEGRET state
          rxControlNs = RX_BFRPOLL_SEGRET;
`endif //RW_BFMEE_EN

        RX_TRIGGER_CMN_INFO:
        //$fsm_s In RX_TRIGGER_CMN_INFO state, the state machine waits Common Info field reception
        if (rxDataValid && rxTriggerType==4'd5 && stateByteCnt==8'd7)
          //$fsm_t When Common Info field is received and the Trigger is GCR MU-BAR,
          //the state machine goes to RX_TRIGGER_DEP_CMN_INFO state
          rxControlNs = RX_TRIGGER_DEP_CMN_INFO;
        else if (rxDataValid && rxTriggerType!=4'd5 && stateByteCnt==8'd7)
          //$fsm_t When Common Info field is received and the Trigger is not GCR MU-BAR,
          //the state machine goes to RX_TRIGGER_USR_INFO state
          rxControlNs = RX_TRIGGER_USR_INFO;
        else
          //$fsm_t While Common Info field is not received, the state machine stays in
          //RX_TRIGGER_CMN_INFO state
          rxControlNs = RX_TRIGGER_CMN_INFO;

        RX_TRIGGER_DEP_CMN_INFO:
        //$fsm_s In RX_TRIGGER_DEP_CMN_INFO state, the state machine waits Trigger Dependent Common
        //Info field reception
        if (rxDataValid && stateByteCnt==8'd3)
          //$fsm_t When Trigger Dependent Common Info field is received,
          //the state machine goes to RX_TRIGGER_USR_INFO state
          rxControlNs = RX_TRIGGER_USR_INFO;
        else
          //$fsm_t While Trigger Dependent Common Info field is not received, 
          //the state machine stays in RX_TRIGGER_DEP_CMN_INFO state
          rxControlNs = RX_TRIGGER_DEP_CMN_INFO;

        RX_TRIGGER_USR_INFO:
        //$fsm_s In RX_TRIGGER_USR_INFO state, the state machine waits User Info fields reception
        if (rxDataValid && stateByteCnt==8'd4)
        begin
          if (rxTriggerAID4095)
             //$fsm_t When the start of padding User Info field is received,
             //the state machine goes to RX_TRIGGER_PADDING state
             rxControlNs = RX_TRIGGER_PADDING;
          else if (rxTriggerType<=4'd2)
             //$fsm_t When User Info field is received and trigger dependent user info subfield is 
             //present in the trigger frame,
             //the state machine goes to RX_TRIGGER_DEP_USR_INFO state
             rxControlNs = RX_TRIGGER_DEP_USR_INFO;
          else if (rxTriggerType>4'd2 && rxByteCnt==16'd4)
             //$fsm_t When Last User Info field is received and trigger dependent user info
             //subfield is not present in the trigger frame,
             //the state machine goes to RX_FCS state
             rxControlNs = RX_FCS;
          else if (rxTriggerType>4'd2 && (rxTriggerUSRInfoMatch || addr1Match))
             //$fsm_t When My User Info field is received and trigger dependent user info
             //subfield is not present in the trigger frame,
             //the state machine goes to RX_TRIGGER_PADDING state
             rxControlNs = RX_TRIGGER_PADDING;
          else if (rxTriggerType>4'd2 &&  rxByteCnt<16'd9)
             //$fsm_t When trigger dependent user info subfield is not present in the trigger
             //frame and not enougth remaining bytes for an another User Info field,
             //the state machine goes to RX_TRIGGER_PADDING state
             rxControlNs = RX_TRIGGER_PADDING;
          else
             //$fsm_t When not My or not Last User Info field is received and trigger dependent 
             //user info subfield is not present in the trigger frame,
             //the state machine stays in RX_TRIGGER_USR_INFO state
             rxControlNs = RX_TRIGGER_USR_INFO;
        end
        else
          //$fsm_t While Last User Info field is not received, the state machine stays in
          //RX_TRIGGER_USR_INFO state
          rxControlNs = RX_TRIGGER_USR_INFO;

        RX_TRIGGER_DEP_USR_INFO:
        //$fsm_s In RX_TRIGGER_DEP_USR_INFO state, the state machine waits Trigger Dependent User
        //Info fields reception
        if (rxDataValid && stateByteCnt==(trigDepUsrInfoLen-8'b1))
        begin
          if (rxByteCnt==16'd4)
             //$fsm_t When Last trigger dependent User Info field is received,
             //the state machine goes to RX_FCS state
             rxControlNs = RX_FCS;
          else if (rxByteCnt<16'd9)
             //$fsm_t When not enougth remaining bytes for an another User Info field,
             //the state machine goes to RX_TRIGGER_PADDING state
             rxControlNs = RX_TRIGGER_PADDING;
          else if (rxTriggerUSRInfoMatch || addr1Match)
            //$fsm_t When Trigger Dependent User Info field is received and My User Info Field,
            //the state machine goes to RX_TRIGGER_PADDING state
            rxControlNs = RX_TRIGGER_PADDING;
          else
            //$fsm_t When Trigger Dependent User Info field is received and not latest User Info
            //Field the state machine goes to RX_TRIGGER_USR_INFO state
            rxControlNs = RX_TRIGGER_USR_INFO;
        end
        else
          //$fsm_t While Trigger Dependent User Info field is not received, the state machine stays
          //in RX_TRIGGER_DEP_USR_INFO state
          rxControlNs = RX_TRIGGER_DEP_USR_INFO;

        RX_TRIGGER_PADDING:
        //$fsm_s In RX_TRIGGER_PADDING state, the state machine waits end of Trigger padding field
        //reception
        if (rxDataValid && rxByteCnt==16'd4)
          //$fsm_t When Trigger padding field is received,
          //the state machine goes to RX_FCS state
          rxControlNs = RX_FCS;
        else
          //$fsm_t While Trigger padding field is not received, the state machine stays
          //in RX_TRIGGER_PADDING state
          rxControlNs = RX_TRIGGER_PADDING;

      CHECK_FCS:
        //$fsm_s In CHECK_FCS state, the state machine waits FCS status
        if (decryptionCheckOk)
          //$fsm_t When fcsOk is correct and frame is encrypted, the state machine goes to CHECK_ENCR state
          rxControlNs = CHECK_ENCR;
        else
        begin
          if (rxFIFOOverFlowError)
            //$fsm_t When rxFIFOOverFlowError is active, the state machine goes to IDLE state
            rxControlNs = IDLE;
          else if (acceptRcved && !respError && ((!fcsAndLenOk && acceptErrorFrames) || fcsAndLenOk))
            //$fsm_t When control frame has to be forwarded to SW, the state machine goes to WRITE_STATUS state
            rxControlNs = WRITE_STATUS;
          else
            //$fsm_t While control frame has not to be forwarded to SW, the state machine goes to DISCARD_FRAME state
            rxControlNs = DISCARD_FRAME;
        end
  
  
      CHECK_ENCR:
        //$fsm_s In CHECK_ENCR state, the state machine waits status from Encryption engine
        if (rxFIFOOverFlowError)
          //$fsm_t When rxFIFOOverFlowError is active, the state machine goes to IDLE state
          rxControlNs = IDLE;
        else if (decrStatusValidHold)
        begin
          if (acceptRcved && !respError &&
              !discardDecrypt &&
             (((fcsErrorRcved || invalidProtectPktLen) && acceptErrorFrames) ||
              (fcsOkRcved && !invalidProtectPktLen)))
          begin
            //$fsm_t When control frame has to be forwarded to SW, the state machine goes to WRITE_STATUS state
            rxControlNs = WRITE_STATUS;
          end
          else
            //$fsm_t While frame has not to be forwarded to SW, the state machine goes to DISCARD_FRAME state
            rxControlNs = DISCARD_FRAME;
        end
        else
          //$fsm_t While status is not received, the state machine stays in CHECK_ENCR state
          rxControlNs = CHECK_ENCR;
  
      WRITE_STATUS:
        //$fsm_s In WRITE_STATUS state, the state machine waits from the completion of status update in RXFIFO
        if (rxFIFODone_p || rxFIFOOverFlowError) 
          //$fsm_t When rxFIFODone_p is received or rxFIFOOverFlowError is active, the state machine goes to IDLE state
          rxControlNs = IDLE; 
        else
          //$fsm_t While rxFIFODone_p is not received, the state machine stays in WRITE_STATUS state
          rxControlNs = WRITE_STATUS;
  
      RX_ERROR:
        //$fsm_s In RX_ERROR state, the state machine checks acceptErrorFrames register
        if ((macPhyIfRxEndForTimingHoldPatch || (rxAggregation && !rxNDP)))
        begin
          if (acceptErrorFrames && !rxNDP)
            //$fsm_t If acceptErrorFrames is active, the state machine goes to WRITE_STATUS state
            rxControlNs = WRITE_STATUS;
          else if (!discardDone) 
            //$fsm_t While acceptErrorFrames is not active, the state machine goes to DISCARD_FRAME state
            rxControlNs = DISCARD_FRAME;
          else  
            //$fsm_t While acceptErrorFrames is not active, the state machine goes to DISCARD_FRAME state
            rxControlNs = IDLE;
        end
        else
          //$fsm_t While macPhyIfRxEndForTiming is not received, the state machine stays in RX_ERROR state
          rxControlNs = RX_ERROR; 
        
      DISCARD_FRAME:
        //$fsm_s In DISCARD_FRAME state, the state machine goes to IDLE state
        if ((macPhyIfRxEndForTimingHold || (rxAggregation && !rxNDP)))
          //$fsm_t After one clock cycle, the state machine goes to IDLE state
          rxControlNs = IDLE; 
        else
          //$fsm_t While macPhyIfRxEndForTiming is not received, the state machine stays in DISCARD_FRAME state
          rxControlNs = DISCARD_FRAME; 

      // Disable coverage on the default state because it cannot be reached.
      // pragma coverage block = off 
       default:   
          rxControlNs = IDLE; 
      // pragma coverage block = on
      
    endcase
  end
  
end


reg rxEndOfFrame;

// Generation of discardDone which indicates a received frame  
// delayed of one clock cycle
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxEndOfFrame     <= 1'b0;
    rxEndOfFrameRC_p <= 1'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0)
  begin
    rxEndOfFrame     <= 1'b0;
    rxEndOfFrameRC_p <= 1'b0;
  end
  else if (rxEndOfFrameDA_p)
  begin
    rxEndOfFrame     <= 1'b1;
  end
  else if (rxEndOfFrame && (rxControlCs == IDLE))
  begin
    rxEndOfFrameRC_p <= 1'b1;
    rxEndOfFrame     <= 1'b0;
  end
  else
    rxEndOfFrameRC_p <= 1'b0;
  
end

// Generation of discardDone which indicates a received frame  
// delayed of one clock cycle
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    discardDone <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    discardDone <= 1'b0;
  else if (rxControlCs == DISCARD_FRAME)
    discardDone <= 1'b1;
end


// Generation of macPhyIfRxErrInt_p which is macPhyIfRxErr_p
// delayed of one clock cycle
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    macPhyIfRxErrInt_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    macPhyIfRxErrInt_p <= 1'b0;
  else 
    macPhyIfRxErrInt_p <= macPhyIfRxErr_p;
end

// Byte counter by states. This counter maintains the number of
// valid pulses from the A-MPDU deaggregator to be expected in each state.
// Each valid pulse corresponds to a byte received.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset 
    stateByteCnt <= 8'h00;
  else if(macCoreClkSoftRst_n == 1'b0)
    stateByteCnt <= 8'h00;
  else if ((rxControlCs == IDLE) || (rxControlCs == RX_ERROR) ||
           ((rxByteCnt == 16'h0004) && rxDataValid))
    stateByteCnt <= 8'h00;
  else if (rxDataValid)
  begin
    case (rxControlCs)

      // 2 bytes
      RX_FRM_CTRL, RX_DUR_ID, RX_SEQ_CTRL, RX_QOS_CTRL,
      RX_CAR_FRM_CTRL, RX_BA_CTRL, RX_BA_SEQ_CTRL, RX_BCN_INTRVL,
      RX_CAPABILITY:
        if (stateByteCnt == 8'h01)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // 4 bytes
      RX_HT_CTRL, RX_FCS:
        if (stateByteCnt == 8'h03)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // 6 bytes
      RX_ADDR1, RX_ADDR2, RX_ADDR3, RX_ADDR4:
        if (stateByteCnt == 8'h05)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // 8 bytes
      RX_TIMESTAMP:
        if (stateByteCnt == 8'h07)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // 8 / 32 / 128 bytes
      RX_BA_BITMAP:
        if (((stateByteCnt == 8'h07) && compressedBA8 ) ||
            ((stateByteCnt == 8'h1f) && compressedBA32) ||
            ((stateByteCnt == 8'h7f) /* basicBA */    ))
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // Encryption states
      RX_IV_WAPIKEYIDX:
`ifdef RW_WAPI_EN
        if ( (!cTypeWAPI && (stateByteCnt == 8'h03)) || 
             ( cTypeWAPI && (stateByteCnt == 8'h01))    )
`else
        if (stateByteCnt == 8'h03)
`endif //RW_WAPI_EN
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;
      
      
      RX_EXTIV_WAPIPN:
`ifdef RW_WAPI_EN
        if ( (!cTypeWAPI && (stateByteCnt == 8'h03)) || 
             ( cTypeWAPI && (stateByteCnt == 8'h0F))    )
`else
        if (stateByteCnt == 8'h03)
`endif //RW_WAPI_EN
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;
      
      RX_ICV:
        if (stateByteCnt == 8'h03)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;
          
      RX_CCMP_WAPI_MIC:
        if ( ( cTypeCCMP128 && (stateByteCnt == 8'h07)) ||  // MIC =  8 bytes
             (~cTypeCCMP128 && (stateByteCnt == 8'h0F)) )   // MIC = 16 bytes
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // Element body state
      RX_ELEMENT_BODY:
        if (stateByteCnt == (infoElemLen - 8'd1))
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;
      
`ifdef RW_BFMEE_EN
      // 2 / 4 Bytes
      RX_NDPA_STAINFO1:
        if ((~rxNDPAHE && stateByteCnt == 8'h01) || (stateByteCnt == 8'h03))
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      // 2 / 4 Bytes
      RX_NDPA_STAINFON:
        if ((~rxNDPAHE && stateByteCnt == 8'h01) || (stateByteCnt == 8'h03))
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;
`endif //RW_BFMEE_EN

      RX_TRIGGER_CMN_INFO:
        if (stateByteCnt == 8'd7)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      RX_TRIGGER_DEP_CMN_INFO:
        if (stateByteCnt == 8'd3)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      RX_TRIGGER_USR_INFO:
        if (stateByteCnt == 8'd4)
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      RX_TRIGGER_DEP_USR_INFO:
        if (stateByteCnt == (trigDepUsrInfoLen-8'b1))
          stateByteCnt <= 8'h00;
        else
          stateByteCnt <= stateByteCnt + 8'd1;

      default:
        stateByteCnt <= 8'h00;

    endcase
  end
  else
    stateByteCnt <= stateByteCnt;
end

// generate invalidProtectPktLen indicating that protected bit was set but
// pkt length is wrong (less than 12 -> FCS + IV+ICV) or 
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    invalidProtectPktLen <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    invalidProtectPktLen <= 1'b0;
  else if (rxDataValid && protectedBit && (rxByteCnt < 16'h000d) &&
          ((rxControlCs == RX_SEQ_CTRL) || (rxControlCs == RX_ADDR4)  ||
           (rxControlCs == RX_HT_CTRL) || (rxControlCs == RX_QOS_CTRL)))
     invalidProtectPktLen <= 1'b1;
  else if ((rxControlCs == RX_IV_WAPIKEYIDX) && rxDataValid && (stateByteCnt == 8'h03) &&
            rxData[5] == 1'b1 && (rxByteCnt < 16'h000d))
     invalidProtectPktLen <= 1'b1;
end

// generate invalidPkt indicating that pkt is wrong according to RX_FCS state
assign invalidPkt = (rxDataEnd_p || (rxDataValid && (rxByteCnt == 16'h0003))) &&
                    ((rxControlCs != RX_FCS) && (rxControlCs != WAIT_SHIFTMIC_ICV)) && !protocolError;

// Frame decoder command
assign rxFrmCtrlEn           = (rxControlCs == RX_FRM_CTRL);
assign rxCarFrmCtrlEn        = (rxControlCs == RX_CAR_FRM_CTRL);
assign rxDurationIdEn        = (rxControlCs == RX_DUR_ID);
assign rxAddr1En             = (rxControlCs == RX_ADDR1);
assign rxAddr2En             = (rxControlCs == RX_ADDR2);
assign rxAddr3En             = (rxControlCs == RX_ADDR3);
assign rxQoSCtrlEn           = (rxControlCs == RX_QOS_CTRL);
assign rxHTCtrlEn            = (rxControlCs == RX_HT_CTRL);
assign rxSeqCtrlEn           = (rxControlCs == RX_SEQ_CTRL);
assign rxBACtrlEn            = (rxControlCs == RX_BA_CTRL);
assign rxBASeqCtrlEn         = (rxControlCs == RX_BA_SEQ_CTRL);
assign rxAddr4En             = (rxControlCs == RX_ADDR4);
`ifdef RW_WAPI_EN            
assign rxWAPIKeyIdxEn        = (rxControlCs == RX_IV_WAPIKEYIDX) && (stateByteCnt == 8'h00);
assign rxWAPIPNEn            = (rxControlCs == RX_EXTIV_WAPIPN);
assign rxInitVectorEn        = (rxControlCs == RX_IV_WAPIKEYIDX);
assign rxExtInitVectorEn     = (rxControlCs == RX_EXTIV_WAPIPN);
`else // RW_WAPI_EN                       
assign rxInitVectorEn        = (rxControlCs == RX_IV_WAPIKEYIDX);
assign rxExtInitVectorEn     = (rxControlCs == RX_EXTIV_WAPIPN);
`endif // RW_WAPI_EN          
assign rxTimeStampEn         = (rxControlCs == RX_TIMESTAMP);
assign rxDtimCntEn           = ((rxControlCs == RX_ELEMENT_BODY) &&
                                (elementID == 8'h05) &&
                                (stateByteCnt == 8'h00));
assign rxDtimPeriodEn        = ((rxControlCs == RX_ELEMENT_BODY) &&
                                (elementID == 8'h05) &&
                                (stateByteCnt == 8'h01));
assign rxQuietCount1En       = ((rxControlCs == RX_ELEMENT_BODY) &&
                                (elementID == 8'h28) &&
                                (stateByteCnt == 8'h00));
assign rxQuietPeriod1En      = ((rxControlCs == RX_ELEMENT_BODY) &&
                               (elementID == 8'h28) &&
                               (stateByteCnt == 8'h01));
assign rxQuietDuration1En    = ((rxControlCs == RX_ELEMENT_BODY) &&
                               (elementID == 8'h28) &&
                               ((stateByteCnt == 8'h02) ||
                               ((stateByteCnt == 8'h03))));
assign rxQuietOffset1En      = ((rxControlCs == RX_ELEMENT_BODY) &&
                                (elementID == 8'h28) &&
                               ((stateByteCnt == 8'h04) ||
                               ((stateByteCnt == 8'h05))));
`ifdef RW_BFMEE_EN           
assign rxBFRPollSegRetEn     = (rxControlCs == RX_BFRPOLL_SEGRET   );
assign rxNDPASndDialTokenEn  = (rxControlCs == RX_NDPA_SNDDIALTOKEN);
assign rxNDPASTAInfo1En      = (rxControlCs == RX_NDPA_STAINFO1    );
assign rxNDPASTAInfoNEn      = (rxControlCs == RX_NDPA_STAINFON    );
`endif //RW_BFMEE_EN         
assign rxTriggerCMNInfoEn    = (rxControlCs == RX_TRIGGER_CMN_INFO);
assign rxTriggerUSRInfoEn    = (rxControlCs == RX_TRIGGER_USR_INFO);
assign rxTriggerDepUSRInfoEn = (rxControlCs == RX_TRIGGER_DEP_USR_INFO);
                             
assign fcsCheck_p            = (rxControlCs == CHECK_FCS);
assign rxDurationIdEnd_p     = (rxControlCs == RX_DUR_ID) &&
                               (stateByteCnt == 8'h01) && rxDataValid;

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxAddr1End_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxAddr1End_p <= 1'b0;
  else if ((rxControlCs == RX_ADDR1) && (stateByteCnt == 8'h05) && rxDataValid)
    rxAddr1End_p <= 1'b1;
  else
    rxAddr1End_p <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxAddr2End_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxAddr2End_p <= 1'b0;
  else if ((rxControlCs == RX_ADDR2) && (stateByteCnt == 8'h05) && rxDataValid)
    rxAddr2End_p <= 1'b1;
  else
    rxAddr2End_p <= 1'b0;
end


always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    carFrameCtrlEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    carFrameCtrlEnd_p <= 1'b0;
  else if ((rxControlCs == RX_CAR_FRM_CTRL) && (stateByteCnt == 8'h01) && rxDataValid)
    carFrameCtrlEnd_p <= 1'b1;
  else
    carFrameCtrlEnd_p <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxAddr3End_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxAddr3End_p <= 1'b0;
  else if ((rxControlCs == RX_ADDR3) && (stateByteCnt == 8'h05) && rxDataValid)
    rxAddr3End_p <= 1'b1;
  else
    rxAddr3End_p <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxAddr4End_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxAddr4End_p <= 1'b0;
  else if ((rxControlCs == RX_ADDR4) && (stateByteCnt == 8'h05) && rxDataValid)
    rxAddr4End_p <= 1'b1;
  else
    rxAddr4End_p <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxSeqCtrlEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxSeqCtrlEnd_p <= 1'b0;
  else if ((rxControlCs == RX_SEQ_CTRL) && (stateByteCnt == 8'h01) && rxDataValid)
    rxSeqCtrlEnd_p <= 1'b1;
  else
    rxSeqCtrlEnd_p <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxQoSEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxQoSEnd_p <= 1'b0;
  else if ((rxControlCs == RX_QOS_CTRL) && (stateByteCnt == 8'h01) && rxDataValid)
    rxQoSEnd_p <= 1'b1;
  else
    rxQoSEnd_p <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxHTCtrlEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxHTCtrlEnd_p <= 1'b0;
  else if ((rxControlCs == RX_HT_CTRL) && (stateByteCnt == 8'h03) && rxDataValid)
    rxHTCtrlEnd_p <= 1'b1;
  else
    rxHTCtrlEnd_p <= 1'b0;
end

assign rxBACtrlEnd_p      = (rxControlCs == RX_BA_CTRL) &&
                             (stateByteCnt == 8'h01) && rxDataValid;

assign rxBARInfoEnd_p      = (rxControlCs == RX_BA_SEQ_CTRL) &&
                             (stateByteCnt == 8'h01) && rxDataValid;

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxInitVectorEnd_p <= 1'b0;
    rxExtIV_p         <= 1'b0;
  end
  else if(macCoreClkSoftRst_n == 1'b0)
  begin
    rxInitVectorEnd_p <= 1'b0;
    rxExtIV_p         <= 1'b0;
  end
  else if (acceptRcved && (rxControlCs == RX_IV_WAPIKEYIDX) && ((rxControlNs == RX_EXTIV_WAPIPN) || (rxControlNs == WAIT_KEYSEARCHDONE)))
  begin
    rxInitVectorEnd_p <= 1'b1;
    if(rxControlNs == RX_EXTIV_WAPIPN)
      rxExtIV_p <= 1'b1;
  end
  else
  begin
    rxExtIV_p         <= 1'b0;
    rxInitVectorEnd_p <= 1'b0;
  end
end

`ifdef RW_BFMEE_EN
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxNDPAStaInfoEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxNDPAStaInfoEnd_p <= 1'b0;
  else if (((rxControlCs == RX_NDPA_STAINFO1) || (rxControlCs == RX_NDPA_STAINFON)) && (rxControlNs == RX_FCS))
    rxNDPAStaInfoEnd_p <= 1'b1;
  else
    rxNDPAStaInfoEnd_p <= 1'b0;
end
`endif // RW_BFMEE_EN

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxTriggerUSRInfoEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxTriggerUSRInfoEnd_p <= 1'b0;
  else if (((rxControlCs == RX_TRIGGER_USR_INFO) || (rxControlCs == RX_TRIGGER_DEP_USR_INFO)) &&
           ((rxControlNs == RX_TRIGGER_PADDING)  || (rxControlNs == RX_FCS)))
    rxTriggerUSRInfoEnd_p <= 1'b1;
  else
    rxTriggerUSRInfoEnd_p <= 1'b0;
end

assign rxControlIdle         = (rxControlCs == IDLE) && ~rxDataStart_p;
assign rxControlBackToIdle_p = (rxControlCs != IDLE) && (rxControlNs == IDLE);


// Flag indicating the completion of the MAC Header
assign macHeaderCompleted = (rxControlCs != IDLE) && 
                            (rxControlCs != RX_FRM_CTRL) && 
                            (rxControlCs != RX_DUR_ID) && 
                            (rxControlCs != RX_ADDR1) && 
                            (rxControlCs != RX_ADDR2) && 
                            (rxControlCs != RX_ADDR3) && 
                            (rxControlCs != RX_ADDR4) && 
                            (rxControlCs != RX_SEQ_CTRL) && 
                            (rxControlCs != RX_QOS_CTRL) && 
                            (rxControlCs != RX_CAR_FRM_CTRL) && 
                            (rxControlCs != RX_HT_CTRL) && 
                            (rxControlCs != RX_IV_WAPIKEYIDX) && 
                            (rxControlCs != RX_EXTIV_WAPIPN);

assign searchMACAddr_p = rxAddr2End_p;


// Beacon element sampling
// information element length is sampled and used in
// RX_ELEMENT_BODY state to determine next state
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    infoElemLen <= 8'h00;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    infoElemLen <= 8'h00;
  else if ((rxControlCs == RX_ELEMENT_LEN) && rxDataValid)
    infoElemLen <= rxData;
end

// Sample element ID field of beacon
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    elementID  <= 8'h00;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    elementID  <= 8'h00;
  else if ((rxControlCs == RX_ELEMENT_ID) && rxDataValid)
    elementID <= rxData;
end

// Detect DTIM parameters
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    dtimParameterFound <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    dtimParameterFound <= 1'b0;
  else if (elementID == 8'h05)
    dtimParameterFound <= 1'b1;
end

// Detect Quiet parameters
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    quietParameterFound <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    quietParameterFound <= 1'b0;
  else if (rxControlCs == IDLE)
    quietParameterFound <= 1'b0;
  else if (elementID == 8'h28)
    quietParameterFound <= 1'b1;
end



always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxPayloadEnd_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    rxPayloadEnd_p <= 1'b0;
  else if ((rxControlCs == RX_FCS) && (stateByteCnt == 8'h03))
    rxPayloadEnd_p <= 1'b1;
  else
    rxPayloadEnd_p <= 1'b0;
end



// FCS Control
assign fcsStartRx_p = rxDataStart_p;

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    fcsEnableRx <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    fcsEnableRx <= 1'b0;
  else if (rxDataStart_p)
    fcsEnableRx <= 1'b1;
  else if ((rxControlCs == IDLE) || (rxControlCs == CHECK_FCS) ||
    ((rxControlCs == RX_ERROR) || (rxControlCs == DISCARD_FRAME)))
    fcsEnableRx <= 1'b0;
end


// RX FIFO Interface Controller -> MPDU Status Information
// Frame successful for DMA
assign frmSuccessfulRcved = (fcsOkRcved)           &&
                            (!phyErrorRcved)       &&
                            (!undefErrorRcved)     &&
                            (!rxFIFOOverFlowError) &&
                            (!decryptionError);
                            
// End of Encyption Engine procedure
assign decrStatusValid_p = decryptPassed_p || decryptFailed_p;

// Decryption Error flag.
//   Set   when protected frame is received (decryption error by default)
//   Clear when decryption is done without error.
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    decryptionError <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    decryptionError <= 1'b0;
  else if (decryptFailed_p | rxDurationIdEnd_p & protectedBit)
    decryptionError <= 1'b1;
  else if (decryptPassed_p)
    decryptionError <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    decrStatusValidHold <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
    decrStatusValidHold <= 1'b0;
  else if (decrStatusValid_p)
    decrStatusValidHold <= 1'b1;
end


// Decryption Type :
// 4'b0000: Frame was not encrypted
// 4'b0001: WEP
// 4'b0010: TKIP
// 4'b0011: CCMP-128
// 4'b0100: CCMP-256
// 4'b0101: GCMP-128
// 4'b0110: GCMP-256
// 4'b0111: WAPI 
// 4'b1111: Null Key Found
always @*
begin
   if (!protectedBit)
      decryptionType = 4'b0000;
   else
   begin
      casex ({cTypeKSR,cLenKSR})
      {ENCRYPTION_TYPE_NONE, 2'bxx}: decryptionType = 4'b1111;
      {ENCRYPTION_TYPE_WEP , 2'bxx}: decryptionType = 4'b0001;
      {ENCRYPTION_TYPE_TKIP, 2'bxx}: decryptionType = 4'b0010;
      {ENCRYPTION_TYPE_CCMP, 2'b00}: decryptionType = 4'b0011;
      {ENCRYPTION_TYPE_CCMP, 2'b10}: decryptionType = 4'b0100;
      {ENCRYPTION_TYPE_GCMP, 2'b00}: decryptionType = 4'b0101;
      {ENCRYPTION_TYPE_GCMP, 2'b10}: decryptionType = 4'b0110;
`ifdef  RW_WAPI_EN                           
      {ENCRYPTION_TYPE_WAPI, 2'bxx}: decryptionType = 4'b0111;
`endif//RW_WAPI_EN                           
      default:                       decryptionType = 4'b1111;
      endcase
   end
end


// Flag indicating if the frame has to be discarded due to encryption failure.
assign discardDecrypt = !acceptDecryptErrorFrames && decryptionError;


// Status information control
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    fcsOkRcved          <= 1'b0;
    fcsErrorRcved       <= 1'b0;
    phyErrorRcved       <= 1'b0;
    undefErrorRcved     <= 1'b0;
    rxFIFOOverFlowError <= 1'b0;
    statusInfoCheck_p   <= 1'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
  begin
    fcsOkRcved          <= 1'b0;
    fcsErrorRcved       <= 1'b0;
    phyErrorRcved       <= 1'b0;
    undefErrorRcved     <= 1'b0;
    rxFIFOOverFlowError <= 1'b0;
    statusInfoCheck_p   <= 1'b0;
  end
  else
  begin
    // Status Information pulse
    statusInfoCheck_p <= fcsCheck_p;
    // PHY error status
    if (macPhyIfRxErr_p || rxDataError_p)
      phyErrorRcved <= 1'b1;
    // FIFO overflow status
    if (rxFIFOOverFlow || rxFIFOFull)
      rxFIFOOverFlowError <= 1'b1;
    // FCS status sampling
    if (invalidPkt)
      fcsErrorRcved <= 1'b1;
    else if (fcsCheck_p)
    begin
      fcsOkRcved    <= fcsAndLenOk;
      fcsErrorRcved <= !fcsAndLenOk;
    end
    // Undefined error status
    if ((/*statusInfoCheck_p && */protocolError) || macPHYIFOverflow)
      undefErrorRcved <= 1'b1;
  end
end

// RX FIFO Interface Controller -> Payload length in buffers calculation
assign rxByteCntTrig = ((rxControlNs == RX_FRM_BODY)  ||
                        (rxControlNs == WAIT_KEYSEARCHDONE)  ||
                        (rxControlNs == RX_BA_CTRL)   ||
                        (rxControlNs == RX_TIMESTAMP) ||
`ifdef RW_BFMEE_EN
                        (rxControlNs == RX_BFRPOLL_SEGRET   ) ||
                        (rxControlNs == RX_NDPA_SNDDIALTOKEN) ||
`endif //RW_BFMEE_EN
                        (rxControlNs == RX_TRIGGER_CMN_INFO) ||
                       ((rxControlNs == RX_ADDR2) && ctrlWrapFlag));

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    payloadLength     <= 16'b0;
    rxByteCntTrig_ff1 <= 1'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlCs == IDLE))
  begin
    payloadLength     <= 16'b0;
    rxByteCntTrig_ff1 <= 1'b0;
  end
  else 
  begin
    rxByteCntTrig_ff1 <= rxByteCntTrig;
    if (rxByteCntTrig && !rxByteCntTrig_ff1)
      // payloadLength = rxByteCnt
      payloadLength <= rxByteCnt - 16'h4; //Removed FCS
  end
end

assign payloadEnd_p = (rxByteCnt == payloadLengthSub);

// payloadBufferLength
// Non-encrypted : Payload length - 4 (FCS)
// WEP/TKIP      : Payload length - 4 (ICV) - 4 (FCS)
// CCMP-128      : Payload length - 8 (MIC) - 4 (FCS)
// CCMP-256      : Payload length - 16 (MIC) - 4 (FCS)
// GCMP          : Payload length - 16 (MIC) - 4 (FCS)

always @*
begin
  if (protectedBit && (cTypeWEP || cTypeTKIP))
    // WEP/TKIP
    payloadLengthSub = 16'h0008;      //ICV 4 + FCS 4 = 8
  else if (protectedBit && cTypeCCMP128)
    // CCMP-128
    payloadLengthSub = 16'h000c;      //MIC 8 + FCS 4 = 12
  else if (protectedBit && cTypeCCMP256 ||
           protectedBit && cTypeGCMP    )
    // CCMP-256 / GCMP
    payloadLengthSub = 16'h0014;      //MIC 16 + FCS 4 = 20
`ifdef RW_WAPI_EN
  else if (protectedBit && cTypeWAPI)
    payloadLengthSub = 16'h0014;      //MIC 16 + FCS 4 = 20
`endif // RW_WAPI_EN
  else
    // No frame body
    payloadLengthSub = 16'h0004;
end

// RX FIFO Interface Controller -> Quadlet completion
assign frmBody      = (rxControlCs == RX_FRM_BODY ) || 
                      (rxControlCs == RX_BA_BITMAP) ||
                      (rxControlCs == RX_BA_INFO  ) ||
`ifdef RW_BFMEE_EN
                      (rxControlCs == RX_BFRPOLL_SEGRET) ||
                      (rxControlCs == RX_NDPA_STAINFO1 ) ||
                      (rxControlCs == RX_NDPA_STAINFON ) ||
`endif //RW_BFMEE_EN
                      (rxControlCs == RX_ELEMENT_ID) ||
                      (rxControlCs == RX_ELEMENT_LEN) ||
                      (rxControlCs == RX_ELEMENT_BODY) ||
                      (rxControlCs == RX_TRIGGER_USR_INFO) ||
                      (rxControlCs == RX_TRIGGER_DEP_USR_INFO) ||
                      (rxControlCs == RX_TRIGGER_PADDING);

// RX FIFO Interface Controller -> Tag FIFO
assign startMacHdr  = (rxControlCs == RX_FRM_CTRL      ) ||
                      (rxControlCs == RX_DUR_ID        );

assign macHdr       = (rxControlCs == RX_FRM_CTRL      ) ||
                      (rxControlCs == RX_DUR_ID        ) ||
                      (rxControlCs == RX_ADDR1         ) ||
                      (rxControlCs == RX_ADDR2         ) ||
                      (rxControlCs == RX_ADDR3         ) ||
                      (rxControlCs == RX_ADDR4         ) ||
                      (rxControlCs == RX_SEQ_CTRL      ) ||
                      (rxControlCs == RX_QOS_CTRL      ) ||
                      (rxControlCs == RX_CAR_FRM_CTRL  ) ||
                      (rxControlCs == RX_HT_CTRL       ) ||
                      (rxControlCs == RX_IV_WAPIKEYIDX ) ||
                      (rxControlCs == RX_EXTIV_WAPIPN  );

assign endMacHdr_p  = macHdr && 
                      (rxControlNs != RX_FRM_CTRL      ) &&
                      (rxControlNs != RX_DUR_ID        ) &&
                      (rxControlNs != RX_ADDR1         ) &&
                      (rxControlNs != RX_ADDR2         ) &&
                      (rxControlNs != RX_ADDR3         ) &&
                      (rxControlNs != RX_ADDR4         ) &&
                      (rxControlNs != RX_SEQ_CTRL      ) &&
                      (rxControlNs != RX_QOS_CTRL      ) &&
                      (rxControlNs != RX_CAR_FRM_CTRL  ) &&
                      (rxControlNs != RX_HT_CTRL       ) &&
                      (rxControlNs != RX_IV_WAPIKEYIDX ) &&
                      (rxControlNs != RX_EXTIV_WAPIPN  );

assign discardFrm   = (rxControlCs == DISCARD_FRAME);

assign acceptError_p = (rxControlCs == RX_ERROR) && (rxControlNs == WRITE_STATUS);

// RX FIFO Interface Controller -> Trailer information
assign writeTrailer = (rxControlCs == WRITE_STATUS) && (macPhyIfRxEndForTimingHold || (rxAggregation && !rxNDP)) && !rxFIFODone_p;

// Used to work around a Synplify issue.
assign macPhyIfRxEndForTimingHoldPatch = macPhyIfRxEndForTimingHold; 

// if macPhyIfRxEndForTiming_p is received, set the macPhyIfRxEndForTimingHold flag
// if FSM moves back to IDLE or if macPhyIfRxErrInt_p is received in IDLE state, clear the flag
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    macPhyIfRxEndForTimingHold <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    macPhyIfRxEndForTimingHold <= 1'b0;
  else if (((rxControlCs != IDLE) &&  (rxControlNs == IDLE)) ||
           ((rxControlCs == IDLE) && macPhyIfRxErrInt_p))
    macPhyIfRxEndForTimingHold <= 1'b0;
  else if (macPhyIfRxEndForTiming_p)
    macPhyIfRxEndForTimingHold <=  1'b1;
end

// rxCntrlReady control
assign rxCntrlReady = !((rxControlCs == WRITE_STATUS)      ||
                        ((rxControlCs == RX_SEQ_CTRL)  && !rxIndexSearchDone)      ||  // Guaranty the completion of keySearch for frame containing seqControl field
                        (rxControlCs == WAIT_KEYSEARCHDONE) || (rxControlNs == WAIT_KEYSEARCHDONE)     ||  // Guaranty the completion of keySearch for frame containing seqControl field
                        ((rxControlCs == RX_FCS)  && !rxIndexSearchDone)      ||  // Guaranty the completion of keySearch for frame containing with default Key
                        ((rxControlCs == RX_FCS) && (rxFrmDiscard == 1'b0) && (rxDescAvailable == 1'b0)) ||
                        ((rxControlCs == RX_FCS) && (rxByteCnt <= 16'h0004) &&!macPhyIfRxEndForTimingHold && !rxAggregation) || 
                        (rxControlNs == WAIT_SHIFTMIC_ICV) ||
                        (rxControlCs == CHECK_ENCR) ||
                        (rxControlCs == RX_ERROR) ||
                        endMacHdr_p || // Give one cycle to write MAC Header Length to RX Fifo
                         fcsCheck_p || rxDataEnd_p || discardFrm);
  



always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    lenIs0InFCSCheck <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    lenIs0InFCSCheck <= 1'b0;
  else if (rxControlCs == RX_FCS)
    lenIs0InFCSCheck <=  (rxByteCnt == 16'h0000);
end

// Local frame status based on FCS and frame Length.
assign fcsAndLenOk = fcsOk && lenIs0InFCSCheck;

assign processingEnd_p = fcsCheck_p;

assign fcsOkRcved_p          = processingEnd_p &&  fcsOk;

assign localCorrectRcved_p   = processingEnd_p &&  fcsOk;
assign localIncorrectRcved_p = processingEnd_p && !fcsOk;

// MAC pulse control according to local frame status and error cases
assign correctRcved_p        = localCorrectRcved_p && !rxFIFOOverFlowError && !respError;
assign incorrectRcved_p      = (processingEnd_p && (!fcsOk || rxFIFOOverFlowError || respError || invalidPkt)) ||
                               ((rxControlCs == RX_ERROR) && macPhyIfRxEndForTimingHold && !discardDone)       ||
                               (protocolError && (rxControlCs == RX_FRM_BODY) && (rxByteCnt == 16'h0000) && rxDataValid);


// In case of non-expected response, the flag respError is set. It will generate an incorrectRcved_p
// If a CTS is expected and the frame received if not a CTS or
// If a ACK/BA is expected and the frame received if not a BA or
// If an ACK/BA is expected and the frame received if not an ACK or
// If an ACK/BA(BFR) is expected and the frame received if not an BFR or
// a CTS, BA or ACK is expected and the frame received is part of AMPDU exepted for BFR(VHT format->AMPDU)
assign respError = ((rxCts && !ctsRcved) ||
`ifdef RW_BFMER_EN
                    (rxBfr && !bfrRcved) ||
`endif // RW_BFMER_EN
                    (rxAck && !(baRcved || ackRcved)) ||
                    ((rxCts || rxAck) && rxAggregation)) ? 1'b1 : 1'b0;


// fcsOk Received pulse delayed by 1 clock cycle
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    fcsOkRcvedDly_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    fcsOkRcvedDly_p <= 1'b0;
  else 
    fcsOkRcvedDly_p <= fcsOkRcved_p;
end

// NAV clear
//    Basic NAV is clear when inter-BSS CF-END is received
//    Intra NAV is clear when intra-BSS CF-END is received
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    navClear <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    navClear <= 1'b0;
  else if (rxVector1Valid_p)
    navClear <= 1'b0;
  else if (fcsOkRcved_p & cfEndRcved)
    navClear <= 1'b1;
end

assign basicNAVClear_p = rxEndOfFrameRC_p & navClear & interBSSRcved;
assign intraNAVClear_p = rxEndOfFrameRC_p & navClear & intraBSSRcved;


// Basic NAV Update
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    basicNAVUpdateFromVector <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    basicNAVUpdateFromVector <= 1'b0;
  else if (rxVector1Valid_p)
  begin
    if (rxFormatMod>=4'd5 & rxTxopDuration!={7{1'b1}})
      basicNAVUpdateFromVector <= 1'b1;
    else
      basicNAVUpdateFromVector <= 1'b0;
  end
  else if (fcsOkRcved_p & ~rxFrameDuration[15])
    basicNAVUpdateFromVector <= 1'b0;
end

always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    basicNAVUpdateFromHeader <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    basicNAVUpdateFromHeader <= 1'b0;
  else if (rxVector1Valid_p)
    basicNAVUpdateFromHeader <= 1'b0;
  else if (fcsOkRcved_p & ~rxFrameDuration[15] & ~addr1Match & ~bcMcRcved)
    basicNAVUpdateFromHeader <= 1'b1;
end

assign basicNAVUpdate_p = rxEndOfFrameRC_p & ~intraBSSRcved & 
                          (basicNAVUpdateFromVector | basicNAVUpdateFromHeader);


// Intra NAV update
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    intraNAVUpdateFromVector <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    intraNAVUpdateFromVector <= 1'b0;
  else if (rxVector1Valid_p)
  begin
    if (~txInProgressDly & rxFormatMod>=4'd5 & rxTxopDuration!={7{1'b1}})
      intraNAVUpdateFromVector <= 1'b1;
    else
      intraNAVUpdateFromVector <= 1'b0;
  end
  else if (fcsOkRcved_p & ~rxFrameDuration[15])
    intraNAVUpdateFromVector <= 1'b0;
end

always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    intraNAVUpdateFromHeader <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    intraNAVUpdateFromHeader <= 1'b0;
  else if (rxVector1Valid_p)
    intraNAVUpdateFromHeader <= 1'b0;
  else if (fcsOkRcvedDly_p & ~rxFrameDuration[15])
  // fcsOkRcvedDly_p is used, because frameExpectingResp is updated on fcsOkRcved_p 
  begin
    if (~addr1Match & ~bcMcRcved)
      intraNAVUpdateFromHeader <= 1'b1;
    else if (~txInProgressDly & ~frameExpectingResp)
      intraNAVUpdateFromHeader <= 1'b1;
    else if (~txInProgressDly & triggerRcved)
      intraNAVUpdateFromHeader <= 1'b1;
  end
end

assign intraNAVUpdate_p = rxEndOfFrameRC_p & intraBSSRcved &
                          (intraNAVUpdateFromVector | intraNAVUpdateFromHeader);


// Duration for the NAV update
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    navUpdateDuration <= 15'd0;
  else if(macCoreClkSoftRst_n == 1'b0)
    navUpdateDuration <= 15'd0;
  else if (rxVector1Valid_p & rxFormatMod>=4'd5)
  begin
    if (rxTxopDuration=={7{1'b1}})
    begin
       // Txop is unspecified
       navUpdateDuration <= 15'd0;
    end
    else if (rxTxopDuration[0])
    begin
       // NAV = txopDuration*128 + 512
       navUpdateDuration <= {1'b0, {1'b0,rxTxopDuration[6:1]}+7'd4, 7'd0};
    end
    else
    begin
       // NAV = txopDuration*8
       navUpdateDuration <= {6'd0, rxTxopDuration[6:1], 3'd0};
    end
  end
  else if (fcsOkRcved_p & ~rxFrameDuration[15])
  begin
     // NAV = Duration/ID
     navUpdateDuration <= rxFrameDuration[14:0];
  end
end


// MAC Timer unit pulse control
assign dtimUpdate_p = tsfUpdate_p && dtimParameterFound && bssType && !ap;

// If the TSF Management is disabled, the tsfUpdated_p is not generated.
assign tsfUpdate_p  = !tsfMgtDisable && localCorrectRcved_p && bssIDMatch &&
                     (bcnRcved || (probRespRcved && addr1Match));
            

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    tsfOffsetUpdate_p <= 1'b0;
  else if(macCoreClkSoftRst_n == 1'b0)
    tsfOffsetUpdate_p <= 1'b0;
  else if ((rxControlCs == RX_TIMESTAMP) && (stateByteCnt == 8'h07) && rxDataValid)
    tsfOffsetUpdate_p <= 1'b1;
  else 
    tsfOffsetUpdate_p <= 1'b0;
end

// Last received packet status for EIFS calculation
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    lastRxWrong <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    lastRxWrong <= 1'b0;
  else if (fcsOkRcved)
    lastRxWrong <= 1'b0;
  else if (mpIfTxEn || backOffDone_p || rxDataStart_p)
    lastRxWrong <= 1'b0;
  else if ((rxControlCs == RX_ERROR) || fcsErrorRcved || macPhyIfRxErr_p)
    lastRxWrong <= 1'b1;
end


// Control and Status Register pulse control
assign quietElement1InValid  = tsfUpdate_p && quietParameterFound;
assign dtimPeriodInValid     = dtimUpdate_p && (currentState == ACTIVE_STATE);

// BA Controller control
assign psBitmapUpdate_p  = correctRcved_p && 
                           (((barRcved || qosFrame) && (!qosNullRcved) && addr1Match) ||
                            (muBARTriggerRcved && rxTriggerUSRInfoMatch));
assign psBitmapDiscard_p = incorrectRcved_p ||
                           (correctRcved_p &&
                            ((!(barRcved || qosFrame || muBARTriggerRcved) ||
                            (qosNullRcved) ||
                            (!muBARTriggerRcved && !addr1Match) ||
                            ( muBARTriggerRcved && !rxTriggerUSRInfoMatch))));
assign baEnable          = (rxControlCs != IDLE) &&
                           (barRcved || qosFrame || muBARTriggerRcved) &&
                           (!qosNullRcved);

`ifdef RW_BFMEE_EN
// BFR Controller
assign rxBFRPollValid_p   = correctRcved_p && bfrPollRcved && addr1Match;
assign rxNDPAValid_p      = correctRcved_p && ndpaRcved &&
                            (addr1Match || (broadCastRcved && rxNDPAAIDMatch));
`endif //RW_BFMEE_EN

// MAC Controller RX
assign rxTriggerHWValid_p = correctRcved_p && triggerHWRcved &&
                            rxTriggerUSRInfoMatch && (addr1Match || broadCastRcved);
assign rxTriggerSWValid_p = correctRcved_p && triggerSWRcved && acceptRcved &&
                            rxTriggerUSRInfoMatch && (addr1Match || broadCastRcved);

// RX FIFO interface controller, capture rxTriggerSWValid_p pulse
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxTriggerSWValid <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxTriggerSWValid <= 1'b0;
  else if (rxTriggerSWValid_p)
    rxTriggerSWValid <= 1'b1;
end

// Frame Decoder, Valid trigger frame found flag
// Indicates that a valid trigger frame has been found in the received PPDU.
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxTriggerFound <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0 || rxEndOfFrameRC_p)
    rxTriggerFound <= 1'b0;
  else if (triggerRcved & fcsOkRcved_p)
    rxTriggerFound <= 1'b1;
end

// Decryption FSM control
//assign rxControlToFrmBody_p = (rxControlNs == RX_FRM_BODY) && (rxControlCs != RX_FRM_BODY);
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxControlToFrmBody_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxControlToFrmBody_p <= 1'b0;
  else if ((rxControlNs == RX_FRM_BODY) && (rxControlCs != RX_FRM_BODY))
    rxControlToFrmBody_p <= 1'b1;
  else 
    rxControlToFrmBody_p <= 1'b0;
end

// sample two first byte of frame body to detect Action frame ( Category & Action field)
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrmBodyByte1En <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxFrmBodyByte1En <= 1'b0;
  else if (rxControlToFrmBody_p == 1'b1)
    rxFrmBodyByte1En <= 1'b1;
  else if ((rxFrmBodyByte1En == 1'b1) && (rxDataValid == 1'b1)) 
    rxFrmBodyByte1En <= 1'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrmBodyByte2En <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxFrmBodyByte2En <= 1'b0;
  else if (rxFrmBodyByte1En == 1'b1)
    rxFrmBodyByte2En <= 1'b1;
  else if ((rxFrmBodyByte2En == 1'b1) && (rxDataValid == 1'b1)) 
    rxFrmBodyByte2En <= 1'b0;
end

`ifdef RW_BFMER_EN
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFrmBodyByte2End_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxFrmBodyByte2End_p <= 1'b0;
  else if ((rxFrmBodyByte2En == 1'b1) && (rxDataValid == 1'b1)) 
    rxFrmBodyByte2End_p <= 1'b1;
  else 
    rxFrmBodyByte2End_p <= 1'b0;
end
`endif // RW_BFMER_EN

assign rxControlToError_p   = ((rxControlNs == RX_ERROR) && (rxControlCs != RX_ERROR)) ||
                              ((rxControlNs == DISCARD_FRAME) && (rxControlCs != DISCARD_FRAME));

assign acceptProtected      = acceptRcved && protectedBit && !dontDecrypt;


// Encryption Engine control
assign rxPayLoadLen         = payloadLength[15:0];




//------------------------------------------------------------------------------
// Intra-BSS and inter-BSS frame determination
//------------------------------------------------------------------------------
// Address2 present flag
assign addr2Pres = ~ackRcved & ~ctsRcved;


// TXOP Holder save
//   In case of received frame with both RA/TA present and RA is the BSS ID, the
//   TA is saved.
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txopHolderOfBSS <= 48'h0;
  else if (macCoreClkSoftRst_n == 1'b0)
    txopHolderOfBSS <= 48'h0;
  else if (fcsOkRcved_p & addr1MatchBssID & addr2Pres)
    txopHolderOfBSS <= rxAddr2;                              //ToDo: Maybe should be cleared ?
                                                             // Timeout, other .... ?
end


// A frame is an Inter-BSS frame if
// - received PPDU with BSS Color not null and is not our BSS color.
// - received HE PPDU with BSS Color not null and associated with a non-HE AP.
// - received VHT PPDU with Group ID 0  and Partial AID not equal to the BSS ID 
// - received VHT PPDU with Group ID 63 and Partial AID not equal to the partial
//   BSS color
// - received VHT MU PPDU and is an AP
// - received HE  MU PPDU with Up Link Flag 0 and is an AP
// - received PPDU carries a frame that has a BSSID field and is not our BSS ID 
//   or the wildcard BSS ID.
// - received PPDU carries a frame that does not have a BSSID field but has both
//   an RA and TA fields, neither value of which is equal to the BSSID of the BSS.
assign interBSSFromBssColor =
   (rxFormatMod>=4'd5 & rxBssColor!=6'd0 & rxBssColor!=bssColorCSReg & bssColorEnCSReg);

assign interBSSFromRxVector = 
   interBSSFromBssColor |
   (rxFormatMod==4'd4 & rxGroupID==6'd0  & rxPartialAID!=bssID[47:39]) |
   (rxFormatMod==4'd4 & rxGroupID==6'd63 & rxPartialAID[8:5]!=bssColorCSReg[3:0] & partialBSSColorEnCSReg) |
   (rxFormatMod==4'd4 & rxGroupID>=6'd1  & rxGroupID<=6'd62 & ap) |
   (rxFormatMod==4'd6 & rxUplinkFlag==1'b0 & ap);

always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      interBSSFromHeader <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0)
      interBSSFromHeader <= 1'b0;
   else if (rxVector1Valid_p)
      interBSSFromHeader <= 1'b0;
   else if ((fcsOkRcved_p &  bssIDFrame & ~bssIDMatch & ~wildcardBSSID                   ) |
            (fcsOkRcved_p & ~bssIDFrame & ~addr1MatchBssID & ~addr2MatchBssID & addr2Pres))
      interBSSFromHeader <= 1'b1;
end


// A frame is an Intra-BSS frame if
// - received PPDU with BSS color null or is our BSS color.
// - received VHT PPDU with Group ID 0  and Partial AID equal to the BSS ID
// - received VHT PPDU with Group ID 63 and Partial AID equal to the partial BSS
//   color
// - received PPDU carries a frame that has an RA, TA or BSSID field value that
//   is equal to our BSSID.
// - received PPDU carries a Control frame that does not have a TA field 
//   (ACK or CTS) and that has an RA field value that matches the saved TXOP
//   holder address of the our BSS.
assign intraBSSFromBssColor =
   (rxFormatMod>=4'd5 & rxBssColor==bssColorCSReg & bssColorEnCSReg);

assign intraBSSFromRxVector =
   intraBSSFromBssColor |
   (rxFormatMod==4'd4 & rxGroupID==6'd0  & rxPartialAID==bssID[47:39]) |
   (rxFormatMod==4'd4 & rxGroupID==6'd63 & rxPartialAID[8:5]==bssColorCSReg[3:0] & partialBSSColorEnCSReg);

always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      intraBSSFromHeader <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0)
      intraBSSFromHeader <= 1'b0;
   else if (rxVector1Valid_p)
      intraBSSFromHeader <= 1'b0;
   else if ((fcsOkRcved_p & addr1MatchBssID                                    ) |
            (fcsOkRcved_p & addr2MatchBssID                                    ) |
            (fcsOkRcved_p & bssIDMatch                                         ) |
            (fcsOkRcved_p & ~addr2Pres & (txopHolderOfBSS==rxAddr1 | addr1Match)))
      intraBSSFromHeader <= 1'b1;
end


// Intra-BSS / Inter-BSS Classification
always @*
begin
   if (~interBSSFromRxVector & ~interBSSFromHeader &
       ~intraBSSFromRxVector & ~intraBSSFromHeader)
   // Not Classified
   begin
      interBSSRcved = 1'b0;
      intraBSSRcved = 1'b0;
   end
   else if (~interBSSFromRxVector & ~interBSSFromHeader)
   // Intra-BSS Only
   begin
      interBSSRcved = 1'b0;
      intraBSSRcved = 1'b1;
   end
   else if (~intraBSSFromRxVector & ~intraBSSFromHeader)
   // Inter-BSS Only
   begin
      interBSSRcved = 1'b1;
      intraBSSRcved = 1'b0;
   end
   else if (interBSSFromHeader & intraBSSFromHeader)
   // Intra-BSS & Inter-BSS From MAC Address -> Intra-BSS
   begin
      interBSSRcved = 1'b0;
      intraBSSRcved = 1'b1;
   end
   else if (intraBSSFromBssColor & interBSSFromHeader)
   // Intra-BSS from Received BSS Color & Inter-BSS From MAC Address -> Inter-BSS
   begin
      interBSSRcved = 1'b1;
      intraBSSRcved = 1'b0;
   end
   else
   // Not specified -> Not Classified                                            //ToDo:11ax
   begin
      interBSSRcved = 1'b0;
      intraBSSRcved = 1'b0;
   end
end


///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// rxControlCs FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (rxControlCs)
                  IDLE  :  rxControlCs_str = {"IDLE"};
           RX_FRM_CTRL  :  rxControlCs_str = {"RX_FRM_CTRL"};
             RX_DUR_ID  :  rxControlCs_str = {"RX_DUR_ID"};
              RX_ADDR1  :  rxControlCs_str = {"RX_ADDR1"};
              RX_ADDR2  :  rxControlCs_str = {"RX_ADDR2"};
              RX_ADDR3  :  rxControlCs_str = {"RX_ADDR3"};
              RX_ADDR4  :  rxControlCs_str = {"RX_ADDR4"};
           RX_SEQ_CTRL  :  rxControlCs_str = {"RX_SEQ_CTRL"};
           RX_QOS_CTRL  :  rxControlCs_str = {"RX_QOS_CTRL"};
       RX_CAR_FRM_CTRL  :  rxControlCs_str = {"RX_CAR_FRM_CTRL"};
            RX_HT_CTRL  :  rxControlCs_str = {"RX_HT_CTRL"};
      RX_IV_WAPIKEYIDX  :  rxControlCs_str = {"RX_IV_WAPIKEYIDX"};
       RX_EXTIV_WAPIPN  :  rxControlCs_str = {"RX_EXTIV_WAPIPN"};
           RX_FRM_BODY  :  rxControlCs_str = {"RX_FRM_BODY"};
                RX_ICV  :  rxControlCs_str = {"RX_ICV"};
      RX_CCMP_WAPI_MIC  :  rxControlCs_str = {"RX_CCMP_WAPI_MIC"};
                RX_FCS  :  rxControlCs_str = {"RX_FCS"};
            RX_BA_CTRL  :  rxControlCs_str = {"RX_BA_CTRL"};
        RX_BA_SEQ_CTRL  :  rxControlCs_str = {"RX_BA_SEQ_CTRL"};
          RX_BA_BITMAP  :  rxControlCs_str = {"RX_BA_BITMAP"};
          RX_BA_INFO    :  rxControlCs_str = {"RX_BA_INFO"};
     WAIT_SHIFTMIC_ICV  :  rxControlCs_str = {"WAIT_SHIFTMIC_ICV"};
          RX_TIMESTAMP  :  rxControlCs_str = {"RX_TIMESTAMP"};
         RX_BCN_INTRVL  :  rxControlCs_str = {"RX_BCN_INTRVL"};
         RX_CAPABILITY  :  rxControlCs_str = {"RX_CAPABILITY"};
         RX_ELEMENT_ID  :  rxControlCs_str = {"RX_ELEMENT_ID"};
        RX_ELEMENT_LEN  :  rxControlCs_str = {"RX_ELEMENT_LEN"};
       RX_ELEMENT_BODY  :  rxControlCs_str = {"RX_ELEMENT_BODY"};
             CHECK_FCS  :  rxControlCs_str = {"CHECK_FCS"};
            CHECK_ENCR  :  rxControlCs_str = {"CHECK_ENCR"};
          WRITE_STATUS  :  rxControlCs_str = {"WRITE_STATUS"};
              RX_ERROR  :  rxControlCs_str = {"RX_ERROR"};
         DISCARD_FRAME  :  rxControlCs_str = {"DISCARD_FRAME"};
     WAIT_KEYSEARCHDONE :  rxControlCs_str = {"WAIT_KEYSEARCHDONE"};
`ifdef RW_BFMEE_EN
      RX_BFRPOLL_SEGRET :  rxControlCs_str = {"RX_BFRPOLL_SEGRET"};
   RX_NDPA_SNDDIALTOKEN :  rxControlCs_str = {"RX_NDPA_SNDDIALTOKEN"};
       RX_NDPA_STAINFO1 :  rxControlCs_str = {"RX_NDPA_STAINFO1"};
       RX_NDPA_STAINFON :  rxControlCs_str = {"RX_NDPA_STAINFON"};
`endif //RW_BFMEE_EN
    RX_TRIGGER_CMN_INFO :  rxControlCs_str = {"RX_TRIGGER_CMN_INFO"};
RX_TRIGGER_DEP_CMN_INFO :  rxControlCs_str = {"RX_TRIGGER_DEP_CMN_INFO"};
    RX_TRIGGER_USR_INFO :  rxControlCs_str = {"RX_TRIGGER_USR_INFO"};
RX_TRIGGER_DEP_USR_INFO :  rxControlCs_str = {"RX_TRIGGER_DEP_USR_INFO"};
     RX_TRIGGER_PADDING :  rxControlCs_str = {"RX_TRIGGER_PADDING"};
               default  :  rxControlCs_str = {"XXX"};
   endcase
end

// rxControlNs FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (rxControlNs)
                  IDLE  :  rxControlNs_str = {"IDLE"};
           RX_FRM_CTRL  :  rxControlNs_str = {"RX_FRM_CTRL"};
             RX_DUR_ID  :  rxControlNs_str = {"RX_DUR_ID"};
              RX_ADDR1  :  rxControlNs_str = {"RX_ADDR1"};
              RX_ADDR2  :  rxControlNs_str = {"RX_ADDR2"};
              RX_ADDR3  :  rxControlNs_str = {"RX_ADDR3"};
              RX_ADDR4  :  rxControlNs_str = {"RX_ADDR4"};
           RX_SEQ_CTRL  :  rxControlNs_str = {"RX_SEQ_CTRL"};
           RX_QOS_CTRL  :  rxControlNs_str = {"RX_QOS_CTRL"};
       RX_CAR_FRM_CTRL  :  rxControlNs_str = {"RX_CAR_FRM_CTRL"};
            RX_HT_CTRL  :  rxControlNs_str = {"RX_HT_CTRL"};
      RX_IV_WAPIKEYIDX  :  rxControlNs_str = {"RX_IV_WAPIKEYIDX"};
       RX_EXTIV_WAPIPN  :  rxControlNs_str = {"RX_EXTIV_WAPIPN"};
           RX_FRM_BODY  :  rxControlNs_str = {"RX_FRM_BODY"};
                RX_ICV  :  rxControlNs_str = {"RX_ICV"};
      RX_CCMP_WAPI_MIC  :  rxControlNs_str = {"RX_CCMP_WAPI_MIC"};
                RX_FCS  :  rxControlNs_str = {"RX_FCS"};
            RX_BA_CTRL  :  rxControlNs_str = {"RX_BA_CTRL"};
        RX_BA_SEQ_CTRL  :  rxControlNs_str = {"RX_BA_SEQ_CTRL"};
          RX_BA_BITMAP  :  rxControlNs_str = {"RX_BA_BITMAP"};
            RX_BA_INFO  :  rxControlNs_str = {"RX_BA_INFO"};
     WAIT_SHIFTMIC_ICV  :  rxControlNs_str = {"WAIT_SHIFTMIC_ICV"};
          RX_TIMESTAMP  :  rxControlNs_str = {"RX_TIMESTAMP"};
         RX_BCN_INTRVL  :  rxControlNs_str = {"RX_BCN_INTRVL"};
         RX_CAPABILITY  :  rxControlNs_str = {"RX_CAPABILITY"};
         RX_ELEMENT_ID  :  rxControlNs_str = {"RX_ELEMENT_ID"};
        RX_ELEMENT_LEN  :  rxControlNs_str = {"RX_ELEMENT_LEN"};
       RX_ELEMENT_BODY  :  rxControlNs_str = {"RX_ELEMENT_BODY"};
             CHECK_FCS  :  rxControlNs_str = {"CHECK_FCS"};
            CHECK_ENCR  :  rxControlNs_str = {"CHECK_ENCR"};
          WRITE_STATUS  :  rxControlNs_str = {"WRITE_STATUS"};
              RX_ERROR  :  rxControlNs_str = {"RX_ERROR"};
         DISCARD_FRAME  :  rxControlNs_str = {"DISCARD_FRAME"};
     WAIT_KEYSEARCHDONE :  rxControlNs_str = {"WAIT_KEYSEARCHDONE"};
`ifdef RW_BFMEE_EN
      RX_BFRPOLL_SEGRET :  rxControlNs_str = {"RX_BFRPOLL_SEGRET"};
   RX_NDPA_SNDDIALTOKEN :  rxControlNs_str = {"RX_NDPA_SNDDIALTOKEN"};
       RX_NDPA_STAINFO1 :  rxControlNs_str = {"RX_NDPA_STAINFO1"};
       RX_NDPA_STAINFON :  rxControlNs_str = {"RX_NDPA_STAINFON"};
`endif //RW_BFMEE_EN
    RX_TRIGGER_CMN_INFO :  rxControlNs_str = {"RX_TRIGGER_CMN_INFO"};
RX_TRIGGER_DEP_CMN_INFO :  rxControlNs_str = {"RX_TRIGGER_DEP_CMN_INFO"};
    RX_TRIGGER_USR_INFO :  rxControlNs_str = {"RX_TRIGGER_USR_INFO"};
RX_TRIGGER_DEP_USR_INFO :  rxControlNs_str = {"RX_TRIGGER_DEP_USR_INFO"};
     RX_TRIGGER_PADDING :  rxControlNs_str = {"RX_TRIGGER_PADDING"};
                default :  rxControlNs_str = {"XXX"};
   endcase
end


// Type of encryption displayed in a string to easy simulation and debug
always @*
begin
  casex ({cLenKSR,cTypeKSR})
    {2'bxx,ENCRYPTION_TYPE_NONE} : encryptionType_str = {"NONE"};
    {2'b00,ENCRYPTION_TYPE_WEP}  : encryptionType_str = {"WEP-64"};
    {2'b01,ENCRYPTION_TYPE_WEP}  : encryptionType_str = {"WEP-128"};
    {2'bxx,ENCRYPTION_TYPE_TKIP} : encryptionType_str = {"TKIP"};
    {2'b00,ENCRYPTION_TYPE_CCMP} : encryptionType_str = {"CCMP-128"};
    {2'b10,ENCRYPTION_TYPE_CCMP} : encryptionType_str = {"CCMP-256"};
`ifdef RW_WAPI_EN
    {2'bxx,ENCRYPTION_TYPE_WAPI} : encryptionType_str = {"WAPI"};
`endif // RW_WAPI_EN
    {2'b00,ENCRYPTION_TYPE_GCMP} : encryptionType_str = {"GCMP-128"};
    {2'b10,ENCRYPTION_TYPE_GCMP} : encryptionType_str = {"GCMP-256"};
    default                      : encryptionType_str = {"XXXX"};
  endcase
end


always @*
begin
   if (rxControlCs == WRITE_STATUS)
   begin
      case (decryptionType)
      4'b0000: decryptionType_str = {"Frame was not encrypted"};
      4'b0001: decryptionType_str = {"WEP"};
      4'b0010: decryptionType_str = {"TKIP"};
      4'b0011: decryptionType_str = {"CCMP-128"};
      4'b0100: decryptionType_str = {"CCMP-256"};
      4'b0101: decryptionType_str = {"GCMP-128"};
      4'b0110: decryptionType_str = {"GCMP-256"};
      4'b0111: decryptionType_str = {"WAPI "};
      4'b1111: decryptionType_str = {"Null Key Found"};
      default: decryptionType_str = {"XXX"};
      endcase
   end
   else  
      decryptionType_str = {"-"};
end

`endif // RW_SIMU_ON

endmodule
