//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : RX FIFO Interface of rxController module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewer.
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
 
module rxFIFOIfController( 
            ///////////////////////////////////////////////
            //$port_g Clock and reset
            ///////////////////////////////////////////////
            input wire         macCoreRxClk,            // MAC Core Receive Clock
            input wire         macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain 
                                                        // active low
            input wire         macCoreClkSoftRst_n,     // Soft Reset of the MAC Core Clock domain
                                                        // active low
            ///////////////////////////////////////////////
            //$port_g Timer unit
            ///////////////////////////////////////////////
            input wire [63:0]  tsf,                     // TSF

            ///////////////////////////////////////////////
            //$port_g Encryption Engine
            ///////////////////////////////////////////////
            input wire [7:0]   plainDataOut,            // Decrypted data for writing into RX FIFO
            input wire         plainDataOutValid,       // Decrypted data valid
            input wire         plainDataOutEnd_p,       // Decrypted data valid
            
            ///////////////////////////////////////////////
            //$port_g Key Search Engine
            ///////////////////////////////////////////////
            input wire         keyStorageValid_p,       // Key Storage Index Valid
            input wire [9:0]   rxKeyIndexReturn,        // Key Storage RAM Index

            ///////////////////////////////////////////////
            //$port_g Decryption FSM
            ///////////////////////////////////////////////
            input wire         rxDataSelect,            // Select data from Encryption Engine
            
            ///////////////////////////////////////////////
            //$port_g MAC-PHY interface
            ///////////////////////////////////////////////
            input wire         macPhyIfRxEndForTiming_p,// Rx end for timing from MAC-PHY if. resynchronized
            input wire         macPhyIfRxErr_p,         // Rx error from MAC-PHY if. resynchronized

            ///////////////////////////////////////////////
            //$port_g A-MPDU Deaggregator
            ///////////////////////////////////////////////
            input wire         correctDelimiter,        // Indicate the correct Delimiter is found
            input wire         rxNDP,                   // Null Data Packet
            input wire [11:0]  rxLegLength,             // Legacy length
            //
            input wire [ 3:0]  rxFormatMod,             // Format and modulation
            input wire         rxAggregation,           // MPDU aggregate
            input wire         rxVector1Valid_p,        // Rx vector 1 is available
            input wire         rxVector2Valid_p,        // Rx vector 2 is available
            
            input wire [ 7:0]  rxData,                  // Rx data read from MAC-PHY interface FIFO
            input wire         rxDataValid,             // Rx data is valid
            input wire         rxDataStart_p,           // Start of data flow
            input wire         rxDataEnd_p,             // End of data flow
            input wire         rxDataError_p,           // Rx data error (incorrect length)
            input wire         rxEndOfFrame_p,          // End of frame information
            input wire  [1:0]  rxAMPDUCnt,              // Counter incremented at each A-MPDU reception
            input wire  [5:0]  rxMPDUCnt,               // Count the number of MPDU detected
            input wire [127:0] rxVector1,               // received Vector 1
            input wire  [63:0] rxVector2,               // received Vector 2

            ///////////////////////////////////////////////
            //$port_g RX Controller FSM
            ///////////////////////////////////////////////
            input wire         rxControlIdle,           // FSM in idle
            input wire         rxControlBackToIdle_p,   // FSM changed to idle state
            input wire [15:0]  rxMpduLength,            // Rx MPDU full length

            input wire         frmSuccessfulRcved,      // Frame successful for RX DMA
            input wire         fcsErrorRcved,           // FCS Error
            input wire         phyErrorRcved,           // PHY Error
            input wire         undefErrorRcved,         // Undefined Error
            input wire         rxFIFOOverFlowError,     // RX FIFO overflow
            input wire         decryptionError,         // Decryption Error
            input wire [3:0]   decryptionType,          // Decryption Type
            input wire         frmBody,                 // Frame body completion
            input wire         rxPayloadEnd_p,          // Last Byte of Frame Body
            
            input wire         startMacHdr,             // Start of the MAC Header
            input wire         macHdr,                  // MAC Header processing
            input wire         endMacHdr_p,             // End of the MAC Header pulse
            input wire         discardFrm,              // End of frame with discard command
            input wire         acceptError_p,           // Pulse to indicate that writeTrailer to accept Error frame (others than FCS error)
            input wire         writeTrailer,            // Write trailer info
            input wire         rxTriggerSWValid,        // Valid trigger frame with SW response flag

            output wire        rxFIFODone_p,            // RX FIFO status done
            output reg         rxEndOfFrame,            // RX end of frame holded
            output reg         txInProgressDly,         // Capture txInProgress on rxVector 1 valid pulse
            
            ///////////////////////////////////////////////
            //$port_g Frame decoder
            ///////////////////////////////////////////////
            input  wire        protocolError,           // Protocol error
            input  wire [3:0]  subTypeFrame,            // Subtype
            input  wire [1:0]  typeFrame,               // Type
            input  wire        mgmtFrame,               // Management frame
            input  wire        ctrlFrame,               // Control frame
            input  wire        dataFrame,               // Data frame
            input  wire        extnFrame,               // Extension frame
            input  wire        addr1Match,              // ADDR1 match
            input  wire        bcMcRcved,               // Broadcast/Multicast
            input  wire        ackRespRcved,            // Ack Response received
            input  wire        baRcved,                 // BA
            input  wire        triggerRcved,            // Trigger Frame
            input  wire        otherCtrlRcved,          // Other control frame
            input  wire [15:0] rxFrmCtrl,               // Received frame control field
            output reg         rxBADly,                 // Capture of rxBA

            ///////////////////////////////////////////////
            //$port_g Decrypt FSM
            ///////////////////////////////////////////////
            input wire         keyIndexCapture_p,       // Key Index returned by keySearch capture pulse

            ///////////////////////////////////////////////
            //$port_g RX FIFO
            ///////////////////////////////////////////////
            input wire         rxFIFOAlmostFull,        // RX FIFO is almost full
            
            output reg [3:0]   rxFIFOWrTag,             // RX Tag FIFO write data
            output reg [31:0]  rxFIFOWrData,            // RX FIFO write data
            output reg         rxFIFOWrite,             // RX FIFO write enable

            ///////////////////////////////////////////////
            //$port_g MacController
            ///////////////////////////////////////////////
            input wire         txInProgress,            // Tx is ongoing
            input wire         rxBA,                    // Indicate current tx frame is a single VHT Mpdu

            ///////////////////////////////////////////////
            //$port_g Backoff
            ///////////////////////////////////////////////
            input wire [2:0]   activeAC,                // Indicates which AC is currently selected.

            ///////////////////////////////////////////////
            //$port_g Control and Status Register
            ///////////////////////////////////////////////
            input wire         acceptErrorFrames,       // Accept error frames
            input wire         dataMHStoredwithPld,     // Data Frame Mac Header Stored With Payload
            input wire         mgtMHStoredwithPld,      // Management Frame Mac Header Stored With
                                                        // Payload
            input wire         ctrlMHStoredwithPld,     // Control Frame Mac Header Stored With
                                                        // Payload
            input wire   [1:0] dataFrmWrapMode,         // Data Frame Wrap Mode
            input wire   [1:0] mgtFrmWrapMode,          // Management Frame Wrap Mode
            input wire   [1:0] ctrlFrmWrapMode,         // Control Frame Wrap Mode
            input wire         disableRxBuffer2,        // Disable Rx Buffer 2

            ///////////////////////////////////////////////
            //$port_g Interrupt Controller
            ///////////////////////////////////////////////
            output reg         rxFIFOOverFlow,          // RX FIFO overflow

            output reg  [15:0] debugPortrxFIFOCntrl     // Port for debug purpose
            );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// MPDU Status information concatenation
wire [31:0] mpduStatusInfo;
wire        descriptorDoneRHMask;
reg         keySRamIndexV;
reg  [9:0]  rxKeyIndexReturnCapt;

// RX FIFO overflow condition
wire        rxFIFOOverFlowCond;

// RX Vector2 valid holded
reg         rxVector2Valid;

// correctDelimiter holded
reg         correctDelimiterDly;

// macPhyIfRxErrHold
reg         macPhyIfRxErrHold;

// acceptErrorHold
reg         acceptErrorHold;

// rxDataErrorHold
reg         rxDataErrorHold;

// TSF holded
reg [63:0]  tsfHold;


// 4 bytes counter
reg [3:0]   quadletCnt;
reg         discardFrmHold_p;
reg         writeTrailer_ff1;
reg         stopWrite;

// MAC header byte counter
reg  [6:0]  macHeaderCnt;

// Indicates that MPDU Length & MPDU priority should be write into RX Fifo
wire        lenMPDU_p;

// Indicates that MAC header Length should be write into RX Fifo
reg         lenMacHdr_p;

// Indicates that the 2 first bytes of MAC header should be restored into
// rxFIFOWrData 
reg         restoreMacHdr_p;

// Mux data between Deaggregator or Encryption Engine
wire [7:0]  rxDataMux;
wire        rxDataValidMux;
wire        rxDataEndMux_p;

// wire to caluclate rxFifo length in Quadlet for DMA
wire [15:0] rxMPDUFifoLength;

// Receive MPDU Priority
wire        rxMPDUPriority;

// Receive MPDU Wrap Mode
wire  [1:0] rxFrmWrapMode;

// MAC Header Stored with payload
wire        mhStoredwithPld;

reg         rxNDPInt; // Capture rxNDP

reg [15:0]  rxMPDULengthCapt;   // length captured for each MPDU inside an ampdu

`ifdef RW_SIMU_ON
// String definition to display rxFIFOWrTag state
reg [33*8-1:0] rxFIFOWrTag_str;
`endif // RW_SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    debugPortrxFIFOCntrl <= 16'd0;
  else
  begin
    if(writeTrailer_ff1)
    begin
      debugPortrxFIFOCntrl <= {writeTrailer_ff1,
                               keySRamIndexV,
                               rxKeyIndexReturn,
                               rxFIFOWrTag};
    end
    else
    begin
      debugPortrxFIFOCntrl <= 16'h7FFF;
    end
  end
end

// TSF holded
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    tsfHold <= 64'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    tsfHold <= 64'b0;
  else if (macPhyIfRxEndForTiming_p)
    tsfHold <= tsf;
  else if (rxControlBackToIdle_p)
    tsfHold <= 64'b0;
end

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxMPDULengthCapt <= 16'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxMPDULengthCapt <= 16'b0;
  else if (lenMPDU_p)
  begin
    if(rxFormatMod >= 4'd2)
      // Captured rxHTLength
      rxMPDULengthCapt <= rxMpduLength;
    else
      // Captured  rxLegLength
      rxMPDULengthCapt <= {4'b0000,rxLegLength[11:0]};
  end
end


// rxVector2Valid holded
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxVector2Valid <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxVector2Valid <= 1'b0;
  else if (rxVector2Valid_p)
    rxVector2Valid <=  1'b1;
end

// macPhyIfRxErr_p holded when not rxVector2Valid : avoid stucking in WRITE_STATUS state.
// if rxVector2Valid  then ignore macPhyIfRxErr_p and finish the writeTrailer
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    macPhyIfRxErrHold <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    macPhyIfRxErrHold <= 1'b0;
  else if (macPhyIfRxErr_p && !rxVector2Valid)
    macPhyIfRxErrHold <=  1'b1;
end

// acceptError_p holded when not rxVector2Valid : avoid stucking in WRITE_STATUS state.
// if rxVector2Valid  then ignore acceptError_p and finish the writeTrailer
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    acceptErrorHold <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    acceptErrorHold <= 1'b0;
  else if (acceptError_p && !rxVector2Valid)
    acceptErrorHold <=  1'b1;
end

// rxDataErro_p holded when not rxVector2Valid : avoid stucking in WRITE_STATUS state.
// if rxVector2Valid  then ignore rxDataErro_p and finish the writeTrailer
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxDataErrorHold <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxDataErrorHold <= 1'b0;
  else if (rxDataError_p && !rxVector2Valid)
    rxDataErrorHold <=  1'b1;
end

// correctDelimiterDly holded
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    correctDelimiterDly <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxFIFODone_p == 1'b1) || (rxControlIdle == 1'b1))
    correctDelimiterDly <= 1'b0;
  else if ((correctDelimiterDly == 1'b0) && (rxControlIdle == 1'b0)) 
    correctDelimiterDly <=  correctDelimiter;
end


// rxEndOfFrame holded
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxEndOfFrame <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxEndOfFrame <= 1'b0;
  else if (rxEndOfFrame_p)
    rxEndOfFrame <=  1'b1;
end


always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxNDPInt <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    rxNDPInt <= 1'b0;
  else if (rxDataStart_p)
    rxNDPInt <= rxNDP;
end

// writeTrailer delayed
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    writeTrailer_ff1 <= 1'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxFIFODone_p == 1'b1) || (rxControlIdle == 1'b1))
  begin
    writeTrailer_ff1 <= 1'b0;
  end
  else
  begin
    writeTrailer_ff1 <= (writeTrailer && (correctDelimiterDly || rxEndOfFrame || macPhyIfRxErrHold || acceptErrorHold || rxDataErrorHold));
  end
end


// MAC Header Byte Counter
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    macHeaderCnt <= 7'h0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    macHeaderCnt <= 7'h0;
  else if (macHdr && rxDataValidMux)
    macHeaderCnt <= macHeaderCnt + 7'h1;
end


// MPDU Length & MPDU Priority
//   Wait 2nd byte of MAC header for the MPDU priority selection, except for NDP
assign lenMPDU_p = macHdr & macHeaderCnt==7'd1 & rxDataValid | 
                   rxNDP  & rxDataStart_p;


// Restore MAC Header after MPDU Length
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    restoreMacHdr_p <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    restoreMacHdr_p <= 1'b0;
  else if (macHdr & macHeaderCnt==7'd1 & rxDataValid)
    restoreMacHdr_p <= 1'b1;
  else
    restoreMacHdr_p <= 1'b0;
end


// MAC Header Length Tag
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    lenMacHdr_p <= 1'b0;
  else if ((macCoreClkSoftRst_n == 1'b0) || (rxControlIdle == 1'b1))
    lenMacHdr_p <= 1'b0;
  else
    lenMacHdr_p <= endMacHdr_p;
end


// rxFIFOWrTag control
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFIFOWrTag <= 4'b1100;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxFIFOWrTag <= 4'b1100;
  else if (lenMPDU_p)
    // MPDU Lenght
    rxFIFOWrTag <= 4'b1010;
  else if (discardFrm || discardFrmHold_p || rxFIFOOverFlowCond)
    // End of frame with discard pattern
    rxFIFOWrTag <= 4'b1111;
  else if (startMacHdr && !mhStoredwithPld && !rxFIFOWrite && (macHeaderCnt[6:2] == 4'h0))
    // Start of Header
    rxFIFOWrTag <= 4'b0001;
  else if (endMacHdr_p && !mhStoredwithPld)
    // End of Header
    rxFIFOWrTag <= 4'b0010;
  else if (lenMacHdr_p && !mhStoredwithPld)
    // Length of Header
    rxFIFOWrTag <= 4'b0011;
  else if (rxDataEndMux_p || acceptError_p)
    // End of payload
    rxFIFOWrTag <= 4'b0101;
  else if (acceptError_p)
    // Force End of payload TAG in case of accepted error frame
    rxFIFOWrTag <= 4'b0101;
  else if (writeTrailer_ff1 && (quadletCnt == 4'h8) && rxTriggerSWValid)
    // Indicate a Trigger Frame
    rxFIFOWrTag <= 4'b1110;
  else if ((writeTrailer_ff1 && (quadletCnt == 4'h9)) || rxFIFODone_p)
    // End of frame with save pattern
    rxFIFOWrTag <= 4'b0000;
  else
    // Intermediate Header or payload
    rxFIFOWrTag <= 4'b1100;
end


// Mux data between Deaggregator or Encryption Engine
assign rxDataMux      = (rxDataSelect) ? plainDataOut : 
                        rxData;
assign rxDataValidMux = (rxDataSelect) ? plainDataOutValid : 
                        rxDataValid;
assign rxDataEndMux_p = rxDataEnd_p;

// Convert MPDU length into FIFO Length
assign rxMPDUFifoLength = (rxFormatMod >= 4'd2) ? rxMpduLength : {4'b0000,rxLegLength[11:0]};

// Receive MDPU Priority
assign rxMPDUPriority = (baRcved | triggerRcved) & ~disableRxBuffer2;

// Receive Frame Wrap Mode
assign rxFrmWrapMode = {2{dataFrame}} & dataFrmWrapMode |
                       {2{mgmtFrame}} & mgtFrmWrapMode  |
                       {2{ctrlFrame}} & ctrlFrmWrapMode ;

// Mac Header Stored with payload
assign mhStoredwithPld = dataFrame & dataMHStoredwithPld |
                         mgmtFrame & mgtMHStoredwithPld  |
                         ctrlFrame & ctrlMHStoredwithPld |
                         extnFrame                       |
                         otherCtrlRcved                  |
                         protocolError;

// rxFIFOWrData control
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    rxFIFOWrData <= 32'h0;
    rxFIFOWrite  <= 1'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0)
  begin
    rxFIFOWrData <= 32'h0;
    rxFIFOWrite  <= 1'b0;
  end
  else
  begin
    // FSM in idle
    if (rxControlIdle && !rxDataStart_p  && !discardFrmHold_p)
    begin
      rxFIFOWrData <= 32'h0;
      rxFIFOWrite  <= 1'b0;
    end  
    // Captured rxMPDUFifoLength
    else if(lenMPDU_p)
    begin
      rxFIFOWrData <= {12'h0,rxFrmWrapMode,mhStoredwithPld,rxMPDUPriority,
                       rxMPDUFifoLength[15:0]};
      rxFIFOWrite  <= 1'b1;
    end  
    // MAC Header Length
    else if(lenMacHdr_p && !mhStoredwithPld)
    begin
      rxFIFOWrData <= {25'h0,macHeaderCnt[6:0]};
      rxFIFOWrite  <= 1'b1;
    end  
    // RX FIFO Overflow condition
    else if (rxFIFOOverFlowCond)
    begin
      rxFIFOWrData <= mpduStatusInfo;
      rxFIFOWrite  <= 1'b1;
    end  
    // 4 bytes frame discard
    else if (discardFrmHold_p)
    begin
      rxFIFOWrData <= 32'h0;
      rxFIFOWrite  <= 1'b1;
    end  
    // Force End of payload TAG in case of accepted error frame
    else if (acceptError_p)
    begin
      rxFIFOWrData <= 32'h0;
      rxFIFOWrite  <= 1'b1;
    end  
    // Trailer information
    else if (writeTrailer_ff1)
    begin
      rxFIFOWrite  <= 1'b1;
      case (quadletCnt)
        4'h0:
        begin
          rxFIFOWrData[15:0] <= rxNDPInt ? 16'h0 : rxMPDULengthCapt;
          if (rxAggregation)
            rxFIFOWrData[31:16] <= {rxAMPDUCnt,rxMPDUCnt,8'b0};
          else
            rxFIFOWrData[31:16] <= 16'b0;
        end
        4'h1: rxFIFOWrData <= tsfHold[31:0];
        4'h2: rxFIFOWrData <= tsfHold[63:32];
        4'h3: rxFIFOWrData <= rxVector1[0*32 +: 32];
        4'h4: rxFIFOWrData <= rxVector1[1*32 +: 32];
        4'h5: rxFIFOWrData <= rxVector1[2*32 +: 32];
        4'h6: rxFIFOWrData <= rxVector1[3*32 +: 32];
        4'h7: rxFIFOWrData <= rxVector2[0*32 +: 32];
        4'h8: rxFIFOWrData <= rxVector2[1*32 +: 32];
        4'h9: rxFIFOWrData <= mpduStatusInfo;
        
        // Disable case default state for block coverage
        // pragma coverage block = off
        default :  
          begin
            rxFIFOWrite  <= 1'b0;
            rxFIFOWrData <= rxFIFOWrData;
          end
        // pragma coverage block = on
      endcase
    end
    // MAC Header 3rd byte (add the registred frame control)
    else if (restoreMacHdr_p &&  rxDataValidMux)
    begin
      rxFIFOWrData <= {8'b0, rxDataMux, rxFrmCtrl};
      rxFIFOWrite  <= 1'b0;
    end
    else if (restoreMacHdr_p && !rxDataValidMux)
    begin
      rxFIFOWrData <= {16'h0, rxFrmCtrl};
      rxFIFOWrite  <= 1'b0;
    end
    // Normal operation
    else if (rxDataValidMux && !writeTrailer_ff1 && !rxFIFOOverFlowCond)
    begin
      if (rxDataEndMux_p || (endMacHdr_p && !mhStoredwithPld))
        rxFIFOWrite  <= 1'b1;
      else   
        rxFIFOWrite  <= 1'b0;
      case (quadletCnt[1:0])
        2'h0    : rxFIFOWrData        <= {24'b0, rxDataMux};
        2'h1    : rxFIFOWrData[15:8]  <= rxDataMux;
        2'h2    : rxFIFOWrData[23:16] <= rxDataMux;
        default : 
          begin 
            rxFIFOWrData[31:24] <= rxDataMux;
            rxFIFOWrite         <= 1'b1;
          end  
      endcase
    end
    else
      rxFIFOWrite  <= 1'b0;
  end
end

// quadletCnt control
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    quadletCnt <= 4'h0;
  else if (macCoreClkSoftRst_n == 1'b0)
    quadletCnt <= 4'h0;
  // Clear counter
  else if (rxControlIdle                     ||
           (quadletCnt == 4'hA)              ||
           rxDataEndMux_p                    ||
           (endMacHdr_p && !mhStoredwithPld) ||
           acceptError_p)
    quadletCnt <= 4'h0;
  // Trailer phase -> increment until end of writing
  else if (writeTrailer_ff1 &&
    (rxAggregation || phyErrorRcved || rxFIFOOverFlowError || acceptErrorHold || macPhyIfRxErrHold || rxDataErrorHold ||
    (!rxAggregation && ((quadletCnt < 4'h6) || rxEndOfFrame || rxEndOfFrame_p))))
    quadletCnt <= quadletCnt + 4'h1;
  // Normal operation
  else if (rxDataValidMux && !writeTrailer_ff1)
  begin
    if (quadletCnt == 4'h3)
      quadletCnt <= 4'h0;
    else
      quadletCnt <= quadletCnt + 4'h1;
  end
end

// stopWrite is generated when quadletCnt=6 during write trailer step
// until rxEndOfFrame_p is received
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    stopWrite <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    stopWrite <= 1'b0;
  else if (!(phyErrorRcved || rxFIFOOverFlowError) &&
    (!rxAggregation && (quadletCnt == 4'h6) && rxFIFOWrite) &&
    !(rxEndOfFrame || rxEndOfFrame_p))
    stopWrite <= 1'b1;
  else if (rxControlIdle || rxEndOfFrame_p)
    stopWrite <= 1'b0;
end


// discardFrmHold generation
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    discardFrmHold_p <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    discardFrmHold_p <= 1'b0;
  else if (discardFrmHold_p && (rxFIFOWrTag == 4'hF))
    // Clear discardFrmHold_p when tag is written
    discardFrmHold_p <= 1'b0;
  else if (discardFrm && (rxFIFOWrTag != 4'hF))
    discardFrmHold_p <= 1'b1;
end

// rxFIFODone_p control
assign rxFIFODone_p   = (quadletCnt == 4'hA) ||
                        (discardFrmHold_p && rxFIFOWrite);

// MPDU Status information
assign mpduStatusInfo = {
   (rxNDPInt ? 4'h0 : subTypeFrame),
   (rxNDPInt ? 2'h0 : typeFrame   ),
   (keySRamIndexV && !rxNDPInt),
   rxKeyIndexReturnCapt,
   descriptorDoneRHMask,
   rxNDPInt || frmSuccessfulRcved,
   activeAC[1:0],
   bcMcRcved,
   !addr1Match,
   fcsErrorRcved,
   undefErrorRcved | phyErrorRcved | rxFIFOOverFlowError | rxFIFOOverFlowCond,
   decryptionError,
   decryptionType, 
   (frmSuccessfulRcved && txInProgressDly) || (ackRespRcved && ~txInProgressDly),
   rxVector2Valid || rxVector2Valid_p
};

assign descriptorDoneRHMask = (rxFIFOWrTag == 4'b1111) ? 1'b0 : 1'b1;

always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    keySRamIndexV        <= 1'b0;
    rxKeyIndexReturnCapt <= 10'b0;
  end
  else if ((macCoreClkSoftRst_n == 1'b0) || rxControlIdle)
  begin
    keySRamIndexV        <= 1'b0;
    rxKeyIndexReturnCapt <= 10'b0;
  end
  else if (keyIndexCapture_p)
  begin
    keySRamIndexV        <= keyStorageValid_p;
    rxKeyIndexReturnCapt <= rxKeyIndexReturn;
  end
end

// txInProgressDly and rxBADly generation
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    txInProgressDly    <= 1'b0;
    rxBADly            <= 1'b0;
  end
  else if (macCoreClkSoftRst_n == 1'b0)
  begin
    txInProgressDly    <= 1'b0;
    rxBADly            <= 1'b0;
  end
  else if (rxVector1Valid_p)
  begin
    txInProgressDly    <= txInProgress;
    rxBADly            <= rxBA;
  end
  else if(rxFIFODone_p)
  begin
    txInProgressDly    <= 1'b0;
    rxBADly            <= 1'b0;
  end
end

// rxFIFOOverFlow generation
always @ (posedge macCoreRxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    rxFIFOOverFlow <= 1'b0;
  else if (macCoreClkSoftRst_n == 1'b0)
    rxFIFOOverFlow <= 1'b0;
  else
    rxFIFOOverFlow <= rxFIFOOverFlowCond;
end

// rxFIFOOverFlow condition when FIFO will be full and we are not writing 
// last qualet of the trailer
assign rxFIFOOverFlowCond = rxFIFOWrite && rxFIFOAlmostFull && !quadletCnt[3];



///////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// String definition to display rxFIFOWrTag
always @(*)
begin
    case (rxFIFOWrTag)
       4'b1100  : rxFIFOWrTag_str = "Intermediate Header or payload";
       4'b0000  : rxFIFOWrTag_str = "End of frame with save pattern";
       4'b0101  : rxFIFOWrTag_str = "End of payload";
       4'b0001  : rxFIFOWrTag_str = "Start of Header";
       4'b0010  : rxFIFOWrTag_str = "End of Header";
       4'b0011  : rxFIFOWrTag_str = "Header Length";
       4'b1111  : rxFIFOWrTag_str = "End of frame with discard pattern";
       4'b1010  : rxFIFOWrTag_str = "MPDU Lenght";
       4'b1110  : rxFIFOWrTag_str = "Indicate a Trigger Frame";
       default  : rxFIFOWrTag_str = "XXXXXX";
    endcase
end

`endif // RW_SIMU_ON

endmodule
