//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Trigger Frame decoder of rxController module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                   the waveform viewer.
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module triggerDecoder( 
   ///////////////////////////////////////////////
   //$port_g Clock and reset
   ///////////////////////////////////////////////
   input  wire        macCoreRxClk,                   // MAC Core Receive Clock
   input  wire        macCoreClkHardRst_n,            // Hard Reset of the MAC Core Clock domain 
                                                      // active low
   input  wire        macCoreClkSoftRst_n,            // Soft Reset of the MAC Core Clock domain
                                                      // active low
   ///////////////////////////////////////////////
   //$port_g RX Controller FSM
   ///////////////////////////////////////////////
   input  wire  [7:0] stateByteCnt,                   // Byte counter by states
   input  wire        rxTriggerCMNInfoEn,             // Trigger Common Info field
   input  wire        rxTriggerUSRInfoEn,             // Trigger User Info field
   input  wire        rxTriggerDepUSRInfoEn,          // Trigger Dependent User Info field
   input  wire        rxTriggerUSRInfoEnd_p,          // FSM Trigger User Info end pulse
   input  wire        rxTriggerFound,                 // Indicate that a valid trigger has been
                                                      // found
   input  wire        fcsOkRcved_p,                   // Frame received with correct FCS 
                                                             
   output reg   [7:0] trigDepUsrInfoLen,              // Length of the Trigger Dependent User Info
                                                      // Field
   output wire  [3:0] rxTriggerType,                  // Trigger Type
   output wire [11:0] rxTriggerULLength,              // Trigger frame, UL Length
   output wire        rxTriggerCS,                    // Trigger frame, CS
   output wire  [1:0] rxTriggerULBW,                  // Trigger frame, UL BW
   output wire  [1:0] rxTriggerGIAndLTFType,          // Trigger frame, GI And LTF Type
   output wire        rxTriggerHELTFMode,             // Trigger frame, HE-LTF Mode
   output wire  [2:0] rxTriggerNumHELTFAndMidPeriod,  // Trigger frame, Number of HE-LTF symbols
                                                      // And Midamble Periodicity
   output wire        rxTriggerULSTBC,                // Trigger frame, UL STBC
   output wire        rxTriggerLDPCExtraSymbol,       // Trigger frame, LDPC Extra Symbol Segment
   output wire  [5:0] rxTriggerAPTxPower,             // Trigger frame, AP TX Power
   output wire  [2:0] rxTriggerULPacketExtension,     // Trigger frame, UL Packet Extension
   output wire [15:0] rxTriggerULSpatialReuse,        // Trigger frame, UL Spatial Reuse
   output wire        rxTriggerDoppler,               // Trigger frame, Doppler
   output wire  [8:0] rxTriggerULHESigA2Reserved,     // Trigger frame, UL HE-SIG-A2 Reserved
   output wire        rxTriggerUSRInfoMatch_p,        // Trigger frame, Valid user info field
   output reg         rxTriggerUSRInfoMatch,          // Trigger frame, Valid user info field
   output reg         rxTriggerAIDMatch,              // Trigger frame, AID12 Match
   output reg         rxTriggerAID4095,               // Trigger frame, AID12 is 4095
   output reg   [7:0] rxTriggerRUAllocation,          // Trigger frame, RU Allocation
   output wire        rxTriggerULFECCoding,           // Trigger frame, UL Coding Type
   output wire  [3:0] rxTriggerULMCS,                 // Trigger frame, UL MCS
   output wire        rxTriggerULDCM,                 // Trigger frame, UL DCM
   output wire  [2:0] rxTriggerStartingSSNum,         // Trigger frame, Starting Spatial Stream
   output wire  [2:0] rxTriggerNumSS,                 // Trigger frame, Number Of Spatial Streams
   output wire  [6:0] rxTriggerULTargetRSSSI,         // Trigger frame, UL Target RSSI
   output reg  [79:0] rxTriggerDepUserInfo,           // Trigger frame, Trigger Dependent User Info
                                                      // Field
   output wire        rxTriggerULRUType,              // Trigger Frame, RU Type

   ///////////////////////////////////////////////
   //$port_g frameDecoder
   ///////////////////////////////////////////////
   input  wire        addr2MatchBssID,                // ADDR2 is equal to the BSS ID
   input  wire        triggerRcved,                   // Trigger Frame
            
   ///////////////////////////////////////////////
   //$port_g A-MPDU Deaggregator
   ///////////////////////////////////////////////
   input  wire  [7:0] rxData,                         // Rx data read from MAC-PHY interface FIFO
   input  wire        rxDataValid,                    // Rx data is valid

   ///////////////////////////////////////////////
   //$port_g Control and Status Register
   ///////////////////////////////////////////////
   input  wire [15:0] aid,                            // Association ID
   input  wire  [2:0] primaryChPosition,              // Primary Channel Position
   input  wire  [1:0] maxSupportedBW,                 // max Supported BW
   input  wire        raRUType,                       // Random Access RU Type
   input  wire        raRUEnable,                     // Random Access RU Enable
   input  wire  [3:0] maxMCSInHETB,                   // Max MCS in HE TB
   input  wire  [2:0] eOCW,                           // Exposent of OFDMA Contention Window
   input  wire  [2:0] eOCWMin,                        // Exposent of OFDMA Contention Window Minimum
   output wire  [2:0] eOCWIn,                         // Exposent of OFDMA Contention Window In
   output wire        eOCWInValid,                    // Exposent of OFDMA Contention Window In
                                                      // Valid
   input  wire  [6:0] ocwLFSR,                        // OFDMA Contention Window LFSR
   output wire  [6:0] ocwLFSRIn,                      // OFDMA Contention Window LFSR In
   output wire        ocwLFSRInValid,                 // OFDMA Contention Window LFSR In Valid
   input  wire        dopplerSupport,                 // Doppler Support
   input  wire        dcmSupport,                     // Dual Carrier Modulation Support

   ///////////////////////////////////////////////
   //$port_g Debug Port
   ///////////////////////////////////////////////
   output wire [15:0] debugPortTriggerDecoder         // Debug port
);


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
reg  [95:0] rxTriggerCommonInfo;      // Trigger: Commno Info Field
reg [39:12] rxTriggerUserInfo;        // Trigger: User Info Field
reg  [15:0] rxTriggerBARControl;      // MU-BAR Trigger: BAR Control Field
                                   
reg         rxTriggerAID0;            // Trigger frame, AID12 is 0
reg         rxTriggerAID2045;         // Trigger frame, AID12 is 2045
                                   
reg   [2:0] heTBPrimaryCh;            // Primary Channel Position in the HE-TB Bandwidth
reg   [3:0] ulBWBitmap;               // Uplink Bandwidth bitmap
reg   [3:0] ourBWBitmap;              // Our Bandwidth Bitmap
wire  [3:0] heTBBWBitmap;             // HE-TB Frame Bandwith Bitmap
reg   [7:0] rxTriggerRUAllocationComb;// Combinatorial RU Allocation field
reg   [2:0] ruSizeComb;               // Indicates the size of allocated RU
reg   [6:0] firstRU;                  // First RU allocated
reg   [6:0] lastRU;                   // Last RU allocated
wire  [4:0] numberOfRARUComb;         // Combinatorial Numbers of Random Access RU field
reg   [6:0] eligibleRUNbr;            // Numbers of eligible RU
reg   [6:0] oboCnt;                   // OFDMA Random Access BackOff counter
reg   [6:0] oboCntNxt;                // OFDMA Random Access BackOff counter next 
wire        oboCntLoad;               // OFDMA Random Access BackOff counter load
wire        oboCntUpdate;             // OFDMA Random Access BackOff counter update
reg   [6:0] ocwMask;                  // OFDMA Contention window mask
reg         rxTriggerUSRInfoMatchDly; // Valid user info field delayed
wire        raruUserInfo;             // Indicates that the user info field allocates RA-RUs
wire        raruUserInfoMatch;        // Indicates that the user info field allocates RA-RUs
                                      // of expected type.


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Trigger Frame: Common Info Field Sampling
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
   begin
      rxTriggerCommonInfo <= 96'h0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      rxTriggerCommonInfo <= 96'h0;
   end
   else if (rxDataValid && rxTriggerCMNInfoEn)
   begin
      case (stateByteCnt[3:0])
      4'd0:  rxTriggerCommonInfo[0+:8]  <= rxData[7:0];
      4'd1:  rxTriggerCommonInfo[8+:8]  <= rxData[7:0];
      4'd2:  rxTriggerCommonInfo[16+:8] <= rxData[7:0];
      4'd3:  rxTriggerCommonInfo[24+:8] <= rxData[7:0];
      4'd4:  rxTriggerCommonInfo[32+:8] <= rxData[7:0];
      4'd5:  rxTriggerCommonInfo[40+:8] <= rxData[7:0];
      4'd6:  rxTriggerCommonInfo[48+:8] <= rxData[7:0];
      4'd7:  rxTriggerCommonInfo[56+:8] <= rxData[7:0];
      endcase
   end
end

assign rxTriggerType                 = rxTriggerCommonInfo[3:0];
assign rxTriggerULLength             = rxTriggerCommonInfo[15:4];
assign rxTriggerCS                   = rxTriggerCommonInfo[17];
assign rxTriggerULBW                 = rxTriggerCommonInfo[19:18];
assign rxTriggerGIAndLTFType         = rxTriggerCommonInfo[21:20];
assign rxTriggerHELTFMode            = rxTriggerCommonInfo[22];
assign rxTriggerNumHELTFAndMidPeriod = rxTriggerCommonInfo[25:23];
assign rxTriggerULSTBC               = rxTriggerCommonInfo[26];
assign rxTriggerLDPCExtraSymbol      = rxTriggerCommonInfo[27];
assign rxTriggerAPTxPower            = rxTriggerCommonInfo[33:28];
assign rxTriggerULPacketExtension    = rxTriggerCommonInfo[36:34];
assign rxTriggerULSpatialReuse       = rxTriggerCommonInfo[52:37];
assign rxTriggerDoppler              = rxTriggerCommonInfo[53];
assign rxTriggerULHESigA2Reserved    = rxTriggerCommonInfo[62:54];


// Trigger Frame: User Info Field Sampling
// Store User Info until valid user info is found.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
   begin
      rxTriggerAID0             <= 1'b0;
      rxTriggerAID2045          <= 1'b0;
      rxTriggerAID4095          <= 1'b0;
      rxTriggerAIDMatch         <= 1'b0;
      rxTriggerUserInfo         <= 28'b0;
   end
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
   begin
      rxTriggerAID0             <= 1'b0;
      rxTriggerAID2045          <= 1'b0;
      rxTriggerAID4095          <= 1'b0;
      rxTriggerAIDMatch         <= 1'b0;
      rxTriggerUserInfo         <= 28'b0;
   end
   else if (rxDataValid & rxTriggerUSRInfoEn & ~rxTriggerUSRInfoMatch)
   begin
      if (stateByteCnt[3:0]==4'd0)
      begin
         rxTriggerAID0       <= rxData==8'h00;
         rxTriggerAID2045    <= rxData==8'hfd;
         rxTriggerAID4095    <= rxData==8'hff;
         rxTriggerAIDMatch   <= rxData==aid[7:0];
      end
      else if (stateByteCnt[3:0]==4'd1)
      begin
         rxTriggerAID0       <= rxTriggerAID0     && rxData[3:0]==4'h0;
         rxTriggerAID2045    <= rxTriggerAID2045  && rxData[3:0]==4'h7;
         rxTriggerAID4095    <= rxTriggerAID4095  && rxData[3:0]==4'hf;
         rxTriggerAIDMatch   <= rxTriggerAIDMatch && rxData[3:0]==aid[11:8];
      end

      case (stateByteCnt[3:0])
      4'd1: rxTriggerUserInfo[12+:4] <= rxData[7:4];
      4'd2: rxTriggerUserInfo[16+:8] <= rxData[7:0];
      4'd3: rxTriggerUserInfo[24+:8] <= rxData[7:0];
      4'd4: rxTriggerUserInfo[32+:8] <= rxData[7:0];
      endcase
   end
end

assign rxTriggerULFECCoding   = rxTriggerUserInfo[20];
assign rxTriggerULMCS         = rxTriggerUserInfo[24:21];
assign rxTriggerULDCM         = rxTriggerUserInfo[25];
assign rxTriggerStartingSSNum = raruUserInfo ? 3'd0 : rxTriggerUserInfo[28:26];
assign rxTriggerNumSS         = raruUserInfo ? 3'd0 : rxTriggerUserInfo[31:29];
assign rxTriggerULTargetRSSSI = rxTriggerUserInfo[38:32];


// Trigger Frame: RU Allocation subfield of the User Info Field Sampling
// Store RU Allocation until valid user info is found.
// Updated when RARU is found.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      rxTriggerRUAllocation <= 8'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      rxTriggerRUAllocation <= 8'b0;
   else if (rxDataValid & rxTriggerUSRInfoEn & ~rxTriggerUSRInfoMatch)
   begin
      case (stateByteCnt[3:0])
      // Store
      4'd1: rxTriggerRUAllocation[3:0] <= rxData[7:4];
      4'd2: rxTriggerRUAllocation[7:4] <= rxData[3:0];
      // Update
      4'd4: if (raruUserInfoMatch & oboCntNxt<=eligibleRUNbr)
            begin
               rxTriggerRUAllocation[7:1] <= firstRU + oboCntNxt - {6'd0, |oboCntNxt};
               rxTriggerRUAllocation[0]   <= 1'b0;
            end
      endcase
   end
end


// Trigger Frame: Trigger Dependent User Info Field Sampling
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      rxTriggerDepUserInfo <= 80'd0;
   else if ((macCoreClkSoftRst_n == 1'b0) || (rxTriggerCMNInfoEn == 1'b1)) // Synchronous Reset
      rxTriggerDepUserInfo <= 80'd0;
   else if (rxDataValid && rxTriggerDepUSRInfoEn)
   begin
      case (stateByteCnt[4:0])
      5'd0: rxTriggerDepUserInfo[0+:8]  <= rxData[7:0];
      5'd1: rxTriggerDepUserInfo[8+:8]  <= rxData[7:0];
      5'd2: rxTriggerDepUserInfo[16+:8] <= rxData[7:0];
      5'd3: rxTriggerDepUserInfo[24+:8] <= rxData[7:0];
      5'd4: rxTriggerDepUserInfo[32+:8] <= rxData[7:0];
      5'd5: rxTriggerDepUserInfo[40+:8] <= rxData[7:0];
      5'd6: rxTriggerDepUserInfo[48+:8] <= rxData[7:0];
      5'd7: rxTriggerDepUserInfo[56+:8] <= rxData[7:0];
      5'd8: rxTriggerDepUserInfo[64+:8] <= rxData[7:0];
      5'd9: rxTriggerDepUserInfo[72+:8] <= rxData[7:0];
      endcase
   end
end


// Trigger Frame: RU Type
assign rxTriggerULRUType = raruUserInfo;


// BAR Control is stored even if AID doesn't match, needed to find the length of
// the Trigger Dependent User Info Field of other Users.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      rxTriggerBARControl <= 16'd0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      rxTriggerBARControl <= 16'd0;
   else if (rxDataValid && rxTriggerDepUSRInfoEn && rxTriggerType==4'd2)
   begin
      case (stateByteCnt)
      8'd0: rxTriggerBARControl[0+:8] <= rxData[7:0];
      8'd1: rxTriggerBARControl[8+:8] <= rxData[7:0];
      endcase
   end
end


// Length of the Trigger Dependent User Info Field
//   Only present for Basic Trigger, Beamforming Report Poll Trigger & MU-BAR Trigger
always @*
begin
   if (rxTriggerType==4'd2 && rxTriggerBARControl[3:1]==3'b011)
      //MU-BAR: Multi-TID BlockAckReq
      trigDepUsrInfoLen = {{2'b00,rxTriggerBARControl[15:12]}+6'd1,2'b10}; // (TID_INFO+1)*4+2
   else if (rxTriggerType==4'd2)
      //MU-BAR: Compressed BlockAckReq
      trigDepUsrInfoLen = 8'd4;
   else if (rxTriggerType<=4'd1)
      // Basic Trigger & Beamforming Report Poll Trigger
      trigDepUsrInfoLen = 8'd1;
   else
      // Trigger Frame without trigger dependent user info field
      trigDepUsrInfoLen = 8'd0;
end


//------------------------------------------------------------------------------
// Primary Channel Position in the HE-TB Bandwidth
always @*
begin
   case (rxTriggerULBW)
   2'b00:   heTBPrimaryCh = primaryChPosition & 3'b000;
   2'b01:   heTBPrimaryCh = primaryChPosition & 3'b001;
   2'b10:   heTBPrimaryCh = primaryChPosition & 3'b011;
   default: heTBPrimaryCh = primaryChPosition & 3'b111;
   endcase
end


// Uplink bandwidth bitmap (in 20MHz)
always @*
begin
   case (rxTriggerULBW)
   2'b00:  ulBWBitmap = 4'b0001;
   2'b01:  ulBWBitmap = 4'b0011;
   default:ulBWBitmap = 4'b1111;
   endcase
end


// Our Bandwidth Bitmap (in 20MHz)
always @*
begin
   case ({maxSupportedBW,heTBPrimaryCh[1:0]})
   {2'b00,2'b00}: ourBWBitmap = 4'b0001;
   {2'b00,2'b01}: ourBWBitmap = 4'b0010;
   {2'b00,2'b10}: ourBWBitmap = 4'b0100;
   {2'b00,2'b11}: ourBWBitmap = 4'b1000;
   {2'b01,2'b00}: ourBWBitmap = 4'b0011;
   {2'b01,2'b01}: ourBWBitmap = 4'b0011;
   {2'b01,2'b10}: ourBWBitmap = 4'b1100;
   {2'b01,2'b11}: ourBWBitmap = 4'b1100;
   {2'b10,2'b00}: ourBWBitmap = 4'b1111;
   {2'b10,2'b01}: ourBWBitmap = 4'b1111;
   {2'b10,2'b10}: ourBWBitmap = 4'b1111;
   {2'b10,2'b11}: ourBWBitmap = 4'b1111;
   default      : ourBWBitmap = 4'b0000;//Not Supported
   endcase
end


// Trigger Based frame bandwidth bitmap (in 20MHz)
assign heTBBWBitmap = ulBWBitmap & ourBWBitmap;


// ruSize Comb
//    RU Size extracted from the combinatorial ruAllocation from user info field.
//    Should be used only when byte2 of the user info field is received.
always @*
begin
   rxTriggerRUAllocationComb = {rxData[3:0],rxTriggerUserInfo[15:12]};

   if      (rxTriggerRUAllocationComb[7:1]==68) ruSizeComb = 3'd6; // 2x996-tones
   else if (rxTriggerRUAllocationComb[7:1]==67) ruSizeComb = 3'd5; //   996-tones
   else if (rxTriggerRUAllocationComb[7:1]>=65) ruSizeComb = 3'd4; //   484-tones
   else if (rxTriggerRUAllocationComb[7:1]>=61) ruSizeComb = 3'd3; //   242-tones
   else if (rxTriggerRUAllocationComb[7:1]>=53) ruSizeComb = 3'd2; //   106-tones
   else if (rxTriggerRUAllocationComb[7:1]>=37) ruSizeComb = 3'd1; //    52-tones
   else                                         ruSizeComb = 3'd0; //    26-tones
end


// First eligible RU
//   - Initialized with first eligible RU for the HE-TB Bandwidth
//   - Updated with first allocated RU if greather than first eligible RU
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      firstRU <= 7'd0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      firstRU <= 7'd0;
   else if (rxDataValid & rxTriggerUSRInfoEn & stateByteCnt==8'd2)
   begin
      casex ({heTBBWBitmap,ruSizeComb})
      // 26 tones RU
      {4'bxxx1,3'd0}:   firstRU <= 7'd0;
      {4'bxx10,3'd0}:   firstRU <= 7'd9;
      {4'bx100,3'd0}:   firstRU <= 7'd19;
      {4'b1000,3'd0}:   firstRU <= 7'd28;
      // 52 tones RU
      {4'bxxx1,3'd1}:   firstRU <= 7'd37;
      {4'bxx10,3'd1}:   firstRU <= 7'd41;
      {4'bx100,3'd1}:   firstRU <= 7'd45;
      {4'b1000,3'd1}:   firstRU <= 7'd49;
      // 106 tones RU
      {4'bxxx1,3'd2}:   firstRU <= 7'd53;
      {4'bxx10,3'd2}:   firstRU <= 7'd55;
      {4'bx100,3'd2}:   firstRU <= 7'd57;
      {4'b1000,3'd2}:   firstRU <= 7'd59;
      // 242 tones RU
      {4'bxxx1,3'd3}:   firstRU <= 7'd61;
      {4'bxx10,3'd3}:   firstRU <= 7'd62;
      {4'bx100,3'd3}:   firstRU <= 7'd63;
      {4'b1000,3'd3}:   firstRU <= 7'd64;
      // 484 tones RU
      {4'bxx11,3'd4}:   firstRU <= 7'd65;
      {4'b1100,3'd4}:   firstRU <= 7'd66;
      // 996 tones RU
      {4'b1111,3'd5}:   firstRU <= 7'd67;
      // Incorrect/Unsupported
      default:          firstRU <= 7'h7F;
      endcase
   end
   else if (rxDataValid & rxTriggerUSRInfoEn & stateByteCnt==8'd3 &
            firstRU<rxTriggerRUAllocation[7:1])
      firstRU <= rxTriggerRUAllocation[7:1];
end


// Last eligible RU
//   - Initialized with last eligible RU for the HE-TB Bandwidth
//   - Updated with last allocated RU if lower than last eligible RU
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      lastRU <= 7'd0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      lastRU <= 7'd0;
   else if (rxDataValid & rxTriggerUSRInfoEn & stateByteCnt==8'd2)
   begin
      casex ({heTBBWBitmap,ruSizeComb})
      // 26 tones RU
      {4'b0001,3'd0}:   lastRU  <= 7'd8;
      {4'b001x,3'd0}:   lastRU  <= 7'd17;
      {4'b01xx,3'd0}:   lastRU  <= 7'd27;
      {4'b1xxx,3'd0}:   lastRU  <= 7'd36;
      // 52 tones RU            
      {4'b0001,3'd1}:   lastRU  <= 7'd40;
      {4'b001x,3'd1}:   lastRU  <= 7'd44;
      {4'b01xx,3'd1}:   lastRU  <= 7'd48;
      {4'b1xxx,3'd1}:   lastRU  <= 7'd52;
      // 106 tones RU           
      {4'b0001,3'd2}:   lastRU  <= 7'd54;
      {4'b001x,3'd2}:   lastRU  <= 7'd56;
      {4'b01xx,3'd2}:   lastRU  <= 7'd58;
      {4'b1xxx,3'd2}:   lastRU  <= 7'd60;
      // 242 tones RU           
      {4'b0001,3'd3}:   lastRU  <= 7'd61;
      {4'b001x,3'd3}:   lastRU  <= 7'd62;
      {4'b01xx,3'd3}:   lastRU  <= 7'd63;
      {4'b1xxx,3'd3}:   lastRU  <= 7'd64;
      // 484 tones RU           
      {4'b0011,3'd4}:   lastRU  <= 7'd65;
      {4'b11xx,3'd4}:   lastRU  <= 7'd66;
      // 996 tones RU           
      {4'b1111,3'd5}:   lastRU  <= 7'd67;
      // Incorrect/Unsupported
      default:          lastRU  <= 7'h7F;
      endcase
   end
   else if (rxDataValid & rxTriggerUSRInfoEn & stateByteCnt==8'd3 &
            lastRU>(rxTriggerRUAllocation[7:1]+{2'b0,numberOfRARUComb}))
      lastRU <= rxTriggerRUAllocation[7:1]+{2'b0,numberOfRARUComb};
end


// Number Of Random Access RU
//    Extracted from the combinatorial RARU information from user info field.
//    Should be used only when byte3 of the user info field is received.
assign numberOfRARUComb = raruUserInfo ? rxData[6:2] : 5'h0;


// Number of Eligible RU for current user info field
//    Availabe when bytes4 of user info field is received
//    Note: Doppler, ULDCM & ULMCS are reserved in MU-RTS trigger frame
always @*
begin
   if (rxTriggerDoppler & ~dopplerSupport & rxTriggerType!=4'd3 |
       rxTriggerULDCM   & ~dcmSupport     & rxTriggerType!=4'd3 |  
       rxTriggerULMCS   > maxMCSInHETB    & rxTriggerType!=4'd3 |
       firstRU          > lastRU                                )
      eligibleRUNbr = 7'd0;
   else
      eligibleRUNbr = lastRU-firstRU+7'b1;
end


// Indicates that the user info field allocates RA-RUs
assign raruUserInfo = rxTriggerAID0 | rxTriggerAID2045;


// Indicates that the user info field allocates RA-RUs of expected type.
assign raruUserInfoMatch = rxTriggerAID0    & ~raRUType & raRUEnable & addr2MatchBssID | 
                           rxTriggerAID2045 &  raRUType & raRUEnable ;


// OFDMA Contention Window Mask
always @*
begin
   case (eOCW)
   3'd0:    ocwMask = 7'b000_0000;
   3'd1:    ocwMask = 7'b000_0001;
   3'd2:    ocwMask = 7'b000_0011;
   3'd3:    ocwMask = 7'b000_0111;
   3'd4:    ocwMask = 7'b000_1111;
   3'd5:    ocwMask = 7'b001_1111;
   3'd6:    ocwMask = 7'b011_1111;
   default: ocwMask = 7'b111_1111;
   endcase
end


// OFDMA Contention Windows Update
//   Updated when OFDMA Back off is done and not Basic Trigger Frame.
assign eOCWIn      = eOCWMin;
assign eOCWInValid = raruUserInfo & rxTriggerUSRInfoMatch & rxTriggerType!=4'b0;


// Next OBO Counter
//    Used to decrement backoff counter during reception of trigger frame, 
//    loaded into backoff counter if frame is received correctly.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      oboCntNxt <= 7'd0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      oboCntNxt <= 7'd0;
   else if (rxDataValid & rxTriggerCMNInfoEn)
      // Initialize
      oboCntNxt <= oboCnt & ocwMask;
   else if (raruUserInfoMatch & rxDataValid & rxTriggerUSRInfoEn & stateByteCnt==8'd4)
   begin
      // Update
      if (oboCntNxt<=eligibleRUNbr)
         oboCntNxt <= 7'd0;
      else
         oboCntNxt <= oboCntNxt-eligibleRUNbr;
   end
end


// OBO Random access backoff
//   Initialized with random number when backoff is done
//   Updated when triggerFrame is completly received
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      oboCnt <= 7'd1;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      oboCnt <= ocwLFSRIn;
   else if (oboCntLoad)
      oboCnt <= ocwLFSRIn;
   else if (oboCntUpdate)
      oboCnt <= oboCntNxt;
end


// Random number generator
assign ocwLFSRIn      = {1'b0,ocwLFSR[6:1]}^({7{ocwLFSR[0]}}&7'h53);
assign ocwLFSRInValid = oboCntLoad;


// OFDMA Random access backoff controls
assign oboCntLoad   = triggerRcved    & fcsOkRcved_p & raruUserInfo &
                      ~rxTriggerFound &  rxTriggerUSRInfoMatch;
assign oboCntUpdate = triggerRcved    & fcsOkRcved_p & raruUserInfo & 
                      ~rxTriggerFound & ~rxTriggerUSRInfoMatch;


// User Info Valid flag
//    Indicates that a valid user info field has been received for TB response.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      rxTriggerUSRInfoMatch <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      rxTriggerUSRInfoMatch <= 1'b0;
   else if (rxDataValid & rxTriggerCMNInfoEn)
      rxTriggerUSRInfoMatch <= 1'b0;
   else if (rxDataValid & rxTriggerUSRInfoEn & stateByteCnt==8'd4 & eligibleRUNbr!=7'd0)
   begin
      if (rxTriggerAIDMatch)
         rxTriggerUSRInfoMatch <= 1'b1;
      else if (raruUserInfoMatch & oboCntNxt<=eligibleRUNbr)
         rxTriggerUSRInfoMatch <= 1'b1;
   end
end


// User Info Valid flag Pulse
//    Indicates that a valid user info field has been received for TB response.
always @(posedge macCoreRxClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
      rxTriggerUSRInfoMatchDly <= 1'b0;
   else if (macCoreClkSoftRst_n == 1'b0) // Synchronous Reset
      rxTriggerUSRInfoMatchDly <= 1'b0;
   else
      rxTriggerUSRInfoMatchDly <= rxTriggerUSRInfoMatch;
end

assign rxTriggerUSRInfoMatch_p = rxTriggerUSRInfoMatch & ~rxTriggerUSRInfoMatchDly;


// Debug Port
assign debugPortTriggerDecoder = {1'b0,
                                  rxTriggerAID0,
                                  rxTriggerAID2045,
                                  rxTriggerAIDMatch,
                                  eOCWInValid,
                                  oboCntLoad,
                                  oboCntUpdate,
                                  oboCntNxt,
                                  raruUserInfoMatch,
                                  rxTriggerUSRInfoMatch};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

`ifdef  RW_SIMU_ON
`endif//RW_SIMU_ON

endmodule
