//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Secondary TX Path Controller module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module secController( 
  //$port_g Clock and Reset interface
  input  wire           macCoreTxClk,            // MAC Core Transmit Clock
  input  wire           macCoreClkHardRst_n,     // Hard Reset of the MAC Core Clock domain
                                                 // (active low)
  input  wire           macCoreTxClkSoftRst_n,   // Soft Reset of the MAC Core Clock domain
                                                 // (active low)

  //$port_g Interface with the DMA Engine
  input  wire           trigTxACxMC_p,           // Transmit ACx Trigger on Core Clk domain
  input  wire           statusUpdated_p,         // Indication from the DMA that status have been
                                                 // updated.
  output reg            updateDMAStatus_p,       // trigs the DMA Status update.
  output wire           txMpduDone_p,            // Asserted high after every transmission of MPDU
                                                 // in an AMPDU frame
  output reg            ampduFrm_p,              // Asserted high if the transmitted packet was an
                                                 // AMPDU
  output reg            mpduSuccess_p,           // If the transmitted frame was successful
  output reg            mpduFailed_p,            // If the transmitted frame was not successful
  output reg      [3:0] whichDescriptorSW,       // Indicates the value of the whichDescriptorSW
                                                 // field for Status in THD

  //$port_g Interface with the Tx Parameters Cache
  input  wire           txParameterHDReady_p,    // Indicates that the Header Descriptor fields are
                                                 // usable 
  input  wire           txParameterPTReady_p,    // Indicates that the Policy Table fields are
                                                 // usable
  output wire           toggleHDSet_p,           // Indicates that the Header Descriptor Set can be
                                                 // toggled
  output wire           togglePTSet_p,           // Indicates that the Policy Table Set can be
                                                 // toggled
  output wire           clearSets_p,             // Indicates that the Sets have to be cleared.
  input  wire           aMPDU,                   // Indicates whether this Transmit Header
                                                 // Descriptor belong to an A-MPDU
  input  wire           txSingleVHTMPDU,         // Flag of single VHT MPDU
  input  wire     [6:0] mcsIndex0Tx,             // MCS Index 0 of PPDU for Transmission
  input  wire     [1:0] whichDescriptor,         // Indicates what kind of a descriptor this is
  input  wire    [19:0] aMPDUFrameLengthTx,      // Length of the entire A-MPDU
  input  wire    [19:0] aMPDUOptFrameLength20Tx, // Length of the entire A-MPDU BW drop 20Mhz
  input  wire    [19:0] aMPDUOptFrameLength40Tx, // Length of the entire A-MPDU BW drop 40Mhz
  input  wire    [19:0] aMPDUOptFrameLength80Tx, // Length of the entire A-MPDU BW drop 80Mhz

  //$port_g Interface with the Key Search Engine
  output wire           keySearchIndexTrig_p,    // Trigger for searching RAM index
  input  wire           keyStorageValid_p,       // Pulse signal indicating completion of search
                                                 // operation

  //$port_g Interface with the TX Controller
  output reg            sendData_p,              // Indicates that data packet has to be sent
  output wire           txFromFifo,              // Transmit a Frame from Tx FIFO
  input  wire           txDone_p,                // Indicates that the transmission is completed
  input  wire           mpduDone_p,              // Indicates that the MPDU has been sent
  output reg     [19:0] txHTLength,              // Length of the HT PPDU

  //$port_g Interface with the MAC controller
  input  wire           secTxStart_p,            // Secondary path start trigger pulse
  input  wire           secTxAbort_p,            // If the transmitted frame has to be aborted then
                                                 // this signal is asserted.
  input  wire     [2:0] txBWDropped,             // Indicates that the current transmission BW has
                                                 // been dropped to 20 Mhz, 40Mhz or 80Mhz
  output wire           secTxActive,             // Indicates that the secondary path is active.
                                                 // A frame has to be sent on this path.
  output wire           secTxReady,              // Indicates that the secondary path is ready,
                                                 // Descriptor & Policy Table has been read.
                                                 // secTxNSTS is valid.
  output reg      [2:0] secTxNSTS,               // Indicates the number of nSTS of the secondary
                                                 // frame. Set to '0' when inactive.


  //$port_g Interface with the MAC PHY
  input  wire           mpIfTxErr_p,             // Tx error from MAC-PHY if. resynchronized
  input  wire           macPHYIFUnderRun,        // Underrun from MAC-PHY if.

  //$port_g Debug port
  output wire    [15:0] debugPort                // Debug port
);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
// secControllerFSM FSM states definition
//$fsm_sd secControllerFSM
localparam 
  IDLE          = 3'd0,  
  WAIT_DMA      = 3'd1,  
  KEY_SEARCH    = 3'd2,  
  WAIT_PRIMARY  = 3'd3,  
  TX_DATA       = 3'd4,  
  STATUS_UPDATE = 3'd5; 


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// secControllerFSM FSM signals definition
reg   [2:0] secControllerFSMCs;        // secControllerFSM FSM Current State
reg   [2:0] secControllerFSMNs;        // secControllerFSM FSM Next State
//reg   [2:0] secControllerFSMPs;        // secControllerFSM FSM Previous State

reg         secTxStartInt;             // Captured secondary path start trigger
reg         secTxAbortInt;             // Captured secondary path abort trigger
reg         mpduSuccess;               // Indicates that the transmitted frame was successful
reg         mpduFailed;                // Indicates that the transmitted frame was not successful
wire        dmaStatusUpdateDone_p;     // Indicates that the DMA Engine has updated the status
wire        toggleHDSetForAMPDU_p;     // Pulse to indicate a toggle of Header Descriptor set in case of AMPDU
wire        toggleSetForSingleMPDU_p;  // Pulse to indicate a toggle of Header Descriptor set in case of Singleton MPDU
wire        currentTxBWDroppedTo20;    // Indicates that the current transmission BW has been dropped to 20 Mhz
wire        currentTxBWDroppedTo40;    // Indicates that the current transmission BW has been dropped to 40 Mhz
wire        currentTxBWDroppedTo80;    // Indicates that the current transmission BW has been dropped to 80 Mhz
wire        txErr;                     // Indicates that a error occured during transmission
reg         captureStatus_p;           // DMA Status information capture trigger
reg         captureStatusDly_p;        // Delayed captureStatus_p
reg   [1:0] whichDescriptorCapt;       // Captured version of whichDescriptor
reg         aMPDUCapt;                 // Captured version of aMPDU

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////
// Secondary Tx Controller FSM
//////////////////////////////////////////////////
// secControllerFSM FSM Current State Logic 
always @(posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    secControllerFSMCs <= IDLE; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    secControllerFSMCs <= IDLE; 
  else
    secControllerFSMCs <= secControllerFSMNs; 
end


// secControllerFSM FSM Previous State Logic 
//always @(posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
//begin
//  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
//    secControllerFSMPs <= IDLE; 
//  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
//    secControllerFSMPs <= IDLE; 
//  else
//    secControllerFSMPs <= secControllerFSMCs; 
//end


// secControllerFSM FSM Next State Logic.
always @* 
begin
  case(secControllerFSMCs)
  IDLE:
    //$fsm_s In IDLE state, the state machine waits for a trigger from the DMA engine
    if (trigTxACxMC_p)
      //$fsm_t When trigTxACxMC_p is received meaning that the secondary path is valid, 
      // the state machine goes to WAIT_DMA state
      secControllerFSMNs = WAIT_DMA; 
    else
      //$fsm_t While trigTxACxMC_p is not received, the state machine stays in IDLE state
      secControllerFSMNs = IDLE;

  WAIT_DMA:
    //$fsm_s In WAIT_DMA state, the state machine waits for a trigger from the txParameterCache
    if (secTxAbortInt)
      //$fsm_t When secTxAbortInt is received meaning that the frame should be retried,
      // the state machine goes to IDLE state
      secControllerFSMNs = IDLE; 
    else if (txParameterPTReady_p)
      //$fsm_t When txParameterPTReady_p is received meaning that the PT has been completed
      // loaded in the TX Parameters Cache, the state machine goes to KEY_SEARCH state
      secControllerFSMNs = KEY_SEARCH; 
    else
      //$fsm_t While txParameterPTReady_p is not received, the state machine stays in WAIT_DMA state
      secControllerFSMNs = WAIT_DMA;

  KEY_SEARCH:
    //$fsm_s In KEY_SEARCH state, the state machine waits for the completion of the key search
    // computation
    if (keyStorageValid_p) 
      //$fsm_t When keyStorageValid_p is received meaning that the key search computation is
      // done, the state machine goes to WAIT_PRIMARY state
      secControllerFSMNs = WAIT_PRIMARY; 
    else
      //$fsm_t While keyStorageValid_p is not received, the state machine stays in KEY_SEARCH state
      secControllerFSMNs = KEY_SEARCH;

  WAIT_PRIMARY:
    //$fsm_s IN WAIT_PRIMARY state, the state machine waits for the start trigger from primary 
    // MAC controller.
    if (secTxStartInt)
      //$fsm_t When secTxStartInt is received meaning that the frame should be transmit,
      // the state machine goes to TX_DATA state
      secControllerFSMNs = TX_DATA; 
    else if (secTxAbortInt)
      //$fsm_t When secTxAbortInt is received meaning that the frame should be retried,
      // the state machine goes to IDLE state
      secControllerFSMNs = IDLE; 
    else
      //$fsm_t While secTxStartInt is not received, the state machine stays in WAIT_PRIMARY state
      secControllerFSMNs = WAIT_PRIMARY; 

  TX_DATA:
    //$fsm_s In TX_DATA state, the state machine launches the frame transmission and waits until
    // its completion
    if (txErr) 
      //$fsm_t When txErr is received set meaning that a PHY error has been detected, 
      // the state machine goes to STATUS_UPDATE state.
      secControllerFSMNs = STATUS_UPDATE; 
    else if (txDone_p)
      //$fsm_t When txDone_p is received, the state machine goes to STATUS_UPDATE state
      secControllerFSMNs = STATUS_UPDATE; 
    else if (secTxAbortInt)
      //$fsm_t When secTxAbortInt is received meaning that the frame should be retried,
      // the state machine goes to IDLE state
      secControllerFSMNs = IDLE; 
    else
      //$fsm_t While txDone_p is not received, the state machine stays in TX_DATA state
      secControllerFSMNs = TX_DATA;

  STATUS_UPDATE:
    //$fsm_s In STATUS_UPDATE state, the state machine waits for the completion of the status
    // update on the DMA side.
    if (statusUpdated_p)
      //$fsm_t When the DMA has finished the status update, the state machine moves in IDLE state
      secControllerFSMNs = IDLE;
    else
      //$fsm_t While the DMA has not finished the status update, the state machine stays in
      // STATUS_UPDATE state.
      secControllerFSMNs = STATUS_UPDATE;

  // Disable coverage on the default state because it cannot be reached.
  // pragma coverage block = off 
  default:   
    secControllerFSMNs = IDLE; 
  // pragma coverage block = on 
  endcase
end


//////////////////////////////////////////////////
// MAC Controller
//////////////////////////////////////////////////
// Capture the secondary start trigger pulse, reseted when FSM changed to IDLE state
// The start trigger pulse indicate that the duration is ready and txController can be started.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    secTxStartInt <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    secTxStartInt <= 1'b0; 
  else if ((secControllerFSMCs != IDLE) && (secControllerFSMNs == IDLE))
    secTxStartInt <= 1'b0; 
  else if ((secControllerFSMCs != IDLE) && secTxStart_p)
    secTxStartInt <= 1'b1; 
end

// Capture the secTxAbort pulse, reseted when FSM is in IDLE state
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    secTxAbortInt <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    secTxAbortInt <= 1'b0; 
  else if (secControllerFSMNs == IDLE)
    secTxAbortInt <= 1'b0; 
  else if (secTxAbort_p)
    secTxAbortInt <= 1'b1; 
end

// Indicates that the secondary path is active, a frame has to be sent on this path.
assign secTxActive = (secControllerFSMCs == IDLE) ? 1'b0 : 1'b1;

// Indicates that the secondary path is ready, descriptor & Policy Table has been read. 
// (secTxNSTS is valid).
assign secTxReady  = ((secControllerFSMCs == IDLE    ) ||
                      (secControllerFSMCs == WAIT_DMA)) ? 1'b0 : 1'b1;

// Indicates the number of nSTS of the secondary frame. Set to '0' when inactive.
always @(posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    secTxNSTS <= 3'd0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    secTxNSTS <= 3'd0; 
  else if (secControllerFSMNs == IDLE)
    secTxNSTS <= 3'd0; 
  else if ((secControllerFSMCs == WAIT_DMA) && (secControllerFSMNs != WAIT_DMA))
    secTxNSTS <= mcsIndex0Tx[6:4]+3'd1; 
end


//////////////////////////////////////////////////
// Key Search Engine Control
//////////////////////////////////////////////////
// When the FSM moves to KEY_SEARCH, the key search engine is trigged to return the encryption
// informations.
assign keySearchIndexTrig_p = ((secControllerFSMCs != KEY_SEARCH) &&
                               (secControllerFSMNs == KEY_SEARCH)) ? 1'b1 : 1'b0; 


//////////////////////////////////////////////////
// DMA Status Update
//////////////////////////////////////////////////
// Indicates when the DMA Status has to be updated.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    updateDMAStatus_p  <= 1'b0; 
    captureStatus_p    <= 1'b0; 
    captureStatusDly_p <= 1'b0; 
  end
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
  begin
    updateDMAStatus_p  <= 1'b0; 
    captureStatus_p    <= 1'b0; 
    captureStatusDly_p <= 1'b0; 
  end
  else
  begin
    captureStatusDly_p <= captureStatus_p; 
    updateDMAStatus_p  <= captureStatusDly_p; 
    if ((secControllerFSMCs != STATUS_UPDATE) && (secControllerFSMNs == STATUS_UPDATE))
      captureStatus_p  <= 1'b1; 
    else  
      captureStatus_p  <= 1'b0;
  end     
end


// Indicates when the DMA has updated the status.
assign dmaStatusUpdateDone_p = ((secControllerFSMCs == STATUS_UPDATE) &&
                                (secControllerFSMNs != STATUS_UPDATE)) ? 1'b1 : 1'b0; 


// DMA Status : MPDU part of an A-MPDU transmitted
assign txMpduDone_p = aMPDUCapt && mpduDone_p;


always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    ampduFrm_p          <= 1'b0;
    mpduSuccess_p       <= 1'b0;
    mpduFailed_p        <= 1'b0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || statusUpdated_p)  // Synchronous Reset
  begin
    ampduFrm_p          <= 1'b0;
    mpduSuccess_p       <= 1'b0;
    mpduFailed_p        <= 1'b0;
  end
  else if (captureStatusDly_p)
  begin
    // DMA Status : Transmitted frame was an A-MPDU
    ampduFrm_p     <= aMPDUCapt;

    // DMA Status : frame has been successfully transmitted
    mpduSuccess_p  <= mpduSuccess;

    // DMA Status : frame has not been successfully transmitted
    mpduFailed_p   <= mpduFailed;
  end
end


// if the frame completed the transmission,
// the flag mpduSuccess is set.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    mpduSuccess <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (dmaStatusUpdateDone_p))  // Synchronous Reset
    mpduSuccess <= 1'b0; 
  else
    if ((secControllerFSMCs == TX_DATA) && txDone_p)    
      mpduSuccess <= 1'b1; 
end
 
                    
// When error is detected, the flag mpduFailed is set
// to report a transmission failure.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    mpduFailed <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (dmaStatusUpdateDone_p))  // Synchronous Reset
    mpduFailed <= 1'b0; 
  else
  begin
    if ((secControllerFSMCs == TX_DATA) && txErr)
      mpduFailed <= 1'b1; 
  end    
end

                    
// DMA Status : whichDescriptorSW
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    whichDescriptorSW <= 4'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    whichDescriptorSW <= 4'b0; 
  else if (secControllerFSMCs == STATUS_UPDATE || txMpduDone_p)
    whichDescriptorSW <= {1'b0/*UnderBASetup*/,aMPDUCapt,whichDescriptorCapt}; 
end


//////////////////////////////////////////////////
// 20/40/80/160MHz bandwidth
//////////////////////////////////////////////////
// Indicates if current transmission BW has been changed with drop mechanism
assign currentTxBWDroppedTo20 = txBWDropped[0];
assign currentTxBWDroppedTo40 = txBWDropped[1];
assign currentTxBWDroppedTo80 = txBWDropped[2];


//////////////////////////////////////////////////
// TX Vector Generation
//////////////////////////////////////////////////
// aMPDU length selection in case of bandwidth drop 
always @*
begin
  if (currentTxBWDroppedTo20 && (aMPDUOptFrameLength20Tx != 20'd0))
  begin
    txHTLength = aMPDUOptFrameLength20Tx;
  end
  else if (currentTxBWDroppedTo40 && (aMPDUOptFrameLength40Tx != 20'd0))
  begin
    txHTLength = aMPDUOptFrameLength40Tx;
  end
  else if (currentTxBWDroppedTo80 && (aMPDUOptFrameLength80Tx != 20'd0))
  begin
    txHTLength = aMPDUOptFrameLength80Tx;
  end
  else
  begin
    txHTLength = aMPDUFrameLengthTx;
  end
end


//////////////////////////////////////////////////
// TX Controller management
//////////////////////////////////////////////////
// The sendData_p pulse is generated when the FSM enters in TX_DATA state
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sendData_p <= 1'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    sendData_p <= 1'b0;
  else
  begin
    if ((secControllerFSMCs != TX_DATA) && (secControllerFSMNs == TX_DATA))
      sendData_p <= 1'b1;
    else
      sendData_p <= 1'b0;
  end
end

assign txFromFifo = (secControllerFSMCs == TX_DATA)  ? 1'b1 : 1'b0;

// Indicates that a error occured during transmission
assign txErr = mpIfTxErr_p || macPHYIFUnderRun;


//////////////////////////////////////////////////
// TX Parameter Cache Control
//////////////////////////////////////////////////
assign clearSets_p   = toggleSetForSingleMPDU_p || secTxAbort_p;
assign toggleHDSet_p = toggleSetForSingleMPDU_p || toggleHDSetForAMPDU_p;
assign togglePTSet_p = 1'b0;

assign toggleHDSetForAMPDU_p    = (sendData_p || mpduDone_p) && (whichDescriptorCapt != 2'b11);
assign toggleSetForSingleMPDU_p = ((secControllerFSMCs == STATUS_UPDATE) && statusUpdated_p) ? 1'b1 : 1'b0;


// Capture whichDescriptor coming from txParametersCache
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    whichDescriptorCapt <= 2'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    whichDescriptorCapt <= 2'b0;
  else if (txParameterHDReady_p)
    whichDescriptorCapt <= whichDescriptor;
end 
 
// Capture aMPDU coming from txParametersCache
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    aMPDUCapt <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    aMPDUCapt <= 1'b0; 
  else if (txParameterPTReady_p == 1'b1)
      aMPDUCapt <= aMPDU && !txSingleVHTMPDU;
end

//////////////////////////////////////////////////
// Debug Port
//////////////////////////////////////////////////
assign debugPort = {13'b0,
                    secControllerFSMCs};
                 

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef  RW_SIMU_ON
// Disable coverage on the default state because it cannot be reached.
// pragma coverage block = off 

// String definition to display secControllerFSM current state
reg [21*8-1:0] secControllerFSMCs_str;

// secControllerFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (secControllerFSMCs)
  IDLE          : secControllerFSMCs_str = {"IDLE         "};
  WAIT_DMA      : secControllerFSMCs_str = {"WAIT_DMA     "};
  KEY_SEARCH    : secControllerFSMCs_str = {"KEY_SEARCH   "};
  WAIT_PRIMARY  : secControllerFSMCs_str = {"WAIT_PRIMARY "};
  TX_DATA       : secControllerFSMCs_str = {"TX_DATA      "};
  STATUS_UPDATE : secControllerFSMCs_str = {"STATUS_UPDATE"};
  default       : secControllerFSMCs_str = {"XXX          "};
  endcase
end
`endif//RW_SIMU_ON


endmodule
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
