////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Spatial Reuse Controller
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module srController( 
   ///////////////////////////////////////////////
   //$port_g Clock and reset
   ///////////////////////////////////////////////
   input  wire        macCoreClk,               // MAC Core Clock
   input  wire        macCoreClkHardRst_n,      // Hard Reset of the MAC Core Clock domain
   input  wire        macCoreClkSoftRst_n,      // Soft Reset of the MAC Core Clock domain
   
   ///////////////////////////////////////////////
   //$port_g MAC Timer Unit
   ///////////////////////////////////////////////
   input  wire        tick1us_p,                // Pulse generated every us
   input  wire        tickTBTT_p,               // Pulse every TBTTs

   ///////////////////////////////////////////////
   //$port_g A-MPDU Deaggregator
   ///////////////////////////////////////////////
   input  wire        rxVector1Start_p,         // Indicates that the first byte of RX Vector
                                                // has been received
   input  wire        rxVector1Valid_p,         // Rx vector 1 is available
   input  wire  [3:0] rxFormatMod,              // Format and modulation of received frame
   input  wire  [3:0] rxSpatialReuse1,          // Indicates the spatial reuse parameter value
   input  wire  [7:0] rxRSSILegacy,             // RSSI during Legacy preamble of received frame
   input  wire  [8:0] rxPartialAID,             // rxPartialAID of received frame
   input  wire  [5:0] rxGroupID,                // GroupID of received frame
   input  wire  [5:0] rxBssColor,               // Indicate the BSS color of the AP
   input  wire        rxUplinkFlag,             // Indicates if the HE PPDU is addressed to an AP
   input  wire        rxNDP,                    // Indicate that the received frame is a Null Data
                                                // Packet
   input  wire        rxAggregation,            // MPDU aggregate of received frame
   input  wire        rxSMPDU,                  // Indicate that the aMPDU is an SMPDU 

   ///////////////////////////////////////////////
   //$port_g TX Controller
   ///////////////////////////////////////////////
   input  wire        startTx_p,                // Start Tx trigger                 
   input  wire  [3:0] formatMod,                // TX Vector Format and Modulation           
   input  wire  [3:0] spatialReuse1,            // TX Vector Spatial Reuse 1 Information
   input  wire  [3:0] spatialReuse2,            // TX Vector Spatial Reuse 2 Information
   input  wire  [3:0] spatialReuse3,            // TX Vector Spatial Reuse 3 Information
   input  wire  [3:0] spatialReuse4,            // TX Vector Spatial Reuse 4 Information
   output reg         obssPDSRPeriod,           // Indicates that an OBSS PD SR power restriction
                                                // period
   output reg   [7:0] obssPDSRPeriodTxPwrMax,   // Indicates the maximum transmit power of the 
                                                // OBSS PD SR power restriction period

   ///////////////////////////////////////////////
   //$port_g RX Controller
   ///////////////////////////////////////////////
   input  wire        basicNAVClear_p,          // Clear basic NAV
   input  wire        notMineRcved,             // Frame received but not mine
   input  wire        rtsRcved,                 // RTS received
   input  wire        ctsRcved,                 // CTS received
   input  wire        ackRcved,                 // ACK received
   input  wire        baRcved,                  // Block Ack received
   input  wire        ftmRcved,                 // FTM received
   input  wire        ndpaRcved,                // NDP-A received
   input  wire        publicActionRcved,        // Public Action frame received
   input  wire [47:0] rxAddr1,                  // Addr1 of the received frame (RA)
   input  wire [47:0] rxAddr2,                  // Addr2 of the received frame (TA)
   input  wire  [5:0] partialBSSID,             // BSSID[44:39] of the received Frame
   input  wire        bssIDFrame,               // Frame contains BSSID field

   input  wire        fcsOkRcved_p,             // Frame received with correct FCS
   input  wire        rxEndOfFrame_p,           // End Of Frame information
   input  wire        interBSSRcved,            // The received frame is classified
                                                // as an inter-BSS frame
   output reg         stopRx_p,                 // Stop Reception trigger

   ///////////////////////////////////////////////
   //$port_g MAC Controller
   ///////////////////////////////////////////////
   input  wire        endOfTxOp,                // Indicates the end of the TX OP.

   ///////////////////////////////////////////////
   //$port_g Control and Status Register
   ///////////////////////////////////////////////
   input  wire  [7:0] slotTime,                 // aSlotTime parameter
   input  wire  [7:0] sifs,                     // Provide SIFS duration in us
   input  wire  [7:0] rxStartDelayOFDM,         // Receive Start Delay for OFDM
   input  wire        srParameterSetValid,      // Indicates that the Spatial reuse parameters
                                                // are valid
   input  wire        srpDisallowed,            // Indicates whether SRP-based SR transmissions are 
                                                // allowed or not
   input  wire        nonSRGOBSSPDSRDisallowed, // Indicates whether non-SRG OBSS PD SR 
                                                // transmissions are allowed or not
   input  wire        nonSRGOffsetPresent,      // Indicates that the non-SRG OBSS PD Max Offset
                                                // is valid
   input  wire  [7:0] nonSRGOBSSPDMaxOffset,    // Indicates the non-SRG OBSS PD Max Offset value
   input  wire        srgInformationPresent,    // Indicates that the SRG OBSS PD parameters are
                                                // valid
   input  wire  [7:0] srgOBSSPDMaxOffset,       // Indicates the SRG OBSS PD Max Offset value
   input  wire  [7:0] srgOBSSPDMinOffset,       // Indicates the SRG OBSS PD Min Offset value
   input  wire [63:0] srgBSSColorBitmap,        // Indicates the SRG BSS Color Bitmap
   input  wire [63:0] srgPartialBSSIDBitmap,    // Indicates the SRG Partial BSSID Bitmap
   input  wire        enableOBSSPD,             // Enable OBSS PD Spatial Reuse
   input  wire        enableSRP,                // Enable SRP Spatial Reuse

   ///////////////////////////////////////////////
   //$port_g NAV interface
   ///////////////////////////////////////////////
   output wire        basicNAVUpdateMask,       // Indicates the basic NAV should not be updated due
                                                // to Spatial Reuse opportunity.

   ///////////////////////////////////////////////
   //$port_g Debug port
   ///////////////////////////////////////////////
   output wire [15:0] debugPortSRController     // Debug port for validation
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////

// Spatial Reuse field encoding for an HE SU PPDU, HE ER SU PPDU, and HE MU PPDU
localparam SRP_DISALLOWED                     = 4'd0;
localparam SR_RESTRICTED                      = 4'd13;
localparam SR_DELAY                           = 4'd14;
localparam SRP_AND_NON_SRG_OBSS_PD_PROHIBITED = 4'd15;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg           startTxDly_p;            // Start Tx trigger with one cycle Delay
reg           obssPDSRPPDU;            // Indicates a PPDU that can used for OBSS
                                       // PD Spatial reuse
reg     [7:0] rxRSSILegacyNorm;        // Normalized RSSI during Legacy preamble of
                                       // received frame
wire          srDelayPPDU;             // Indicates a PPDU with Spatial reuse SR_DELAY
wire          srRestrictedPPDU;        // Indicates a PPDU with Spatial reuse SR_RESTRICTED
wire          sr15PPDU;                // Indicates a PPDU with Spatial reuse 
                                       // SRP_AND_NON_SRG_OBSS_PD_PROHIBITED

reg           txSpatialReuse15;        // Indicates that a frame has been transmitted
                                       // With spatial reuse 15 (SRP_AND_NON_SRG_OBSS_PD_PROHIBITED)
                                       // in the current beacon period
reg           txSpatialReuse15Prev;    // Indicates that a frame has been transmitted
                                       // With spatial reuse 15 (SRP_AND_NON_SRG_OBSS_PD_PROHIBITED)
                                       // in the previous beacon period

wire          nonSRGPPDU;              // non-SRG PPDU detected
wire          nonSRGPPDU_p;            // non-SRG PPDU detected Pulse
reg           nonSRGRTS;               // non-SRG RTS received
wire    [7:0] nonSRGOBSSPDMax;         // Maximum receive power for non-SRG OBSS PD SR
reg           nonSRGOBSSPDPPDU;        // Indicates a PPDU that can used for non-SRG OBSS PD
                                       // Spatial reuse
wire          srgPPDU;                 // SRG PPDU detected
wire          srgPPDU_p;               // SRG PPDU detected pulse
wire    [7:0] srgOBSSPDMax;            // Maximum receive power for SRG OBSS PD SR
reg           srgOBSSPDPPDU;           // Indicates a PPDU that can used for SRG OBSS PD
                                       // Spatial reuse
wire          nonSRGRxRSSICorrect;     // Indicates that the receive RSSI match the 
                                       // non-SRG OBSS PD-based requirement
wire          srgRxRSSICorrect;        // Indicates that the receive RSSI match the 
                                       // SRG OBSS PD-based requirement
reg     [9:0] pifsCnt;                 // PIFS Counter
reg           rxPIFSAfterNONSRGRTS;    // Received a frame PIFS after non-SRG RTS

wire          obssPDSRPeriod_p;        // OBSS PD Spatial Reuse Period detected pulse
wire    [7:0] obssPDMinOffset;         // Indicates the OBSS PD Min Offset value 
wire    [9:0] obssPDTxPwrMaxTmp;       // Indicates the OBSS PD Maximum transmit power
reg     [7:0] obssPDTxPwrMax;          // Indicates the OBSS PD Maximum transmit power

reg           fcsOkRcvedDly_p;         // fcsOkRcved_p delayed by 1 clock cycle
wire          addr2Pres;               // Indicates that the received frame has an addr2 field


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------------
// non-SRG OBSS PD-based Spatial Reuse
//------------------------------------------------------------------------------

// Delayed StartTx_p pulse
//    Tx Vector parameters are provided one cycle after the startTx_p pulse.
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      startTxDly_p <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      startTxDly_p <= 1'b0;
   else
      startTxDly_p <= startTx_p;
end

// Transmitted Spatial reuse SRP_AND_NON_SRG_OBSS_PD_PROHIBITED 
//    Indicates that a frame has been transmitted With spatial reuse value 
//    SRP_AND_NON_SRG_OBSS_PD_PROHIBITED in the current beacon period
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      txSpatialReuse15 <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      txSpatialReuse15 <= 1'b0;
   else if (tickTBTT_p)
      txSpatialReuse15 <= 1'b0;
   else if (startTxDly_p & formatMod>=4'd5 &
            (spatialReuse1==SRP_AND_NON_SRG_OBSS_PD_PROHIBITED |
             spatialReuse2==SRP_AND_NON_SRG_OBSS_PD_PROHIBITED |
             spatialReuse3==SRP_AND_NON_SRG_OBSS_PD_PROHIBITED |
             spatialReuse4==SRP_AND_NON_SRG_OBSS_PD_PROHIBITED ))
      txSpatialReuse15 <= 1'b1;
end

// Transmitted Spatial reuse SRP_AND_NON_SRG_OBSS_PD_PROHIBITED 
//    Indicates that a frame has been transmitted With spatial reuse value 
//    SRP_AND_NON_SRG_OBSS_PD_PROHIBITED in the previous beacon period
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      txSpatialReuse15Prev <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      txSpatialReuse15Prev <= 1'b0;
   else if (tickTBTT_p)
      txSpatialReuse15Prev <= txSpatialReuse15;
end

// non-SRG OBSS PD-based RTS received flag
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      nonSRGRTS <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      nonSRGRTS <= 1'b0;
   else if (obssPDSRPeriod_p & rtsRcved)
      nonSRGRTS <= 1'b1;
   else if (rxVector1Start_p)
      nonSRGRTS <= 1'b0;
end

// PIFS Counter
//    Down Counter loaded with PIFS value on non-SRG OBSS PD-based RTS reception
//    and decremented until zero.
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      pifsCnt <= 10'h0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      pifsCnt <= 10'h0;
   else if (obssPDSRPeriod_p & rtsRcved)
      pifsCnt <= {2'b0,sifs} + {2'b0,slotTime} + {2'b0,rxStartDelayOFDM};
   else if (rxVector1Start_p)
      pifsCnt <= 10'h0;
   else if (tick1us_p & pifsCnt!=10'h0)
      pifsCnt <= pifsCnt - 10'h1;
end

// Received frame PIFS after non-SRG OBSS PD-based RTS
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      rxPIFSAfterNONSRGRTS <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      rxPIFSAfterNONSRGRTS <= 1'b0;
   else if (rxVector1Start_p & nonSRGRTS & pifsCnt!=10'd0)
      rxPIFSAfterNONSRGRTS <= 1'b1;
   else if (rxVector1Start_p)
      rxPIFSAfterNONSRGRTS <= 1'b0;
end

// non-SRG OBSS PD Max
// Must be clarified:
//    Why non-SRG OBSS PD Max is defined (in table 27-10, 802.11ax 3.0) when
//    nonSRGOBSSPDSRDisallowed is '1' ?
assign nonSRGOBSSPDMax = (srParameterSetValid &
                          nonSRGOffsetPresent ) ? nonSRGOBSSPDMaxOffset - 8'd82 :
                                                  -8'd62;

// RSSI Legacy compare with non-SRG OBSS PD Max
//    Indicates that the receive RSSI during Legacy preamble is correct for the
//    non-SRG OBSS PD spatial Reuse
assign nonSRGRxRSSICorrect = (rxRSSILegacyNorm^8'h80) <= (nonSRGOBSSPDMax^8'h80);

// non-SRG PPDU Detection
//    Inter-BSS received frame
//    and non-SRG OBSS Spatial Reuse not prohibited
//    and not an NDP
//    and not a non-HT response frame (Ack, BlockAck or CTS)
//    or CTS Frame received PIFS after a non-SRG RTS.
//
//    Restricted Spatial Reuse is not yet supported
//    (SR TXOP must be limited to the duration of the HE-MU PPDU)
assign nonSRGPPDU = enableOBSSPD                                                  &
                    ~(srParameterSetValid & nonSRGOBSSPDSRDisallowed)             &
                    ~txSpatialReuse15                                             &
                    ~txSpatialReuse15Prev                                         &
                    ~sr15PPDU                                                     &
                    ~srRestrictedPPDU                                             &
                    ~rxNDP                                                        &
                    ~(rxFormatMod[3:1]==3'd0 & ackRcved             )             &
                    ~(rxFormatMod[3:1]==3'd0 & baRcved              )             &
                    (~(rxFormatMod[3:1]==3'd0 & ctsRcved) | rxPIFSAfterNONSRGRTS) &
                    nonSRGRxRSSICorrect                                           ;

// non-SRG OBSS PD-based PPDU Detection Pulse
assign nonSRGPPDU_p = nonSRGPPDU & fcsOkRcved_p;

// non-SRG OBSS PD-based PPDU Flag
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
       nonSRGOBSSPDPPDU <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
       nonSRGOBSSPDPPDU <= 1'b0;
   else if (rxVector1Start_p | rxEndOfFrame_p)
       nonSRGOBSSPDPPDU <= 1'b0;
   else if (nonSRGPPDU_p)
       nonSRGOBSSPDPPDU <= 1'b1;
end


//------------------------------------------------------------------------------
// SRG OBSS PD
//------------------------------------------------------------------------------

// Address2 present flag
assign addr2Pres = ~ackRcved & ~ctsRcved;

// SRG OBSS PD Max
assign srgOBSSPDMax = srgOBSSPDMaxOffset - 8'd82;

// RSSI Legacy compare with SRG OBSS PD Max
//    Indicates that the receive RSSI during Legacy preamble is correct for the
//    SRG OBSS PD spatial Reuse
assign srgRxRSSICorrect = (rxRSSILegacyNorm^8'h80) <= (srgOBSSPDMax^8'h80);

// SRG PPDU Detection 
//    Restricted Spatial Reuse is not yet supported
//    (SR TXOP must be limited to the duration of the HE-MU PPDU)
assign srgPPDU =
   enableOBSSPD                  &
   srParameterSetValid           &
   srgInformationPresent         &
   srgRxRSSICorrect              &
   ~srRestrictedPPDU             &
   ~rxNDP                        &
   (rxFormatMod>=4'd5 & srgBSSColorBitmap[rxBssColor]                                        |
    rxFormatMod==4'd4 & srgPartialBSSIDBitmap[rxPartialAID[5:0]] & rxGroupID==6'd0           |
    bssIDFrame        & srgPartialBSSIDBitmap[partialBSSID]                                  |
    rxFormatMod==4'd4 & srgPartialBSSIDBitmap[rxAddr1[44:39]] & rxGroupID==6'd0              |
    rxFormatMod==4'd4 & srgPartialBSSIDBitmap[rxAddr2[44:39]] & rxGroupID==6'd63 & addr2Pres |
    rxFormatMod==4'd5 & srgPartialBSSIDBitmap[rxAddr1[44:39]] & rxUplinkFlag                 |
    rxFormatMod==4'd6 & srgPartialBSSIDBitmap[rxAddr1[44:39]] & rxUplinkFlag                 |
    rxFormatMod==4'd7 & srgPartialBSSIDBitmap[rxAddr1[44:39]] & rxUplinkFlag                 );

// SRG OBSS PD-based PPDU Detection Pulse
assign srgPPDU_p = srgPPDU & fcsOkRcved_p;

// SRG OBSS PD-based PPDU
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
       srgOBSSPDPPDU <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
       srgOBSSPDPPDU <= 1'b0;
   else if (rxVector1Start_p | rxEndOfFrame_p)
       srgOBSSPDPPDU <= 1'b0;
   else if (srgPPDU_p)
       srgOBSSPDPPDU <= 1'b1;
end


//------------------------------------------------------------------------------
// Common OBSS PD-based Spatial Reuse
//------------------------------------------------------------------------------

// OBSS PD SR PPDU
//    Indicates that the received frame can be used for OBSS PD Spatial Reuse
//    This flag is set when RX Vector reception is started and clear when 
//    forbidden MPDU is received.
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      obssPDSRPPDU <= 1'b1;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      obssPDSRPPDU <= 1'b1;
   else if (rxVector1Start_p)
      obssPDSRPPDU <= 1'b1;
   else if (rxVector1Valid_p & rxNDP)
      obssPDSRPPDU <= 1'b0;
   else if (fcsOkRcved_p & rxFormatMod<4'd5 &
            (~notMineRcved | publicActionRcved | ndpaRcved | ftmRcved))
      obssPDSRPPDU <= 1'b0;
end

// Normalized received RSSI during Legacy preamble
//   In an HE ER SU PPDU, the power of the L-STF/L-LTF symbols is boosted 3 dB,
//   the RSSI Legacy shall be decreased by 3 dB to compensate for the power boost
//   factor when compared to the OBSS PD level.
always @*
begin
   if (rxFormatMod!=4'd7)
      // Received PDDU is not an HE-ER-SU PPDU
      rxRSSILegacyNorm = rxRSSILegacy;
   else if (rxRSSILegacy[7] & rxRSSILegacy[6:0]<7'h3)
      // Received PPDU is an HE-ER-SU and RSSI Legacy lower than -125
      rxRSSILegacyNorm = 8'h80;
   else
      // Received PPDU is an HE-ER-SU and RSSI Legacy higher or equal to -125
      rxRSSILegacyNorm = rxRSSILegacy - 8'd3;
end

// Received frame with Spatial Reuse field set to SR_DELAY
//    Reception should not be stopped before end of PPDU.
assign srDelayPPDU      = (rxFormatMod==4'd5 |
                           rxFormatMod==4'd7 ) & rxSpatialReuse1==SR_DELAY;

// Received frame with Spatial Reuse field set to SR_RESTRICTED
assign srRestrictedPPDU = (rxFormatMod==4'd5 |
                           rxFormatMod==4'd6 |
                           rxFormatMod==4'd7 ) & rxSpatialReuse1==SR_RESTRICTED;

// Received frame with Spatial Reuse field set to SR_RESTRICTED
assign sr15PPDU         = (rxFormatMod>=4'd5 ) & rxSpatialReuse1==SRP_AND_NON_SRG_OBSS_PD_PROHIBITED;

// Maximum transmit power for the OBSS PD-based
//    In case of the PPDU match both  non-SRG OBSS PD-based & SRG OBSS PD-based
//    requirement, the minimum OBSS-PD Offset from non-SRG OBSS PD-based PPDU is
//    used, more restrictive than the SRG OBSS PD-based Minimum offset.
assign obssPDMinOffset   = nonSRGOBSSPDPPDU ? 8'd0 : srgOBSSPDMinOffset;
assign obssPDTxPwrMaxTmp = {2'b00,obssPDMinOffset} -
                           10'd61                  -
                           {rxRSSILegacyNorm[7],rxRSSILegacyNorm[7],rxRSSILegacyNorm};

always @*
begin
   if (obssPDTxPwrMaxTmp[9] & obssPDTxPwrMaxTmp[8:7]!=2'b11)
      // Negative overflow
      obssPDTxPwrMax = -8'd128;
   else if ((obssPDTxPwrMaxTmp^10'h200)>=(10'd21^10'h200))
      // Unconstrained transmit power (TxPwrMax > 21)
      obssPDTxPwrMax = 8'd127;
   else
      obssPDTxPwrMax = obssPDTxPwrMaxTmp[7:0];
end


//------------------------------------------------------------------------------
// OBSS PD Spatial Reuse transmit power restriction period
//------------------------------------------------------------------------------

// fcsOkRcved_p delayed by 1 clock cycle
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      fcsOkRcvedDly_p <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      fcsOkRcvedDly_p <= 1'b0;
   else
      fcsOkRcvedDly_p <= fcsOkRcved_p;
end

// Stop reception
//   In case of OBSS PD PPDU detected on correctly received frame, the reception is aborted.
//   The reception is never aborted in case of spatial reuse SR_DELAY
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      stopRx_p <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      stopRx_p <= 1'b0;
   else if (~srDelayPPDU & rxAggregation & ~rxSMPDU & basicNAVUpdateMask & fcsOkRcvedDly_p)
      stopRx_p <= 1'b1;
   else
      stopRx_p <= 1'b0;
end

// BASIC NAV update mask
//    The BASIC NAV update is masked when OBSS PPDU and reception is not stopped.
assign basicNAVUpdateMask = nonSRGOBSSPDPPDU & obssPDSRPPDU & interBSSRcved |
                            srgOBSSPDPPDU    & obssPDSRPPDU & interBSSRcved;

// OBSS PD Spatial Reuse Period detected pulse
assign obssPDSRPeriod_p = basicNAVUpdateMask & rxEndOfFrame_p;

// OBSS PD Power Restriction period
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      obssPDSRPeriod <= 1'b0;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      obssPDSRPeriod <= 1'b0;
   else if (obssPDSRPeriod_p)
      obssPDSRPeriod <= 1'b1;
   else if (endOfTxOp | basicNAVClear_p)
      obssPDSRPeriod <= 1'b0;
end

// OBSS PD SR Power Restriction
always @(posedge macCoreClk or negedge macCoreClkHardRst_n)
begin
   if (macCoreClkHardRst_n==1'b0)      // Asynchronous Reset
      obssPDSRPeriodTxPwrMax <= 8'd127;
   else if (macCoreClkSoftRst_n==1'b0) // Synchronous Reset
      obssPDSRPeriodTxPwrMax <= 8'd127;
   else if (obssPDSRPeriod_p & ((obssPDTxPwrMax^8'h80)<(obssPDSRPeriodTxPwrMax^8'h80)))
      obssPDSRPeriodTxPwrMax <= obssPDTxPwrMax;
   else if (endOfTxOp | basicNAVClear_p)
      obssPDSRPeriodTxPwrMax <= 8'd127;
end


//------------------------------------------------------------------------------
// SRP Responder
//------------------------------------------------------------------------------
//ToDo...


//------------------------------------------------------------------------------
// Debug Port
//------------------------------------------------------------------------------
assign debugPortSRController = {1'b0,
                                txSpatialReuse15Prev,
                                txSpatialReuse15,
                                obssPDSRPPDU,
                                srgRxRSSICorrect,
                                1'b0,
                                srgPPDU_p,
                                srgOBSSPDPPDU,
                                nonSRGRxRSSICorrect,
                                1'b0,
                                nonSRGPPDU_p,
                                nonSRGOBSSPDPPDU,
                                obssPDSRPeriod_p,
                                obssPDSRPeriod,
                                basicNAVUpdateMask,
                                stopRx_p};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef  RW_SIMU_ON

`endif//RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef  RW_ASSERT_ON

`endif//RW_ASSERT_ON


endmodule
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
